/*
 * PMX shape information handler
 *
 * License : The MIT License
 * Copyright(c) 2015 MikuToga Partners
 */

package jp.sfjp.mikutoga.pmx.parser;

import jp.sfjp.mikutoga.bin.parser.LoopHandler;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;

/**
 * PMXモデルの頂点通知用ハンドラ.
 * <p>0から始まる. 頂点ID順に頂点は出現する.
 */
public interface PmxVertexHandler extends LoopHandler {

    /** 頂点抽出ループ。 */
    ParseStage VERTEX_LIST = new ParseStage();

    /**
     * 頂点の座標の通知を受け取る.
     * <p>{@link #VERTEX_LIST}ループの構成要素
     * @param xPos X座標
     * @param yPos Y座標
     * @param zPos Z座標
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxVertexPosition(float xPos, float yPos, float zPos)
            throws MmdFormatException;

    /**
     * 頂点の法線情報の通知を受け取る.
     * <p>{@link #VERTEX_LIST}ループの構成要素
     * <p>※単位ベクトル化必須？
     * @param xVec 法線ベクトルX成分
     * @param yVec 法線ベクトルY成分
     * @param zVec 法線ベクトルZ成分
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxVertexNormal(float xVec, float yVec, float zVec)
            throws MmdFormatException;

    /**
     * 頂点のUVマッピング情報の通知を受け取る.
     * <p>{@link #VERTEX_LIST}ループの構成要素
     * @param uVal テクスチャのU座標
     * @param vVal テクスチャのV座標
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxVertexUV(float uVal, float vVal )
            throws MmdFormatException;

    /**
     * 頂点の追加UVマッピング情報の通知を受け取る.
     * <p>{@link #VERTEX_LIST}ループの構成要素
     * @param i 追加UVテクスチャのインデックス
     * @param xVal テクスチャのx座標
     * @param yVal テクスチャのy座標
     * @param zVal テクスチャのz座標
     * @param wVal テクスチャのw座標
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxVertexExUV(byte i, float xVal, float yVal, float zVal, float wVal)
            throws MmdFormatException;

    /**
     * 頂点のボーン間ウェイトバランス情報の通知を受け取る。
     * <p>{@link #VERTEX_LIST}ループの構成要素
     * @param boneIds ボーン識別子
     * @param weight ボーンの影響度. 0(min)～1(max). 合計は 1 になるはず.
     * @param sdef SDEF 3ベクトル. BDEFの場合は null.
     * [0-2] SDEF-C値(x,y,z)
     * [3-5] SDEF-R0値(x,y,z)
     * [6-8] SDEF-R1値(x,y,z)
     * 
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxVertexWeight(int [] boneIds, float [] weight,float [] sdef)
            throws MmdFormatException;

    /**
     * 頂点のエッジ表現の倍率の情報の通知を受け取る。
     * <p>{@link #VERTEX_LIST}ループの構成要素
     * @param edge 素材で指定されたエッジ厚さにかける倍率.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxVertexEdge(float edge)
            throws MmdFormatException;

}
