/* 
 * Copyright (C) 2001-2007 Jacek Sieka, arnetheduck on gmail point com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "stdinc.h"
#include "DCPlusPlus.h"

#include "Client.h"
#include "version.h"

#include "BufferedSocket.h"

#include "FavoriteManager.h"
#include "TimerManager.h"
#include "ResourceManager.h"
#include "ClientManager.h"
#include "UploadManager.h"
#include "PGManager.h"

Client::Counts Client::counts;

Client::Client(const string& hubURL, char separator_, bool secure_) : 
	myIdentity(ClientManager::getInstance()->getMe(), 0),
	reconnDelay(120), lastActivity(GET_TICK()), registered(false), autoReconnect(false),
	encoding(const_cast<string*>(&Text::systemCharset)), state(STATE_DISCONNECTED), socket(0),
	hubUrl(hubURL), port(0), separator(separator_),
	secure(secure_), countType(COUNT_UNCOUNTED), availableBytes(0)
{
	string file;
	Util::decodeUrl(hubURL, address, port, file);

	TimerManager::getInstance()->addListener(this);
}

Client::~Client() throw() {
	dcassert(!socket);
	
	// In case we were deleted before we Failed
	FavoriteManager::getInstance()->removeUserCommand(getHubUrl());
	TimerManager::getInstance()->removeListener(this);
	updateCounts(true);
}

void Client::reconnect() {
	disconnect(true);
	setAutoReconnect(true);
	setReconnDelay(0);
}

void Client::shutdown() {

	if(socket) {
		BufferedSocket::putSocket(socket);
		socket = 0;
	}
}

void Client::reloadSettings(bool updateNick) {
	const FavoriteHubEntry* hub = FavoriteManager::getInstance()->getFavoriteHubEntry(getHubUrl());
	
	string speedDescription = Util::emptyString;

#ifdef BETA
	if(!SETTING(BETAUSR).empty())
		speedDescription += "[Auth:" + SETTING(BETAUSR) + "]";
#endif

	if (getStealth() == false) {
		if(BOOLSETTING(SHOW_DESCRIPTION_SLOTS))
			speedDescription += "[" + Util::toString(UploadManager::getInstance()->getFreeSlots()) + "]";
		if(BOOLSETTING(SHOW_DESCRIPTION_SPEED))
			speedDescription += "["+SETTING(DOWN_SPEED)+"/"+SETTING(UP_SPEED)+"]";
		if(BOOLSETTING(SHOW_DESCRIPTION_LIMIT) && SETTING(THROTTLE_ENABLE) && SETTING(MAX_UPLOAD_SPEED_LIMIT) != 0)
			speedDescription += "[L:" + Util::toString(SETTING(MAX_UPLOAD_SPEED_LIMIT)) + "KB]";
	}

	if(hub) {
		if(updateNick) {
			setCurrentNick(checkNick(hub->getNick(true)));
		}		

		if(!hub->getUserDescription().empty()) {
			setCurrentDescription(speedDescription + hub->getUserDescription());
		} else {
			setCurrentDescription(speedDescription + SETTING(DESCRIPTION));
		}
		if(!hub->getEmail().empty()) {
			setCurrentEmail(hub->getEmail());
		} else {
			setCurrentEmail(SETTING(EMAIL));
		}
		if(!hub->getPassword().empty())
			setPassword(hub->getPassword());
		setHideShare(hub->getHideShare());
		setStealth(hub->getStealth());
		setExclChecks(hub->getExclChecks());
		setAutoOpenOpChat(hub->getAutoOpenOpChat());
		setLogChat(hub->getLogChat() || BOOLSETTING(LOG_MAIN_CHAT));
		setFavIp(hub->getIP());
		setOpChat(hub->getHubChats().find(";") ? hub->getHubChats().substr(0, hub->getHubChats().find(";")) : hub->getHubChats());
		setHubChats(Text::toT(hub->getHubChats()));
		setProtectedPrefixes(hub->getProtectedPrefixes().empty() ? SETTING(PROT_USERS) : (SETTING(PROT_USERS).empty() ? hub->getProtectedPrefixes() : SETTING(PROT_USERS) + ";" + hub->getProtectedPrefixes()));
	} else {
		if(updateNick) {
			setCurrentNick(checkNick(SETTING(NICK)));
		}
		setCurrentDescription(speedDescription + SETTING(DESCRIPTION));
		setCurrentEmail(SETTING(EMAIL));
		setHideShare(false);
		setStealth(true);
		setExclChecks(false);
		setAutoOpenOpChat(false);
		setLogChat(BOOLSETTING(LOG_MAIN_CHAT));
		setFavIp(Util::emptyString);
		setOpChat(Util::emptyString);
		setHubChats(Util::emptyStringT);
		setProtectedPrefixes(SETTING(PROT_USERS));
	}

	if(getCurrentDescription().length() > 35 && getStealth())
		setCurrentDescription(getCurrentDescription().substr(0, 35));
}

bool Client::isActive() const {
	return ClientManager::getInstance()->getMode(hubUrl) != SettingsManager::INCOMING_FIREWALL_PASSIVE;
}

void Client::connect() {
	if(socket)
		BufferedSocket::putSocket(socket);

	availableBytes = 0;

	setAutoReconnect(true);
	setReconnDelay(120 + Util::rand(0, 60));
	reloadSettings(true);
	setRegistered(false);
	setMyIdentity(Identity(ClientManager::getInstance()->getMe(), 0));
	setHubIdentity(Identity());

	try {
		if(SETTING(PG_ENABLE)) {
			// suerly people don't want to connect to hubs which is in their blocklist, well
			// and if they do that's just too bad (this is done this way to prevent abuse)
			string company = PGManager::getInstance()->getIPBlock(Socket::resolve(address));
			if(!company.empty()) {
				if(SETTING(PG_LOG)) {
					PGManager::getInstance()->log("Blocked outgoing hub connection to: " + address + " (" + company + ")");
				}
				throw Exception(STRING(PG_BLOCKED) + ": " + company);
			}
		}

		socket = BufferedSocket::getSocket(separator);
		socket->addListener(this);
		socket->connect(address, port, secure, BOOLSETTING(ALLOW_UNTRUSTED_HUBS), true);
	} catch(const Exception& e) {
		if(socket) {
			BufferedSocket::putSocket(socket);
			socket = 0;
		}
		fire(ClientListener::Failed(), this, e.getError());
	}
	updateActivity();
	state = STATE_CONNECTING;
}

void Client::on(Connected) throw() {
	updateActivity(); 
	ip = socket->getIp(); 
	fire(ClientListener::Connected(), this);
	state = STATE_PROTOCOL;
}

void Client::on(Failed, const string& aLine) throw() {
	updateActivity(); 
	state = STATE_DISCONNECTED;
	FavoriteManager::getInstance()->removeUserCommand(getHubUrl());
	socket->removeListener(this);
	fire(ClientListener::Failed(), this, aLine);
}

void Client::disconnect(bool graceLess) {
	if(socket) 
		socket->disconnect(graceLess);
}

void Client::updateCounts(bool aRemove) {
	// We always remove the count and then add the correct one if requested...
	if(countType == COUNT_NORMAL) {
		Thread::safeDec(counts.normal);
	} else if(countType == COUNT_REGISTERED) {
		Thread::safeDec(counts.registered);
	} else if(countType == COUNT_OP) {
		Thread::safeDec(counts.op);
	}

	countType = COUNT_UNCOUNTED;

	if(!aRemove) {
		if(getMyIdentity().isOp()) {
			Thread::safeInc(counts.op);
			countType = COUNT_OP;
		} else if(getMyIdentity().isRegistered()) {
			Thread::safeInc(counts.registered);
			countType = COUNT_REGISTERED;
		} else {
			Thread::safeInc(counts.normal);
			countType = COUNT_NORMAL;
		}
	}
}

string Client::getLocalIp() const {
	// Favorite hub Ip
	if(!getFavIp().empty())
		return getFavIp();

	// Best case - the server detected it
	if((!BOOLSETTING(NO_IP_OVERRIDE) || SETTING(EXTERNAL_IP).empty()) && !getMyIdentity().getIp().empty()) {
		return getMyIdentity().getIp();
	}

	if(!SETTING(EXTERNAL_IP).empty()) {
		return Socket::resolve(SETTING(EXTERNAL_IP));
	}

	string lip;
	if(socket)
		lip = socket->getLocalIp();

	if(lip.empty())
		return Util::getLocalIp();
	return lip;
}

void Client::on(Line, const string& aLine) throw() {
	updateActivity();
	COMMAND_DEBUG(aLine, DebugManager::HUB_IN, getIpPort());
}

void Client::on(Second, uint64_t aTick) throw() {
	if(state == STATE_DISCONNECTED && getAutoReconnect() && (aTick > (getLastActivity() + getReconnDelay() * 1000)) ) {
		// Try to reconnect...
		connect();
	}
}

/**
 * @file
 * $Id: Client.cpp 317 2007-08-04 14:52:24Z bigmuscle $
 */
