' table/KeyboardOperation.vb
'
' Copyright (c) 2008, SystemBase Co.,Ltd.
' All rights reserved.
'
' Redistribution and use in source and binary forms, with or without 
' modification, are permitted provided that the following conditions are met:
'
'    1. Redistributions of source code must retain the above copyright 
'       notice, this list of conditions and the following disclaimer.
'    2. Redistributions in binary form must reproduce the above copyright 
'       notice, this list of conditions and the following disclaimer in the
'       documentation and/or other materials provided with the distribution.
'
' THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
' IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
' ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
' LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
' CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
' SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
' INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
' CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
' ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
' POSSIBILITY OF SUCH DAMAGE.

Public Interface IKeyboardOperation
    Function ProcessDialogKey(ByVal key As Keys, ByVal table As UTable) As Boolean
    Sub LeaveEdit(ByVal direction As String, ByVal table As UTable)
End Interface

Public Class CKeyboardOperation
    Implements IKeyboardOperation

    Public Enum EBehavior
        NONE
        NEXT_FIELD
        NEXT_EDITABLE_FIELD
        NEXT_RECORD
        NEXT_CONTROL
    End Enum

    Public EnterBehavior As EBehavior = EBehavior.NEXT_RECORD
    Public TabBehavior As EBehavior = EBehavior.NEXT_FIELD
    Public StartEditKey As Keys = Keys.F2
    Public LeaveFromLastField As Boolean = True
    Public ClearByDelete As Boolean = True

    Public Overridable Function ProcessDialogKey(ByVal key As Keys, ByVal table As UTable) As Boolean Implements IKeyboardOperation.ProcessDialogKey
        If key = Me.StartEditKey Then
            If table.StartEditing() IsNot Nothing Then
                Return True
            End If
        End If
        Select Case key
            Case Keys.Up, Keys.Down, Keys.Left, Keys.Right
                If table.FocusField Is Nothing Then
                    table.FocusField = table.Content.DefaultField
                Else
                    Select Case key
                        Case Keys.Up
                            Me.prevRow(table)
                        Case Keys.Down
                            Me.nextRow(table)
                        Case Keys.Left
                            Me.prevCol(table)
                        Case Keys.Right
                            Me.nextCol(table)
                    End Select
                End If
                Return True
            Case Keys.Home, Keys.End
                If table.FocusField IsNot Nothing Then
                    Select Case key
                        Case Keys.Home
                            Me.startCol(table)
                        Case Keys.End
                            Me.endCol(table)
                    End Select
                End If
                Return True
            Case Keys.PageDown
                Me.pageDown(table, table.VScrollBar)
                Return True
            Case Keys.PageUp
                Me.pageUp(table, table.VScrollBar)
                Return True
            Case Keys.Enter
                Me.tabEnter(table, Me.EnterBehavior)
                Return True
            Case Keys.Enter Or Keys.Shift
                Me.tabEnterPrev(table, Me.EnterBehavior)
                Return True
            Case Keys.Tab
                Me.tabEnter(table, Me.TabBehavior)
                Return True
            Case Keys.Tab Or Keys.Shift
                Me.tabEnterPrev(table, Me.TabBehavior)
                Return True
            Case Keys.Delete
                If Me.ClearByDelete AndAlso table.FocusField IsNot Nothing Then
                    If table.FocusField.Editable = UTable.EEditable.ALLOW Then
                        table.FocusField.Clear()
                    End If
                    Return True
                End If
        End Select
        Return False
    End Function

    Public Overridable Sub LeaveEdit(ByVal direction As String, ByVal table As UTable) Implements IKeyboardOperation.LeaveEdit
        Select Case direction
            Case "UP"
                Me.prevRow(table)
            Case "DOWN"
                Me.nextRow(table)
            Case "LEFT"
                Me.prevCol(table)
            Case "RIGHT"
                Me.nextCol(table)
            Case "ENTER"
                Me.tabEnter(table, Me.EnterBehavior)
            Case "ENTER_PREV"
                Me.tabEnterPrev(table, Me.EnterBehavior)
            Case "TAB"
                Me.tabEnter(table, Me.TabBehavior)
            Case "TAB_PREV"
                Me.tabEnterPrev(table, Me.TabBehavior)
        End Select
    End Sub

    Protected Overridable Sub nextRow(ByVal table As UTable)
        table.FocusGrid.Row = table.FocusField.Desc.Layout.BottomRow
        Dim field As UTable.CField = Me.nextRowField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub prevRow(ByVal table As UTable)
        table.FocusGrid.Row = table.FocusField.Desc.Layout.Row
        Dim field As UTable.CField = Me.prevRowField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub nextCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.RightCol
        Dim field As UTable.CField = Me.nextColField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub prevCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = Me.prevColField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub startCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = Me.nextColField(New UTable.CGrid.CPoint(table.FocusGrid.Row, -1), table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub endCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = Me.prevColField(New UTable.CGrid.CPoint(table.FocusGrid.Row, table.Cols.Count), table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub pageDown(ByVal table As UTable, ByVal scrollBar As ScrollBar)
        table.SetScrollValue(scrollBar, scrollBar.Value + scrollBar.LargeChange)
    End Sub

    Protected Overridable Sub pageUp(ByVal table As UTable, ByVal scrollBar As ScrollBar)
        table.SetScrollValue(scrollBar, scrollBar.Value - scrollBar.LargeChange)
    End Sub

    Protected Sub tabEnter(ByVal table As UTable, ByVal behavior As EBehavior)
        If table.FocusField Is Nothing Then
            table.FocusField = table.Content.DefaultField
        End If
        If table.FocusField Is Nothing Then
            Me.doNextControl(table)
        Else
            Select Case behavior
                Case EBehavior.NEXT_FIELD
                    Me.doNextField(table, False)
                Case EBehavior.NEXT_EDITABLE_FIELD
                    Me.doNextField(table, True)
                Case EBehavior.NEXT_RECORD
                    Me.nextRecord(table)
                Case EBehavior.NEXT_CONTROL
                    Me.doNextControl(table)
            End Select
        End If
    End Sub

    Protected Sub tabEnterPrev(ByVal table As UTable, ByVal behavior As EBehavior)
        If table.FocusField Is Nothing Then
            table.FocusField = table.Content.DefaultField
        End If
        If table.FocusField Is Nothing Then
            Me.doPrevControl(table)
        Else
            Select Case behavior
                Case EBehavior.NEXT_FIELD
                    Me.doPrevField(table, False)
                Case EBehavior.NEXT_EDITABLE_FIELD
                    Me.doPrevField(table, True)
                Case EBehavior.NEXT_RECORD
                    Me.prevRecord(table)
                Case EBehavior.NEXT_CONTROL
                    Me.doPrevControl(table)
            End Select
        End If
    End Sub

    Protected Sub doNextField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim handled As Boolean = False
        table.raiseFieldFocusNext(table.FocusField, True, handled)
        If Not handled Then
            Me.[nextField](table, editableOnly)
        End If
    End Sub

    Protected Sub doPrevField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim handled As Boolean = False
        table.raiseFieldFocusNext(table.FocusField, False, handled)
        If Not handled Then
            Me.prevField(table, editableOnly)
        End If
    End Sub

    Protected Overridable Sub nextField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim f As UTable.CField
        If editableOnly Then
            f = table.FocusField.NextEditableField
        Else
            f = table.FocusField.NextField
        End If
        If f IsNot Nothing Then
            table.FocusField = f
        ElseIf LeaveFromLastField Then
            Me.doNextControl(table)
        End If
    End Sub

    Protected Overridable Sub prevField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim f As UTable.CField
        If editableOnly Then
            f = table.FocusField.PrevEditableField
        Else
            f = table.FocusField.PrevField
        End If
        If f IsNot Nothing Then
            table.FocusField = f
        ElseIf LeaveFromLastField Then
            Me.doPrevControl(table)
        End If
    End Sub

    Protected Overridable Sub nextRecord(ByVal table As UTable)
        Dim r As UTable.CRecord = table.FocusRecord.LayoutCache.NextRecord
        If r IsNot Nothing Then
            If r.Fields.ContainsKey(table.FocusField.Key) Then
                table.FocusField = r.Fields(table.FocusField.Key)
            Else
                table.FocusField = r.DefaultField
            End If
        End If
    End Sub

    Protected Overridable Sub prevRecord(ByVal table As UTable)
        Dim r As UTable.CRecord = table.FocusRecord.LayoutCache.PrevRecord
        If r IsNot Nothing Then
            If r.Fields.ContainsKey(table.FocusField.Key) Then
                table.FocusField = r.Fields(table.FocusField.Key)
            Else
                table.FocusField = r.DefaultField
            End If
        End If
    End Sub

    Protected Sub doNextControl(ByVal table As UTable)
        Dim handled As Boolean = False
        table.raiseFocusNext(table.FocusField, True, handled)
        If Not handled Then
            Me.nextControl(table)
        End If
    End Sub

    Protected Sub doPrevControl(ByVal table As UTable)
        Dim handled As Boolean = False
        table.raiseFocusNext(table.FocusField, False, handled)
        If Not handled Then
            Me.prevControl(table)
        End If
    End Sub

    Protected Overridable Sub nextControl(ByVal table As UTable)
        If table.TopLevelControl IsNot Nothing Then
            table.TopLevelControl.SelectNextControl(table, True, True, True, True)
        End If
    End Sub

    Protected Overridable Sub prevControl(ByVal table As UTable)
        If table.TopLevelControl IsNot Nothing Then
            table.TopLevelControl.SelectNextControl(table, False, True, True, True)
        End If
    End Sub

    Public Function nextColField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row, point.Col + 1)
        Do While p.Col < record.Table.Cols.Count
            Dim field As UTable.CField = record.FindFocusableField(p)
            If field IsNot Nothing Then
                Return field
            End If
            p.Col += 1
        Loop
        Return Nothing
    End Function

    Public Function prevColField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row, point.Col - 1)
        Do While p.Col >= 0
            Dim field As UTable.CField = record.FindFocusableField(p)
            If field IsNot Nothing Then
                Return field
            End If
            p.Col -= 1
        Loop
        Return Nothing
    End Function

    Public Function nextRowField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row + 1, point.Col)
        Do While p.Row < record.Rows.Count
            Dim field As UTable.CField = record.FindFocusableField(p)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.nextColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.prevColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            p.Row += 1
        Loop
        If record.LayoutCache.NextRecord IsNot Nothing Then
            Dim r As UTable.CRecord = record.LayoutCache.NextRecord
            Return Me.nextRowField(New UTable.CGrid.CPoint(-1, point.Col), r)
        Else
            Return Nothing
        End If
    End Function

    Public Function prevRowField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row - 1, point.Col)
        Do While p.Row >= 0
            Dim field As UTable.CField = record.FindFocusableField(p)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.nextColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.prevColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            p.Row -= 1
        Loop
        If record.LayoutCache.PrevRecord IsNot Nothing Then
            Dim r As UTable.CRecord = record.LayoutCache.PrevRecord
            Return Me.prevRowField(New UTable.CGrid.CPoint(r.Rows.Count, point.Col), r)
        Else
            Return Nothing
        End If
    End Function

End Class
