package net.osdn.util.ssdp;

import java.io.IOException;
import java.io.Reader;
import java.net.InetAddress;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

/** デバイス情報を保持しています。
 * 
 */
public class Device {
	
	private static XmlPullParserFactory factory;
	
	public static Device parse(Reader reader) throws XmlPullParserException, IOException {
		if(factory == null) {
			factory = XmlPullParserFactory.newInstance();
			factory.setNamespaceAware(true);
		}
		
		Device device = new Device();
		
		XmlPullParser parser = factory.newPullParser();
		parser.setInput(reader);
		int eventType = parser.getEventType();
		boolean isInDeviceTag = false;
		while(eventType != XmlPullParser.END_DOCUMENT) {
			String name = parser.getName();
			switch(eventType) {
			case XmlPullParser.START_TAG:
				if("device".equalsIgnoreCase(name)) {
					isInDeviceTag = true;
				} else if(isInDeviceTag) {
					if("UDN".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setUdn(parser.getText());
						}
						continue;
					} else if("friendlyName".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setFriendlyName(parser.getText());
						}
						continue;
					} else if("deviceType".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setDeviceType(parser.getText());
						};
						continue;
					} else if("manufacturer".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setManufacturer(parser.getText());
						}
						continue;
					} else if("modelName".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setModelName(parser.getText());
						}
						continue;
					} else if("modelNumber".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setModelNumber(parser.getText());
						}
						continue;
					} else if("modelDescription".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setModelDescription(parser.getText());
						}
						continue;
					} else if("serialNumber".equalsIgnoreCase(name)) {
						if(parser.next() == XmlPullParser.TEXT) {
							device.setSerialNumber(parser.getText());
						}
						continue;
					}
				}
				break;
			case XmlPullParser.END_TAG:
				if("device".equalsIgnoreCase(name)) {
					isInDeviceTag = false;
				}
				break;
			}
			eventType = parser.next();
		}
		
		return device;
	}
	
	private static String escape(String value) {
		if(value != null) {
			return ("!" + value.replace("\r\n", "\n").replace('\r', ' ').replace('\n', ' ')).trim().substring(1);
		}
		return null;
	}

	private String udn;
	private String friendlyName;
	private String deviceType;
	private String manufacturer;
	private String modelName;
	private String modelNumber;
	private String modelDescription;
	private String serialNumber;
	private InetAddress addr;
	private boolean isSelf;
	
	/** UDN(Unique Device Name)を設定します。
	 * 
	 * @param udn Unique Device Name
	 */
	public void setUdn(String udn) {
		this.udn = udn;
	}
	
	/** UDN(Unique Device Name)を取得します。
	 * 
	 * @return UDN
	 */
	public String getUdn() {
		if(this.udn == null) {
			return "";
		}
		return this.udn;
	}
	
	/** フレンドリー名を設定します。
	 * 
	 * @param friendlyName フレンドリー名
	 */
	public void setFriendlyName(String friendlyName) {
		this.friendlyName = escape(friendlyName);
	}

	/** フレンドリー名を取得します。
	 * 
	 * @return フレンドリー名
	 */
	public String getFriendlyName() {
		if(this.friendlyName == null) {
			return "";
		}
		return this.friendlyName;
	}
	
	/** デバイス・タイプを設定します。
	 * 
	 * @param deviceType デバイス・タイプ
	 */
	public void setDeviceType(String deviceType) {
		this.deviceType = escape(deviceType);
	}
	
	/** デバイス・タイプを取得します。
	 * 
	 * @return デバイス・タイプ
	 */
	public String getDeviceType() {
		if(this.deviceType == null) {
			return "";
		}
		return this.deviceType;
	}
	
	/** 製造元を設定します。
	 * 
	 * @param manufacturer 製造元
	 */
	public void setManufacturer(String manufacturer) {
		this.manufacturer = escape(manufacturer);
	}
	
	/** 製造元を取得します。
	 * 
	 * @return 製造元
	 */
	public String getManufacturer() {
		if(this.manufacturer == null) {
			return "";
		}
		return this.manufacturer;
	}
	
	/** モデル名を設定します。
	 * 
	 * @param modelName モデル名
	 */
	public void setModelName(String modelName) {
		this.modelName = escape(modelName);
	}
	
	/** モデル名を取得します。
	 * 
	 * @return モデル名
	 */
	public String getModelName() {
		if(this.modelName == null) {
			return "";
		}
		return this.modelName;
	}
	
	/** モデル番号を設定します。
	 * 
	 * @param modelNumber モデル番号
	 */
	public void setModelNumber(String modelNumber) {
		this.modelNumber = escape(modelNumber);
	}
	
	/** モデル番号を取得します。
	 * 
	 * @return モデル番号
	 */
	public String getModelNumber() {
		if(this.modelNumber == null) {
			return "";
		}
		return this.modelNumber;
	}
	
	/** モデルの説明を設定します。
	 * 
	 * @param modelDescription モデルの説明
	 */
	public void setModelDescription(String modelDescription) {
		this.modelDescription = escape(modelDescription);
	}
	
	/** モデルの説明を取得します。
	 * 
	 * @return モデルの説明
	 */
	public String getModelDescription() {
		if(this.modelDescription == null) {
			return "";
		}
		return this.modelDescription;
	}
	
	/** シリアル番号を設定します。
	 * 
	 * @param serialNumber シリアル番号
	 */
	public void setSerialNumber(String serialNumber) {
		this.serialNumber = escape(serialNumber);
	}
	
	/** シリアル番号を取得します。
	 * 
	 * @return シリアル番号
	 */
	public String getSerialNumber() {
		if(this.serialNumber == null) {
			return "";
		}
		return this.serialNumber;
	}
	
	/** IPアドレスを設定します。
	 * 
	 * @param addr IPアドレス
	 */
	public void setAddress(InetAddress addr) {
		this.addr = addr;
	}
	
	/** IPアドレスを取得します。
	 * 
	 * @return IPアドレス
	 */
	public InetAddress getAddress() {
		return this.addr;
	}

	/** この端末から返されたデバイス情報であることを示すフラグを設定します。
	 * 
	 * @param isSelf
	 */
	public void setSelf(boolean isSelf) {
		this.isSelf = isSelf;
	}
	
	/** この端末から返されたデバイス情報であることを示すフラグを取得します。
	 * 
	 * @return ローカルホストから返されたデバイス情報の場合は true、そうでなければ false が返されます。
	 */
	public boolean isSelf() {
		return this.isSelf;
	}

	/** オブジェクトのハッシュコード値を返します。
	 * 
	 * @return このオブジェクトのハッシュコード値。
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((addr == null) ? 0 : addr.hashCode());
		result = prime * result + ((udn == null) ? 0 : udn.hashCode());
		return result;
	}

	/** このオブジェクトと他のオブジェクトが等しいかどうかを示します。
	 * デバイス情報に含まれるUDN、IPアドレスが等しい場合にオブジェクトが等しいと見なされます。
	 * 
	 * @param obj 比較対象の参照オブジェクト。
	 * @return このオブジェクトが obj 引数と同じである場合は true、それ以外の場合は false。
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Device other = (Device) obj;
		if (addr == null) {
			if (other.addr != null)
				return false;
		} else if (!addr.equals(other.addr))
			return false;
		if (udn == null) {
			if (other.udn != null)
				return false;
		} else if (!udn.equals(other.udn))
			return false;
		return true;
	}
}
