package net.osdn.util.ssdp.server;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.DatagramPacket;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.InterfaceAddress;
import java.net.MulticastSocket;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import net.osdn.util.ssdp.Device;

public class SsdpServer extends NanoHTTPD {
	public static boolean DEBUG = false;
	public static final InetSocketAddress MULTICAST_ADDR_IPv4;
	public static final InetSocketAddress MULTICAST_ADDR_IPv6;
	public static final int SSDP_PORT = 1900;
	public static final int HTTP_PORT = 1980;
	
	private static final Charset UTF_8 = Charset.forName("UTF-8");
	
	static {
		InetSocketAddress mcastAddrIPv4 = null;
		try {
			mcastAddrIPv4 = new InetSocketAddress(InetAddress.getByName("239.255.255.250"), SSDP_PORT);
		} catch(Exception e) {
			if(SsdpServer.DEBUG) {
				e.printStackTrace();
			}
		}
		MULTICAST_ADDR_IPv4 = mcastAddrIPv4;

		InetSocketAddress mcastAddrIPv6 = null;
		try {
			mcastAddrIPv6 = new InetSocketAddress(InetAddress.getByName("FF02::C"), SSDP_PORT);
		} catch(Exception e) {
			if(SsdpServer.DEBUG) {
				e.printStackTrace();
			}
		}
		MULTICAST_ADDR_IPv6 = mcastAddrIPv6;
	}

	private Device device;
	private String server;
	private String locationIPv4;
	private String locationIPv6;
	private volatile boolean isRunning;
	private Thread ssdpThread;
	private MulticastSocket ssdpSocket;

	public SsdpServer(NetworkInterface netIf, Device device) {
		this(getInet4Address(netIf), getInet6Address(netIf), device, null);
	}

	public SsdpServer(NetworkInterface netIf, Device device, String server) {
		this(getInet4Address(netIf), getInet6Address(netIf), device, server);
	}

	public SsdpServer(Inet4Address ipv4addr, Inet6Address ipv6addr, Device device) {
		this(ipv4addr, ipv6addr, device, null);
	}
	
	public SsdpServer(Inet4Address ipv4addr, Inet6Address ipv6addr, Device device, String server) {
		super(HTTP_PORT);

		if(ipv4addr != null) {
			try {
				ipv4addr = (Inet4Address)InetAddress.getByAddress(ipv4addr.getAddress());
				locationIPv4 = "http://" + ipv4addr.getHostAddress() + ":" + HTTP_PORT + "/";
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		if(ipv6addr != null) {
			try {
				ipv6addr = (Inet6Address)InetAddress.getByAddress(ipv6addr.getAddress());
				locationIPv6 = "http://[" + ipv6addr.getHostAddress() + "]:" + HTTP_PORT + "/";
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		
		this.device = device;
		
		if(server != null) {
			this.server = server;
		} else {
			this.server = "UPnP/1.0 UPnP-Device-Host/1.0";
		}
	}

	@Override
	public void start(int timeout, boolean daemon) throws IOException {
		//Start NanoHTTPD
		super.start(timeout, daemon);
		
		//Start SSDP
		ssdpThread = new Thread(new Runnable() {
			@Override
			public void run() {
				isRunning = true;
				while(isRunning) {
					try {
						ssdpRun();
					} catch (Exception e) {
						if(DEBUG) {
							e.printStackTrace();
						}
					}
				}
			}
		});
		ssdpThread.setDaemon(true);
		ssdpThread.setName("SSDP Listener");
		ssdpThread.start();
	}

	@Override
	public void stop() {
		//Stop SSDP
		if(isRunning) {
			isRunning = false;
			if(ssdpSocket != null) {
				try { ssdpSocket.close(); } catch(Exception e) {}
			}
			if(ssdpThread != null) {
				ssdpThread.interrupt();
				try { ssdpThread.join(); } catch(InterruptedException e) {}
			}
			ssdpThread = null;
			ssdpSocket = null;
		}
		
		// Stop NanoHTTPD
		super.stop();
	}

	@Override
	public Response serve(IHTTPSession session) {
		String xml =
				"<?xml version=\"1.0\"?>\r\n" + 
				"<root xmlns=\"urn:schemas-upnp-org:device-1-0\">\r\n" +
				"	<specVersion>\r\n" +
				"		<major>1</major>\r\n" +
				"		<minor>0</minor>\r\n" +
				"	</specVersion>\r\n" +
				"	<device>\r\n" +
				"		<UDN>${UDN}</UDN>\r\n" + 
				"		<friendlyName>${friendlyName}</friendlyName>\r\n" +
				"		<deviceType>${deviceType}</deviceType>\r\n" +
				"		<manufacturer>${manufacturer}</manufacturer>\r\n" +
				"		<modelName>${modelName}</modelName>\r\n" +
				"		<modelNumber>${modelNumber}</modelNumber>\r\n" + 
				"		<modelDescription>${modelDescription}</modelDescription>\r\n" + 
				"		<serialNumber>${serialNumber}</serialNumber>\r\n" + 
				"		<iconList>\r\n" +
				"		</iconList>\r\n" +
				"		<serviceList>\r\n" + 
				"		</serviceList>\r\n" +
				"	</device>\r\n" +
				"</root>\r\n";
			
		String udn = device.getUdn();
		String friendlyName = device.getFriendlyName();
		String deviceType = device.getDeviceType();
		String manufacturer = device.getManufacturer();
		String modelName = device.getModelName();
		String modelNumber = device.getModelNumber();
		String modelDescription = device.getModelDescription();
		String serialNumber = device.getSerialNumber();
			
		if(udn == null) {
			udn = "Unknown";
		}
		if(friendlyName == null || friendlyName.length() == 0) {
			friendlyName = "Unknown";
		}
		if(deviceType == null || deviceType.length() == 0) {
			deviceType = "Unknown";
		}
		if(manufacturer == null || manufacturer.length() == 0) {
			manufacturer = "Unknown";
		}
		if(modelName == null || modelName.length() == 0) {
			modelName = "Unknown";
		}
		if(modelNumber == null || modelNumber.length() == 0) {
			modelNumber = "Unknown";
		}
		if(modelDescription == null || modelDescription.length() == 0) {
			modelDescription = "Unknown";
		}
		if(serialNumber == null || serialNumber.length() == 0) {
			serialNumber = "Unknown";
		}
			
		xml = xml
			.replace("${UDN}", udn)
			.replace("${friendlyName}", friendlyName)
			.replace("${deviceType}", deviceType)
			.replace("${manufacturer}", manufacturer)
			.replace("${modelName}", modelName)
			.replace("${modelNumber}", modelNumber)
			.replace("${modelDescription}", modelDescription)
			.replace("${serialNumber}", serialNumber);
			
		return newFixedLengthResponse(Response.Status.OK, "text/xml", xml);
	}
	
	protected void ssdpRun() throws IOException {
		String response = "HTTP/1.1 200 OK\r\n"
						+ "ST:upnp:rootdevice\r\n"
						+ "USN:${USN}\r\n"
						+ "Location:${Location}\r\n"
						+ "OPT:\"http://schemas.upnp.org/upnp/1/0/\"; ns=01\r\n"
						+ "Server:${Server}\r\n"
						+ "\r\n";
		
		ssdpSocket = new MulticastSocket(SSDP_PORT);

		Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
		while(interfaces.hasMoreElements()) {
			boolean isIPv4joined = false;
			boolean isIPv6joined = false;
			NetworkInterface netIf = interfaces.nextElement();
			if(netIf.isLoopback() || netIf.isVirtual() || !netIf.isUp() || !netIf.supportsMulticast()) {
				continue;
			}
			for(InterfaceAddress ifAddr : netIf.getInterfaceAddresses()) {
				InetAddress addr = ifAddr != null ? ifAddr.getAddress() : null;
				if(addr != null) {
					addr = InetAddress.getByAddress(addr.getAddress());
				}
				if(MULTICAST_ADDR_IPv4 != null && addr instanceof Inet4Address && !isIPv4joined) {
					System.out.println("IPv4: netIf=" + netIf.getName() + ", addr=" + addr);
					ssdpSocket.joinGroup(MULTICAST_ADDR_IPv4, netIf);
					isIPv4joined = true;
				}
				if(MULTICAST_ADDR_IPv6 != null && addr instanceof Inet6Address && !isIPv6joined) {
					System.out.println("IPv6: netIf=" + netIf.getName() + ", addr=" + addr);
					ssdpSocket.joinGroup(MULTICAST_ADDR_IPv6, netIf);
					isIPv6joined = true;
				}
			}
		}
		
		byte[] buf = new byte[4096];
		DatagramPacket request = new DatagramPacket(buf, buf.length);
		for(;;) {
			ssdpSocket.receive(request);
			Map<String, String> searchHeaders = parseSearchHeaders(request);
			if(searchHeaders == null) {
				continue;
			}
			String man = searchHeaders.get("MAN");
			if(!man.equalsIgnoreCase("\"ssdp:discover\"")) {
				continue;
			}
			String st = searchHeaders.get("ST");
			if(st == null || (!st.equalsIgnoreCase("upnp:rootdevice") && !st.equalsIgnoreCase("ssdp:all"))) {
				continue;
			}
			String host = searchHeaders.get("HOST");
			if(host == null) {
				continue;
			}
			String hostAddr;
			int i = host.lastIndexOf(':');
			if(i < 0) {
				hostAddr = host;
			} else {
				hostAddr = host.substring(0, i);
			}
			if(hostAddr.startsWith("[") && hostAddr.endsWith("]")) {
				hostAddr = hostAddr.substring(1, hostAddr.length() - 1);
			}
			
			String usn = device.getUdn() + "::upnp:rootdevice";
			String location = null;
			InetAddress remoteAddr = InetAddress.getByAddress(request.getAddress().getAddress());
			if(remoteAddr instanceof Inet4Address) {
				location = locationIPv4;
			}
			if(remoteAddr instanceof Inet6Address) {
				location = locationIPv6;
			}
			if(location != null) {
				byte[] data = response
						.replace("${USN}", usn)
						.replace("${Location}", location)
						.replace("${Server}", server)
						.getBytes();
				DatagramPacket packet = new DatagramPacket(data, data.length, remoteAddr, request.getPort());
				try {
					ssdpSocket.send(packet);
					if(DEBUG) {
						String ipv = (remoteAddr instanceof Inet4Address) ? "IPv4" : "IPv6";
						System.out.println("# SSDP: send: " + ipv + ": OK, addr=" + remoteAddr + ", port=" + request.getPort() + ", data.length=" + data.length);
					}
				} catch(SocketException e) {
					String ipv = (remoteAddr instanceof Inet4Address) ? "IPv4" : "IPv6";
					System.err.println("# SSDP: send: " + ipv + ": ERROR, addr=" + remoteAddr + ", port=" + request.getPort() + ", data.length=" + data.length);
					e.printStackTrace();
				}
			}
		}
	}
	
	protected Map<String, String> parseSearchHeaders(DatagramPacket packet) throws IOException {
		BufferedReader r = null;
		try {
			r = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(packet.getData(), packet.getOffset(), packet.getLength()), UTF_8));
			String header = r.readLine();
			if(header.toUpperCase().startsWith("M-SEARCH")) {
				Map<String, String> headers = new HashMap<String, String>();
				String line;
				while((line = r.readLine()) != null) {
					int i = line.indexOf(':');
					if(i > 0) {
						String key = line.substring(0,  i);
						String value = line.substring(i + 1).trim();
						headers.put(key.toUpperCase(), value);
					}
				}
				return headers;
			} else {
				return null;
			}
		} finally {
			if(r != null) {
				r.close();
			}
		}
	}
	
	/** 指定したネットワークインターフェースに設定されているIPv4アドレスを返します。
	 * 
	 * @param netIf ネットワークインターフェース
	 * @return はじめに見つかったIPv4アドレス。見つからない場合はnullが返されます。
	 */
	private static Inet4Address getInet4Address(NetworkInterface netIf) {
		for(InterfaceAddress ifAddr : netIf.getInterfaceAddresses()) {
			InetAddress addr = ifAddr != null ? ifAddr.getAddress() : null;
			if(addr instanceof Inet4Address) {
				return (Inet4Address)addr;
			}
		}
		return null;
	}
	
	/** 指定したネットワークインターフェースに設定されているIPv6アドレスを返します。
	 * 
	 * @param netIf ネットワークインターフェース
	 * @return はじめに見つかったIPv6アドレス。見つからない場合はnullが返されます。
	 */
	private static Inet6Address getInet6Address(NetworkInterface netIf) {
		for(InterfaceAddress ifAddr : netIf.getInterfaceAddresses()) {
			InetAddress addr = ifAddr != null ? ifAddr.getAddress() : null;
			if(addr instanceof Inet6Address) {
				return (Inet6Address)addr;
			}
		}
		return null;
	}
}
