package net.osdn.util.jersey;

import java.io.IOException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.ext.ParamConverter;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.CollectionType;
import com.fasterxml.jackson.databind.type.TypeFactory;

public class JacksonParamConverter<T> implements ParamConverter<T> {

	private ObjectMapper mapper;
	private Class<T> rawType;
	private Type genericType;
	private Class<?> jsonBaseType;
	
	public JacksonParamConverter(ObjectMapper mapper, Class<T> rawType, Type genericType, Class<?> jsonBaseType) {
		this.mapper = mapper;
		this.rawType = rawType;
		this.genericType = genericType;
		this.jsonBaseType = jsonBaseType;
	}
	
	@Override
	public T fromString(String json) {
		//System.out.println("# JacksonParamConverter: rawType=" + rawType + ", genericType=" + genericType + ", json=<" + json + ">");

		if(json == null || json.equals("null")) {
			return null;
		}
		
		try {
			if(jsonBaseType == String.class) {
				json = normalize(json);
			}
			
			if(genericType instanceof ParameterizedType) {
				ParameterizedType pType = (ParameterizedType)genericType;
				@SuppressWarnings("unchecked")
				Class<? extends Collection<?>> collectionClass = (Class<? extends Collection<?>>)rawType;
				Class<?> elementClass = (Class<?>)pType.getActualTypeArguments()[0];
				CollectionType collectionType = TypeFactory.defaultInstance().constructCollectionType(collectionClass, elementClass);
				return mapper.readValue(json, collectionType);
			} else {
				return mapper.readValue(json, rawType);
			}
		} catch (Exception e) {
			//e.printStackTrace();
			String message = "Invalid parameter as a " + rawType.getSimpleName() + ": " + json;
			Response response = Response
					.status(Status.BAD_REQUEST)
					.type(MediaType.TEXT_PLAIN_TYPE)
					.entity("400 Bad Request\r\n\r\n" + message)
					.build();
			throw new WebApplicationException(e, response);
		}
	}

	@Override
	public String toString(T object) {
		try {
			return mapper.writeValueAsString(object);
		} catch (Exception e) {
			e.printStackTrace(); //レスポンスにはエラー詳細を出力せず、スタックとレースは標準出力に出力します。(サーバーによってログに記録されたりします)
			Response response = Response
					.status(Status.INTERNAL_SERVER_ERROR)
					.type(MediaType.TEXT_PLAIN_TYPE)
					.entity("500 Internal Server Error")
					.build();
			throw new WebApplicationException(e, response);
		}
	}
	
	/** 指定された文字列をJSON文字列として標準化します。
	 * 指定された文字列がJSON文字列の場合はそのまま返します。
	 * 指定された文字列がJSON文字列ではない場合(ダブルクォーテーションでエスケープされていないケースなど)は指定文字列をJSON文字列に変換して返します。
	 * これにより返される文字列はダブルクォーテーションで囲まれ適切にエスケープ処理されたJSON文字列であることが保証されます。
	 * 
	 * @param string JSON文字列または通常の文字列
	 * @return JSON文字列
	 * @throws JsonProcessingException JSON変換でエラーが発生した場合
	 */
	private String normalize(String string) throws JsonProcessingException {
		boolean isValid = false;
		try {
			JsonParser parser = mapper.getFactory().createParser(string);
			JsonToken firstToken = parser.nextToken();
			if(firstToken == JsonToken.VALUE_STRING) {
				JsonToken secondToken = parser.nextToken();
				if(secondToken == null) {
					isValid = true;
				}
			}
		} catch (JsonParseException e) {
		} catch (IOException e) {
		}
		if(!isValid) {
			string = mapper.writeValueAsString(string);
		}
		return string;
	}
}
