package net.osdn.util.jersey.fastpack;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;

public class ByteUnpacker {
	
	private static byte BYTE_STRING_EMPTY = (byte)0x00;
	private static int MAX_STRING_LENGTH = 16383;
	private static int  INT_LONG_0 = 0xC0;
	private static byte BYTE_LONG_0 = (byte)0xC0;
	private static byte BYTE_LONG_1 = (byte)0xC1;
	private static byte BYTE_LONG_2 = (byte)0xC2;
	private static byte BYTE_LONG_3 = (byte)0xC3;
	private static byte BYTE_LONG_4 = (byte)0xC4;
	private static byte BYTE_LONG_5 = (byte)0xC5;
	private static byte BYTE_LONG_6 = (byte)0xC6;
	private static byte BYTE_LONG_7 = (byte)0xC7;
	private static byte BYTE_LONG_8 = (byte)0xC8;
	private static byte BYTE_LONG_9 = (byte)0xC9;
	private static byte BYTE_LONG_10 = (byte)0xCA;
	private static byte BYTE_LONG_11 = (byte)0xCB;
	private static byte BYTE_LONG_12 = (byte)0xCC;
	private static byte BYTE_LONG_13 = (byte)0xCD;
	private static byte BYTE_LONG_14 = (byte)0xCE;
	private static byte BYTE_LONG_15 = (byte)0xCF;
	private static byte BYTE_LONG_N1 = (byte)0xD0;
	private static byte BYTE_LONG_NEXT1 = (byte)0xD1;
	private static byte BYTE_LONG_NEXT2 = (byte)0xD2;
	private static byte BYTE_LONG_NEXT3 = (byte)0xD3;
	private static byte BYTE_LONG_NEXT4 = (byte)0xD4;
	private static byte BYTE_LONG_NEXT5 = (byte)0xD5;
	private static byte BYTE_LONG_NEXT6 = (byte)0xD6;
	private static byte BYTE_LONG_NEXT7 = (byte)0xD7;
	private static byte BYTE_LONG_NEXT8 = (byte)0xD8;
	private static byte BYTE_LONG_NEXT1_N = (byte)0xD9;
	private static byte BYTE_LONG_NEXT2_N = (byte)0xDA;
	private static byte BYTE_LONG_NEXT3_N = (byte)0xDB;
	private static byte BYTE_LONG_NEXT4_N = (byte)0xDC;
	private static byte BYTE_LONG_NEXT5_N = (byte)0xDD;
	private static byte BYTE_LONG_NEXT6_N = (byte)0xDE;
	private static byte BYTE_LONG_NEXT7_N = (byte)0xDF;
	private static int  INT_LONG_NEXT7_N = 0xDF;
	private static int  INT_DOUBLE_0 = 0xE0;
	private static byte BYTE_DOUBLE_0 = (byte)0xE0;
	private static byte BYTE_DOUBLE_1 = (byte)0xE1;
	private static byte BYTE_DOUBLE_2 = (byte)0xE2;
	private static byte BYTE_DOUBLE_3 = (byte)0xE3;
	private static byte BYTE_DOUBLE_4 = (byte)0xE4;
	private static byte BYTE_DOUBLE_5 = (byte)0xE5;
	private static byte BYTE_DOUBLE_6 = (byte)0xE6;
	private static byte BYTE_DOUBLE_7 = (byte)0xE7;
	private static byte BYTE_DOUBLE_8 = (byte)0xE8;
	private static byte BYTE_DOUBLE_9 = (byte)0xE9;
	private static byte BYTE_DOUBLE_10 = (byte)0xEA;
	private static byte BYTE_DOUBLE_11 = (byte)0xEB;
	private static byte BYTE_DOUBLE_12 = (byte)0xEC;
	private static byte BYTE_DOUBLE_13 = (byte)0xED;
	private static byte BYTE_DOUBLE_14 = (byte)0xEE;
	private static byte BYTE_DOUBLE_15 = (byte)0xEF;
	private static byte BYTE_DOUBLE_N1 = (byte)0xF0;
	private static int  INT_DOUBLE_N1 = 0xF0;
	private static byte BYTE_DOUBLE_NEXT8 = (byte)0xF8;
	private static byte BYTE_NULL = (byte)0xFF;
	private static Charset UTF8 = Charset.forName("UTF-8");
	
	private InputStream in;
	private byte[] buf = new byte[MAX_STRING_LENGTH];
	private boolean wasNull;
	
	public ByteUnpacker(InputStream in) {
		this.in = in;
	}
	
	public boolean wasNull() {
		return wasNull;
	}
	
	/** オブジェクトを読み取ります。
	 * 返される型は null, String, Long, Double, EOF のいずれかです。
	 * ストリームの終端に達した場合は EOF オブジェクトが返されます。
	 * 
	 * @return 読み取ったオブジェクトデータ
	 * @throws IOException IOException
	 */
	public Object read() throws IOException {
		int i = in.read();
		if(i < 0) {
			return EOF.instance;
		}
		byte firstByte = (byte)i;
		wasNull = (firstByte == BYTE_NULL);
		if(firstByte == BYTE_NULL) {
			return null;
		} else if(firstByte == BYTE_STRING_EMPTY) {
			return "";
		} else if(firstByte == BYTE_DOUBLE_NEXT8) {
			return readPrimitiveDouble(firstByte);
		} else if(((firstByte & 0x80) == 0x00) || ((firstByte & 0xC0) == 0x80)) {
			return readString(firstByte);
		} else {
			i = (int)firstByte & 0xFF;
			if(INT_LONG_0 <= i && i <= INT_LONG_NEXT7_N) {
				return readPrimitiveLong(firstByte);
			} else if(INT_DOUBLE_0 <= i && i <= INT_DOUBLE_N1) {
				return readPrimitiveDouble(firstByte);
			}
		}
		
		throw new IOException("not object");
	}
	
	public String readString() throws IOException {
		byte firstByte = read(in);
		if((wasNull = (firstByte == BYTE_NULL)) == true) {
			return null;
		}
		return readString(firstByte);
	}
	
	public long readPrimitiveLong() throws IOException {
		byte firstByte = read(in);
		if((wasNull = (firstByte == BYTE_NULL)) == true) {
			return 0L;
		}
		return readPrimitiveLong(firstByte);
	}
	
	public Long readLong() throws IOException {
		byte firstByte = read(in);
		if((wasNull = (firstByte == BYTE_NULL)) == true) {
			return null;
		}
		return readPrimitiveLong(firstByte);
	}
	
	public double readPrimitiveDouble() throws IOException {
		byte firstByte = read(in);
		if((wasNull = (firstByte == BYTE_NULL)) == true) {
			return 0.0D;
		}
		return readPrimitiveDouble(firstByte);
	}
	
	public Double readDouble() throws IOException {
		byte firstByte = read(in);
		if((wasNull = (firstByte == BYTE_NULL)) == true) {
			return null;
		}
		return readPrimitiveDouble(firstByte);
	}
	
	private String readString(byte firstByte) throws IOException {
		byte b = firstByte;
		if(b == BYTE_NULL) {
			return null;
		} else if(b == BYTE_STRING_EMPTY) {
			return "";
		} else if((b & 0x80) == 0x00) { //上位1ビットが 0 (2進数)
			int len = b;
			read(in, buf, 0, len);
			return new String(buf, 0, len, UTF8);
		} else if((b & 0xC0) == 0x80) { //上位2ビットが 10 (2進数)
			byte b2 = read(in);
			int len = (b & 0x3F) << 8 | (b2 & 0xFF);
			read(in, buf, 0, len);
			return new String(buf, 0, len, UTF8);
		}

		throw new IOException("not string");
	}
	
	private long readPrimitiveLong(byte firstByte) throws IOException {
		byte b = firstByte;
		if(b == BYTE_LONG_0) {
			return 0L;
		} else if(b == BYTE_LONG_1) {
			return 1L;
		} else if(b == BYTE_LONG_2) {
			return 2L;
		} else if(b == BYTE_LONG_3) {
			return 3L;
		} else if(b == BYTE_LONG_4) {
			return 4L;
		} else if(b == BYTE_LONG_5) {
			return 5L;
		} else if(b == BYTE_LONG_6) {
			return 6L;
		} else if(b == BYTE_LONG_7) {
			return 7L;
		} else if(b == BYTE_LONG_8) {
			return 8L;
		} else if(b == BYTE_LONG_9) {
			return 9L;
		} else if(b == BYTE_LONG_10) {
			return 10L;
		} else if(b == BYTE_LONG_11) {
			return 11L;
		} else if(b == BYTE_LONG_12) {
			return 12L;
		} else if(b == BYTE_LONG_13) {
			return 13L;
		} else if(b == BYTE_LONG_14) {
			return 14L;
		} else if(b == BYTE_LONG_15) {
			return 15L;
		} else if(b == BYTE_LONG_N1) {
			return -1L;
		} else {
			buf[0] = 0;
			buf[1] = 0;
			buf[2] = 0;
			buf[3] = 0;
			buf[4] = 0;
			buf[5] = 0;
			buf[6] = 0;
			boolean isNegative = false;
			if(b == BYTE_LONG_NEXT1) {
				read(in, buf, 7, 1);
			} else if(b == BYTE_LONG_NEXT2) {
				read(in, buf, 6, 2);
			} else if(b == BYTE_LONG_NEXT3) {
				read(in, buf, 5, 3);
			} else if(b == BYTE_LONG_NEXT4) {
				read(in, buf, 4, 4);
			} else if(b == BYTE_LONG_NEXT5) {
				read(in, buf, 3, 5);
			} else if(b == BYTE_LONG_NEXT6) {
				read(in, buf, 2, 6);
			} else if(b == BYTE_LONG_NEXT7) {
				read(in, buf, 1, 7);
			} else if(b == BYTE_LONG_NEXT8) {
				read(in, buf, 0, 8);
			} else if(b == BYTE_LONG_NEXT1_N) {
				isNegative = true;
				read(in, buf, 7, 1);
			} else if(b == BYTE_LONG_NEXT2_N) {
				isNegative = true;
				read(in, buf, 6, 2);
			} else if(b == BYTE_LONG_NEXT3_N) {
				isNegative = true;
				read(in, buf, 5, 3);
			} else if(b == BYTE_LONG_NEXT4_N) {
				isNegative = true;
				read(in, buf, 4, 4);
			} else if(b == BYTE_LONG_NEXT5_N) {
				isNegative = true;
				read(in, buf, 3, 5);
			} else if(b == BYTE_LONG_NEXT6_N) {
				isNegative = true;
				read(in, buf, 2, 6);
			} else if(b == BYTE_LONG_NEXT7_N) {
				isNegative = true;
				read(in, buf, 1, 7);
			} else {
				throw new IOException("not long");
			}
			long value = (((long)buf[0] & 0xFF) << 56)
					   | (((long)buf[1] & 0xFF) << 48)
					   | (((long)buf[2] & 0xFF) << 40)
					   | (((long)buf[3] & 0xFF) << 32)
					   | (((long)buf[4] & 0xFF) << 24)
					   | (((long)buf[5] & 0xFF) << 16)
					   | (((long)buf[6] & 0xFF) <<  8)
					   | (((long)buf[7] & 0xFF) <<  0);
			if(isNegative) {
				value *= -1;
			}
			return value;
		}
	}
	
	private double readPrimitiveDouble(byte firstByte) throws IOException {
		byte b = firstByte;
		if(b == BYTE_DOUBLE_0) {
			return 0.0D;
		} else if(b == BYTE_DOUBLE_1) {
			return 1.0D;
		} else if(b == BYTE_DOUBLE_2) {
			return 2.0D;
		} else if(b == BYTE_DOUBLE_3) {
			return 3.0D;
		} else if(b == BYTE_DOUBLE_4) {
			return 4.0D;
		} else if(b == BYTE_DOUBLE_5) {
			return 5.0D;
		} else if(b == BYTE_DOUBLE_6) {
			return 6.0D;
		} else if(b == BYTE_DOUBLE_7) {
			return 7.0D;
		} else if(b == BYTE_DOUBLE_8) {
			return 8.0D;
		} else if(b == BYTE_DOUBLE_9) {
			return 9.0D;
		} else if(b == BYTE_DOUBLE_10) {
			return 10.0D;
		} else if(b == BYTE_DOUBLE_11) {
			return 11.0D;
		} else if(b == BYTE_DOUBLE_12) {
			return 12.0D;
		} else if(b == BYTE_DOUBLE_13) {
			return 13.0D;
		} else if(b == BYTE_DOUBLE_14) {
			return 14.0D;
		} else if(b == BYTE_DOUBLE_15) {
			return 15.0D;
		} else if(b == BYTE_DOUBLE_N1) {
			return -1.0D;
		} else if(b == BYTE_DOUBLE_NEXT8) {
			read(in, buf, 0, 8);
			long v = (((long)buf[0] & 0xFF) << 56)
				   | (((long)buf[1] & 0xFF) << 48)
				   | (((long)buf[2] & 0xFF) << 40)
				   | (((long)buf[3] & 0xFF) << 32)
				   | (((long)buf[4] & 0xFF) << 24)
				   | (((long)buf[5] & 0xFF) << 16)
				   | (((long)buf[6] & 0xFF) <<  8)
				   | (((long)buf[7] & 0xFF) <<  0);
			return Double.longBitsToDouble(v);
		}
		
		throw new IOException("not double");
	}

	private static byte read(InputStream in) throws IOException {
		int i = in.read();
		if(i == -1) {
			throw new EOFException();
		}
		return (byte)i;
	}
	
	private static void read(InputStream in, byte[] buf, int off, int len) throws IOException {
		int size;
		while(len > 0) {
			size = in.read(buf, off, len);
			if(size < 0) {
				throw new EOFException();
			}
			off += size;
			len -= size;
		}
	}
}
