package net.osdn.util.concurrent;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/** 複数のオブジェクトを対象としてスレッドを待機させるモニターです。
 * 
 * <p>Javaのすべてのオブジェクトはスレッドを待機・再開させるためのモニターを持っていますが、
 * 待機させたスレッドは特定のオブジェクトのnotify()メソッドによってしか再開させることができません。
 * このMonitorクラスは複数のオブジェクトで構成される1つのモニターを提供します。</p>
 * 
 * <p>オブジェクトAとオブジェクトBでモニターを構成した場合、
 * オブジェクトA、オブジェクトBいずれかの通知によって待機させたスレッドを再開させることができます。</p>
 */
public class Monitor {

	private final Object notifySynchronizer = new Object();
	private final Object lockSynchronizer = new Object();
	private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock(true);
	private volatile Object object;

	/** 新しいモニターを作成します。
	 * 
	 */
	public Monitor() {
	}
	
	/** 複数のオブジェクトをまとめたMultipleObjectsを作成します。
	 * 
	 * @param objects スレッドを待機・再開させるための対象オブジェクト
	 * @return 指定されたオブジェクトをまとめたMultipleObjects
	 */
	public MultipleObjects newMultipleObjects(Object... objects) {
		return new MultipleObjects(0, objects);
	}
	
	/** 複数のオブジェクトをまとめたMultipleObjectsを作成します。
	 * 
	 * @param objects スレッドを待機・再開させるための対象オブジェクト
	 * @return 指定されたオブジェクトをまとめたMultipleObjects
	 */
	public MultipleObjects newMultipleObjects(Collection<?> objects) {
		return new MultipleObjects(0, objects);
	}
	
	/** 待機時間を指定して複数のオブジェクトをまとめたMultipleObjectsを作成します。
	 * 
	 * <p>MultipleObjectsを作成した時点で待機時間の開始点が設定されます。
	 * waitForMultipleObjectsメソッドの呼び出し時が待機時間の開始点にならないことに注意してください。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param objects スレッドを待機・再開させるための対象オブジェクト
	 * @return 指定されたオブジェクトをまとめたMultipleObjects
	 */
	public MultipleObjects newMultipleObjects(long timeout, Object... objects) {
		return new MultipleObjects(timeout, objects);
	}
	
	/** 待機時間を指定して複数のオブジェクトをまとめたMultipleObjectsを作成します。
	 * 
	 * <p>MultipleObjectsを作成した自邸で待機時間の開始点が設定されます。
	 * waitForMultipleObjectsメソッドの呼び出し時が待機時間の開始点にならないことに注意してください。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param objects スレッドを待機・再開させるための対象オブジェクト
	 * @return 指定されたオブジェクトをまとめたMultipleObjects
	 */
	public MultipleObjects newMultipleObjects(long timeout, Collection<?> objects) {
		return new MultipleObjects(timeout, objects);
	}
	
	/** 指定したオブジェクトをまとめているMultipleObjectsのwaitForMultipleObjects()メソッドで待機しているすべてのスレッドを再開させます。
	 * 
	 * @param object MultipleObjectsに含まれているいずれかのオブジェクト
	 */
	public void notifyAll(Object object) {
		if(object == null) {
			throw new IllegalArgumentException();
		}
		synchronized(lockSynchronizer) {
			
			this.object = object;
			
			synchronized(notifySynchronizer) {
				notifySynchronizer.notifyAll();
			}
			try {
				lock.writeLock().lock();
				this.object = null;
			} finally {
				lock.writeLock().unlock();
			}
		}
	}
	
	/** 複数のオブジェクトを対象としてスレッドを待機・再開させるために複数のオブジェクトをまとめるクラスです。
	 * 
	 */
	public class MultipleObjects {
		
		private long startNanoTime = System.nanoTime();
		private long timeout;
		private Set<Object> objects;
		
		protected MultipleObjects(long timeout, Object... objects) {
			if(timeout < 0) {
				throw new IllegalArgumentException();
			}
			if(objects == null) {
				throw new IllegalArgumentException();
			}
			this.timeout = timeout;
			this.objects = new HashSet<Object>(Arrays.asList(objects));
		}
		
		protected MultipleObjects(long timeout, Collection<?> objects) {
			if(objects == null) {
				throw new IllegalArgumentException();
			}
			if(objects.size() == 0) {
				throw new IllegalArgumentException();
			}
			this.timeout = timeout;
			this.objects = new HashSet<Object>(objects);
		}
		
		/** スレッドを待機させる時間を設定します。
		 * 
		 * <p>waitForMultipleObjectsメソッドの呼び出し時ではなく、
		 * このメソッドを呼び出した時点で待機時間の開始点が設定されます。</p>
		 * 
		 * @param timeout ミリ秒単位の待機時間
		 */
		public void setTimeout(long timeout) {
			if(timeout < 0) {
				throw new IllegalArgumentException();
			}
			this.timeout = timeout;
			this.startNanoTime = System.nanoTime();
		}
		
		/** ほかのスレッドがこのMultipleObjectに含まれるいずれかのオブジェクトを対象としてMonitor.notifyAll(Object)メソッドを呼び出すか、
		 * ほかのスレッドが現在のスレッドに割り込みをかけたり、あらかじめ指定されている量の実時間が経過するまで、現在のスレッドを待機させます。
		 * 
		 * <p>このメソッドの呼び出しによって待機したスレッドはMonitor.notifyAll(object)メソッドによって再開されます。</p>
		 * 
		 * <p>待機時間が設定されている場合の待機時間の開始点はMultipleObjectsインスタンスが生成時、または、setTimeout(long)メソッドを呼び出し時です。
		 * waitForMultipleObjects()メソッドを繰り返し呼び出す場合、残りの待機時間は減少していきます。</p>
		 * 
		 * @return スレッド再開の要因となったオブジェクト。これはMonitor.notifyAll(Object)メソッドで指定されたオブジェクトです。待機時間が経過した場合はnull。
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 */
		public Object waitForMultipleObjects() throws InterruptedException {
			if(timeout <= 0) {
				for(;;) {
					try {
						lock.readLock().lock();
						synchronized (notifySynchronizer) {
							notifySynchronizer.wait();
						}
						if(object != null && objects.contains(object)) {
							return object;
						}
					} finally {
						lock.readLock().unlock();
					}
					synchronized(lockSynchronizer) { }
				}
			} else {
				long restMillis;
				while((restMillis = timeout - (System.nanoTime() - startNanoTime) / 1000000) > 0) {
					try {
						lock.readLock().lock();
						synchronized (notifySynchronizer) {
							notifySynchronizer.wait(restMillis);
						}
						if(object != null && objects.contains(object)) {
							return object;
						}
					} finally {
						lock.readLock().unlock();
					}
					synchronized(lockSynchronizer) { }
				}
			}
			return null;
		}
		
		/** ほかのスレッドがこのMultipleObjectに含まれるいずれかのオブジェクトを対象としてMonitor.notifyAll(Object)メソッドを呼び出すか、
		 * ほかのスレッドが現在のスレッドに割り込みをかけたり、指定された量の実時間が経過するまで、現在のスレッドを待機させます。
		 *
		 * <p>このメソッドの呼び出しによって待機したスレッドはMonitor.notifyAll(object)メソッドによって再開されます。</p>
		 * 
		 * @param timeout ミリ秒単位の待機時間
		 * @return スレッド再開の要因となったオブジェクト。これはMonitor.notifyAll(Object)メソッドで指定されたオブジェクトです。待機時間が経過した場合はnull。
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 */
		public Object waitForMultipleObjects(long timeout) throws InterruptedException {
			if(timeout < 0) {
				throw new IllegalArgumentException();
			}
			if(timeout == 0) {
				for(;;) {
					try {
						lock.readLock().lock();
						synchronized (notifySynchronizer) {
							notifySynchronizer.wait();
						}
						if(object != null && objects.contains(object)) {
							return object;
						}
					} finally {
						lock.readLock().unlock();
					}
					synchronized(lockSynchronizer) { }
				}
			} else {
				long startNanoTime = System.nanoTime();
				long restMillis;
				while((restMillis = timeout - (System.nanoTime() - startNanoTime) / 1000000) > 0) {
					try {
						lock.readLock().lock();
						synchronized (notifySynchronizer) {
							notifySynchronizer.wait(restMillis);
						}
						if(object != null && objects.contains(object)) {
							return object;
						}
					} finally {
						lock.readLock().unlock();
					}
					synchronized (lockSynchronizer) { }
				}
			}
			return null;
		}
	}
}
