package net.osdn.util.sql;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.PreparedStatement;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.osdn.util.sql.NamedParameter.Method;

/** 名前付きパラメーターを使用してSQL文を作成するためのユーティリティー・クラスです。
 * 
 * <p>JDBCのPreparedStatementはパラメーターのプレースホルダーとして {@literal ?} をサポートしていますが、
 * これは名前によるパラメーターの指定をサポートしておらず、{@literal ?} の出現順によってパラメーターを指定する必要があります。
 * 一部のデータベース製品のJDBCドライバーには独自に名前付きパラメーターをサポートしているものもありますが、
 * データベース製品固有の方法で名前付きパラメーターを実装しているため、汎用性がなくなります。</p>
 * 
 * <p>このユーティリティークラスは特定のJDBCドライバー実装に依存することなく名前付きパラメーターをサポートします。</p>
 * 
 * <p>NamedParameterStatementではパラメーターのプレースホルダーとして {@literal :name} という形式を使用します。
 * {@literal :}に続く文字列がパラメーター名です。</p>
 * 
 * <p>id列に値を指定してPersonテーブルを取得する場合は以下のようになります。</p>
 * 
 * <pre><code>
 * String sql = "SELECT * FROM Person WHERE id = :id";
 * NamedParameterStatement npst = new NamedParameterStatement(sql);
 * npst.setInt("id", 3);
 * 
 * PreparedStatement st = cn.prepareStatement(npst.getSql());
 * for(NamedParameter parameter : npst.getParameters()) {
 *     parameter.applyTo(st);
 * }
 * 
 * ResultSet rs = st.executeQuery();
 * </code></pre>
 * 
 * <p>{@link NamedParameterStatement#getSql()}は名前付きパラメーターのプレースホルダーを
 * 名前なしパラメーターのプレースホルダー {@literal ?} に置き換えたSQL文を返します。
 * そのため、{@link #getSql()}が返すSQL文はJDBC標準のPreparedStatementを作成するのに使えます。
 * NamedParameterStatementにパラメーターの値を設定するメソッド({@link #setInt(String, int)} など)は
 * 第1引数にパラメーターの出現順ではなく、パラメーターの名前を指定することができます。
 * NamedParameterStatementにパラメーターを設定すると、内部に{@link NamedParameter}が作成されます。
 * {@link NamedParameter}はパラメーターの名前と値だけでなくパラメーターの出現順序も保持しています。
 * {@link NamedParameter#applyTo(PreparedStatement)}を呼び出すと、パラメーターの出現順序を使用して
 * PreparedStatementにパラメーターを設定します。</p>
 * 
 * <pre><code>
 * PreparedStatement st = cn.prepareStatement(npst.getSql());
 * for(NamedParameter parameter : npst.getParameters()) {
 *     parameter.applyTo(st);
 * }
 * </code></pre>
 * 
 * このコードは以下のように書くこともできます。
 * 
 * <pre><code>
 * PreparedStatement st = cn.prepareStatement(npst.getSql());
 *     for(NamedParameter parameter : npst.getParameters()) {
 *     st.setObject(parameter.getIndex(), parameter.getValue());
 * }
 * </code></pre>
 * 
 * <p>NamedParameterStatementは名前付きパラメーターを持つSQL文から、
 * パラメーターのプレースホルダー {@literal ?} を使ったJDBC標準のSQL文と
 * パラメーターのプレースホルダー出現順(インデックス)を保持したパラメーターに分解する機能を提供します。
 * これにより任意のPreparedStatemntで名前付きパラメーターを利用できるようになります。
 * NamedParameterStatementはユーティリティー・クラスであり、
 * 実際にデータベースへアクセスするのはJDBCドライバー固有のPreparedStatement実装です。
 * そのため、JDBCドライバー固有のPreparedStatementが持つプリコンパイル機能や最適化が損なわれることもありません。</p>
 * 
 */
public class NamedParameterStatement {

	private static final Pattern PARAMETER_PATTERN = Pattern.compile(
			"'(?:''|[^'])*'|--.*?$|\\/\\*.*?\\*\\/|:(.+?)\\b|(\\?)",
			Pattern.MULTILINE | Pattern.DOTALL);
	
	private String original;
	private String sql;
	private Set<String> parameterNames = new LinkedHashSet<String>();
	private Map<String, int[]> indexesMap = new LinkedHashMap<String, int[]>();
	private Map<Integer, NamedParameter> parameters = new TreeMap<Integer, NamedParameter>();
	/* package private */ boolean hasRowVersionColumn;
	
	/** 指定したSQL文で名前付きパラメーター・ステートメントを作ります。
	 * 
	 * <p>パラメーターのプレースホルダーには {@literal ?} ではなく {@literal :name} 形式を使用します。</p>
	 * 
	 * @param sql 名前付きパラーメーターのプレースホルダーを含むSQL文
	 */
	public NamedParameterStatement(String sql) {
		original = sql;
		
		List<String> names = new ArrayList<String>();
		StringBuffer sb = new StringBuffer();
		Matcher m = PARAMETER_PATTERN.matcher(sql);
		while(m.find()) {
			if(m.group(1) != null) {
				// :name
				names.add(m.group(1));
				m.appendReplacement(sb, "?");
			} else if(m.group(2) != null) {
				// ?
				names.add(m.group(2));
				m.appendReplacement(sb, "?");
			} else {
				// quoted string or comment
			}
		}
		m.appendTail(sb);
		this.sql = sb.toString();

		Map<String, List<Integer>> map = new HashMap<String, List<Integer>>();
		for(int i = 0; i < names.size(); i++) {
			String name = names.get(i);
			List<Integer> indexes = map.get(name.toLowerCase());
			if(indexes == null) {
				indexes = new ArrayList<Integer>();
				map.put(name.toLowerCase(), indexes);
			}
			indexes.add(i + 1);
			if(!parameterNames.contains(name.toLowerCase())) {
				this.parameterNames.add(name);
			}
		}
		
		for(String name : this.parameterNames) {
			List<Integer> list = map.get(name.toLowerCase());
			int[] indexes = new int[list.size()];
			for(int j = 0; j < indexes.length; j++) {
				indexes[j] = list.get(j);
			}
			indexesMap.put(name.toLowerCase(), indexes);
		}
	}
	
	/** コンストラクタで指定された元のSQL文を取得します。
	 * 
	 * <p>このメソッドが返すSQL文は加工されていないため、
	 * 名前付きパラメーターのプレースホルダー {@literal :name} を含みます。</p>
	 * 
	 * @return コンストラクタで指定された元のSQL文
	 */
	public String getOriginalSql() {
		return original;
	}
	
	/** JDBCで使用するSQL文を取得します。
	 * 
	 * <p>このメソッドが返すSQL文は名前付きパラメーターのプレースホルダー {@literal :name} が
	 * JDBC標準のパラメーター・プレースホルダー {@literal ?} に置き換えられています。</p>
	 * 
	 * @return パラメーターのプレースホルダーを {@literal ?} に置き換えたSQL文
	 */
	public String getSql() {
		return sql;
	}
	
	/** このNamedParameterStatementが持つパラメーター名のセットを取得します。
	 * 
	 * <p>"SELECT * FROM Product WHERE category = :category AND price &gt; :price" というSQL文を
	 * コンストラクタで指定した場合、このメソッドが返すパラメーター名のセットには "category" と "price" が含まれます。
	 * パラメータ名には {@literal :} は含まれません。</p>
	 * 
	 * @return パラメーター名のセット
	 */
	public Set<String> getParameterNames() {
		return parameterNames;
	}
	
	/** 指定したパラメーターの出現順を配列で取得します。
	 * 
	 * <p>このメソッドが返すパラメーターの出現順は0ではなく1から始まります。
	 * この値はPreparedStatementのパラメーター指定に使用することができます。</p>
	 * 
	 * <p>"SELECT * FROM Product WHERE price &gt;= :price AND (flag = :param1 OR (:param1 is null AND flag is null))" というSQL文を
	 * コンストラクタで指定した場合、getIndexesBy("param1") は int[] { 2, 3 } を返します。</p>
	 * 
	 * @param parameterName パラメーターの名前
	 * @return パラメーターの出現順配列
	 */
	public int[] getIndexesBy(String parameterName) {
		int[] indexes = indexesMap.get(parameterName.toLowerCase());
		if(indexes == null) {
			throw new IllegalArgumentException();
		}
		return indexes;
	}
	
	/** パラメーターに設定していた値をすべてクリアします。
	 * 
	 */
	public void clearParameters() {
		parameters.clear();
	}
	
	/** 設定したパラメーターの値を持つ{@link NamedParameter}配列を返します。
	 * 
	 * @return 設定したパラメーターの配列。パラメーターが設定されていない場合はサイズ 0 の配列。null が返されることはありません。
	 */
	public NamedParameter[] getParameters() {
		return parameters.values().toArray(new NamedParameter[]{});
	}
	
	/** 指定されたパラメーターを指定されたjava.sql.Arrayオブジェクトに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x SQL ARRAY値をマッピングするArrayオブジェクト
	 */
	public void setArray(String parameterName, java.sql.Array x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_ARRAY, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定された入力ストリームに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x ASCIIパラメーター値を含むJava入力ストリーム
	 */
	public void setAsciiStream(String parameterName, InputStream x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_ASCII_STREAM, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたバイト数を持つ指定された入力ストリームに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x ASCIIパラメーター値を含むJava入力ストリーム
	 * @param length ストリームのバイト数
	 */
	public void setAsciiStream(String parameterName, InputStream x, int length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_ASCII_STREAM, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたバイト数を持つ指定された入力ストリームに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x ASCIIパラメーター値を含むJava入力ストリーム
	 * @param length ストリームのバイト数
	 */
	public void setAsciiStream(String parameterName, InputStream x, long length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_ASCII_STREAM, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.math.BigDecimal値に設定します。
	 * データベースに送るときに、ドライバーはこれを SQL NUMERIC値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setBigDecimal(String parameterName, BigDecimal x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_BIG_DECIMAL, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定された入力ストリームに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x バイナリ・パラメーター値を含むJava入力ストリーム
	 */
	public void setBinaryStream(String parameterName, InputStream x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_BINARY_STREAM, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたバイト数を持つ指定された入力ストリームに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x バイナリ・パラメーター値を含むJava入力ストリーム
	 * @param length ストリームのバイト数
	 */
	public void setBinaryStream(String parameterName, InputStream x, int length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_BINARY_STREAM, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたバイト数を持つ指定された入力ストリームに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x バイナリ・パラメーター値を含むJava入力ストリーム
	 * @param length ストリームのバイト数
	 */
	public void setBinaryStream(String parameterName, InputStream x, long length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_BINARY_STREAM, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.sql.Blobオブジェクトに設定します。
	 * データベースに送るときに、ドライバーはこれをSQL BLOB値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x SQL BLOB値をマッピングするBlobオブジェクト
	 */
	public void setBlob(String parameterName, java.sql.Blob x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_BLOB, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをInputStreamオブジェクトに設定します。
	 * このメソッドは{@link #setBinaryStream(String, InputStream)}メソッドと異なり、
	 * パラメーター値をBLOBとしてサーバーに送信するべきであることをドライバーに通知します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーター値に設定されるデータを含むオブジェクト
	 */
	public void setBlob(String parameterName, InputStream x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_BLOB, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをInputStreamオブジェクトに設定します。
	 * ストリームにはlengthで指定される文字数が含まれる必要があります。
	 * そうでない場合は、PreparedStatementの実行時にSQLExceptionが生成されます。
	 * このメソッドは{@link #setBinaryStream(String, InputStream, int)}メソッドと異なり、
	 * パラメーター値をBLOBとしてサーバーに送信するべきであることをドライバーに通知します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーター値に設定されるデータを含むオブジェクト
	 * @param length パラメーター・データ内のバイト数
	 */
	public void setBlob(String parameterName, InputStream x, long length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_BLOB, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのboolean値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL BITまたはBOOLEAN値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setBoolean(String parameterName, boolean x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_BOOLEAN, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのbyte値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL TINYINT値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setByte(String parameterName, byte x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_BYTE, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのバイト配列に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL VARBINARYまたはLONGVARBINARYに変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setBytes(String parameterName, byte[] x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_BYTES, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたReaderオブジェクトに設定します。
	 * JDBCドライバーはデータをUNICODEからデータベースのchar形式に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x Unicodeを格納するjava.io.Readerオブジェクト
	 */
	public void setCharacterStream(String parameterName, Reader x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_CHARACTER_STREAM, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定された文字数である指定されたReaderオブジェクトに設定します。
	 * JDBCドライバーはデータをUNICODEからデータベースのchar形式に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x Unicodeデータを格納するjava.io.Readerオブジェクト
	 * @param length ストリーム内の文字数
	 */
	public void setCharacterStream(String parameterName, Reader x, int length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_CHARACTER_STREAM, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定された文字数である指定されたReaderオブジェクトに設定します。
	 * JDBCドライバーはデータをUNICODEからデータベースのchar形式に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x Unicodeデータを格納するjava.io.Readerオブジェクト
	 * @param length ストリーム内の文字数
	 */
	public void setCharacterStream(String parameterName, Reader x, long length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_CHARACTER_STREAM, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.sql.Clobオブジェクトに設定します。
	 * データベースに送るときに、ドライバーはこれをSQL CLOB値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x SQL CLOB値をマッピングするClobオブジェクト
	 */
	public void setClob(String parameterName, java.sql.Clob x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_CLOB, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをReaderオブジェクトに設定します。
	 * このメソッドは{@link #setCharacterStream(String, Reader)}メソッドと異なり、
	 * パラメーター値をCLOBとしてサーバーに送信するべきであることをドライバーに通知します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーター値に設定されるデータを含むオブジェクト
	 */
	public void setClob(String parameterName, Reader x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_CLOB, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをReaderオブジェクトに設定します。
	 * Reader xにはlengthで指定される文字数が含まれる必要があります。
	 * そうでない場合、PreparedStatementの実行時にSQLExceptionが生成されます。
	 * このメソッドは{@link #setCharacterStream(String, Reader, int)}メソッドと異なり、
	 * パラメーター値をCLOBとしてサーバーに送信するべきであることをドライバーに通知します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーター値に設定されるデータを含むオブジェクト
	 * @param length パラメーター・データ内の文字数
	 */
	public void setClob(String parameterName, Reader x, long length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_CLOB, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** アプリケーションを実行している仮想マシンのデフォルトのタイムゾーンを使用して、
	 * 指定されたパラメーターを指定されたjava.sql.Date値に設定します。
	 * データベースに送るときに、ドライバーはこれを SQL DATE値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setDate(String parameterName, java.sql.Date x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_DATE, index, parameterName, x));
		}
	}
	
	/** 指定されたCalendarオブジェクトを使用して、指定されたパラメーターを指定されたjava.sql.Date値に設定します。
	 * ドライバーはCalendarオブジェクトを使用してSQL DATE値を作成し、続いてそれをデータベースに送ります。
	 * Calendarオブジェクトを使用すると、ドライバーはカスタム・タイムゾーンを考慮して日付を計算できます。
	 * Calendarオブジェクトを指定しない場合、ドライバーはアプリケーションを
	 * 実行している仮想マシンのタイムゾーンであるデフォルトのタイムゾーンを使用します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 * @param calendar ドライバーが日付を作成するために使用するCalendarオブジェクト
	 */
	public void setDate(String parameterName, java.sql.Date x, Calendar calendar) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_DATE, index, parameterName, x);
			param.setCalendar(calendar);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのdouble値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL DOUBLE値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setDouble(String parameterName, double x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_DOUBLE, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのfloat値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL REAL値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setFloat(String parameterName, float x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_FLOAT, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのint値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL INTEGER値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setInt(String parameterName, int x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_INT, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのlong値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL BIGINT値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setLong(String parameterName, long x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_LONG, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをReaderオブジェクトに設定します。
	 * ドライバーは必要に応じて、Javaの文字列表現をデータベース内の各国文字セットに変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setNCharacterStream(String parameterName, Reader x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_NCHARACTER_STREAM, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをReaderオブジェクトに設定します。
	 * ドライバーは必要に応じて、Javaの文字列表現をデータベース内の各国文字セットに変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 * @param length パラメーター・データ内の文字数
	 */
	public void setNCharacterStream(String parameterName, Reader x, long length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_NCHARACTER_STREAM, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターをjava.sql.NClobオブジェクトに設定します。
	 * データベースに送るときに、ドライバーはこれをSQL NCLOB値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setNClob(String parameterName, java.sql.NClob x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_NCLOB, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをReaderオブジェクトに設定します。
	 * このメソッドは、{@link #setCharacterStream(String, Reader)}メソッドと異なり、
	 * パラメーター値をNCLOBとしてサーバーに送信するべきであることをドライバーに通知します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setNClob(String parameterName, Reader x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_NCLOB, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをReaderオブジェクトに設定します。
	 * Reader xにはlengthで指定される文字数が含まれる必要があります。
	 * そうでない場合、PreparedStatementの実行時にSQLExceptionが生成されます。
	 * このメソッドは、{@link #setCharacterStream(String, Reader, int)}メソッドと異なり、
	 * パラメーター値をNCLOBとしてサーバーに送信するべきであることをドライバーに通知します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 * @param length パラメーター・データ内の文字数
	 */
	public void setNClob(String parameterName, Reader x, long length) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_NCLOB, index, parameterName, x);
			param.setLength(length);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたStringオブジェクトに設定します。
	 * データベースに送るときに、ドライバーはこれをSQL NCHAR、またはLONGVARCHAR値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setNString(String parameterName, String x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_NSTRING, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターをSQL NULLに設定します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param sqlType java.sql.Typesで定義されるSQL型コード
	 */
	public void setNull(String parameterName, int sqlType) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_NULL, index, parameterName, null);
			param.setType(sqlType);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターをSQL NULLに設定します。
	 * ユーザー定義型およびREF型のパラメーターでは、このバージョンのsetNullメソッドを使用するべきです。
	 * ユーザー定義型の例には、STRUCT、DISTINCT、JAVA_OBJECT、および名前付き配列があります。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param sqlType java.sql.Typesで定義されるSQL型コード
	 * @param typeName SQLユーザー定義型の完全指定の名前。パラメーターがユーザー定義型でもREFでもない場合は無視されます。
	 */
	public void setNull(String parameterName, int sqlType, String typeName) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_NULL, index, parameterName, null);
			param.setType(sqlType);
			param.setTypeName(typeName);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターの値を指定されたオブジェクトを使用して設定します。
	 * 
	 * <p>JDBCの仕様では、JavaのObject型からSQL型への標準マッピングを規定しています。
	 * 指定された引数は、データベースに送られる前に、対応するSQL型に変換されます。</p>
	 * 
	 * <p>このメソッドは、ドライバー固有のJava型を使用して、データベース固有の抽象データ型を渡すために使用することに注意してください。
	 * オブジェクトがインターフェースSQLDataを実装するクラスのインスタンスである場合、
	 * JDBCドライバーはSQLData.writeSQLメソッドを呼び出して、そのオブジェクトをSQLデータ・ストリームへ書き込む必要があります。
	 * また、オブジェクトがRef、Blob、Clob、NClob、Struct、java.net.URL、RowId、SQLXML、またはArrayを実装するクラスのオブジェクトである場合、
	 * ドライバーはこのオブジェクトを対応するSQL型の値としてデータベースに渡すべきです。</p>
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x 入力パラメーター値を含むオブジェクト
	 */
	public void setObject(String parameterName, Object x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_OBJECT, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーター値を指定されたオブジェクトで設定します。
	 * 
	 * <p>このメソッドは、{@link #setObject(String, Object, int, int)} に似ていますが、scaleOrLengthに0を仮定している点が異なります。</p>
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x 入力パラメーター値を含むオブジェクト
	 * @param targetSqlType データベースに送られる(java.sql.Typesで定義される)SQL型
	 */
	public void setObject(String parameterName, Object x, int targetSqlType) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_OBJECT, index, parameterName, x);
			param.setType(targetSqlType);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターの値を指定されたオブジェクトで設定します。
	 * 
	 * <p>2番目の引数がInputStreamである場合、ストリームにはscaleOrLengthで指定されるバイト数が含まれる必要があります。
	 * 2番目の引数がReaderである場合、リーダーにはscaleOrLengthで指定される文字数が含まれる必要があります。
	 * そうでない場合は、PreparedStatementの実行時にSQLExceptionが生成されます。</p>
	 * 
	 * <p>指定されたJavaオブジェクトは、データベースに送られる前に、指定されたtargetSqlTypeに変換されます。
	 * オブジェクトがカスタム・マッピングを持つ場合(オブジェクトがインターフェースSQLDataを実装するクラスのインスタンスである場合)、
	 * JDBCドライバーはSQLData.writeSQLメソッドを呼び出して、そのオブジェクトをSQLデータ・ストリームへ書き込む必要があります。
	 * また、オブジェクトがRef、Blob、Clob、NClob、Struct、java.net.URL、またはArrayを実装するクラスのオブジェクトである場合、
	 * ドライバーはこのオブジェクトを対応するSQL型の値としてデータベースに渡す必要があります。</p>
	 * 
	 * <p>このメソッドは、データベース固有の抽象データ型を渡すために使用することに注意してください。</p>
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x 入力パラメーター値を含むオブジェクト
	 * @param targetSqlType データベースに送られる(java.sql.Typesで定義される)SQL型。scaleOrLength引数で、さらにこの型を限定できます。
	 * @param scaleOrLength java.sql.Types.DECIMALまたはjava.sql.Types.NUMERICの場合、これは小数点のあとの桁数になります。Javaオブジェクト型InputStreamおよびReaderの場合はストリームまたはReaderのデータ長。ほかのすべての型ではこの値は無視されます。
	 */
	public void setObject(String parameterName, Object x, int targetSqlType, int scaleOrLength) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_OBJECT, index, parameterName, x);
			param.setType(targetSqlType);
			param.setLength(scaleOrLength);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたREF(&lt;structured-type&gt;)値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL REF値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x SQL REF値
	 */
	public void setRef(String parameterName, java.sql.Ref x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_REF, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.sql.RowIdオブジェクトに設定します。
	 * データベースに送るときに、ドライバーはこれをSQL ROWID値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setRowId(String parameterName, java.sql.RowId x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_ROWID, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定したJavaのshort値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL SMALLINT値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setShort(String parameterName, short x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_SHORT, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.sql.SQLXMLオブジェクトに設定します。
	 * データベースに送るときに、ドライバーはこれをSQL XML値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x SQL XML値をマッピングするSQLXMLオブジェクト
	 */
	public void setSQLXML(String parameterName, java.sql.SQLXML x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_SQLXML, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたJavaのString値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL VARCHARまたはLONGVARCHAR値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setString(String parameterName, String x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_STRING, index, parameterName, x));
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.sql.Time値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL TIME値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setTime(String parameterName, java.sql.Time x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_TIME, index, parameterName, x));
		}
	}
	
	/** 指定されたCalendarオブジェクトを使用して、指定されたパラメーターを指定されてjava.sql.Time値に設定します。
	 * ドライバーはCalendarオブジェクトを使用してSQL TIME値を作成し、続いてそれをデータベースに送ります。
	 * Calendarオブジェクトを使用すると、ドライバーはカスタム・タイムゾーンを考慮して時間を計算できます。
	 * Calendarオブジェクトを指定しない場合、ドライバーはアプリケーションを実行している仮想マシンのタイムゾーンであるデフォルトのタイムゾーンを使用します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 * @param calendar ドライバーが時間作成するために使用するCalendarオブジェクト
	 */
	public void setTime(String parameterName, java.sql.Time x, Calendar calendar) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_TIME, index, parameterName, x);
			param.setCalendar(calendar);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.sql.Timestamp値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL TIMESTAMP値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setTimestamp(String parameterName, java.sql.Timestamp x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_TIMESTAMP, index, parameterName, x));
		}
	}
	
	/** 指定されたCalendarオブジェクトを使用して、指定されたパラメーターを指定されたjava.sql.Timestamp値に設定します。
	 * ドライバーはCalendarオブジェクトを使用してSQL TIMESTAMP値を作成し、続いてそれをデータベースに送ります。
	 * Calendarオブジェクトを使用すると、ドライバーはカスタム・タイムゾーンを考慮してタイムスタンプを計算できます。
	 * Calendarオブジェクトを指定しない場合、ドライバーはアプリケーションを実行している仮想マシンのタイムゾーンであるデフォルトのタイムゾーンを使用します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 * @param calendar ドライバーがタイムスタンプを作成するために使用するCalendarオブジェクト
	 */
	public void setTimestamp(String parameterName, java.sql.Timestamp x, Calendar calendar) {
		for(int index : getIndexesBy(parameterName)) {
			NamedParameter param = new NamedParameter(Method.SET_TIMESTAMP, index, parameterName, x);
			param.setCalendar(calendar);
			parameters.put(index, param);
		}
	}
	
	/** 指定されたパラメーターを指定されたjava.net.URL値に設定します。
	 * データベースに送るときに、ドライバーはこれをSQL DATALINK値に変換します。
	 * 
	 * @param parameterName パラメーターの名前
	 * @param x パラメーターの値
	 */
	public void setURL(String parameterName, URL x) {
		for(int index : getIndexesBy(parameterName)) {
			parameters.put(index, new NamedParameter(Method.SET_URL, index, parameterName, x));
		}
	}
	
	/** オブジェクトの文字列表現を返します。
	 * このメソッドは、プレースホルダーを設定されたパラメーターで置き換えたSQL文を返します。
	 * 
	 * 通常、JDBCドライバーはパラメーターを文字列置換することなくSQLとパラメーターをデータベースに送信します。
	 * このメソッドが返すSQL文はデータベースで実行されるSQLとは厳密には一致しないことに注意してください。
	 * 
	 */
	@Override
	public String toString() {
		return new Sql(this).dump();
	}
}
