package jp.sourceforge.masasa.architecture.framework.validation.enumeration;

import java.text.SimpleDateFormat;

import jp.sourceforge.masasa.architecture.framework.validation.validator.IValidator;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateByte;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateCreditCard;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateDate;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateDouble;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateDoubleRange;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateEmail;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateFloat;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateFloatRange;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateIntRange;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateInteger;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateLong;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateMask;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateMaxLength;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateMinLength;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateRequired;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateRequiredIf;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateShort;
import jp.sourceforge.masasa.architecture.framework.validation.validator.ValidateUrl;

/**
 * バリデーションクラスをシングルトン・インスタンスで公開するための管理クラス.
 *
 * @author masasa.
 */
public enum CheckRulesEnum {
    /**
     * Checks if the field can safely be converted to a byte primitive.
     *
     * @see ValidateByte
     */
    ByteCheck(ValidateByte.class),
    /**
     * Checks if the field matches the regular expression in the field's mask attribute.
     *
     * @see ValidateMask
     */
    MaskCheck(ValidateMask.class),
    /**
     * Checks if the field matches the regular expression in the field's mask attribute.
     *
     * @see ValidateCreditCard
     */
    CreditCardCheck(ValidateCreditCard.class),
    /**
     * Checks if the field is a valid date. If the field has a datePattern variable, that will be used to format
     * {@link SimpleDateFormat}. If the field has a datePatternStrict variable, that will be used to format
     * {@link SimpleDateFormat} and the length will be checked so '2/12/1999' will not pass validation with the format
     * 'MM/dd/yyyy' because the month isn't two digits. If no datePattern variable is specified, then the field gets the
     * DateFormat.SHORT format for the locale. The setLenient method is set to <code>false</code> for all variations.
     *
     * @see ValidateDate
     */
    DateCheck(ValidateDate.class),
    /**
     * Checks if the field can safely be converted to a double primitive.
     *
     * @see ValidateDouble
     */
    DoubleCheck(ValidateDouble.class),
    /**
     * Checks if a fields value is within a range (min &amp; max specified in the vars attribute).
     *
     * @see ValidateDoubleRange
     */
    DoubleRangeCheck(ValidateDoubleRange.class),
    /**
     * Checks if a field has a valid e-mail address.
     *
     * @see ValidateEmail
     */
    EmailCheck(ValidateEmail.class),
    /**
     * Checks if the field can safely be converted to a float primitive.
     *
     * @see ValidateFloat
     */
    FloatCheck(ValidateFloat.class),
    /**
     * Checks if a fields value is within a range (min &amp; max specified in the vars attribute).
     *
     * @see ValidateFloatRange
     */
    FloatRangeCheck(ValidateFloatRange.class),
    /**
     * Checks if the field can safely be converted to an int primitive.
     *
     * @see ValidateInteger
     */
    IntegerCheck(ValidateInteger.class),
    /**
     * Checks if a fields value is within a range (min &amp; max specified in the vars attribute).
     *
     * @see ValidateIntRange
     */
    IntRangeCheck(ValidateIntRange.class),
    /**
     * Checks if the field can safely be converted to a long primitive.
     *
     * @see ValidateLong
     */
    LongCheck(ValidateLong.class),
    /**
     * Checks if the field's length is less than or equal to the maximum value. A <code>Null</code> will be considered
     * an error.
     *
     * @see ValidateMaxLength
     */
    MaxLengthCheck(ValidateMaxLength.class),
    /**
     * Checks if the field's length is greater than or equal to the minimum value. A <code>Null</code> will be
     * considered an error.
     *
     * @see ValidateMinLength
     */
    MinLengthCheck(ValidateMinLength.class),
    /**
     * Checks if the field isn't null and length of the field is greater than zero not including whitespace.
     *
     * @see ValidateRequired
     */
    RequiredCheck(ValidateRequired.class),
    /**
     * Checks if the field isn't null based on the values of other fields.
     *
     * @see ValidateRequiredIf
     */
    RequiredIfCheck(ValidateRequiredIf.class),
    /**
     * Checks if the field can safely be converted to a short primitive.
     *
     * @see ValidateShort
     */
    ShortCheck(ValidateShort.class),
    /**
     * Checks if a field has a valid url. Four optional variables can be specified to configure url validation.
     *
     * @see ValidateUrl
     */
    UrlCheck(ValidateUrl.class);

    /**
     * バリデーションインスタンス.
     */
    private IValidator clz;

    /**
     * コンストラクタ.
     *
     * @param clz
     *            バリデーションクラス.
     */
    @SuppressWarnings("hiding")
    private CheckRulesEnum(final Class<? extends IValidator> clz) {
        try {
            this.clz = clz.newInstance();
        } catch (InstantiationException e) {
            // do nothing
        } catch (IllegalAccessException e) {
            // do nothing
        }
    }

    /**
     * インスタンス化されているオブジェクトを返却します.
     *
     * @return インスタンス化されているオブジェクト.
     */
    public IValidator getInstance() {
        return this.clz;
    }

    /**
     * インスタンス化されているオブジェクトのクラス名を返却します.
     *
     * @return インスタンス化されているオブジェクトのクラス名.
     */
    public String getName() {
        return this.clz.getClass().getSimpleName();
    }
}
