#ifndef __OGMSTREAMS_H__
#define __OGMSTREAMS_H__

#include "../ogg/ogg.h"

/* 
 * Taken from http://tobias.everwicked.com/packfmt.htm
 *
 
 First packet (header)
 ---------------------
 
 pos    | content                 | description
 -------+-------------------------+----------------------------------
 0x0000 | 0x01                    | indicates 'header packet'
 -------+-------------------------+----------------------------------
 0x0001 | stream_header           | the size is indicated in the
        |                         | size member


 Second packet (comment)
 -----------------------
 
 pos    | content                 | description
 -------+-------------------------+----------------------------------
 0x0000 | 0x03                    | indicates 'comment packet'
 -------+-------------------------+----------------------------------
 0x0001 | data                    | see vorbis doc on www.xiph.org

 Data packets
 ------------
 
 pos      | content                 | description
 ---------+-------------------------+----------------------------------
 0x0000   | Bit0  0                 | indicates data packet
          | Bit1  Bit 2 of lenbytes |
          | Bit2  unused            |
          | Bit3  keyframe          |
          | Bit4  unused            |
          | Bit5  unused            |
          | Bit6  Bit 0 of lenbytes |
          | Bit7  Bit 1 of lenbytes |
 ---------+-------------------------+----------------------------------
 0x0001   | LowByte                 | Length of this packet in samples
          | ...            lenbytes | (frames for video, samples for
          | HighByte                | audio, 1ms units for text)
 ---------+-------------------------+----------------------------------
 0x0001+  | data                    | packet contents
 lenbytes |                         |

 *
 *
 */

//// OggDS headers
// Header for the new header format
typedef struct stream_header_video {
  ogg_int32_t  width;
  ogg_int32_t  height;
} stream_header_video;
  
typedef struct stream_header_audio {
  ogg_int16_t  channels;
  ogg_int16_t  blockalign;
  ogg_int32_t  avgbytespersec;
} stream_header_audio;

typedef struct stream_header {
  char        streamtype[8];
  char        subtype[4];

  ogg_int32_t size;             // size of the structure

  ogg_int64_t time_unit;        // in reference time
  ogg_int64_t samples_per_unit;
  ogg_int32_t default_len;      // in media time

  ogg_int32_t buffersize;
  ogg_int16_t bits_per_sample;

  union
  {
    // Video specific
    stream_header_video  video;
    // Audio specific
    stream_header_audio  audio;
  } sh;
} stream_header;

/*
 * Some infos :
 * size       = size of the structure  (sizeof(stream_header))
 * blockalign = number of bytes for 1 sample including all channels
 * This mean we should have :
 * nblockalign = bits_per_sample * channels / 8;
 * bits_per_sample = avgbytespersec * 8 / sample_rate;
 *
 * What I think of the rest :
 * - time_unit is the unit time expressed in a 100ns time reference
 *   (100ns is the reference time for Windows' DirectShow)
 *   So the real time unit is time_unit / 10000000
 * - samples_per_unit is the number of samples per unit of time
 *   So the number of samples per second is :
 *   samples_per_second = samples_per_unit / (time_unit / 10000000)
 *         samples_per_second      10000000
 *   Or : -------------------- = -----------
 *          samples_per_unit      time_unit
 * ATTENTION : time_unit and samples_per_second are the one corresponding
 * to the decoded stream, not the container (i.e. even if an AC3 Frame
 * contains 1536 samples, and a Frame lasts 1/30th of a second, if the
 * audio stream is 48kHz, then samples_per_second is 48000, and not 30*1536)
 *
 * Application :
 * - for a video stream there is only 1 sample per unit (a packet represents
 *  a unit here), so samples_per_second = fps = 10000000 / time_unit
 * - for an audio stream the unit is 1s, so time_unit = 10000000 and
 *  samples_per_unit = samples_per_second
 * - for a text stream, the unit is 1ms, so time_unit = 10000. As one sample
 *  lasts 1ms, samples_per_second = 1000 => samples_per_unit = 1
 */


/// Some defines
#define PACKET_TYPE_HEADER       0x01
#define PACKET_TYPE_COMMENT      0x03
#define PACKET_TYPE_DATA         0x00
#define PACKET_TYPE_BITS         0x07
#define PACKET_LEN_BITS01        0xc0
#define PACKET_LEN_BITS2         0x02
#define PACKET_IS_SYNCPOINT      0x08

#endif /* __OGMSTREAMS_H__ */
