//	VirtualDubMod - Video processing and capture application based on VirtualDub by Avery Lee
//	Copyright (C) 1998-2002 VirtualDubMod Developers
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

//  Adapted from a sample by Todd Jeffreys

//#include "stdafx.h"
#include <windows.h>
#include <tchar.h>
#include "..\resource.h"
#include "openglwnd.h"
#include "credits_mod.h"
#include "starwars.h"
#include <stdlib.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include "credits_mod.h"

extern HINSTANCE g_hInst;

// Definitions for Multisampling
typedef const char * (WINAPI * PFNWGLGETEXTENSIONSSTRINGARBPROC) (HDC hdc);
typedef BOOL (WINAPI * PFNWGLCHOOSEPIXELFORMATARBPROC) (HDC hdc, const int *piAttribIList, const FLOAT *pfAttribFList, UINT nMaxFormats, int *piFormats, UINT *nNumFormats);

#define WGL_NUMBER_PIXEL_FORMATS_ARB   0x2000
#define WGL_DRAW_TO_WINDOW_ARB         0x2001
#define WGL_DRAW_TO_BITMAP_ARB         0x2002
#define WGL_ACCELERATION_ARB           0x2003
#define WGL_NEED_PALETTE_ARB           0x2004
#define WGL_NEED_SYSTEM_PALETTE_ARB    0x2005
#define WGL_SWAP_LAYER_BUFFERS_ARB     0x2006
#define WGL_SWAP_METHOD_ARB            0x2007
#define WGL_NUMBER_OVERLAYS_ARB        0x2008
#define WGL_NUMBER_UNDERLAYS_ARB       0x2009
#define WGL_TRANSPARENT_ARB            0x200A
#define WGL_TRANSPARENT_RED_VALUE_ARB  0x2037
#define WGL_TRANSPARENT_GREEN_VALUE_ARB 0x2038
#define WGL_TRANSPARENT_BLUE_VALUE_ARB 0x2039
#define WGL_TRANSPARENT_ALPHA_VALUE_ARB 0x203A
#define WGL_TRANSPARENT_INDEX_VALUE_ARB 0x203B
#define WGL_SHARE_DEPTH_ARB            0x200C
#define WGL_SHARE_STENCIL_ARB          0x200D
#define WGL_SHARE_ACCUM_ARB            0x200E
#define WGL_SUPPORT_GDI_ARB            0x200F
#define WGL_SUPPORT_OPENGL_ARB         0x2010
#define WGL_DOUBLE_BUFFER_ARB          0x2011
#define WGL_STEREO_ARB                 0x2012
#define WGL_PIXEL_TYPE_ARB             0x2013
#define WGL_COLOR_BITS_ARB             0x2014
#define WGL_RED_BITS_ARB               0x2015
#define WGL_RED_SHIFT_ARB              0x2016
#define WGL_GREEN_BITS_ARB             0x2017
#define WGL_GREEN_SHIFT_ARB            0x2018
#define WGL_BLUE_BITS_ARB              0x2019
#define WGL_BLUE_SHIFT_ARB             0x201A
#define WGL_ALPHA_BITS_ARB             0x201B
#define WGL_ALPHA_SHIFT_ARB            0x201C
#define WGL_ACCUM_BITS_ARB             0x201D
#define WGL_ACCUM_RED_BITS_ARB         0x201E
#define WGL_ACCUM_GREEN_BITS_ARB       0x201F
#define WGL_ACCUM_BLUE_BITS_ARB        0x2020
#define WGL_ACCUM_ALPHA_BITS_ARB       0x2021
#define WGL_DEPTH_BITS_ARB             0x2022
#define WGL_STENCIL_BITS_ARB           0x2023
#define WGL_AUX_BUFFERS_ARB            0x2024
#define WGL_NO_ACCELERATION_ARB        0x2025
#define WGL_GENERIC_ACCELERATION_ARB   0x2026
#define WGL_FULL_ACCELERATION_ARB      0x2027
#define WGL_SWAP_EXCHANGE_ARB          0x2028
#define WGL_SWAP_COPY_ARB              0x2029
#define WGL_SWAP_UNDEFINED_ARB         0x202A
#define WGL_TYPE_RGBA_ARB              0x202B
#define WGL_TYPE_COLORINDEX_ARB        0x202C

#define GL_MULTISAMPLE_ARB                0x809D
#define GL_SAMPLE_ALPHA_TO_COVERAGE_ARB   0x809E
#define GL_SAMPLE_ALPHA_TO_ONE_ARB        0x809F
#define GL_SAMPLE_COVERAGE_ARB            0x80A0
#define GL_SAMPLE_BUFFERS_ARB             0x80A8
#define GL_SAMPLES_ARB                    0x80A9
#define GL_SAMPLE_COVERAGE_VALUE_ARB      0x80AA
#define GL_SAMPLE_COVERAGE_INVERT_ARB     0x80AB
#define GL_MULTISAMPLE_BIT_ARB            0x20000000

/* NV_multisample_filter_hint */
#define GL_MULTISAMPLE_FILTER_HINT_NV     0x8534

#define WGL_SAMPLE_BUFFERS_ARB		 0x2041
#define WGL_SAMPLES_ARB			     0x2042

#define M_PI			3.14152


/* Basic object.  
Fill in the angle, color, starting coordinates and slope, and the StarWarsController
will update the current position for you.  Override draw to do the work.  You will
be translated, so 0,0,0 will be your origin
*/
CObject::CObject() 
{
  m_start.x = 0.0;
  m_start.y = -4.0f;
  m_start.z = 0.0;
  m_slope.x = 0.0;
  m_slope.y = 0.5f;
  m_slope.z = -0.5f;

  // based on their slopes, we calculate the angle that the words move along
  m_fAngle = atanf(m_slope.z/m_slope.y)*180.0f/float(M_PI);
}

/* TexturedQuad.  Supply the .RAW file and the dimensions of it, and it will display
a polygon with the texture on it */
class CTexturedQuad : public CObject
{
private:
  GLuint m_texId;
  float m_endX;
  float m_endY;
public:
  // id is the resource id of 'RAW' (32 bit + alpha channel) type.  
  // X and Y are the dimensions of it.  startX is where to start, endX is where to end
  // endY is the height of the poly.
  CTexturedQuad(UINT id,UINT x,UINT y,float startX,float endX,float endY,float timeOffset) : CObject()
  {
	m_start.x = startX;
	m_fColor[0] = 1.0;
	m_fColor[1] = 1.0;
	m_fColor[2] = 1.0;

	m_endX = endX;
	m_endY = endY;
	m_fTimeOffset = timeOffset;

	// ok now load the texture
	// load the data now
	HRSRC hrsrc = FindResource(g_hInst,(LPCTSTR)id,_T("STUFF"));
	HGLOBAL hGlob = LoadResource(g_hInst,hrsrc);
	LPVOID pData = LockResource(hGlob);
	if (!pData)
	  throw CError();

	glGenTextures(1,&m_texId);
	glEnable(GL_TEXTURE_2D);
	glBindTexture ( GL_TEXTURE_2D, m_texId );
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
	glTexEnvi ( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE ); 
	gluBuild2DMipmaps ( GL_TEXTURE_2D, 4, x, y, GL_RGBA, GL_UNSIGNED_BYTE, pData);
	glDisable(GL_TEXTURE_2D);
  }
  virtual ~CTexturedQuad()
  {
	glDeleteTextures(1,&m_texId);
  }
  virtual void Draw()
  {
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D,m_texId);
	// draw the quad now
	glBegin(GL_QUADS);

	  glTexCoord2f(0.0,0.0);
	  glVertex3f(0.0,0.0,0.0);	  

	  glTexCoord2f(1.0,0.0);
	  glVertex3f(m_endX,0.0,0.0);

	  glTexCoord2f(1.0,1.0);
	  glVertex3f(m_endX,-m_endY,0.0);

	  glTexCoord2f(0.0,1.0);
	  glVertex3f(0.0,-m_endY,0.0);

	glEnd();

	glDisable(GL_TEXTURE_2D);
  }
};
/* This is a simple text object, for displaying text */
class CTextObject : public CObject
{
private:
  // Calculates the length of a string
  float CalculateXLength(LPGLYPHMETRICSFLOAT agmf,const char *str)
  {	
	int len = strlen(str);
	float length = 0.0;
	for (int loop=0;loop<len;loop++)	// Loop To Find Text Length
	{
	  length+=agmf[str[loop]].gmfCellIncX;		// Increase Length By Each Characters Width
	}
	return length;
  }
public:
  // Simple creation.  Supply text, and the colors and time offset
  CTextObject(LPGLYPHMETRICSFLOAT agmf,const char *pText,
	float colorR,float colorG,float colorB,float timeOffset) : CObject()
  {
	m_pText = pText;
	m_lenText = strlen(pText);

	m_fColor[0] = colorR;
	m_fColor[1] = colorG;
	m_fColor[2] = colorB;

	m_fTimeOffset = timeOffset;

	m_start.x = -CalculateXLength(agmf,m_pText)/2;
  }
  virtual ~CTextObject()
  {
  }
  virtual void Draw()
  {
	glListBase(1000); // Indicates the m_start of display lists for the glyphs 
	glCallLists(m_lenText, GL_UNSIGNED_BYTE, m_pText);
  }  
private:
  const char *m_pText;
  int m_lenText;
};

/* The star wars controller. It shows stars and scrolling CObject's.  The CObjects
are created in Start().  You can control the star speed,number of stars, and if
the stars are points or polygons via the constructor.  The CObjects are hard-coded.
It wouldn't be hard to allow dynamic changes.  
The Idle function moves everything.  Draw displays everything
*/
CStarWarsController::CStarWarsController(float fStarSpeed,int iNumStars,bool bPointStars)
{
  m_bStarted = false;
  m_bMultiSample = false;
  m_fStarSpeed = fStarSpeed;
  m_iNumStars = iNumStars;
  m_bPointStars = bPointStars;
  m_pStars = new STAR[iNumStars];
  if (!m_pStars)
	throw CError();

/*  for (int i=0;i<NUMOBJECTS;++i)
	pObjects[i] = NULL;*/

  QueryPerformanceFrequency(&m_freq);
}
// cleanup
CStarWarsController::~CStarWarsController()
{
  if (m_pStars)
	delete [] m_pStars;

  // Ok, the following code should be in to get rid of the objects, but then
  // it fubars on the second call on Win98. Don't ask me why.
/*  vector<CObject *>::iterator walkit;
  for(walkit = pObjects.begin();walkit != pObjects.end(); walkit++)
	if ((*walkit)) {
		delete *walkit;
		*walkit = NULL;
	}
  pObjects.clear();
*/
}
// determines if a GL extension exists.  First it uses wglGetExtensionsStringARB.
// I chose to do that first because my Geforce 4 required that call in order to 
// initialize extensions.  
bool CStarWarsController::GLExtensionExists(const char *extension)
{
  PFNWGLGETEXTENSIONSSTRINGARBPROC wglGetExtensionsStringARB = 0;
  wglGetExtensionsStringARB = (PFNWGLGETEXTENSIONSSTRINGARBPROC)wglGetProcAddress("wglGetExtensionsStringARB");
  if(wglGetExtensionsStringARB)
  {
	const char *winsys_extensions = wglGetExtensionsStringARB(wglGetCurrentDC());
	if (winsys_extensions && (strstr(winsys_extensions,extension)))
	  return true;
  }
  const char *glExtensions = (const char*)glGetString(GL_EXTENSIONS);
  if (!glExtensions)
	return false;

  return (strstr(glExtensions,extension) != NULL);
}
// Returns a random number between min and max
float CStarWarsController::GetRandom(float min,float max)
{
  float random = float(rand())/float(RAND_MAX);
  return (min + ((max-min)*random));
} 
void CStarWarsController::InitStars()
{
  srand(GetTickCount());

  for (int i=0;i<m_iNumStars;++i)
  {
	m_pStars[i].m_start.x = GetRandom(-5.0,5.0);
	m_pStars[i].m_start.y = GetRandom(-5.0,5.0);
	m_pStars[i].m_start.z = GetRandom(-10.0,1.0);

	m_pStars[i].speed.x = 0.0;
	m_pStars[i].speed.y = 0.0;
	m_pStars[i].speed.z = m_fStarSpeed;

	// 80% of the stars are white.  The rest are yellowish - redish
#define PERCENTAGE_WHITE 0.8f
	m_pStars[i].m_fColor[0] = 1.0;
	if (GetRandom(0.0,1.0) > PERCENTAGE_WHITE)
	{
	  // make it a colorful star
	  m_pStars[i].m_fColor[1] = GetRandom(0.5,1.0);
	  m_pStars[i].m_fColor[2] = 0.0;
	}
	else
	{
	  // make it white
	  m_pStars[i].m_fColor[1] = 1.0;
	  m_pStars[i].m_fColor[2] = 1.0;
	}

	m_pStars[i].timeOffset = 0.0f;
  }
}
/* First function called before we enter the rendering loop, determine the starting time  */
void CStarWarsController::Start()
{
//	AObject *pObject;  
	CObject *pObject;

	QueryPerformanceCounter(&m_start);

	if (m_bStarted) return;

	if (!m_bStarted) // only initialize the stars the first time through
	{
		m_starStart = m_start;
		InitStars();
	}

	m_bStarted = true;

	// create objects now (only if they don't exist)
	pObject = new CTexturedQuad(IDR_LOGO,274,148,-2.5,5,3,0.0f);
	if (!pObject)
	  throw CError();
	pObjects.push_back(pObject);

	float time = 5.0f;

	for(int i=0; modCredits[i].credit != NULL; i++) {
		if (modCredits[i].credit != "") {
			time+=1.2f;
			pObject = new CTextObject(m_agmf,modCredits[i].credit,modCredits[i].color.r,modCredits[i].color.g,modCredits[i].color.b,time);
			if (!pObject)
				throw CError();
			pObjects.push_back(pObject);
		} else
			time+=0.4f;
	}

	// Easter Eggs =D

	SYSTEMTIME st;
	GetLocalTime(&st);

	if ((st.wMonth==12) && (st.wDay>=24) && (st.wDay<=26)) {

		time+=7;

		pObject = new CTexturedQuad(IDR_BAND,476,100,-5,10,1.5,time);
		if (!pObject)
			throw CError();
		pObjects.push_back(pObject);
		pObject = new CTextObject(m_agmf,"Merry X-mas everybody!",1.0,0.0,0.0,time+3);
		if (!pObject)
			throw CError();
		pObjects.push_back(pObject);

	}

	if ((st.wMonth==1) && (st.wDay>=1) && (st.wDay<=2)) {

		time+=7;

		pObject = new CTexturedQuad(IDR_NEW,250,356,-2,4,5.0,time);
		if (!pObject)
			throw CError();
		pObjects.push_back(pObject);
		pObject = new CTextObject(m_agmf,"Happy New Year!",1.0,1.0,0.0,time+9);
		if (!pObject)
			throw CError();
		pObjects.push_back(pObject);

	}

	/*	pObject = new AObject(new CTextObject(m_agmf,"OpenGL/FSAA/StarWars Test",1.0f,0.0f,0.0f,5.0f));
	if (!pObject->get())
	  throw CError();
	pObjects.push_back(*pObject);

  if (!pObjects[0])
  {
	pObjects[0] = new CTextObject(m_agmf,"OpenGL/FSAA/StarWars Test",1.0f,0.0f,0.0f,0.0f);
	if (!pObjects[0])
	  throw CError();
  }
  if (!pObjects[1])
  {
	pObjects[1] = new CTextObject(m_agmf,"Written by",1.0f,1.0f,1.0f,2.0f);
	if (!pObjects[1])
	  throw CError();
  }
  if (!pObjects[2])
  {
	pObjects[2] = new CTextObject(m_agmf,"Todd Jeffreys",0.0f,0.0f,1.0f,4.0f);
	if (!pObjects[2])
	  throw CError();
  }

  if (!pObjects[3])
  {
	pObjects[3] = new CTextObject(m_agmf,"paladine@grnet.com",1.0f,1.0f,1.0f,7.0f);
	if (!pObjects[3])
	  throw CError();
  }
*/
}
/* Idle function.  Moves the text/stars using the current time */
void CStarWarsController::Idle()
{
  LARGE_INTEGER now;

  // get current time
  QueryPerformanceCounter(&now);

  m_fTimeElapsed = ((float)(now.QuadPart - m_start.QuadPart)/(float)m_freq.QuadPart);

  // move the objects
  vector<CObject *>::iterator walkit;
  for (walkit = pObjects.begin();walkit != pObjects.end();walkit++)
  {
	(*walkit)->m_curPos.x = (*walkit)->m_start.x;
	(*walkit)->m_curPos.y = (*walkit)->m_start.y + (*walkit)->m_slope.y * (m_fTimeElapsed - (*walkit)->m_fTimeOffset);
	(*walkit)->m_curPos.z = (*walkit)->m_start.z + (*walkit)->m_slope.z * (m_fTimeElapsed - (*walkit)->m_fTimeOffset);
  }
  // move the stars, calculate new time based on star m_start time
  m_fTimeElapsed = ((float)(now.QuadPart - m_starStart.QuadPart)/(float)m_freq.QuadPart);
  for (int i=0;i<m_iNumStars;++i)
  {
	// update their z position
	m_pStars[i].m_curPos[2] = m_pStars[i].m_start.z + m_pStars[i].speed.z * (m_fTimeElapsed - m_pStars[i].timeOffset);
	// ok they're out of view, respawn a new star
	if (m_pStars[i].m_curPos[2] >= EYE_Z)
	{
	  m_pStars[i].m_start.x = GetRandom(-5.0,5.0);
	  m_pStars[i].m_start.y = GetRandom(-5.0,5.0);
	  m_pStars[i].m_start.z = -10.0f;
	  m_pStars[i].timeOffset = m_fTimeElapsed;
	}
	else
	{
	  m_pStars[i].m_curPos[0] = m_pStars[i].m_start.x;
	  m_pStars[i].m_curPos[1] = m_pStars[i].m_start.y;
	}
  }
}
// Overridden to enable multisampling (FSAA)
int  CStarWarsController::ValidatePixelFormat(HDC hdc,int suggestedFormat)
{
  HDC hDC = wglGetCurrentDC();

  PFNWGLCHOOSEPIXELFORMATARBPROC wglChoosePixelFormatARB = (PFNWGLCHOOSEPIXELFORMATARBPROC)wglGetProcAddress("wglChoosePixelFormatARB");	
  if (!wglChoosePixelFormatARB)
	return suggestedFormat;

  if (!GLExtensionExists("WGL_ARB_multisample "))
	return suggestedFormat;

  int pixelFormat;
  BOOL bStatus;
  UINT numFormats;
  float fAttributes[] = {0,0};
  int iAttributes[] = { WGL_DRAW_TO_WINDOW_ARB,GL_TRUE,
	WGL_SUPPORT_OPENGL_ARB,GL_TRUE,
	WGL_ACCELERATION_ARB,WGL_FULL_ACCELERATION_ARB,
	WGL_COLOR_BITS_ARB,24,
	WGL_ALPHA_BITS_ARB,8,
	WGL_DEPTH_BITS_ARB,16,
	WGL_STENCIL_BITS_ARB,0,
	WGL_DOUBLE_BUFFER_ARB,GL_TRUE,
	WGL_SAMPLE_BUFFERS_ARB,GL_TRUE,
	WGL_SAMPLES_ARB,4,
	0,0};
  bStatus = wglChoosePixelFormatARB(hDC,iAttributes,fAttributes,1,&pixelFormat,&numFormats);
  if ((bStatus == GL_TRUE) && (numFormats == 1))
  {
	m_bMultiSample = true;
	return pixelFormat;	  
  }
  // ok that failed, try using 2 samples now instead of 4
  iAttributes[19] = 2;
  bStatus = wglChoosePixelFormatARB(hDC,iAttributes,fAttributes,1,&pixelFormat,&numFormats);
  if ((bStatus == GL_TRUE) && (numFormats == 1))
  {
	m_bMultiSample = true;
	return pixelFormat;	  
  }
  // failed, return the suggested format and continue
  return suggestedFormat;
}
/* Called when the window is sized, setup the viewport */
void CStarWarsController::WindowSized(int cx,int cy)
{
  glMatrixMode (GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(60.0,(float)cx/(float)cy,1.0,90.0);

  glViewport (0, 0, cx, cy);
}
/* Method to actually draw on the control */
void CStarWarsController::vDraw()
{
  glClearColor(0.0,0.0,0.0,0.0);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  if (!m_bStarted)
	return;

  glHint(GL_MULTISAMPLE_FILTER_HINT_NV,GL_NICEST);
  glEnable(GL_MULTISAMPLE_ARB);
  glDisable(GL_BLEND);
  glCullFace(GL_BACK); 
  glEnable(GL_CULL_FACE); 

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  gluLookAt(0.0,0.0,EYE_Z,0.0,0.0,0.0,0.0,1.0,0.0);

  // now draw stars - as points
  if (m_bPointStars)
  {
	glBegin(GL_POINTS);
	for (int i=0;i<m_iNumStars;++i)
	{
	  glColor3fv(m_pStars[i].m_fColor);
	  glVertex3fv(m_pStars[i].m_curPos);
	}
	glEnd();
  }
  else // draw stars as quads
  {
	glBegin(GL_QUADS);
	for (int i=0;i<m_iNumStars;++i)
	{
#define LENGTH 0.02f
	  glColor3fv(m_pStars[i].m_fColor);
	  glVertex3f(m_pStars[i].m_curPos[0]-LENGTH,m_pStars[i].m_curPos[1]-LENGTH,m_pStars[i].m_curPos[2]);
	  glVertex3f(m_pStars[i].m_curPos[0]-LENGTH,m_pStars[i].m_curPos[1]+LENGTH,m_pStars[i].m_curPos[2]);
	  glVertex3f(m_pStars[i].m_curPos[0]+LENGTH,m_pStars[i].m_curPos[1]+LENGTH,m_pStars[i].m_curPos[2]);
	  glVertex3f(m_pStars[i].m_curPos[0]+LENGTH,m_pStars[i].m_curPos[1]-LENGTH,m_pStars[i].m_curPos[2]);
	}
	glEnd();
  }
  // now draw text

  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);

  float distance,alpha;
  bool onegotpainted=false;

  vector<CObject *>::iterator walkit;
  for (walkit = pObjects.begin();walkit != pObjects.end();walkit++)
  {
	if (!(*walkit))
	  continue;
	// determine distance from us
	distance = sqrtf((*walkit)->m_curPos.x*(*walkit)->m_curPos.x +
	  (*walkit)->m_curPos.y*(*walkit)->m_curPos.y + 
	  (*walkit)->m_curPos.z*(*walkit)->m_curPos.z);
	if ((*walkit)->m_curPos.z>5)
	  continue;

	// approximate the alpha value based on the distance away from us
//	alpha = 3.75f - sqrtf(distance);
	alpha = 3.45f - sqrtf(distance);
	if (alpha > 1.0f)
	  alpha = 1.0f;
	else if (alpha < 0.0)
	  alpha = 0.0;

	if (alpha == 0)
	  continue;

	glPushMatrix();

	// move everything into position
	glScalef(0.50f,0.50f,0.50f);
	glTranslatef((*walkit)->m_curPos.x,(*walkit)->m_curPos.y,(*walkit)->m_curPos.z);
	glRotatef((*walkit)->m_fAngle,1.0,0.0,0.0);
	glColor4f((*walkit)->m_fColor[0],(*walkit)->m_fColor[1],(*walkit)->m_fColor[2],alpha);

	(*walkit)->Draw();

	glPopMatrix();

	onegotpainted = true;
  }
  // ok now we check the last alpha value, if it's <= 0.0, everything has faded away, and we restart
  if (!onegotpainted)
	Start();
}
/* Init function, create our text */
void CStarWarsController::Init()
{
  HFONT hOld;
  HFONT hFont = CreateFont(12, 0, 0, 0, FW_NORMAL,
	FALSE, FALSE, 0, ANSI_CHARSET,
	OUT_DEFAULT_PRECIS, 
	CLIP_DEFAULT_PRECIS,
	DEFAULT_QUALITY, 
	DEFAULT_PITCH, _T("Arial"));

  HDC hdc = wglGetCurrentDC();
  hOld = (HFONT)SelectObject(hdc,hFont);

  wglUseFontOutlines(hdc, 0, MAX_TEXT, 1000, 0.0f, 0.1f,WGL_FONT_POLYGONS, m_agmf);

  SelectObject(hdc,hOld);
  DeleteObject(hFont);
}
/* Destroy our text */
void CStarWarsController::Close()
{
  glDeleteLists(1000,MAX_TEXT);
  // delete our objects now
/*  vector<CObject *>::iterator walkit;
  for(walkit = pObjects.begin();walkit != pObjects.end(); walkit++)
	if ((*walkit)) {
		delete *walkit;
		*walkit = NULL;
	}
  pObjects.clear();*/
}	
