//	VirtualDub - Video processing and capture application
//	Copyright (C) 1998-2001 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <windows.h>
#include <commctrl.h>
#include <crtdbg.h>
#include <assert.h>
#include <stdio.h>
#include <math.h>

#include "ScriptInterpreter.h"
#include "ScriptValue.h"
#include "ScriptError.h"

#include "misc.h"
#include "cpuaccel.h"
#include "resource.h"
#include "gui.h"
#include "filter.h"
#include "resample.h"
#include "vbitmap.h"

extern HINSTANCE g_hInst;

#define round(x)	( (x)-floor(x)>=ceil(x)-(x) ? ceil(x) : floor(x) )

///////////////////////

enum {
	FILTER_NONE				= 0,
	FILTER_BILINEAR			= 1,
	FILTER_BICUBIC			= 2,
	FILTER_TABLEBILINEAR	= 3,
	FILTER_TABLEBICUBIC075	= 4,
	FILTER_TABLEBICUBIC060	= 5,
	FILTER_TABLEBICUBIC100	= 6,
	FILTER_LANZCOS3			= 7
};

static char *filter_names[]={
	"Nearest neighbor",
	"Bilinear",
	"Bicubic",
	"Precise bilinear",
	"Precise bicubic (A=-0.75)",
	"Precise bicubic (A=-0.60)",
	"Precise bicubic (A=-1.00)",
	"Lanczos3"
};

typedef struct MyFilterData {
	long new_x, new_y, new_xf, new_yf;
	int filter_mode;
	COLORREF	rgbColor;

	HBRUSH		hbrColor;
	IFilterPreview *ifp;

	Resampler *resampler;

	long in_x, in_y;
	long out_x, out_y;
	long out_xf, out_yf;
	long ratio_x, ratio_y;
	long round_x, round_y;
	long width_percent, height_percent;
	long idLastChanged, lastValue;
	long focus;

	bool	fLetterbox;
	bool	fInterlaced;
	bool  fPixel;
	bool  fMaintainRatio; //maintain ratio for the input image
	bool  fMaintainImageRatio; //maintain ratio for the letterbox too
	bool  fRoundDim;
} MyFilterData;

////////////////////

int revcolor(int c) {
	return ((c>>16)&0xff) | (c&0xff00) | ((c&0xff)<<16);
}

////////////////////

static int resize_run(const FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	unsigned long x_inc, y_inc;
	long dstw = mfd->new_x;
	long dsth = mfd->new_y;
	long h_margin=0, w_margin=0, h_margin_half=0;
	Pixel *dst, *src;

	dst = fa->dst.data;
	src = fa->src.data;

	// Draw letterbox bound

	if (mfd->fLetterbox) {
		Pixel *dst2 = dst, *dst3;
		Pixel fill = revcolor(mfd->rgbColor);

		long w1, w2, w, h;

		if (mfd->fInterlaced)
			dsth = (dsth+1)&~1;

		w_margin = fa->dst.w - dstw;
		h_margin = fa->dst.h - dsth;

		if (w_margin < 0)
			w_margin = 0;

		if (h_margin < 0)
			h_margin = 0;

		h_margin_half = h_margin/2;

		if (mfd->fInterlaced)
			h_margin_half &= ~1;

		h = h_margin - h_margin_half;
		if (h>0) do {
			dst3  = dst2;
			w = fa->dst.w;
			do {
				*dst3++ = fill;
			} while(--w);

			dst2 = (Pixel32 *)((char *)dst2 + fa->dst.pitch);
		} while(--h);

		w1 = w_margin/2;
		w2 = w_margin - w_margin/2;

		h = dsth;
		do {
			dst3 = dst2;

			// fill left

			w = w1;
			if (w) do {
				*dst3++ = fill;
			} while(--w);

			// skip center

			dst3 += mfd->new_x;

			// fill right

			w = w2;
			if (w) do {
				*dst3++ = fill;
			} while(--w);


			dst2 = (Pixel32 *)((char *)dst2 + fa->dst.pitch);
		} while(--h);

		h = h_margin_half;
		if (h>0) do {
			dst3  = dst2;
			w = fa->dst.w;
			do {
				*dst3++ = fill;
			} while(--w);

			dst2 = (Pixel32 *)((char *)dst2 + fa->dst.pitch);
		} while(--h);

		// offset resampled rectangle

		dst = (Pixel *)((char *)dst + fa->dst.pitch * (h_margin/2)) + (w_margin/2);
	}

	if (mfd->fInterlaced) {
		VBitmap vbHalfSrc, vbHalfDst;

		vbHalfSrc = fa->src;
		vbHalfSrc.modulo += vbHalfSrc.pitch;
		vbHalfSrc.pitch *= 2;
		vbHalfSrc.data = fa->src.Address32i(0, fa->src.h&1);
		vbHalfSrc.h >>= 1;

		vbHalfDst = fa->dst;
		vbHalfDst.modulo += vbHalfDst.pitch;
		vbHalfDst.pitch *= 2;
		vbHalfDst.h >>= 1;

		double dy = 0.25 * (1.0 - (double)vbHalfSrc.h / (double)vbHalfDst.h);

		mfd->resampler->Process(&vbHalfDst, w_margin/2, h_margin/4, &vbHalfSrc, 0, -dy, false);

		vbHalfSrc.data = fa->src.Address32i(0, 1+(fa->src.h&1));
		vbHalfDst.data = fa->dst.Address32i(0, 1);

		mfd->resampler->Process(&vbHalfDst, w_margin/2, h_margin/4, &vbHalfSrc, 0, +dy, false);
	} else
		mfd->resampler->Process(&fa->dst, w_margin/2, h_margin/2, &fa->src, 0, 0, false);

	return 0;
}

static long resize_param(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->in_x = fa->src.w;
	mfd->in_y = fa->src.h;

	if (mfd->fLetterbox) {
		fa->dst.w		= max(mfd->new_x, mfd->new_xf);
		fa->dst.h		= max(mfd->new_y, mfd->new_yf);
	} else {
		fa->dst.w		= mfd->new_x;
		fa->dst.h		= mfd->new_y;
	}

	if (mfd->fInterlaced)
		fa->dst.h = (fa->dst.h+1)&~1;

	fa->dst.AlignTo8();

	return FILTERPARAM_SWAP_BUFFERS;
}

static void update_params(MyFilterData *mfd, int idValue, long new_value) {
	if(!mfd)
		return;

	mfd->idLastChanged = idValue;
	mfd->lastValue = new_value;

	long new_x = mfd->new_x;
	long new_y = mfd->new_y;
	long new_xf = mfd->new_xf;
	long new_yf = mfd->new_yf;
	long out_x = mfd->out_x;
	long out_y = mfd->out_y;
	long out_xf = mfd->out_xf;
	long out_yf = mfd->out_yf;
	long width_percent = mfd->width_percent;
	long height_percent = mfd->height_percent;

	switch(idValue) {
		case IDC_WIDTH:
		case IDC_WIDTHP:
			if(idValue == IDC_WIDTH)
				out_x = new_value;
			if(idValue == IDC_WIDTHP) {
				width_percent = new_value;
				out_x = mfd->in_x * width_percent / 100;
			}
			if(mfd->fMaintainRatio)
				out_y = round((double)out_x * (double)mfd->ratio_y / (double)mfd->ratio_x);
			if(mfd->fLetterbox && mfd->fMaintainImageRatio)
				out_yf = round((double)max(out_x, out_xf) * (double)mfd->ratio_y / (double)mfd->ratio_x);
			if(mfd->fRoundDim) {
				if(!mfd->fLetterbox) {
					new_x = round((double)out_x / (double)mfd->round_x) * mfd->round_x;
					if(mfd->fMaintainRatio)
						new_y = round((double)new_x * (double)mfd->ratio_y / (double)mfd->ratio_x);
					else
						new_y = out_y;
					new_y = round((double)new_y / (double)mfd->round_y) * mfd->round_y;
				} else {
					new_xf = round((double)out_xf / (double)mfd->round_x) * mfd->round_x;
					if(out_x >= new_xf)
						new_x = new_xf = round((double)out_x / (double)mfd->round_x) * mfd->round_x;
					else
						new_x = out_x;
					if(mfd->fMaintainRatio)
						new_y = round((double)new_x * (double)mfd->ratio_y / (double)mfd->ratio_x);
					else
						new_y = out_y;
					if(mfd->fMaintainImageRatio)
						new_yf = round((double)max(new_x, new_xf) * (double)mfd->ratio_y / (double)mfd->ratio_x);
					else
						new_yf = out_yf;
					new_yf = round((double)new_yf / (double)mfd->round_y) * mfd->round_y;
					if(new_y >= new_yf)
						new_y = new_yf = round((double)new_y / (double)mfd->round_y) * mfd->round_y;
				}
			} else {
				new_x = out_x;
				new_y = out_y;
			}
			if(mfd->fLetterbox) {
				if(new_xf < new_x)
					new_xf = new_x;
				if(new_yf < new_y)
					new_yf = new_y;
				if(out_xf < out_x)
					out_xf = out_x;
				if(out_yf < out_y)
					out_yf = out_y;
			}
			if(idValue != IDC_WIDTHP)
				width_percent = round(100 * (double)out_x / (double)mfd->in_x);
			height_percent = round(100 * (double)out_y / (double)mfd->in_y);
			break;

		case IDC_FRAMEWIDTH:
			out_xf = new_value;
			if(mfd->fMaintainImageRatio)
				out_yf = round((double)out_xf * (double)mfd->ratio_y / (double)mfd->ratio_x);
			if(mfd->fRoundDim) {
				new_xf = round((double)out_xf / (double)mfd->round_x) * mfd->round_x;
				if(mfd->fMaintainImageRatio)
					new_yf = round((double)new_xf * (double)mfd->ratio_y / (double)mfd->ratio_x);
				else
					new_yf = out_yf;
				new_yf = round((double)new_yf / (double)mfd->round_y) * mfd->round_y;
			} else {
				new_xf = out_xf;
				new_yf = out_yf;
			}
			if(new_xf <= 0)
				new_xf = 1;
			if(new_yf <= 0)
				new_yf = 1;
			if(new_x > new_xf)
				new_x = new_xf;
			if(new_y > new_yf)
				new_y = new_yf;
			if(out_xf <= 0)
				out_xf = 1;
			if(out_yf <= 0)
				out_yf = 1;
			if(out_x > out_xf) {
				update_params(mfd, IDC_WIDTH, out_xf);
				return;
			}
			if(out_y > out_yf)
				out_y = out_yf;
			width_percent = round(100 * (double)out_x / (double)mfd->in_x);
			height_percent = round(100 * (double)out_y / (double)mfd->in_y);
			break;

		case IDC_HEIGHTP:
		case IDC_HEIGHT:
			if(idValue == IDC_HEIGHT)
				out_y = new_value;
			if(idValue == IDC_HEIGHTP) {
				height_percent = new_value;
				out_y = mfd->in_y * height_percent / 100;
			}
			if(mfd->fMaintainRatio)
				out_x = round((double)out_y * (double)mfd->ratio_x / (double)mfd->ratio_y);
			if(mfd->fLetterbox && mfd->fMaintainImageRatio)
				out_xf = round((double)max(out_y, out_yf) * (double)mfd->ratio_x / (double)mfd->ratio_y);
			if(mfd->fRoundDim) {
				if(!mfd->fLetterbox) {
					new_y = round((double)out_y / (double)mfd->round_y) * mfd->round_y;
					if(mfd->fMaintainRatio)
						new_x = round((double)new_y * (double)mfd->ratio_x / (double)mfd->ratio_y);
					else
						new_x = out_x;
					new_x = round((double)new_x / (double)mfd->round_x) * mfd->round_x;
				} else {
					new_yf = round((double)out_yf / (double)mfd->round_y) * mfd->round_y;
					if(out_y >= new_yf)
						new_y = new_yf = round((double)out_y / (double)mfd->round_y) * mfd->round_y;
					else
						new_y = out_y;
					if(mfd->fMaintainRatio)
						new_x = round((double)new_y * (double)mfd->ratio_x / (double)mfd->ratio_y);
					else
						new_x = out_x;
					if(mfd->fMaintainImageRatio)
						new_xf = round((double)max(new_y, new_yf) * (double)mfd->ratio_x / (double)mfd->ratio_y);
					else
						new_xf = out_xf;
					if(new_x >= new_xf)
						new_x = new_xf = round((double)new_x / (double)mfd->round_x) * mfd->round_x;
					else
						new_xf = round((double)new_xf / (double)mfd->round_x) * mfd->round_x;
				}
			} else {
				new_x = out_x;
				new_y = out_y;
			}
			if(mfd->fLetterbox) {
				if(new_xf < new_x)
					new_xf = new_x;
				if(new_yf < new_y)
					new_yf = new_y;
				if(out_xf < out_x)
					out_xf = out_x;
				if(out_yf < out_y)
					out_yf = out_y;
			}
			width_percent = round(100 * (double)out_x / (double)mfd->in_x);
			if(idValue != IDC_HEIGHTP)
				height_percent = round(100 * (double)out_y / (double)mfd->in_y);
			break;

		case IDC_FRAMEHEIGHT:
			out_yf = new_value;
			if(mfd->fMaintainImageRatio)
				out_xf = round((double)out_yf * (double)mfd->ratio_x / (double)mfd->ratio_y);
			if(mfd->fRoundDim) {
				new_yf = round((double)out_yf / (double)mfd->round_y) * mfd->round_y;
				if(mfd->fMaintainImageRatio)
					new_xf = round((double)new_yf * (double)mfd->ratio_x / (double)mfd->ratio_y);
				else
					new_xf = out_xf;
				new_xf = round((double)new_xf / (double)mfd->round_x) * mfd->round_x;
			} else {
				new_xf = out_xf;
				new_yf = out_yf;
			}
			if(new_xf <= 0)
				new_xf = 1;
			if(new_yf <= 0)
				new_yf = 1;
			if(new_x > new_xf)
				new_x = new_xf;
			if(new_y > new_yf)
				new_y = new_yf;
			if(out_xf <= 0)
				out_xf = 1;
			if(out_yf <= 0)
				out_yf = 1;
			if(out_x > out_xf)
				out_x = out_xf;
			if(out_y > out_yf) {
				update_params(mfd, IDC_HEIGHT, out_yf);
				return;
			}
			width_percent = round(100 * (double)out_x / (double)mfd->in_x);
			height_percent = round(100 * (double)out_y / (double)mfd->in_y);
			break;
	}

	if(!mfd->fLetterbox) {
		new_xf = new_x;
		new_yf = new_y;
		out_xf = out_x;
		out_yf = out_y;
	}

	mfd->new_x = new_x;
	mfd->new_y = new_y;
	mfd->out_x = out_x;
	mfd->out_y = out_y;
	mfd->new_xf = new_xf;
	mfd->new_yf = new_yf;
	mfd->out_xf = out_xf;
	mfd->out_yf = out_yf;
	mfd->width_percent = width_percent;
	mfd->height_percent = height_percent;
}

static void update_params(MyFilterData *mfd) {
	if(!mfd)
		return;
	update_params(mfd, mfd->idLastChanged, mfd->lastValue);
}

static BOOL APIENTRY resizeDlgProc( HWND hDlg, UINT message, UINT wParam, LONG lParam) {
	MyFilterData *mfd = (struct MyFilterData *)GetWindowLong(hDlg, DWL_USER);

    switch (message)
    {
        case WM_INITDIALOG:
			{
				HWND hwndItem;
				int i;

				mfd = (MyFilterData *)lParam;

				hwndItem = GetDlgItem(hDlg, IDC_FILTER_MODE);

				for(i=0; i<(sizeof filter_names/sizeof filter_names[0]); i++)
					SendMessage(hwndItem, CB_ADDSTRING, 0, (LPARAM)filter_names[i]);

				SendMessage(hwndItem, CB_SETCURSEL, mfd->filter_mode, 0);

				CheckDlgButton(hDlg, IDC_INTERLACED, mfd->fInterlaced ? BST_CHECKED : BST_UNCHECKED);

				if(mfd->fPixel) {
					CheckDlgButton(hDlg, IDC_PIXEL, BST_CHECKED);
					CheckDlgButton(hDlg, IDC_PERCENTAGE, BST_UNCHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_WIDTHP), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_HEIGHTP), FALSE);
				} else {
					CheckDlgButton(hDlg, IDC_PIXEL, BST_UNCHECKED);
					CheckDlgButton(hDlg, IDC_PERCENTAGE, BST_CHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_WIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT), FALSE);
				}

				if (mfd->fLetterbox) {
					CheckDlgButton(hDlg, IDC_LETTERBOX, BST_CHECKED);
				} else {
					CheckDlgButton(hDlg, IDC_LETTERBOX, BST_UNCHECKED);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_STATIC_FILLCOLOR), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_COLOR), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_PICKCOLOR), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDC_KEEPIMAGERATIO), FALSE);
				}

				SendDlgItemMessage(hDlg, IDC_SPIN_PIXELWIDTH, UDM_SETRANGE, 0, MAKELONG(0x7FFF, 1));
				SendDlgItemMessage(hDlg, IDC_SPIN_PIXELHEIGHT, UDM_SETRANGE, 0, MAKELONG(0x7FFF, 1));
				SendDlgItemMessage(hDlg, IDC_SPIN_PERCENTAGEWIDTH, UDM_SETRANGE, 0, MAKELONG(0x7FFF, 1));
				SendDlgItemMessage(hDlg, IDC_SPIN_PERCENTAGEHEIGHT, UDM_SETRANGE, 0, MAKELONG(0x7FFF, 1));
				SendDlgItemMessage(hDlg, IDC_SPIN_CANVASWIDTH, UDM_SETRANGE, 0, MAKELONG(0x7FFF, 1));
				SendDlgItemMessage(hDlg, IDC_SPIN_CANVASHEIGHT, UDM_SETRANGE, 0, MAKELONG(0x7FFF, 1));
				CheckDlgButton(hDlg, IDC_KEEPRATIO, mfd->fMaintainRatio ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_KEEPIMAGERATIO, mfd->fMaintainImageRatio ? BST_CHECKED : BST_UNCHECKED);
				CheckDlgButton(hDlg, IDC_ROUNDING, mfd->fRoundDim ? BST_CHECKED : BST_UNCHECKED);
				SetDlgItemInt(hDlg, IDC_ROUND_WIDTH, mfd->round_x, FALSE);
				SetDlgItemInt(hDlg, IDC_ROUND_HEIGHT, mfd->round_y, FALSE);
				char szError[64];
				memset(szError, 0, sizeof(szError));
				_snprintf(szError, sizeof(szError)-1, "%.15f to 1", (double)mfd->in_x / (double)mfd->in_y);
				SetDlgItemText(hDlg, IDC_STATIC_INPUTRATIO, szError);
				// Let's do some easy maths :p
				int nWidth = mfd->new_x;
				int nHeight = mfd->new_y;
				while(nWidth != nHeight) {
					if(nWidth > nHeight)
						nWidth -= nHeight;
					else
						nHeight -= nWidth;
				}
				mfd->ratio_x = mfd->new_x / nWidth;
				mfd->ratio_y = mfd->new_y / nHeight;
				SetDlgItemInt(hDlg, IDC_RATIO_WIDTH, mfd->ratio_x, FALSE);
				SetDlgItemInt(hDlg, IDC_RATIO_HEIGHT, mfd->ratio_y, FALSE);
				mfd->focus = IDC_WIDTH;

				mfd->hbrColor = CreateSolidBrush(mfd->rgbColor);

				SetWindowLong(hDlg, DWL_USER, (LONG)mfd);

				SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);

				mfd->ifp->InitButton(GetDlgItem(hDlg, IDC_PREVIEW));
			}
            return (TRUE);

			case WM_NOTIFY:
			{
				LPNMHDR pnmh = (LPNMHDR)lParam;
				switch(LOWORD(wParam)) {
				case IDC_SPIN_PIXELWIDTH:
				case IDC_SPIN_CANVASWIDTH:
				{
					if( (pnmh->code == UDN_DELTAPOS) && mfd && mfd->fRoundDim) {
						LPNMUPDOWN lpnmud = (LPNMUPDOWN)lParam;
						long out = lpnmud->iPos + lpnmud->iDelta;
						long round_out = mfd->round_x;
						long rounded = 0;
						if(lpnmud->iDelta < 0)
							rounded = floor((double)out / (double)round_out) * round_out;
						else if(lpnmud->iDelta > 0)
							rounded = ceil((double)out / (double)round_out) * round_out;
						else
							rounded = round((double)out / (double)round_out) * round_out;
						lpnmud->iDelta = rounded - lpnmud->iPos;
					}
				}
				return TRUE;

				case IDC_SPIN_PIXELHEIGHT:
				case IDC_SPIN_CANVASHEIGHT:
				{
					if( (pnmh->code == UDN_DELTAPOS) && mfd && mfd->fRoundDim) {
						LPNMUPDOWN lpnmud = (LPNMUPDOWN)lParam;
						long out = lpnmud->iPos + lpnmud->iDelta;
						long round_out = mfd->round_y;
						long rounded = 0;
						if(lpnmud->iDelta < 0)
							rounded = floor((double)out / (double)round_out) * round_out;
						else if(lpnmud->iDelta > 0)
							rounded = ceil((double)out / (double)round_out) * round_out;
						else
							rounded = round((double)out / (double)round_out) * round_out;
						lpnmud->iDelta = rounded - lpnmud->iPos;
					}
				}
				return TRUE;

				}
			}
			return FALSE;

			case WM_COMMAND:
			if(HIWORD(wParam) == EN_SETFOCUS)
				mfd->focus = LOWORD(wParam);
			switch(LOWORD(wParam)) {
			case IDC_REFRESH_LIST:
				{
					SendDlgItemMessage(hDlg, IDC_SPIN_PIXELWIDTH, UDM_SETPOS, 0, mfd->out_x);
					SendDlgItemMessage(hDlg, IDC_SPIN_PIXELHEIGHT, UDM_SETPOS, 0, mfd->out_y);
					SendDlgItemMessage(hDlg, IDC_SPIN_PERCENTAGEWIDTH, UDM_SETPOS, 0, mfd->width_percent);
					SendDlgItemMessage(hDlg, IDC_SPIN_PERCENTAGEHEIGHT, UDM_SETPOS, 0, mfd->height_percent);
					SendDlgItemMessage(hDlg, IDC_SPIN_CANVASWIDTH, UDM_SETPOS, 0, mfd->out_xf);
					SendDlgItemMessage(hDlg, IDC_SPIN_CANVASHEIGHT, UDM_SETPOS, 0, mfd->out_yf);
					/*BOOL success;
					long val = 0;
					val = GetDlgItemInt(hDlg, IDC_RATIO_WIDTH, &success, FALSE);
					if(!success || val != mfd->ratio_x)
						SetDlgItemInt(hDlg, IDC_RATIO_WIDTH, mfd->ratio_x, FALSE);
					val = GetDlgItemInt(hDlg, IDC_RATIO_HEIGHT, &success, FALSE);
					if(!success || val != mfd->ratio_y)
						SetDlgItemInt(hDlg, IDC_RATIO_HEIGHT, mfd->ratio_y, FALSE);
					val = GetDlgItemInt(hDlg, IDC_ROUND_WIDTH, &success, FALSE);
					if(!success || val != mfd->round_x)
						SetDlgItemInt(hDlg, IDC_ROUND_WIDTH, mfd->round_x, FALSE);
					val = GetDlgItemInt(hDlg, IDC_ROUND_HEIGHT, &success, FALSE);
					if(!success || val != mfd->round_y)
						SetDlgItemInt(hDlg, IDC_ROUND_HEIGHT, mfd->round_y, FALSE);*/
					double currentRatio = 0;
					if(mfd->fLetterbox)
						currentRatio = (double)(max(mfd->new_x, mfd->new_xf))
							/ (double)(max(mfd->new_y, mfd->new_yf));
					else
						currentRatio = (double)mfd->new_x / (double)mfd->new_y;
					char szRatio[128];
					memset(szRatio, 0, sizeof(szRatio));
					_snprintf(szRatio, sizeof(szRatio)-1, "Filter : resize [ %3ldx%3ld -> %3ldx%3ld (image : %3ldx%3ld) ]"
						, mfd->in_x, mfd->in_y
						, max(mfd->new_x, mfd->new_xf), max(mfd->new_y, mfd->new_yf)
						, mfd->new_x, mfd->new_y);
					SetWindowText(hDlg, szRatio);
					_snprintf(szRatio, sizeof(szRatio)-1, "%3ld x %3ld"
						, max(mfd->new_x, mfd->new_xf), max(mfd->new_y, mfd->new_yf));
					SetDlgItemText(hDlg, IDC_STATIC_OUTPUTSIZE, szRatio);
					_snprintf(szRatio, sizeof(szRatio)-1, "%.15f to 1", currentRatio);
					SetDlgItemText(hDlg, IDC_STATIC_RATIO, szRatio);
					double wantedRatio = (double)mfd->ratio_x / (double)mfd->ratio_y;
					double deviationRatio = 0;
					if(mfd->fMaintainRatio)
						deviationRatio = 100 * (currentRatio - wantedRatio) / wantedRatio;
					_snprintf(szRatio, sizeof(szRatio)-1, "% +7.3f%%", deviationRatio);
					SetDlgItemText(hDlg, IDC_STATIC_RATIODEVIATION, szRatio);
				}
				return TRUE;

			case IDOK:
				mfd->ifp->Close();
				if(mfd->new_x < 16) {
					SetFocus((HWND)GetDlgItem(hDlg, mfd->fPixel ? IDC_WIDTH : IDC_WIDTHP));
					MessageBeep(MB_ICONQUESTION);
					return TRUE;
				} else if(mfd->new_y < 16) {
					SetFocus((HWND)GetDlgItem(hDlg, mfd->fPixel ? IDC_HEIGHT : IDC_HEIGHTP));
					MessageBeep(MB_ICONQUESTION);
					return TRUE;
				}
				EndDialog(hDlg, 0);
				return TRUE;

			case IDCANCEL:
				mfd->ifp->Close();
				EndDialog(hDlg, 1);
				return TRUE;

			case IDC_WIDTH:
				if( (HIWORD(wParam) == EN_UPDATE) && mfd && (mfd->focus == LOWORD(wParam)) ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_WIDTH, &success, FALSE);
					if(!success)
						return TRUE;

					mfd->ifp->UndoSystem();
					update_params(mfd, LOWORD(wParam), new_val);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_WIDTHP:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd && (mfd->focus == LOWORD(wParam)) ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_WIDTHP, &success, FALSE);
					if(!success)
						return TRUE;

					mfd->ifp->UndoSystem();
					update_params(mfd, LOWORD(wParam), new_val);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_HEIGHT:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd && (mfd->focus == LOWORD(wParam)) ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_HEIGHT, &success, FALSE);
					if(!success)
						return TRUE;

					mfd->ifp->UndoSystem();
					update_params(mfd, LOWORD(wParam), new_val);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_HEIGHTP:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd && (mfd->focus == LOWORD(wParam)) ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_HEIGHTP, &success, FALSE);
					if(!success)
						return TRUE;

					mfd->ifp->UndoSystem();
					update_params(mfd, LOWORD(wParam), new_val);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_FRAMEWIDTH:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd && (mfd->focus == LOWORD(wParam)) ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_FRAMEWIDTH, &success, FALSE);
					if(!success)
						return TRUE;

					mfd->ifp->UndoSystem();
					update_params(mfd, LOWORD(wParam), new_val);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_FRAMEHEIGHT:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd && (mfd->focus == LOWORD(wParam)) ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_FRAMEHEIGHT, &success, FALSE);
					if(!success)
						return TRUE;

					mfd->ifp->UndoSystem();
					update_params(mfd, LOWORD(wParam), new_val);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_RATIO_WIDTH:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_RATIO_WIDTH, &success, FALSE);
					if(!success)
						return TRUE;
					if(new_val <= 0) {
						new_val = mfd->ratio_x;
						SetDlgItemInt(hDlg, IDC_RATIO_WIDTH, new_val, FALSE);
						return TRUE;
					}

					if(mfd->ratio_x != new_val) {
						mfd->ratio_x = new_val;

						mfd->ifp->UndoSystem();
						update_params(mfd);
						mfd->ifp->RedoSystem();
						SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
					}
				}
				return TRUE;

			case IDC_RATIO_HEIGHT:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_RATIO_HEIGHT, &success, FALSE);
					if(!success)
						return TRUE;
					if(new_val <= 0) {
						new_val = mfd->ratio_y;
						SetDlgItemInt(hDlg, IDC_RATIO_HEIGHT, new_val, FALSE);
						return TRUE;
					}

					if(mfd->ratio_y != new_val) {
						mfd->ratio_y = new_val;

						mfd->ifp->UndoSystem();
						update_params(mfd);
						mfd->ifp->RedoSystem();
						SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
					}
				}
				return TRUE;

			case IDC_ROUND_WIDTH:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_ROUND_WIDTH, &success, FALSE);
					if(!success)
						return TRUE;
					if(new_val <= 0) {
						new_val = mfd->round_x;
						SetDlgItemInt(hDlg, IDC_ROUND_WIDTH, new_val, FALSE);
						return TRUE;
					}

					if(mfd->round_x != new_val) {
						mfd->round_x = new_val;

						mfd->ifp->UndoSystem();
						update_params(mfd);
						mfd->ifp->RedoSystem();
						SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
					}
				}
				return TRUE;

			case IDC_ROUND_HEIGHT:
				if( (HIWORD(wParam) == EN_CHANGE) && mfd ) {
					long new_val;
					BOOL success;

					new_val = GetDlgItemInt(hDlg, IDC_ROUND_HEIGHT, &success, FALSE);
					if(!success)
						return TRUE;
					if(new_val <= 0) {
						new_val = mfd->round_y;
						SetDlgItemInt(hDlg, IDC_ROUND_HEIGHT, new_val, FALSE);
						return TRUE;
					}

					if(mfd->round_y != new_val) {
						mfd->round_y = new_val;

						mfd->ifp->UndoSystem();
						update_params(mfd);
						mfd->ifp->RedoSystem();
						SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
					}
				}
				return TRUE;

			case IDC_PREVIEW:
				mfd->ifp->Toggle(hDlg);
				return TRUE;

			case IDC_FILTER_MODE:
				if (HIWORD(wParam) == CBN_SELCHANGE) {
					mfd->ifp->UndoSystem();
					mfd->filter_mode = SendDlgItemMessage(hDlg, IDC_FILTER_MODE, CB_GETCURSEL, 0, 0);
					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_INTERLACED:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_INTERLACED);

					mfd->ifp->UndoSystem();
					mfd->fInterlaced = !!f;

					mfd->ifp->RedoSystem();
				}
				return TRUE;

			case IDC_LETTERBOX:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_LETTERBOX);

					mfd->ifp->UndoSystem();
					mfd->fLetterbox = !!f;

					EnableWindow(GetDlgItem(hDlg, IDC_STATIC_FILLCOLOR), f);
					EnableWindow(GetDlgItem(hDlg, IDC_COLOR), f);
					EnableWindow(GetDlgItem(hDlg, IDC_PICKCOLOR), f);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEWIDTH), f);
					EnableWindow(GetDlgItem(hDlg, IDC_FRAMEHEIGHT), f);
					EnableWindow(GetDlgItem(hDlg, IDC_KEEPIMAGERATIO), f);

					if (mfd->fLetterbox) {
						if (mfd->new_xf < mfd->new_x)
							mfd->new_xf = mfd->new_x;
						if (mfd->new_yf < mfd->new_y)
							mfd->new_yf = mfd->new_y;
						if (mfd->out_xf < mfd->out_x)
							mfd->out_xf = mfd->out_x;
						if (mfd->out_yf < mfd->out_y)
							mfd->out_yf = mfd->out_y;
					} else {
						mfd->new_xf = mfd->new_x;
						mfd->new_yf = mfd->new_y;
						mfd->out_xf = mfd->out_x;
						mfd->out_yf = mfd->out_y;
					}
					update_params(mfd);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_KEEPRATIO:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_KEEPRATIO);

					mfd->fMaintainRatio = !!f;
					mfd->ifp->UndoSystem();
					update_params(mfd);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_KEEPIMAGERATIO:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_KEEPIMAGERATIO);

					mfd->fMaintainImageRatio = !!f;
					mfd->ifp->UndoSystem();
					update_params(mfd);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_ROUNDING:
				if (HIWORD(wParam) == BN_CLICKED) {
					BOOL f = IsDlgButtonChecked(hDlg, IDC_ROUNDING);

					mfd->fRoundDim = !!f;

					mfd->ifp->UndoSystem();
					update_params(mfd);
					mfd->ifp->RedoSystem();
					SendMessage(hDlg, WM_COMMAND, IDC_REFRESH_LIST, (LPARAM)hDlg);
				}
				return TRUE;

			case IDC_PIXEL:
			case IDC_PERCENTAGE:
				if (HIWORD(wParam) == BN_CLICKED) {
					if( (mfd->fPixel && LOWORD(wParam)==IDC_PIXEL)
					|| (!mfd->fPixel && LOWORD(wParam)==IDC_PERCENTAGE) )
						return TRUE;
					mfd->fPixel = !mfd->fPixel;

					if(mfd->fPixel) {
						CheckDlgButton(hDlg, IDC_PIXEL, BST_CHECKED);
						CheckDlgButton(hDlg, IDC_PERCENTAGE, BST_UNCHECKED);
						EnableWindow(GetDlgItem(hDlg, IDC_WIDTH), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_WIDTHP), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_HEIGHTP), FALSE);
					} else {
						CheckDlgButton(hDlg, IDC_PIXEL, BST_UNCHECKED);
						CheckDlgButton(hDlg, IDC_PERCENTAGE, BST_CHECKED);
						EnableWindow(GetDlgItem(hDlg, IDC_WIDTH), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_HEIGHT), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDC_WIDTHP), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDC_HEIGHTP), TRUE);
					}
				}
				return TRUE;

			case IDC_PICKCOLOR:
				if (guiChooseColor(hDlg, mfd->rgbColor)) {
					DeleteObject(mfd->hbrColor);
					mfd->hbrColor = CreateSolidBrush(mfd->rgbColor);
					RedrawWindow(GetDlgItem(hDlg, IDC_COLOR), NULL, NULL, RDW_ERASE|RDW_INVALIDATE|RDW_UPDATENOW);
				}
				break;
            }
            break;

		case WM_CTLCOLORSTATIC:
			if (GetWindowLong((HWND)lParam, GWL_ID) == IDC_COLOR)
				return (BOOL)mfd->hbrColor;
			break;
    }
    return FALSE;
}

static int resize_config(FilterActivation *fa, const FilterFunctions *ff, HWND hWnd) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	MyFilterData mfd2 = *mfd;
	int ret;

	mfd->hbrColor = NULL;
	mfd->ifp = fa->ifp;

	mfd->fPixel = true;
	mfd->fMaintainRatio = true;
	mfd->fMaintainImageRatio = false;
	mfd->fRoundDim = false;
	mfd->width_percent = 100;
	mfd->height_percent = 100;
	mfd->round_x = 16;
	mfd->round_y = 16;

	mfd->new_x = fa->src.w;
	mfd->new_y = fa->src.h;
	if (mfd->new_x < 16)
		mfd->new_x = 320;
	if (mfd->new_y < 16)
		mfd->new_y = 240;
	mfd->out_x = mfd->out_xf = mfd->new_xf = mfd->new_x;
	mfd->out_y = mfd->out_yf = mfd->new_yf = mfd->new_y;

	ret = DialogBoxParam(g_hInst, MAKEINTRESOURCE(IDD_FILTER_RESIZE), hWnd, resizeDlgProc, (LONG)mfd);

	if (mfd->hbrColor) {
		DeleteObject(mfd->hbrColor);
		mfd->hbrColor = NULL;
	}

	if (ret)
		*mfd = mfd2;

	return ret;
}

static void resize_string(const FilterActivation *fa, const FilterFunctions *ff, char *buf) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	if (mfd->fLetterbox)
		wsprintf(buf, " (%s, lbox %dx%d #%06x)", filter_names[mfd->filter_mode],
				mfd->new_xf, mfd->new_yf, revcolor(mfd->rgbColor));
	else
		wsprintf(buf, " (%s)", filter_names[mfd->filter_mode]);
}

static int resize_start(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	long dstw = mfd->new_x;
	long dsth = mfd->new_y;

	if (dstw<16 || dsth<16)
		return 1;

	Resampler::eFilter fmode;

	switch(mfd->filter_mode) {
	case FILTER_NONE:			fmode = Resampler::eFilter::kPoint; break;
	case FILTER_BILINEAR:		fmode = Resampler::eFilter::kLinearInterp; break;
	case FILTER_BICUBIC:		fmode = Resampler::eFilter::kCubicInterp; break;
	case FILTER_TABLEBILINEAR:	fmode = Resampler::eFilter::kLinearDecimate; break;
	case FILTER_TABLEBICUBIC060:	fmode = Resampler::eFilter::kCubicDecimate060; break;
	case FILTER_TABLEBICUBIC075:	fmode = Resampler::eFilter::kCubicDecimate075; break;
	case FILTER_TABLEBICUBIC100:	fmode = Resampler::eFilter::kCubicDecimate100; break;
	case FILTER_LANZCOS3:			fmode = Resampler::eFilter::kLanzcos3; break;
	}

	mfd->resampler = new Resampler();

	if (mfd->fInterlaced)
		mfd->resampler->Init(fmode, fmode, dstw, (dsth+1)/2, fa->src.w, fa->src.h/2);
	else
		mfd->resampler->Init(fmode, fmode, dstw, dsth, fa->src.w, fa->src.h);

	return 0;
}

static int resize_stop(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	delete mfd->resampler;	mfd->resampler = NULL;

	return 0;
}

static void resize_script_config(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;

	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->new_x	= argv[0].asInt();
	mfd->new_y	= argv[1].asInt();

	if (argv[2].isInt())
		mfd->filter_mode = argv[2].asInt();
	else {
		char *s = *argv[2].asString();

		if (!stricmp(s, "point") || !stricmp(s, "nearest"))
			mfd->filter_mode = 0;
		else if (!stricmp(s, "bilinear"))
			mfd->filter_mode = 1;
		else if (!stricmp(s, "bicubic"))
			mfd->filter_mode = 2;
		else
			EXT_SCRIPT_ERROR(FCALL_UNKNOWN_STR);
	}

	mfd->fInterlaced = false;

	if (mfd->filter_mode & 128) {
		mfd->fInterlaced = true;
		mfd->filter_mode &= 127;
	}

	mfd->fLetterbox = false;

	if (argc > 3) {
		mfd->new_xf = argv[3].asInt();
		mfd->new_yf = argv[4].asInt();
		mfd->fLetterbox = true;
		mfd->rgbColor = revcolor(argv[5].asInt());
	}
}

static ScriptFunctionDef resize_func_defs[]={
	{ (ScriptFunctionPtr)resize_script_config, "Config", "0iii" },
	{ (ScriptFunctionPtr)resize_script_config, NULL, "0iis" },
	{ (ScriptFunctionPtr)resize_script_config, NULL, "0iiiiii" },
	{ (ScriptFunctionPtr)resize_script_config, NULL, "0iisiii" },
	{ NULL },
};

static CScriptObject resize_obj={
	NULL, resize_func_defs
};

static bool resize_script_line(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	int filtmode = mfd->filter_mode + (mfd->fInterlaced ? 128 : 0);

	if (mfd->fLetterbox)
		_snprintf(buf, buflen, "Config(%d,%d,%d,%d,%d,0x%06x)", mfd->new_x, mfd->new_y, filtmode, mfd->new_xf, mfd->new_yf,
			revcolor(mfd->rgbColor));
	else
		_snprintf(buf, buflen, "Config(%d,%d,%d)", mfd->new_x, mfd->new_y, filtmode);

	return true;
}

FilterDefinition filterDef_resize={
	0,0,NULL,
	"resize",
	"Resizes the image to a new size."
#ifdef USE_ASM
			"\n\n[Assembly optimized] [FPU optimized] [MMX optimized]"
#endif
			,
	NULL,NULL,
	sizeof(MyFilterData),
	NULL,NULL,
	resize_run,
	resize_param,
	resize_config,
	resize_string,
	resize_start,
	resize_stop,

	&resize_obj,
	resize_script_line,
};