#include "asynclog.h"
#include "color.h"
#include <iostream>
#include <syslog.h>

namespace VFIELD {


AsyncLog::AsyncLog(bool syslog) :
	use_syslog(syslog),
	end_flag(false),
	writer( boost::bind(&AsyncLog::WriterThread, this) )
{
	if( use_syslog ) {
		::openlog("VFIELD", 0, LOG_USER);
	}
}

AsyncLog::~AsyncLog()
{
	queue.push( std::make_pair(format("Stopping asynclog thread"), ASLOG_INFO) );
	end_flag = true;
	queue.push( std::make_pair(format("End of log"), ASLOG_INFO) );	// スレッドを開始させる
	writer.join();

	if( use_syslog ) {
		::closelog();
	}
}


void AsyncLog::WriterThread(void)
{
	while(1) {
		queue_type::value_ptr msg( queue.blocked_pop() );
		const char* color = "";
		int syslog_level = LOG_DEBUG;
		switch(msg->second) {
		case ASLOG_DEBUG:
			color = Color::NORMAL;
			syslog_level = -1;
			break;
		case ASLOG_INFO:
			color = Color::WHITE;
			syslog_level = LOG_DEBUG;
			break;
		case ASLOG_NOTICE:
			color = Color::YELLOW;
			syslog_level = LOG_INFO;
			break;
		case ASLOG_WARNING:
			color = Color::MAGENTA;
			syslog_level = LOG_WARNING;
			break;
		case ASLOG_ERR:
			color = Color::RED;
			syslog_level = LOG_ERR;
			break;
		case ASLOG_CRIT:
			color = Color::RED;
			syslog_level = LOG_CRIT;
			break;
		case ASLOG_ALERT:
			color = Color::RED;
			syslog_level = LOG_ALERT;
			break;
		case ASLOG_EMERG:
			color = Color::RED;
			syslog_level = LOG_EMERG;
			break;
		}

		msg->first.exceptions( boost::io::no_error_bits );
		std::cout << color << msg->first << Color::NORMAL << std::endl;

		if( use_syslog && syslog_level >= 0 ) {
			::syslog(syslog_level, "%s", boost::io::str(msg->first).c_str());
		}

		if( end_flag && queue.empty() ) return;	// ログは最後まで出力してから終わる
	}

	// コンソールの色を戻す
	std::cout << Color::NORMAL << std::flush;
}


}  // namespace VFIELD
