#include "ip46.h"
#include "exception.h"
#include "vfield.h"
#include <string>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <boost/lexical_cast.hpp>

#include <cstdio>

namespace VFIELD {


////
// helper functions
//
namespace {
typedef enum {
	TCP,
	UDP,
} proto_type;

int listen46IMPL(proto_type proto, uint16_t port)
{
	// make address
	struct sockaddr_in6 addr;
	::memset(&addr, 0, sizeof(addr));
	addr.sin6_port = port;
	addr.sin6_family = AF_INET6;	// AF_INET6はIPv4射影アドレスでIPv4も受け付ける
	addr.sin6_addr = in6addr_any;


	// create socket
	int sock = -1;
	// --  TCP / UDP Switch {  --//
	switch(proto) {
	case TCP:
		sock = ::socket(PF_INET6, SOCK_STREAM, 0);
		break;
	case UDP:
		sock = ::socket(PF_INET6, SOCK_DGRAM,  0);
		break;
	}
	// --  } TCP / UDP Switch  --//
	if( sock < 0 ) {
		return -2;	// socket() error
	}


	// bind
	int on = 1;
	if( ::setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0 ) {
		return -3;	// setsockopt() error
	}

	if( ::bind(sock, (struct sockaddr*)&addr, sizeof(addr)) < 0 ) {
		return -4;	// bind() error
	}


	// listen if proto is TCP
	// --  TCP / UDP Switch {  --//
	if( proto == TCP ) {
		static const int backlog = 5;	// FIXME: ?
		if( ::listen(sock, backlog) < 0 ) {
			return -5;	// listen() error
		}
	}
	// --  } TCP / UDP Switch  --//


	return sock;
}
}  // noname namespace


////
// addr46_t
//
const char* addr46_t::addressString(char* dst, size_t cnt) const
{
	if(isIPv6()) {
		return ::inet_ntop(AF_INET6, addr_in.sin6.sin6_addr.s6_addr, dst, cnt);
	} else {
		return ::inet_ntop(AF_INET, &addr_in.sin.sin_addr.s_addr, dst, cnt);
	}
}
std::string addr46_t::addressString(void) const
{
	char addr_dst[50];	// FIXME: おそらく41で十分
	addressString(addr_dst, sizeof(addr_dst));
	if(addr_dst == NULL) {
		return std::string();
	} else {
		return std::string(addr_dst);
	}
}
std::ostream& operator<< (std::ostream& stream, const addr46_t& rhl)
{
	return stream << rhl.addressString();
}


////
// NodeIdentity
//
NodeIdentity::NodeIdentity(const char* numeric_host, const char* port)
{
	// stream/stream.cc StrippedNodeIdentity::StrippedNodeIdentityと連動

	::memset(m_data, 0, sizeof(m_data));

	////
	// inet_ptonバージョン
	//
	uint16_t port_num;
	try {
		port_num = htons( boost::lexical_cast<uint16_t>(port) );
	} catch (...) {
		throw InvalidAddressException(EINVAL, "Invalid port number");
	}


	char addr[ sizeof(struct in6_addr) ];	// IPv6アドレスの方が大きい

	int res;
	if( (res = ::inet_pton(AF_INET6, numeric_host, addr)) > 0 ) {
		// IPv6アドレス
		m_data[0] |= NODE_IDENTITY_FLAG_IPV6;
		::memcpy(&m_data[1], &((struct in6_addr*)addr)->s6_addr, 16);
		::memcpy(&m_data[17], &port_num, 2);

	} else if( res == 0 && ::inet_pton(AF_INET, numeric_host, addr) > 0 ) {
		// IPv4アドレス
		// m_data[0] |= 0;
		const struct in_addr* sin = (struct in_addr*)addr;
		::memcpy(&m_data[13], &sin->s_addr, 4);
		::memcpy(&m_data[17], &port_num, 2);

	} else {
		throw InvalidAddressException(errno, "Invalid address");
	}


#if 0
	////
	// getaddrinfoバージョン
	//
	// getaddrinfoを使うと、libcをスタティックリンクしても実行時にlibcが必要になってしまう
	// getaddrinfoほどの機能は必要ない

	struct addrinfo hints;
	::memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_flags = AI_NUMERICHOST;// | AI_NUMERICSERV;	// FIXME: AI_NUMERICSERVはどうする？
					// AI_NUMERICHOST: hostは数値的な文字列のみ（名前解決しない）
					// AI_NUMERICSERV: serviceは数字の文字列に限る
	hints.ai_socktype = SOCK_STREAM;

	struct addrinfo* res = NULL;
	int err = ::getaddrinfo(numeric_host, port, &hints, &res);
	if( err ) {
		throw InvalidAddressException(errno, "Invalid address or port number");
	}

	struct addrinfo* ai_first(res);		// 最初の一つを使う
	if( ai_first->ai_family == AF_INET6 ) {
		m_data[0] |= NODE_IDENTITY_FLAG_IPV6;
		const struct sockaddr_in6* const sin6 = (const struct sockaddr_in6*)ai_first->ai_addr;
		::memcpy(&m_data[1], sin6->sin6_addr.s6_addr, 16);
		::memcpy(&m_data[17], &sin6->sin6_port, 2);
	} else {
		// m_data[0] |= 0;
		const struct sockaddr_in* const sin  = (const struct sockaddr_in*)ai_first->ai_addr;
		for(int i = 0; i < 12; i++ ) {
			printf("%02X", ((unsigned char*)&sin->sin_addr)[i]);
		}
		::memcpy(&m_data[13], &sin->sin_addr.s_addr, 4);
		::memcpy(&m_data[17], &sin->sin_port, 2);
	}
#endif // if 0
}
bool NodeIdentity::isIPv6(void) const
{
	return (m_data[0] & NODE_IDENTITY_FLAG_IPV6);
}
uint16_t NodeIdentity::getPortHostByteOrder(void) const
{
	return ntohs(getPortN());
}
const char* NodeIdentity::addressString(char* dst, size_t cnt) const
{
	if(isIPv6()) {
		struct sockaddr_in6 sin6;
		::memset(&sin6, 0, sizeof(sin6));
		sin6.sin6_family = PF_INET6;
		sin6.sin6_port   = getPortN();
		::memcpy(sin6.sin6_addr.s6_addr, getIPv6Address(), 16);
		return ::inet_ntop(AF_INET6, sin6.sin6_addr.s6_addr, dst, cnt);
	} else {
		struct sockaddr_in sin;
		::memset(&sin, 0, sizeof(sin));
		sin.sin_family = PF_INET;
		sin.sin_port   = getPortN();
		sin.sin_addr.s_addr = getIPv4Address();
		return ::inet_ntop(AF_INET, &sin.sin_addr.s_addr, dst, cnt);
	}
}
std::string NodeIdentity::addressString(void) const
{
	char addr_dst[50];	// FIXME: おそらく41で十分
	addressString(addr_dst, sizeof(addr_dst));
	if(addr_dst == NULL) {
		return std::string();
	} else {
		return std::string(addr_dst);
	}
}




////
// public functions
//
// for receive
int listenTCP46(const char* numeric_port_str)
{
	uint16_t port = 0;
	::sscanf(numeric_port_str, "%hd", &port);
	return listen46IMPL(TCP, htons(port));
}

int listenTCP46(uint16_t port)
{
	return listen46IMPL(TCP, htons(port));
}

int listenUDP46(const char* numeric_port_str)
{
	uint16_t port = 0;
	::sscanf(numeric_port_str, "%hd", &port);
	return listen46IMPL(UDP, htons(port));
}

int listenUDP46(uint16_t port)
{
	return listen46IMPL(UDP, htons(port));
}

int accept46(int sock, addr46_t* from) throw()
{
	struct sockaddr_storage from_ss;
	socklen_t len_ss = sizeof(from_ss);
	int retval = accept(sock, (struct sockaddr*)&from_ss, &len_ss);

	if( from != NULL ) {
		if( ((struct sockaddr*)&from_ss)->sa_family == AF_INET6 ) {
			::memcpy(&from->addr_in.sin6, (struct sockaddr_in6*)&from_ss, sizeof(struct sockaddr_in6));
		} else {
			::memcpy(&from->addr_in.sin,  (struct sockaddr_in*) &from_ss, sizeof(struct sockaddr_in) );
		}
	}
	return retval;
}

ssize_t recvfrom46(int sock, void* buf, size_t len, int flags, addr46_t* from) throw()
{
	struct sockaddr_storage from_ss;
	socklen_t len_ss = sizeof(from_ss);
	int retval = recvfrom(sock, buf, len, flags, (struct sockaddr*)&from_ss, &len_ss);
	if( from != NULL ) {
		if( ((struct sockaddr*)&from_ss)->sa_family == AF_INET6 ) {
			::memcpy(&from->addr_in.sin6, (struct sockaddr_in6*)&from_ss, sizeof(struct sockaddr_in6));
		} else {
			::memcpy(&from->addr_in.sin,  (struct sockaddr_in*) &from_ss, sizeof(struct sockaddr_in) );
		}
	}
	return retval;
}


// for send
int connectTCP46(const addr46_t& to) throw()
{
	int sock;
	sock = ::socket(
			(to.isIPv6() ? AF_INET6 : AF_INET),
			SOCK_STREAM,
			0
		       );
	if( sock < 0 ) {
		return -errno;	// socket()エラー
	}

	int ret;
	if( to.isIPv6() ) {
		ret = ::connect(sock, (struct sockaddr*)to.getsin6(), sizeof(struct sockaddr_in6));
	} else {
		ret = ::connect(sock, (struct sockaddr*)to.getsin(),  sizeof(struct sockaddr_in));
	}

	if( ret < 0 ) {
		::close(sock);
		return -errno;	// connect()エラー
	} else {
		return sock;
	}
}

int timed_connectTCP46(const addr46_t& to, long sec, long usec) throw()
{
	int sock;
	sock = ::socket(
			(to.isIPv6() ? AF_INET6 : AF_INET),
			SOCK_STREAM,
			0
		       );
	if( sock <= 0 ) {
		return -errno;	// socket()エラー
	}

	// NonBlockingモードにセット
	if( ::fcntl(sock, F_SETFL, O_NONBLOCK) < 0 ) {
		::close(sock);
		return -errno;	// fcntlエラー
	}

	// connectする
	int ret;
	if( to.isIPv6() ) {
		ret = ::connect(sock, (struct sockaddr*)to.getsin6(), sizeof(struct sockaddr_in6));
	} else {
		ret = ::connect(sock, (struct sockaddr*)to.getsin(),  sizeof(struct sockaddr_in));
	}
	if( ret > 0 ) {
		// すぐに成功した。問題なし
		return sock;
	}

	// connect待ち
	fd_set fds;
	struct timeval tv;

	FD_ZERO(&fds);
	FD_SET(sock, &fds);

	tv.tv_sec = sec;
	tv.tv_usec = usec;

	ret = ::select(sock+1, NULL, &fds, NULL, &tv);
	if( ret > 0 ) {
		// connect成功
		// Blockingモードに戻さずにreturn
		return sock;
	} else if( ret == 0 ) {
		// タイムアウト
		::close(sock);
		return 0;	// タイムアウト
	} else if( errno == EINTR ) {
		// EINTRはタイムアウトとみなす
		::close(sock);
		return 0;
	} else {
		// select()エラー
		::close(sock);
		return -errno;
	}
}

// PF_INET6ではシステムによってAF_INETを送れないので、IPv4版とIPv6版を分ける
int socketUDP6(void)
{
	return ::socket(PF_INET6, SOCK_DGRAM, 0);
}

int socketUDP4(void)
{
	return ::socket(PF_INET, SOCK_DGRAM, 0);
}

int sendto46(int sock, const void* buf, size_t len, int flags, const addr46_t& to) throw()
{
	if( to.isIPv6() ) {
		return ::sendto(sock, buf, len, flags, (struct sockaddr*)to.getsin6(), sizeof(struct sockaddr_in6));
	} else {
		return ::sendto(sock, buf, len, flags, (struct sockaddr*)to.getsin(),  sizeof(struct sockaddr_in) );
	}
}



// misc
int setTimeout(int sock, long send_sec, long send_usec, long recv_sec, long recv_usec) throw()
{
	struct timeval tv;

	tv.tv_sec = recv_sec;
	tv.tv_usec = recv_usec;
	if( ::setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, (const void*)&tv, sizeof(tv)) < 0 ) {
		return -errno;
	}

	tv.tv_sec = send_sec;
	tv.tv_usec = send_usec;
	if( ::setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, (const void*)&tv, sizeof(tv)) < 0 ) {
		return -errno;
	}

	return 0;
}

namespace {
typedef std::vector< std::pair<std::string, std::string> > pair_vector;
void ioctlIfList(pair_vector& result)
{
	static const unsigned int MAX_IFR = 10;

	// XXX: getifaddrs(3)を使う
	// ifaddrs.h

	struct ifreq ifr[MAX_IFR];
	struct ifconf ifc;

	ifc.ifc_len = sizeof(ifr);
	ifc.ifc_ifcu.ifcu_buf = (char*)ifr;

	int fd = ::socket(AF_INET, SOCK_DGRAM, 0);
	::ioctl(fd, SIOCGIFCONF, &ifc);
	::close(fd);

	unsigned int nifs = ifc.ifc_len / sizeof(struct ifreq);

	for(unsigned int i=0; i < nifs; i++) {
		char buf[41];
		// FIXME: IPv6に対応していない
		::inet_ntop(AF_INET, &((struct sockaddr_in*)&ifr[i].ifr_addr)->sin_addr.s_addr, buf, sizeof(buf));
		result.push_back( std::pair<std::string, std::string>(ifr[i].ifr_name, buf) );
	}
}
}  // noname namespace

void getIfList(std::vector<std::string>& result)
{
	pair_vector iflist;
	ioctlIfList(iflist);
	for(pair_vector::const_iterator ifa(iflist.begin()), ifa_end(iflist.end());
			ifa != ifa_end;
			++ifa ) {
		result.push_back(ifa->first);
	}
}

std::string getIfAddr(const std::string& ifname)
{
	pair_vector iflist;
	ioctlIfList(iflist);
	for(pair_vector::const_iterator ifa(iflist.begin()), ifa_end(iflist.end());
			ifa != ifa_end;
			++ifa ) {
		if( ifa->first == ifname ) {
			return ifa->second;
		}
	}
	return "";
}



}  // namespace VFIELD
