#include "rpcclient.h"
#include "log.h"
#include <limits>
#include <iostream>

namespace VFIELD {


RPCClient::RPCClient(	const char* bcast_udp_port,
			const NodeIdentity& self_identity
			) :
	// XXX: ブロードキャストアドレスはリッスンインターフェースから取得する
	m_bcast_addr( NodeIdentity("255.255.255.255", bcast_udp_port) ),
	m_self_identity(self_identity),
	m_self_range(	std::numeric_limits<pos_type>::max(),
			std::numeric_limits<pos_type>::max()
		    ),		// 初期Range  後でStorageから更新してもらう
	m_image_id(0)		// image_idの初期値は0
{
	// 送信用のソケットを作る
	for(unsigned int i=0; i < 4; ++i) {
		if( (m_send_socks4[i] = socketUDP4()) < 0 ) {
			throw RPCSendException(errno, "Can't make IPv4 UDP socket for sending");
		}
	}
	for(unsigned int i=0; i < 4; ++i) {
		if( (m_send_socks6[i] = socketUDP6()) < 0 ) {
			throw RPCSendException(errno, "Can't make IPv6 UDP socket for sending");
		}
	}
}

RPCClient::~RPCClient() {}


void RPCClient::setSelfRange(const DataRange& range)
{
	m_self_range = range;
}

void RPCClient::setImageId(uint16_t image_id)
{
	m_image_id = image_id;
}


////
// Unicast Notify Up
//
void RPCClient::ucastNotifySelfUp(const addr46_t& to)
	{ ucastNotifyOtherUp(to, m_self_identity, m_self_range); }
void RPCClient::ucastNotifyOtherUp(const addr46_t& to, const NodeIdentity& target, const DataRange& target_range)
{
	LogDebug0( Log::format("Sending RPC Notify Up (%1%: %2%) to %3%") % addr46_t(target) % target_range % to);
	char buf[RPC_COMMON_HEADER_SIZE + RPC_NOTIFY_UP_SIZE];
	makeNotifyUp(buf, target, target_range);
	sendUcast(to, buf, sizeof(buf));
}

////
// Broadcast Notify Up
//
void RPCClient::bcastNotifySelfUp(void)
	{ bcastNotifyOtherUp(m_self_identity, m_self_range); }
void RPCClient::bcastNotifyOtherUp(const NodeIdentity& target, const DataRange& target_range)
{
	LogDebug0( Log::format("Sending RPC Notify Up (%1%: %2%) to broadcast") % addr46_t(target) % target_range );
	char buf[RPC_COMMON_HEADER_SIZE + RPC_NOTIFY_UP_SIZE];
	makeNotifyUp(buf, target, target_range);
	sendBcast(buf, sizeof(buf));
}


////
// Unicast Notify Down
//
void RPCClient::ucastNotifySelfDown(const addr46_t& to)
	{ ucastNotifyOtherDown(to, m_self_identity); }
void RPCClient::ucastNotifyOtherDown(const addr46_t& to, const NodeIdentity& target)
{
	LogDebug0( Log::format("Sending RPC Notify Down (%1%) to %2%") % addr46_t(target) % addr46_t(target) );
	char buf[RPC_COMMON_HEADER_SIZE + RPC_NOTIFY_DOWN_SIZE];
	makeNotifyDown(buf, target);
	sendUcast(to, buf, sizeof(buf));
}

////
// Broadcast Notify Down
//
void RPCClient::bcastNotifySelfDown(void)
	{ bcastNotifyOtherDown(m_self_identity); }
void RPCClient::bcastNotifyOtherDown(const NodeIdentity& target)
{
	LogDebug0( Log::format("Sending RPC Notify Down (%1%) to broadcast") % addr46_t(target) );
	char buf[RPC_COMMON_HEADER_SIZE + RPC_NOTIFY_DOWN_SIZE];
	makeNotifyDown(buf, target);
	sendBcast(buf, sizeof(buf));
}


////
// Unicast Find
//
void RPCClient::ucastFind(const addr46_t& to, const DataRange& range)
{
	LogDebug0( Log::format("Sending RPC Find (%1%) to %2%") % range % to );
	char buf[RPC_COMMON_HEADER_SIZE + RPC_FIND_SIZE];
	makeFind(buf, range);
	sendUcast(to, buf, sizeof(buf));
}

////
// Broadcast Find
//
void RPCClient::bcastFind(const DataRange& range)
{
	LogDebug0( Log::format("Sending RPC Find (%1%) to broadcast") % range );
	char buf[RPC_COMMON_HEADER_SIZE + RPC_FIND_SIZE];
	makeFind(buf, range);
	sendBcast(buf, sizeof(buf));
}


////
// Unicast Ping
//
void RPCClient::ucastPing(const addr46_t& to)
{
	LogDebug0( Log::format("Sending RPC Ping to %1%") % to);
	char buf[RPC_COMMON_HEADER_SIZE + RPC_PING_SIZE];
	makePing(buf, m_self_identity, m_self_range);
	sendUcast(to, buf, sizeof(buf));
}


// private:
inline int RPCClient::getSocket(const addr46_t& to, const char* buf, size_t len)
{
	LogDebug0( Log::format("Using socks[%1%] for sending RPC") % static_cast<unsigned short>(buf[(len>>1)] & 0x03) );
	if(to.isIPv6()) {
		return m_send_socks6[ buf[(len>>1)] & 0x03 ];
	} else {
		return m_send_socks4[ buf[(len>>1)] & 0x03 ];
	}
}

void RPCClient::makeNotifyUp(char* buf, const NodeIdentity& target, const DataRange& target_range) const throw()
{  // バッファの長さは RPC_COMMON_HEADER_SIZE + RPC_NOTIFY_UP_SIZE
	buf[0] = RPC_NOTIFY_UP_MAGIC;
	::memcpy(&buf[1], &m_image_id, 2);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+0],  target.getNetRaw(), NodeIdentity::raw_size);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+19], target_range.getNetRaw().get(), DataRange::raw_size);
}

void RPCClient::makeNotifyDown(char* buf, const NodeIdentity& target) const throw()
{  // バッファの長さは RPC_COMMON_HEADER_SIZE + RPC_NOTIFY_DOWN_SIZE
	buf[0] = RPC_NOTIFY_DOWN_MAGIC;
	::memcpy(&buf[1], &m_image_id, 2);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+0], target.getNetRaw(), NodeIdentity::raw_size);
}

void RPCClient::makeFind(char* buf, const DataRange& range) const throw()
{  // バッファの長さは RPC_COMMON_HEADER_SIZE + RPC_FIND_SIZE
	buf[0] = RPC_FIND_MAGIC;
	::memcpy(&buf[1], &m_image_id, 2);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+0], range.getNetRaw().get(), DataRange::raw_size);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+16], m_self_identity.getNetRaw(), NodeIdentity::raw_size);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+37], m_self_range.getNetRaw().get(), DataRange::raw_size);
}

void RPCClient::makePing(char* buf, const NodeIdentity& self, const DataRange& self_range) const throw()
{  // バッファの長さは RPC_COMMON_HEADER_SIZE + RPC_PING_SIZE
	buf[0] = RPC_PING_MAGIC;
	::memcpy(&buf[1], &m_image_id, 2);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+0],  self.getNetRaw(), NodeIdentity::raw_size);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+19], self_range.getNetRaw().get(), DataRange::raw_size);
}


inline void RPCClient::sendUcast(const addr46_t& to, const char* buf, size_t len)
{
	int sock = getSocket(to, buf, len);
	if( sendto46(sock, buf, len, 0, to) < 0 ) {
		throw RPCSendException(errno, "Failed to send RPC");
	}
}

inline void RPCClient::sendBcast(const char* buf, size_t len)
{
	int sock = getSocket(m_bcast_addr, buf, len);
	int on = 1;
	if( setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &on, sizeof(on)) < 0 ) {
		throw RPCSendException(errno, "Can't enable broadcast");
	}
	if( sendto46(sock, buf, len, 0, m_bcast_addr) < 0 ) {
		throw RPCSendException(errno, "Failed to send RPC");
	}
}


}  // namespace VFIELD
