#include "controller.h"
#include "engine.h"
#include "datarange.h"
#include "bufferedreq.h"
#include "commander.h"
#include "threadpool.h"
#include "log.h"
#include "exception.h"
#include <map>

namespace VFIELD {


class SearchControllerIMPL {
public:
	SearchControllerIMPL(	StreamManager& smgr,
				RPCClient& rpcc,
				VTable& vtable,
				ThreadPool& threadpool );
	~SearchControllerIMPL();
public:
	inline void submitRequests(std::vector<BufferedRequest>& requests);
	inline void submitRequest(BufferedRequest& request);
public:
	inline void rpcNotifyUp(const NodeIdentity& node, const DataRange& range);
private:
	EngineToolSet m_toolset;
private:
	boost::mutex m_mutex;
	ThreadPool& m_threadpool;
	typedef std::multimap<DataRange, EngineCommander*, CompareStartLess> thread_map_type;
	typedef thread_map_type::iterator thread_map_iterator;
	thread_map_type m_thread_map;
private:
	void EngineWrapper(BufferedRequest& request);
};


SearchController::SearchController(StreamManager& smgr, RPCClient& rpcc, VTable& vtable, ThreadPool& threadpool) :
	impl(new SearchControllerIMPL(smgr, rpcc, vtable, threadpool)) {}
SearchControllerIMPL::SearchControllerIMPL(StreamManager& smgr, RPCClient& rpcc, VTable& vtable, ThreadPool& threadpool) :
	m_toolset(smgr, rpcc, vtable, threadpool),
	m_threadpool(threadpool)
{}

SearchController::~SearchController() {}
SearchControllerIMPL::~SearchControllerIMPL()
{
	LogDebug("Destructing StreamController");
	// すべてjoinしないと、子スレッドが持っているリファレンスが無効になって、
	// セグメンテーションフォールトする
	m_threadpool.joinThreads();
}

void SearchControllerIMPL::EngineWrapper(BufferedRequest& request)
{
	thread_map_type::iterator progress;
	EngineCommander commander;
	{  // m_thread_mapをロック
		boost::mutex::scoped_lock lk(m_mutex);
		// m_thread_mapに追加
		progress = m_thread_map.insert( std::make_pair(request, &commander) );
	}  // m_thread_mapをアンロック

	// ダウンロードを実行
	SearchEngine(request, m_toolset, commander);
	// ダウンロード終了（または失敗）

	{  // m_thread_mapをロック
		boost::mutex::scoped_lock lk(m_mutex);
		// m_thread_mapから削除
		m_thread_map.erase(progress);
	}  // m_thread_mapをアンロック
}


void SearchController::submitRequests(std::vector<BufferedRequest>& requests)
	{ return impl->submitRequests(requests); }
void SearchControllerIMPL::submitRequests(std::vector<BufferedRequest>& requests)
{
	std::for_each(	requests.begin(),
			requests.end(),
			boost::bind(&SearchControllerIMPL::submitRequest, this, _1) );
}

void SearchController::submitRequest(BufferedRequest& request)
	{ return impl->submitRequest(request); }
void SearchControllerIMPL::submitRequest(BufferedRequest& request)
{
	LogDebug0( Log::format("Download request: %1%") % request );

	m_threadpool.submitTask(
			boost::bind(
				&SearchControllerIMPL::EngineWrapper,
				this,
				request
				)
			);
}

namespace {
struct NotifyUpMapping {
	NotifyUpMapping(const NodeIdentity& node, const DataRange& range) :
		m_node(node), m_range(range) {}
	void operator() (std::pair<const DataRange, EngineCommander*>& tm)
	{
		// FIXME: 前は線形検索する
		if( tm.first.contains_part(m_range) ) {
			tm.second->rpcNotifyUp(m_node, m_range);
			LogDebug0( Log::format("RPC Notify Up (%1%) is mapped to (%2%) processor") % m_range % tm.first );
		}
	}
private:
	const NodeIdentity& m_node;
	const DataRange& m_range;
	NotifyUpMapping();
};
}
void SearchController::rpcNotifyUp(const NodeIdentity& node, const DataRange& range)
	{ return impl->rpcNotifyUp(node, range); }
void SearchControllerIMPL::rpcNotifyUp(const NodeIdentity& node, const DataRange& range)
{
	// m_thread_mapをロック
	boost::mutex::scoped_lock lk(m_mutex);

	// 部分一致を検索する
	// FIXME: 前は2分検索、後ろは線形検索 エントリが多くなると遅くなるが、エントリは多くならない
	thread_map_iterator match_upper(
			m_thread_map.upper_bound( DataRange(range.end(), range.end()) )
			);

	// マッチしたEngineCommanderに知らせる
	std::for_each(m_thread_map.begin(), match_upper, NotifyUpMapping(node,range) );

	// m_thread_mapをアンロック
}


}  // namespace VFIELD
