#ifndef VFIELD_STORAGE_H__
#define VFIELD_STORAGE_H__

#include "request.h"
#include "interface/receptor.h"
#include "exception.h"
#include <vector>
#include <boost/utility.hpp>
#include <boost/thread.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/scoped_array.hpp>

namespace VFIELD {


////
// Storage
//
class DataRange;
class NodeIdentity;
class AutoSock;
class RPCClient;
class Storage : private boost::noncopyable {
public:
	Storage() : m_image_size(0) {}
	virtual ~Storage() {}
public:
	virtual uint16_t getImageId(void) const { return m_image_id; }
	virtual uint64_t getImageSize(void) const { return m_image_size; }
	virtual void rpcFind(const NodeIdentity& from, const DataRange& find_range) = 0;
	virtual void streamGetData(AutoSock& asock, const DataRange& range) = 0;
protected:
	uint16_t m_image_id;
	uint64_t m_image_size;
};


////
// RootStorage
//
// ファイルをエクスポートするノード用のStorage
class VTable;
class RootStorage : public Storage {
public:
	RootStorage(const std::string& image_path, RPCClient& rpc);	// throw(FileSystemException)
	~RootStorage();
public:  // 応答処理
	// Storageインターフェース
	void rpcFind(const NodeIdentity& from, const DataRange& find_range);
	void streamGetData(AutoSock& asock, const DataRange& range);	// throw(FileSystemException, StreamSocketException)
public:
	// EdgeStorage::invokeJoinに対応
	void initialize(const NodeIdentity& self, VTable& vtable);
public:
	uint32_t request(char* buf, uint64_t from, uint32_t length);	// throw(FileSystemException)
private:
	int m_fd;
	RPCClient& m_rpc;
private:
	uint32_t readData(char* buf, uint64_t from, uint32_t length);
private:
	RootStorage();
};


////
// EdgeStorage
//
// ファイルをエクスポートしないノード用のStorage
class ThreadPool;
class StreamManager;
class RPCServer;
class SearchController;
class addr46_t;
class EdgeStorage : public Storage {
public:
	EdgeStorage(RPCClient& rpcc, VTable& vtable, uint64_t store_limit, ThreadPool& threadpool);
	virtual ~EdgeStorage();
public:
	// Storageインターフェース
	void rpcFind(const NodeIdentity& from, const DataRange& find_range);
public:
	// EdgeStorageインターフェース
	virtual void request(	DataRequest& request,
				std::vector<DataReceptor>* result_cached,
				std::vector<BufferedRequest>*  result_uncached	) = 0;
	virtual void invokeJoin(StreamManager& smgr, RPCServer& rpcs, SearchController& engine, const addr46_t& bootstrap, unsigned int join_delay = 0, unsigned short join_boost = STREAM_JOIN_DEFAULT_BOOST, bool exec_this_thread = false);
		// ブロックしない
	//virtual void invokeJoin(RPCServer& rpcs);
	//	// ブロックしない
protected:
	// EdgeStorageインターフェース
	virtual void setSelfRange(const DataRange& range, SearchController& engine, unsigned short join_boost = STREAM_JOIN_DEFAULT_BOOST) = 0;
	virtual void invalidate(void);
protected:
	RPCClient& m_rpcc;
	VTable& m_vtable;
	uint64_t m_store_limit;
	DataRange m_self_range;
	ThreadPool& m_threadpool;
private:
	void invokeJoinIMPL(StreamManager* smgr, RPCServer* rpcs, SearchController* engine, const addr46_t* bootstrap, unsigned int join_delay, unsigned short join_boost);
private:
	EdgeStorage();
};


////
// EdgeStorageToRAM
//
// メモリにデータを保存する
class BufferedRequest;
class EdgeStorageToRAM : public EdgeStorage {
public:
	EdgeStorageToRAM(uint64_t store_limit, RPCClient& rpcc, VTable& vtable, ThreadPool& threadpool); // throw(std::bad_alloc)
	~EdgeStorageToRAM();
public:  // 応答処理
	// Storageインターフェース
	void streamGetData(AutoSock& asock, const DataRange& range);
public:
	// EdgeStorageインターフェース
	void request(	DataRequest& request,
			std::vector<DataReceptor>* result_cached,
			std::vector<BufferedRequest>*  result_uncached	);
protected:
	// EdgeStorageインターフェース
	void setSelfRange(const DataRange& range, SearchController& engine, unsigned short join_boost = STREAM_JOIN_DEFAULT_BOOST);
	void invalidate(void);
private:
	BufferedRequest m_ram;	// m_bufferよりm_ramの方が寿命が短くないと危険
	boost::shared_array<char> m_buffer;
private:
	EdgeStorageToRAM();
};


////
// EdgeStorageToFile
//
// ファイルにデータを保存する
class EdgeStorageToFile : public EdgeStorage {
public:
	EdgeStorageToFile(size_t store_size, const std::string& file_path);
	~EdgeStorageToFile();
public:
	void request(	char* buf,
			uint64_t from,
			uint32_t length,
			std::vector<DataReceptor>* result_cached,
			std::vector<BufferedRequest>*  result_uncached	);
private:
	char* m_cache;
	size_t store_size;	// 0 = すべてキャッシュ
	std::string file_path;
private:
	EdgeStorageToFile();
};



////
// Exception
//
struct FileSystemException : public SystemCallException {
	FileSystemException(int errno_, const std::string& message) :
		SystemCallException(errno_, message) {}
};

struct JoinFailedException : public std::runtime_error {
	JoinFailedException(const std::string& message) :
		std::runtime_error(message) {}
};



}  // namespace VFIELD

#endif /* storage.h */
