#include "storage.h"
#include "rpc/rpcclient.h"
#include "rpc/rpcserver.h"
#include "stream/stream.h"
#include "vtable/vtable.h"
#include "node.h"
#include "threadpool.h"
#include "log.h"
#include <limits>
#include <algorithm>

namespace VFIELD {


void EdgeStorage::invalidate(void)
{
	LogDebug("EdgeStorage inivalidating");
	m_self_range = DataRange( std::numeric_limits<pos_type>::max(), std::numeric_limits<pos_type>::max() );
}


EdgeStorage::EdgeStorage(RPCClient& rpcc, VTable& vtable, uint64_t store_limit, ThreadPool& threadpool) :
	m_rpcc(rpcc),
	m_vtable(vtable),
	m_store_limit(store_limit),
	m_self_range(	std::numeric_limits<pos_type>::max(),
			std::numeric_limits<pos_type>::max()
		    ),		// 初期Range
	m_threadpool(threadpool)
{
	LogInfo( Log::format("Storage layer is initialized with %1% bytes limit") % m_store_limit );
}

EdgeStorage::~EdgeStorage() {}


void EdgeStorage::rpcFind(const NodeIdentity& from, const DataRange& find_range)
{
	// データを一部でも持っているならNotifySelfUpを送る
	if( m_self_range.contains_part(find_range) ) {
		LogDebug0( Log::format("Asked data range %1% is part of self data range %2%") % find_range % m_self_range );
		m_rpcc.ucastNotifySelfUp(from);
	} else {
		LogDebug0( Log::format("Asked data range %1% is out of self data range %2%") % find_range % m_self_range );
	}
}


namespace {
struct DuplexLess {
	inline bool operator() (const DuplexRange& lhl, const DuplexRange& rhl) const
		{ return lhl.duplex() < rhl.duplex(); }
};
struct LengthGreaterOrSame {
	LengthGreaterOrSame(DataRange::size_type length) : m_length(length) {}
	bool operator() (const DataRange& range) const
		{ return m_length <= range.length(); }
private:
	const DataRange::size_type m_length;
};
struct LengthGreater {
	bool operator() (const DuplexRange& lhl, const DuplexRange& rhl)
		{ return lhl.length() < rhl.length(); }
};

struct LogDuplex {
	void operator() (const DuplexRange& duplex) {
		LogDebug0( Log::format("  duplex %1%") % duplex );
	}
};
struct LogRange {
	void operator() (const DataRange& range) {
		LogDebug0( Log::format("  range %1%") % range );
	}
};

// store_limit == 0 || image_size == 0 は弾いた上で呼びす
DataRange determineSelfRange(const VTable& vtable, pos_type image_size, uint64_t store_limit, const NodeIdentity& self)
{
	// image_sizeがstore_limitより大きければ、全部持つ
	if( image_size <= store_limit ) {
		return DataRange(0, image_size-1);
	}

	// ここからimage_size > store_limit


	// VTableのエントリから、startとendを分離したsorted vectorを作成
	typedef VTable::duplex_type duplex_type;
	duplex_type duplex;
	vtable.calcDuplex(image_size, duplex);

	std::sort( duplex.begin(), duplex.end(), DuplexLess() );	// 多重化度が少ない順にソート
#ifndef NDEBUG
	std::for_each( duplex.begin(), duplex.end(), LogDuplex() );	// 多重化度をログに書き出す
#endif

	// 多重化度が大小の範囲群を取り出す
	duplex_type::iterator min_lower( duplex.begin() );
	if( unlikely(min_lower == duplex.end()) ) {
		throw JoinFailedException("No duplex range found");
	}
	duplex_type::iterator min_upper( std::upper_bound(min_lower, duplex.end(), *min_lower, DuplexLess()) );

	// すべてシーケンシャルに持てる範囲があるかどうか調べる
	std::vector<DataRange> candidates;
	// store_limitと同じ長さかより大きい範囲について、
	for( duplex_type::iterator more( std::find_if(min_lower, min_upper, LengthGreaterOrSame(store_limit)) );
			more != min_upper;
			more = std::find_if( (more+1), min_upper, LengthGreaterOrSame(store_limit)) ) {
		// 前から順にstore_limitごとのブロックに区切っていく（ぴったりでなければ最後に空白ができる）
		for(pos_type cur_end( more->start() + store_limit - 1 );
				cur_end < more->end();
				cur_end += store_limit ) {
			candidates.push_back( DataRange(1 + cur_end - store_limit, cur_end) );
		}
		// ここでendに吸着する範囲を追加
		candidates.push_back( DataRange(1 + more->end() - store_limit, more->end()) );
	}

	if( !candidates.empty() ) {
		LogDebug0( Log::format("Found %1% sequentially storable data ranges in the fewest duplexed ranges")
				% candidates.size() );
#ifndef NDEBUG
		std::for_each( candidates.begin(), candidates.end(), LogRange() );	// 候補をログに書き出す
#endif
		// IPアドレスの下位2バイトから分散
		// 一斉に起動された場合でも高い確率で均等に分散できる
		uint16_t rseed( *((uint16_t*)(&self.getNetRaw()[16])) );
		return candidates[rseed % candidates.size()];
	} else {
		LogDebug0("Sequentially storable data range is not found in the fewest duplexed ranges");
		// すべてシーケンシャルに持てるデータが無い
		// 最もデータ範囲が大きい範囲＋前store_limitか0に達するまで＋後ろstore_limitに達するまで
		DataRange& base( *( std::max_element(min_lower, min_upper, LengthGreater()) ) );
		LogDebug0( Log::format("Longest data range is %1%") % base );
		if( base.end() > store_limit ) {
			return DataRange(
					base.end() - store_limit + 1,
					base.end()
					);
		} else {
			return DataRange(
					0,
					store_limit - 1
					);
		}
	}
}

}  // noname namespace


void EdgeStorage::invokeJoin(StreamManager& smgr, RPCServer& rpcs, SearchController& engine, const addr46_t& bootstrap, unsigned int join_delay, unsigned short join_boost, bool exec_this_thread)
{
	if( !exec_this_thread ) {
		// 参照のバインドには問題がある
		//m_threadpool.submitTask( boost::bind(&EdgeStorage::invokeJoinIMPL, this, smgr, rpcs, engine, bootstrap) );
		m_threadpool.submitTask(
				boost::bind(&EdgeStorage::invokeJoinIMPL, this, &smgr, &rpcs, &engine, &bootstrap, join_delay, join_boost)
				);
	} else {
		invokeJoinIMPL(&smgr, &rpcs, &engine, &bootstrap, join_delay, join_boost);
	}
}

void EdgeStorage::invokeJoinIMPL(StreamManager* smgr, RPCServer* rpcs, SearchController* engine, const addr46_t* bootstrap, unsigned int join_delay, unsigned short join_boost)
{
	unsigned short retry = STREAM_JOIN_RETRY_LIMIT;	// STREAM_JOIN_RETRY_LIMIT回までリトライ
	while(1) {
	try {

		// 1. smgrにstreamClientJoin(bootstrap)を要請
		//    ここでVTableは初期化される
		//    image_idとimage_sizeがセットされる
		uint16_t image_id;
		uint64_t image_size;
		smgr->streamClientJoin(*bootstrap, m_vtable, image_id, image_size);
		if( image_size == 0 ) {
			throw JoinFailedException("Image size is 0");
		}
		m_image_id = image_id;
		m_image_size = image_size;
		LogInfo( Log::format("Stream Join succeeded, image size is %1% bytes") % m_image_size );

		// 2. RPCClientとRPCServerにimage_idを知らせる
		m_rpcc.setImageId(m_image_id);
		////m_rpcs.setImageId(m_image_id);

		// データを一切キャッシュしない場合
		if( m_store_limit == 0 ) {
			// m_self_rangeを初期Rangeで初期化する
			m_self_range = DataRange(
					std::numeric_limits<pos_type>::max(),
					std::numeric_limits<pos_type>::max()
					);

			// RPCClientにself rangeをセット
			m_rpcc.setSelfRange(m_self_range);

			// rpcNotifySelfDownのブロードキャストして、VTableから削除してもらう
			m_rpcc.bcastNotifySelfDown();

			// イメージはダウンロードしない
			break;
		}

		if( join_delay > 0 ) {
			// delayする際もイメージサイズはすぐに必要なので、
			// 最初にJoinだけした後、再度VTableを取得する
			LogDebug0( Log::format("Delaying stream join for %1% seconds...")
					% join_delay );
			sleep(join_delay);
			join_delay = 0;	// delayはリトライ時に引き継がない

			smgr->streamClientJoin(*bootstrap, m_vtable, image_id, image_size);
			if( image_size == 0 ) {
				throw JoinFailedException("Image size is 0");
			}
			m_image_id = image_id;
			m_image_size = image_size;
			LogInfo( Log::format("Stream Join succeeded, image size is %1% bytes") % m_image_size );
		}

		// 3. VTabaleを基に、自分の持つべき範囲を決める
		DataRange self_range( determineSelfRange(m_vtable, m_image_size, m_store_limit, rpcs->getSelfIdentity()) );
		LogInfo( Log::format("Storing data range is %1%") % self_range );

		// 4. SearchEngineに頼んでデータをダウンロードしてくる（this->setSelfRange()）
		setSelfRange(self_range, *engine, join_boost);	// ブロックする

		// 5. RPCClientにself rangeをセット
		// FIXME: データをダウンロードする前にセットしてしまう？
		m_self_range = self_range;
		m_rpcc.setSelfRange(m_self_range);

		// 6. rpcNotifyUpをブロードキャスト
		// FIXME: VTable全体にユニキャストにする？
		// FIXME: データをダウンロードする前に送信してしまう？
		m_rpcc.bcastNotifySelfUp();

		// 7. VTableに自分を追加
		m_vtable.rpcNotifyUp(rpcs->getSelfIdentity(), self_range);

		LogDebug( Log::format("Self data range %1% is successfully downloaded") % m_self_range );

		break;

	} catch( const std::runtime_error& ex ) {
		if( retry == 0 ) {
			invalidate();
			throw JoinFailedException(ex.what());
		} else {
			LogError( Log::format("Can't complete Stream Join: %1%") % ex.what() );
			--retry;
		}
	} catch( const std::bad_alloc& ex ) {
		if( retry == 0 ) {
			invalidate();
			throw JoinFailedException("memory error");
		} else {
			LogError("Can't complete Stream Join: memory error");
			--retry;
		}
	} catch(...) {
		if( retry == 0 ) {
			invalidate();
			throw JoinFailedException("unknown error");
		} else {
			LogError("Can't complete Stream Join: unknown error");
			--retry;
		}

	}  // try
	}  // while(1)
}


// ブロードキャストを使ってJOIN
/* TODO: 未実装
void EdgeStorage::invokeJoinBcast(const char* port, const NodeIdentity& self)
{
	lis = listenUDP6(,port);
	lis_self = NodeIdentity(self.addressString().c_str(), port);

	char buf[RPC_COMMON_HEADER_SIZE + RPC_FIND_SIZE];
	DataRange req_range(0, std::numeric_limits<pos_type>::max() - 1);
	DataRange lis_range(std::numeric_limits<pos_type>::max(), std::numeric_limits<pos_type>::max());

	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+0], req_range.getNetRaw().get(), DataRange::raw_size);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+16], lis_self.getNetRaw(), NodeIdentity::raw_size);
	::memcpy(&buf[RPC_COMMON_HEADER_SIZE+37], lis_range.getNetRaw(), DataRange::raw_size);

	int on = 1;
	if( setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &on, sizeof(on)) < 0 ) {
		throw;
	}
	NodeIdentity bcast_addr("255.255.255.255", port);
	if( sendto46(lis, buf, sizeof(buf), 0, bcast_addr) < 0 ) {
		throw
	}

	char recv_buf[256];
	recvfrom46(lis, recv_buf, sizeof(recv_buf), 0, NULL)

	::close(lis);
}
*/

}  // namespace VFIELD
