#include <sys/select.h>
#include "stream.h"
#include "ip46.h"
#include "datarange.h"
#include "storage/storage.h"
#include "vtable/vtable.h"
#include "threadpool.h"
#include "log.h"
#include "vfield.h"
#include "stdext_hash_map.h"
#include <netinet/in.h>
#include <sys/types.h>
#include <unistd.h>
#include <deque>
#include <algorithm>
#include <boost/thread.hpp>
#include <boost/bind.hpp>
#include <boost/shared_array.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/random.hpp>
#include <boost/functional/hash.hpp>

namespace VFIELD {

////
// stripped_addr46_t
//
class stripped_addr46_t {
friend struct stripped_addr46_hash;
public:
	stripped_addr46_t(const addr46_t& from);	// impicit
	~stripped_addr46_t() {}
public:
	inline bool isIPv6(void) const { return is_ipv6; }
	const struct in6_addr& getv6addr(void) const { return address.v6addr; }
	const struct in_addr   getv4addr(void) const { return address.v4addr; }
public:
	// hash_map用
	bool operator== (const stripped_addr46_t& rhl) const;
private:
	bool is_ipv6;
	typedef union {
		struct in6_addr v6addr;
		struct in_addr  v4addr;
	} address_type;
	address_type address;
private:
	stripped_addr46_t();
};
stripped_addr46_t::stripped_addr46_t(const addr46_t& from)
{
	if( from.isIPv6() ) {
		is_ipv6 = true;
		::memcpy(address.v6addr.s6_addr, from.getsin6()->sin6_addr.s6_addr, 16);
	} else {
		is_ipv6 = false;
		//::memset(&address.v6addr.s6_addr, 0, 12);
		address.v4addr.s_addr = from.getsin()->sin_addr.s_addr;
	}
}

struct stripped_addr46_hash {
	size_t operator() (const stripped_addr46_t& x) const {
		if( x.is_ipv6 ) {
			return boost::hash_range(
					//(const char*)(&x.address.v6addr),
					//(const char*)(&x.address.v6addr) + sizeof(struct in6_addr)
					(const int32_t*)(&x.address.v6addr),
					(const int32_t*)(&x.address.v6addr) + (sizeof(struct in6_addr) / sizeof(int32_t))	// sizeof(in6_addr) == 16で、4で割り切れる
					);
		} else {
			return boost::hash_value( (uint32_t)x.address.v4addr.s_addr );
			//return stdext::hash<uint32_t>()(x.address.v4addr.s_addr);
		}
	}
};

bool stripped_addr46_t::operator== (const stripped_addr46_t& rhl) const
{
	if( is_ipv6 && rhl.is_ipv6 ) {
		return ::memcmp(&address, &rhl.address, sizeof(address_type)) == 0;
	} else if( !is_ipv6 && !is_ipv6 ) {
		return address.v4addr.s_addr == rhl.address.v4addr.s_addr;
	} else {
		return false;
	}
	//return (::memcmp(&address.v6addr, &rhl.address.v6addr, 16) == 0) && (is_ipv6 == rhl.is_ipv6);
}

addr46_t::addr46_t(const stripped_addr46_t& strip)
{
	::memset(&addr_in.sin6, 0, sizeof(struct sockaddr_in6));
	if( strip.isIPv6() ) {
		addr_in.sin6.sin6_family = PF_INET6;
		addr_in.sin6.sin6_port = 0;
		::memcpy(addr_in.sin6.sin6_addr.s6_addr, strip.getv6addr().s6_addr, 16);
	} else {
		addr_in.sin.sin_family = PF_INET;
		addr_in.sin.sin_port = 0;
		addr_in.sin.sin_addr.s_addr = strip.getv4addr().s_addr;
	}
}

bool addr46_t::operator== (const addr46_t& rhl) const throw()
{
	if( isIPv6() && rhl.isIPv6() ) {
		return ::memcmp(&addr_in.sin6, &rhl.addr_in.sin6, sizeof(struct sockaddr_in6)) == 0;
	} else if( !isIPv6() && !isIPv6() ) {
		return ::memcmp(&addr_in.sin, &rhl.addr_in.sin, sizeof(struct sockaddr_in)) == 0;
	} else {
		return false;
	}
}

namespace {
struct addr46_hash {
	size_t operator() (const addr46_t& x) const
	{
		if( x.isIPv6() ) {
			return boost::hash_range(
					(const char*)(x.getsin6()),
					(const char*)(x.getsin6()) + sizeof(struct sockaddr_in6)
					);
		} else {
			return boost::hash_range(
					(const char*)(x.getsin()),
					(const char*)(x.getsin()) + sizeof(struct sockaddr_in)
					);
		}
	}
};
}  // noname namespace



////
// StripeedNodeIdentity
//
StrippedNodeIdentity::StrippedNodeIdentity(const stripped_addr46_t& strip)
{
	// common/ip46.cc NodeIdentity::NodeIdentityと連動

	::memset(m_data_part, 0, sizeof(m_data_part));
	if( strip.isIPv6() ) {
		m_data_part[0] |= NODE_IDENTITY_FLAG_IPV6;
		::memcpy(&m_data_part[1], strip.getv6addr().s6_addr, 16);
	} else {
		// m_data_part[0] |= 0;
		::memcpy(&m_data_part[13], &strip.getv4addr().s_addr, 4);
	}
}



////
// FDNotify
//
class FDNotify {
public:
	FDNotify(); 	 // throw(StreamListenException)
	~FDNotify();
public:
	int getClient(void) throw();
	void notify(void) throw();
	void reset(void) throw();
private:
	static const int READ  = 0;
	static const int WRITE = 1;
	int m_pipe[2];
};

FDNotify::FDNotify()
{
	if( ::pipe(m_pipe) < 0
			|| ::fcntl(m_pipe[0], F_SETFL, O_NONBLOCK) < 0
			|| ::fcntl(m_pipe[1], F_SETFL, O_NONBLOCK) < 0 ) {
		throw StreamListenException(errno, "Failed to initialize a pair of pipe");
	}
}
FDNotify::~FDNotify()
{
	::close(m_pipe[READ]);
	::close(m_pipe[WRITE]);
}

int FDNotify::getClient(void) throw()
{
	return m_pipe[READ];
}

void FDNotify::notify(void) throw()
{
	::write(m_pipe[WRITE], "", 1);
}

void FDNotify::reset(void) throw()
{
	char buf[128];
	::read(m_pipe[READ], buf, sizeof(buf));
}


////
// TimeoutDB
//
class TimeoutDB;
class TimeoutDBAccessor {
public:
	typedef unsigned int count_type;
	typedef stdext::hash_map<addr46_t, unsigned int, addr46_hash> timeout_db_type;
public:
	TimeoutDBAccessor(TimeoutDB& db, timeout_db_type::iterator iterator) :
		m_db(db),
		m_iterator(iterator) {}
	~TimeoutDBAccessor() {}
public:
	inline count_type get(void) throw();
	// 1つのTimeoutDBAccessorインスタンスで
	inline TimeoutDBAccessor& operator++ (void) throw();
	inline TimeoutDBAccessor& operator-- (void) throw();
private:
	TimeoutDB& m_db;
	timeout_db_type::iterator m_iterator;
private:
	TimeoutDBAccessor();
};

class TimeoutDB {
friend class TimeoutDBAccessor;
public:
	TimeoutDB(VTable& vtable, ThreadPool& threadpool) : m_threadpool(threadpool), m_vtable(vtable){}
	~TimeoutDB() {}
public:
	typedef TimeoutDBAccessor accessor_type;
public:
	inline TimeoutDBAccessor getEntry(const addr46_t& to)
	{
		// FIXME: タイムアウトカウントはコストが高すぎて効果に見合わない？
		// FIXME: timeout_db内の要素数は増える一方

		// m_timeout_dbをロック
		boost::mutex::scoped_lock lk(m_timeout_db_mutex);
		std::pair<timeout_db_type::iterator, bool> ins(
				m_timeout_db.insert( std::make_pair(to, 0) )
				);
		if( ins.second ) {
			LogDebug0( Log::format("New timeout DB entry for %1% is inserted, size of DB is %2%")
					% to
					% m_timeout_db.size() );
		} else {
			LogDebug0( Log::format("Timeout count of %1% is %2%, size of DB is %3%")
					% to
					% ins.first->second
					% m_timeout_db.size() );
		}
		return TimeoutDBAccessor(*this, ins.first);
		// m_timeout_dbをアンロック
	}
private:
	boost::mutex m_timeout_db_mutex;
	typedef TimeoutDBAccessor::timeout_db_type timeout_db_type;
	timeout_db_type m_timeout_db;

	ThreadPool& m_threadpool;
	VTable& m_vtable;
};

TimeoutDBAccessor::count_type TimeoutDBAccessor::get (void) throw()
{
	return m_iterator->second;
}
TimeoutDBAccessor& TimeoutDBAccessor::operator++ (void) throw()
{
	// TODO: 正確なカウントにはロックが必要だが、気にしない
	if( unlikely(m_iterator->second >= STREAM_NODE_DOWN_DETECT_TIMEOUT_COUNT) ) {
		// タイムアウト回数が指定回数を超過したので、
		// VTableから削除する & RPCNotifyDownOtherを送ってもらう
		LogDebugError(
				Log::format("Time out count of stream connection to %1% exceeds %2% times limits")
				% m_iterator->first
				% STREAM_NODE_DOWN_DETECT_TIMEOUT_COUNT );
		m_db.m_threadpool.submitTask(
				boost::bind(
					&VTable::strippedNodeDownDetect,
					&m_db.m_vtable,
					StrippedNodeIdentity(m_iterator->first)  // IPv4射影アドレスを同一として比較
					)
				);
		// TODO: ここでTimeoutDBから削除すると、2回目のoperator++でSegmentation Faultする
	}
	if( likely(m_iterator->second < std::numeric_limits<count_type>::max()) ) {
		++(m_iterator->second);
	}
	return *this;
}
TimeoutDBAccessor& TimeoutDBAccessor::operator-- (void) throw()
{
	// TODO: 正確なカウントにはロックが必要だが、気にしない
	if( likely(m_iterator->second > std::numeric_limits<count_type>::min()) ) {
		--(m_iterator->second);
	}
	return *this;
}



////
// StreamManagerIMPL
//
class StreamManagerIMPL {
public:
	StreamManagerIMPL(const NodeIdentity& self_identity, Storage& storage, VTable& vtable, ThreadPool& threadpool, size_t max_connection, unsigned int sweep_interval);
	~StreamManagerIMPL() throw();
public:
	inline bool isConnected(const addr46_t& node);

	// nodeにコネクションを張る
	// 接続に失敗するとfalseを返す
	inline bool tryConnect(const addr46_t& node) throw();

	// StreamManagerIMPL内部で使用する
	// 1. 新たなコネクションが接続されたとき
	// 2. AutoSockでAutoReturnが指定されたとき
	void addNewConnection(int sock, const stripped_addr46_t& from);
			// 内部で呼び出すときはデッドロック注意
public:
	// クライアント機能

	// ネゴシエーションが成立済みのソケットを得る
	// AutoSockはAutoReturn
	// ソケットはブロッキングモード（m_managedに管理されたソケットとは異なる）
	// タイムアウトすると例外を投げる
	AutoSock getReadyConnection(
			const addr46_t& to,
			const char* request_buf,
			size_t request_len,
			uint8_t ack_magic );

	// SocketClosedException、SocketErrorExceptionを投げる
	void streamClientJoin(const addr46_t& to, VTable& vtable, uint16_t& result_image_id, uint64_t& result_image_size);
public:
	// StreamManagerIMPL内部で使用する
	// 1. 相手からコネクションを切断されたとき
	// 2. AutoSocketでAutoDisconnectが指定されたとき
	// 3. デストラクタ
	void disconnectStream(int sock) throw();

	// getReadyConnection()のためのヘルパー関数
	// ネゴシエーションを開始するために、新たにコネクションを確立する
	// AutoSockはAutoDisconnect
	// ソケットは、ブロッキングモードそうでないか"分からない"
	// 接続に失敗すると例外を投げる
	int makeNewConnectionForSend(const addr46_t& to);
private:
	typedef stdext::hash_multimap<stripped_addr46_t, int, stripped_addr46_hash> managed_type;
	managed_type m_managed;
	boost::mutex m_managed_mutex;
private:
	ThreadPool& m_threadpool;
	Storage& m_storage;
	VTable& m_vtable;

	FDNotify m_thread_notify;
	void StreamWorker(int sock, stripped_addr46_t& from, boost::shared_array<char> init_buf, ssize_t len);

	int m_listen_sock;

	TimeoutDB m_timeout_db;

	void ManagingThread(void);
	void stopManagingThread(void);

	void SweepingThread(size_t max_connection, unsigned int interval);
	void stopSweepingThread(void);

	boost::mutex m_sweep_thread_mutex;
	boost::condition m_sweep_thread_cond;

	volatile bool m_end_flag;

	boost::thread m_managing_thread;	// スレッドは必ず最後に初期化
	boost::thread m_sweeping_thread;	// スレッドは必ず最後に初期化
};


////
// AutoSockFinalizer
//
namespace {
class AutoReturn : public AutoSockFinalizer {
public:
	AutoReturn(const stripped_addr46_t& addr, StreamManagerIMPL& smgr) :
		m_smgr(smgr), m_addr(addr) {}
	void operator() (int sock) throw()
	{
		try {
			m_smgr.addNewConnection(sock, m_addr);
		} catch (const std::runtime_error& ex) {
			LogDebugError(ex.what());
			::close(sock);
		} catch (...) {
			LogDebugError("Unknown error has been occurred while adding returning connection to manager");
			::close(sock);
		}
	}
private:
	StreamManagerIMPL& m_smgr;
	stripped_addr46_t m_addr;
};
class AutoDisconnect : public AutoSockFinalizer {
public:
	AutoDisconnect(StreamManagerIMPL& smgr) :
		m_smgr(smgr) {}
	void operator() (int sock) throw()
	{
		// disconnectStream()は例外を投げない
		m_smgr.disconnectStream(sock);
	}
private:
	StreamManagerIMPL& m_smgr;
};
}  // noname namespace



// Constructor / Destructor
StreamManager::StreamManager(const NodeIdentity& self_identity, Storage& storage, VTable& vtable, ThreadPool& threadpool, size_t max_connection, unsigned int sweep_interval) :
	impl( new StreamManagerIMPL(self_identity, storage, vtable, threadpool, max_connection, sweep_interval) ) {}
StreamManager::~StreamManager() {}

StreamManagerIMPL::StreamManagerIMPL(const NodeIdentity& self_identity, Storage& storage, VTable& vtable, ThreadPool& threadpool, size_t max_connection, unsigned int sweep_interval) :
	m_threadpool(threadpool),
	m_storage(storage),
	m_vtable(vtable),
	m_listen_sock(-1),
	m_timeout_db(m_vtable, m_threadpool),
	m_end_flag(false),
	m_managing_thread( boost::bind(&StreamManagerIMPL::ManagingThread, this) ),
	m_sweeping_thread( boost::bind(&StreamManagerIMPL::SweepingThread, this, max_connection, sweep_interval) )
{
	if( self_identity.isIPv6() ) {
		m_listen_sock = listenTCP6(self_identity.getIPv6Address(), self_identity.getPortHostByteOrder());
	} else {
		m_listen_sock = listenTCP4(self_identity.getIPv4Address(), self_identity.getPortHostByteOrder());
	}
	if( m_listen_sock < 0 || ::fcntl(m_listen_sock, F_SETFL, O_NONBLOCK) < 0 ) {
		int err = errno;
		m_end_flag = true;
		m_managing_thread.join();
		throw StreamListenException(err, "Can't listen Stream");
	}
	LogInfo( Log::format("Stream is listened on %1%:%2%") % addr46_t(self_identity) % self_identity.getPortHostByteOrder() );
	LogDebug( Log::format("Stream listen socket: %1%") % m_listen_sock );
}

StreamManagerIMPL::~StreamManagerIMPL() throw()
{
	try {
		stopManagingThread();
		stopSweepingThread();

		// すべてjoinしないと、子スレッドが持っているリファレンスが無効になって、
		// セグメンテーションフォールトする
		m_threadpool.joinThreads();

	} catch (const std::runtime_error& ex) {
		LogError( Log::format("Can't stop stream managing thread regularly: %1%") % ex.what() );
	} catch (...) {
		LogError("Can't stop stream managing thread regularly");
	}
}



// public:
bool StreamManager::isConnected(const addr46_t& node) { return impl->isConnected(node); }
bool StreamManagerIMPL::isConnected(const addr46_t& node)
{
	if( m_managed.find(node) != m_managed.end() ) {
		return true;
	} else {
		return false;
	}
}

void StreamManager::addNewConnection(int sock, const addr46_t& from) { return impl->addNewConnection(sock, from); }
void StreamManagerIMPL::addNewConnection(int sock, const stripped_addr46_t& from)
{
	if( unlikely( ::fcntl(sock, F_SETFL, O_NONBLOCK) < 0 ) ) {
		throw SocketErrorException(errno, "Invalid stream socket is detected");
	}
	LogDebug0( Log::format("New managed connection %1%: %2%") % addr46_t(from) % sock );
	// m_managedをロック
	boost::mutex::scoped_lock lk(m_managed_mutex);
	m_managed.insert( managed_type::value_type(from, sock) );
	m_thread_notify.notify();
	// m_managedをアンロック
	LogDebug0( Log::format("...Connection insert ok %1%: %2%") % addr46_t(from) % sock );
}

bool StreamManager::tryConnect(const addr46_t& node) throw() { return impl->tryConnect(node); }
bool StreamManagerIMPL::tryConnect(const addr46_t& node) throw()
{
	if( isConnected(node) ) return true;
	try {
		int new_sock = makeNewConnectionForSend(node);
		// makeNewConnectionForSendは、返ってきたソケットが
		// ブロッキングモードかそうでないかは"分からない"が、
		// addNewConnection時にBlockingモードにされる
		addNewConnection(new_sock, node);
		return true;
	} catch (...) {
		// 例外の種類は、どれにしても処理は同じなので気にしない
		//asock.close();
		return false;
	}
}

// Basic helper functions
int StreamManagerIMPL::makeNewConnectionForSend(const addr46_t& to)
{
	// AutoSockのFinalizerはAutoReturn

	TimeoutDB::accessor_type timeout_count( m_timeout_db.getEntry(to) );

	LogDebug0(
		Log::format("Making new connection with %1% sec %2% usec time out")
		% (STREAM_FAST_TIMEOUT_SEC  * (timeout_count.get() + 1))
		% (STREAM_FAST_TIMEOUT_USEC * (timeout_count.get() + 1))
		);
	int new_sock = timed_connectTCP46(
			to,
			STREAM_FAST_TIMEOUT_SEC  * (timeout_count.get() + 1),
			STREAM_FAST_TIMEOUT_USEC * (timeout_count.get() + 1)
			);

	if( likely(new_sock > 0) ) {
		// 接続成功
		// connectではタイムアウトカウントを減らさない
		//--timeout_count;
		// タイムアウト時間をセット
		if( unlikely( setTimeout(
					new_sock,
					STREAM_SOCKET_SEND_TIMEOUT_SEC, STREAM_SOCKET_SEND_TIMEOUT_USEC,
					STREAM_SOCKET_RECV_TIMEOUT_SEC, STREAM_SOCKET_RECV_TIMEOUT_USEC
			      ) < 0 ) ) {
			int err = errno;
			disconnectStream(new_sock);
			throw SocketErrorException(err, "Making new stream connection timed out");
		} else {
			LogDebug( Log::format("Stream connection to %1% succeeded: %2%") % to % new_sock );
			return new_sock;
		}

	} else if( new_sock == 0 ) {
		// タイムアウト
		++timeout_count;
		throw SocketErrorException(ETIMEDOUT, "Making new stream connection timed out");

	} else {	// new_sock < 0
		// 自分側のエラーで接続失敗
		throw SocketErrorException(-new_sock, "Failed to make new stream connection");
		// RPCNotifyDownは送らない（相手が存在するかどうか不明）
	}
}

namespace {
void disconnectStreamThread(int sock)
{
	LogDebug( Log::format("Disconnecting stream socket: %1%") % sock );
	char buf[STREAM_HEADER_SIZE + STREAM_CLOSE_SIZE];
	buf[0] = STREAM_CLOSE_MAGIC;

	// すでに閉じている可能性がある
	if( ::fcntl(sock, F_SETFL, O_NONBLOCK) < 0 ) {
		::close(sock);				// エラーは無視
		return;
	}

	// XXX: タイムアウト時間が適当
	unsigned short retry = 5;
	while(retry > 0) {
		if( ::write(sock, buf, sizeof(buf)) < 0 && (errno == EAGAIN || errno == EINTR) ) {
			--retry;
			sleep(2);
		} else {
			break;
		}
	}

	::close(sock);	// エラーは無視
	return;
}
}  // noname namespace

void StreamManagerIMPL::disconnectStream(int sock) throw()
{
	// 例外を投げない
	// ブロックしない
	m_threadpool.submitTask( boost::bind(&disconnectStreamThread, sock) );
}

void StreamManager::streamClientJoin(const addr46_t& to, VTable& vtable, uint16_t& result_image_id, uint64_t& result_image_size)
	{ return impl->streamClientJoin(to, vtable, result_image_id, result_image_size); }
void StreamManagerIMPL::streamClientJoin(const addr46_t& to, VTable& vtable, uint16_t& result_image_id, uint64_t& result_image_size)
{
	LogDebug( Log::format("Requesting Stream Join to %1%") % to );

	// リクエストバッファを作成
	char reqbuf[STREAM_HEADER_SIZE + STREAM_JOIN_SIZE];
	reqbuf[0] = STREAM_JOIN_MAGIC;

	boost::scoped_ptr<AutoSock> asock;
	char ackhead[STREAM_JOIN_ACK_HEADER_SIZE];

	// コネクションを開始＆ネゴシエーション
	asock.reset( new AutoSock(getReadyConnection(to, reqbuf, sizeof(reqbuf), STREAM_JOIN_ACK_MAGIC)) );

	// ネゴシエーション成立

	// Stream Join Ack Headerを受信
	if( unlikely( asock->read(ackhead, sizeof(ackhead)) < sizeof(ackhead) ) ) {
		// 受信したデータが足りない時は、ソケットを閉じて再試行
		// 突然切断するので相手のVTableから自分が削除されるかもしれないが、
		// JOINの最中なので問題ない
		asock->close();
		throw SocketErrorException(EINTR, "Received stream join ack header is too small");
	}

	// ヘッダの受信に成功
	result_image_id   = ntohs ( *((uint16_t*)&ackhead[0]) );
	result_image_size = ntohll( *((uint64_t*)&ackhead[2]) );
	uint32_t table_size = ntohl ( *((uint32_t*)&ackhead[10]) );

	LogDebug( Log::format("Stream Join Ack Header: ImageID = %1%,  ImageSize = %2%,  TableSize = %3%")
			% result_image_id
			% result_image_size
			% table_size );

	// VTableを初期化
	vtable.setImageSize(result_image_size);

	// テーブルの受信を開始
	char buffer[STREAM_BUFFER_SIZE + RPC_NOTIFY_UP_SIZE-1];
		// RPC_NOTIFY_UP_SIZE-1は、前回受信した分の端数の繰越分のスペース
	uint64_t accum = 0;
	uint32_t carry_size = 0;
	uint32_t goal;
	uint32_t pos;
	uint32_t rl;
	while( accum < (table_size * RPC_NOTIFY_UP_SIZE) ) {
		rl = asock->read(buffer + carry_size, STREAM_BUFFER_SIZE);

		accum += rl;
			// 継続条件: rl + carry_size - pos >= RPC_NOTIFY_UP_SIZE
			//     移項: rl + carry_size - RPC_NOTIFY_UP_SIZE >= pos
			//     左右: pos <= rl + carry_size - RPC_NOTIFY_UP_SIZE
		goal = rl + carry_size - RPC_NOTIFY_UP_SIZE;
		for(pos=0; pos <= goal; pos += RPC_NOTIFY_UP_SIZE ) {
			NodeIdentity up_node( &buffer[pos] );
			DataRange up_range( &buffer[pos+19] );
			vtable.rpcNotifyUp(up_node, up_range);
		}
		if( pos <= (rl + carry_size) ) {
			// 端数が発生
			carry_size = rl + carry_size - pos;
			::memcpy(buffer, buffer+STREAM_BUFFER_SIZE, carry_size);
		}
	}

	LogDebug( Log::format("V-Table is successfully downloaded from %1%") % to );

	// getReadyConnectionはAutoReturnのAutoSockを返すので、asockは自動的に返却される
}



////
// getReadyConnection
//
// Active stream negotiation
// AutoReturnのAutoSockを返す
namespace {
struct StreamNegotiationConflictsException : public std::runtime_error {
	StreamNegotiationConflictsException() :
		std::runtime_error("") {}
};
inline void streamNegotiation(
		AutoSock& asock,
		const char* request_buf,
		size_t request_len,
		uint8_t ack_magic,
		TimeoutDB::accessor_type& timeout_count
		)
{
	// *** 例外を投げた（ネゴシエーションに失敗した）ときはソケットを閉じないといけない ***
	// 例外は呼び出し元でキャッチ

	// request_bufを送信
	// タイムアウトするとSocketTimedOutExceptionがthrowされる
	LogDebug0(
		Log::format("Stream negotiationg with %1% sec %2% usec time out...")
		% (STREAM_FAST_TIMEOUT_SEC  * (timeout_count.get() + 1))
		% (STREAM_FAST_TIMEOUT_USEC * (timeout_count.get() + 1))
		);
	asock.timed_write(
			request_buf,
			request_len,
			STREAM_FAST_TIMEOUT_SEC * (timeout_count.get() + 1),
			STREAM_FAST_TIMEOUT_USEC * (timeout_count.get() + 1)
			);


	// パケット受信
	// タイムアウトするとSocketTimedOutExceptionがthrowされる
	LogDebug0(
		Log::format("Waiting Stream negotiation response with %1% sec %2% usec time out...")
		% (STREAM_FAST_TIMEOUT_SEC  * (timeout_count.get() + 1))
		% (STREAM_FAST_TIMEOUT_USEC * (timeout_count.get() + 1))
		);
	char negbuf[STREAM_HEADER_SIZE];
	if( unlikely( asock.timed_read(
			negbuf,
			sizeof(negbuf),
			STREAM_FAST_TIMEOUT_SEC * (timeout_count.get() + 1),
			STREAM_FAST_TIMEOUT_USEC * (timeout_count.get() + 1)
			) != sizeof(negbuf) ) ) {
		// コンフリクトしたソケットは、状態が不定になるので使いものにならない
		asock.close();
		throw StreamNegotiationConflictsException();
	}

	if( negbuf[0] != ack_magic ) {
		// コンフリクトしたソケットは、状態が不定になるので使いものにならない
		asock.close();
		throw StreamNegotiationConflictsException();
	}

	// ネゴシエーション成功
	LogDebug("...Stream negotiation succeeded");
}
}  // noname namespace

AutoSock StreamManager::getReadyConnection(
		const addr46_t& to,
		const char* request_buf,
		size_t request_len,
		uint8_t ack_magic ) { return impl->getReadyConnection(to, request_buf, request_len, ack_magic); }
AutoSock StreamManagerIMPL::getReadyConnection(
		const addr46_t& to,
		const char* request_buf,
		size_t request_len,
		uint8_t ack_magic )
{
	unsigned short retry = 0;
	boost::scoped_ptr<AutoSock> asock;
	TimeoutDB::accessor_type timeout_count( m_timeout_db.getEntry(to) );
	do {
		// m_managedをロック
		boost::mutex::scoped_lock lk(m_managed_mutex);
		managed_type::iterator found( m_managed.find(to) );
		try {
			if( found != m_managed.end() ) {
				// m_managedの中に入っているので、それを使う
				// asockに既にソケットが入っていた場合は、ここでFinalizer(AutoReturn)が呼ばれる
				asock.reset( new AutoSock(
							makeAutoSock(
								found->second,
								AutoDisconnect(*this)
								)
							) );
				// m_managedの中から削除し、ManagingThreadにread()されないようにする
				m_managed.erase(found);
				lk.unlock();	// m_managedをアンロック

				// ソケットのO_NONBLOCKを解除
				asock->setBlock();

			} else {
				lk.unlock();	// m_managedをアンロック
				// 新しく接続を確立
				// 接続に失敗した場合は例外が投げられる
					// makeNewConnectionForSendは、返ってきたソケットが
					// ブロッキングモードかそうでないかは"分からない"
				int new_sock = makeNewConnectionForSend(to);
				asock.reset( new AutoSock( makeAutoSock(new_sock, AutoDisconnect(*this)) ) );
				asock->setBlock();
			}

			streamNegotiation(*asock, request_buf, request_len, ack_magic, timeout_count);
			// エラーが起きたら例外が投げられる

			// リトライループを抜ける
			break;

		// FIXME: リトライ回数の精査が必要
		// asockはデストラクタでcloseされる
		} catch (const SocketTimedOutException& ex) {
			// タイムアウト
			LogDebugError("...Stream negotiation failed: timed out");
			++timeout_count;
			// if( asock ) asock->close();	// 初回の試行でmakeAutoSockがタイムアウトすると、asockがNULLになる
			// AutoSock::read/write()でcloseされる
			if( retry >= STREAM_NEGOTIATION_RETRY_LIMIT_TIMEOUT )
				throw StreamConnectFailedException("Stream negotiation timed out");
			++retry;

		} catch (const SocketClosedException& ex) {
			// 接続先が明示的に接続を切断
			LogDebugError("...Stream negotiation failed: socket is closed by remote host");
			if( retry >= STREAM_NEGOTIATION_RETRY_LIMIT_SOCKET_CLOSED )
				throw StreamConnectFailedException("socket is closed by remote host");
			++retry;

		} catch (const StreamNegotiationConflictsException&) {
			// ネゴシエーションが衝突
			LogDebugError("...Stream negotiation failed: command conflicts");
			if( retry >= STREAM_NEGOTIATION_RETRY_LIMIT_CONFLICT )
				throw StreamConnectFailedException("Stream negotiation conflicts");
			++retry;

		} catch (const SocketErrorException& ex) {
			LogDebugError("...Stream negotiation failed: socket error");
			if( retry >= STREAM_NEGOTIATION_RETRY_LIMIT_SOCKET_ERROR )
				throw StreamConnectFailedException(ex.what());
			++retry;

		} catch (const std::runtime_error& ex) {
			LogDebugError( Log::format("...Stream negotiation failed: %1%") % ex.what() );
			if( retry >= STREAM_NEGOTIATION_RETRY_LIMIT_RUNTIME_ERROR )
				throw StreamConnectFailedException(ex.what());
			++retry;

		} catch (...) {
			LogDebugError("...Stream negotiation failed: unknown error");
			if( retry >= STREAM_NEGOTIATION_RETRY_LIMIT_UNKNOWN_ERROR )
				throw StreamConnectFailedException();
			++retry;
		}

		// asockはFinalizerでcloseされる
	} while(1);

	// asockはAutoDisconnect
	// AutoReturnにして返す
	asock->setFinalizer( AutoReturn(to, *this) );

	// 接続できたので、タイムアウトカウントを減らす
	--timeout_count;

	return *asock;
}


// Passive stream negotiation
void StreamManagerIMPL::StreamWorker(int sock, stripped_addr46_t& from, boost::shared_array<char> init_buf, ssize_t len)
{
	AutoSock asock( makeAutoSock(sock, AutoReturn(from, *this)) );

	switch(init_buf[0]) {
	////
	// Stream Get Data
	//
	case STREAM_GET_DATA_MAGIC:
		if( len - STREAM_HEADER_SIZE < STREAM_GET_DATA_SIZE ) {
			LogDebugError("Receive invalid Stream Get Data (too small request size)");
		} else {
			try {
				DataRange req_range(&init_buf[STREAM_HEADER_SIZE+0]);

				LogDebug(
					Log::format("Receive Stream Get Data %1% (%2% bytes)")
					% req_range % req_range.length() );

				m_storage.streamGetData(asock, req_range);

				LogDebug(
					Log::format("...Stream Get Data %1% processed")
					% req_range );

			} catch (const std::runtime_error& ex) {
				LogWarn(
					Log::format("An error occurred while proceseding Stream Get Data: %1%")
					% ex.what() );
			} catch (...) {
				LogWarn("An error occurred while proceeding Stream Get Data");
			}
		}
		break;

	////
	// Stream Join
	//
	case STREAM_JOIN_MAGIC:
		if( len - STREAM_HEADER_SIZE < STREAM_JOIN_SIZE ) {
			LogDebugError("Receive invalid Stream Join (too small request size)");
		} else {
			try {
				LogDebug0("Receive Stream Join");

				m_vtable.streamJoin(asock, m_storage.getImageId(), m_storage.getImageSize());

			} catch (const std::runtime_error& ex) {
				LogWarn(
					Log::format("An error occurred while proceeding Stream Join: %1%")
					% ex.what() );
			} catch (...) {
				LogWarn("An error occurred while proceeding Stream Join");
			}
		}
		break;

	////
	// Stream Close
	//
	case STREAM_CLOSE_MAGIC:
		if( len - STREAM_HEADER_SIZE < STREAM_CLOSE_SIZE ) {
		} else {
			try {
				LogDebug0("Receive Stream Close");

				// VTableに残したままソケットを閉じる
				asock.close();

			} catch (const std::runtime_error& ex) {
				LogWarn(
					Log::format("An error occurred while proceeding Stream Close: %1%")
					% ex.what() );
			} catch (...) {
				LogWarn("An error occurred while proceeding Stream Close");
			}
		}
		break;

	default:
		LogDebugError( Log::format("Receive unknown Stream (number 0x%x)") % (short)init_buf[0] );
	}
}


// for_eachの関数用
namespace {
class CloseManagedSocket {
public:
	CloseManagedSocket(StreamManagerIMPL& smgr) :
		m_smgr(smgr) {}
	void operator() (const std::pair<stripped_addr46_t, int>& mng) {
		m_smgr.disconnectStream(mng.second);
	}
private:
	StreamManagerIMPL& m_smgr;
};
class SetFDSet {
public:
	SetFDSet(fd_set* pfds, int& nfds) : m_pfds(pfds), m_nfds(nfds) {}
	inline void operator() (const std::pair<stripped_addr46_t, int>& pair)
	{
		FD_SET(pair.second, m_pfds);
		if( m_nfds < pair.second ) {
			m_nfds = pair.second;
		}
	}
private:
	fd_set* m_pfds;
	int& m_nfds;
};
struct StreamManagingErrorException : public SystemCallException {
	StreamManagingErrorException(int errno_, const std::string& message) :
		SystemCallException(errno_, message) {}
};
}  // noname namespace

// Stream managing thread
void StreamManagerIMPL::ManagingThread(void)
{
	// コンストラクタでm_listen_sockが初期化されるまでend_flagを確認しながら待つ
	while( m_listen_sock < 0 ) {
		::usleep(100*1000);
		if( m_end_flag ) return;
	}

	LogDebug("Start stream managing thread");
	fd_set fds;
	int nfds;
	int ret_select;


	unsigned short retry = STREAM_MANAGER_RETRY_LIMIT;
	while(1) {  // retry loop
	try {


	while(!m_end_flag) {
		FD_ZERO(&fds);

		// fdsを作成
		// notifyをlisten_sockを最初に入れる
		nfds = std::max( m_thread_notify.getClient(), m_listen_sock );
		FD_SET(m_thread_notify.getClient(), &fds);
		FD_SET(m_listen_sock, &fds);
		{  // m_managedをロック
			boost::mutex::scoped_lock lk(m_managed_mutex);
			LogDebug0( Log::format("Number of managed sockets is %1%") % m_managed.size() );

			std::for_each(m_managed.begin(), m_managed.end(), SetFDSet(&fds, nfds));

			m_thread_notify.reset();	// notifyはロックの中でリセット
		}  // m_managedをアンロック

		// select()
		LogDebug0("Waiting for stream");
		ret_select = ::select(nfds+1, &fds, NULL, NULL, NULL);
		LogDebug0( Log::format("Stream select() wake up: %1%") % ret_select );

		if( unlikely( ret_select < 0 ) ) {
			if( m_end_flag ) break;
			if( errno == EINTR) continue;	// シグナルを受け取った
			// ENOMEMかEINVALかEBADF
			throw StreamManagingErrorException(errno, "An error is occurred while waiting Stream");
		}

		// listen_sockをチェック
		if( FD_ISSET(m_listen_sock, &fds) ) {
			addr46_t from;
			int new_sock = accept46(m_listen_sock, &from);
			if( likely(new_sock > 0) ) {
				LogDebug( Log::format("New stream is connected from %1%") % from);
				try {
					// addNewConnectionは自分でロックを取得するので、
					// ここでロックを取得するとデッドロックする
					addNewConnection(new_sock, from);

					// acceptしたソケットはそのままデータが送られてきている可能性が高い
					// たとえデータが送られてきていなくても、EAGAINをチェックしてるので大丈夫
					FD_SET(new_sock, &fds);
				} catch (const SocketErrorException& ex) {
					LogWarn( Log::format("Failed to accept new socket: %1%") % ex.what() );
				}
			} else if( new_sock < 0 && (errno == EAGAIN || errno == EINTR) ) {
				// 何もしない、単に無視
			} else if( new_sock < 0 && errno != EAGAIN && errno != EINTR ) {
				if( m_end_flag ) break;
				throw StreamManagingErrorException(errno, "Can't accept new stream socket");
			} else { // new_sock == 0
				if( m_end_flag ) break;
				// m_end_flagがtrueで無いのにソケットがクローズされた
				throw StreamManagingErrorException(errno, "Stream listen sock is closed unexpectedly");
			}
		}

		// m_managedをチェック
		{  // m_managedをロック
			boost::mutex::scoped_lock lk(m_managed_mutex);

			// m_managedでチェックする
			for(managed_type::iterator mng( m_managed.begin() ), mng_end(m_managed.end());
					mng != mng_end; ) {
				if( FD_ISSET(mng->second, &fds) ) {
					boost::shared_array<char> init_buf(new char[STREAM_BUFFER_SIZE]);
					ssize_t len = ::read(mng->second, init_buf.get(), STREAM_BUFFER_SIZE);
					if( likely(len > 0) ) {
						LogDebug0(
							Log::format("Stream data from %1% (%2% bytes)")
							% addr46_t(mng->first)
							% len );
						// ここでAutoReturn(addNewConnection)で初期化したAutoSockを作ると、この
						// ブロックの中でデストラクトされたときにデッドロックするので、
						// StreamWorkerには生のソケットを渡す
						m_threadpool.submitTask(
								boost::bind(
									&StreamManagerIMPL::StreamWorker,
									this,
									mng->second,
									mng->first,
									init_buf,
									len
									)
								);
						m_managed.erase(mng++);
					} else if( len == 0 ) {
						// ソケットが突然クローズされた
						LogDebug( Log::format("Stream connection from %1% is closed") % addr46_t(mng->first) );
						disconnectStream(mng->second); // closeを予約してから削除
						// VTableから削除する & RPCNotifyDownOtherを送ってもらう
						m_threadpool.submitTask(
								boost::bind(
									&VTable::strippedNodeDownDetect,
									&m_vtable,
									mng->first
									)
								);
						// 最後にコンテナから削除
						m_managed.erase(mng++);
					} else if( len < 0 && errno != EAGAIN && errno != EINTR ) {
						// ソケットが突然異常にクローズされた
						LogDebugError( Log::format("Stream connection from %1% is closed by error") % addr46_t(mng->first) );
						disconnectStream(mng->second); // closeを予約してから削除
						// エラーはこちら側で起こったかもしれない
						// FIXME: VTableから削除する？

						// 最後にコンテナから削除
						m_managed.erase(mng++);
					} else {  // len < 0 && (errno == EAGAIN || errno == EINTR)
						++mng;
					}
				} else {
					++mng;
				}
			}
		}  // m_managedをアンロック
	}  // while(!m_end_flag) {}
	break;  // retry loopを抜ける


	}  // try
	catch (const std::runtime_error& ex) {
		LogWarn(Log::format("An error is occurred while proceeding managed socket: %1%")
			% ex.what() );
	} catch (const std::bad_alloc& ex) {
		LogWarn("Memory error is occurred while proceeding managed socket");
	} catch (...) {
		LogWarn("Unknown error is occurred while proceeding managed socket");
	}

	// すべてのソケットをクローズし、m_managedをクリア
	LogDebug("Closing all managed sockets");
	std::for_each(m_managed.begin(), m_managed.end(), CloseManagedSocket(*this));
	m_managed.clear();
	// FIXME: listen_sockも再初期化が必要

	--retry;

	if( retry <= 0 ) {
		LogWarn("Can't recover stream managing thread");
		// FIXME: main()スレッドにエラーを知らせる方法が無い
		exit(92);
	}
	LogWarn("Restarting stream managing thread");

	}  // retry loop
}

void StreamManagerIMPL::stopManagingThread(void)
{
	LogDebug("Stopping Stream listen thread");
	m_end_flag = true;

	LogDebug("Closing all managed sockets");
	std::for_each(m_managed.begin(), m_managed.end(), CloseManagedSocket(*this));

	m_thread_notify.notify();

	if( ::close(m_listen_sock) < 0 ) {
		throw StreamManagingErrorException(errno, "An error is occurred while closing stream listen socket");
	}
	m_managing_thread.join();

	LogInfo("Stream listen socket is closed");
}

void StreamManagerIMPL::SweepingThread(size_t max_connection, unsigned int interval)
{
	LogDebug0( Log::format("Start stream sweeping thread with %1% connection allowed, %2% fuzzy coefficient, %3% seconds interval")
			% max_connection
			% STREAM_MANAGER_FUZZY_COEFFICIENT
			% interval );

	size_t fuzzy_limit = static_cast<size_t>(max_connection * (STREAM_MANAGER_FUZZY_COEFFICIENT + 1.0));
	boost::xtime xt;
	boost::mt19937 rgen;	// 乱数系列
	while( !m_end_flag ) {
		{  // m_managedをロック
			boost::mutex::scoped_lock lk(m_sweep_thread_mutex);
			LogDebug0( Log::format("Stream sweeping thread, number of managed sockets is %1%")
					% m_managed.size() );

			while( m_managed.size() > fuzzy_limit ) {
				// ランダムな個数分だけ進めて削除
				// [0 , m_manged.size()-1] の範囲からランダムに数字を取り出す
				boost::variate_generator< boost::mt19937, boost::uniform_int<> >
					rand( rgen, boost::uniform_int<>(0,m_managed.size()-1) );

				managed_type::iterator target(m_managed.begin());
				std::advance(target, rand());

				LogDebug( Log::format("Stream sweep socket %1%, number of managed sockets is %2%")
						% target->second
						% (m_managed.size() - 1) );
				disconnectStream(target->second);
				m_managed.erase(target);
			}
		}  // m_managedをアンロック

		boost::xtime_get(&xt, boost::TIME_UTC);
		xt.sec += interval;
		{  // タイムアウトまで待つ
			boost::mutex::scoped_lock lk(m_sweep_thread_mutex);
			// シグナルを受け取ったときの対応は適当
			m_sweep_thread_cond.timed_wait(lk, xt);
		}
	}
}

void StreamManagerIMPL::stopSweepingThread(void)
{
	LogDebug("Stopping Stream sweeping thread");
	// stopManagingThreadが先に呼ばれているので、既にm_end_flag == true
	boost::mutex::scoped_lock lk(m_sweep_thread_mutex);
	m_sweep_thread_cond.notify_all();
}


}  // namespace VFIELD
