/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "avifmt.h"
#include "gsrfmt.h"


/******************************************************************************
*                                                                             *
* ja:AVI基本                                                                  *
*                                                                             *
******************************************************************************/
const guint8 rgb2[] = {0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00};
const guint8 rgb16[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00,
        0x00, 0xff, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00,
        0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
        0xff, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0xff, 0x00, 0x00,
        0xff, 0xff, 0xff, 0x00};
const guint8 rgb256[] = {
        0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00,
        0xff, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x55, 0x24, 0x00, 0x00,
        0xaa, 0x24, 0x00, 0x00, 0xff, 0x24, 0x00, 0x00, 0x00, 0x49, 0x00, 0x00,
        0x55, 0x49, 0x00, 0x00, 0xaa, 0x49, 0x00, 0x00, 0xff, 0x49, 0x00, 0x00,
        0x00, 0x6d, 0x00, 0x00, 0x55, 0x6d, 0x00, 0x00, 0xaa, 0x6d, 0x00, 0x00,
        0xff, 0x6d, 0x00, 0x00, 0x00, 0x92, 0x00, 0x00, 0x55, 0x92, 0x00, 0x00,
        0xaa, 0x92, 0x00, 0x00, 0xff, 0x92, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x00,
        0x55, 0xb6, 0x00, 0x00, 0xaa, 0xb6, 0x00, 0x00, 0xff, 0xb6, 0x00, 0x00,
        0x00, 0xdb, 0x00, 0x00, 0x55, 0xdb, 0x00, 0x00, 0xaa, 0xdb, 0x00, 0x00,
        0xff, 0xdb, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x55, 0xff, 0x00, 0x00,
        0xaa, 0xff, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00,
        0x55, 0x00, 0x24, 0x00, 0xaa, 0x00, 0x24, 0x00, 0xff, 0x00, 0x24, 0x00,
        0x00, 0x24, 0x24, 0x00, 0x55, 0x24, 0x24, 0x00, 0xaa, 0x24, 0x24, 0x00,
        0xff, 0x24, 0x24, 0x00, 0x00, 0x49, 0x24, 0x00, 0x55, 0x49, 0x24, 0x00,
        0xaa, 0x49, 0x24, 0x00, 0xff, 0x49, 0x24, 0x00, 0x00, 0x6d, 0x24, 0x00,
        0x55, 0x6d, 0x24, 0x00, 0xaa, 0x6d, 0x24, 0x00, 0xff, 0x6d, 0x24, 0x00,
        0x00, 0x92, 0x24, 0x00, 0x55, 0x92, 0x24, 0x00, 0xaa, 0x92, 0x24, 0x00,
        0xff, 0x92, 0x24, 0x00, 0x00, 0xb6, 0x24, 0x00, 0x55, 0xb6, 0x24, 0x00,
        0xaa, 0xb6, 0x24, 0x00, 0xff, 0xb6, 0x24, 0x00, 0x00, 0xdb, 0x24, 0x00,
        0x55, 0xdb, 0x24, 0x00, 0xaa, 0xdb, 0x24, 0x00, 0xff, 0xdb, 0x24, 0x00,
        0x00, 0xff, 0x24, 0x00, 0x55, 0xff, 0x24, 0x00, 0xaa, 0xff, 0x24, 0x00,
        0xff, 0xff, 0x24, 0x00, 0x00, 0x00, 0x49, 0x00, 0x55, 0x00, 0x49, 0x00,
        0xaa, 0x00, 0x49, 0x00, 0xff, 0x00, 0x49, 0x00, 0x00, 0x24, 0x49, 0x00,
        0x55, 0x24, 0x49, 0x00, 0xaa, 0x24, 0x49, 0x00, 0xff, 0x24, 0x49, 0x00,
        0x00, 0x49, 0x49, 0x00, 0x55, 0x49, 0x49, 0x00, 0xaa, 0x49, 0x49, 0x00,
        0xff, 0x49, 0x49, 0x00, 0x00, 0x6d, 0x49, 0x00, 0x55, 0x6d, 0x49, 0x00,
        0xaa, 0x6d, 0x49, 0x00, 0xff, 0x6d, 0x49, 0x00, 0x00, 0x92, 0x49, 0x00,
        0x55, 0x92, 0x49, 0x00, 0xaa, 0x92, 0x49, 0x00, 0xff, 0x92, 0x49, 0x00,
        0x00, 0xb6, 0x49, 0x00, 0x55, 0xb6, 0x49, 0x00, 0xaa, 0xb6, 0x49, 0x00,
        0xff, 0xb6, 0x49, 0x00, 0x00, 0xdb, 0x49, 0x00, 0x55, 0xdb, 0x49, 0x00,
        0xaa, 0xdb, 0x49, 0x00, 0xff, 0xdb, 0x49, 0x00, 0x00, 0xff, 0x49, 0x00,
        0x55, 0xff, 0x49, 0x00, 0xaa, 0xff, 0x49, 0x00, 0xff, 0xff, 0x49, 0x00,
        0x00, 0x00, 0x6d, 0x00, 0x55, 0x00, 0x6d, 0x00, 0xaa, 0x00, 0x6d, 0x00,
        0xff, 0x00, 0x6d, 0x00, 0x00, 0x24, 0x6d, 0x00, 0x55, 0x24, 0x6d, 0x00,
        0xaa, 0x24, 0x6d, 0x00, 0xff, 0x24, 0x6d, 0x00, 0x00, 0x49, 0x6d, 0x00,
        0x55, 0x49, 0x6d, 0x00, 0xaa, 0x49, 0x6d, 0x00, 0xff, 0x49, 0x6d, 0x00,
        0x00, 0x6d, 0x6d, 0x00, 0x55, 0x6d, 0x6d, 0x00, 0xaa, 0x6d, 0x6d, 0x00,
        0xff, 0x6d, 0x6d, 0x00, 0x00, 0x92, 0x6d, 0x00, 0x55, 0x92, 0x6d, 0x00,
        0xaa, 0x92, 0x6d, 0x00, 0xff, 0x92, 0x6d, 0x00, 0x00, 0xb6, 0x6d, 0x00,
        0x55, 0xb6, 0x6d, 0x00, 0xaa, 0xb6, 0x6d, 0x00, 0xff, 0xb6, 0x6d, 0x00,
        0x00, 0xdb, 0x6d, 0x00, 0x55, 0xdb, 0x6d, 0x00, 0xaa, 0xdb, 0x6d, 0x00,
        0xff, 0xdb, 0x6d, 0x00, 0x00, 0xff, 0x6d, 0x00, 0x55, 0xff, 0x6d, 0x00,
        0xaa, 0xff, 0x6d, 0x00, 0xff, 0xff, 0x6d, 0x00, 0x00, 0x00, 0x92, 0x00,
        0x55, 0x00, 0x92, 0x00, 0xaa, 0x00, 0x92, 0x00, 0xff, 0x00, 0x92, 0x00,
        0x00, 0x24, 0x92, 0x00, 0x55, 0x24, 0x92, 0x00, 0xaa, 0x24, 0x92, 0x00,
        0xff, 0x24, 0x92, 0x00, 0x00, 0x49, 0x92, 0x00, 0x55, 0x49, 0x92, 0x00,
        0xaa, 0x49, 0x92, 0x00, 0xff, 0x49, 0x92, 0x00, 0x00, 0x6d, 0x92, 0x00,
        0x55, 0x6d, 0x92, 0x00, 0xaa, 0x6d, 0x92, 0x00, 0xff, 0x6d, 0x92, 0x00,
        0x00, 0x92, 0x92, 0x00, 0x55, 0x92, 0x92, 0x00, 0xaa, 0x92, 0x92, 0x00,
        0xff, 0x92, 0x92, 0x00, 0x00, 0xb6, 0x92, 0x00, 0x55, 0xb6, 0x92, 0x00,
        0xaa, 0xb6, 0x92, 0x00, 0xff, 0xb6, 0x92, 0x00, 0x00, 0xdb, 0x92, 0x00,
        0x55, 0xdb, 0x92, 0x00, 0xaa, 0xdb, 0x92, 0x00, 0xff, 0xdb, 0x92, 0x00,
        0x00, 0xff, 0x92, 0x00, 0x55, 0xff, 0x92, 0x00, 0xaa, 0xff, 0x92, 0x00,
        0xff, 0xff, 0x92, 0x00, 0x00, 0x00, 0xb6, 0x00, 0x55, 0x00, 0xb6, 0x00,
        0xaa, 0x00, 0xb6, 0x00, 0xff, 0x00, 0xb6, 0x00, 0x00, 0x24, 0xb6, 0x00,
        0x55, 0x24, 0xb6, 0x00, 0xaa, 0x24, 0xb6, 0x00, 0xff, 0x24, 0xb6, 0x00,
        0x00, 0x49, 0xb6, 0x00, 0x55, 0x49, 0xb6, 0x00, 0xaa, 0x49, 0xb6, 0x00,
        0xff, 0x49, 0xb6, 0x00, 0x00, 0x6d, 0xb6, 0x00, 0x55, 0x6d, 0xb6, 0x00,
        0xaa, 0x6d, 0xb6, 0x00, 0xff, 0x6d, 0xb6, 0x00, 0x00, 0x92, 0xb6, 0x00,
        0x55, 0x92, 0xb6, 0x00, 0xaa, 0x92, 0xb6, 0x00, 0xff, 0x92, 0xb6, 0x00,
        0x00, 0xb6, 0xb6, 0x00, 0x55, 0xb6, 0xb6, 0x00, 0xaa, 0xb6, 0xb6, 0x00,
        0xff, 0xb6, 0xb6, 0x00, 0x00, 0xdb, 0xb6, 0x00, 0x55, 0xdb, 0xb6, 0x00,
        0xaa, 0xdb, 0xb6, 0x00, 0xff, 0xdb, 0xb6, 0x00, 0x00, 0xff, 0xb6, 0x00,
        0x55, 0xff, 0xb6, 0x00, 0xaa, 0xff, 0xb6, 0x00, 0xff, 0xff, 0xb6, 0x00,
        0x00, 0x00, 0xdb, 0x00, 0x55, 0x00, 0xdb, 0x00, 0xaa, 0x00, 0xdb, 0x00,
        0xff, 0x00, 0xdb, 0x00, 0x00, 0x24, 0xdb, 0x00, 0x55, 0x24, 0xdb, 0x00,
        0xaa, 0x24, 0xdb, 0x00, 0xff, 0x24, 0xdb, 0x00, 0x00, 0x49, 0xdb, 0x00,
        0x55, 0x49, 0xdb, 0x00, 0xaa, 0x49, 0xdb, 0x00, 0xff, 0x49, 0xdb, 0x00,
        0x00, 0x6d, 0xdb, 0x00, 0x55, 0x6d, 0xdb, 0x00, 0xaa, 0x6d, 0xdb, 0x00,
        0xff, 0x6d, 0xdb, 0x00, 0x00, 0x92, 0xdb, 0x00, 0x55, 0x92, 0xdb, 0x00,
        0xaa, 0x92, 0xdb, 0x00, 0xff, 0x92, 0xdb, 0x00, 0x00, 0xb6, 0xdb, 0x00,
        0x55, 0xb6, 0xdb, 0x00, 0xaa, 0xb6, 0xdb, 0x00, 0xff, 0xb6, 0xdb, 0x00,
        0x00, 0xdb, 0xdb, 0x00, 0x55, 0xdb, 0xdb, 0x00, 0xaa, 0xdb, 0xdb, 0x00,
        0xff, 0xdb, 0xdb, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x55, 0xff, 0xdb, 0x00,
        0xaa, 0xff, 0xdb, 0x00, 0xff, 0xff, 0xdb, 0x00, 0x00, 0x00, 0xff, 0x00,
        0x55, 0x00, 0xff, 0x00, 0xaa, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00,
        0x00, 0x24, 0xff, 0x00, 0x55, 0x24, 0xff, 0x00, 0xaa, 0x24, 0xff, 0x00,
        0xff, 0x24, 0xff, 0x00, 0x00, 0x49, 0xff, 0x00, 0x55, 0x49, 0xff, 0x00,
        0xaa, 0x49, 0xff, 0x00, 0xff, 0x49, 0xff, 0x00, 0x00, 0x6d, 0xff, 0x00,
        0x55, 0x6d, 0xff, 0x00, 0xaa, 0x6d, 0xff, 0x00, 0xff, 0x6d, 0xff, 0x00,
        0x00, 0x92, 0xff, 0x00, 0x55, 0x92, 0xff, 0x00, 0xaa, 0x92, 0xff, 0x00,
        0xff, 0x92, 0xff, 0x00, 0x00, 0xb6, 0xff, 0x00, 0x55, 0xb6, 0xff, 0x00,
        0xaa, 0xb6, 0xff, 0x00, 0xff, 0xb6, 0xff, 0x00, 0x00, 0xdb, 0xff, 0x00,
        0x55, 0xdb, 0xff, 0x00, 0xaa, 0xdb, 0xff, 0x00, 0xff, 0xdb, 0xff, 0x00,
        0x00, 0xff, 0xff, 0x00, 0x55, 0xff, 0xff, 0x00, 0xaa, 0xff, 0xff, 0x00,
        0xff, 0xff, 0xff, 0x00, 0x70, 0x62, 0x63, 0x6c, 0x65, 0x76, 0x74, 0x75,
        0x67, 0x20, 0x28, 0x70, 0x29, 0x20, 0x31, 0x39, 0x39, 0x38, 0x2d, 0x32,
        0x30, 0x30, 0x36, 0x20, 0x58, 0x6e, 0x6d, 0x68, 0x78, 0x76, 0x20, 0x56,
        0x4a, 0x4e, 0x5a, 0x42, 0x47, 0x42, 0x20, 0x75, 0x67, 0x67, 0x63, 0x3a,
        0x2f, 0x2f, 0x6a, 0x6a, 0x6a, 0x2e, 0x7a, 0x6e, 0x76, 0x71, 0x2e, 0x62,
        0x65, 0x74, 0x2f, 0x20, 0x76, 0x6a, 0x7a, 0x40, 0x7a, 0x6e, 0x76, 0x71,
        0x2e, 0x62, 0x65, 0x74};


/*  ja:AVIファイルを求める
    avi_edit,AVI編集ハンドル
         pos,サンプル番号                                                   */
void
avi_base_get_file (AviEdit    *avi_edit,
                   const gint  pos)
{
  while (pos < avi_edit->offset && avi_edit->file->prev)
    {
      avi_edit->file = avi_edit->file->prev;
      avi_edit->offset -= avi_edit->file->length;
    }
  while (avi_edit->offset + avi_edit->file->length <= pos
                                                    && avi_edit->file->next)
    {
      avi_edit->offset += avi_edit->file->length;
      avi_edit->file = avi_edit->file->next;
    }
}


/*  ja:AVIファイルを分離する
    avi_edit,AVI編集ハンドル
         pos,サンプル番号                                                   */
void
avi_base_split_file (AviEdit    *avi_edit,
                     const gint  pos)
{
  AviFile *avi_file;

  if (pos < 0 || avi_edit_length (avi_edit) <= pos)
    return;

  /* ja:位置を求める */
  avi_base_get_file (avi_edit, pos);/* ja:AVIファイルを求める */
  if (avi_edit->offset == pos)
    return;/* ja:ファイルの切れ目に一致した */

  /* ja:そのファイルを分離する */
  avi_file = g_malloc (sizeof (AviFile));
  avi_file->fio = avi_edit->file->fio;
  if (avi_file->fio)
    fileio_ref (avi_file->fio);
  avi_file->avi_memory = avi_edit->file->avi_memory;
  if (avi_file->avi_memory)
    avi_memory_ref (avi_file->avi_memory);
  avi_file->name = g_strdup (avi_edit->file->name);
  avi_file->start = pos-avi_edit->offset + avi_edit->file->start;
  avi_file->length = avi_edit->file->length - (pos - avi_edit->offset);
  avi_file->entries = avi_edit->file->entries;
  avi_file->param = avi_edit->file->param;
  avi_file->index = g_memdup (avi_edit->file->index,
                                avi_edit->file->entries * sizeof (AviIndex));
  /* ja:ビデオ固有 */
  avi_file->handler = avi_edit->file->handler;
  avi_file->bmih = avi_edit->file->bmih ? g_memdup (avi_edit->file->bmih,
                                bm_header_bytes (avi_edit->file->bmih)) : NULL;
  /* ja:オーディオ固有 */
  avi_file->samples = avi_edit->file->samples;
  avi_file->wfx = avi_edit->file->wfx ? g_memdup (avi_edit->file->wfx,
                                wf_header_bytes (avi_edit->file->wfx)) : NULL;
  /* ja:元のファイルの変更 */
  avi_edit->file->length -= avi_file->length;
  /* ja:リストに挿入する */
  avi_file->prev = avi_edit->file;
  avi_file->next = avi_edit->file->next;
  avi_file->prev->next = avi_file;
  if (avi_file->next)
    avi_file->next->prev = avi_file;
  avi_base_get_file (avi_edit, pos);
}


/******************************************************************************
*                                                                             *
* ja:AVIファイル構造体関数                                                    *
*                                                                             *
******************************************************************************/
/*  ja:AVIからAVIファイル構造体を求める
        file,ファイル名
       param,ストリーム番号
        type,AVI_STREAM_AUDIO/AVI_STREAM_VIDEO
        rate,レート
       scale,スケール
    priority,優先度
    language,言語
        name,名前
         RET,AVIファイル構造体,NULL:エラー                                  */
static AviFile *
avi_file_open_avi (const gchar  *file,
                   const gint    param,
                   gint         *type,
                   guint32      *rate,
                   guint32      *scale,
                   guint16      *priority,
                   guint16      *language,
                   gchar       **name)
{
  gchar *strn = NULL;
  gint i;
  guint32 id, size;
  goffset offset;               /* ja:AVIのムービーチャンク */
  AviFile *avi_file;
  AviStreamHeader ash;
  Chunk *ck;

  avi_file = g_malloc0 (sizeof (AviFile));
  /* ja:ファイルを開く */
  avi_file->fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING);
  if (!avi_file->fio)
    {
      g_free (avi_file);
      return NULL;
    }
  /* ja:ファイルディスクリターからチャンクポインタを作る */
  ck = chunk_open (avi_file->fio);
  /* ja:RIFFチャンクに入る */
  if (!chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:フォームがhdrlのLISTチャンクを探す */
  while (chunk_form (avi_file->fio) != CK_LIST_AVIHEADER)
    if (!chunk_next (avi_file->fio, ck))
      {
        chunk_free (ck);
        fileio_close (avi_file->fio);
        g_free (avi_file);
        return NULL;
      }
  /* ja:フォームがhdrlのLISTチャンクに入る(RIFF'AVI ' -> LIST'hdrl') */
  if (!chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:フォームがstrlのLISTチャンクを探す */
  for (i = 0;i <= param; i++)
    do
      if (!chunk_next (avi_file->fio, ck))
        {
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file);
          return NULL;
        }
    while (chunk_form (avi_file->fio) != CK_LIST_STREAMHEADER);
  /* ja:フォームがstrlのLISTチャンクに入る
                                    (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl') */
  if (!chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:strhチャンクを探す(AviStreamHeader) */
  while (chunk_id (avi_file->fio) != CK_ID_STREAMHEADER)
    if (!chunk_next (avi_file->fio, ck))
      {
        chunk_free (ck);
        fileio_close (avi_file->fio);
        g_free (avi_file);
        return NULL;
      }
  /* ja:strhチャンクのサイズ(AviStreamHeader) */
  size = chunk_size (avi_file->fio);
  if (size == -1 || size < ASH_SIZE)
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:strhチャンクに入る(AVIStreamHeader)
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' -> strh) */
  if (!chunk_in (avi_file->fio, ck)
      /* ja:AviStreamHeaderを読み込む */
                || fileio_read (avi_file->fio, &ash, ASH_SIZE) != ASH_SIZE
      /* ja:タイプ */
                || (ash_get_type (&ash) != AVI_STREAM_AUDIO
                                    && ash_get_type (&ash) != AVI_STREAM_VIDEO)
      /* ja:strhチャンクから出る(AVIStreamHeader)
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' <- strh) */
                || !chunk_out (avi_file->fio, ck)
      /* ja:フォームがstrlのLISTチャンクから出る
                                    (RIFF'AVI ' -> LIST'hdrl' <- LIST'strl') */
                || !chunk_out (avi_file->fio, ck)
      /* ja:フォームがstrlのLISTチャンクに入る
                                    (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl') */
                || !chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:strfチャンクを探す(BitmapInfoHeader)|(WaveFormatEx) */
  while (chunk_id (avi_file->fio) != CK_ID_STREAMFORMAT)
    if (!chunk_next (avi_file->fio, ck))
      {
        chunk_free (ck);
        fileio_close (avi_file->fio);
        g_free (avi_file);
        return NULL;
      }
  /* ja:strfチャンクのサイズ(BitmapInfoHeader)|(WaveFormatEx) */
  size = chunk_size (avi_file->fio);
  if (size == -1)
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  if (ash_get_type (&ash) == AVI_STREAM_VIDEO)
    {
      if (size < BMIH_SIZE)
        {
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file);
          return NULL;
      }
      avi_file->bmih = g_malloc (size);
    }
  else
    {
      if (size < WFX_SIZE - sizeof (guint16))
        {
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file);
          return NULL;
        }
      avi_file->wfx = g_malloc (MAX (size, WFX_SIZE));
      wfx_set_size (avi_file->wfx, 0);
    }
  /* ja:strfチャンクに入る(BitmapInfoHeader)|(WaveFormatEx)
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' -> strf) */
  if (!chunk_in (avi_file->fio, ck)
      /* ja:BitmapInfoHeader|WaveFormatExを読み込む */
        || fileio_read (avi_file->fio, ash_get_type (&ash) == AVI_STREAM_VIDEO
            ? (gpointer)avi_file->bmih : (gpointer)avi_file->wfx, size) != size
      /* ja:strfチャンクから出る(BitmapInfoHeader)|(WaveFormatEx)
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' <- strh) */
        || !chunk_out (avi_file->fio, ck)
      /* ja:フォームがstrlのLISTチャンクから出る
                                    (RIFF'AVI ' -> LIST'hdrl' <- LIST'strl') */
        || !chunk_out (avi_file->fio, ck)
      /* ja:フォームがstrlのLISTチャンクに入る
                                    (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl') */
        || !chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file->bmih);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
   }
  if (name)
    {
      /*ja: strnチャンクを探す */
      while ((id = chunk_id (avi_file->fio)) != CK_ID_STREAMNAME)
        if (!chunk_next (avi_file->fio, ck))
          break;
      if (id == CK_ID_STREAMNAME)
        {
          /* ja:strnチャンクのサイズ */
          size = chunk_size (avi_file->fio);
          if (size == -1)
            {
              chunk_free (ck);
              fileio_close (avi_file->fio);
              g_free (avi_file->bmih);
              g_free (avi_file->wfx);
              g_free (avi_file);
              return NULL;
            }
          strn = g_malloc0 (size + 1);
          /* ja:strnチャンクに入る
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' -> strn) */
          if (!chunk_in (avi_file->fio, ck)
              /* ja:strnを読み込む */
                    || fileio_read (avi_file->fio, strn, size) != size
              /* strnチャンクから出る
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' <- strh) */
                    || !chunk_out (avi_file->fio, ck))
            {
              g_free (strn);
              chunk_free (ck);
              fileio_close (avi_file->fio);
              g_free (avi_file->bmih);
              g_free (avi_file->wfx);
              g_free (avi_file);
              return NULL;
            }
        }
      /* ja:フォームがstrlのLISTチャンクから出る
                                    (RIFF'AVI ' -> LIST'hdrl' <- LIST'strl') */
      if (!chunk_out (avi_file->fio, ck)
          /* ja:フォームがstrlのLISTチャンクに入る
                                    (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl') */
                || !chunk_in (avi_file->fio, ck))
        {
          g_free (strn);
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      /* ja:UTF-8としての正統性をチェック */
      if (strn)
        {
          gchar *end_ptr;

          if (!g_utf8_validate (strn, -1, (const gchar **)&end_ptr))
            *end_ptr = '\0';
          if (*strn == '\0')
            {
              g_free (strn);
              strn = NULL;
            }
        }
    }

  /* ja:スーパーインデックスチャンクへ移動する */

  /* ja:スーパーインデックスチャンクを探す */
  while ((id = chunk_id (avi_file->fio)) != CK_ID_AVISUPERINDEX)
    if (!chunk_next (avi_file->fio, ck))
      break;
  if (id == CK_ID_AVISUPERINDEX)
    {
      AviSuperIndex asi;

      /* ja:スーパーインデックスチャンクのサイズ */
      size = chunk_size (avi_file->fio);
      if (size == -1  /* ja:スーパーインデックスチャンクに入る
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' -> indx) */
                    || !chunk_in (avi_file->fio, ck)
                    /* ja:indxを読み込む */
                    || fileio_read (avi_file->fio, &asi, ASI_SIZE) != ASI_SIZE
                    || size < ASI_SIZE + asi_get_entries (&asi) * ASE_SIZE)
        {
          g_free (strn);
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      if (asi_get_longs (&asi) == 4
                            && asi_get_sub_type (&asi) == AVI_IDX_SUB_DEFAULT
                                && asi_get_type (&asi) == AVI_IDX_OF_INDEXES)
        for (i = 0; i < asi_get_entries (&asi); i++)
          {
            AviSuperIndexEntry ase;

            /* ja:スーパーインデックスエントリーを読み込む */
            if (fileio_read (avi_file->fio, &ase, ASE_SIZE) != ASE_SIZE)
              {
                g_free (strn);
                chunk_free (ck);
                fileio_close (avi_file->fio);
                g_free (avi_file->index);
                g_free (avi_file);
                return NULL;
              }
            if (ase_get_offset (&ase) != 0)
              {
                AviBaseIndex abi;

                /* ja:ファイルのオフセットを取得する */
                offset = fileio_seek (avi_file->fio, 0, FILEIO_SEEK_CUR);
                /* ja:ファイルのオフセットを設定する */
                if (offset == -1 || fileio_seek (avi_file->fio,
                                ase_get_offset (&ase) + sizeof (guint32) * 2,
                                                        FILEIO_SEEK_SET) == -1
                            /* ix##を読み込む */
                                || fileio_read (avi_file->fio, &abi, ABI_SIZE)
                                                                != ABI_SIZE)
                  {
                    g_free (strn);
                    chunk_free (ck);
                    fileio_close (avi_file->fio);
                    g_free (avi_file->index);
                    g_free (avi_file->bmih);
                    g_free (avi_file->wfx);
                    g_free (avi_file);
                    return NULL;
                  }
                if (abi_get_longs (&abi) == 2
                        && asi_get_sub_type (&asi) == abi_get_sub_type (&abi)
                                && abi_get_type (&abi) == AVI_IDX_OF_CHUNKS)
                  {
                    gint j;

                    avi_file->index = g_realloc (avi_file->index,
                                (avi_file->entries + abi_get_entries (&abi))
                                                        * sizeof(AviIndex));
                    for (j = 0; j < abi_get_entries (&abi); j++)
                      {
                        AviBaseIndexEntry abe;

                        /* ja:スーパーインデックスエントリーを読み込む */
                        if (fileio_read (avi_file->fio, &abe, ABE_SIZE)
                                                                != ABE_SIZE)
                          {
                            g_free (strn);
                            chunk_free (ck);
                            fileio_close (avi_file->fio);
                            g_free (avi_file->index);
                            g_free (avi_file->bmih);
                            g_free (avi_file->wfx);
                            g_free (avi_file);
                            return NULL;
                          }
                        avi_file->index[avi_file->entries].offset
                                    = abi_get_offset (&abi)
                                                    + abe_get_offset (&abe);
                        avi_file->index[avi_file->entries].size
                                    = (abe_get_size (&abe) & 0x7fffffff);
                        avi_file->index[avi_file->entries].key
                                    = (abe_get_size (&abe) & 0x80000000) != 0;
                        avi_file->entries++;
                      }
                  }
                /* ja:ファイルのオフセットを設定する */
                if (fileio_seek (avi_file->fio, offset, FILEIO_SEEK_SET) == -1)
                  {
                    g_free (strn);
                    chunk_free (ck);
                    fileio_close (avi_file->fio);
                    g_free (avi_file->index);
                    g_free (avi_file->bmih);
                    g_free (avi_file->wfx);
                    g_free (avi_file);
                    return NULL;
                  }
              }
          }
    }

  /* ja:チャンクポインタを解放する */
  if (!chunk_free (ck))
    {
      g_free (strn);
      fileio_close (avi_file->fio);
      g_free (avi_file->index);
      g_free (avi_file->bmih);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }

  if (avi_file->entries <= 0)
    {
      /* ja:ムービーチャンクへ移動する */

      /* ja:ファイルディスクリターからチャンクポインタを作る */
      ck = chunk_open (avi_file->fio);
      /* ja:RIFFチャンクに入る */
      if (!chunk_in (avi_file->fio, ck))
        {
          g_free (strn);
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      /* ja:moviチャンクを探す */
      while (chunk_form (avi_file->fio) != CK_LIST_AVIMOVIE)
        if (!chunk_next (avi_file->fio, ck))
          {
            g_free (strn);
            chunk_free (ck);
            fileio_close (avi_file->fio);
            g_free (avi_file->bmih);
            g_free (avi_file->wfx);
            g_free (avi_file);
            return NULL;
          }
      /* ja:ファイルのオフセットを取得する */
      offset = fileio_seek (avi_file->fio, 0, FILEIO_SEEK_CUR);
      if (offset == -1)
        {
          g_free (strn);
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      offset += sizeof (guint32) + sizeof (guint32);/* ja:moviのオフセット */
      /* ja:チャンクポインタを解放する */
      if (!chunk_free (ck))
        {
          g_free (strn);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }

      /* ja:インデックスチャンクへ移動する */

      /* ja:ファイルディスクリターからチャンクポインタを作る */
      ck = chunk_open (avi_file->fio);
      /* ja:RIFFチャンクに入る */
      if (!chunk_in (avi_file->fio, ck))
        {
          g_free (strn);
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      /* ja:idx1チャンクを探す */
      while (chunk_id (avi_file->fio) != CK_ID_AVINEWINDEX)
        if (!chunk_next (avi_file->fio, ck))
          {
            g_free (strn);
            chunk_free (ck);
            fileio_close (avi_file->fio);
            g_free (avi_file->bmih);
            g_free (avi_file->wfx);
            g_free (avi_file);
            return NULL;
          }
      /* ja:idx1チャンクのサイズ */
      /* ja:idx1チャンクに入る(RIFF'AVI ' -> idx1) */
      size = chunk_size (avi_file->fio);
      if (size == -1 || !chunk_in (avi_file->fio, ck))
        {
          g_free (strn);
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      /* ja:インデックスを読み込む */
      for (i = 0; i < size / AIE_SIZE; i++)
        {
          AviIndexEntry aie;

          if (fileio_read (avi_file->fio, &aie, AIE_SIZE) != AIE_SIZE)
            {
              g_free (strn);
              chunk_free (ck);
              fileio_close (avi_file->fio);
              g_free (avi_file->index);
              g_free (avi_file->bmih);
              g_free (avi_file->wfx);
              g_free (avi_file);
              return NULL;
            }
          if (((aie_get_ckid (&aie) & 0xff) - '0') * 10
                        + ((aie_get_ckid (&aie) >> 8) & 0xff) - '0' == param)
            {
              avi_file->index = g_realloc (avi_file->index,
                                (avi_file->entries + 1) * sizeof (AviIndex));
              avi_file->index[avi_file->entries].offset
                                = aie_get_offset (&aie)
                                + offset + sizeof (guint32) + sizeof(guint32);
              avi_file->index[avi_file->entries].size = aie_get_length (&aie);
              avi_file->index[avi_file->entries].key
                            = (aie_get_flags (&aie) & AVI_IF_KEYFRAME) != 0;
              avi_file->entries++;
            }
        }
      /* ja:チャンクポインタを解放する */
      if (!chunk_free (ck))
        {
          g_free (strn);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
    }
  /* ja:インデックスとヘッタの整合性 */
  if (ash_get_type (&ash) == AVI_STREAM_VIDEO)
    {
      avi_file->length = avi_file->entries;
    }
  else
    {
      for (i = 0; i < avi_file->entries; i++)
        avi_file->samples += avi_file->index[i].size
                                        / wfx_get_block_align (avi_file->wfx);
      if (avi_file->samples <= 0)
        {
          g_free (strn);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      avi_file->length = MAX ((glonglong)avi_file->samples
                        * AVI_PCM_SAMPLES_PER_SEC
                        * wfx_get_block_align (avi_file->wfx)
                        / wfx_get_average_bytes_per_sec (avi_file->wfx), 1);
    }
  avi_file->handler = ash_get_handler (&ash);
  /* ja:情報 */
  if (type)
    *type = ash_get_type (&ash) == AVI_STREAM_VIDEO
                                            ? AVI_TYPE_VIDEO : AVI_TYPE_AUDIO;
  if (rate)
    *rate = ash_get_rate (&ash);
  if (scale)
    *scale = ash_get_scale (&ash);
  if (priority)
    *priority = ash_get_priority (&ash);
  if (language)
    *language = ash_get_language (&ash);
  if (name)
    *name = strn;
  return avi_file;
}


/*  ja:GtkShot RawからAVIファイル構造体を求める
     file,ファイル名
    param,ストリーム番号
     type,AVI_STREAM_AUDIO/AVI_STREAM_VIDEO
     rate,レート
    scale,スケール
      RET,AVIファイル構造体,NULL:エラー                                     */
static AviFile *
avi_file_open_gshot (const gchar *file,
                     const gint   param,
                     gint        *type,
                     guint32     *rate,
                     guint32     *scale)
{
  gint i;
  gsize format;
  AviFile *avi_file;
  GsrMainHeader gmh;
  GsrStreamHeader gsh;

  avi_file = g_malloc0 (sizeof (AviFile));
  /* ja:ファイルを開く */
  avi_file->fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING);
  if (!avi_file->fio)
    {
      g_free (avi_file);
      return NULL;
    }
  /* ja:ストリームの数 */
  if (fileio_read (avi_file->fio, &gmh, GMH_SIZE) != GMH_SIZE
                                            || gmh_get_streams (&gmh) <= param)
    {
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:目的のストリームまでシーク */
  for (i = 0; i < param; i++)
    if (fileio_read (avi_file->fio, &gsh, GSH_SIZE) != GSH_SIZE
            || fileio_seek (avi_file->fio,
                                gsh_get_format (&gsh), FILEIO_SEEK_CUR) == -1)
      {
        fileio_close (avi_file->fio);
        g_free (avi_file);
        return NULL;
      }
  /* ja:目的のストリームを読み込む */
  if (fileio_read (avi_file->fio, &gsh, GSH_SIZE) != GSH_SIZE)
    {
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:フォーマットを読み込む */
  format = gsh_get_format (&gsh);
  if (gsh_get_type (&gsh) == GSR_STREAM_VIDEO)
    {
      if (format < BMIH_SIZE)
        {
          fileio_close (avi_file->fio);
          g_free (avi_file);
          return NULL;
      }
      avi_file->bmih = g_malloc (gsh_get_format (&gsh));
      if (fileio_read (avi_file->fio, avi_file->bmih, format) != format)
        {
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file);
          return NULL;
        }
    }
  else
    {
      if (format < WFX_SIZE - sizeof (guint16))
        {
          fileio_close (avi_file->fio);
          g_free (avi_file);
          return NULL;
        }
      avi_file->wfx = g_malloc (MAX (format, WFX_SIZE));
      wfx_set_size (avi_file->wfx, 0);
      if (fileio_read (avi_file->fio, avi_file->wfx, format) != format)
        {
          fileio_close (avi_file->fio);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
    }
  /* ja:インデックスを読み込む */
  if (fileio_seek (avi_file->fio,
                                gsh_get_offset (&gsh), FILEIO_SEEK_SET) == -1)
    {
      fileio_close (avi_file->fio);
      g_free (avi_file->bmih);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }
  avi_file->entries = gsh_get_index (&gsh);
  avi_file->index = g_malloc (avi_file->entries * sizeof (AviIndex));
  for (i = 0; i < avi_file->entries; i++)
    {
      GsrIndexEntry gie;

      if (fileio_read (avi_file->fio, &gie, GIE_SIZE) != GIE_SIZE)
        {
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      avi_file->index[i].offset = gie_get_offset (&gie);
      avi_file->index[i].size = gie_get_length (&gie);
      avi_file->index[i].key = (gie_get_flags (&gie) & GSR_IF_KEYFRAME) != 0;
    }
  /* ja:インデックスとヘッタの整合性 */
  if (gsh_get_type (&gsh) == GSR_STREAM_VIDEO)
    {
      avi_file->length = avi_file->entries;
    }
  else
    {
      for (i = 0; i < avi_file->entries; i++)
        avi_file->samples += avi_file->index[i].size
                                        / wfx_get_block_align (avi_file->wfx);
      if (avi_file->samples <= 0)
        {
          fileio_close (avi_file->fio);
          g_free (avi_file->index);
          g_free (avi_file->bmih);
          g_free (avi_file->wfx);
          g_free (avi_file);
          return NULL;
        }
      avi_file->length = MAX ((glonglong)avi_file->samples
                        * AVI_PCM_SAMPLES_PER_SEC
                        * wfx_get_block_align (avi_file->wfx)
                        / wfx_get_average_bytes_per_sec (avi_file->wfx), 1);
    }
  avi_file->handler = gsh_get_handler (&gsh);
  /* ja:情報 */
  if (gsh_get_type (&gsh) == GSR_STREAM_VIDEO)
    {
      if (type)
        *type = AVI_TYPE_VIDEO;
      if (rate)
        *rate = 1000000;
      if (scale)
        *scale = gsh_get_micro_sec_per_frame (&gsh);
    }
  else
    {
      if (type)
        *type = AVI_TYPE_AUDIO;
      if (rate)
        *rate = wfx_get_average_bytes_per_sec (avi_file->wfx);
      if (scale)
        *scale = wfx_get_block_align (avi_file->wfx);
    }
  return avi_file;
}


/*  ja:ビットマップからAVIファイル構造体を求める
    file,ファイル名
     RET,AVIファイル構造体,NULL:エラー                                      */
static AviFile *
avi_file_open_bitmap (const gchar *file)
{
  guint32 size;
  AviFile *avi_file;

  avi_file = g_malloc0 (sizeof (AviFile));
  /* ja:ファイルを開く */
  avi_file->fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING);
  if (!avi_file->fio)
    {
      g_free (avi_file);
      return NULL;
    }
  /* ja:仮にヘッタを読み込む */
  if (fileio_seek (avi_file->fio, BMFH_SIZE, FILEIO_SEEK_SET) == -1)
    {
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  avi_file->bmih = g_malloc (BMIH_SIZE);
  if (fileio_read (avi_file->fio, avi_file->bmih, BMIH_SIZE) != BMIH_SIZE)
    {
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:ヘッタのサイズを計算する */
  size = bm_header_bytes (avi_file->bmih);
  /* ja:ヘッタの読み込み */
  avi_file->bmih = g_realloc (avi_file->bmih, size);
  if (fileio_seek (avi_file->fio, BMFH_SIZE, FILEIO_SEEK_SET) == -1
                || fileio_read (avi_file->fio, avi_file->bmih, size) != size)
    {
      fileio_close (avi_file->fio);
      g_free (avi_file->bmih);
      g_free (avi_file);
      return NULL;
    }
  /* ja:イメージを読み込む */
  if (bmih_get_compression (avi_file->bmih) == BI_COMP_RGB
                || bmih_get_compression (avi_file->bmih) == BI_COMP_BITFIELDS)
    {
      /* ja:無圧縮 */
      size = bm_image_bytes (avi_file->bmih);
      avi_file->avi_memory = avi_memory_malloc (size);
      if (fileio_read (avi_file->fio,
                    avi_memory_get_data (avi_file->avi_memory), size) != size)
        {
          avi_memory_free (avi_file->avi_memory);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file);
          return NULL;
        }
    }
  else if (bmih_get_compression (avi_file->bmih) == BI_COMP_RLE4
                    || bmih_get_compression (avi_file->bmih) == BI_COMP_RLE8)
    {
      BitmapInfoHeader *bmih0, *bmih1;

      /* ja:圧縮 */
      bmih0 = g_malloc (size + bmih_get_size_image (avi_file->bmih));
      g_memmove (bmih0, avi_file->bmih, size);
      if (fileio_read (avi_file->fio, (guint8 *)bmih0 + size,
                bmih_get_size_image (bmih0)) != bmih_get_size_image (bmih0))
        {
          g_free (bmih0);
          fileio_close (avi_file->fio);
          g_free (avi_file->bmih);
          g_free (avi_file);
          return NULL;
        }
      bmih1 = g_malloc (bx_all_bytes
                            (bmih_get_width (bmih0), bmih_get_height (bmih0),
                                    bmih_get_bit_count (bmih0), BI_COMP_RGB,
                                                bmih_get_color_used (bmih0)));
      bitmap_expand (bmih0, bmih1);
      g_free (bmih0);
      /* ja:展開後のDIBをコピー */
      size = bm_header_bytes (bmih1);
      avi_file->bmih = g_realloc (avi_file->bmih, size);
      g_memmove (avi_file->bmih, bmih1, size);
      avi_file->avi_memory = avi_memory_dup ((guint8 *)bmih1 + size,
                                            bm_image_bytes (avi_file->bmih));
      g_free (bmih1);
    }
  else
    {
      fileio_close (avi_file->fio);
      g_free (avi_file->bmih);
      g_free (avi_file);
      return NULL;
    }
  if (!fileio_close (avi_file->fio))
    {
      avi_memory_free (avi_file->avi_memory);
      g_free (avi_file->bmih);
      g_free (avi_file);
      return NULL;
    }
  avi_file->handler = AVI_STREAM_COMP_DIB;
  /* ja:ファイル */
  avi_file->fio = NULL;                 /* ja:ファイルポインタ */
  avi_file->length = 1;
  avi_file->entries = 1;                /* ja:インデックスの数 */
  /* ja:インデックス */
  avi_file->index = g_malloc (sizeof (AviIndex));
  avi_file->index[0].offset = 0;
  avi_file->index[0].size = size;
  avi_file->index[0].key = TRUE;
  return avi_file;
}


/*  ja:WAVEファイルからAVIファイル構造体を求める
    file,ファイル名
     RET,AVIファイル構造体,NULL:エラー                                      */
static AviFile *
avi_file_open_wave (const gchar *file)
{
  guint32 size;
  goffset offset;               /* ja:AVIのムービーチャンク */
  AviFile *avi_file;
  Chunk *ck;

  avi_file = g_malloc0 (sizeof (AviFile));
  /* ja:ファイルを開く */
  avi_file->fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING);
  if (!avi_file->fio)
    {
      g_free (avi_file);
      return NULL;
    }
  /* ja:ファイルディスクリターからチャンクポインタを作る */
  ck = chunk_open (avi_file->fio);
  /* ja:RIFFチャンクに入る */
  if (!chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  /* ja:fmtチャンクを探す */
  while (chunk_id (avi_file->fio) != CK_ID_WAVEFORMAT)
    {
      if (!chunk_next (avi_file->fio, ck))
        {
          chunk_free (ck);
          fileio_close (avi_file->fio);
          g_free (avi_file);
          return NULL;
        }
    }
  /* ja:fmtチャンクのサイズ */
  /* ja:fmtチャンクに入る(RIFF'AVI ' -> fmt) */
  size = chunk_size (avi_file->fio);
  if (size == -1 || !chunk_in (avi_file->fio,ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  if (size < WFX_SIZE - sizeof (guint16))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file);
      return NULL;
    }
  avi_file->wfx = g_malloc (MAX (size, WFX_SIZE));
  wfx_set_size (avi_file->wfx, 0);
  /* ja:WaveFormatExを読み込む */
  if (fileio_read (avi_file->fio, avi_file->wfx, size) != size)
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }
  /* ja:チャンクポインタを解放する */
  if (!chunk_free (ck))
    {
      fileio_close (avi_file->fio);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }

  /* ja:ファイルディスクリターからチャンクポインタを作る */
  ck = chunk_open (avi_file->fio);
  /* ja:RIFFチャンクに入る */
  if (!chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }
  /* ja:dataチャンクを探す */
  while (chunk_id (avi_file->fio) != CK_ID_WAVEDATA)
    if (!chunk_next (avi_file->fio, ck))
      {
        chunk_free (ck);
        fileio_close (avi_file->fio);
        g_free (avi_file->wfx);
        g_free (avi_file);
        return NULL;
      }
  /* ja:dataチャンクのサイズ */
  /* ja:dataチャンクに入る(RIFF'AVI ' -> data) */
  size = chunk_size (avi_file->fio);
  if (size == -1 || !chunk_in (avi_file->fio, ck))
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }
  /* ja:ファイルのオフセットを取得する */
  offset = fileio_seek (avi_file->fio, 0, FILEIO_SEEK_CUR);
  if (offset == -1)
    {
      chunk_free (ck);
      fileio_close (avi_file->fio);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }
  /* ja:チャンクポインタを解放する */
  if (!chunk_free (ck))
    {
      fileio_close (avi_file->fio);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }
  /* ja:ファイル */
  avi_file->samples = size / wfx_get_block_align (avi_file->wfx);
  if (avi_file->samples <= 0)
    {
      fileio_close (avi_file->fio);
      g_free (avi_file->wfx);
      g_free (avi_file);
      return NULL;
    }
  avi_file->length = MAX ((glonglong)avi_file->samples
                        * AVI_PCM_SAMPLES_PER_SEC
                        * wfx_get_block_align (avi_file->wfx)
                        / wfx_get_average_bytes_per_sec (avi_file->wfx), 1);
  avi_file->entries = 1;                /* ja:インデックスの数 */
  /* ja:インデックス */
  avi_file->index = g_malloc (sizeof (AviIndex));
  avi_file->index[0].offset = offset;
  avi_file->index[0].size = size;
  avi_file->index[0].key = FALSE;
  return avi_file;
}


/*  ja:AVIからAVIファイル構造体を求める
        file,ファイル名
       param,ストリーム番号
        type,AVI_STREAM_AUDIO/AVI_STREAM_VIDEO
        rate,レート
       scale,スケール
    priority,優先度
    language,言語
        name,名前
         RET,AVIファイル構造体,NULL:エラー                                  */
AviFile *
avi_file_open (const gchar  *file,
               const gint    param,
               gint         *type,
               guint32      *rate,
               guint32      *scale,
               guint16      *priority,
               guint16      *language,
               gchar       **name)
{
  AviFile *avi_file = NULL;

  /* ja:ファイルの形式毎に分岐 */
  switch (avi_main_type (file))
    {
      case AVI_FILE_AVI:/* en:AVI */
        avi_file = avi_file_open_avi (file, param, type, rate, scale,
                                                    priority, language, name);
        break;
      case AVI_FILE_GSR:/* en:GtkShot Raw */
        avi_file = avi_file_open_gshot (file, param, type, rate, scale);
        if (avi_file)
          {
            if (priority)
              *priority = 0;
            if (language)
              *language = 0;
            if (name)
              *name = NULL;
          }
        break;
      case AVI_FILE_BITMAP:/* ja:ビットマップ */
        avi_file = param == 0 ? avi_file_open_bitmap (file) : NULL;
        if (avi_file)
          {
            if (type)
              *type = AVI_TYPE_VIDEO;
            if (rate)
              *rate = 15;
            if (scale)
              *scale = 1;
            if (priority)
              *priority = 0;
            if (language)
              *language = 0;
            if (name)
              *name = g_strdup ("Video #1");
          }
        break;
      case AVI_FILE_WAVE:/* ja:WAVEファイル */
        avi_file = param == 0 ? avi_file_open_wave (file) : NULL;
        if (avi_file)
          {
            if (type)
              *type = AVI_TYPE_AUDIO;
            if (rate)
              *rate = wfx_get_average_bytes_per_sec (avi_file->wfx);
            if (scale)
              *scale = wfx_get_block_align (avi_file->wfx);
            if (priority)
              *priority = 0;
            if (language)
              *language = 0;
            if (name)
              *name = g_strdup ("Audio #1");
          }
    }
  if (avi_file)
    {
      avi_file->name = fileio_get_full_path (file);     /* ja:ファイル名 */
      avi_file->param = param;                          /* ja:ストリーム番号 */
    }
  return avi_file;
}


/*  ja:AVIファイル構造体を解放する
    avi_file,AVIファイル構造体
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_file_close (AviFile *avi_file)
{
  gboolean result = FALSE;

  if (avi_file)
    {
      result = !avi_file->fio || fileio_close (avi_file->fio);
      avi_memory_free (avi_file->avi_memory);
      g_free (avi_file->name);
      g_free (avi_file->index);
      g_free (avi_file->bmih);
      g_free (avi_file->wfx);
      g_free (avi_file);
    }
  return result;
}


/******************************************************************************
*                                                                             *
* ja:AVIファイル入力関数                                                      *
*                                                                             *
******************************************************************************/
#ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
#endif /* USE_THREAD */


/*  ja:AVI(ビデオ)のデータのバイト数を取得する
    avi_file,AVIファイル構造体
         pos,読み込むサンプル番号
         RET,バイト数                                                       */
gsize
avi_read_video_size (AviFile    *avi_file,
                     const gint  pos)
{
  gint i;

  /* ja:スタート位置の調整 */
  if (!avi_file || !avi_file->bmih || avi_file->start + pos < 0
                                || avi_file->entries <= avi_file->start + pos)
    return 0;
  /* ja:ヌルフレームのチェック */
  for (i = pos + avi_file->start; i > 0; i--)
    if (avi_file->index[i].size > 0)
      break;
  return avi_file->index[i].size;
}


/*  ja:AVI(ビデオ)のデータを読み込む
    avi_file,AVIファイル構造体
         pos,読み込むサンプル番号
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_video (AviFile    *avi_file,
                const gint  pos,
                gpointer    buffer)
{
  gint i;
  guint32 size;
  goffset offset;

  /* ja:スタート位置の調整 */
  if (!avi_file || !avi_file->bmih  || !buffer || avi_file->start + pos < 0
                                || avi_file->entries <= avi_file->start + pos)
    return FALSE;
  /* ja:ヌルフレームのチェック */
  for (i = pos + avi_file->start; i > 0; i--)
    if (avi_file->index[i].size > 0)
      break;
  offset = avi_file->index[i].offset;
  size = avi_file->index[i].size;
  if (avi_file->avi_memory)
    {
      /* ja:メモリ */
      g_memmove (buffer,
        (guint8 *)avi_memory_get_data (avi_file->avi_memory) + offset, size);
    }
  else
    {
      /* ja:ファイル */
      gboolean result;

#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      result = fileio_seek (avi_file->fio, offset, FILEIO_SEEK_SET) != -1
                        && fileio_read (avi_file->fio, buffer, size) == size;
# ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
# endif /* USE_THREAD */
      if (!result)
        return FALSE;
    }
  return TRUE;
}


/*  ja:AVI(オーディオ)のデータのバイト数を取得する
    avi_file,AVIファイル構造体
       start,取得する実サンプル番号
     samples,実サンプル数
         RET,バイト数                                                       */
gsize
avi_read_audio_size (AviFile    *avi_file,
                     const gint  start,
                     const gint  samples)
{
  return avi_file && avi_file->wfx
                        ? samples * wfx_get_block_align (avi_file->wfx) : 0;
}


/*  ja:AVI(オーディオ)のデータを読み込む
    avi_file,AVIファイル構造体
       start,取得する実サンプル番号
     samples,実サンプル数
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_audio (AviFile    *avi_file,
                const gint  start,
                const gint  samples,
                gpointer    buffer)
{
  gint i, real_start, real_samples, buf_bytes = 0, buf_samples = 0;
  gint dif = 0;/* ja:先頭のエントリーの読み飛ばすサンプル数 */

  if (!avi_file || !avi_file->wfx || !buffer || start < 0 || samples <= 0)
    return FALSE;                   /* ja:範囲外 */
  real_samples = CLAMP (avi_audio_real_from_virtual_roundup (avi_file->length,
                                        avi_file->wfx), 1, avi_file->samples);
  real_start = MIN (avi_audio_real_from_virtual (avi_file->start,
                            avi_file->wfx), avi_file->samples - real_samples);
  if (real_samples < start + samples)
    return FALSE;                   /* ja:範囲外 */
  real_start += start;
  /* ja:データの先頭があるエントリーを求める */
  for (i = 0; i < avi_file->entries; i++)
    {
      gint n;

      n = avi_file->index[i].size / wfx_get_block_align (avi_file->wfx);
      if (real_start < dif + n)
        break;
      dif += n;
    }
  dif = real_start - dif;
  /* ja:バッファにファイルのサンプルを確保する */
  while (buf_samples < samples && i < avi_file->entries)
    {
      goffset off;
      gint read_samples, read_bytes;

      read_samples = MIN (samples - buf_samples,
        (avi_file->index[i].size / wfx_get_block_align (avi_file->wfx)) - dif);
      read_bytes = read_samples * wfx_get_block_align (avi_file->wfx);
      off = (goffset)dif * wfx_get_block_align (avi_file->wfx)
                                                + avi_file->index[i].offset;
      if (avi_file->avi_memory)
        {
          /* ja:メモリ */
          g_memmove ((guint8 *)buffer + buf_bytes,
                    (guint8 *)avi_memory_get_data (avi_file->avi_memory) + off,
                                                                read_bytes);
        }
      else
        {
          gboolean result;

#ifdef USE_THREAD
          G_LOCK (critical);
          critical = TRUE;
#endif /* USE_THREAD */
          /* ja:ファイル */
          result = fileio_seek (avi_file->fio, off, FILEIO_SEEK_SET) != -1
                    && fileio_read (avi_file->fio,
                    (guint8 *)buffer + buf_bytes, read_bytes) == read_bytes;
# ifdef USE_THREAD
          critical = FALSE;
          G_UNLOCK (critical);
# endif /* USE_THREAD */
          if (!result)
            return FALSE;
        }
      buf_bytes += read_bytes;
      buf_samples += read_samples;
      dif = 0;
      i++;
    }
  return buf_samples > 0;
}
