/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "aviclip.h"
#ifdef G_OS_WIN32
# include <gdk/gdkwin32.h>
# include <tchar.h>
#endif /* G_OS_WIN32 */


enum
{
  PASTE_SIGNAL,
  LAST_SIGNAL
};
#ifndef G_OS_WIN32
enum
{
  TARGET_VMAID
};
#endif /* not G_OS_WIN32 */


static void avi_clip_class_init (AviClipClass *klass);
static void avi_clip_init       (AviClip      *clip);
static void avi_clip_dispose    (GObject      *object);


static GObjectClass *parent_class = NULL;
static gint avi_clip_signals[LAST_SIGNAL] = {0};
#ifdef G_OS_WIN32
static UINT uFormat;            /* ja:クリップボード */
#else /* not G_OS_WIN32 */
static GdkAtom atom_targets;    /* ja:ターゲット */
static GdkAtom atom_scenario;   /* ja:シナリオオブジェクト */
#endif /* not G_OS_WIN32 */


/******************************************************************************
*                                                                             *
******************************************************************************/
GType
avi_clip_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      const static GTypeInfo info =
      {
        sizeof (AviClipClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)avi_clip_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (AviClip),
        0,              /* n_preallocs */
        (GInstanceInitFunc)avi_clip_init,
      };

      type = g_type_register_static (G_TYPE_OBJECT, "AviClip", &info, 0);
    }

  return type;
}


static void
avi_clip_class_init (AviClipClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass *)klass;

  object_class->dispose = avi_clip_dispose;

  klass->paste = NULL;

  avi_clip_signals[PASTE_SIGNAL]
        = g_signal_new ("paste",
                G_TYPE_FROM_CLASS (klass),
                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                G_STRUCT_OFFSET (AviClipClass, paste),
                NULL, NULL,
                g_cclosure_marshal_VOID__POINTER,
                G_TYPE_NONE, 1,
                G_TYPE_POINTER);

#ifdef G_OS_WIN32
  uFormat = RegisterClipboardFormat (_T("Video maid Scenario Object"));
#else /* not G_OS_WIN32 */
  atom_targets = gdk_atom_intern ("TARGETS", FALSE);
  atom_scenario = gdk_atom_intern ("Video maid Scenario Object", FALSE);
#endif /* not G_OS_WIN32 */
}


/*  ja:新規作成
    RET,オブジェクト                                                        */
static void
avi_clip_init (AviClip *clip)
{
  clip->handler_destroy  = 0;
  clip->window           = NULL;
#ifndef G_OS_WIN32
  clip->scenario         = NULL;
  clip->handler_clear    = 0;
  clip->handler_get      = 0;
  clip->handler_received = 0;
#endif /* not G_OS_WIN32 */
}


static void
avi_clip_dispose (GObject *object)
{
  AviClip *clip = AVI_CLIP (object);

#ifndef G_OS_WIN32
  g_free (clip->scenario);
  if (clip->handler_clear)
    g_signal_handler_disconnect (G_OBJECT (clip->window), clip->handler_clear);
  if (clip->handler_get)
    g_signal_handler_disconnect (G_OBJECT (clip->window), clip->handler_get);
  if (clip->handler_received)
    g_signal_handler_disconnect (G_OBJECT (clip->window),
                                                    clip->handler_received);
#endif /* not G_OS_WIN32 */
  if (clip->handler_destroy)
    g_signal_handler_disconnect (G_OBJECT (clip->window),
                                                        clip->handler_destroy);

  if (G_OBJECT_CLASS (parent_class)->dispose)
    G_OBJECT_CLASS (parent_class)->dispose (object);
}


#ifndef G_OS_WIN32
static gboolean
avi_clip_selection_clear (GtkWidget         *widget,
                          GdkEventSelection *event,
                          AviClip           *clip)
{
  if (event->selection == GDK_SELECTION_CLIPBOARD)
    {
      /* ja:クリップボード */
      g_free (clip->scenario);
      clip->scenario = NULL;
    }
  return TRUE;
}


static void
avi_clip_selection_get (GtkWidget        *widget,
                        GtkSelectionData *data,
                        guint             info,
                        guint             time,
                        AviClip          *clip)
{
  /* ja:クリップボード */
  if (data->selection == GDK_SELECTION_CLIPBOARD
                                    && clip->scenario && info == TARGET_VMAID)
    gtk_selection_data_set (data, atom_scenario, 8,
                        (guchar *)clip->scenario, g_strlen (clip->scenario));
}


static void
avi_clip_selection_received (GtkWidget        *widget,
                             GtkSelectionData *data,
                             guint             time,
                             AviClip          *clip)
{
  if (data->selection == GDK_SELECTION_CLIPBOARD && data->length >= 0)
    {
      /* ja:クリップボード */
      if (data->type == GDK_SELECTION_TYPE_ATOM)
        {
          gint i, count;
          GdkAtom *atoms;

          atoms = (GdkAtom *)data->data;
          count = data->length / sizeof (GdkAtom);
          for (i = 0; i < count; i++)
            if (atoms[i] == atom_scenario)
              break;
          if (i < count)
            gtk_selection_convert (widget, data->selection, atoms[i],
                                                            GDK_CURRENT_TIME);
        }
      else if (data->type == atom_scenario)
        {
          AviEdit **avi_edit;

          avi_edit = avi_edit_from_scenario ((gchar *)data->data);
          if (avi_edit)
            {
              gint i;

              g_signal_emit_by_name (G_OBJECT (clip), "paste", avi_edit);
              for (i = 0; avi_edit[i]; i++)
                avi_edit_close (avi_edit[i]);
              g_free (avi_edit);
            }
        }
    }
}
#endif /* not G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:クリップボード関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    window,ウィジェット
       RET,オブジェクト                                                     */
GObject *
avi_clip_new (GtkWidget *window)
{
  AviClip *clip;
#ifndef G_OS_WIN32
  /* ja:提供可能なセレクション */
  GtkTargetEntry targets[1]
                        = {{"Video maid Scenario Object", 0, TARGET_VMAID}};
#endif /* not G_OS_WIN32 */

  if (!window)
    return NULL;
  clip = AVI_CLIP (g_object_new (AVI_TYPE_CLIP, NULL));
  clip->window = window;
  clip->handler_destroy = g_signal_connect_swapped (G_OBJECT (window),
                    "destroy", G_CALLBACK (g_object_unref), G_OBJECT (clip));
#ifndef G_OS_WIN32
  gtk_selection_add_targets (window, GDK_SELECTION_CLIPBOARD,
                                            targets, G_N_ELEMENTS (targets));
  clip->handler_clear = g_signal_connect (G_OBJECT (window),
                            "selection-clear-event",
                            G_CALLBACK (avi_clip_selection_clear), clip);
  clip->handler_get = g_signal_connect (G_OBJECT (window),
                            "selection-get",
                            G_CALLBACK (avi_clip_selection_get), clip);
  clip->handler_received = g_signal_connect (G_OBJECT (window),
                            "selection-received",
                            G_CALLBACK (avi_clip_selection_received), clip);
#endif /* not G_OS_WIN32 */
  return G_OBJECT (clip);
}


/*  ja:クリップボードにコピーする
        clip,オブジェクト
    avi_edit,AVI編集ハンドル                                                */
void
avi_clip_copy (AviClip  *clip,
               AviEdit **avi_edit)
{
#ifdef G_OS_WIN32
  if (clip)
    {
      gchar *scenario;
      gint i;
      HGLOBAL hGlobalScenario = NULL, hGlobalBitmap = NULL;

      scenario = avi_edit_to_scenario (avi_edit, FALSE);
      if (scenario)
        {
          hGlobalScenario = GlobalAlloc (GMEM_MOVEABLE,
                                (g_strlen (scenario) + 1) * sizeof (gchar));
          if (hGlobalScenario)
            {
              LPSTR lpScenario;

              lpScenario = GlobalLock (hGlobalScenario);
              if (lpScenario)
                {
                  g_strcpy (lpScenario, scenario);
                  GlobalUnlock (hGlobalScenario);
                }
              else
                {
                  GlobalFree (hGlobalScenario);
                  hGlobalScenario = NULL;
                }
            }
          g_free (scenario);
        }
      for (i = 0; avi_edit[i]; i++)
        if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
          {
            AviFrame *avi_frame;

            avi_frame = avi_frame_open (avi_edit[i]);
            if (avi_frame)
              {
                BitmapInfoHeader *bmih;

                bmih = avi_frame_get_bitmap (avi_frame, 0,
                                        avi_edit_get_width (avi_edit[i]),
                                        avi_edit_get_height (avi_edit[i]),
                                        avi_edit_get_bit_count (avi_edit[i]));
                if (bmih)
                  {
                    hGlobalBitmap = GlobalAlloc (GMEM_MOVEABLE,
                                                        bm_all_bytes (bmih));
                    if (hGlobalBitmap)
                      {
                        LPBITMAPINFOHEADER lpBmih;;

                        lpBmih = GlobalLock (hGlobalBitmap);
                        if (lpBmih)
                          {
                            g_memmove (lpBmih, bmih, bm_all_bytes (bmih));
                            GlobalUnlock (hGlobalBitmap);
                          }
                        else
                          {
                            GlobalFree (hGlobalBitmap);
                            hGlobalBitmap = NULL;
                          }
                      }
                  }
                avi_frame_close (avi_frame);
                if (hGlobalBitmap)
                  break;
              }
          }
      if (hGlobalScenario || hGlobalBitmap)
        {
          if (OpenClipboard (GDK_WINDOW_HWND (clip->window->window)))
            {
              if (EmptyClipboard ())
                {
                  if (hGlobalScenario
                                && SetClipboardData (uFormat, hGlobalScenario))
                    hGlobalScenario = NULL;
                  if (hGlobalBitmap
                                && SetClipboardData (CF_DIB, hGlobalBitmap))
                    hGlobalBitmap = NULL;
                }
              CloseClipboard ();
            }
          if (hGlobalScenario)
            GlobalFree (hGlobalScenario);
          if (hGlobalBitmap)
            GlobalFree (hGlobalBitmap);
        }
    }
#else /* not G_OS_WIN32 */
  if (clip)
    {
      g_free (clip->scenario);
      clip->scenario = avi_edit_to_scenario (avi_edit, FALSE);
      if (clip->scenario && !gtk_selection_owner_set (clip->window,
                                    GDK_SELECTION_CLIPBOARD, GDK_CURRENT_TIME))
        {
          g_free (clip->scenario);
          clip->scenario = NULL;
        }
    }
#endif /* not G_OS_WIN32 */
}


/*  ja:クリップボードから貼り付ける
    clip,オブジェクト                                                       */
void
avi_clip_paste (AviClip *clip)
{
#ifdef G_OS_WIN32
  if (clip && OpenClipboard (GDK_WINDOW_HWND (clip->window->window)))
    {
      AviEdit **avi_edit = NULL;

      if (IsClipboardFormatAvailable (uFormat))
        {
          HGLOBAL hGlobal;

          hGlobal = GetClipboardData (uFormat);
          if (hGlobal)
            {
              LPSTR lpScenario;

              lpScenario = GlobalLock (hGlobal);
              if (lpScenario)
                {
                  avi_edit = avi_edit_from_scenario (lpScenario);
                  GlobalUnlock (hGlobal);
                }
            }
        }
      else
        {
          AviEdit *avi_edit_video = NULL, *avi_edit_audio = NULL;

          if (IsClipboardFormatAvailable (CF_DIB))
            {
              HGLOBAL hGlobal;

              hGlobal = GetClipboardData (CF_DIB);
              if (hGlobal)
                {
                  LPBITMAPINFOHEADER lpBmih;

                  lpBmih = GlobalLock (hGlobal);
                  if (lpBmih)
                    {
                      avi_edit_video = avi_edit_open_from_memory (lpBmih);
                      GlobalUnlock (hGlobal);
                    }
                }
            }
#ifdef XXX
          if (IsClipboardFormatAvailable (CF_WAVE))
            {
              HGLOBAL hGlobal;

              hGlobal = GetClipboardData (CF_WAVE);
              if (hGlobal)
                {
                  LPVOID lpMem;

                  lpMem = GlobalLock (hGlobal);
                  if (lpMem)
                    {
                      avi_edit_audio = avi_edit_open_from_memory (lpMem);
                      GlobalUnlock (hGlobal);
                    }
                }
            }
#endif
          if (avi_edit_video || avi_edit_audio)
            {
              gint i = 0;

              avi_edit = g_malloc ((avi_edit_video && avi_edit_audio ? 3 : 2)
                                                        * sizeof (AviEdit *));
              if (avi_edit_video)
                avi_edit[i++] = avi_edit_video;
              if (avi_edit_audio)
                avi_edit[i++] = avi_edit_audio;
              avi_edit[i] = NULL;
            }
        }
      CloseClipboard ();
      if (avi_edit)
        {
          gint i;

          g_signal_emit_by_name (G_OBJECT (clip), "paste", avi_edit);
          for (i = 0; avi_edit[i]; i++)
            avi_edit_close (avi_edit[i]);
          g_free (avi_edit);
        }
    }
#else /* not G_OS_WIN32 */
  if (clip)
    gtk_selection_convert (clip->window, GDK_SELECTION_CLIPBOARD, atom_targets,
                                                            GDK_CURRENT_TIME);
#endif /* not G_OS_WIN32 */
}
