/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "avifmt.h"
#include "gsrfmt.h"
#include "misc/pixbuf.h"


#define AVI_HEADERSIZE 2048
#define AVI_ALIGNSIZE (AVI_HEADERSIZE * 8)
#define AVI_RIFFSIZE (1024 * 1024 * 1024)
#define AVI_SUPERINDEX_ENTRY                                                \
                ((AVI_ALIGNSIZE - sizeof (guint32) * 2 - ASI_SIZE) / ASE_SIZE)


/******************************************************************************
*                                                                             *
* ja:保存オプション関数                                                       *
*                                                                             *
******************************************************************************/
/*  ja:保存用のオプションを解放する
    avi_save,オプション
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_save_free (AviSave *avi_save)
{
  if (!avi_save)
    return FALSE;
  if (avi_save->avi_opt)
    {
      gint i;

      for (i = 0; avi_save->avi_opt[i].type != AVI_TYPE_UNKNOWN; i++)
        {
          g_free (avi_save->avi_opt[i].param);
          g_free (avi_save->avi_opt[i].bmih);
          g_free (avi_save->avi_opt[i].wfx);
        }
      g_free (avi_save->avi_opt);
    }
  g_free (avi_save->extension);
  g_free (avi_save->name);
  g_strfreev (avi_save->keys);
  g_strfreev (avi_save->values);
  g_free (avi_save);
  return TRUE;
}


/*  ja:保存されるファイルのタイプを返す
    avi_save,オプション
         RET,ファイルタイプ,-1:エラー                                       */
gint
avi_save_get_type (AviSave *avi_save)
{
  return avi_save ? avi_save->type : AVI_FILE_UNKNOWN;
}


/*  ja:保存されるファイルの拡張子を返す
    avi_save,オプション
         RET,拡張子,NULL:エラー                                             */
const gchar *
avi_save_get_extension (AviSave *avi_save)
{
  return avi_save ? avi_save->extension : NULL;
}


/*  ja:すべて保存されるかチェックする
    avi_edit,AVI編集ハンドル
    avi_save,オプション
         RET,TRUE:すべて保存される,FALSE:すべて保存されない                 */
gboolean
avi_save_is_all (AviEdit **avi_edit,
                 AviSave  *avi_save)
{
  return avi_edit && avi_edit[0] && avi_save
    && (avi_save_get_type (avi_save) == AVI_FILE_AVI
        || avi_save_get_type (avi_save) == AVI_FILE_GSR
        || avi_save_get_type (avi_save) == AVI_FILE_SCENARIO
        || (!avi_edit[0] && avi_save_get_type (avi_save) == AVI_FILE_WAVE));
}


/******************************************************************************
*                                                                             *
* ja:ビットマップ&WAVE保存関数                                                *
*                                                                             *
******************************************************************************/
/*  ja:ビットマップファイルに保存する
     avi_edit,AVI編集ハンドル
         file,ファイル名
        start,最初のフレーム
          end,最後のフレーム
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_bitmap (AviEdit      *avi_edit,
                 const gchar  *file,
                 const gint    start,
                 const gint    end,
                 gboolean    (*func)(gint, gpointer),
                 gpointer      user_data)
{
  gboolean result = TRUE;
  gchar *format;
  gint i;
  AviFrame *avi_frame;
  BitmapFileHeader bmfh;

  if (!avi_edit || !file || start > end)
    return FALSE;
  if (start == end)
    {
      format = g_strdup (file);
    }
  else
    {
      gchar *dir, *name, *ext;
      gint s;

      dir = g_path_get_dirname (file);
      name = g_path_get_basename (file);
      for (i = g_strlen (name) - 1; i >= 0; i--)
        if (name[i] == '.')
          break;
      if (i >= 0)
        {
          ext = g_strdup (name + i);
          name[i] = '\0';
        }
      else
        {
          ext = g_strdup ("");
        }
      for (i = 0, s = end; s != 0; s /= 10)
        i++;
      format = g_strdup_printf ("%s/%s%%0%dd%s", dir, name, i, ext);
      g_free (dir);
      g_free (name);
      g_free (ext);
    }

  avi_frame = avi_frame_open (avi_edit);
  if (!avi_frame)
    {
      g_free (format);
      return FALSE;
    }

  g_memset (&bmfh, 0, BMFH_SIZE);
  bmfh_set_type (&bmfh, BF_TYPE);
  bmfh_set_size (&bmfh, bm_all_bytes (avi_edit->bmih) + BMFH_SIZE);
  bmfh_set_off_bits (&bmfh, bm_header_bytes (avi_edit->bmih) + BMFH_SIZE);

  for (i = start; i <= end; i++)
    {
      gchar *name;
      gint length;
      BitmapInfoHeader *bmih;
      FileIO *fio;

      /* ja:中断チェック */
      if (func && !func ((i - start) * 100 / (end - start + 1), user_data))
        break;

      bmih = avi_frame_get_bitmap (avi_frame, i,
                                        bmih_get_width (avi_edit->bmih),
                                        bmih_get_height (avi_edit->bmih),
                                        bmih_get_bit_count (avi_edit->bmih));
      if (!bmih)
        {
          result = FALSE;
          break;
        }
      bmih_set_x_pixels_per_meter (bmih,
                                bmih_get_x_pixels_per_meter (avi_edit->bmih));
      bmih_set_y_pixels_per_meter (bmih,
                                bmih_get_y_pixels_per_meter (avi_edit->bmih));
      length = bm_all_bytes (bmih);

      name = g_strdup_printf (format, i);
      /* ja:bmihファイルを開く */
      fio = fileio_open (name, FILEIO_ACCESS_ALL,
                                    FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW);
      g_free (name);
      if (!fio)
        {
          result = FALSE;
          break;
        }
      if (fileio_write (fio, &bmfh, BMFH_SIZE) != BMFH_SIZE
                                || fileio_write (fio, bmih, length) != length)
        {
          fileio_close (fio);
          result = FALSE;
          break;
        }
      if (!fileio_close (fio))
        {
          result = FALSE;
          break;
        }
    }
  g_free (format);
  return avi_frame_close (avi_frame) && result;
}


/*  ja:WAVEファイルに保存する
      avi_edit,AVI編集ハンドル
          file,ファイル名
           wfx,WaveFormatEx構造体へのポインタ(NULL:PCM)
    recompress,TRUE:再圧縮,FALSE:差分圧縮(wfxがNULLならば無視)
          func,コールバック関数
     user_data,データ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
avi_save_wave (AviEdit             *avi_edit,
               const gchar         *file,
               const WaveFormatEx  *wfx,
               const gboolean       recompress,
               gboolean           (*func)(gint, gpointer),
               gpointer             user_data)
{
  FileIO *fio;
  gpointer head;
  gsize header_bytes;
  guint8 *p;
  gint written = 0, header;
  AviPcm *avi_pcm;
  AcmObject *acm_object = NULL;

  if (!avi_edit || !file || (wfx && wfx_get_format_tag (wfx) == WAVE_FMT_PCM))
    return FALSE;
  header_bytes = wfx ? wf_header_bytes (wfx) : WFX_SIZE - sizeof (guint16);
  /* ja:ヘッタサイズを計算する */
  header = sizeof (guint32)                                     /* en:'RIFF' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + sizeof (guint32)                                  /* en:'WAVE' */
            + sizeof (guint32)                                  /* en:'fmt ' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + ((header_bytes + 1) & ~1)                         /* ja:構造体 */
            + sizeof (guint32)                                  /* en:'data' */
            + sizeof (guint32);                                 /* ja:サイズ */
  /* ja:ファイルを開く */
  fio = fileio_open (file, FILEIO_ACCESS_ALL,
                                    FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW);
  if (!fio)
    return FALSE;
  if (fileio_seek (fio, header, FILEIO_SEEK_SET) == -1)
    {
      fileio_close (fio);
      return FALSE;
    }
  /* ja:データ */
  avi_pcm = avi_pcm_open (avi_edit);
  if (!avi_pcm)
    {
      fileio_close (fio);
      return FALSE;
    }
  if (wfx && !(acm_object = acm_open (wfx_get_format_tag (wfx))))
    {
      avi_pcm_close (avi_pcm);
      fileio_close (fio);
      return FALSE;
    }
  if (wfx && !recompress)
    { /* ja:差分圧縮 */
      gboolean flag = TRUE, acm = FALSE;
      gint offset = 0;
      AviFile *avi_file;

      for (avi_file = avi_edit->file; avi_file->prev;
                                                    avi_file = avi_file->prev);
      while (avi_file && flag)
        {
          if (wf_header_cmp (avi_file->wfx, wfx))
            { /* ja:再利用 */
              gint pos = 0, samples, leng;

              leng = CLAMP (avi_audio_real_from_virtual_roundup
                    (avi_file->length, avi_file->wfx), 1, avi_file->samples);
              samples = MAX (wfx_get_average_bytes_per_sec (wfx)
                                            / wfx_get_block_align (wfx), 1);
              while (pos < leng)
                {
                  gpointer buffer;
                  gint length;

                  /* ja:中断チェック */
                  if (func
                    && !(flag = func ((offset + avi_file->length * pos / leng)
                            * 100 / avi_edit_length (avi_edit), user_data)))
                    break;
                  if (leng < pos + samples)
                    samples = leng - pos;
                  buffer = g_malloc (avi_read_audio_size (avi_file,
                                                                pos, samples));
                  if (!avi_read_audio (avi_file, pos, samples, buffer))
                    {
                      acm_close (acm_object);
                      avi_pcm_close (avi_pcm);
                      fileio_close (fio);
                      return FALSE;
                    }
                  pos += samples;
                  length = samples * wfx_get_block_align (wfx);
                  if (fileio_write (fio, buffer, length) != length)
                    {
                      g_free (buffer);
                      acm_close (acm_object);
                      avi_pcm_close (avi_pcm);
                      fileio_close (fio);
                      return FALSE;
                    }
                  g_free (buffer);
                  written += length;
                }
            }
          else
            { /* ja:圧縮 */
              gint pos = 0, samples = AVI_PCM_SAMPLES_PER_SEC;

              if (!acm)
                {
                  if (!acm_compress_begin (acm_object, avi_edit->wfx, wfx))
                    {
                      acm_close (acm_object);
                      avi_pcm_close (avi_pcm);
                      fileio_close (fio);
                      return FALSE;
                    }
                  acm = TRUE;
                }
              while (pos < avi_file->length)
                {
                  gpointer out_buf, acm_buf;
                  gsize out_samples, acm_samples, length;

                  /* ja:中断チェック */
                  if (func && !(flag = func ((pos + offset) * 100
                                    / avi_edit_length (avi_edit), user_data)))
                    break;
                  if (avi_edit->length < pos + samples)
                    samples = avi_edit->length - pos;
                  if (!avi_pcm_get_raw (avi_pcm, pos + offset, samples,
                                                    &out_buf, &out_samples))
                    {
                      acm_compress_end (acm_object, NULL, NULL);
                      acm_close (acm_object);
                      avi_pcm_close (avi_pcm);
                      fileio_close (fio);
                      return FALSE;
                    }
                  pos += samples;
                  if (!acm_compress (acm_object,
                                out_buf, out_samples, &acm_buf, &acm_samples))
                    {
                      g_free (out_buf);
                      acm_compress_end (acm_object, NULL, NULL);
                      acm_close (acm_object);
                      avi_pcm_close (avi_pcm);
                      fileio_close (fio);
                      return FALSE;
                    }
                  g_free (out_buf);
                  length = acm_samples * wfx_get_block_align (wfx);
                  if (length > 0
                            && fileio_write (fio, acm_buf, length) != length)
                    {
                      g_free (acm_buf);
                      acm_compress_end (acm_object, NULL, NULL);
                      acm_close (acm_object);
                      avi_pcm_close (avi_pcm);
                      fileio_close (fio);
                      return FALSE;
                    }
                  g_free (acm_buf);
                  written += length;
                }
              if (!avi_file->next || wf_header_cmp (avi_file->next->wfx, wfx))
                {
                  gpointer acm_buf;
                  gsize acm_samples, length;

                  if (!acm_compress_end (acm_object, &acm_buf, &acm_samples))
                    {
                      acm_close (acm_object);
                      avi_pcm_close (avi_pcm);
                      g_free (acm_buf);
                      fileio_close (fio);
                      return FALSE;
                    }
                  length = acm_samples * wfx_get_block_align (wfx);
                  if (length > 0
                            && fileio_write (fio, acm_buf, length) != length)
                    {
                      g_free (acm_buf);
                      avi_pcm_close (avi_pcm);
                      fileio_close (fio);
                      return FALSE;
                    }
                  g_free (acm_buf);
                  written += length;
                  acm = FALSE;
                }
            }
          offset = avi_file->length;
          avi_file = avi_file->next;
        }
    }
  else
    { /* ja:PCMまたは再圧縮 */
      gint pos = 0, samples = AVI_PCM_SAMPLES_PER_SEC;

      if (acm_object && !acm_compress_begin (acm_object, avi_edit->wfx, wfx))
        {
          acm_close (acm_object);
          avi_pcm_close (avi_pcm);
          fileio_close (fio);
          return FALSE;
        }
      while (pos < avi_edit_length (avi_edit))
        {
          gpointer out_buf;
          gsize out_samples, length;

          /* ja:中断チェック */
          if (func && !func (pos * 100
                                    / avi_edit_length (avi_edit), user_data))
            break;
          if (avi_edit_length (avi_edit) < pos + samples)
            samples = avi_edit_length (avi_edit) - pos;
          if (!avi_pcm_get_raw (avi_pcm, pos, samples, &out_buf, &out_samples))
            {
              if (acm_object)
                {
                  acm_compress_end (acm_object, NULL, NULL);
                  acm_close (acm_object);
                }
              avi_pcm_close (avi_pcm);
              fileio_close (fio);
              return FALSE;
            }
          pos += samples;
          if (acm_object)
            {
              gpointer acm_buf;
              gsize acm_samples;

              if (!acm_compress (acm_object,
                                out_buf, out_samples, &acm_buf, &acm_samples))
                {
                  g_free (out_buf);
                  acm_compress_end (acm_object, NULL, NULL);
                  acm_close (acm_object);
                  avi_pcm_close (avi_pcm);
                  fileio_close (fio);
                  return FALSE;
                }
              g_free (out_buf);
              out_buf = acm_buf;
              length = acm_samples * wfx_get_block_align (wfx);
            }
          else
            {
              length = out_samples * wfx_get_block_align (avi_edit->wfx);
            }
          if (length > 0 && fileio_write (fio, out_buf, length) != length)
            {
              g_free (out_buf);
              if (acm_object)
                {
                  acm_compress_end (acm_object, NULL, NULL);
                  acm_close (acm_object);
                }
              avi_pcm_close (avi_pcm);
              fileio_close (fio);
              return FALSE;
            }
          g_free (out_buf);
          written += length;
        }
      if (acm_object)
        {
          gpointer acm_buf;
          gsize acm_samples, length;

          if (!acm_compress_end (acm_object, &acm_buf, &acm_samples))
            {
              acm_close (acm_object);
              avi_pcm_close (avi_pcm);
              g_free (acm_buf);
              fileio_close (fio);
              return FALSE;
            }
          length = acm_samples * wfx_get_block_align (wfx);
          if (length > 0 && fileio_write (fio, acm_buf, length) != length)
            {
              g_free (acm_buf);
              acm_close (acm_object);
              avi_pcm_close (avi_pcm);
              fileio_close (fio);
              return FALSE;
            }
          g_free (acm_buf);
          written += length;
        }
    }
  if ((acm_object && !acm_close (acm_object)) | !avi_pcm_close (avi_pcm))
    {
      fileio_close (fio);
      return FALSE;
    }
  if (written & 1)
    {
      const static guint8 dmy = 0;

      if (fileio_write (fio, &dmy, 1) != 1)
        {
          fileio_close (fio);
          return FALSE;
        }
    }
  /* ja:ヘッタを設定する */
  p = head = g_malloc (header);
  *(guint32 *)p = GUINT32_TO_LE (CK_DEF_RIFF);
  p += sizeof (guint32);                                        /* en:'RIFF' */
  *(guint32 *)p = GUINT32_TO_LE (header - sizeof (guint32) * 2
                                                    + ((written + 1) & ~1));
  p += sizeof (guint32);                                        /* ja:サイズ */
  *(guint32 *)p = GUINT32_TO_LE (CK_FORM_WAVE);
  p += sizeof (guint32);                                        /* en:'WAVE' */
  *(guint32 *)p = GUINT32_TO_LE (CK_ID_WAVEFORMAT);
  p += sizeof (guint32);                                        /* en:'fmt ' */
  *(guint32 *)p = GUINT32_TO_LE (header_bytes);
  p += sizeof (guint32);                                        /* ja:サイズ */
  g_memmove (p, wfx ? wfx : avi_edit->wfx, header_bytes);
  if (header_bytes & 1)
    p[header_bytes] = 0;
  p += (header_bytes + 1) & ~1;
  *(guint32 *)p = GUINT32_TO_LE (CK_ID_WAVEDATA);
  p += sizeof (guint32);                                        /* en:'data' */
  *(guint32 *)p = GUINT32_TO_LE (written);                      /* ja:サイズ */
  if (fileio_seek (fio, 0 , FILEIO_SEEK_SET) == -1
                                || fileio_write (fio, head, header) != header)
    {
      fileio_close (fio);
      g_free (head);
      return FALSE;
    }
  g_free (head);
  return fileio_close (fio);
}


/******************************************************************************
*                                                                             *
* ja:GdkPixbuf保存関数                                                        *
*                                                                             *
******************************************************************************/
/*  ja:GdkPixbufで保存する
     avi_edit,AVI編集ハンドル
         file,ファイル名
     avi_save,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
static gboolean
avi_save_pixbuf (AviEdit      *avi_edit,
                 const gchar  *file,
                 AviSave      *avi_save,
                 gboolean    (*func)(gint, gpointer),
                 gpointer      user_data)
{
  gboolean result = TRUE;
  gchar *format;
  gint i;
  AviFrame *avi_frame;

  if (!avi_edit || !file)
    return FALSE;
  if (avi_save->start == avi_save->end)
    {
      format = g_strdup (file);
    }
  else
    {
      gchar *dir, *name, *ext;
      gint s;

      dir = g_path_get_dirname (file);
      name = g_path_get_basename (file);
      for (i = g_strlen (name) - 1; i >= 0; i--)
        if (name[i] == '.')
          break;
      if (i >= 0)
        {
          ext = g_strdup (name + i);
          name[i] = '\0';
        }
      else
        {
          ext = g_strdup ("");
        }
      for (i = 0, s = avi_save->end; s != 0; s /= 10)
        i++;
      format = g_strdup_printf ("%s/%s%%0%dd%s", dir, name, i, ext);
      g_free (dir);
      g_free (name);
      g_free (ext);
    }

  avi_frame = avi_frame_open (avi_edit);
  if (!avi_frame)
    {
      g_free (format);
      return FALSE;
    }
  for (i = avi_save->start; i <= avi_save->end; i++)
    {
      BitmapInfoHeader *bmih;
      GdkPixbuf *pixbuf;

      /* ja:中断チェック */
      if (func && !func ((i - avi_save->start) * 100
                        / (avi_save->end - avi_save->start + 1), user_data))
        break;

      bmih = avi_frame_get_bitmap (avi_frame, i,
                                        bmih_get_width (avi_edit->bmih),
                                        bmih_get_height (avi_edit->bmih),
                                        bmih_get_bit_count (avi_edit->bmih));
      if (!bmih)
        {
          result = FALSE;
          break;
        }
      bmih_set_x_pixels_per_meter (bmih,
                                bmih_get_x_pixels_per_meter (avi_edit->bmih));
      bmih_set_y_pixels_per_meter (bmih,
                                bmih_get_y_pixels_per_meter (avi_edit->bmih));

      /* ja:GdkPixbufで保存 */
      pixbuf = pixbuf_from_bitmap (bmih);
      if (pixbuf)
        {
          gchar *name;

          name = g_strdup_printf (format, i);
          result = gdk_pixbuf_savev (pixbuf, name, avi_save->name,
                                    avi_save->keys, avi_save->values, NULL);
          g_free (name);
          g_object_unref (G_OBJECT (pixbuf));
        }
      else
        {
          result = FALSE;
        }
      if (!result)
        break;
    }
  g_free (format);
  return avi_frame_close (avi_frame) && result;
}


/******************************************************************************
*                                                                             *
* ja:AVI保存補助関数                                                          *
*                                                                             *
******************************************************************************/
typedef struct _AviAcm
{
  gpointer tmp_buf;
  gboolean recompress;
  gboolean acm_flag;    /* ja:TRUE:acm_compress_begin呼出済 */
  gint tmp_samples;     /* ja:テンポラリの実サンプル数 */
  gint in_samples;      /* ja:累積入力仮想サンプル数 */
  gint out_samples;     /* ja:累積出力実サンプル数 */
  gint pos;             /* ja:前回の終了点 */
  gint marker;          /* ja:読み込みの終了点 */
  AviEdit *avi_edit;
  AviPcm *avi_pcm;
  AcmObject *acm_object;
  WaveFormatEx *wfx;
} AviAcm;


/*  ja:ACMで圧縮されたオーディオを開く
      avi_edit,AVI編集ハンドル
           wfx,WaveFormatEx構造体へのポインタ
    recompress,TRUE:再圧縮,FALSE:差分圧縮
           RET,AVIACM構造体,NULL:エラー                                     */
static AviAcm *
avi_acm_open (AviEdit            *avi_edit,
              const WaveFormatEx *wfx,
              const gboolean      recompress)
{
  AviAcm *avi_acm;

  if (!avi_edit || avi_edit_type (avi_edit) != AVI_TYPE_AUDIO
                        || (wfx && wfx_get_format_tag (wfx) == WAVE_FMT_PCM))
    return NULL;
  avi_acm = g_malloc0 (sizeof (AviAcm));
  avi_acm->avi_edit = avi_edit;
  avi_acm->avi_pcm = avi_pcm_open (avi_edit);
  if (!avi_acm->avi_pcm)
    {
      g_free (avi_acm);
      return NULL;
    }
  if (wfx)
    {
      avi_acm->recompress = recompress;
      avi_acm->acm_object = acm_open (wfx_get_format_tag (wfx));
      if (!avi_acm->acm_object)
        {
          avi_pcm_close (avi_acm->avi_pcm);
          g_free (avi_acm);
          return NULL;
        }
      avi_acm->wfx = g_memdup (wfx, wf_header_bytes (wfx));
    }
  return avi_acm;
}


/*  ja:ACMで圧縮されたオーディオを閉じる
    avi_pcm,AVIACM構造体
        RET,TRUE:正常終了,FALSE:エラー                                      */
static gboolean
avi_acm_close (AviAcm *avi_acm)
{
  gboolean result;

  if (!avi_acm)
    return FALSE;
  result = (!avi_acm->avi_pcm || avi_pcm_close (avi_acm->avi_pcm))
         & (!avi_acm->acm_flag
                        || acm_compress_end (avi_acm->acm_object, NULL, NULL))
         & (!avi_acm->acm_object || acm_close (avi_acm->acm_object));
  g_free (avi_acm->wfx);
  g_free (avi_acm->tmp_buf);
  g_free (avi_acm);
  return result;
}


/*  ja:バッファに追加する
        avi_acm,AVIACM構造体
            buf,追加するデータ
        samples,追加するデータの実サンプル数                                */
static void
avi_acm_add_buffer (AviAcm        *avi_acm,
                    gconstpointer  buf,
                    const gint     samples)
{
  avi_acm->tmp_buf = g_realloc (avi_acm->tmp_buf,
        (avi_acm->tmp_samples + samples) * wfx_get_block_align (avi_acm->wfx));
  g_memmove ((guint8 *)avi_acm->tmp_buf + avi_acm->tmp_samples
                                        * wfx_get_block_align (avi_acm->wfx),
                            buf, samples * wfx_get_block_align (avi_acm->wfx));
  avi_acm->tmp_samples += samples;
}


/*  ja:PCMに展開されたオーディオを取得する
        avi_pcm,AVIPCM構造体
          start,取得する仮想サンプル番号
        samples,仮想サンプル数
        out_buf,PCMデータ
    out_samples,PCMデータの実サンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
static gboolean
avi_acm_get_raw (AviAcm      *avi_acm,
                 const gsize  samples,
                 gpointer    *out_buf,
                 gsize       *out_samples)
{
  gpointer result_buf;
  gsize real_samples, virtual_samples, result_samples, result_bytes, tmp_bytes;
  AviEdit *avi_edit;

  if (!avi_acm || samples <= 0)
    return FALSE;
  if (!avi_acm->wfx)
    {
      gboolean result;

      result = avi_pcm_get_raw (avi_acm->avi_pcm, avi_acm->pos, samples,
                                                        out_buf, out_samples);
      avi_acm->pos += samples;
      return result;
    }
  avi_edit = avi_acm->avi_edit;
  real_samples = avi_audio_real_from_virtual_roundup
        (avi_acm->in_samples + samples, avi_acm->wfx) - avi_acm->out_samples;
  virtual_samples = avi_audio_real_to_virtual
                (MAX (real_samples - avi_acm->tmp_samples, 1), avi_acm->wfx);
  while (avi_acm->tmp_samples < real_samples)
    {
      gboolean raw = FALSE;

      if (avi_acm->pos >= avi_edit_length (avi_edit))
        {
          if (avi_acm->acm_flag)
            {
              gpointer acm_buf;
              gsize acm_samples;

              if (!acm_compress_end (avi_acm->acm_object,
                                                    &acm_buf, &acm_samples))
                return FALSE;
              avi_acm->acm_flag = FALSE;
              avi_acm_add_buffer (avi_acm, acm_buf, acm_samples);
              g_free (acm_buf);
            }
          break;
        }
      if (!avi_acm->recompress)
        {
          /* ja:差分圧縮 */
          /* ja:AVIファイルを求める */
          avi_base_get_file (avi_edit, avi_acm->pos);
          if (wf_header_cmp (avi_acm->wfx, avi_edit->file->wfx))
            raw = TRUE;
        }
      if (raw)
        {
          /* ja:元の圧縮済みのデータを使う */
          gpointer buffer;
          gint file_length, file_samples;

          if (avi_acm->acm_flag)
            {
              gpointer acm_buf;
              gsize acm_samples;

              if (!acm_compress_end (avi_acm->acm_object,
                                                    &acm_buf, &acm_samples))
                return FALSE;
              avi_acm->acm_flag = FALSE;
              avi_acm_add_buffer (avi_acm, acm_buf, acm_samples);
              g_free (acm_buf);
            }
          file_length = CLAMP (avi_audio_real_from_virtual_roundup
                                (avi_edit->file->length, avi_edit->file->wfx),
                                                1, avi_edit->file->samples);
          file_samples = MIN (file_length - avi_acm->marker,
                                        real_samples - avi_acm->tmp_samples);
          buffer = g_malloc (avi_read_audio_size (avi_edit->file,
                                            avi_acm->marker, file_samples));
          if (!avi_read_audio (avi_edit->file,
                                        avi_acm->marker, file_samples, buffer))
            {
              g_free (buffer);
              return FALSE;
            }
          avi_acm_add_buffer (avi_acm, buffer, file_samples);
          g_free (buffer);
          avi_acm->marker += file_samples;
          if (avi_acm->marker >= file_length)
            {
              avi_acm->pos += avi_edit->file->length;
              avi_acm->marker = 0;
            }
          virtual_samples -= avi_edit->file->length;
          if (virtual_samples <= 0)
            virtual_samples = avi_audio_real_to_virtual
                        (real_samples - avi_acm->tmp_samples, avi_acm->wfx);
        }
      else
        {
          /* ja:再圧縮 */
          gpointer pcm_buf, acm_buf;
          gsize length, pcm_samples, acm_samples;

          length = avi_acm->recompress ? virtual_samples
            : MIN (avi_edit->file->length + avi_edit->offset - avi_acm->pos,
                                                            virtual_samples);
          if (!avi_pcm_get_raw (avi_acm->avi_pcm, avi_acm->pos, length,
                                                    &pcm_buf, &pcm_samples))
            return FALSE;
          if (!avi_acm->acm_flag && !acm_compress_begin (avi_acm->acm_object,
                                                avi_edit->wfx, avi_acm->wfx))
            {
              g_free (pcm_buf);
              return FALSE;
            }
          avi_acm->acm_flag = TRUE;
          if (!acm_compress (avi_acm->acm_object, pcm_buf, pcm_samples,
                                                    &acm_buf, &acm_samples))
            {
              g_free (pcm_buf);
              return FALSE;
            }
          g_free (pcm_buf);
          avi_acm_add_buffer (avi_acm, acm_buf, acm_samples);
          g_free (acm_buf);
          avi_acm->pos += length;
          virtual_samples -= length;
          if (virtual_samples <= 0)
            virtual_samples = CLAMP (avi_audio_real_to_virtual_roundup
                        (real_samples - avi_acm->tmp_samples, avi_acm->wfx),
                                1, avi_edit_length (avi_edit) - avi_acm->pos);
        }
    }
  result_samples = MIN (avi_acm->tmp_samples, real_samples);
  result_bytes = result_samples * wfx_get_block_align (avi_acm->wfx);
  result_buf = g_malloc (result_bytes);
  g_memmove (result_buf, avi_acm->tmp_buf, result_bytes);
  avi_acm->tmp_samples -= result_samples;
  tmp_bytes = avi_acm->tmp_samples * wfx_get_block_align (avi_acm->wfx);
  g_memmove (avi_acm->tmp_buf,
                        (guint8 *)avi_acm->tmp_buf + result_bytes, tmp_bytes);
  avi_acm->tmp_buf = g_realloc (avi_acm->tmp_buf, tmp_bytes);
  if (out_buf)
    *out_buf = result_buf;
  else
    g_free (result_buf);
  if (out_samples)
    *out_samples = result_samples;
  avi_acm->in_samples += samples;
  avi_acm->out_samples += result_samples;
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:AVI保存関数                                                              *
*                                                                             *
******************************************************************************/
#define AVI_SAVE_WRITE_KEYFRAME 0
#define AVI_SAVE_WRITE_VIDEO 1
#define AVI_SAVE_WRITE_AUDIO 2
#define AVI_SAVE_WRITE_FINALIZE -1


typedef struct _AviSaveAviStream
{
  gboolean active;          /* ja:TRUE:圧縮on,FALSE:圧縮off */
  gboolean recompress;      /* ja:TRUE:次のキーフレームまで再圧縮中 */
  gsize bmih_size;          /* ja:圧縮されたときの最大サイズ */
  gpointer fmt;             /* ja:ストリームのヘッタ */
  gsize fmt_size;           /* ja:ストリームのヘッタのサイズ */
  gint pos;                 /* ja:ストリームの現在位置 */
  AviFrame *avi_frame;
  AviAcm *avi_acm;
  IcmObject *icm_object;
  gint entries;             /* ja:AviBaseIndexEntry/GsrIndexEntryの数 */
  /* ja:AVI固有 */
  gsize max;                /* ja:書き込んだチャンクの最大サイズ */
  guint32 fourcc;           /* ja:4文字コード */
  gint samples;             /* ja:RIFF内のサンプル数 */
  gint written;             /* ja:書き込んだサンプル数 */
  AviSuperIndexEntry super[AVI_SUPERINDEX_ENTRY];
  AviBaseIndexEntry *base;
  /* ja:GSR固有 */
  GsrIndexEntry *index;
} AviSaveAviStream;


/*  ja:ファイルに書き込む
          fio,ファイルポインタ
         data,データ
       length,dataのバイト数
      samples,サンプル数(0以下ならばファイナライズ)
    key_frame,0:キーフレーム,1:非キーフレーム,2:オーディオ
        index,AviIndexEntry構造体
      entries,AviIndexEntry構造体の数
         riff,現在のRIFFのオフセット
         movi,現在のmoviのオフセット
       stream,ストリームのデータ
      streams,ストリームの数
      current,データの属するストリーム(0未満ならばファイナライズ)
          RET,TRUE:正常終了,FALSE:エラー                                    */
static gboolean
avi_save_avi_write (FileIO            *fio,
                    gconstpointer      data,
                    gsize              length,
                    gint               samples,
                    gint               key_frame,
                    AviIndexEntry    **index,
                    gint              *entries,
                    goffset           *riff,
                    goffset           *movi,
                    AviSaveAviStream  *stream,
                    gint               streams,
                    gint               current)
{
  gint i, size;
  guint8 *p, *dummy;
  guint32 fcc, cb, type;
  goffset offset;

  if (!fio || !index || !riff || !movi || !stream || streams <= 0)
    return FALSE;
  offset = fileio_seek (fio, 0, FILEIO_SEEK_CUR);
  if (offset == -1)
    return FALSE;
  if (samples > 0)
    {
      if (current < 0 || streams < current)
        return FALSE;
      size = offset - *riff - sizeof (guint32) * 2;         /* ja:既存データ */
      /* ja:書き込み後の現在のRIFFのデータサイズを計算する */
      if (data && length > 0)
        size +=     sizeof (guint32)                        /* ja:FOURCC */
                  + sizeof (guint32)                        /* ja:サイズ */
                  + ((length + 1) & ~1);                    /* ja:データ */
      /* ja:各ストリームのAviBaseIndexを加える */
      for (i = 0; i < streams; i++)
        if (stream[i].entries > 0 || i == current)
          {
            gint s;

            s = sizeof (guint32) * 2 + ABI_SIZE + stream[i].entries * ABE_SIZE;
            if (i == current)
              s += ABE_SIZE;
            if (s % AVI_ALIGNSIZE != 0)
              s = (s / AVI_ALIGNSIZE + 1) * AVI_ALIGNSIZE;
            size += s;
          }
      /* ja:最初のRIFFならばidx1チャンクも加える */
      if (*riff <= 0)
        size += sizeof (guint32) * 2 + (*entries + 1) * AIE_SIZE;
    }
  else
    { /* ja:ファイナライズ */
      if (data || length > 0 || current >= 0)
        return FALSE;
      size = -1;
    }
  if (samples <= 0 || AVI_RIFFSIZE <= size)
    { /* ja:RIFFを更新する処理 */
      if (*riff <= 0)
        { /* ja:最初のRIFF */
          /* ja:すべてのストリームは出力されている */
          for (i = 0; i < streams; i++)
            if (stream[i].entries <= 0)
              return FALSE;
        }
      else
        {
          /* ja:最低でも1つのストリームは出力されている */
          for (i = 0; i < streams; i++)
            if (stream[i].entries > 0)
              break;
          if (i >= streams)
            return FALSE;
        }
      /* ja:ix??の出力 */
      for (i = 0; i < streams; i++)
        if (stream[i].entries > 0)
          {
            gint j, s;
            AviBaseIndex abi;

            s = stream[i].entries * ABE_SIZE;
            size = sizeof (guint32) * 2 + ABI_SIZE + s;
            if (size % AVI_ALIGNSIZE != 0)
              size = (size / AVI_ALIGNSIZE + 1) * AVI_ALIGNSIZE;
            /* en:AviSuperIndexEntry */
            for (j = 0, p = (guint8 *)stream[i].super;
                                j < AVI_SUPERINDEX_ENTRY; j++, p += ASE_SIZE)
              if (ase_get_size (p) == 0)
                break;
            if (j >= AVI_SUPERINDEX_ENTRY)
              return FALSE;
            ase_set_offset (p, offset);
            ase_set_size (p, size);
            ase_set_duration (p, stream[i].samples);
            stream[i].samples = 0;
            /* en:AviBaseIndex */
            offset += size;
            size -= sizeof (guint32) * 2;
            fcc = GUINT32_TO_LE (make4cc ('i', 'x',
                                                i / 10 + '0', i % 10 + '0'));
            cb = GUINT32_TO_LE (size);
            abi_set_longs (&abi, 2);
            abi_set_sub_type (&abi, AVI_IDX_SUB_DEFAULT);
            abi_set_type (&abi, AVI_IDX_OF_CHUNKS);
            abi_set_entries (&abi, stream[i].entries);
            abi_set_ckid (&abi, stream[i].fourcc);
            abi_set_offset (&abi, *movi);
            abi_set_reserve (&abi, 0);
            if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
                    || fileio_write (fio, &cb, sizeof (guint32))
                                                            != sizeof (guint32)
                    || fileio_write (fio, &abi, ABI_SIZE) != ABI_SIZE
                    || fileio_write (fio, stream[i].base, s) != s)
              return FALSE;
            /* ja:ダミー */
            size -= ABI_SIZE + s;
            if (size > 0)
              {
                dummy = g_malloc0 (size);
                if (fileio_write (fio, dummy, size) != size)
                  {
                    g_free (dummy);
                    return FALSE;
                  }
                g_free (dummy);
              }
            stream[i].entries = 0;
          }
      /* moviのサイズ */
      cb = GUINT32_TO_LE (offset - *movi - sizeof (guint32) * 2);
      if (fileio_seek (fio, *movi + sizeof (guint32), SEEK_SET) == -1
            || fileio_write (fio, &cb, sizeof (guint32)) != sizeof (guint32))
        return FALSE;
      if (*riff <= 0)
        { /* 最初のRIFF */
          gint  s;

          /* インデックス */
          s = *entries * AIE_SIZE;
          fcc = GUINT32_TO_LE (CK_ID_AVINEWINDEX);
          cb = GUINT32_TO_LE (s);
          if (!*index || *entries < 0
                                || fileio_seek (fio, 0, FILEIO_SEEK_END) == -1
                                || fileio_write (fio, &fcc, sizeof (guint32))
                                                            != sizeof (guint32)
                                || fileio_write (fio, &cb, sizeof (guint32))
                                                            != sizeof (guint32)
                                || fileio_write (fio, *index, s) != s)
            return FALSE;
          offset += sizeof (guint32) * 2 + s;
          g_free (*index);
          *index = NULL;
          *entries = 0;
        }
      /* ja:RIFFのサイズ */
      cb = GUINT32_TO_LE (offset - *riff - sizeof (guint32) * 2);
      if (fileio_seek (fio, *riff + sizeof (guint32), SEEK_SET) == -1
            || fileio_write (fio, &cb, sizeof (guint32)) != sizeof (guint32)
                                || fileio_seek (fio, 0, FILEIO_SEEK_END) == -1)
        return FALSE;
      /* ja:新しいRIFF */
      if (samples > 0)
        {
          *riff = offset;
          fcc = GUINT32_TO_LE (CK_DEF_RIFF);
          type = GUINT32_TO_LE (CK_FORM_AVIX);
          if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
                || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
                || fileio_write (fio, &type, sizeof (guint32))
                                                        != sizeof (guint32))
            return FALSE;
          offset += sizeof (guint32) * 3;
          if ((offset + sizeof (guint32) * 3) % AVI_HEADERSIZE != 0)
            { /* en:JUNK */
              size = ((offset + sizeof (guint32) * 5) / AVI_HEADERSIZE + 1)
                            * AVI_HEADERSIZE - (offset + sizeof (guint32) * 5);
              fcc = GUINT32_TO_LE (CK_ID_AVIPADDING);
              cb = GUINT32_TO_LE (size);
              dummy = g_malloc0 (size);
              if (fileio_write (fio, &fcc, sizeof (guint32))
                                                            != sizeof (guint32)
                                || fileio_write (fio, &cb, sizeof (guint32))
                                                            != sizeof (guint32)
                                    || fileio_write (fio, dummy, size) != size)
                {
                  g_free (dummy);
                  return FALSE;
                }
              g_free (dummy);
              offset += sizeof (guint32) * 2 + size;
            }
          *movi = offset;
          fcc = GUINT32_TO_LE (CK_DEF_LIST);
          type = GUINT32_TO_LE (CK_LIST_AVIMOVIE);
          if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
                || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
                || fileio_write (fio, &type, sizeof (guint32))
                                                        != sizeof (guint32))
            return FALSE;
          offset += sizeof (guint32) * 3;
        }
    }
  /* ja:データを書き加える */
  if (samples > 0)
    {
      const static guint8 dmy = 0;

      /* en:AviBaseIndexEntry */
      stream[current].base = g_realloc (stream[current].base,
                                    (stream[current].entries + 1) * ABE_SIZE);
      p = (guint8 *)stream[current].base
                                        + stream[current].entries++ * ABE_SIZE;
      abe_set_offset (p, data && length > 0
                                ? offset + sizeof (guint32) * 2 - *movi : 0);
      abe_set_size (p, length
                        | (key_frame == AVI_SAVE_WRITE_VIDEO ? 1 << 31 : 0));
      if (*riff <= 0)
        { /* ja:最初のRIFF */
          /* ja:インデックス */
          *index = g_realloc (*index, (*entries + 1) * AIE_SIZE);
          p = (guint8 *)*index + (*entries)++ * AIE_SIZE;
          aie_set_type (p, stream[current].fourcc);
          aie_set_flags (p,
                key_frame == AVI_SAVE_WRITE_KEYFRAME ? AVI_IF_KEYFRAME : 0);
          aie_set_offset (p, data && length > 0
                                ? offset - *movi - sizeof (guint32) * 2 : 0);
          aie_set_length (p, length);
        }
      /* ja:データ */
      if (data && length > 0)
        {
          fcc = GUINT32_TO_LE (stream[current].fourcc);
          cb = GUINT32_TO_LE ((length + 1) & ~1);
          if (fileio_write (fio, &fcc, sizeof (guint32)) != sizeof (guint32)
            || fileio_write (fio, &cb, sizeof (guint32)) != sizeof (guint32)
            || fileio_write (fio, data, length) != length
            || (length % 2 != 0
            && fileio_write (fio, &dmy, sizeof (guint8)) != sizeof (guint8)))
            return FALSE;
        }
      stream[current].samples += samples;
      if (stream[current].max < length)
        stream[current].max = length;
    }
  stream[current].written += samples;
  return TRUE;
}


/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
      avi_opt,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
static gboolean
avi_save_avi (AviEdit     **avi_edit,
              const gchar  *file,
              AviOptions   *avi_opt,
              gboolean    (*func)(gint, gpointer),
              gpointer      user_data)
{
  FileIO *fio;
  gint current;         /* ja:最も1秒間のフレーム数が少ないストリーム */
  gint longer = 0;      /* ja:最も時間が長いストリーム */
  gint times;           /* ja:最も1秒間のフレーム数が少ないストリームの時間 */
  gint timer = 0;       /* ja:current終了後の書き込みの基準となる時間 */
  gint streams;         /* ja:ストリームの数 */
  gint head;            /* ja:アライメントされたヘッタのサイズ */
  gint header;          /* ja:ヘッタのサイズ */
  gint i;
  guint8 *p, *data = NULL;
  guint32 size;
  AviIndexEntry *index = NULL;
  gint entries = 0;     /* ja:AviIndexEntry構造体の数 */
  goffset riff = 0;     /* ja:RIFFのオフセット */
  goffset movi;         /* ja:moviのオフセット */
  AviSaveAviStream *stream;
  static guint32 fcc0 = GUINT32_TO_LE (CK_DEF_RIFF);
  static guint32 type0 = GUINT32_TO_LE (CK_FORM_AVI);
  static guint32 fcc1 = GUINT32_TO_LE (CK_DEF_LIST);
  static guint32 type1 = GUINT32_TO_LE (CK_LIST_AVIMOVIE);

  if (!avi_edit || !file)
    return FALSE;
  for (streams = 0; avi_edit[streams]; streams++);
  if (streams <= 0)
    return FALSE;
  if (streams == 1 && avi_edit_type (avi_edit[0]) == AVI_TYPE_AUDIO)
    return avi_save_wave (avi_edit[0], file,
                    avi_opt[0].wfx, avi_opt[0].recompress, func, user_data);
  stream = g_malloc0 (streams * sizeof (AviSaveAviStream));

  /* ja:ヘッタサイズを計算する */
  header = sizeof (guint32)                                     /* en:'RIFF' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + sizeof (guint32)                                  /* en:'AVI ' */
                + sizeof (guint32)                              /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32)                              /* en:'hdrl' */
                    + sizeof (guint32)                          /* en:'avih' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((AMH_SIZE + 1) & ~1)                     /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'LIST' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + sizeof (guint32)                          /* en:'odml' */
                        + sizeof (guint32)                      /* en:'odmh' */
                        + sizeof (guint32)                      /* ja:サイズ */
                        + DML_SIZE                              /* ja:構造体 */
                + sizeof (guint32)                              /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32);                             /* en:'movi' */
  for (i = 0; i < streams; i++)
    {
      switch (avi_edit_type (avi_edit[i]))
        {
          case AVI_TYPE_VIDEO:/* ja:ビデオ */
            if (avi_opt && avi_opt[i].handler != 0
                    && (stream[i].icm_object = icm_open (avi_opt[i].handler,
                                                            ICM_MODE_COMPRESS))
                    && (avi_opt[i].param_size <= 0
                            || icm_set_state (stream[i].icm_object,
                                    avi_opt[i].param, avi_opt[i].param_size))
                    && icm_compress_frames_info (stream[i].icm_object,
                                                 avi_edit[i]->bmih,
                                                 avi_opt[i].bmih,
                                                 avi_opt[i].bmih_size,
                                                 avi_opt[i].key_frame,
                                                 avi_opt[i].quality,
                                                 avi_opt[i].data_rate,
                                             avi_edit_get_rate (avi_edit[i]),
                                             avi_edit_get_scale (avi_edit[i]))
                    && (stream[i].fmt_size = icm_compress_get_format_size
                                            (stream[i].icm_object,
                                            avi_edit[i]->bmih)) >= BMIH_SIZE)
              {
                /* ja:圧縮 */
                stream[i].fmt = g_malloc (stream[i].fmt_size);
                if (!icm_compress_get_format (stream[i].icm_object,
                                            avi_edit[i]->bmih, stream[i].fmt))
                  {
                    g_free (stream[i].fmt);
                    stream[i].fmt = NULL;
                  }
                stream[i].fourcc = make4cc (i / 10 + '0',
                                                    i % 10 + '0', 'd', 'c');
              }
            if (!stream[i].fmt)
              {
                /* ja:未圧縮 */
                if (stream[i].icm_object)
                  {
                    icm_close (stream[i].icm_object);
                    stream[i].icm_object = NULL;
                  }
                stream[i].fmt_size = bx_header_bytes
                                (bmih_get_width (avi_edit[i]->bmih),
                                bmih_get_height (avi_edit[i]->bmih),
                                bmih_get_bit_count (avi_edit[i]->bmih),
                                bmih_get_compression (avi_edit[i]->bmih), 0);
                stream[i].fmt = g_malloc (stream[i].fmt_size);
                g_memmove (stream[i].fmt, avi_edit[i]->bmih,
                                                        stream[i].fmt_size);
                bmih_set_color_used (stream[i].fmt, 0);
                bmih_set_color_important (stream[i].fmt, 0);
                switch (bmih_get_bit_count (stream[i].fmt))
                  {
                    case 1: g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb2, RGBQUAD_SIZE * 2); break;
                    case 4: g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb16, RGBQUAD_SIZE * 16); break;
                    case 8: g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb256, RGBQUAD_SIZE * 256);
                  }
                stream[i].fourcc = make4cc (i / 10 + '0',
                                                    i % 10 + '0', 'd', 'b');
              }
            break;
          case AVI_TYPE_AUDIO:/* ja:オーディオ */
            if (avi_opt && avi_opt[i].wfx)
              { /* ja:圧縮 */
                stream[i].fmt_size = wf_header_bytes (avi_opt[i].wfx);
                stream[i].fmt = g_memdup (avi_opt[i].wfx, stream[i].fmt_size);
              }
            else
              { /* ja:未圧縮 */
                stream[i].fmt_size = WFX_SIZE - sizeof (guint16);
                stream[i].fmt = g_memdup (avi_edit[i]->wfx,
                                                        stream[i].fmt_size);
              }
            stream[i].fourcc = make4cc (i / 10 + '0', i % 10 + '0', 'w', 'b');
        }
      header += sizeof (guint32)                                /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32)                              /* en:'strl' */
                    + sizeof (guint32)                          /* en:'strh' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((ASH_SIZE + 1) & ~1)                     /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'strf' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((stream[i].fmt_size + 1) & ~1)           /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'indx' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + AVI_ALIGNSIZE - sizeof (guint32) * 2;     /* ja:配列 */
      if (avi_edit_get_name (avi_edit[i]))
        header +=     sizeof (guint32)                          /* en:'strn' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((g_strlen (avi_edit_get_name (avi_edit[i])) + 2) & ~1);
                                                                /* ja:文字列 */
    }
  head = header % AVI_HEADERSIZE != 0
                        ? ((header + sizeof(guint32) * 2) / AVI_HEADERSIZE + 1)
                                                    * AVI_HEADERSIZE : header;
  movi = head - sizeof (guint32) * 3;

  /* ja:ファイルを開く */
  fio = fileio_open (file, FILEIO_ACCESS_ALL,
                                    FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW);
  if (!fio
      || fileio_write (fio, &fcc0, sizeof (guint32)) != sizeof (guint32)
      || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
      || fileio_write (fio, &type0, sizeof (guint32)) != sizeof (guint32)
      || fileio_seek (fio, head - sizeof (guint32) * 3, FILEIO_SEEK_SET) == -1
      || fileio_write (fio, &fcc1, sizeof (guint32)) != sizeof (guint32)
      || fileio_seek (fio, sizeof (guint32), FILEIO_SEEK_CUR) == -1
      || fileio_write (fio, &type1, sizeof (guint32)) != sizeof (guint32))
    {
      fileio_close (fio);
      for (i = 0; i < streams; i++)
        {
          g_free (stream[i].fmt);
          if (stream[i].icm_object)
            icm_close (stream[i].icm_object);
        }
      g_free (stream);
      return FALSE;
    }

  /* ja:最も1秒間のフレーム数が少ないストリームを探す */
  for (current = 0; current < streams; current++)
    if (avi_edit_type (avi_edit[current]) == AVI_TYPE_VIDEO)
      break;
  if (current >= streams)
    {
      for (i = 0; i < streams; i++)
        if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                    && (glonglong)avi_edit_get_rate (avi_edit[i])
                                    * avi_edit_get_scale (avi_edit[current])
                            < (glonglong)avi_edit_get_rate (avi_edit[current])
                                            * avi_edit_get_scale (avi_edit[i]))
          current = i;
      times = avi_time_length (avi_edit[current])
                                        / avi_edit_length (avi_edit[current]);
    }
  else
    {
      /* ja:ビデオストリームがないとき */
      current = -1;
      times = AVI_PCM_BUFFER_TIMES;
    }
  /* ja:最も時間が長いストリームを探す */
  for (i = 1; i < streams; i++)
    if (avi_time_length (avi_edit[longer]) < avi_time_length (avi_edit[i]))
      longer = i;
  /* ja:フレーム */
  for (i = 0; i < streams; i++)
    if ((avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                && !(stream[i].avi_frame = avi_frame_open (avi_edit[i])))
        || (avi_edit_type (avi_edit[i]) == AVI_TYPE_AUDIO
                && !(stream[i].avi_acm = avi_acm_open (avi_edit[i],
                                    avi_opt ? avi_opt[i].wfx : NULL,
                                    avi_opt ? avi_opt[i].recompress : TRUE))))
      {
        while (i >= 0)
          {
            if (stream[i].avi_frame)
              avi_frame_close (stream[i].avi_frame);
            if (stream[i].avi_acm)
              avi_acm_close (stream[i].avi_acm);
            i--;
          }
        fileio_close (fio);
        for (i = 0; i < streams; i++)
          {
            g_free (stream[i].fmt);
            if (stream[i].icm_object)
              icm_close (stream[i].icm_object);
          }
        g_free (stream);
        return FALSE;
      }
  while (TRUE)
    {
      /* ja:中断チェック */
      if (func && !func (stream[longer].pos * 100
                            / avi_edit_length (avi_edit[longer]), user_data))
        break;
      /* ja:すべて出力したら終了 */
      for (i = 0; i < streams; i++)
        if (stream[i].pos < avi_edit_length (avi_edit[i]))
          break;
      if (i >= streams)
        break;
      for (i = 0; i < streams; i++)
        {
          gint samples;

          if (avi_edit_length (avi_edit[i]) <= stream[i].pos)
            continue;
          /* ja:出力サンプル数を計算 */
          samples = MIN (current >= 0
                ? i == current ? 1 : avi_time_sync_sample (avi_edit[current],
                        stream[current].pos + 1, avi_edit[i]) - stream[i].pos
                : avi_time_to_sample (avi_edit[i], timer) - stream[i].pos,
                                avi_edit_length (avi_edit[i]) - stream[i].pos);
          /* ja:オーディオのとき、AVI_PCM_BUFFER_LENGTH未満か
                                                    最後に達しないならばパス */
          if (samples <= 0 || (avi_edit_type (avi_edit[i]) == AVI_TYPE_AUDIO
                && samples < AVI_PCM_BUFFER_LENGTH
                && stream[i].pos + samples < avi_edit_length (avi_edit[i])))
            continue;
          if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            { /* ja:ビデオ */
              gint j;

              for (j = 0; j < samples; j++)
                {
                  gboolean nullframe;   /* ja:TRUE:ヌルフレーム,FALSE:通常 */

                  nullframe = avi_opt[i].nullframe && stream[i].pos + j > 0;
                  if (nullframe)
                    {
                      AviFile *avi_file;

                      avi_base_get_file (avi_edit[i], stream[i].pos + j);
                      avi_file = avi_edit[i]->file;
                      avi_base_get_file (avi_edit[i], stream[i].pos + j - 1);
                      if (avi_edit[i]->file->avi_memory
                                                    || avi_file->avi_memory)
                        { /* ja:どちらかがメモリのとき */
                          nullframe = avi_edit[i]->file->avi_memory
                                                    == avi_file->avi_memory;
                        }
                      else if (g_strfilecmp (avi_edit[i]->file->name,
                                                        avi_file->name) == 0)
                        { /* ja:同じファイルのとき */
                          gint k, l;

                          for (k = stream[i].pos + j - 1 - avi_edit[i]->offset
                                    + avi_edit[i]->file->start; k >= 0; k--)
                            if (avi_edit[i]->file->index[k].size > 0)
                              break;
                          avi_base_get_file (avi_edit[i], stream[i].pos + j);
                          for (l = stream[i].pos + j - avi_edit[i]->offset
                                    + avi_edit[i]->file->start; l >= 0; l--)
                            if (avi_edit[i]->file->index[l].size > 0)
                              break;
                          nullframe = k == l;
                        }
                      else
                        { /* ja:異なるファイルのとき */
                          nullframe = FALSE;
                        }
                      if (!nullframe)
                        {
                          /* ja:実画像を比較 */
                          gsize length;
                          BitmapInfoHeader *bmih0, *bmih1;

                          bmih0 = avi_frame_get_bitmap (stream[i].avi_frame,
                                        stream[i].pos + j - 1,
                                        bmih_get_width (stream[i].fmt),
                                        bmih_get_height (stream[i].fmt),
                                        bmih_get_bit_count (stream[i].fmt));
                          bmih1 = avi_frame_get_bitmap (stream[i].avi_frame,
                                        stream[i].pos + j,
                                        bmih_get_width (stream[i].fmt),
                                        bmih_get_height (stream[i].fmt),
                                        bmih_get_bit_count (stream[i].fmt));
                          length = bm_image_bytes (bmih0);
                          if (bm_image_bytes (bmih1) == length
                                        && g_memcmp (bm_image_pointer (bmih0),
                                        bm_image_pointer (bmih1), length) == 0)
                            nullframe = TRUE;
                        }
                    }
                  if (nullframe)
                    {
                      if (!avi_save_avi_write (fio, NULL, 0, 1,
                        AVI_SAVE_WRITE_VIDEO,
                        &index, &entries, &riff, &movi, stream, streams, i))
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].base);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_acm)
                                avi_acm_close (stream[i].avi_acm);
                            }
                          g_free (stream);
                          g_free (data);
                          g_free (index);
                          return FALSE;
                        }
                      continue;
                    }
                  if (!stream[i].icm_object)
                    {
                      BitmapInfoHeader *bmih;

                      /* ja:未圧縮 */
                      bmih = avi_frame_get_bitmap (stream[i].avi_frame,
                                        stream[i].pos + j,
                                        bmih_get_width (stream[i].fmt),
                                        bmih_get_height (stream[i].fmt),
                                        bmih_get_bit_count (stream[i].fmt));
                      if (!bmih || !avi_save_avi_write (fio,
                                    bm_image_pointer (bmih),
                                    bm_image_bytes (bmih), 1,
                                    AVI_SAVE_WRITE_KEYFRAME,
                                    &index, &entries, &riff, &movi,
                                                        stream, streams, i))
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].base);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_acm)
                                avi_acm_close (stream[i].avi_acm);
                            }
                          g_free (stream);
                          g_free (data);
                          g_free (index);
                          return FALSE;
                        }
                    }
                  else
                    {
                      gboolean key_frame;
                                /* ja:TRUE:キーフレーム,FALSE:非キーフレーム */
                      gboolean recompress;
                                /* ja:TRUE:再圧縮あり,FALSE:再圧縮なし */
                      gsize length;

                      /* ja:圧縮 */
                      recompress = avi_opt[i].recompress;
                      key_frame = avi_frame_is_key (avi_edit[i],
                                                            stream[i].pos + j);
                      if (key_frame)
                        stream[i].recompress = FALSE;
                      if (!recompress)
                        {
                          /* ja:AVIファイルを求める */
                          avi_base_get_file (avi_edit[i],stream[i].pos + j);
                          recompress = stream[i].recompress
                                || avi_edit[i]->file->handler
                                                        != avi_opt[i].handler
                                || bmih_get_size (avi_edit[i]->file->bmih)
                                        != bmih_get_size (stream[i].fmt)
                                || bmih_get_width (avi_edit[i]->file->bmih)
                                        != bmih_get_width (stream[i].fmt)
                                || bmih_get_height (avi_edit[i]->file->bmih)
                                        != bmih_get_height (stream[i].fmt)
                                || bmih_get_planes (avi_edit[i]->file->bmih)
                                        != bmih_get_planes (stream[i].fmt)
                                || bmih_get_bit_count (avi_edit[i]->file->bmih)
                                        != bmih_get_bit_count (stream[i].fmt)
                                || bmih_get_compression
                                                    (avi_edit[i]->file->bmih)
                                        != bmih_get_compression (stream[i].fmt)
                                || (avi_edit[i]->offset == stream[i].pos + j
                                                                && !key_frame);
                        }
                      if (recompress)
                        {
                          BitmapInfoHeader *bmih;

                          stream[i].recompress = TRUE;
                          /* ja:再圧縮 */
                          if (!stream[i].active)
                            {
                              icm_compress_begin (stream[i].icm_object);
                              stream[i].active = TRUE;
                              stream[i].bmih_size = icm_compress_get_size
                                                        (stream[i].icm_object);
                            }
                          data = g_realloc (data, stream[i].bmih_size);
                          if (!(bmih = avi_frame_get_bitmap
                                    (stream[i].avi_frame,
                                    stream[i].pos + j,
                                    bmih_get_width (avi_edit[i]->bmih),
                                    bmih_get_height (avi_edit[i]->bmih),
                                    bmih_get_bit_count (avi_edit[i]->bmih)))
                                || (length = icm_compress
                                        (stream[i].icm_object, &key_frame,
                                        bm_image_pointer (bmih), data)) <= 0)
                            {
                              fileio_close (fio);
                              for (i = 0; i < streams; i++)
                                {
                                  g_free (stream[i].fmt);
                                  g_free (stream[i].base);
                                  if (stream[i].active)
                                    icm_compress_end (stream[i].icm_object);
                                  if (stream[i].icm_object)
                                    icm_close (stream[i].icm_object);
                                  if (stream[i].avi_frame)
                                    avi_frame_close (stream[i].avi_frame);
                                  if (stream[i].avi_acm)
                                    avi_acm_close (stream[i].avi_acm);
                                }
                              g_free (stream);
                              g_free (data);
                              g_free (index);
                              return FALSE;
                            }
                        }
                      else
                        {
                          /* ja:再圧縮なし */
                          if (stream[i].active)
                            {
                              icm_compress_end (stream[i].icm_object);
                              stream[i].active = FALSE;
                            }
                          length = avi_read_video_size (avi_edit[i]->file,
                                    stream[i].pos + j - avi_edit[i]->offset);
                          data = g_realloc (data, length);
                          avi_read_video (avi_edit[i]->file,
                                stream[i].pos + j - avi_edit[i]->offset, data);
                        }
                      if (!avi_save_avi_write (fio, data, length, 1,
                                            key_frame ? AVI_SAVE_WRITE_KEYFRAME
                                                      : AVI_SAVE_WRITE_VIDEO,
                                                &index, &entries, &riff, &movi,
                                                        stream, streams, i))
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].base);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_acm)
                                avi_acm_close (stream[i].avi_acm);
                            }
                          g_free (stream);
                          g_free (data);
                          g_free (index);
                          return FALSE;
                        }
                    }
                }
            }
          else
            { /* ja:オーディオ */
              gpointer out_buf = NULL;
              gsize out_samples;

              if (!avi_acm_get_raw (stream[i].avi_acm,
                                            samples, &out_buf, &out_samples)
                    || (out_samples > 0 && !avi_save_avi_write (fio, out_buf,
                            out_samples * wfx_get_block_align (stream[i].fmt),
                                            out_samples, AVI_SAVE_WRITE_AUDIO,
                                                &index, &entries, &riff, &movi,
                                                        stream, streams, i)))
                {
                  g_free (out_buf);
                  fileio_close (fio);
                  for (i = 0; i < streams; i++)
                    {
                      g_free (stream[i].fmt);
                      g_free (stream[i].base);
                      if (stream[i].active)
                        icm_compress_end (stream[i].icm_object);
                      if (stream[i].icm_object)
                        icm_close (stream[i].icm_object);
                      if (stream[i].avi_frame)
                        avi_frame_close (stream[i].avi_frame);
                      if (stream[i].avi_acm)
                        avi_acm_close (stream[i].avi_acm);
                    }
                  g_free (stream);
                  g_free (data);
                  g_free (index);
                  return FALSE;
                }
              g_free (out_buf);
            }
          if (i != current)
            stream[i].pos += samples;
        }
      if (current >= 0)
        {
          stream[current].pos++;
          if (avi_edit_length (avi_edit[current]) <= stream[current].pos)
            {
              timer = avi_time_length (avi_edit[current]);
              current = -1;
            }
        }
      else
        {
          timer += times;
        }
    }
  g_free (data);
  /* ja:フレーム/圧縮 */
  for (i = 0; i < streams; i++)
    if ((stream[i].active && !icm_compress_end (stream[i].icm_object))
        | (stream[i].icm_object && !icm_close (stream[i].icm_object))
        | (stream[i].avi_frame && !avi_frame_close (stream[i].avi_frame))
        | (stream[i].avi_acm && !avi_acm_close (stream[i].avi_acm)))
      {
        fileio_close (fio);
        while (++i < streams)
          {
            if (stream[i].active)
              icm_compress_end (stream[i].icm_object);
            if (stream[i].icm_object)
              icm_close (stream[i].icm_object);
            if (stream[i].avi_frame)
              avi_frame_close (stream[i].avi_frame);
          }
        for (i = 0; i < streams; i++)
          {
            g_free (stream[i].fmt);
            g_free (stream[i].base);
          }
        g_free (stream);
        g_free (index);
        return FALSE;
      }
  /* ja:ファイナライズ */
  if (!avi_save_avi_write (fio, NULL, 0, 0, AVI_SAVE_WRITE_FINALIZE,
                        &index, &entries, &riff, &movi, stream, streams, -1))
    {
      fileio_close (fio);
      for (i = 0; i < streams; i++)
        {
          g_free (stream[i].fmt);
          g_free (stream[i].base);
        }
      g_free (stream);
      g_free (index);
      return FALSE;
    }
  /* ja:最も長いビデオストリームを求める */
  current = -1;
  for (i = 0;i < streams; i++)
    if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                    && (current == -1 || stream[i].pos > stream[current].pos))
      current = i;
  if (current < 0)
    {
      fileio_close (fio);
      for (i = 0; i < streams; i++)
        {
          g_free (stream[i].fmt);
          g_free (stream[i].base);
        }
      g_free (stream);
      return FALSE;
    }

  /* ja:ヘッタを設定する */
  p = data = g_malloc0 (head - sizeof (guint32) * 6);

  *(guint32 *)p = GUINT32_TO_LE (CK_DEF_LIST);
  p += sizeof (guint32);                                        /* en:'LIST' */
  size = sizeof (guint32)                                       /* en:'hdrl' */
            + sizeof (guint32)                                  /* en:'avih' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + AMH_SIZE                                          /* ja:構造体 */
            + sizeof (guint32)                                  /* en:'LIST' */
            + sizeof (guint32)                                  /* ja:サイズ */
            + sizeof (guint32)                                  /* en:'odml' */
                + sizeof (guint32)                              /* en:'odmh' */
                + sizeof (guint32)                              /* ja:サイズ */
                + DML_SIZE;                                     /* ja:構造体 */
  for (i = 0; i < streams; i++)
    {
      size += sizeof (guint32)                                  /* en:'LIST' */
                + sizeof (guint32)                              /* ja:サイズ */
                + sizeof (guint32)                              /* en:'strl' */
                    + sizeof (guint32)                          /* en:'strh' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ASH_SIZE                                  /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'strf' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((stream[i].fmt_size + 1) & ~1)           /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'indx' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + AVI_ALIGNSIZE - sizeof (guint32) * 2;     /* ja:配列 */
      if (avi_edit_get_name (avi_edit[i]))
        size +=       sizeof (guint32)                          /* en:'strn' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((g_strlen (avi_edit_get_name (avi_edit[i])) + 2) & ~1);
                                                                /* ja:文字列 */
    }
  *(guint32 *)p = GUINT32_TO_LE (size);
  p += sizeof (guint32);                                        /* ja:サイズ */
  *(guint32 *)p = GUINT32_TO_LE (CK_LIST_AVIHEADER);
  p += sizeof (guint32);                                        /* en:'hdrl' */
  *(guint32 *)p = GUINT32_TO_LE (CK_ID_AVIMAINHDR);
  p += sizeof (guint32);                                        /* en:'avih' */
  *(guint32 *)p = GUINT32_TO_LE (AMH_SIZE);
  p += sizeof (guint32);                                        /* ja:サイズ */
  amh_set_micro_sec_per_frame (p,0);
  amh_set_padding_granularity (p, AVI_HEADERSIZE);
  amh_set_flags (p, AVI_FILE_HASINDEX | AVI_FILE_MUSTUSEINDEX);
  amh_set_total_frames (p, stream[current].pos);
  for (i = 0; i < streams; i++)
    if (stream[i].pos > 0)
      amh_set_streams (p, amh_get_streams (p) + 1);
  amh_set_width (p, bmih_get_width (stream[current].fmt));
  amh_set_height (p, bmih_get_height (stream[current].fmt));
  p += AMH_SIZE;                                                /* ja:構造体 */

  for (i = 0; i < streams; i++)
    {
      if (stream[i].pos > 0)
        {
          gint j;
          guint8 *q;

          *(guint32 *)p = GUINT32_TO_LE (CK_DEF_LIST);
          p += sizeof (guint32);                                /* en:'LIST' */
          size = sizeof (guint32)                               /* en:'strl' */
                    + sizeof (guint32)                          /* en:'strh' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ASH_SIZE                                  /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'strf' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + ((stream[i].fmt_size + 1) & ~1)           /* ja:構造体 */
                    + sizeof (guint32)                          /* en:'indx' */
                    + sizeof (guint32)                          /* ja:サイズ */
                    + AVI_ALIGNSIZE - sizeof (guint32) * 2;     /* ja:配列 */
          if (avi_edit_get_name (avi_edit[i]))
            size +=   sizeof(guint32)                           /* en:'strn' */
                    + sizeof(guint32)                           /* ja:サイズ */
                    + ((g_strlen (avi_edit_get_name (avi_edit[i])) + 2) & ~1);
                                                                /* ja:文字列 */
          *(guint32 *)p = GUINT32_TO_LE (size);
          p += sizeof (guint32);                                /* ja:サイズ */
          *(guint32 *)p = GUINT32_TO_LE (CK_LIST_STREAMHEADER);
          p += sizeof (guint32);                                /* en:'strl' */

          *(guint32 *)p = GUINT32_TO_LE (CK_ID_STREAMHEADER);
          p += sizeof (guint32);                                /* en:'strh' */
          *(guint32 *)p = GUINT32_TO_LE (ASH_SIZE);
          p += sizeof (guint32);                                /* ja:サイズ */
          ash_set_type (p, avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                                        ? AVI_STREAM_VIDEO : AVI_STREAM_AUDIO);
          if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            ash_set_handler (p, !avi_opt || avi_opt[i].handler == 0
                                                    || !stream[i].icm_object
                                ? AVI_STREAM_COMP_DIB : avi_opt[i].handler);
          ash_set_priority (p, avi_edit_get_priority (avi_edit[i]));
          ash_set_language (p, avi_edit_get_language (avi_edit[i]));
          if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            {
              ash_set_scale (p, avi_edit_get_scale (avi_edit[i]));
              ash_set_rate (p, avi_edit_get_rate (avi_edit[i]));
            }
          else
            {
              ash_set_scale (p, wfx_get_block_align (stream[i].fmt));
              ash_set_rate (p, wfx_get_average_bytes_per_sec (stream[i].fmt));
            }
          ash_set_length (p, stream[i].written);
          ash_set_suggested_buffer_size (p, stream[i].max);
          ash_set_quality (p, -1);
          if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            {
              ash_set_frame_right (p, bmih_get_width (avi_edit[i]->bmih));
              ash_set_frame_bottom (p, bmih_get_height (avi_edit[i]->bmih));
            }
          p += ASH_SIZE;                                        /* ja:構造体 */

          *(guint32 *)p = GUINT32_TO_LE (CK_ID_STREAMFORMAT);
          p += sizeof (guint32);                                /* en:'strf' */
          *(guint32 *)p = GUINT32_TO_LE ((stream[i].fmt_size + 1) & ~1);
          p += sizeof (guint32);                                /* ja:サイズ */
          g_memmove (p, stream[i].fmt, stream[i].fmt_size);
          p += (stream[i].fmt_size + 1) & ~1;                   /* ja:構造体 */

          if (avi_edit_get_name (avi_edit[i]))
            {
              *(guint32 *)p = GUINT32_TO_LE (CK_ID_STREAMNAME);
              p += sizeof (guint32);                            /* en:'strn' */
              *(guint32 *)p = GUINT32_TO_LE ((g_strlen
                                (avi_edit_get_name (avi_edit[i])) + 2) & ~1);
              p += sizeof (guint32);                            /* ja:サイズ */
              g_memmove (p, avi_edit_get_name (avi_edit[i]),
                                g_strlen (avi_edit_get_name (avi_edit[i])));
              p += (g_strlen (avi_edit_get_name (avi_edit[i])) + 2) & ~1;
                                                                /* ja:文字列 */
            }

          *(guint32 *)p = GUINT32_TO_LE (CK_ID_AVISUPERINDEX);
          p += sizeof (guint32);                            /* en:'indx' */
          *(guint32 *)p = AVI_ALIGNSIZE - sizeof (guint32) * 2;
          p += sizeof (guint32);                                /* ja:サイズ */
          for (j = 0, q = (guint8 *)stream[i].super;
                                j < AVI_SUPERINDEX_ENTRY; j++, q += ASE_SIZE)
            if (ase_get_size (q) == 0)
              break;
          asi_set_longs (p, 4);
          asi_set_sub_type (p, 0);
          asi_set_type (p, AVI_IDX_OF_INDEXES);
          asi_set_entries (p, j);
          asi_set_ckid (p, stream[i].fourcc);
          p += ASI_SIZE;
          g_memmove (p, stream[i].super, AVI_SUPERINDEX_ENTRY * ASE_SIZE);
          p += AVI_ALIGNSIZE - sizeof (guint32) * 2 - ASI_SIZE; /* ja:配列 */
        }
      g_free (stream[i].fmt);
    }
  *(guint32 *)p = GUINT32_TO_LE (CK_DEF_LIST);
  p += sizeof (guint32);                                        /* en:'LIST' */
  *(guint32 *)p = GUINT32_TO_LE (sizeof (guint32)               /* en:'odml' */
                                    + sizeof (guint32)          /* en:'odmh' */
                                    + sizeof (guint32)          /* ja:サイズ */
                                    + DML_SIZE);                /* ja:構造体 */
  p += sizeof (guint32);                                        /* ja:サイズ */
  *(guint32 *)p = GUINT32_TO_LE (CK_LIST_OPENDML);
  p += sizeof (guint32);                                        /* en:'odml' */

  *(guint32 *)p = GUINT32_TO_LE (CK_ID_OPENDML);
  p += sizeof (guint32);                                        /* en:'dmlh' */
  *(guint32 *)p = GUINT32_TO_LE (DML_SIZE);
  p += sizeof (guint32);                                        /* ja:サイズ */
  dml_set_total_frames (p, stream[current].pos);
  p += DML_SIZE;                                                /* ja:構造体 */
  g_free (stream);

  if (head - header > 0)
    {
      *(guint32 *)p = GUINT32_TO_LE (CK_ID_AVIPADDING);
      p += sizeof (guint32);                                    /* en:'JUNK' */
      *(guint32 *)p = GUINT32_TO_LE (head - header - sizeof (guint32) * 2);
    }                                                           /* ja:サイズ */

  if (fileio_seek (fio, sizeof (guint32) * 3, FILEIO_SEEK_SET) == -1
                    || fileio_write (fio, data, head - sizeof (guint32) * 6)
                                                != head - sizeof (guint32) * 6)
    {
      fileio_close (fio);
      g_free (data);
    }
  g_free (data);
  return fileio_close (fio);
}


/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
      avi_opt,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
static gboolean
avi_save_gsr (AviEdit     **avi_edit,
              const gchar  *file,
              AviOptions   *avi_opt,
              gboolean    (*func)(gint, gpointer),
              gpointer      user_data)
{
  FileIO *fio;
  gint current;         /* ja:最も1秒間のフレーム数が少ないストリーム */
  gint longer = 0;      /* ja:最も時間が長いストリーム */
  gint times;           /* ja:最も1秒間のフレーム数が少ないストリームの時間 */
  gint timer = 0;       /* ja:current終了後の書き込みの基準となる時間 */
  gint streams;         /* ja:ストリームの数 */
  gint header;          /* ja:ヘッタのサイズ */
  gint i;
  guint8 *p, *data = NULL;
  goffset offset = 0;
  AviSaveAviStream *stream;

  if (!avi_edit || !file)
    return FALSE;
  for (streams = 0; avi_edit[streams]; streams++);
  if (streams <= 0)
    return FALSE;
  stream = g_malloc0 (streams * sizeof (AviSaveAviStream));

  /* ja:ヘッタサイズを計算する */
  header = GMH_SIZE;
  for (i = 0; i < streams; i++)
    {
      switch (avi_edit_type (avi_edit[i]))
        {
          case AVI_TYPE_VIDEO:/* ja:ビデオ */
            if (avi_opt && avi_opt[i].handler != 0
                    && (stream[i].icm_object = icm_open (avi_opt[i].handler,
                                                            ICM_MODE_COMPRESS))
                    && (avi_opt[i].param_size <= 0
                            || icm_set_state (stream[i].icm_object,
                                    avi_opt[i].param, avi_opt[i].param_size))
                    && icm_compress_frames_info (stream[i].icm_object,
                                                 avi_edit[i]->bmih,
                                                 avi_opt[i].bmih,
                                                 avi_opt[i].bmih_size,
                                                 avi_opt[i].key_frame,
                                                 avi_opt[i].quality,
                                                 avi_opt[i].data_rate,
                                             avi_edit_get_rate (avi_edit[i]),
                                             avi_edit_get_scale (avi_edit[i]))
                    && (stream[i].fmt_size = icm_compress_get_format_size
                                            (stream[i].icm_object,
                                            avi_edit[i]->bmih)) >= BMIH_SIZE)
              {
                /* ja:圧縮 */
                stream[i].fmt = g_malloc (stream[i].fmt_size);
                if (!icm_compress_get_format (stream[i].icm_object,
                                            avi_edit[i]->bmih, stream[i].fmt))
                  {
                    g_free (stream[i].fmt);
                    stream[i].fmt = NULL;
                  }
              }
            if (!stream[i].fmt)
              {
                /* ja:未圧縮 */
                if (stream[i].icm_object)
                  {
                    icm_close (stream[i].icm_object);
                    stream[i].icm_object = NULL;
                  }
                stream[i].fmt_size = bx_header_bytes
                                (bmih_get_width (avi_edit[i]->bmih),
                                bmih_get_height (avi_edit[i]->bmih),
                                bmih_get_bit_count (avi_edit[i]->bmih),
                                bmih_get_compression (avi_edit[i]->bmih), 0);
                stream[i].fmt = g_malloc (stream[i].fmt_size);
                g_memmove (stream[i].fmt, avi_edit[i]->bmih,
                                                        stream[i].fmt_size);
                bmih_set_color_used (stream[i].fmt, 0);
                bmih_set_color_important (stream[i].fmt, 0);
                switch (bmih_get_bit_count (stream[i].fmt))
                  {
                    case 1: g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb2, RGBQUAD_SIZE * 2); break;
                    case 4: g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb16, RGBQUAD_SIZE * 16); break;
                    case 8: g_memmove ((guint8 *)stream[i].fmt + BMIH_SIZE,
                                            rgb256, RGBQUAD_SIZE * 256);
                  }
              }
            break;
          case AVI_TYPE_AUDIO:/* ja:オーディオ */
            if (avi_opt && avi_opt[i].wfx)
              { /* ja:圧縮 */
                stream[i].fmt_size = wf_header_bytes (avi_opt[i].wfx);
                stream[i].fmt = g_memdup (avi_opt[i].wfx, stream[i].fmt_size);
              }
            else
              { /* ja:未圧縮 */
                stream[i].fmt_size = WFX_SIZE - sizeof (guint16);
                stream[i].fmt = g_memdup (avi_edit[i]->wfx,
                                                        stream[i].fmt_size);
              }
        }
      header += GSH_SIZE + stream[i].fmt_size;
    }

  /* ja:ファイルを開く */
  offset = header;
  fio = fileio_open (file, FILEIO_ACCESS_ALL,
                                    FILEIO_SHARE_READ, FILEIO_MODE_CREATE_NEW);
  if (!fio || fileio_seek (fio, offset, FILEIO_SEEK_SET) == -1)
    {
      fileio_close (fio);
      for (i = 0; i < streams; i++)
        {
          g_free (stream[i].fmt);
          if (stream[i].icm_object)
            icm_close (stream[i].icm_object);
        }
      g_free (stream);
      return FALSE;
    }

  /* ja:最も1秒間のフレーム数が少ないストリームを探す */
  for (current = 0; current < streams; current++)
    if (avi_edit_type (avi_edit[current]) == AVI_TYPE_VIDEO)
      break;
  if (current >= streams)
    {
      for (i = 0; i < streams; i++)
        if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                    && (glonglong)avi_edit_get_rate (avi_edit[i])
                                    * avi_edit_get_scale (avi_edit[current])
                            < (glonglong)avi_edit_get_rate (avi_edit[current])
                                            * avi_edit_get_scale (avi_edit[i]))
          current = i;
      times = avi_time_length (avi_edit[current])
                                        / avi_edit_length (avi_edit[current]);
    }
  else
    {
      /* ja:ビデオストリームがないとき */
      current = -1;
      times = AVI_PCM_BUFFER_TIMES;
    }
  /* ja:最も時間が長いストリームを探す */
  for (i = 1; i < streams; i++)
    if (avi_time_length (avi_edit[longer]) < avi_time_length (avi_edit[i]))
      longer = i;
  /* ja:フレーム */
  for (i = 0; i < streams; i++)
    if ((avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO
                && !(stream[i].avi_frame = avi_frame_open (avi_edit[i])))
        || (avi_edit_type (avi_edit[i]) == AVI_TYPE_AUDIO
                && !(stream[i].avi_acm = avi_acm_open (avi_edit[i],
                                    avi_opt ? avi_opt[i].wfx : NULL,
                                    avi_opt ? avi_opt[i].recompress : TRUE))))
      {
        while (i >= 0)
          {
            if (stream[i].avi_frame)
              avi_frame_close (stream[i].avi_frame);
            if (stream[i].avi_acm)
              avi_acm_close (stream[i].avi_acm);
            i--;
          }
        fileio_close (fio);
        for (i = 0; i < streams; i++)
          {
            g_free (stream[i].fmt);
            if (stream[i].icm_object)
              icm_close (stream[i].icm_object);
          }
        g_free (stream);
        return FALSE;
      }
  while (TRUE)
    {
      /* ja:中断チェック */
      if (func && !func (stream[longer].pos * 100
                            / avi_edit_length (avi_edit[longer]), user_data))
        break;
      /* ja:すべて出力したら終了 */
      for (i = 0; i < streams; i++)
        if (stream[i].pos < avi_edit_length (avi_edit[i]))
          break;
      if (i >= streams)
        break;
      for (i = 0; i < streams; i++)
        {
          gint samples;
          guint32 flags = 0;
          gssize length = 0;

          if (avi_edit_length (avi_edit[i]) <= stream[i].pos)
            continue;
          /* ja:出力サンプル数を計算 */
          samples = MIN (current >= 0
                ? i == current ? 1 : avi_time_sync_sample (avi_edit[current],
                        stream[current].pos + 1, avi_edit[i]) - stream[i].pos
                : avi_time_to_sample (avi_edit[i], timer) - stream[i].pos,
                                avi_edit_length (avi_edit[i]) - stream[i].pos);
          /* ja:オーディオのとき、AVI_PCM_BUFFER_LENGTH未満か
                                                    最後に達しないならばパス */
          if (samples <= 0 || (avi_edit_type (avi_edit[i]) == AVI_TYPE_AUDIO
                && samples < AVI_PCM_BUFFER_LENGTH
                && stream[i].pos + samples < avi_edit_length (avi_edit[i])))
            continue;
          if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
            { /* ja:ビデオ */
              gint j;

              for (j = 0; j < samples; j++)
                {
                  gboolean nullframe;   /* ja:TRUE:ヌルフレーム,FALSE:通常 */

                  nullframe = avi_opt[i].nullframe && stream[i].pos + j > 0;
                  if (nullframe)
                    {
                      AviFile *avi_file;

                      avi_base_get_file (avi_edit[i], stream[i].pos + j);
                      avi_file = avi_edit[i]->file;
                      avi_base_get_file (avi_edit[i], stream[i].pos + j - 1);
                      if (avi_edit[i]->file->avi_memory
                                                    || avi_file->avi_memory)
                        { /* ja:どちらかがメモリのとき */
                          nullframe = avi_edit[i]->file->avi_memory
                                                    == avi_file->avi_memory;
                        }
                      else if (g_strfilecmp (avi_edit[i]->file->name,
                                                        avi_file->name) == 0)
                        { /* ja:同じファイルのとき */
                          gint k, l;

                          for (k = stream[i].pos + j - 1 - avi_edit[i]->offset
                                    + avi_edit[i]->file->start; k >= 0; k--)
                            if (avi_edit[i]->file->index[k].size > 0)
                              break;
                          avi_base_get_file (avi_edit[i], stream[i].pos + j);
                          for (l = stream[i].pos + j - avi_edit[i]->offset
                                    + avi_edit[i]->file->start; l >= 0; l--)
                            if (avi_edit[i]->file->index[l].size > 0)
                              break;
                          nullframe = k == l;
                        }
                      else
                        { /* ja:異なるファイルのとき */
                          nullframe = FALSE;
                        }
                    }
                  if (nullframe)
                    {
                      length = 0;
                    }
                  else if (!stream[i].icm_object)
                    {
                      BitmapInfoHeader *bmih;

                      /* ja:未圧縮 */
                      bmih = avi_frame_get_bitmap (stream[i].avi_frame,
                                        stream[i].pos + j,
                                        bmih_get_width (stream[i].fmt),
                                        bmih_get_height (stream[i].fmt),
                                        bmih_get_bit_count (stream[i].fmt));
                      if (!bmih)
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].index);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_acm)
                                avi_acm_close (stream[i].avi_acm);
                            }
                          g_free (stream);
                          g_free (data);
                          return FALSE;
                        }
                      length = bm_image_bytes (bmih);
                      if (fileio_write (fio, bm_image_pointer (bmih), length)
                                                                    != length)
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].index);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_acm)
                                avi_acm_close (stream[i].avi_acm);
                            }
                          g_free (stream);
                          g_free (data);
                          return FALSE;
                        }
                      flags = GSR_IF_KEYFRAME;
                    }
                  else
                    {
                      gboolean key_frame;
                                /* ja:TRUE:キーフレーム,FALSE:非キーフレーム */
                      gboolean recompress;
                                /* ja:TRUE:再圧縮あり,FALSE:再圧縮なし */

                      /* ja:圧縮 */
                      recompress = avi_opt[i].recompress;
                      key_frame = avi_frame_is_key (avi_edit[i],
                                                            stream[i].pos + j);
                      if (key_frame)
                        stream[i].recompress = FALSE;
                      if (!recompress)
                        {
                          /* ja:AVIファイルを求める */
                          avi_base_get_file (avi_edit[i],stream[i].pos + j);
                          recompress = stream[i].recompress
                                || avi_edit[i]->file->handler
                                                        != avi_opt[i].handler
                                || bmih_get_size (avi_edit[i]->file->bmih)
                                        != bmih_get_size (stream[i].fmt)
                                || bmih_get_width (avi_edit[i]->file->bmih)
                                        != bmih_get_width (stream[i].fmt)
                                || bmih_get_height (avi_edit[i]->file->bmih)
                                        != bmih_get_height (stream[i].fmt)
                                || bmih_get_planes (avi_edit[i]->file->bmih)
                                        != bmih_get_planes (stream[i].fmt)
                                || bmih_get_bit_count (avi_edit[i]->file->bmih)
                                        != bmih_get_bit_count (stream[i].fmt)
                                || bmih_get_compression
                                                    (avi_edit[i]->file->bmih)
                                        != bmih_get_compression (stream[i].fmt)
                                || (avi_edit[i]->offset == stream[i].pos + j
                                                                && !key_frame);
                        }
                      if (recompress)
                        {
                          BitmapInfoHeader *bmih;

                          stream[i].recompress = TRUE;
                          /* ja:再圧縮 */
                          if (!stream[i].active)
                            {
                              icm_compress_begin (stream[i].icm_object);
                              stream[i].active = TRUE;
                              stream[i].bmih_size = icm_compress_get_size
                                                        (stream[i].icm_object);
                            }
                          data = g_realloc (data, stream[i].bmih_size);
                          if (!(bmih = avi_frame_get_bitmap
                                    (stream[i].avi_frame,
                                    stream[i].pos + j,
                                    bmih_get_width (avi_edit[i]->bmih),
                                    bmih_get_height (avi_edit[i]->bmih),
                                    bmih_get_bit_count (avi_edit[i]->bmih)))
                                || (length = icm_compress
                                        (stream[i].icm_object, &key_frame,
                                        bm_image_pointer (bmih), data)) <= 0)
                            {
                              fileio_close (fio);
                              for (i = 0; i < streams; i++)
                                {
                                  g_free (stream[i].fmt);
                                  g_free (stream[i].index);
                                  if (stream[i].active)
                                    icm_compress_end (stream[i].icm_object);
                                  if (stream[i].icm_object)
                                    icm_close (stream[i].icm_object);
                                  if (stream[i].avi_frame)
                                    avi_frame_close (stream[i].avi_frame);
                                  if (stream[i].avi_acm)
                                    avi_acm_close (stream[i].avi_acm);
                                }
                              g_free (stream);
                              g_free (data);
                              return FALSE;
                            }
                        }
                      else
                        {
                          /* ja:再圧縮なし */
                          if (stream[i].active)
                            {
                              icm_compress_end (stream[i].icm_object);
                              stream[i].active = FALSE;
                            }
                          length = avi_read_video_size (avi_edit[i]->file,
                                    stream[i].pos + j - avi_edit[i]->offset);
                          data = g_realloc (data, length);
                          avi_read_video (avi_edit[i]->file,
                                stream[i].pos + j - avi_edit[i]->offset, data);
                        }
                      if (fileio_write (fio, data, length) != length)
                        {
                          fileio_close (fio);
                          for (i = 0; i < streams; i++)
                            {
                              g_free (stream[i].fmt);
                              g_free (stream[i].index);
                              if (stream[i].active)
                                icm_compress_end (stream[i].icm_object);
                              if (stream[i].icm_object)
                                icm_close (stream[i].icm_object);
                              if (stream[i].avi_frame)
                                avi_frame_close (stream[i].avi_frame);
                              if (stream[i].avi_acm)
                                avi_acm_close (stream[i].avi_acm);
                            }
                          g_free (stream);
                          g_free (data);
                          return FALSE;
                        }
                      if (key_frame)
                        flags = GSR_IF_KEYFRAME;
                    }
                }
            }
          else
            { /* ja:オーディオ */
              gpointer out_buf = NULL;
              gsize out_samples;

              if (!avi_acm_get_raw (stream[i].avi_acm,
                                            samples, &out_buf, &out_samples))
                {
                  fileio_close (fio);
                  for (i = 0; i < streams; i++)
                    {
                      g_free (stream[i].fmt);
                      g_free (stream[i].index);
                      if (stream[i].active)
                        icm_compress_end (stream[i].icm_object);
                      if (stream[i].icm_object)
                        icm_close (stream[i].icm_object);
                      if (stream[i].avi_frame)
                        avi_frame_close (stream[i].avi_frame);
                      if (stream[i].avi_acm)
                        avi_acm_close (stream[i].avi_acm);
                    }
                  g_free (stream);
                  g_free (data);
                  return FALSE;
                }
              offset = fileio_seek (fio, 0, FILEIO_SEEK_CUR);
              length = out_samples * wfx_get_block_align (stream[i].fmt);
              if (offset == -1 || (out_samples > 0
                            && fileio_write (fio, out_buf, length) != length))
                {
                  g_free (out_buf);
                  fileio_close (fio);
                  for (i = 0; i < streams; i++)
                    {
                      g_free (stream[i].fmt);
                      g_free (stream[i].index);
                      if (stream[i].active)
                        icm_compress_end (stream[i].icm_object);
                      if (stream[i].icm_object)
                        icm_close (stream[i].icm_object);
                      if (stream[i].avi_frame)
                        avi_frame_close (stream[i].avi_frame);
                      if (stream[i].avi_acm)
                        avi_acm_close (stream[i].avi_acm);
                    }
                  g_free (stream);
                  g_free (data);
                  return FALSE;
                }
              g_free (out_buf);
            }
          stream[i].index = g_realloc (stream[i].index,
                                        GIE_SIZE * (stream[i].entries + 1));
          gie_set_offset (stream[i].index + stream[i].entries, offset);
          gie_set_length (stream[i].index + stream[i].entries, length);
          gie_set_flags (stream[i].index + stream[i].entries, flags);
          stream[i].entries++;
          offset += length;
          if (i != current)
            stream[i].pos += samples;
        }
      if (current >= 0)
        {
          stream[current].pos++;
          if (avi_edit_length (avi_edit[current]) <= stream[current].pos)
            {
              timer = avi_time_length (avi_edit[current]);
              current = -1;
            }
        }
      else
        {
          timer += times;
        }
    }
  g_free (data);
  /* ja:フレーム/圧縮 */
  for (i = 0; i < streams; i++)
    if ((stream[i].active && !icm_compress_end (stream[i].icm_object))
        | (stream[i].icm_object && !icm_close (stream[i].icm_object))
        | (stream[i].avi_frame && !avi_frame_close (stream[i].avi_frame))
        | (stream[i].avi_acm && !avi_acm_close (stream[i].avi_acm)))
      {
        fileio_close (fio);
        while (++i < streams)
          {
            if (stream[i].active)
              icm_compress_end (stream[i].icm_object);
            if (stream[i].icm_object)
              icm_close (stream[i].icm_object);
            if (stream[i].avi_frame)
              avi_frame_close (stream[i].avi_frame);
          }
        for (i = 0; i < streams; i++)
          {
            g_free (stream[i].fmt);
            g_free (stream[i].index);
          }
        g_free (stream);
        return FALSE;
      }

  /* ja:ヘッタを設定する */
  p = data = g_malloc0 (header);

  gmh_set_signature (p, GSR_MAIN_SIGNATURE);
  gmh_set_flags (p, 0);
  gmh_set_streams (p, streams);
  p += GMH_SIZE;
  for (i = 0; i < streams; i++)
    {
      gssize length;

      if (avi_edit_type (avi_edit[i]) == AVI_TYPE_VIDEO)
        {
          gsh_set_type (p, GSR_STREAM_VIDEO);
          gsh_set_micro_sec_per_frame (p,
                        (glonglong)avi_edit_get_rate (avi_edit[i]) * 1000000
                                        / avi_edit_get_scale (avi_edit[i]));
          gsh_set_handler (p, !avi_opt || avi_opt[i].handler == 0
                                                    || !stream[i].icm_object
                                ? AVI_STREAM_COMP_DIB : avi_opt[i].handler);
        } else {
          gsh_set_type (p, GSR_STREAM_AUDIO);
          gsh_set_micro_sec_per_frame (p, 0);
          gsh_set_handler (p, 0);
        }
      gsh_set_index (p, stream[i].entries);
      gsh_set_offset (p, offset);
      gsh_set_format (p, stream[i].fmt_size);
      p += GSH_SIZE;
      g_memmove (p, stream[i].fmt, stream[i].fmt_size);
      p += stream[i].fmt_size;
      length = GIE_SIZE * stream[i].entries;
      if (fileio_write (fio, stream[i].index, length) != length)
        {
          fileio_close (fio);
          g_free (data);
          do
            {
              g_free (stream[i].fmt);
              g_free (stream[i].index);
            }
          while (++i < streams);
          g_free (stream);
          return FALSE;
        }
      offset += length;
      g_free (stream[i].fmt);
      g_free (stream[i].index);
    }
  g_free (stream);

  if (fileio_seek (fio, 0, FILEIO_SEEK_SET) == -1
                                || fileio_write (fio, data, header) != header)
    {
      fileio_close (fio);
      g_free (data);
    }
  g_free (data);
  return fileio_close (fio);
}


/******************************************************************************
*                                                                             *
* ja:保存関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
     avi_save,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_with_options (AviEdit     **avi_edit,
                       const gchar  *file,
                       AviSave      *avi_save,
                       gboolean    (*func)(gint, gpointer),
                       gpointer      user_data)
{
  gboolean result = FALSE;

  switch (avi_save_get_type (avi_save))
    {
      case AVI_FILE_AVI:
        result = avi_save_avi (avi_edit, file,
                                        avi_save->avi_opt, func, user_data);
        break;
      case AVI_FILE_GSR:
        result = avi_save_gsr (avi_edit, file,
                                        avi_save->avi_opt, func, user_data);
        break;
      case AVI_FILE_BITMAP:
        result = avi_save_bitmap (avi_edit[avi_save->stream], file,
                            avi_save->start, avi_save->end, func, user_data);
        break;
      case AVI_FILE_PIXBUF:
        result = avi_save_pixbuf (avi_edit[avi_save->stream], file, avi_save,
                                                            func, user_data);
        break;
      case AVI_FILE_WAVE:
        return avi_save_wave (avi_edit[avi_save->stream], file,
                        avi_save->avi_opt->wfx, avi_save->avi_opt->recompress,
                                                            func, user_data);
        break;
      case AVI_FILE_SCENARIO:
        {
          gchar *scenario;

          scenario = avi_edit_to_scenario (avi_edit, FALSE);
          if (scenario)
            {
              result = fileio_save (file, scenario, g_strlen (scenario));
              g_free (scenario);
            }
        }
    }
  return result;
}
