/*
    avicore
    copyright (c) 1998-2008 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __AVIBASE_H__
#define __AVIBASE_H__


#include "avicore.h"
#include "icm.h"
#include "acm.h"
#include "misc/fileio.h"


G_BEGIN_DECLS


#define AVI_EDIT_CACHE 20
#define AVI_PCM_SAMPLES_PER_SEC 1000
#define AVI_PCM_BUFFER_TIMES 500
#define AVI_PCM_BUFFER_LENGTH                                               \
                            (AVI_PCM_BUFFER_TIMES*AVI_PCM_SAMPLES_PER_SEC/1000)


typedef struct _AviIndex
{
  goffset offset;
  guint32 size;
  gboolean key;
} AviIndex;
typedef struct _AviMemory AviMemory;
typedef struct _AviFile
{
  FileIO *fio;                  /* ja:ファイルポインタ */
  AviMemory *avi_memory;        /* ja:メモリ */
  gchar *name;                  /* ja:ファイル名 */
  gint start, length;           /* ja:有効な範囲 */
  gint entries;                 /* ja:インデックスの数 */
  gint param;                   /* ja:ストリーム番号 */
  AviIndex *index;              /* ja:インデックス */
  /* ja:リスト */
  struct _AviFile *prev, *next;
  /* ja:ビデオ固有 */
  guint32 handler;
  BitmapInfoHeader *bmih;       /* ja:オリジナルのフォーマット */
  /* ja:オーディオ固有 */
  gint samples;
  WaveFormatEx *wfx;            /* ja:オリジナルのフォーマット */
} AviFile;
typedef struct _AviCache
{
  gpointer data;                /* ja:キャッシュ */
  /* ja:ビデオ固有 */
  gboolean raw;                 /* ja:TRUE:32ビットRAW,FALSE:ビットマップ */
  gint pos;                     /* ja:位置 */
  gint width, height;           /* ja:サイズ(RAW固有) */
} AviCache;
struct _AviEdit
{
  gint offset;
  AviFile *file;
  gint type;                    /* ja:AVI_TYPE_VIDEO/AVI_TYPE_AUDIO */
  gint length;                  /* ja:サンプル数 */
  guint32 rate;                 /* ja:rate÷scale=1秒間のフレーム数 */
  guint32 scale;                /* ja:rate=サンプリング周波数,scale=1 */
  guint16 priority;             /* ja:優先度 */
  guint16 language;             /* ja:言語 */
  gchar *name;                  /* ja:名前 */
  AviCache buf[AVI_EDIT_CACHE]; /* ja:キャッシュ */
  /* ja:ビデオ固有 */
  BitmapInfoHeader *bmih;       /* ja:展開後のフォーマット */
  /* ja:オーディオ固有 */
  WaveFormatEx *wfx;            /* ja:展開後のフォーマット */
};
typedef struct _AviOptions
{
  gint type;                    /* ja:AVI_TYPE_VIDEO/AVI_TYPE_AUDIO */
  gboolean recompress;          /* ja:TRUE:再圧縮あり,FALSE:再圧縮なし */
  /* ja:ビデオ固有 */
  gboolean nullframe;           /* ja:TRUE:ヌルフレームあり */
  guint32 handler;              /* ja:4文字コード */
  gint key_frame;               /* ja:周期(0:すべてキーフレーム) */
  gint quality;                 /* ja:品質0～10000,-1:デフォルト */
  gsize data_rate;              /* ja:1秒間のサイズ(0:無効) */
  gpointer param;               /* ja:オプションデータ */
  gsize param_size;             /* ja:オプションデータのサイズ */
  BitmapInfoHeader *bmih;       /* ja:圧縮後のフォーマット */
  gsize bmih_size;              /* ja:圧縮後のフォーマットのサイズ */
  /* ja:オーディオ固有 */
  WaveFormatEx *wfx;            /* ja:圧縮後のフォーマット */
} AviOptions;
struct _AviSave
{
  gchar *extension;             /* ja:保存するファイルの推奨拡張子 */
  gint type;                    /* ja:保存するファイルの種類 */
  /* ja:AVI/WAVE固有 */
  AviOptions *avi_opt;          /* ja:オプション */
  /* ja:ビットマップ/GdkPixbuf/WAVE固有 */
  gint stream;                  /* ja:対象ストリーム */
  /* ja:ビットマップ/GdkPixbuf固有 */
  gint start;                   /* ja:最初のフレーム */
  gint end;                     /* ja:最後のフレーム */
  /* ja:GdkPixbuf固有 */
  gchar *name;                  /* en:bmp,ico,jpeg,png,tiff,... */
  gchar **keys, **values;       /* ja:オプション */
};


extern const guint8 rgb2[];
extern const guint8 rgb16[];
extern const guint8 rgb256[];
extern AviLanguage *avi_lang;


/******************************************************************************
*                                                                             *
* ja:AVIオーディオサンプル変換マクロ                                          *
*                                                                             *
******************************************************************************/
/*  ja:AVI(オーディオ)の仮想サンプル番号から実サンプル番号を求める
    sample,仮想サンプル番号
       wfx,WaveFormatEx構造体
       RET,実サンプル番号                                                   */
#define avi_audio_real_from_virtual(sample,wfx)                             \
            ((wfx)?(glonglong)(sample)*wfx_get_average_bytes_per_sec(wfx)   \
            /((glonglong)wfx_get_block_align(wfx)*AVI_PCM_SAMPLES_PER_SEC):0)


/*  ja:AVI(オーディオ)の実サンプル番号から仮想サンプル番号を求める
    sample,実サンプル番号
       wfx,WaveFormatEx構造体
       RET,仮想サンプル番号                                                 */
#define avi_audio_real_to_virtual(sample,wfx)                               \
            ((wfx)?((glonglong)(sample)*wfx_get_block_align(wfx)            \
            *AVI_PCM_SAMPLES_PER_SEC)/wfx_get_average_bytes_per_sec(wfx):0)


/*  ja:AVI(オーディオ)の仮想サンプル番号から実サンプル番号を求める
    sample,仮想サンプル番号
       wfx,WaveFormatEx構造体
       RET,実サンプル番号                                                   */
#define avi_audio_real_from_virtual_roundup(sample,wfx)                     \
        ((wfx)?(glonglong)(sample)*wfx_get_average_bytes_per_sec(wfx)       \
        /((glonglong)wfx_get_block_align(wfx)*AVI_PCM_SAMPLES_PER_SEC)      \
        +((glonglong)(sample)*wfx_get_average_bytes_per_sec(wfx)            \
        %((glonglong)wfx_get_block_align(wfx)*AVI_PCM_SAMPLES_PER_SEC)!=0):0)


/*  ja:AVI(オーディオ)の実サンプル番号から仮想サンプル番号を求める
    sample,実サンプル番号
       wfx,WaveFormatEx構造体
       RET,仮想サンプル番号                                                 */
#define avi_audio_real_to_virtual_roundup(sample,wfx)                       \
            ((wfx)?((glonglong)(sample)*wfx_get_block_align(wfx)            \
            *AVI_PCM_SAMPLES_PER_SEC)/wfx_get_average_bytes_per_sec(wfx)    \
            +(((glonglong)(sample)*wfx_get_block_align(wfx)                 \
            *AVI_PCM_SAMPLES_PER_SEC)%wfx_get_average_bytes_per_sec(wfx)!=0):0)


/******************************************************************************
*                                                                             *
* ja:AVI基本                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:AVIファイルを求める
    avi_edit,AVI編集ハンドル
         pos,サンプル番号                                                   */
void
avi_base_get_file (AviEdit    *avi_edit,
                   const gint  pos);


/*  ja:AVIファイルを分離する
    avi_edit,AVI編集ハンドル
         pos,サンプル番号                                                   */
void
avi_base_split_file (AviEdit    *avi_edit,
                     const gint  pos);


/******************************************************************************
*                                                                             *
* ja:AVIファイル構造体関数                                                    *
*                                                                             *
******************************************************************************/
/*  ja:AVIからAVIファイル構造体を求める
        file,ファイル名
       param,ストリーム番号
        type,AVI_TYPE_AUDIO/AVI_TYPE_VIDEO
        rate,レート
       scale,スケール
    priority,優先度
    language,言語
        name,名前
         RET,AVIファイル構造体,NULL:エラー                                  */
AviFile *
avi_file_open (const gchar  *file,
               const gint    param,
               gint         *type,
               guint32      *rate,
               guint32      *scale,
               guint16      *priority,
               guint16      *language,
               gchar       **name);


/*  ja:メモリからAVIファイル構造体を求める
    data,データ
    type,AVI_TYPE_AUDIO/AVI_TYPE_VIDEO
     RET,AVIファイル構造体,NULL:エラー                                      */
AviFile *
avi_file_open_memory (gconstpointer  data,
                      gint          *type);


/*  ja:GdkPixbufからAVIファイル構造体を求める
    pixbuf,GdkPixbuf
       RET,AVIファイル構造体,NULL:エラー                                    */
AviFile *
avi_file_open_pixbuf (GdkPixbuf *pixbif);


/*  ja:AVIファイル構造体を解放する
    avi_file,AVIファイル構造体
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_file_close (AviFile *avi_file);


/******************************************************************************
*                                                                             *
* ja:AVIメモリ関数                                                            *
*                                                                             *
******************************************************************************/
/*  ja:AVIメモリを登録する
      data,登録するメモリ
    length,登録するバイト数
       RET,AVIメモリ,NULL:エラー                                            */
AviMemory *
avi_memory_register (gpointer    data,
                     const gsize length);


/*  ja:AVIメモリを複製する
      data,複製するメモリ
    length,複製するバイト数
       RET,AVIメモリ,NULL:エラー                                            */
AviMemory *
avi_memory_alloc (gconstpointer data,
                  const gsize   length);


/*  ja:AVIメモリのデータを取得する
    avi_memory,AVIメモリ
           RET,データ,NULL:エラー                                           */
gconstpointer
avi_memory_get_data (AviMemory *avi_memory);


/*  ja:AVIメモリのバイト数を取得する
    avi_memory,AVIメモリ
           RET,バイト数                                                     */
gsize
avi_memory_get_length (AviMemory *avi_memory);


/*  ja:AVIメモリの参照数を増やす
    avi_memory,AVIメモリ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
avi_memory_ref (AviMemory *avi_memory);


/*  ja:AVIメモリの参照数を減らす
    avi_memory,AVIメモリ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
avi_memory_unref (AviMemory *avi_memory);


/*  ja:AVIメモリを解放する
    avi_memory,AVIメモリ
           RET,TRUE:正常終了,FALSE:エラー                                   */
#define avi_memory_free(avi_memory) avi_memory_unref(avi_memory)


/******************************************************************************
*                                                                             *
* ja:AVIファイル入力関数                                                      *
*                                                                             *
******************************************************************************/
/*  ja:AVI(ビデオ)のデータのバイト数を取得する
    avi_file,AVIファイル構造体
         pos,読み込むサンプル番号
         RET,バイト数                                                       */
gsize
avi_read_video_size (AviFile    *avi_file,
                     const gint  pos);


/*  ja:AVI(ビデオ)のデータを読み込む
    avi_file,AVIファイル構造体
         pos,読み込むサンプル番号
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_video (AviFile    *avi_file,
                const gint  pos,
                gpointer    buffer);


/*  ja:AVI(オーディオ)のデータのバイト数を取得する
    avi_file,AVIファイル構造体
       start,取得する実サンプル番号
     samples,実サンプル数
         RET,バイト数                                                       */
gsize
avi_read_audio_size (AviFile    *avi_file,
                     const gint  start,
                     const gint  samples);


/*  ja:AVI(オーディオ)のデータを読み込む
    avi_file,AVIファイル構造体
       start,取得する実サンプル番号
     samples,実サンプル数
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_audio (AviFile    *avi_file,
                const gint  start,
                const gint  samples,
                gpointer    buffer);


G_END_DECLS


#endif /* __AVIBASE_H__ */
