/*
    avicore
    copyright (c) 1998-2008 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __ICM_H__
#define __ICM_H__


#include "misc/bitmap.h"


G_BEGIN_DECLS


#define ICM_ATTR_NATIVE             0
#define ICM_ATTR_WIN32              1

#define ICM_FLAG_QUALITY            0x0001
#define ICM_FLAG_CRUNCH             0x0002
#define ICM_FLAG_TEMPORAL           0x0004
#define ICM_FLAG_COMPRESSFRAMES     0x0008
#define ICM_FLAG_DRAW               0x0010
#define ICM_FLAG_FASTTEMPORALC      0x0020
#define ICM_FLAG_FASTTEMPORALD      0x0080
#define ICM_FLAG_QUALITYTIME        0x0040

#define ICM_MODE_COMPRESS           1
#define ICM_MODE_DECOMPRESS         2
#define ICM_MODE_FASTDECOMPRESS     3
#define ICM_MODE_QUERY              4
#define ICM_MODE_FASTCOMPRESS       5
#define ICM_MODE_DRAW               8

#define ICM_DECOMPRESS_HURRYUP      0x80000000
#define ICM_DECOMPRESS_UPDATE       0x40000000
#define ICM_DECOMPRESS_PREROLL      0x20000000
#define ICM_DECOMPRESS_NULLFRAME    0x10000000
#define ICM_DECOMPRESS_NOTKEYFRAME  0x08000000


typedef struct _IcmObject IcmObject;


/******************************************************************************
*                                                                             *
* ja:ICM初期化                                                                *
*                                                                             *
******************************************************************************/
/*  ja:ICMを初期化する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_init (void);


/*  ja:ICMを終了する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_exit (void);


/******************************************************************************
*                                                                             *
* ja:ICM4文字コード処理                                                       *
*                                                                             *
******************************************************************************/
/*  ja:4文字コード→文字列
    handler,4文字コード
        RET,文字列                                                          */
gchar *
icm_handler_to_string (const guint32 handler);


/*  ja:4文字コード←文字列
    text,文字列
     RET,4文字コード                                                        */
guint32
icm_handler_from_string (const gchar *text);


/******************************************************************************
*                                                                             *
* ja:ICM基本                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:登録されている4文字コードを取得する
    RET,4文字コードのリスト,NULL:登録なし                                   */
GList *
icm_get_handler_list (void);


/*  ja:Codecを開く
    handler,4文字コード
       mode,モード
        RET,IcmObject,NULL:エラー                                           */
IcmObject *
icm_open (const guint32 handler,
          const gint    mode);


/*  ja:Codecを閉じる
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
#define icm_close(icm_object) icm_unref(icm_object)


/*  ja:Codecの参照数を増やす
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_ref (IcmObject *icm_object);


/*  ja:Codecの参照数を減らす
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_unref (IcmObject *icm_object);


/*  ja:Codecを複製する
    icm_object,IcmObject
           RET,IcmObject,NULL:エラー                                        */
IcmObject *
icm_dup (IcmObject *icm_object);


/******************************************************************************
*                                                                             *
* ja:ICM情報                                                                  *
*                                                                             *
******************************************************************************/
/*  ja:Codecの4文字コードを取得する
    icm_object,IcmObject
           RET,4文字コード,0:エラー                                         */
guint32
icm_get_handler (IcmObject *icm_object);


/*  ja:Codecの名前を取得する
    icm_object,IcmObject
           RET,名前,NULL:エラー                                             */
const gchar *
icm_get_name (IcmObject *icm_object);


/*  ja:Codecのフラグを取得する
    icm_object,IcmObject
           RET,フラグ                                                       */
gint
icm_get_flags (IcmObject *icm_object);


/*  ja:Codecの属性を取得する
    icm_object,IcmObject
           RET,ICM_ATTR_NATIVE,ICM_ATTR_WIN32                               */
gint
icm_get_attribute (IcmObject *icm_object);


/******************************************************************************
*                                                                             *
* ja:ICMダイアログ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:Aboutダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_about (IcmObject *icm_object);


/*  ja:Aboutダイアログを表示する
    icm_object,IcmObject
        widget,親ウインドウ                                                 */
void
icm_dialog_about (IcmObject *icm_object,
                  GtkWidget *widget);


/*  ja:設定ダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_configure (IcmObject *icm_object);


/*  ja:設定ダイアログを表示する
    icm_object,IcmObject
        widget,親ウインドウ                                                 */
void
icm_dialog_configure (IcmObject *icm_object,
                      GtkWidget *widget);


/*  ja:ドライバ設定ダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_preference (IcmObject *icm_object);


/*  ja:ドライバ設定ダイアログを表示する
    icm_object,IcmObject
        widget,親ウインドウ                                                 */
void
icm_dialog_preference (IcmObject *icm_object,
                       GtkWidget *widget);


/******************************************************************************
*                                                                             *
* ja:ICMパラメータ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:パラメータのサイズを取得する
    icm_object,IcmObject
           RET,サイズ                                                       */
gsize
icm_get_state_size (IcmObject *icm_object);


/*  ja:パラメータを取得する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_get_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size);


/*  ja:パラメータを設定する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_set_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size);


/******************************************************************************
*                                                                             *
* en:ICM Compress                                                             *
*                                                                             *
******************************************************************************/
/*  ja:圧縮可能か確認する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
icm_compress_query (IcmObject              *icm_object,
                    const BitmapInfoHeader *bmih);


/*  ja:圧縮のデフォルトの周期を取得する
    icm_object,IcmObject
           RET,デフォルト(0:未対応)                                         */
gint
icm_compress_get_default_key_frame (IcmObject *icm_object);


/*  ja:圧縮のデフォルトの品質を取得する
    icm_object,IcmObject
           RET,デフォルト(-1:未対応)                                        */
gint
icm_compress_get_default_quality (IcmObject *icm_object);


/*  ja:圧縮後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_format_size (IcmObject              *icm_object,
                              const BitmapInfoHeader *bmih);


/*  ja:圧縮後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_get_format (IcmObject              *icm_object,
                         const BitmapInfoHeader *bmih_in,
                         BitmapInfoHeader       *bmih_out);


/*  ja:圧縮後のデータの最大サイズを取得する
    icm_object,IcmObject
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_size (IcmObject *icm_object);


/*  ja:圧縮を設定する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
     bmih_size,圧縮されたビットマップのヘッタのサイズ
     key_frame,周期(0:すべてキーフレーム)
       quality,品質0～10000,-1:デフォルト
     data_rate,1秒間のサイズ(0:無効)
          rate,レート
         scale,スケール
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_frames_info (IcmObject              *icm_object,
                          const BitmapInfoHeader *bmih_in,
                          const BitmapInfoHeader *bmih_out,
                          const gsize             bmih_size,
                          const gint              key_frame,
                          const gint              quality,
                          const gsize             data_rate,
                          const guint32           rate,
                          const guint32           scale);


/*  ja:圧縮を開始する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_begin (IcmObject *icm_object);


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_end (IcmObject *icm_object);


/*  ja:圧縮する
    icm_object,IcmObject
     key_frame,TRUE:キーフレーム,FALSE:非キーフレーム
         input,圧縮するデータ
        output,圧縮されたデータ
           RET,フレームのサイズ,0:エラー                                    */
gsize
icm_compress (IcmObject      *icm_object,
              gboolean       *key_frame,
              const gpointer  input,
              gpointer        output);


/******************************************************************************
*                                                                             *
* en:ICM Decompress                                                           *
*                                                                             *
******************************************************************************/
/*  ja:展開可能か確認する
    icm_object,IcmObject
         bmih,圧縮されたビットマップのヘッタ
          RET,TRUE:可能,FALSE:不可能                                        */
gboolean
icm_decompress_query (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih);


/*  ja:展開後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_decompress_get_format_size (IcmObject              *icm_object,
                                const BitmapInfoHeader *bmih);


/*  ja:展開後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮されたビットマップのヘッタ
      bmih_out,展開されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_get_format (IcmObject              *icm_object,
                           const BitmapInfoHeader *bmih_in,
                           BitmapInfoHeader       *bmih_out);


/*  ja:展開を開始する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_begin (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih);


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_end (IcmObject *icm_object);


/*  ja:展開する
    icm_object,IcmObject
         flags,フラグ
         input,圧縮されたデータ
        output,展開されたデータ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress (IcmObject      *icm_object,
                const guint     flags,
                const gpointer  input,
                gpointer        output);


G_END_DECLS


#endif /* __ICM_H__ */
