/*
    peimage
    copyright (c) 1998-2010 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "peimage.h"
#include "misc/fileio.h"


/******************************************************************************
*                                                                             *
* ja:PEイメージ関数                                                           *
*                                                                             *
******************************************************************************/
/*  ja:PEイメージの仮想相対アドレスをファイルオフセットに変換する
      image,PEイメージ(ヘッダ)
    address,仮想相対アドレス
        RET,ファイルオフセット,-1:エラー                                    */
gint
peimage_file_address_to_offset (const guint8  *image,
                                const guint32  address)
{
  gint offset = -1;

  if (image)
    {
      gint i;
      guint16 sections;
      ImageSectionHeader *ish;

      sections = pe_ifh_get_number_of_sections (image);
      ish = pe_image_section_header (image);
      for (i = 0; i < sections; i++)
        {
          if (ish_get_virtual_address (ish) <= address
                                && address < ish_get_virtual_address (ish)
                                           + ish_get_virtual_size (ish))
            {
              offset = address - ish_get_virtual_address (ish)
                               + ish_get_pointer_to_raw_data (ish);
              break;
            }
          ish++;
        }
    }
  return offset;
}


/*  ja:PEイメージのファイルオフセットを仮想相対アドレスに変換する
     image,PEイメージ(ヘッダ)
    offset,ファイルオフセット
       RET,仮想相対アドレス,-1:エラー                                       */
guint32
peimage_file_offset_to_address (const guint8 *image,
                                const gint    offset)
{
  guint32 address = (guint32)-1;

  if (image)
    {
      gint i;
      guint16 sections;
      ImageSectionHeader *ish;

      sections = pe_ifh_get_number_of_sections (image);
      ish = pe_image_section_header (image);
      for (i = 0; i < sections; i++)
        {
          if (ish_get_pointer_to_raw_data (ish) <= offset
                                && offset < ish_get_pointer_to_raw_data (ish)
                                          + ish_get_size_of_raw_data (ish))
            {
              address = offset - ish_get_pointer_to_raw_data (ish)
                               + ish_get_virtual_address (ish);
              break;
            }
          ish++;
        }
    }
  return address;
}


/*  ja:PEイメージを読み込む
    file,ファイル名
     RET,PEイメージ,NULL:エラー                                             */
guint8 *
peimage_file_load (const gchar *file)
{
  guint8 *image;
  gint i, number_of_sections;
  gssize length, header_bytes, size_of_image;
  FileIO *fio;
  ImageSectionHeader *ish;

  if (!file)
    return NULL;
  fio = fileio_open (file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (!fio)
    return NULL;
  image = g_malloc (sizeof (ImageDosHeader));
  if (!image || fileio_read (fio, image, sizeof (ImageDosHeader))
                                                    != sizeof (ImageDosHeader)
                        || pe_idh_get_magic (image) != PEIMAGE_DOS_SIGNATURE)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  length = idh_get_lfanew (image) + sizeof (guint32)
                                                    + sizeof (ImageFileHeader);
  if (length > sizeof (ImageDosHeader))
    {
      image = g_realloc (image, length);
      length -= sizeof (ImageDosHeader);
      if (!image || fileio_read (fio, image + sizeof (ImageDosHeader), length)
                                                                    != length)
        {
          g_free (image);
          fileio_close (fio);
          return NULL;
        }
      length += sizeof (ImageDosHeader);
    }
  else
    {
      length = sizeof (ImageDosHeader);
    }
  if (pe_get_signature (image) != PEIMAGE_NT_SIGNATURE
                || pe_ifh_get_machine (image) != PEIMAGE_FILE_MACHINE_I386)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  header_bytes = peimage_file_header_bytes (image);
  image = g_realloc (image, header_bytes);
  header_bytes -= length;
  if (!image || fileio_read (fio, image + length, header_bytes) != header_bytes
                || pe_get_signature (image) != PEIMAGE_NT_SIGNATURE
                || pe_ifh_get_machine (image) != PEIMAGE_FILE_MACHINE_I386
                || pe_ioh_get_magic (image) != PEIMAGE_NT_OPTIONAL_HDR_MAGIC)
    {
      g_free (image);
      fileio_close (fio);
      return NULL;
    }
  header_bytes += length;
  number_of_sections = pe_ifh_get_number_of_sections (image);
  length = 0;
  ish = pe_image_section_header (image);
  for (i = 0; i < number_of_sections; i++)
    {
      gssize leng;

      leng = ish_get_virtual_address (ish) + ish_get_virtual_size (ish);
      if (length < leng)
        length = leng;
      ish++;
    }
  size_of_image = pe_ioh_get_size_of_image (image);
  if (length > size_of_image)
    {
      size_of_image = length;
      pe_ioh_set_size_of_image (image, size_of_image);
    }
  image = g_realloc (image, size_of_image);
  if (!image)
    {
      fileio_close (fio);
      return NULL;
    }
  length = pe_ioh_get_size_of_headers (image);
  g_memset (image + length, 0, size_of_image - length);
  if (length > header_bytes)
    {
      length -= header_bytes;
      if (fileio_read (fio, image + header_bytes, length) != length)
        {
          g_free (image);
          fileio_close (fio);
          return NULL;
        }
    }
  ish = pe_image_section_header (image);
  for (i = 0; i < number_of_sections; i++)
    {
      if (fileio_seek (fio, ish_get_pointer_to_raw_data (ish),
                                                        FILEIO_SEEK_SET) >= 0)
        fileio_read (fio, image + ish_get_virtual_address (ish),
            MIN (ish_get_virtual_size (ish), ish_get_size_of_raw_data (ish)));
      ish++;
    }
  if (!fileio_close (fio))
    {
      g_free (image);
      return NULL;
    }
  return image;
}


/*  ja:PEイメージを再配置する
    image,PEイメージ
    delta,差                                                                */
void
peimage_file_relocate (guint8     *image,
                       const gint  delta)
{
  if (image && delta != 0 && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC))
    {
      guint32 address, size;

      size = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC);
      while (address + sizeof (ImageBaseRelocation) <= size)
        {
          gint i, count;
          guint32 base;
          ImageBaseRelocation *ibr;

          ibr = (ImageBaseRelocation *)(image + address);
          base = ibr_get_virtual_address (ibr);
          if (base == 0 || address + ibr_get_size_of_block (ibr) > size)
            break;
          count = (ibr_get_size_of_block (ibr) - sizeof (guint32) * 2)
                                                            / sizeof (guint16);
          for (i = 0; i < count; i++)
            {
              guint16 typeoffset;

              typeoffset = ibr_get_type_offset (ibr, i);
              if (((typeoffset >> 12) & 0xf) == PEIMAGE_REL_BASED_HIGHLOW
                                        && base + (typeoffset & 0xfff) <= size)
                {
                  guint32 *p;

                  p = (guint32 *)(image + base + (typeoffset & 0xfff));
                  *p = GUINT32_TO_LE (GUINT32_FROM_LE (*p) + delta);
                }
            }
          address += ibr_get_size_of_block (ibr);
        }
    }
}


/*  ja:PEイメージからAPIを取得する
    image,PEイメージ
     name,API名
      RET,アドレス,NULL:エラー                                              */
gconstpointer
peimage_file_get_func (const guint8 *image,
                       const gchar  *name)
{
  gconstpointer func = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      guint32 address, size;
      const ImageExportDirectory *ied;

      size = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      ied = (const ImageExportDirectory *)(image + address);
      if (address + sizeof (ImageExportDirectory) <= size
            && ied_get_address_of_functions (ied)
                + ied_get_number_of_functions (ied) * sizeof (guint32) <= size
            && ied_get_address_of_names (ied)
                + ied_get_number_of_names (ied) * sizeof (guint32) <= size
            && ied_get_address_of_name_ordinals (ied)
                + ied_get_number_of_names (ied) * sizeof (guint16) <= size)
        {
          gint i;
          const guint16 *ordinals;
          const guint32 *functions, *names;

          functions = (const guint32 *)(image
                                    + ied_get_address_of_functions (ied));
          names = (const guint32 *)(image
                                    + ied_get_address_of_names (ied));
          ordinals = (const guint16 *)(image
                                    + ied_get_address_of_name_ordinals (ied));
          for (i = 0; i < ied_get_number_of_names (ied); i++)
            {
              gint j;

              address = GUINT32_FROM_LE (names[i]);
              for (j = 0; address + j < size
                            && image[address + j] != '\0' && name[j] != '\0'
                            && image[address + j] == name[j]; j++);
              if (address + j < size && image[address + j] == name[j])
                {
                  gint index;

                  index = GUINT16_FROM_LE (ordinals[i]);
                  if (index < ied_get_number_of_functions (ied))
                    {
                      address = GUINT32_FROM_LE (functions[index]);
                      if (address < size)
                        func = image + address;
                    }
                  break;
                }
            }
        }
    }
  return func;
}


/*  ja:PEイメージからAPIを取得する
      image,PEイメージ
    ordinal,オーディナル値
        RET,アドレス,NULL:エラー                                            */
gconstpointer
peimage_file_get_proc (const guint8  *image,
                       const guint16  ordinal)
{
  gconstpointer proc = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      guint32 address, size;
      const ImageExportDirectory *ied;

      size = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      ied = (const ImageExportDirectory *)(image + address);
      if (address + sizeof (ImageExportDirectory) <= size
            && ied_get_address_of_functions (ied)
                + ied_get_number_of_functions (ied) * sizeof (guint32) <= size)
        {
          gint index;
          const guint32 *functions;

          functions = (const guint32 *)(image
                                    + ied_get_address_of_functions (ied));
          index = ordinal - ied_get_base (ied);
          if (index < ied_get_number_of_functions (ied))
            {
              address = GUINT32_FROM_LE (functions[index]);
              if (address < size)
                proc = image + address;
            }
        }
    }
  return proc;
}


/*  ja:PEイメージからリソースを取得する
    image,PEイメージ
      key,リソース
      RET,データ,NULL:エラー                                                */
gconstpointer
peimage_file_get_resource (const guint8 *image,
                           const gchar  *key)
{
  gconstpointer resource = NULL;

  if (image && key && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE))
    {
      gchar **element;
      guint32 size;

      size = pe_ioh_get_size_of_image (image);
      element = g_strsplit (key[0] == '/' ? key + 1 : key, "/", G_MAXINT);
      if (element)
        {
          gint i;
          guint32 address, offset = 0;

          address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE);
          for (i = 0; element[i]; i++)
            {
              gint j, count;
              const ImageResourceDirectory *ird;
              const ImageResourceDirectoryEntry *irde;

              ird = (const ImageResourceDirectory *)(image + address + offset);
              irde = (const ImageResourceDirectoryEntry *)(ird + 1);
              if (address + offset + sizeof (ImageResourceDirectory) > size)
                break;
              count = ird_get_number_of_named_entries (ird)
                                        + ird_get_number_of_id_entries (ird);
              if (address + offset + sizeof (ImageResourceDirectory)
                        + count * sizeof (ImageResourceDirectoryEntry) > size)
                break;
              for (j = 0; j < count; j++)
                {
                  gboolean result;
                  gchar *name = NULL;

                  if (irde_get_id (irde) & 0x80000000)
                    {
                      guint32 offset;

                      offset = address + (irde_get_id (irde) & 0x7fffffff);
                      if (offset + sizeof (guint16) <= size)
                        {
                          const ImageResourceDirStringU *irdsu;

                          irdsu = (const ImageResourceDirStringU *)
                                                            (image + offset);
                          if (sizeof (guint16)
                                + irdsu_get_length (irdsu) * sizeof (gunichar2)
                                                                    <= size)
                            {
                              gchar *utf8str;

                              utf8str = g_utf16_to_utf8
                                (irdsu_get_name_string (irdsu),
                                irdsu_get_length (irdsu), NULL, NULL, NULL);
                              name = g_utf8_strup (utf8str, -1);
                              g_free (utf8str);
                            }
                        }
                    }
                  else
                    {
                      name = g_strdup_printf ("0x%x", irde_get_id (irde));
                    }
                  result = g_strcmp (element[i], name) == 0;
                  g_free (name);
                  if (result)
                    break;
                  irde++;
                }
              if (j < count && irde_get_offset_to_data (irde) & 0x80000000)
                offset = irde_get_offset_to_data (irde) & 0x7fffffff;
              else if (j < count && !element[i + 1])
                resource = image + address + irde_get_offset_to_data (irde);
              else
                break;
            }
          g_strfreev (element);
        }
    }
  return resource;
}


/*  ja:PEイメージのエクスポートテーブルを列挙する
    image,PEイメージ
      RET,リスト,NULL:エラー                                                */
GList *
peimage_file_enum_export (const guint8 *image)
{
  GList *glist = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      guint32 address, size, size_of_image;
      const ImageExportDirectory *ied;

      size_of_image = pe_ioh_get_size_of_image (image);
      address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      size = pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      ied = (const ImageExportDirectory *)(image + address);
      if (address + size <= size_of_image
            && size >= sizeof (ImageExportDirectory)
            && ied_get_address_of_functions (ied)
                + ied_get_number_of_functions (ied) * sizeof (guint32)
                                                            <= size_of_image
            && ied_get_address_of_names (ied)
                + ied_get_number_of_names (ied) * sizeof (guint32)
                                                            <= size_of_image
            && ied_get_address_of_name_ordinals (ied)
                + ied_get_number_of_names (ied) * sizeof (guint16)
                                                            <= size_of_image)
        {
          gint i;
          const guint16 *ordinals;
          const guint32 *functions, *names;

          functions = (const guint32 *)(image
                                    + ied_get_address_of_functions (ied));
          names = (const guint32 *)(image
                                    + ied_get_address_of_names (ied));
          ordinals = (const guint16 *)(image
                                    + ied_get_address_of_name_ordinals (ied));
          for (i = 0; i < ied_get_number_of_names (ied); i++)
            {
              guint32 func;

              func = GUINT32_FROM_LE (functions[i]);
              if (func < size_of_image)
                {
                  gint j;
                  PeExport *peexp;

                  peexp = g_malloc0 (sizeof (PeExport));
                  if (address <= func && func < address + size)
                    peexp->forward = (const gchar *)(image + func);
                  else
                    peexp->address = func;
                  peexp->ordinal = i + ied_get_base (ied);
                  for (j = 0; j < ied_get_number_of_names (ied); j++)
                    if (i == GUINT16_FROM_LE (ordinals[j]))
                      {
                        gint k;
                        guint32 name;

                        name = GUINT32_FROM_LE (names[j]);
                        for (k = name; 0 <= k && k < size_of_image; k++)
                          if (image[k] == '\0')
                            {
                              peexp->name = (const gchar *)(image + name);
                              break;
                            }
                      }
                  glist = g_list_append (glist, peexp);
                }
            }
        }
    }
  return glist;
}


/*  ja:PEイメージのインポートテーブルを列挙する
    image,PEイメージ
      RET,リスト,NULL:エラー                                                */
GList *
peimage_file_enum_import (const guint8 *image)
{
  GList *glist = NULL;

  if (image && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT))
    {
      guint16 sections;
      guint32 address, size;

      size = pe_ioh_get_size_of_image (image);
      sections = pe_ifh_get_number_of_sections (image);
      for (address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT);
                        address + sizeof (ImageImportDescriptor) <= size;
                                    address += sizeof (ImageImportDescriptor))
        {
          gint i;
          guint32 lookup;
          const gchar *file;
          const ImageSectionHeader *ish;
          const ImageImportDescriptor *iid;

          iid = (const ImageImportDescriptor *)(image + address);
          for (i = iid_get_name (iid); 0 < i && i < size; i++)
            if (image[i] == '\0')
              break;
          if (i <= 0 || size <= i)
            break;
          file = (const gchar *)(image + iid_get_name (iid));
          ish = pe_image_section_header (image);
          for (i = 0; i < sections; i++)
            {
              if (ish_get_virtual_address (ish)
                                        <= iid_get_original_first_thunk (iid)
                    && iid_get_original_first_thunk (iid)
                                                < ish_get_virtual_address (ish)
                                                + ish_get_virtual_size (ish))
                break;
              ish++;
            }
          lookup = i < sections ? iid_get_original_first_thunk (iid)
                                : iid_get_first_thunk (iid);
          while (lookup + sizeof (guint32) <= size)
            {
              PeImport *peimp;
              const ImageThunkData *ilt;

              ilt = (const ImageThunkData *)(image + lookup);
              if (!itd_get_address_of_data (ilt))
                break;
              peimp = g_malloc0 (sizeof (PeImport));
              peimp->file = file;
              if (itd_get_ordinal (ilt) & PEIMAGE_ORDINAL_FLAG)
                {
                  peimp->ordinal = itd_get_ordinal (ilt)
                                                    & ~PEIMAGE_ORDINAL_FLAG;
                }
              else if (itd_get_address_of_data (ilt)
                                                    + sizeof (guint16) <= size)
                {
                  for (i = itd_get_address_of_data (ilt) + sizeof (guint16);
                                                    0 <= i && i < size; i++)
                    if (image[i] == '\0')
                      {
                        const ImageImportByName *iibn;

                        iibn = (const ImageImportByName *)(image
                                            + itd_get_address_of_data (ilt));
                        peimp->name = iibn_get_name (iibn);
                        break;
                      }
                }
              else
                {
                  g_free (peimp);
                  peimp = NULL;
                }
              if (peimp)
                glist = g_list_append (glist, peimp);
              lookup += sizeof (guint32);
            }
        }
    }
  return glist;
}


/******************************************************************************
*                                                                             *
* ja:モジュール関数群                                                         *
*                                                                             *
******************************************************************************/
typedef struct _PeModule
{
  guint8 *image;
  gchar *file, *name;
  gint counter;
  GList *library;
} PeModule;


# ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
# endif /* USE_THREAD */


static gchar *
peimage_get_basename (const gchar *file)
{
  gchar *name = NULL;

  if (file)
    {
      name = g_path_get_basename (file);
      if (!g_strrchr (name, '.'))
        {
          gchar *tmp;

          tmp = g_strconcat (name, ".dll", NULL);
          g_free (name);
          name = tmp;
        }
    }
  return name;
}


/*  ja:既にロードされているPEイメージを取得する
    process,プロセス
       file,イメージ名
        RET,PEイメージ,NULL:ロードされていない                              */
guint8 *
peimage_module_get_library (PeProcess   *process,
                            const gchar *file)
{
  guint8 *image = NULL;

  if (file)
    {
      gchar *base, *name;
      GList *glist;

      base = g_path_get_basename (file);
      name = peimage_get_basename (file);
#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
        if (g_ascii_strcasecmp (((PeModule *)glist->data)->name, base) == 0
         || g_ascii_strcasecmp (((PeModule *)glist->data)->name, name) == 0)
          {
            image = ((PeModule *)glist->data)->image;
            break;
          }
#ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
      g_free (base);
      g_free (name);
    }
  return image;
}


/*  ja:モジュール構造体を取得する
    process,プロセス
      image,PEイメージ
        RET,モジュール構造体,NULL:エラー                                    */
static PeModule *
peimage_module_get_info (PeProcess    *process,
                         const guint8 *image)
{
  PeModule *module = NULL;

  if (image)
    {
      GList *glist;

#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
        if (((PeModule *)glist->data)->image == image)
          {
            module = glist->data;
            break;
          }
#ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
    }
  return module;
}


/*  ja:イメージをロードする
    process,プロセス
       file,イメージ名
        RET,イメージ,NULL:エラー                                            */
guint8 *
peimage_module_load_library (PeProcess        *process,
                             const gchar      *file,
                             PeModuleSystem    pemodule_system,
                             PeModuleRelocate  pemodule_relocate,
                             PeModuleImport    pemodule_import,
                             PeModuleEntry     pemodule_entry,
                             gpointer          user_data)
{
  gboolean result = TRUE;
  guint8 *image;
  gint i;
  PeModule *module;

  if (!process || !file)
    return NULL;
  image = peimage_module_get_library (process, file);
  if (image)
    {
      /* ja:既読 */
      module = peimage_module_get_info (process, image);
      if (module)
        {
          module->counter++;
          return module->image;
        }
    }
  module = g_malloc0 (sizeof (PeModule));
  module->counter = 1;
  module->name = peimage_get_basename (file);
  if (pemodule_system)
    module->image = pemodule_system (process, module->name, user_data);
  if (module->image)
    {
      module->file = g_build_filename
            (peimage_dir_get_path (PEIMAGE_DIR_SYSTEM), module->name, NULL);
#ifdef USE_THREAD
      G_LOCK (critical);
      critical = TRUE;
#endif /* USE_THREAD */
      process->module = g_list_append (process->module, module);
#ifdef USE_THREAD
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
      return module->image;
    }
  g_free (module->name);

  /* ja:ファイル読み込み */
  module->name = g_path_get_basename (file);
  module->file = fileio_get_full_path (file);
  module->image = peimage_file_load (module->file);
  for (i = 0; i < 8; i++)
    if (!module->image)
      {
        gchar *path;

        g_free (module->name);
        g_free (module->file);
        module->name = i % 2 == 0 ? g_path_get_basename (file)
                                  : peimage_get_basename (file);
        path = i < 4 ? g_build_filename (peimage_dir_get_path
                (i / 2 == 0 ? PEIMAGE_DIR_SYSTEM : PEIMAGE_DIR_WINDOWS),
                                                        module->name, NULL)
                : g_build_filename (peimage_dir_get_path (PEIMAGE_DIR_HOME),
                                    i < 6 ? "bin" : "lib", module->name, NULL);
        module->file = peimage_dir_get_case_insensitive (path);
        g_free (path);
        module->image = peimage_file_load (module->file);
      }
  if (!module->image)
    {
      g_free (module->name);
      g_free (module->file);
      g_free (module);
      return NULL;
    }

  /* ja:インポート */
  if (pe_ioh_get_data_directory_size (module->image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT))
    {
      guint16 sections;
      guint32 address, size;

      sections = pe_ifh_get_number_of_sections (module->image);
      size = pe_ioh_get_size_of_image (module->image);
      for (address = pe_ioh_get_data_directory_virtual_address (module->image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT);
                            address + sizeof (ImageImportDescriptor) <= size;
                                    address += sizeof (ImageImportDescriptor))
        {
          gchar *name;
          ImageImportDescriptor *iid;

          iid = (ImageImportDescriptor *)(module->image + address);
          for (i = iid_get_name (iid); 0 < i && i < size; i++)
            if (module->image[i] == '\0')
              break;
          if (i <= 0 || size <= i)
            break;
          name = (gchar *)(module->image + iid_get_name (iid));
          image = peimage_module_load_library (process,
                                               name,
                                               pemodule_system,
                                               pemodule_relocate,
                                               pemodule_import,
                                               pemodule_entry,
                                               user_data);
          if (image)
            {
              guint32 lookup, thunk;
              ImageSectionHeader *ish;

              ish = pe_image_section_header (module->image);
              for (i = 0; i < sections; i++)
                {
                  if (ish_get_virtual_address (ish)
                                        <= iid_get_original_first_thunk (iid)
                    && iid_get_original_first_thunk (iid)
                                                < ish_get_virtual_address (ish)
                                                + ish_get_virtual_size (ish))
                    break;
                  ish++;
                }
              lookup = i < sections ? iid_get_original_first_thunk (iid)
                                    : iid_get_first_thunk (iid);
              thunk = iid_get_first_thunk (iid);
              while (lookup + sizeof (guint32) <= size
                                        && thunk + sizeof (guint32) <= size)
                {
                  gconstpointer func = NULL;
                  ImageThunkData *ilt, *iat;

                  ilt = (ImageThunkData *)(module->image + lookup);
                  iat = (ImageThunkData *)(module->image + thunk);
                  if (!itd_get_address_of_data (ilt)
                                            || !itd_get_address_of_data (iat))
                    break;
                  if (itd_get_ordinal (ilt) & PEIMAGE_ORDINAL_FLAG)
                    {
                      guint16 ordinal;

                      ordinal = itd_get_ordinal (ilt) & ~PEIMAGE_ORDINAL_FLAG;
                      func = peimage_file_get_proc (image, ordinal);
                    }
                  else if (itd_get_address_of_data (ilt)
                                                    + sizeof (guint16) <= size)
                    {
                      for (i = itd_get_address_of_data (ilt)
                                + sizeof (guint16); 0 <= i && i < size; i++)
                        if (module->image[i] == '\0')
                          {
                            const ImageImportByName *iibn;

                            iibn = (const ImageImportByName *)(module->image
                                            + itd_get_address_of_data (ilt));
                            func = peimage_file_get_func (image,
                                                        iibn_get_name (iibn));
                            break;
                          }
                    }
                  itd_set_function (iat, pemodule_import
                                ? pemodule_import (process, func, user_data)
                                : GPOINTER_TO_UINT (func));
                  lookup += sizeof (guint32);
                  thunk += sizeof (guint32);
                }
              module->library = g_list_append (module->library, image);
            }
          else
            {
              result = FALSE;
            }
        }
    }

  if (!result)
    {
      for (module->library = g_list_first (module->library); module->library;
                                        module->library = g_list_delete_link
                                            (module->library, module->library))
        peimage_module_free_library (process, module->library->data,
                                                    pemodule_entry, user_data);
      g_free (module->image);
      g_free (module->name);
      g_free (module->file);
      g_free (module);
      return NULL;
    }

  /* ja:再配置 */
  if (pemodule_relocate)
    peimage_file_relocate (module->image,
                        pemodule_relocate (process, module->image, user_data));

  /* ja:プロセス登録 */
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  process->module = g_list_append (process->module, module);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */

  if (pemodule_entry
                && !pemodule_entry (process, module->image, TRUE, user_data))
    {
      peimage_module_free_library (process,
                                    module->image, pemodule_entry, user_data);
      return NULL;
    }
  return module->image;
}


/*  ja:イメージを解放する
    process,プロセス
      image,イメージ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
peimage_module_free_library (PeProcess     *process,
                             const guint8  *image,
                             PeModuleEntry  pemodule_entry,
                             gpointer       user_data)
{
  gboolean result = FALSE;
  PeModule *module;

  module = peimage_module_get_info (process, image);
  if (module)
    {
      result = TRUE;
      module->counter--;
      if (module->counter <= 0)
        {
          if (pemodule_entry)
            result = pemodule_entry (process, module->image, FALSE, user_data);
#ifdef USE_THREAD
          G_LOCK (critical);
          critical = TRUE;
#endif /* USE_THREAD */
          process->module = g_list_remove (process->module, module);
#ifdef USE_THREAD
          critical = FALSE;
          G_UNLOCK (critical);
#endif /* USE_THREAD */
          for (module->library = g_list_first (module->library);
                                                            module->library;
                                        module->library = g_list_delete_link
                                            (module->library, module->library))
            if (!peimage_module_free_library (process, module->library->data,
                                                    pemodule_entry, user_data))
              result = FALSE;
          g_free (module->image);
          g_free (module->name);
          g_free (module->file);
          g_free (module);
        }
    }
  return result;
}


/*  ja:イメージの名前を取得する
    process,プロセス
      image,イメージ
        RET,名前,NULL:エラー                                                */
const gchar *
peimage_module_get_filename (PeProcess    *process,
                             const guint8 *image)
{
  PeModule *module;

  module = peimage_module_get_info (process, image);
  return module ? module->file : NULL;
}


/*  ja:プロセスのイメージ一覧を取得する
    process,プロセス
        RET,リスト,NULL:エラー                                              */
GList *
peimage_module_get_list (PeProcess *process)
{
  GList *module = NULL, *glist;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
    module = g_list_append (module, ((PeModule *)glist->data)->image);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
  return module;
}


/******************************************************************************
*                                                                             *
* ja:ディレクトリ関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ディレクトリを取得する
    dir,ディレクトリの種類
    RET,ディレクトリ                                                        */
const gchar *
peimage_dir_get_path (const guint dir)
{
  gchar *file;
  const gchar *key, *path = NULL;
  const static gchar *dir_windows = NULL, *dir_system = NULL, *dir_temp = NULL;
  const static gchar *dir_data = NULL, *dir_home = NULL;
  gint i;
  GKeyFile *key_file;

  switch (dir)
    {
      case PEIMAGE_DIR_WINDOWS: if (dir_windows) return dir_windows; break;
      case PEIMAGE_DIR_SYSTEM:  if (dir_system)  return dir_system;  break;
      case PEIMAGE_DIR_TEMP:    if (dir_temp)    return dir_temp;    break;
      case PEIMAGE_DIR_DATA:    if (dir_data)    return dir_data;    break;
      case PEIMAGE_DIR_HOME:    if (dir_home)    return dir_home;
    }
  for (i = 0; i < 2 && !path; i++)
    {
      switch (dir)
        {
          case PEIMAGE_DIR_WINDOWS:
            file = peimage_dir_get_filename ("system.ini",
                                i == 0 ? PEIMAGE_DIR_HOME : PEIMAGE_DIR_DATA);
            key = "windows";
            break;
          case PEIMAGE_DIR_SYSTEM:
            file = peimage_dir_get_filename ("system.ini",
                                i == 0 ? PEIMAGE_DIR_HOME : PEIMAGE_DIR_DATA);
            key = "system";
            break;
          case PEIMAGE_DIR_TEMP:
            file = peimage_dir_get_filename ("system.ini",
                                i == 0 ? PEIMAGE_DIR_HOME : PEIMAGE_DIR_DATA);
            key = "temp";
            break;
          case PEIMAGE_DIR_DATA:
            if (i == 0)
              {
                file = g_build_filename (g_get_home_dir (),
                                    ".maid.org", "w32ldr", "system.ini", NULL);
                key = "data";
                break;
              }
          default:
            file = NULL;
            key = NULL;
        }
      if (file && key)
        {
          gchar *tmp, *utf8str;

          key_file = g_key_file_new ();
          tmp = peimage_dir_get_case_insensitive (file);
          g_key_file_load_from_file (key_file, tmp, G_KEY_FILE_NONE, NULL);
          g_free (tmp);
          utf8str = g_key_file_get_string (key_file, "directory", key, NULL);
          g_key_file_free (key_file);
          if (utf8str)
            {
              path = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
            }
        }
      g_free (file);
    }
  switch (dir)
    {
      case PEIMAGE_DIR_WINDOWS:
        return dir_windows = path ? path
#ifdef BINDIR
                                  : g_build_filename (BINDIR, "w32ldr", NULL);
#else /* not BINDIR */
                                  : "/usr/local/bin/w32ldr";
#endif /* not BINDIR */
      case PEIMAGE_DIR_SYSTEM:
        return dir_system = path ? path
#ifdef LIBDIR
                                 : g_build_filename (LIBDIR, "w32ldr", NULL);
#else /* not LIBDIR */
                                 : "/usr/local/lib/w32ldr";
#endif /* not LIBDIR */
      case PEIMAGE_DIR_TEMP:
        return dir_temp = path ? path : g_get_tmp_dir ();
      case PEIMAGE_DIR_DATA:
        return dir_data = path ? path
#ifdef SYSCONFDIR
                               : g_build_filename (SYSCONFDIR, "w32ldr", NULL);
#else /* not SYSCONFDIR */
                               : "/usr/local/etc/w32ldr";
#endif /* not SYSCONFDIR */
      case PEIMAGE_DIR_HOME:
        if (!dir_home)
          dir_home = g_build_filename (g_get_home_dir (),
                                                ".maid.org", "w32ldr", NULL);
        return dir_home;
    }
  return NULL;
}


/*  ja:大文字小文字を区別しないパス名を求める
    path,ファイル名
     RET,大文字小文字を区別しないパス名                                     */
gchar *
peimage_dir_get_case_insensitive (const gchar *path)
{
  gchar **element, *file = NULL;
  gint i;

  if (!path)
    return NULL;
  if (path[0] == '\0')
    return g_malloc0 (sizeof (gchar));
  element = g_strsplit (path, G_DIR_SEPARATOR_S, 0);
  for (i = 0; element[i]; i++)
    if ((element[i])[0] == '\0')
      {
        if (!file)
          file = g_strdup (G_DIR_SEPARATOR_S);
      }
    else
      {
        gchar *tmp;

        tmp = file ? g_build_filename (file, element[i], NULL)
                   : g_strdup (element[i]);
        if (g_file_test (tmp, G_FILE_TEST_EXISTS))
          {
            g_free (file);
            file = tmp;
          }
        else
          {
            const gchar *name;
            GDir *dir;

            g_free (tmp);
            tmp = NULL;
            dir = g_dir_open (file ? file : ".", 0, NULL);
            while ((name = g_dir_read_name (dir)))
              if (g_ascii_strcasecmp (element[i], name) == 0)
                {
                  tmp = file ? g_build_filename (file, name, NULL)
                             : g_strdup (name);
                  break;
                }
            g_dir_close (dir);
            if (!tmp)
              {
                while (element[i])
                  {
                    if ((element[i])[0] != '\0')
                      {
                        tmp = file ? g_build_filename (file, element[i], NULL)
                                   : g_strdup (element[i]);
                        g_free (file);
                        file = tmp;
                      }
                    i++;
                  }
                break;
              }
            g_free (file);
            file = tmp;
          }
      }
  g_strfreev (element);
  return file;
}


/*  ja:特定のディレクトリのファイルを取得する
    file,元のファイル名
     dir,ディレクトリの種類
     RET,ファイル                                                           */
gchar *
peimage_dir_get_filename (const gchar *file,
                          const guint  dir)
{
  gchar *name, *tmp, *result;

  name = g_path_get_basename (file);
  tmp = g_build_filename (peimage_dir_get_path (dir), name, NULL);
  g_free (name);
  result = peimage_dir_get_case_insensitive (tmp);
  g_free (tmp);
  return result;
}
