/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "kernel32.h"
#include "commctrl.h"
#include <gdk/gdkkeysyms.h>


#define MAX_COLUMN     32
#define LVIEW_LPARAM   MAX_COLUMN
#define LVIEW_EDITABLE (MAX_COLUMN+1)


static LRESULT WINAPI
w32lview_DefWindowProc (HWND   hWnd,
                        UINT   Msg,
                        WPARAM wParam,
                        LPARAM lParam)
{
  GtkWidget *lview;
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (hWnd), "user_data");
  if (!wd)
    return 0;
  lview = gtk_bin_get_child (GTK_BIN (hWnd));
  switch (Msg)
    {
      case LVM_GETBKCOLOR:
        return RGB (255, 255, 255);
      case LVM_SETBKCOLOR:
        return FALSE;
      case LVM_GETIMAGELIST:
        return GPOINTER_TO_INT (NULL);
      case LVM_SETIMAGELIST:
        return GPOINTER_TO_INT (NULL);
      case LVM_GETITEMCOUNT:
        {
          GtkTreeModel *model;

          model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
          return gtk_tree_model_iter_n_children (model, NULL);
        }
      case LVM_GETITEMA:
        if (lParam)
          {
            GtkTreeIter iter;
            GtkTreeModel *model;
            LRESULT lResult = FALSE;
            LV_ITEMA *lvi;

            lvi = GINT_TO_POINTER (lParam);
            model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
            if (gtk_tree_model_iter_nth_child (model, &iter, NULL, lvi->iItem))
              {
                gchar *utf8str;
                gint param;

                lResult = TRUE;
                gtk_tree_model_get (model, &iter, lvi->iSubItem, &utf8str,
                                                    LVIEW_LPARAM, &param, -1);
                if (lvi->mask & LVIF_STATE)
                  {
                    GtkTreeSelection *select;

                    select = gtk_tree_view_get_selection
                                                    (GTK_TREE_VIEW (lview));
                    lvi->state = gtk_tree_selection_iter_is_selected
                                        (select, &iter) ? LVIS_SELECTED : 0;
                    lvi->stateMask = LVIS_SELECTED;
                  }
                if (lvi->mask & LVIF_TEXT)
                  {
                    gchar *mb;

                    mb = lvi->pszText && lvi->cchTextMax > 0
                                        ? w32ldr_utf8_to_mb (utf8str) : NULL;
                    if (mb)
                      {
                        g_strncpy (lvi->pszText, mb, lvi->cchTextMax);
                        g_free (mb);
                      }
                    else
                      {
                        if (lvi->pszText && lvi->cchTextMax > 0)
                          lvi->pszText[0] = '\0';
                        lvi->mask &= ~LVIF_TEXT;
                        lResult = FALSE;
                      }
                  }
                if (lvi->mask & LVIF_IMAGE)
                  {
                    lvi->iImage = 0;
                    lvi->mask &= ~LVIF_IMAGE;
                    lResult = FALSE;
                  }
                if (lvi->mask & LVIF_PARAM)
                  lvi->lParam = param;
                g_free (utf8str);
              }
            return lResult;
          }
        return FALSE;
      case LVM_GETITEMW:
        if (lParam)
          {
            GtkTreeIter iter;
            GtkTreeModel *model;
            LRESULT lResult = FALSE;
            LV_ITEMW *lvi;

            lvi = GINT_TO_POINTER (lParam);
            model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
            if (gtk_tree_model_iter_nth_child (model, &iter, NULL, lvi->iItem))
              {
                gchar *utf8str;
                gint param;

                lResult = TRUE;
                gtk_tree_model_get (model, &iter, lvi->iSubItem, &utf8str,
                                                    LVIEW_LPARAM, &param, -1);
                if (lvi->mask & LVIF_STATE)
                  {
                    GtkTreeSelection *select;

                    select = gtk_tree_view_get_selection
                                                    (GTK_TREE_VIEW (lview));
                    lvi->state = gtk_tree_selection_iter_is_selected
                                        (select, &iter) ? LVIS_SELECTED : 0;
                    lvi->stateMask = LVIS_SELECTED;
                  }
                if (lvi->mask & LVIF_TEXT)
                  {
                    gunichar2 *wc;

                    wc = utf8str && lvi->pszText && lvi->cchTextMax > 0
                            ? g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL)
                            : NULL;
                    if (wc)
                      {
                        lstrcpynW (lvi->pszText, wc, lvi->cchTextMax);
                        g_free (wc);
                      }
                    else
                      {
                        if (lvi->pszText && lvi->cchTextMax > 0)
                          lvi->pszText[0] = '\0';
                        lvi->mask &= ~LVIF_TEXT;
                        lResult = FALSE;
                      }
                  }
                if (lvi->mask & LVIF_IMAGE)
                  {
                    lvi->iImage = 0;
                    lvi->mask &= ~LVIF_IMAGE;
                    lResult = FALSE;
                  }
                if (lvi->mask & LVIF_PARAM)
                  lvi->lParam = param;
                g_free (utf8str);
              }
            return lResult;
          }
        return FALSE;
      case LVM_SETITEMA:
        if (lParam)
          {
            GtkListStore *store;
            GtkTreeIter iter;
            LRESULT lResult = FALSE;
            LV_ITEMA *lvi;

            lvi = GINT_TO_POINTER (lParam);
            store = GTK_LIST_STORE (gtk_tree_view_get_model
                                                    (GTK_TREE_VIEW (lview)));
            if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                    &iter, NULL, lvi->iItem))
              {
                lResult = TRUE;
                if (lvi->mask & LVIF_STATE)
                  {
                    if (lvi->stateMask & LVIS_SELECTED)
                      {
                        GtkTreeSelection *select;

                        select = gtk_tree_view_get_selection
                                                    (GTK_TREE_VIEW (lview));
                        if (lvi->state & LVIS_SELECTED)
                          gtk_tree_selection_select_iter (select, &iter);
                        else
                          gtk_tree_selection_unselect_iter (select, &iter);
                      }
                    if (lvi->stateMask & ~LVIS_SELECTED)
                      lResult = FALSE;
                  }
                if (lvi->mask & LVIF_TEXT)
                  {
                    gchar *utf8str;

                    utf8str = w32ldr_utf8_from_mb (lvi->pszText);
                    gtk_list_store_set (store, &iter,
                                                lvi->iSubItem, utf8str, -1);
                    g_free (utf8str);
                  }
                if (lvi->mask & LVIF_IMAGE)
                  lResult = FALSE;
                if (lvi->mask & LVIF_PARAM)
                  gtk_list_store_set (store, &iter,
                                                LVIEW_LPARAM, lvi->lParam, -1);
              }
            return lResult;
          }
        return FALSE;
      case LVM_SETITEMW:
        if (lParam)
          {
            GtkListStore *store;
            GtkTreeIter iter;
            LRESULT lResult = FALSE;
            LV_ITEMW *lvi;

            lvi = GINT_TO_POINTER (lParam);
            store = GTK_LIST_STORE (gtk_tree_view_get_model
                                                    (GTK_TREE_VIEW (lview)));
            if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                    &iter, NULL, lvi->iItem))
              {
                lResult = TRUE;
                if (lvi->mask & LVIF_STATE)
                  {
                    if (lvi->stateMask & LVIS_SELECTED)
                      {
                        GtkTreeSelection *select;

                        select = gtk_tree_view_get_selection
                                                    (GTK_TREE_VIEW (lview));
                        if (lvi->state & LVIS_SELECTED)
                          gtk_tree_selection_select_iter (select, &iter);
                        else
                          gtk_tree_selection_unselect_iter (select, &iter);
                      }
                    if (lvi->stateMask & ~LVIS_SELECTED)
                      lResult = FALSE;
                  }
                if (lvi->mask & LVIF_TEXT)
                  {
                    gchar *utf8str;

                    utf8str = lvi->pszText ? g_utf16_to_utf8 (lvi->pszText, -1,
                                                    NULL, NULL, NULL) : NULL;
                    gtk_list_store_set (store, &iter,
                                                lvi->iSubItem, utf8str, -1);
                    g_free (utf8str);
                  }
                if (lvi->mask & LVIF_IMAGE)
                  lResult = FALSE;
                if (lvi->mask & LVIF_PARAM)
                  gtk_list_store_set (store, &iter,
                                                LVIEW_LPARAM, lvi->lParam, -1);
              }
            return lResult;
          }
        return FALSE;
      case LVM_INSERTITEMA:
        if (lParam)
          {
            GtkListStore *store;
            GtkTreeIter iter;
            LV_ITEMA *lvi;

            lvi = GINT_TO_POINTER (lParam);
            store = GTK_LIST_STORE (gtk_tree_view_get_model
                                                    (GTK_TREE_VIEW (lview)));
            gtk_list_store_insert (store, &iter, lvi->iItem);
            gtk_list_store_set (store, &iter,
                            LVIEW_EDITABLE, wd->dwStyle & LVS_EDITLABELS, -1);
            if (lvi->mask & LVIF_STATE && lvi->stateMask & LVIS_SELECTED)
              {
                GtkTreeSelection *select;

                select = gtk_tree_view_get_selection (GTK_TREE_VIEW (lview));
                if (lvi->state & LVIS_SELECTED)
                  gtk_tree_selection_select_iter (select, &iter);
                else
                  gtk_tree_selection_unselect_iter (select, &iter);
              }
            if (lvi->mask & LVIF_TEXT)
              {
                gchar *utf8str;

                utf8str = w32ldr_utf8_from_mb (lvi->pszText);
                gtk_list_store_set (store, &iter, lvi->iSubItem, utf8str, -1);
                g_free (utf8str);
              }
            if (lvi->mask & LVIF_PARAM)
              gtk_list_store_set (store, &iter, LVIEW_LPARAM, lvi->lParam, -1);
            return lvi->iItem;
          }
        return -1;
      case LVM_INSERTITEMW:
        if (lParam)
          {
            GtkListStore *store;
            GtkTreeIter iter;
            LV_ITEMW *lvi;

            lvi = GINT_TO_POINTER (lParam);
            store = GTK_LIST_STORE (gtk_tree_view_get_model
                                                    (GTK_TREE_VIEW (lview)));
            gtk_list_store_insert (store, &iter, lvi->iItem);
            gtk_list_store_set (store, &iter,
                            LVIEW_EDITABLE, wd->dwStyle & LVS_EDITLABELS, -1);
            if (lvi->mask & LVIF_STATE && lvi->stateMask & LVIS_SELECTED)
              {
                GtkTreeSelection *select;

                select = gtk_tree_view_get_selection (GTK_TREE_VIEW (lview));
                if (lvi->state & LVIS_SELECTED)
                  gtk_tree_selection_select_iter (select, &iter);
                else
                  gtk_tree_selection_unselect_iter (select, &iter);
              }
            if (lvi->mask & LVIF_TEXT)
              {
                gchar *utf8str;

                utf8str = lvi->pszText ? g_utf16_to_utf8 (lvi->pszText, -1,
                                                    NULL, NULL, NULL) : NULL;
                gtk_list_store_set (store, &iter, lvi->iSubItem, utf8str, -1);
                g_free (utf8str);
              }
            if (lvi->mask & LVIF_PARAM)
              gtk_list_store_set (store, &iter, LVIEW_LPARAM, lvi->lParam, -1);
            return lvi->iItem;
          }
        return -1;
      case LVM_DELETEITEM:
        {
          GtkListStore *store;
          GtkTreeIter iter;

          store = GTK_LIST_STORE (gtk_tree_view_get_model
                                                    (GTK_TREE_VIEW (lview)));
          if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, wParam))
            {
              gtk_list_store_remove (store, &iter);
              return TRUE;
            }
        }
        return FALSE;
      case LVM_DELETEALLITEMS:
        {
          GtkListStore *store;

          store = GTK_LIST_STORE (gtk_tree_view_get_model
                                                    (GTK_TREE_VIEW (lview)));
          gtk_list_store_clear (store);
        }
        return TRUE;
      case LVM_GETCALLBACKMASK:
        return 0;
      case LVM_SETCALLBACKMASK:
        return FALSE;
      case LVM_GETNEXTITEM:
        {
          gint i, length;
          GtkTreeModel *model;
          GtkTreeSelection *select;

          if (LOWORD (lParam) & (LVNI_FOCUSED | LVNI_CUT | LVNI_DROPHILITED))
            return -1;
          model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
          select = gtk_tree_view_get_selection (GTK_TREE_VIEW (lview));
          length = gtk_tree_model_iter_n_children (model, NULL);
          i = wParam < 0 ? 0
                    : wParam >= length ? length - 1
                    : LOWORD (lParam) & (LVNI_ABOVE | LVNI_TOLEFT) ? wParam - 1
                    : wParam + 1;
            
          while (0 <= i && i < length)
            {
              GtkTreeIter iter;

              if (!(LOWORD (lParam) & LVNI_SELECTED)
                    || (gtk_tree_model_iter_nth_child (model, &iter, NULL, i)
                    && gtk_tree_selection_iter_is_selected (select, &iter)))
                break;
              if (LOWORD (lParam) & (LVNI_ABOVE | LVNI_TOLEFT))
                i--;
              else
                i++;
            }
          return i < length ? i : -1;
        }
      case LVM_FINDITEMA:
      case LVM_FINDITEMW:
        if (lParam)
          {
            gchar *utf8str;
            gint i, length, param;
            guint flags;
            GtkTreeModel *model;

            if (Msg == LVM_FINDITEMW)
              {
                LV_FINDINFOW *lvfi;

                lvfi = GINT_TO_POINTER (lParam);
                flags = lvfi->flags;
                utf8str = lvfi->psz ? g_utf16_to_utf8 (lvfi->psz, -1,
                                                    NULL, NULL, NULL) : NULL;
                param = lvfi->lParam;
              }
            else
              {
                LV_FINDINFOA *lvfi;

                lvfi = GINT_TO_POINTER (lParam);
                flags = lvfi->flags;
                utf8str = w32ldr_utf8_from_mb (lvfi->psz);
                param = lvfi->lParam;
              }
            model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
            length = gtk_tree_model_iter_n_children (model, NULL);
            for (i = wParam + 1; i < length; i++)
              {
                GtkTreeIter iter;

                if (gtk_tree_model_iter_nth_child (model, &iter, NULL, i))
                  {
                    gchar *utf8str_tmp;
                    gint result, param_tmp;

                    gtk_tree_model_get (model, &iter, 0, &utf8str_tmp,
                                                LVIEW_LPARAM, &param_tmp, -1);
                    result = flags & LVFI_PARAM ? param - param_tmp
                        : flags & LVFI_STRING ? g_strcmp (utf8str, utf8str_tmp)
                        : flags & LVFI_PARTIAL ? g_strncmp (utf8str,
                                            utf8str_tmp, g_strlen (utf8str))
                        : 0;
                    g_free (utf8str_tmp);
                    if (result == 0)
                      {
                        g_free (utf8str);
                        return i;
                      }
                  }
              }
            if (flags & LVFI_WRAP)
              for (i = 0; i < wParam; i++)
                {
                  GtkTreeIter iter;

                  if (gtk_tree_model_iter_nth_child (model, &iter, NULL, i))
                    {
                      gchar *utf8str_tmp;
                      gint result, param_tmp;

                      gtk_tree_model_get (model, &iter, 0, &utf8str_tmp,
                                                LVIEW_LPARAM, &param_tmp, -1);
                      result = flags & LVFI_PARAM ? param - param_tmp
                        : flags & LVFI_STRING ? g_strcmp (utf8str, utf8str_tmp)
                        : flags & LVFI_PARTIAL ? g_strncmp (utf8str,
                                            utf8str_tmp, g_strlen (utf8str))
                        : 0;
                      g_free (utf8str_tmp);
                      if (result == 0)
                        {
                          g_free (utf8str);
                          return i;
                        }
                    }
                }
          }
        return -1;
      case LVM_GETITEMRECT:
        if (lParam)
          {
            GtkTreeIter iter;
            GtkTreeModel *model;
            GtkTreePath *path;

            model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
            if (gtk_tree_model_iter_nth_child (model, &iter, NULL, wParam)
                            && (path = gtk_tree_model_get_path (model, &iter)))
              {
                GdkRectangle rect;
                LPRECT lprc;

                lprc = GINT_TO_POINTER (lParam);
                gtk_tree_view_get_cell_area (GTK_TREE_VIEW (lview), path,
                        gtk_tree_view_get_column (GTK_TREE_VIEW (lview), 0),
                                                                        &rect);
                gtk_tree_path_free (path);
                lprc->left = rect.x;
                lprc->top = rect.y;
                lprc->right = rect.x + rect.width;
                lprc->bottom = rect.y + rect.height;
                return TRUE;
              }
          }
        return FALSE;
      case LVM_SETITEMPOSITION:
        return TRUE;
      case LVM_GETITEMPOSITION:
        if (lParam)
          {
            GtkTreeIter iter;
            GtkTreeModel *model;
            GtkTreePath *path;

            model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
            if (gtk_tree_model_iter_nth_child (model, &iter, NULL, wParam)
                            && (path = gtk_tree_model_get_path (model, &iter)))
              {
                GdkRectangle rect;
                LPPOINT lppt;

                lppt = GINT_TO_POINTER (lParam);
                gtk_tree_view_get_cell_area (GTK_TREE_VIEW (lview),
                                                            path, NULL, &rect);
                gtk_tree_path_free (path);
                lppt->x = rect.x;
                lppt->y = rect.y;
                return TRUE;
              }
          }
        return FALSE;
      case LVM_GETSTRINGWIDTHA:
      case LVM_GETSTRINGWIDTHW:
        if (lParam)
          {
            gchar *utf8str;
            gint width;
            PangoLayout *layout;

            utf8str = lParam ? Msg == LVM_GETSTRINGWIDTHW
                             ? g_utf16_to_utf8 (GINT_TO_POINTER (lParam), -1,
                                                            NULL, NULL, NULL)
                             : w32ldr_utf8_from_mb (GINT_TO_POINTER (lParam))
                             : g_strdup ("");
            layout = gtk_widget_create_pango_layout (lview, utf8str);
            pango_layout_get_pixel_size (layout, &width, NULL);
            g_object_unref (G_OBJECT (layout));
            g_free (utf8str);
            return width;
          }
        return 0;
      case LVM_HITTEST:
        if (lParam)
          {
            GtkTreePath *path;
            LV_HITTESTINFO *lvhi;

            lvhi = GINT_TO_POINTER (lParam);
            lvhi->flags = LVHT_NOWHERE;
            lvhi->iItem = -1;
            if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (lview),
                            lvhi->pt.x, lvhi->pt.y, &path, NULL, NULL, NULL))
              {
                gint *indices;

                indices = gtk_tree_path_get_indices (path);
                if (indices)
                  {
                    lvhi->flags = LVHT_ONITEM;
                    lvhi->iItem = indices[0];
                  }
                gtk_tree_path_free (path);
              }
            return lvhi->iItem;
          }
        return -1;
      case LVM_ENSUREVISIBLE:
        {
          GtkTreeIter iter;
          GtkTreeModel *model;
          GtkTreePath *path;

          model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
          if (gtk_tree_model_iter_nth_child (model, &iter, NULL, wParam)
                            && (path = gtk_tree_model_get_path (model, &iter)))
            {
              gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (lview),
                                                    path, NULL, FALSE, 0, 0);
              gtk_tree_path_free (path);
              return TRUE;
            }
        }
        return FALSE;
      case LVM_SCROLL:
        return TRUE;
      case LVM_REDRAWITEMS:
        return TRUE;
      case LVM_ARRANGE:
        return TRUE;
      case LVM_EDITLABELA:
        return GPOINTER_TO_INT (NULL);
      case LVM_EDITLABELW:
        return GPOINTER_TO_INT (NULL);
      case LVM_GETEDITCONTROL:
        return GPOINTER_TO_INT (NULL);
      case LVM_GETCOLUMNA:
        {
          GtkTreeViewColumn *column;
          LRESULT lResult = FALSE;

          column = gtk_tree_view_get_column (GTK_TREE_VIEW (lview), wParam);
          if (column && lParam)
            {
              LV_COLUMNA *lpc;

              lResult = TRUE;
              lpc = GINT_TO_POINTER (lParam);
              if (lpc->mask & LVCF_FMT)
                {
                  gfloat xalign;

                  xalign = gtk_tree_view_column_get_alignment (column);
                  lpc->fmt = xalign < 0.33 ? LVCFMT_LEFT
                           : xalign > 0.66 ? LVCFMT_RIGHT : LVCFMT_CENTER;
                }
              if (lpc->mask & LVCF_WIDTH)
                lpc->cx = gtk_tree_view_column_get_width (column);
              if (lpc->mask & LVCF_TEXT)
                {
                  gchar *mb;

                  mb = lpc->pszText && lpc->cchTextMax > 0 ? w32ldr_utf8_to_mb
                            (gtk_tree_view_column_get_title (column)) : NULL;
                  if (mb)
                    {
                      g_strncpy (lpc->pszText, mb, lpc->cchTextMax);
                      g_free (mb);
                    }
                  else
                    {
                      if (lpc->pszText && lpc->cchTextMax > 0)
                        lpc->pszText[0] = '\0';
                      lpc->mask &= ~LVCF_TEXT;
                      lResult = FALSE;
                    }
                }
              if (lpc->mask & LVCF_SUBITEM)
                lpc->iSubItem = wParam;
            }
          return lResult;
        }
      case LVM_GETCOLUMNW:
        {
          GtkTreeViewColumn *column;
          LRESULT lResult = FALSE;

          column = gtk_tree_view_get_column (GTK_TREE_VIEW (lview), wParam);
          if (column && lParam)
            {
              LV_COLUMNW *lpc;

              lResult = TRUE;
              lpc = GINT_TO_POINTER (lParam);
              if (lpc->mask & LVCF_FMT)
                {
                  gfloat xalign;

                  xalign = gtk_tree_view_column_get_alignment (column);
                  lpc->fmt = xalign < 0.33 ? LVCFMT_LEFT
                           : xalign > 0.66 ? LVCFMT_RIGHT : LVCFMT_CENTER;
                }
              if (lpc->mask & LVCF_WIDTH)
                lpc->cx = gtk_tree_view_column_get_width (column);
              if (lpc->mask & LVCF_TEXT)
                {
                  const gchar *title;
                  gunichar2 *wc;

                  title = gtk_tree_view_column_get_title (column);
                  wc = title && lpc->pszText && lpc->cchTextMax > 0
                        ? g_utf8_to_utf16 (title, -1, NULL, NULL, NULL) : NULL;
                  if (wc)
                    {
                      lstrcpynW (lpc->pszText, wc, lpc->cchTextMax);
                      g_free (wc);
                    }
                  else
                    {
                      if (lpc->pszText && lpc->cchTextMax > 0)
                        lpc->pszText[0] = '\0';
                      lpc->mask &= ~LVCF_TEXT;
                      lResult = FALSE;
                    }
                }
              if (lpc->mask & LVCF_SUBITEM)
                lpc->iSubItem = wParam;
            }
          return lResult;
        }
      case LVM_SETCOLUMNA:
        {
          GtkTreeViewColumn *column;

          column = gtk_tree_view_get_column (GTK_TREE_VIEW (lview), wParam);
          if (column && lParam)
            {
              LV_COLUMNA *lpc;

              lpc = GINT_TO_POINTER (lParam);
              if (lpc->mask & LVCF_FMT)
                {
                  gfloat xalign;

                  switch (lpc->fmt)
                    {
                      case LVCFMT_CENTER: xalign = 0.5; break;
                      case LVCFMT_RIGHT: xalign = 1;
                      default: xalign = 0;
                    }
                  gtk_tree_view_column_set_alignment (column, xalign);
                }
              if (lpc->mask & LVCF_TEXT)
                {
                  gchar *utf8str;

                  utf8str = w32ldr_utf8_from_mb (lpc->pszText);
                  gtk_tree_view_column_set_title (column, utf8str);
                  g_free (utf8str);
                }
              return TRUE;
            }
        }
        return FALSE;
      case LVM_SETCOLUMNW:
        {
          GtkTreeViewColumn *column;

          column = gtk_tree_view_get_column (GTK_TREE_VIEW (lview), wParam);
          if (column && lParam)
            {
              LV_COLUMNW *lpc;

              lpc = GINT_TO_POINTER (lParam);
              if (lpc->mask & LVCF_FMT)
                {
                  gfloat xalign;

                  switch (lpc->fmt)
                    {
                      case LVCFMT_CENTER: xalign = 0.5; break;
                      case LVCFMT_RIGHT: xalign = 1;
                      default: xalign = 0;
                    }
                  gtk_tree_view_column_set_alignment (column, xalign);
                }
              if (lpc->mask & LVCF_TEXT)
                {
                  gchar *utf8str;

                  utf8str = lpc->pszText ? g_utf16_to_utf8 (lpc->pszText, -1,
                                                    NULL, NULL, NULL) : NULL;
                  gtk_tree_view_column_set_title (column, utf8str);
                  g_free (utf8str);
                }
              return TRUE;
            }
        }
        return FALSE;
      case LVM_INSERTCOLUMNA:
      case LVM_INSERTCOLUMNW:
        if (lParam && 0 <= wParam && wParam < MAX_COLUMN - 1)
          {
            gchar *utf8str;
            gint i;
            gfloat xalign = 0;
            GList *glist_columns;
            GtkCellRenderer *renderer;
            GtkTreeViewColumn *column;

            glist_columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (lview));
            for (i = wParam; i < g_list_length (glist_columns); i++)
              {
                GList *glist;

                column = g_list_nth_data (glist_columns, i);
                glist = gtk_tree_view_column_get_cell_renderers (column);
                if (glist)
                  {
                    GtkCellRenderer *renderer;

                    renderer = glist->data;
                    gtk_tree_view_column_set_attributes (column, renderer,
                                                    "text", (i + 1) * 2, NULL);
                    g_list_free (glist);
                  }
              }
            g_list_free (glist_columns);
            if (Msg == LVM_INSERTCOLUMNW)
              {
                LV_COLUMNW *lpc;

                lpc = GINT_TO_POINTER (lParam);
                utf8str = lpc->mask & LVCF_TEXT && lpc->pszText
                        ? g_utf16_to_utf8 (lpc->pszText, -1, NULL, NULL, NULL)
                        : NULL;
                if (lpc->mask & LVCF_FMT)
                  switch (lpc->fmt)
                    {
                      case LVCFMT_CENTER: xalign = 0.5; break;
                      case LVCFMT_RIGHT: xalign = 1;
                    }
              }
            else
              {
                LV_COLUMNA *lpc;

                lpc = GINT_TO_POINTER (lParam);
                utf8str = lpc->mask & LVCF_TEXT
                                ? w32ldr_utf8_from_mb (lpc->pszText) : NULL;
                if (lpc->mask & LVCF_FMT)
                  switch (lpc->fmt)
                    {
                      case LVCFMT_CENTER: xalign = 0.5; break;
                      case LVCFMT_RIGHT: xalign = 1;
                    }
              }
            renderer = gtk_cell_renderer_text_new ();
            column = gtk_tree_view_column_new_with_attributes
                                        (utf8str, renderer, "text", wParam * 2,
                                            "editable", LVIEW_EDITABLE, NULL);
            g_free (utf8str);
            gtk_tree_view_column_set_alignment (column, xalign);
            gtk_tree_view_insert_column (GTK_TREE_VIEW (lview),
                                                            column, wParam);
            return wParam;
          }
        return -1;
      case LVM_DELETECOLUMN:
        {
          GList *glist_columns;

          glist_columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (lview));
          if (glist_columns && wParam < g_list_length (glist_columns))
            {
              gint i;
              GtkTreeViewColumn *column;

              column = g_list_nth_data (glist_columns, wParam);
              gtk_tree_view_remove_column (GTK_TREE_VIEW (lview), column);
              for (i = wParam + 1; i < g_list_length (glist_columns); i++)
                {
                  GList *glist;

                  column = g_list_nth_data (glist_columns, i);
                  glist = gtk_tree_view_column_get_cell_renderers (column);
                  if (glist)
                    {
                      GtkCellRenderer *renderer;

                      renderer = glist->data;
                      gtk_tree_view_column_set_attributes (column, renderer,
                                                    "text", (i - 1) * 2, NULL);
                      g_list_free (glist);
                    }
                }
              g_list_free (glist_columns);
              return TRUE;
            }
        }
        return FALSE;
      case LVM_GETCOLUMNWIDTH:
        {
          GtkTreeViewColumn *column;

          column = gtk_tree_view_get_column (GTK_TREE_VIEW (lview), wParam);
          return column ? gtk_tree_view_column_get_width (column) : 0;
        }
      case LVM_SETCOLUMNWIDTH:
        return TRUE;
      case LVM_GETHEADER:
        return GPOINTER_TO_INT (NULL);
      case LVM_CREATEDRAGIMAGE:
        return GPOINTER_TO_INT (NULL);
      case LVM_GETVIEWRECT:
        if (lParam)
          {
            GdkRectangle rect;
            LPRECT lprc;

            lprc = GINT_TO_POINTER (lParam);
            gtk_tree_view_get_visible_rect (GTK_TREE_VIEW (lview), &rect);
            lprc->left = rect.x;
            lprc->top = rect.y;
            lprc->right = rect.x + rect.width;
            lprc->bottom = rect.x + rect.height;
            return TRUE;
          }
        return FALSE;
      case LVM_GETTEXTCOLOR:
        return RGB (0, 0, 0);
      case LVM_SETTEXTCOLOR:
        return FALSE;
      case LVM_GETTEXTBKCOLOR:
        return RGB (255, 255, 255);
      case LVM_SETTEXTBKCOLOR:
        return FALSE;
      case LVM_GETTOPINDEX:
#if GTK_CHECK_VERSION(2,8,0)
        {
          GtkTreePath *path;

          if (gtk_tree_view_get_visible_range (GTK_TREE_VIEW (lview),
                                                                &path, NULL))
            {
              gint *indices, index;

              indices = gtk_tree_path_get_indices (path);
              index = indices[0];
              gtk_tree_path_free (path);
              return index;
            }
        }
#endif /* GTK_CHECK_VERSION(2,8,0) */
        return 0;
      case LVM_GETCOUNTPERPAGE:
        return w32lview_DefWindowProc (hWnd, LVM_GETITEMCOUNT, 0, 0);
      case LVM_GETORIGIN:
        if (lParam && ((wd->dwStyle & LVS_TYPEMASK) == LVS_ICON
                            || (wd->dwStyle & LVS_TYPEMASK) == LVS_SMALLICON))
          {
            LPPOINT lppt;

            lppt = GINT_TO_POINTER (lParam);
            lppt->x = lppt->y = 0;
            return TRUE;
          }
        return FALSE;
      case LVM_UPDATE:
        return TRUE;
      case LVM_SETITEMSTATE:
        if (lParam)
          {
            LV_ITEMW lvi;

            g_memmove (&lvi, GINT_TO_POINTER (lParam), sizeof (LV_ITEMW));
            lvi.mask = LVIF_STATE;
            lvi.iItem = wParam;
            return w32lview_DefWindowProc (hWnd, LVM_SETITEMW,
                                                    0, GPOINTER_TO_INT (&lvi));
          }
        return FALSE;
      case LVM_GETITEMSTATE:
        {
          GtkTreeIter iter;
          GtkTreeModel *model;

          model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
          if (gtk_tree_model_iter_nth_child (model, &iter, NULL, wParam))
            {
              GtkTreeSelection *select;

              select = gtk_tree_view_get_selection (GTK_TREE_VIEW (lview));
              return (gtk_tree_selection_iter_is_selected (select, &iter)
                                                ? LVIS_SELECTED : 0) & lParam;
            }
        }
        return 0;
      case LVM_GETITEMTEXTA:
        if (lParam)
          {
            LV_ITEMA lvi;

            g_memmove (&lvi, GINT_TO_POINTER (lParam), sizeof (LV_ITEMA));
            lvi.mask = LVIF_TEXT;
            lvi.iItem = wParam;
            w32lview_DefWindowProc (hWnd, LVM_GETITEMA,
                                                    0, GPOINTER_TO_INT (&lvi));
          }
        return 0;
      case LVM_GETITEMTEXTW:
        if (lParam)
          {
            LV_ITEMW lvi;

            g_memmove (&lvi, GINT_TO_POINTER (lParam), sizeof (LV_ITEMW));
            lvi.mask = LVIF_TEXT;
            lvi.iItem = wParam;
            w32lview_DefWindowProc (hWnd, LVM_GETITEMW,
                                                    0, GPOINTER_TO_INT (&lvi));
          }
        return 0;
      case LVM_SETITEMTEXTA:
        if (lParam)
          {
            LV_ITEMA lvi;

            g_memmove (&lvi, GINT_TO_POINTER (lParam), sizeof (LV_ITEMA));
            lvi.mask = LVIF_TEXT;
            lvi.iItem = wParam;
            return w32lview_DefWindowProc (hWnd, LVM_SETITEMA,
                                                    0, GPOINTER_TO_INT (&lvi));
          }
        return FALSE;
      case LVM_SETITEMTEXTW:
        if (lParam)
          {
            LV_ITEMW lvi;

            g_memmove (&lvi, GINT_TO_POINTER (lParam), sizeof (LV_ITEMW));
            lvi.mask = LVIF_TEXT;
            lvi.iItem = wParam;
            return w32lview_DefWindowProc (hWnd, LVM_SETITEMW,
                                                    0, GPOINTER_TO_INT (&lvi));
          }
        return FALSE;
      case LVM_SETITEMCOUNT:
        return 0;
      case LVM_SORTITEMS:
#if GTK_CHECK_VERSION(2,2,0)
        if (lParam)
          {
            gint i;
            GtkListStore *store;
            int CALLBACK (*CompareFunc) (LPARAM, LPARAM, LPARAM);

            CompareFunc = GINT_TO_POINTER (lParam);
            store = GTK_LIST_STORE (gtk_tree_view_get_model
                                                    (GTK_TREE_VIEW (lview)));
            for (i = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (store),
                                                        NULL) - 1; i > 0; i--)
              {
                gint j;
                GtkTreeIter iter1;
                LPARAM lParam1;

                if (!gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                            &iter1, NULL, 0))
                  return FALSE;
                gtk_tree_model_get (GTK_TREE_MODEL (store), &iter1,
                                                LVIEW_LPARAM, &lParam1, -1);
                for (j = 1; j <= i; j++)
                  {
                    GtkTreeIter iter2;
                    LPARAM lParam2;

                    if (!gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                            &iter2, NULL, j))
                      return FALSE;
                    gtk_tree_model_get (GTK_TREE_MODEL (store), &iter2,
                                                LVIEW_LPARAM, &lParam2, -1);
                    if (CompareFunc (lParam1, lParam2, wParam) > 0)
                      {
                        gtk_list_store_swap (store, &iter1, &iter2);
                      }
                    else
                      {
                        iter1 = iter2;
                        lParam1 = lParam2;
                      }
                  }
              }
            return TRUE;
          }
        return FALSE;
#else /* not GTK_CHECK_VERSION(2,2,0) */
        return TRUE;
#endif /* not GTK_CHECK_VERSION(2,2,0) */
      case LVM_SETITEMPOSITION32:
        return TRUE;
      case LVM_GETSELECTEDCOUNT:
        {
          gint i, count = 0;
          GtkTreeIter iter;
          GtkTreeModel *model;
          GtkTreeSelection *select;

          model = gtk_tree_view_get_model (GTK_TREE_VIEW (lview));
          select = gtk_tree_view_get_selection (GTK_TREE_VIEW (lview));
          for (i = 0;
                    gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
            if (gtk_tree_selection_iter_is_selected (select, &iter))
              count++;
          return count;
        }
      case LVM_GETITEMSPACING:
        return 0;
      case LVM_GETISEARCHSTRINGA:
        if (lParam)
          {
            gchar *mb;

            mb = GINT_TO_POINTER (lParam);
            mb[0] = '\0';
          }
        return 0;
      case LVM_GETISEARCHSTRINGW:
        if (lParam)
          {
            gunichar2 *wc;

            wc = GINT_TO_POINTER (lParam);
            wc[0] = '\0';
          }
        return 0;
    }
  return wd->widechar ? DefWindowProcW (hWnd, Msg, wParam, lParam)
                       : DefWindowProcA (hWnd, Msg, wParam, lParam);
}

/* ja:列が変更されたとき */
static void
w32lview_row_changed (GtkTreeModel *model,
                      GtkTreePath  *path,
                      GtkTreeIter  *iter,
                      GtkWidget    *widget)
{
  gint *indices;

  indices = gtk_tree_path_get_indices (path);
  if (indices)
    {
      GtkTreeSelection *select;
      GtkWidget *parent;
      W32LdrWindowData *wd;
      NM_LISTVIEW nml;

      select = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));
      parent = gtk_widget_get_parent (widget);
      wd = g_object_get_data (G_OBJECT (parent), "user_data");
      nml.hdr.hwndFrom = parent;
      nml.hdr.idFrom = wd->wID;
      nml.hdr.code = LVN_ITEMCHANGING;
      nml.iItem = indices[0];
      nml.iSubItem = 0;
      nml.uNewState = gtk_tree_selection_path_is_selected (select, path)
                                                        ? LVIS_SELECTED : 0;
      nml.uOldState = nml.uNewState;
      nml.uChanged = 0;
      nml.ptAction.x = 0;
      nml.ptAction.y = 0;
      gtk_tree_model_get (model, iter, LVIEW_LPARAM, &nml.lParam, -1);
      if (wd->widechar)
        SendMessageW (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nml));
      else
        SendMessageA (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nml));
      nml.hdr.code = LVN_ITEMCHANGED;
      if (wd->widechar)
        SendMessageW (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nml));
      else
        SendMessageA (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nml));
    }
}

/* ja:列が削除されたとき */
static void
w32lview_row_deleted (GtkTreeModel *model,
                      GtkTreePath  *path,
                      GtkWidget    *widget)
{
  GtkWidget *parent;
  W32LdrWindowData *wd;
  NM_LISTVIEW nml;

  parent = gtk_widget_get_parent (widget);
  wd = g_object_get_data (G_OBJECT (parent), "user_data");
  nml.hdr.hwndFrom = parent;
  nml.hdr.idFrom = wd->wID;
  nml.hdr.code = LVN_DELETEITEM;
  nml.iItem = -1;
  nml.iSubItem = 0;
  nml.uNewState = 0;
  nml.uOldState = 0;
  nml.uChanged = 0;
  nml.ptAction.x = 0;
  nml.ptAction.y = 0;
  nml.lParam = 0;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nml));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nml));
}


/* ja:列が挿入されたとき */
static void
w32lview_row_inserted (GtkTreeModel *model,
                       GtkTreePath  *path,
                       GtkWidget    *widget)
{
  gint *indices;

  indices = gtk_tree_path_get_indices (path);
  if (indices)
    {
      GtkWidget *parent;
      W32LdrWindowData *wd;
      NM_LISTVIEW nml;

      parent = gtk_widget_get_parent (widget);
      wd = g_object_get_data (G_OBJECT (parent), "user_data");
      nml.hdr.hwndFrom = parent;
      nml.hdr.idFrom = wd->wID;
      nml.hdr.code = LVN_INSERTITEM;
      nml.iItem = indices[0];
      nml.iSubItem = 0;
      nml.uNewState = 0;
      nml.uOldState = 0;
      nml.uChanged = 0;
      nml.ptAction.x = 0;
      nml.ptAction.y = 0;
      nml.lParam = 0;
      if (wd->widechar)
        SendMessageW (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nml));
      else
        SendMessageA (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nml));
    }
}


/* ja:ヘッダがクリックされたとき */
static void
w32lview_clicked (GtkTreeViewColumn *column,
                  GtkWidget         *widget)
{
  GList *glist;
  GtkWidget *parent;
  W32LdrWindowData *wd;
  NM_LISTVIEW nml;

  parent = gtk_widget_get_parent (widget);
  wd = g_object_get_data (G_OBJECT (parent), "user_data");
  nml.hdr.hwndFrom = parent;
  nml.hdr.idFrom = wd->wID;
  nml.hdr.code = LVN_COLUMNCLICK;
  nml.iItem = -1;
  glist = gtk_tree_view_get_columns (GTK_TREE_VIEW (widget));
  for (nml.iSubItem = 0; nml.iSubItem < g_list_length (glist); nml.iSubItem++)
    if (g_list_nth_data (glist, nml.iSubItem) == column)
      break;
  g_list_free (glist);
  nml.uNewState = 0;
  nml.uOldState = 0;
  nml.uChanged = 0;
  nml.ptAction.x = 0;
  nml.ptAction.y = 0;
  nml.lParam = 0;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nml));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nml));
}


/* ja:クリックされたとき */
static gboolean
w32lview_button_press_event (GtkWidget      *widget,
                             GdkEventButton *event,
                             gpointer        user_data)
{
  GtkWidget *parent;
  W32LdrWindowData *wd;
  NMHDR nmh;

  switch (event->type)
    {
      case GDK_BUTTON_PRESS:
        switch (event->button)
          {
            case 1: nmh.code = NM_CLICK; break;
            case 3: nmh.code = NM_RCLICK; break;
            default: return FALSE;
          }
        break;
      case GDK_2BUTTON_PRESS:
        switch (event->button)
          {
            case 1: nmh.code = NM_DBLCLK; break;
            case 3: nmh.code = NM_RDBLCLK; break;
            default: return FALSE;
          }
        break;
      default: return FALSE;
    }
  parent = gtk_widget_get_parent (widget);
  wd = g_object_get_data (G_OBJECT (parent), "user_data");
  nmh.hwndFrom = parent;
  nmh.idFrom = wd->wID;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  return FALSE;
}


/* ja:フォーカス取得 */
static gboolean
w32lview_focus_in (GtkWidget     *widget,
                   GdkEventFocus *event,
                   gpointer       user_data)
{
  GtkWidget *parent;
  W32LdrWindowData *wd;
  NMHDR nmh;

  parent = gtk_widget_get_parent (widget);
  wd = g_object_get_data (G_OBJECT (parent), "user_data");
  nmh.hwndFrom = parent;
  nmh.idFrom = wd->wID;
  nmh.code = NM_SETFOCUS;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  return FALSE;
}


/* ja:フォーカス喪失 */
static gboolean
w32lview_focus_out (GtkWidget     *widget,
                    GdkEventFocus *event,
                    gpointer       user_data)
{
  GtkWidget *parent;
  W32LdrWindowData *wd;
  NMHDR nmh;

  parent = gtk_widget_get_parent (widget);
  wd = g_object_get_data (G_OBJECT (parent), "user_data");
  nmh.hwndFrom = parent;
  nmh.idFrom = wd->wID;
  nmh.code = NM_KILLFOCUS;
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  else
    SendMessageA (wd->hWndParent, WM_NOTIFY, wd->wID, GPOINTER_TO_INT (&nmh));
  return FALSE;
}


/* ja:キーが押された */
static gboolean
w32lview_key_press_event (GtkWidget   *widget,
                          GdkEventKey *event,
                          gpointer     user_data)
{
  if (event->keyval == GDK_Return)
    {
      GtkWidget *parent;
      W32LdrWindowData *wd;
      NMHDR nmh;

      parent = gtk_widget_get_parent (widget);
      wd = g_object_get_data (G_OBJECT (parent), "user_data");
      nmh.hwndFrom = parent;
      nmh.idFrom = wd->wID;
      nmh.code = NM_RETURN;
      if (wd->widechar)
        SendMessageW (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nmh));
      else
        SendMessageA (wd->hWndParent, WM_NOTIFY,
                                            wd->wID, GPOINTER_TO_INT (&nmh));
    }
  return FALSE;
}


GtkWidget *
w32ldr_control_create_lview (const gchar    *windowname,
                             const guint32   style,
                             const guint32   exstyle,
                             const gint      width,
                             const gint      height,
                             const guint16   id,
                             HWND            hWndParent,
                             HINSTANCE       hInstance,
                             const gboolean  widechar)
{
  gint i;
  GType types[MAX_COLUMN + 2];
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeSelection *select;
  GtkTreeViewColumn *column;
  GtkWidget *lview, *scroll;
  W32LdrWindowData *wd;

  /* ja:リストビュー */
  for (i = 0; i < MAX_COLUMN; i++)
    types[i] = G_TYPE_STRING;
  types[LVIEW_LPARAM] = G_TYPE_INT;
  types[LVIEW_EDITABLE] = G_TYPE_BOOLEAN;
  store = gtk_list_store_newv (G_N_ELEMENTS (types), types);
  lview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (lview),
                                                !(style & LVS_NOCOLUMNHEADER));
  gtk_tree_view_set_enable_search (GTK_TREE_VIEW (lview),
                                                !(style & LVS_NOSORTHEADER));
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
                                "text", 0, "editable", LVIEW_EDITABLE, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (lview), column);
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (lview));
  gtk_tree_selection_set_mode (select,
        style & LVS_SINGLESEL ? GTK_SELECTION_SINGLE : GTK_SELECTION_MULTIPLE);
  g_signal_connect (G_OBJECT (store), "row-changed",
                            G_CALLBACK (w32lview_row_changed), lview);
  g_signal_connect (G_OBJECT (store), "row-deleted",
                            G_CALLBACK (w32lview_row_deleted), lview);
  g_signal_connect (G_OBJECT (store), "row-inserted",
                            G_CALLBACK (w32lview_row_inserted), lview);
  g_signal_connect (G_OBJECT (column), "clicked",
                            G_CALLBACK (w32lview_clicked), lview);
  g_signal_connect (G_OBJECT (lview), "button-press-event",
                            G_CALLBACK (w32lview_button_press_event), NULL);
  g_signal_connect (G_OBJECT (lview), "focus-in-event",
                            G_CALLBACK (w32lview_focus_in), NULL);
  g_signal_connect (G_OBJECT (lview), "focus-out-event",
                            G_CALLBACK (w32lview_focus_out), NULL);
  g_signal_connect (G_OBJECT (lview), "key-press-event",
                            G_CALLBACK (w32lview_key_press_event), NULL);
  gtk_widget_add_events (lview, GDK_BUTTON_PRESS_MASK | GDK_FOCUS_CHANGE_MASK
                                                        | GDK_KEY_PRESS_MASK);
  g_object_unref (G_OBJECT (store));
  gtk_widget_show (lview);
  /* ja:スクロールウインドウ */
  scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
            style & LVS_NOSCROLL ? GTK_POLICY_NEVER : GTK_POLICY_AUTOMATIC,
            style & LVS_NOSCROLL ? GTK_POLICY_NEVER : GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scroll), lview);
  wd = g_malloc (sizeof (W32LdrWindowData));
  wd->dwStyle = style;
  wd->dwExStyle = exstyle;
  wd->hInstance = hInstance;
  wd->lpfnWndProc = w32lview_DefWindowProc;
  wd->hWndParent = hWndParent;
  wd->wID = id;
  wd->dwUserData = 0;
  wd->lpDialogFunc = NULL;
  wd->lResult = -1;
  wd->dwUser = 0;
  wd->csa = NULL;
  wd->csw = NULL;
  wd->dwInitParam = 0;
  wd->widechar = widechar;
  wd->resizable = TRUE;
  wd->classname = g_strdup ("SysListView32");
  wd->focus = lview;
  wd->child = NULL;
  g_object_set_data (G_OBJECT (scroll), "user_data", wd);
  return scroll;
}
