/*
    riffdp.c
    copyright (c) 1998-2019 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include "chunk.h"


#define HEADER_TYPE_VALUE 0
#define HEADER_TYPE_HEX 1
#define HEADER_TYPE_FOURCC 2


typedef struct headers
{
  int mode;     /* 種類 */
  int size;     /* サイズ */
  char *name;   /* 名前 */
} HEADERS;


/*  ヘッタを表示する
       fp,ファイルポインタ
     head,ヘッタ
    space,木の深さ                                                          */
static void
printavi (FILE      *fp,
          HEADERS   *head,
          const int  space)
{
  int i;

  for (i = 0; head[i].size > 0; i++)
    {
      int j;
      long value = 0;

      for (j = 0; j < space; j++)
        fputs ("    ", stdout);
      for (j = 0; j < head[i].size; j++)
        value += fgetc (fp) << (j * 8);
      printf ("             %s : ", head[i].name);
      switch (head[i].mode)
        {
          case HEADER_TYPE_VALUE:
            printf ("%ld\n", value);
            break;
          case HEADER_TYPE_HEX:
            switch (head[i].size)
              {
                case 1:  printf ("%02lXh\n", value); break;
                case 2:  printf ("%04lXh\n", value); break;
                case 4:  printf ("%08lXh\n", value); break;
                default: printf ("%lXh\n", value);
              }
            break;
          case HEADER_TYPE_FOURCC:
            printf ("%08lXh", value);
            for (j = 0; j < 4; j++)
              {
                int c;

                c = (value >> (j * 8)) & 0xff;
                if (c < '\x20' || '\x7e' < c)
                  break;
              }
            if (j == 4)
              {
                fputs (" \'", stdout);
                for (j = 0; j < 4; j++)
                  putchar ((value >> (j * 8)) & 0xff);
                putchar ('\'');
              }
            putchar ('\n');
        }
    }
}


/*  AVIのRIFFファイルをツリー表示する
       fp,ファイルポインタ
       ck,チャンクポインタ
    space,木の深さ
      RET,0:成功,-1:エラー                                                  */
static int
chunkavi (FILE      *fp,
          CHUNK     *ck,
          const int  space)
{
  do
    {
      int i;
      cksize_t size;
      fourcc_t id;
      static int type = 0;
      HEADERS wave[] = {
            {HEADER_TYPE_HEX,    2, "Format Tag"},
            {HEADER_TYPE_VALUE,  2, "Channels"},
            {HEADER_TYPE_VALUE,  4, "Samples Per Sec"},
            {HEADER_TYPE_VALUE,  4, "Avg Bytes Per Sec"},
            {HEADER_TYPE_VALUE,  2, "Block Align"},
            {HEADER_TYPE_VALUE,  2, "Bits Per Sample"},
            {0, 0, NULL}
      };
      HEADERS wavex[] = {
            {HEADER_TYPE_HEX,    2, "Format Tag"},
            {HEADER_TYPE_VALUE,  2, "Channels"},
            {HEADER_TYPE_VALUE,  4, "Samples Per Sec"},
            {HEADER_TYPE_VALUE,  4, "Avg Bytes Per Sec"},
            {HEADER_TYPE_VALUE,  2, "Block Align"},
            {HEADER_TYPE_VALUE,  2, "Bits Per Sample"},
            {HEADER_TYPE_VALUE,  2, "Size"},
            {0, 0, NULL}
      };

      if ((id = chunkid (fp)) == (fourcc_t)-1
                                    || (size = chunksize (fp)) == (cksize_t)-1)
        return -1;
      printf ("%08lX ", ftell (fp));
      for (i = 0; i < space; i++)
        fputs ("    ", stdout);
      printf ("%c%c%c%c (%08lX)", getidcc (id, 1), getidcc (id, 2),
                        getidcc (id, 3), getidcc (id, 4), (unsigned long)size);
      if (id == make4cc ('R', 'I', 'F', 'F')
                                        || id == make4cc ('L', 'I', 'S', 'T'))
        {
          fourcc_t form;

          form = chunkform (fp);
          if (form == (fourcc_t)-1)
            return -1;
          printf (" \'%c%c%c%c\'\n", getidcc (form, 1), getidcc (form, 2),
                                        getidcc (form, 3), getidcc (form, 4));
          if (form != make4cc ('m', 'o', 'v', 'i')
                                        && (chunkin (fp, ck) != 0
                                        || chunkavi (fp, ck, space + 1) != 0
                                                    || chunkout (fp, ck) != 0))
            return -1;
        }
      else
        {
          putchar ('\n');
        }
      /* 詳細表示 */
      if (chunkin (fp, ck) != 0)
        return -1;
      if (id == make4cc ('a', 'v', 'i', 'h'))
        {
          HEADERS avimain[] = {
            {HEADER_TYPE_VALUE,  4, "Micro Sec Per Frame"},
            {HEADER_TYPE_VALUE,  4, "Max Byte Per Sec"},
            {HEADER_TYPE_VALUE,  4, "Padding Granularity"},
            {HEADER_TYPE_HEX,    4, "Flags"},
            {HEADER_TYPE_VALUE,  4, "Total Frames"},
            {HEADER_TYPE_VALUE,  4, "Initial Frames"},
            {HEADER_TYPE_VALUE,  4, "Streams"},
            {HEADER_TYPE_VALUE,  4, "Suggested Buffer Size"},
            {HEADER_TYPE_VALUE,  4, "Width"},
            {HEADER_TYPE_VALUE,  4, "Height"},
            {0, 0, NULL}
          };

          printavi (fp, avimain, space);
        }
      else if (id == make4cc ('s', 't', 'r', 'h'))
        {
          HEADERS avistream[] = {
            {HEADER_TYPE_FOURCC, 4, "Type"},
            {HEADER_TYPE_FOURCC, 4, "Handler"},
            {HEADER_TYPE_HEX,    4, "Flags"},
            {HEADER_TYPE_VALUE,  2, "Priority"},
            {HEADER_TYPE_VALUE,  2, "Language"},
            {HEADER_TYPE_VALUE,  4, "Initial Frames"},
            {HEADER_TYPE_VALUE,  4, "Scale"},
            {HEADER_TYPE_VALUE,  4, "Rate"},
            {HEADER_TYPE_VALUE,  4, "Start"},
            {HEADER_TYPE_VALUE,  4, "Length"},
            {HEADER_TYPE_VALUE,  4, "Suggested Buffer Size"},
            {HEADER_TYPE_VALUE,  4, "Quality"},
            {HEADER_TYPE_VALUE,  4, "Sample Size"},
            {HEADER_TYPE_VALUE,  2, "Frame Left"},
            {HEADER_TYPE_VALUE,  2, "Frame Top"},
            {HEADER_TYPE_VALUE,  2, "Frame Right"},
            {HEADER_TYPE_VALUE,  2, "Frame Bottom"},
            {0, 0, NULL}
          };

          if (fread (&type, sizeof (fourcc_t), 1, fp) != 1
                            || fseek (fp, -sizeof (fourcc_t), SEEK_CUR) != 0)
            return -1;
          printavi (fp, avistream, space);
        }
      else if (id == make4cc ('s', 't', 'r', 'f'))
        {
          HEADERS bitmap[] = {
            {HEADER_TYPE_VALUE,  4, "Size"},
            {HEADER_TYPE_VALUE,  4, "Width"},
            {HEADER_TYPE_VALUE,  4, "Height"},
            {HEADER_TYPE_VALUE,  2, "Planes"},
            {HEADER_TYPE_VALUE,  2, "Bit Count"},
            {HEADER_TYPE_FOURCC, 4, "Compression"},
            {HEADER_TYPE_VALUE,  4, "Size Image"},
            {HEADER_TYPE_VALUE,  4, "X Pels Per Meter"},
            {HEADER_TYPE_VALUE,  4, "Y Pels Per Meter"},
            {HEADER_TYPE_VALUE,  4, "Color Used"},
            {HEADER_TYPE_VALUE,  4, "Color Important"},
            {0, 0, NULL}
          };

          if (type == make4cc ('v', 'i', 'd', 's'))
            printavi (fp, bitmap, space);
          else if (type == make4cc ('a', 'u', 'd', 's'))
            printavi (fp, size <= 16 ? wave : wavex, space);
        }
      else if (id == make4cc ('f', 'm', 't', ' '))
        {
          printavi (fp, size <= 16 ? wave : wavex, space);
        }
      if (chunkout (fp, ck) != 0)
        return -1;
    }
  while (chunknext (fp, ck) == 0);
  return 0;
}


/*  RIFFファイルをツリー表示する
       fp,ファイルポインタ
       ck,チャンクポインタ
    space,木の深さ
      RET,0:成功,-1:エラー                                                  */
static int
chunktree (FILE      *fp,
           CHUNK     *ck,
           const int  space)
{

  do
    {
      int i;
      cksize_t size;
      fourcc_t id;

      if ((id = chunkid (fp)) == (fourcc_t)-1
                                    || (size = chunksize (fp)) == (cksize_t)-1)
        return -1;
      printf ("%08lX ", ftell (fp));
      for (i = 0; i < space; i++)
        fputs ("    ", stdout);
      printf ("%c%c%c%c (%08lX)", getidcc (id, 1), getidcc (id, 2),
                        getidcc (id, 3), getidcc (id, 4), (unsigned long)size);
      if (id == make4cc ('R', 'I', 'F', 'F')
                                        || id == make4cc ('L', 'I', 'S', 'T'))
        {
          fourcc_t form;

          form = chunkform (fp);
          if (form == (fourcc_t)-1)
            return -1;
          printf (" \'%c%c%c%c\'\n", getidcc (form, 1), getidcc (form, 2),
                                        getidcc (form, 3), getidcc (form, 4));
          if (chunkin (fp, ck) != 0 || chunktree (fp, ck, space + 1 ) != 0
                                                    || chunkout (fp, ck) != 0)
            return -1;
        }
      else
        {
          putchar ('\n');
        }
    }
  while (chunknext (fp, ck) == 0);
  return 0;
}


int
main (int   argc,
      char *argv[])
{
  int i;        /* ループカウンタ */
  int mode = 0; /* 0:AVI表示,1:標準表示 */
  int s = 0;    /* argvの中のファイル名 */

  for (i = 1; i < argc; i++)
    if ((argv[i])[0] != '-')
      {
        s++;
      }
    else if (tolower ((argv[i])[1]) == 's' && (argv[i])[2] == '\0')
      {
        mode = 1;
      }
    else
      {
        s = 0;
        break;
      }
  if (s <= 0)
    {
      /* 引数にファイル名がないとき */
      fprintf (stderr,
"copyright (c) 1998-2019 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org\n"
"This program comes with NO WARRANTY, to the extent permitted by law.\n"
"You may redistribute it under the terms of the GNU General Public License;\n"
"see the file named COPYING for details.\n"
"\n"
"Usage: %s [OPTION...] <rifffile...>\n"
"\n"
" -s Standard dump\n"
"\n", argv[0]);
      return 0;
    }

  for (i = 1; i < argc; i++)
    if ((argv[i])[0] != '-')
      {
        CHUNK *ck;  /* チャンクポインタ */
        FILE *fp;   /* ファイルポインタ */

        /* ファイルを開く */
        fp = fopen (argv[i], "rb");
        if (!fp)
          {
            fprintf (stderr, "%s can not open.\n", argv[s]);
            return -1;
          }
        /* ファイルポインタからチャンクポインタを作る */
        ck = chunkopen (fp);
        if (!ck)
          {
            fputs ("chunk open error.\n", stderr);
            return -1;
          }
        /* チャンクを表示する */
        if ((mode == 0 ? chunkavi (fp, ck, 0) : chunktree (fp, ck, 0)) != 0)
          {
            fputs ("chunk error.\n", stderr);
            return -1;
          }
        /* チャンクポインタを解放する */
        if (chunkfree (ck) != 0)
          {
            fputs ("chunk free error.\n", stderr);
            return -1;
          }
      }
  return 0;
}
