/***************************************************************************
                          cecaiamanager.cpp  -  description
                             -------------------
    begin                : 2004ǯ  5  4  17:36:12 JST
    copyright            : (C) 2004 by Tomoaki Murakami
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *   Copyright (C) 2004 by Tomoaki Murakami                                *
 *   solarisphrere@yahoo.co.jp                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <vector>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include <ktempfile.h>
#include "cvoxstormglobal.h"
#include "cecaiamanager.h"
#include "cewf.h"

/////////////////////////////////////////////////////////////////////////////

CEcaIAManager::CEcaIAManager()
{
  nTimerIDFinished = 0;
  nTimerIDCurrentPos = 0;
  nMasterPosition = 0;
  rStatus = NotRunning;
  nBufferSize = 1024;
  //! \todo 󥻥åȥå "TEMP" ơ
  //! (ÿ֤)νԤ
}

/////////////////////////////////////////////////////////////////////////////

CEcaIAManager::~CEcaIAManager()
{
  //! \todo 󥻥åȥå "TEMP" 롣
}

/////////////////////////////////////////////////////////////////////////////

bool
CEcaIAManager::isExisting(const QString& strTrackName) const
{
  if ( getIndexByName(strTrackName) != -1 ) {
    return true;
  } else {
    return false;
  }
}

/////////////////////////////////////////////////////////////////////////////

CEcaIAManager::RunningStatus
CEcaIAManager::runningStatus() const
{
  return rStatus;
}

/////////////////////////////////////////////////////////////////////////////

int
CEcaIAManager::numberOfInputs()
{
  command("ai-list", false);
  return lastCountOfStringList();
}

/////////////////////////////////////////////////////////////////////////////

int 
CEcaIAManager::numberOfOutputs()
{
  command("ao-list", false);
  return lastCountOfStringList();
}

/////////////////////////////////////////////////////////////////////////////

CSourceFormat 
CEcaIAManager::soundSourceFormat() const
{
  return formatRecording;
}

/////////////////////////////////////////////////////////////////////////////

int
CEcaIAManager::bufferSize() const
{
  return nBufferSize;
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::setSoundSourceFormat(const CSourceFormat& format)
{
  formatRecording = format;
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::setBufferSize(int nSize)
{
  nBufferSize = nSize;
}

/////////////////////////////////////////////////////////////////////////////

QTime
CEcaIAManager::pos()
{
  command("cs-get-position", false);

  return QTime().addMSecs(static_cast<int>(lastFloat()) * 1000);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotCreateNewChainsetup(const QString& strChainsetup)
{
  slotReleaseCurrentChainsetup();

  nMasterPosition = 0;
  QString strStatus = "cs-add " + strChainsetup;
  command(strStatus);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotReleaseCurrentChainsetup()
{
  vTrackChains.clear();
  slotStop();
  command("cs-disconnect");
  command("cs-remove");
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotPlay(bool bRecord)
{
  switch ( engineStatus() ) {
  case NotReady:
  case Error:
    emit finished();
    return;
  case Finished:
    slotSetMasterPosition(0);
    break;
  case Running:
    return;
  default:
    break;
  }

  if ( !start(bRecord) ) {
    return;
  }

  command("start");
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotStop()
{
  killTimers(); // kill all timers
  command("cs-disconnect");
  slotSetMasterPosition(this->nMasterPosition);
  // need to reconnect? !!!!!!!!!!!!!!!!!!!!!!!!!!!!
  doDetailStopped();
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotPause(bool bRecord)
{
  if ( runningStatus() == NotRunning ) {
    if ( !start(bRecord) ) {
      return;
    }
  } else {
    killTimers();
  }

  command("stop");
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotResume()
{
  nTimerIDFinished   = startTimer(500); // 500-milisecond timer
  nTimerIDCurrentPos = startTimer(100); // 100-milisecond timer
  command("start");
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotSetMasterPosition(const QTime& time)
{
  int nSec = 0;

  nSec += time.hour() * 3600;
  nSec += time.minute() * 60;
  nSec += time.second();

  slotSetMasterPosition(nSec);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotSetMasterPosition(int nSec)
{
  command(QString("cs-set-position %1").arg(nSec));

  if ( nSec == this->nMasterPosition ) {
    return;
  }

  this->nMasterPosition = nSec;

  int nHour   = nSec / 3600;
  emit masterPositionChanged(QTime(nHour, 
                             (nSec - (nHour * 3600)) / 60,
                             (nSec - (nHour * 3600)) % 60));
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRewind(int nSecond)
{
  if ( rStatus == Playing ) {
    command("cs-rewind " + QString("%1").arg(nSecond));
  } else if ( rStatus == NotRunning ) {
    this->nMasterPosition -= nSecond;
    if ( nMasterPosition < 0 ) {
      this->nMasterPosition = 0;
    }
    int nHour   = this->nMasterPosition / 3600;
    emit masterPositionChanged(QTime(nHour,
                               (this->nMasterPosition - (nHour * 3600)) / 60,
                               (this->nMasterPosition - (nHour * 3600)) % 60));
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotForward(int nSecond)
{
  if ( rStatus == Playing ) {
    command("cs-forward " + QString("%1").arg(nSecond));
  } else if ( rStatus == NotRunning) {
    this->nMasterPosition += nSecond;
    int nHour   = this->nMasterPosition / 3600;
    emit masterPositionChanged(QTime(nHour,
                               (this->nMasterPosition - (nHour * 3600)) / 60,
                               (this->nMasterPosition - (nHour * 3600)) % 60));
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotMixdown(
  bool bWithPlaying, 
  const QString& strChainsetup, 
  const QString& strFileName)
{
  switch ( engineStatus() ) {
  case NotReady:
  case Error:
    emit finished();
    return;
  case Finished:
    slotSetMasterPosition(0);
    break;
  case Running:
    return;
  default:
    break;
  }

  command("cs-add " + strChainsetup);
  command("cs-select " + strChainsetup);

  QValueVector<CTrackInf>::iterator it = vTrackChains.begin();
  while ( it != vTrackChains.end() ) {
    QFileInfo FileInfo((*it).strFileName);
    if ( (*it).strFileName.isEmpty() ) {
      continue;
    } else if ( !FileInfo.exists() ) {
      continue;
    } else if ( !CVoxStormGlobal::isSupportedFileType((*it).strFileName) ) {
      continue;
    }

    // create chain
    QString strCommand = "c-add " + (*it).strTrackName;
    command(strCommand);
    
    // select chain
    slotSelectChain((*it).strTrackName);

    // setup mute condition !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    // if already this name track is existing...???
    if ( (*it).bMute ) {
      slotToggleMute((*it).strTrackName, (*it).bMute);
    }

    // NOTE : don't change order after parameter (level, pan, ...)
    // CEcaIAManager object processes implicitly this order
    // setup level
    strCommand = "cop-add -ea:" + QString("%1").arg((*it).nLevel);
    command(strCommand);
    slotChangeLevel((*it).strTrackName, (*it).nLevel);

    // setup pan
    strCommand = "cop-add -epp:" + QString("%1").arg((*it).nPan);
    command(strCommand);
    slotChangePan((*it).strTrackName, (*it).nPan);

    it++;
  }

  setMixdownIOToCurrentChainsetup(bWithPlaying, strFileName);
  slotSetMasterPosition(this->nMasterPosition);

  if ( !this->isValidChainsetup() ) {
    emit finished();
    return;
  }

  killTimers(); // kill all timers
  nTimerIDFinished   = startTimer(500); // 500-milisecond timer
  nTimerIDCurrentPos = startTimer(100); // 100-milisecond timer

  rStatus = Mixing;

  command("cs-connect");
  command("start");
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotAddChain(const CTrackInf& TrackInfAdd)
{
  // if no input or no output, not be added
  QFileInfo FileInfo(TrackInfAdd.strFileName);
  if ( TrackInfAdd.strFileName.isEmpty() ) {
    return;
  } else if ( (!TrackInfAdd.bRecPort && !FileInfo.exists() ) || 
              !CVoxStormGlobal::isSupportedFileType(TrackInfAdd.strFileName) ) {
    return;
  } else {
    if ( TrackInfAdd.bRecPort ) {
      if ( TrackInfAdd.strInputDevice.isEmpty() ) {
        return;
      }
    } else {
      if ( TrackInfAdd.strOutputDevice.isEmpty() ) {
        return;
      }
    }
  }

  // should be push to vector AT FIRST for after process (other slots or functions)
  vTrackChains.push_back(TrackInfAdd);

  // create chain
  QString strCommand = "c-add " + TrackInfAdd.strTrackName;
  command(strCommand);

  // select chain
  slotSelectChain(TrackInfAdd.strTrackName);

  // setup mute condition !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  // if already this name track is existing...???
  if ( TrackInfAdd.bMute ) {
    slotToggleMute(TrackInfAdd.strTrackName, TrackInfAdd.bMute);
  }

  // NOTE : don't change order after parameter (level, pan, ...)
  // CEcaIAManager object processes implicitly this order
  // setup level
  strCommand = "cop-add -ea:" + QString("%1").arg(TrackInfAdd.nLevel);
  command(strCommand);
  slotChangeLevel(TrackInfAdd.strTrackName, TrackInfAdd.nLevel);

  // setup pan
  strCommand = "cop-add -epp:" + QString("%1").arg(TrackInfAdd.nPan);
  command(strCommand);
  slotChangePan(TrackInfAdd.strTrackName, TrackInfAdd.nPan);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveChain(const QString& strTrackName)
{
  // select chain
  slotSelectChain(strTrackName);

  QString strCommand = "c-remove " + strTrackName;
  command(strCommand);

  QValueVector<CTrackInf>::iterator it = vTrackChains.begin();
  while ( it != vTrackChains.end() ) {
    if ( it->strTrackName == strTrackName ) {
      vTrackChains.erase(it);
      break;
    }
    it++;
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveChain(int nID)
{
  if ( (nID < 0) || (nID >= static_cast<int>(vTrackChains.size())) ) {
    return;
  }

  slotRemoveChain(vTrackChains[nID].strTrackName);
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotRemoveAllChains()
{
  command("c-select-all");
  command("c-remove");
  vTrackChains.clear();
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveAllInputs()
{
  int nInput = numberOfInputs();
  int i;

  for ( i = 0; i < nInput; i++ ) {
    command("ai-index-select 1");
    command("ai-remove");
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotRemoveAllOutputs()
{
  int nOutput = numberOfOutputs();
  int i;

  for ( i = 0; i < nOutput; i++ ) {
    command("ao-index-select 1");
    command("ao-remove");
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangePlaybackDevice(const QString& strTrackName,
                                        const QString& strDevice)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].strOutputDevice = strDevice;

  // setup I/O file (device setting will be done slotPlay)
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeRecordingDevice(const QString& strTrackName,
                                         const QString& strDevice)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].strInputDevice = strDevice;

  // setup I/O file (device setting will be done slotPlay)
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::slotChangeFileName(
  const QString& strTrackName, 
  const QString& strFileName)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  if ( strFileName.isEmpty() ) { // removed file name
    slotRemoveChain(nIndex);
  }

  vTrackChains[nIndex].strFileName = strFileName;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeLevel(const QString& strTrackName, int nLevel)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].nLevel = nLevel;

  // select chain
  slotSelectChain(strTrackName);

  // select level operator
  // NOTE : parameter id is ALWAYS 1 and 1
  QString strCommand = "cop-select 1";
  command(strCommand);
  // select level operator parameter
  strCommand = "copp-select 1";
  command(strCommand);

  // set level
  strCommand = "copp-set " + QString("%1").arg(nLevel);
  command(strCommand);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangePan(const QString& strTrackName, int nPan)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].nPan = nPan;

  // select chain
  slotSelectChain(strTrackName);

  // select pan operator
  // NOTE : parameter id is ALWAYS 2 and 1
  QString strCommand = "cop-select 2";
  command(strCommand);
  // select pan operator parameter
  strCommand = "copp-select 1";
  command(strCommand);

  // set pan
  strCommand = "copp-set " + QString("%1").arg(nPan);
  command(strCommand);
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotToggleMute(const QString& strTrackName, bool bMute)
{
  int nIndex;

  if ( (nIndex = this->getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].bMute = bMute;

  slotSelectChain(strTrackName);
  command("c-muting");
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotToggleRec(const QString& strTrackName, bool bRec)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].bRecPort = bRec;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeStartPosition(
                 const QString& strTrackName,
                 const QTime& startPosition)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].timeStartPosition = startPosition;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeOffset(
                 const QString& strTrackName,
                 const QTime& offset)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].timeOffset = offset;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::slotChangeLength(
                 const QString& strTrackName,
                 const QTime& length)
{
  int nIndex;

  if ( (nIndex = getIndexByName(strTrackName)) == -1 ) {
    return;
  }

  vTrackChains[nIndex].timeLength = length;
}

/////////////////////////////////////////////////////////////////////////////

inline void
CEcaIAManager::slotSelectChain(const QString& strTrackName)
{
  QString strCommand = "c-select " + strTrackName;
  command(strCommand);
}

/////////////////////////////////////////////////////////////////////////////

int
CEcaIAManager::getIndexByName(const QString& strTrackName) const
{
  int i, nTrack;

  nTrack = vTrackChains.size();
  for ( i = 0; i < nTrack; i++ ) {
    if ( strTrackName == vTrackChains[i].strTrackName ) {
      return i;
    }
  }

  return -1;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::setIOToCurrentChainsetup(bool bRecord)
{
  prepareTracks();

  int nChains, i;
  nChains = vPrepareTrackChains.size();

  QString      strCommand;
  QStringList  strlInputDevices;
  QStringList  strlOutputDevices;
  QStringList  strlInputFiles;
  QStringList  strlOutputFiles;
  int          nResult;

  // setup I/O
  slotRemoveAllInputs();
  slotRemoveAllOutputs();
  for ( i = 0; i < nChains; i++ ) {
    if ( bRecord && vPrepareTrackChains[i].bRecPort ) { // recording
      // setup input (device)
      nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strInputDevice,
                              strlInputDevices);
      if ( nResult == -1 ) {
        strlInputDevices.append(vPrepareTrackChains[i].strInputDevice);
        command("ai-add " + vPrepareTrackChains[i].strInputDevice);
      }

      // setup output (file)
      if ( !vPrepareTrackChains[i].strFileName.isEmpty() ) {
        nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strFileName, 
                              strlOutputFiles);
        if ( nResult == -1 ) {
          strlOutputFiles.append(vPrepareTrackChains[i].strFileName);
          command("ao-add " + DQ + vPrepareTrackChains[i].strFileName + DQ);
        }
      }
    } else {                                     // playback
      // setup input (file)
      QFileInfo FileInfo(vPrepareTrackChains[i].strFileName);
      //      if ( !vPrepareTrackChains[i].strFileName.isEmpty() ) {
      if ( FileInfo.exists() ) {
        nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strFileName,
                              strlInputFiles);
        if ( nResult == -1 ) {
          strlInputFiles.append(vPrepareTrackChains[i].strFileName);
          command("ai-add " + DQ + vPrepareTrackChains[i].strFileName + DQ);
        }
      }

      // setup output (device)
      nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strOutputDevice,
                              strlOutputDevices);
      if ( nResult == -1 ) {
        strlOutputDevices.append(vPrepareTrackChains[i].strOutputDevice);
        command("ao-add " + vPrepareTrackChains[i].strOutputDevice);
      }
    }
  }

  QString  strSelection;
  QStringList::Iterator itIO;
  QValueVector<CTrackInf>::Iterator itTrack = vPrepareTrackChains.begin();

  // >>>> attach inputs
  // attach intput devices
  itIO = strlInputDevices.begin();
  while ( itIO != strlInputDevices.end() ){
    command("ai-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( bRecord && (*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strInputDevice ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ai-attach");
    itIO++;
  }

  // attach intput files
  itIO = strlInputFiles.begin();
  while ( itIO != strlInputFiles.end() ){
    command("ai-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( !bRecord || !(*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strFileName ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ai-attach");
    itIO++;
  }
  // <<<< attach intputs

  // >>>> attach oututs
  // attach output devices
  itIO = strlOutputDevices.begin();
  while ( itIO != strlOutputDevices.end() ){
    command("ao-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( !bRecord || !(*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strOutputDevice ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ao-attach");
    itIO++;
  }

  // attach output files
  itIO = strlOutputFiles.begin();
  while ( itIO != strlOutputFiles.end() ){
    command("ao-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( bRecord && (*itTrack).bRecPort ) {
        if ( (*itIO) == (*itTrack).strFileName ) {
          strSelection += (*itTrack).strTrackName + ",";
        }
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ao-attach");
    itIO++;
  }
  // <<<< attach outputs

  destroyPreparings();
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::setMixdownIOToCurrentChainsetup(
  bool, // not use
  const QString& strFileName)
{
  prepareTracks();

  int nChains, i;
  nChains = vPrepareTrackChains.size();

  QString      strCommand;
  QStringList  strlOutputDevices;
  QStringList  strlInputFiles;
  int          nResult;

  // setup I/O
  for ( i = 0; i < nChains; i++ ) {
    // setup input (file)
    QFileInfo FileInfo(vPrepareTrackChains[i].strFileName);
    //      if ( !vPrepareTrackChains[i].strFileName.isEmpty() ) {
    if ( FileInfo.exists() ) {
      nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strFileName,
                              strlInputFiles);
      if ( nResult == -1 ) {
        strlInputFiles.append(vPrepareTrackChains[i].strFileName);
        command("ai-add " + DQ + vPrepareTrackChains[i].strFileName + DQ);
      }

      // setup output (device)
      nResult = CVoxStormGlobal::findStringFromList(
                              vPrepareTrackChains[i].strOutputDevice,
                              strlOutputDevices);
      if ( nResult == -1 ) {
        strlOutputDevices.append(vPrepareTrackChains[i].strOutputDevice);
        command("ao-add " + vPrepareTrackChains[i].strOutputDevice);
      }
    }
  }

  // set mixdown file name
  command("ao-add " + DQ + strFileName + DQ);

  QString  strSelection;
  QStringList::Iterator itIO;
  QValueVector<CTrackInf>::Iterator itTrack = vPrepareTrackChains.begin();

  // >>>> attach inputs
  // attach intput files
  itIO = strlInputFiles.begin();
  while ( itIO != strlInputFiles.end() ){
    command("ai-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( (*itIO) == (*itTrack).strFileName ) {
        strSelection += (*itTrack).strTrackName + ",";
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ai-attach");
    itIO++;
  }
  // <<<< attach intputs

  // >>>> attach oututs
  // attach output devices
  itIO = strlOutputDevices.begin();
  while ( itIO != strlOutputDevices.end() ){
    command("ao-select " + CVoxStormGlobal::fenceString(*itIO));

    strSelection = "";
    itTrack = vPrepareTrackChains.begin();
    while ( itTrack != vPrepareTrackChains.end() ) {
      if ( (*itIO) == (*itTrack).strOutputDevice ) {
        strSelection += (*itTrack).strTrackName + ",";
      }
      itTrack++;
    }

    slotSelectChain(strSelection.left(strSelection.length() - 1));
    command("ao-attach");
    itIO++;
  }

  // attach output files
  command("ao-select " + CVoxStormGlobal::fenceString(strFileName));

  strSelection = "";
  itTrack = vPrepareTrackChains.begin();
  while ( itTrack != vPrepareTrackChains.end() ) {
    strSelection += (*itTrack).strTrackName + ",";
    itTrack++;
  }

  slotSelectChain(strSelection.left(strSelection.length() - 1));
  command("ao-attach");
  // <<<< attach outputs

  destroyPreparings();
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::timerEvent(QTimerEvent* TimerEvent)
{
  if ( TimerEvent->timerId() == nTimerIDFinished ) {
    if ( engineStatus() == Finished ) { // when finished
      killTimer(nTimerIDFinished);
      slotStop();
    }
  } else if ( TimerEvent->timerId() == nTimerIDCurrentPos ) {
    emit moved(pos());
  }

  QObject::timerEvent(TimerEvent);
}

/////////////////////////////////////////////////////////////////////////////

inline void
CEcaIAManager::writeEwf(const QString& strTrackName, const QString& strFileName)
{
  writeEwf(getIndexByName(strTrackName), strFileName);
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::writeEwf(int nIndex, const QString& strFileName)
{
  if ( (nIndex < 0) || (nIndex >= static_cast<int>(vTrackChains.size())) ) {
    return;
  }

  QFile fileEwf(strFileName);
  fileEwf.open(IO_WriteOnly);

  QString strBuffer = "source = ";
  strBuffer += DQ + vTrackChains[nIndex].strFileName + DQ + "\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  QTime time = vTrackChains[nIndex].timeOffset;
  strBuffer = "offset = ";
  strBuffer += QString("%1").arg(time.secsTo(time)) + ".0\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  time = vTrackChains[nIndex].timeStartPosition;
  strBuffer = "start-position = ";
  strBuffer += QString("%1").arg(time.secsTo(time)) + ".0\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  time = vTrackChains[nIndex].timeLength;
  strBuffer = "length = ";
  strBuffer += QString("%1").arg(time.secsTo(time)) + ".0\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  strBuffer = "looping = false\n";
  fileEwf.writeBlock(strBuffer.ascii(), qstrlen(strBuffer));

  fileEwf.close();
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::removeTmpFiles()
{
  QFile fileTmp;
  QStringList::iterator it = strlTmpFiles.begin();

  while ( it != strlTmpFiles.end() ) {
    fileTmp.setName(*it);
    fileTmp.remove();
    it++;
  }

  strlTmpFiles.clear();
}

/////////////////////////////////////////////////////////////////////////////

void 
CEcaIAManager::doDetailStopped()
{
  if ( rStatus == Mixing ) {
    command("cs-remove");
    command("cs-index-select 1");
  }

  rStatus = NotRunning;
  removeTmpFiles();
  emit finished();
}

/////////////////////////////////////////////////////////////////////////////

bool 
CEcaIAManager::start(bool bRecord)
{
  setIOToCurrentChainsetup(bRecord);
  slotSetMasterPosition(this->nMasterPosition);

  if ( !this->isValidChainsetup() ) {
    emit finished();
    return false;
  }

  killTimers(); // kill all timers
  nTimerIDFinished   = startTimer(500); // 500-milisecond timer
  nTimerIDCurrentPos = startTimer(100); // 100-milisecond timer

  if ( bRecord ) {
    rStatus = Recording;
  } else {
    rStatus = Playing;
  }

  command("cs-set-audio-format " + formatRecording.sourceFormatInString());
  command("cs-set-param -b:" + QString("%1").arg(nBufferSize));
  command("cs-connect");

  return true;
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::prepareTracks(bool bIgnoresRecord)
{
  vPrepareTrackChains.clear();

  int nTracks = vTrackChains.size();
  int i;

  for ( i = 0; i < nTracks; i++ ) {
    QFileInfo FileInfo(vTrackChains[i].strFileName);
    bool bEnabledRec;
    if ( bIgnoresRecord ) {
      bEnabledRec = false;
    } else {
      bEnabledRec = vTrackChains[i].bRecPort;
    }

    if ( (!bEnabledRec && !FileInfo.exists()) ||
         !CVoxStormGlobal::isSupportedFileType(vTrackChains[i].strFileName) ) {
      continue;
    }

    CEwf Ewf(vTrackChains[i]);
    if ( Ewf.hasAdvantage() && !bEnabledRec ) {
      KTempFile TempFile(QString::null, ".ewf");
      CTrackInf TrackEwf = vTrackChains[i];

      TempFile.close();
      TrackEwf.strFileName = TempFile.name();
      Ewf.writeEcasoundWaveFile(TempFile.name());

      vPrepareTrackChains.push_back(TrackEwf);
      strlTmpFiles.push_back(TrackEwf.strFileName);
    } else {
      vPrepareTrackChains.push_back(vTrackChains[i]);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void
CEcaIAManager::destroyPreparings()
{
  vPrepareTrackChains.clear();
}

/////////////////////////////////////////////////////////////////////////////

#include "cecaiamanager.moc"
