package Walrus::Session::Lite;
use strict;
use vars qw($VERSION @log);
use Data::Dumper;
use Digest::Perl::MD5 'md5_hex';

=head1 NAME

Walrus::Session::Lite - Simple implementation of storeing session data.

=head1 SYNOPSIS

Use session.

 use Data::Dumper;
 use Walrus::Session::Lite;
 
 print "[ tie ]\n";
 my %session;
 my $session_obj = tie(%session, 'Walrus::Session::Lite', './');
 my $session_id  = $session{'_session_id'};
 
 $session{'sex'} = 'male';
 $session{'name'} = ['Makio', 'Tsukamoto'];
 $session{'birthday'} = {
 	'year' => 1975,
 	'month' => 1,
 	'day' => 9
 };
 print Dumper(\%session);
 
 print "\n[ untie ]\n";
 untie %session;
 print Dumper(\%session);
 
 print "\n[ tie again ]\n";
 $session_obj = tie(%session, 'Walrus::Session::Lite', './', $session_id);
 print Dumper(\%session);
 
 print "\n[ remove file ]\n";
 tied(%session)->delete;
 printf(qq("%s" removed.\n), tied(%session)->{'data_file'});

Manage past session datas.

 use Walrus::Session::Lite;
 
 my %session;
 my $session_obj = tie(%session, 'Walrus::Session::Lite', './');
 
 print "\n[ stored session files ]\n";
 print join("\n", tied(%session)->list_session_files), "\n";
 
 print "\n[ stored session id ]\n";
 print join("\n", tied(%session)->list_session_id), "\n";
 
 print "\n[ delete expired session ]\n";
 my $result = tied(%session)->delete_expired(3 * 24 * 60 * 60); # removes three days past
 print $result ? "Secceeded\n" : "Failed\n";
 
 tied(%session)->delete;      # remove ownself when finishing

=head1 DESCRIPTION

This modules generate session id, and ties hash and file.
Hash values are stored in file when untieing or destroying,
loaded tieing with past session id.

=head1 REQUIREMENT

L<Digest::MD5>

=head1 COPYRIGHT

This library is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

 Copyright 2003 TSUKAMOTO Makio, Oki Software Co.

=head1 AUTHOR

This release was made by TSUKAMOTO Makio <walrus@digit.que.ne.jp>

=cut

$VERSION = '0.2';

sub TIEHASH {
	my $class = shift;
	my $database   = shift;
	my $session_id = (defined($_[0]) and ref($_[0]) eq '') ? shift : undef;
	my $args       = (ref($_[0]) eq 'HASH') ? shift : (@_) ? {@_} : {};
	$database =~ s/\/?$/\//;
	my $self = {
		data         => { _session_id => $session_id },
		data_dir     => $database,
		data_file    => undef,
		lock_dir     => undef,
		lock_file    => undef,
		lock_handle  => undef,
		log          => [],
		status       => undef,
		temp_dir     => undef,
		temp_file    => undef,
	};
	bless $self, $class;
	unless (-d $database) {
		$self->add_log(qq("$database" does not exist.));
		return undef;
	}
	$self->{'lock_dir'} = $args->{'LockDirectory'} if ($args->{'LockDirectory'});
	$self->{'temp_dir'} = $args->{'TempDirectory'} if ($args->{'TempDirectory'});
	$self->set_file_path;
	if (defined $session_id  && $session_id) {
		unless ($self->is_valid_id) {
			$self->add_log(qq("$session_id" is invalid.));
			return undef;
		}
		$self->load;
		unless ($self->{'status'} eq 'SYNCED') {
			$self->add_log(qq("$session_id" loading is failed.));
			return undef;
		}
	} else {
		for (0..10) {
			$self->{'data'}->{'_session_id'} = &md5_hex(time(). {}. rand(). $$);
			$self->set_file_path;
			next if (-e $self->{'data_file'});
			next unless ($self->save);
			last;
		}
		unless ($self->{'status'} eq 'SYNCED') {
			$self->add_log(qq("$session_id" loading is failed.));
			return undef;
		}
		$self->{'status'} = 'NEW'
	}
	return $self;
}

sub FETCH {
	my $self = shift;
	my $key  = shift;
	return $self->{data}->{$key};
}

sub STORE {
	my $self  = shift;
	my $key   = shift;
	my $value = shift;
	return undef if ($key eq '_session_id');
	$self->{data}->{$key} = $value;
	$self->{'status'} = 'MODIFIED';
	return $self->{data}->{$key};
}

sub DELETE {
	my $self = shift;
	my $key  = shift;
	return undef if ($key eq '_session_id');
	$self->{'status'} = 'MODIFIED';
	delete $self->{data}->{$key};
}

sub CLEAR {
	my $self = shift;
	$self->{'status'} = 'MODIFIED';
	$self->{data} = {'_session_id' => $self->{data}->{'_session_id'}};
}

sub EXISTS {
	my $self = shift;
	my $key  = shift;
	return exists $self->{data}->{$key};
}

sub FIRSTKEY {
	my $self = shift;
	my $reset = keys %{$self->{data}};
	return each %{$self->{data}};
}

sub NEXTKEY {
	my $self = shift;
	return each %{$self->{data}};
}

sub UNTIE {
	my $self = shift;
	$self->save;
}

sub DESTROY {
	my $self = shift;
	$self->save;
}

sub is_valid_id {
	my $self = shift;
	my $session_id = (@_) ? shift : $self->{'data'}->{'_session_id'};
	return 1 if ($session_id =~ /^[0-9a-f]+$/);
	$self->add_log(qq(Session id "$session_id" is invalid.));
	return 0;
}

sub load {
	my $self = shift;
	return if ($self->{'status'} eq 'SYNCED');
	return if ($self->{'status'} eq 'NEW');
	unless (open(IN, $self->{'data_file'})) {
		$self->add_log(qq(Can't open session data file: $!));
		return undef;
	}
	my $csv = join('', <IN>);
	close IN;
	$self->{'data'} = {'_session_id' => $self->{'data'}->{'_session_id'}};
	$self->csv_to_session($csv);
	$self->{'status'} = 'SYNCED';
	return 1;
}

sub save {
	my $self = shift;
	return if ($self->{'status'} eq 'SYNCED');
	my $retry = $self->{'args'}->{'lock_retry'} ? $self->{'args'}->{'lock_retry'} : 3;
	$self->{'lock_handle'} = undef;
	for (0..$retry) { last if ($self->create_lock); }
	unless (defined($self->{'lock_handle'})) {
		$self->add_log(qq(Can't create lock: $!));
		return;
	}
	if ($self->{'status'} eq 'DELETED') {
		return unless (-f $self->{'data_file'});
		unless (unlink($self->{'data_file'})) {
			$self->add_log(qq(Can't delete session data file: $!));
			$self->remove_lock;
			return;
		}
	} else {
		my $csv = $self->session_to_csv;
		unless (open(OUT, '>'.$self->{'temp_file'})) {
			$self->add_log(qq(Can't write session temporary file: $!));
			$self->remove_lock;
			return undef;
		}
		print OUT $csv;
		close OUT;
		unless (rename($self->{'temp_file'}, $self->{'data_file'})) {
			$self->add_log(qq(Can't write session data file: $!));
			$self->remove_lock;
			return undef;
		}
	}
	$self->remove_lock;
	$self->{'status'} = 'SYNCED';
	return 1;
}

sub delete {
	my $self = shift;
	$self->{'status'} = 'DELETED';
	$self->save;
}

sub session_to_csv {
	my $self = shift;
	my @keys = grep { $_ ne '_session_id' } keys(%{$self->{'data'}});
	my @records = map { [$_, ref($self->{'data'}->{$_}), $self->{'data'}->{$_}] } @keys;
	for (my $i = 0; $i < @records; $i++) {
		my ($key, $ref, $value) = @{$records[$i]};
		next unless (defined $value);
		my @values;
		if    ($ref eq '')       { @values = ($value); }
		elsif ($ref eq 'SCALAR') { @values = ($value); }
		elsif ($ref eq 'ARRAY')  { @values = @{$value}; }
		elsif ($ref eq 'HASH')   { @values = map { $_ => $value->{$_} } keys(%{$value}); }
		for (my $j = 0; $j < @values; $j++) {
			if    (not defined($values[$j])) {
				$values[$j] = '';
			} elsif (ref($values[$j]) eq '') {
				$values[$j] =~ s/"/""/g;
				$values[$j] = qq("$values[$j]");
			} elsif (ref($values[$j]) eq 'SCALAR') {
				push(@records, ['', '', ${$values[$j]}]);
				$values[$j] = "ref(record ".scalar(@records).")";
			} elsif (ref($values[$j]) eq 'ARRAY' or ref($values[$j]) eq 'HASH' ) {
				push(@records, ['', ref($values[$j]), $values[$j]]);
				$values[$j] = "ref(record ".scalar(@records).")";
			} else {
				($ref, @values) = ('undef');
			}
		}
		$key =~ s/"/""/g;
		$key =~ qq("$key");
		$records[$i] = join(',', $key, $ref, @values);
	}
	my $csv = join("\n", @records);
	return $csv;
}

sub csv_to_session {
	my $self = shift;
	my $csv  = shift;
	my @records;
	my @lines = map { $_ = "$_\n"; } split(/\n/, $csv);
	while (@lines) {
		my $record = shift(@lines);
		$record .= shift(@lines) while ($record =~ tr/"// % 2 and @lines);
		$record =~ s/\s*$/,/;
		push(@records, $record);
	}
	for (my $i = $#records; $i >= 0; $i--) {
		my $record = $records[$i];
		my ($key, $ref, @values) = ($record =~ /("[^"]*(?:""[^"]*)*"|[^,]*),/g);
		$key =~ s/^"(.*)"$/$1/;
		$key =~ s/""/"/g;
		for (my $i = 0; $i < @values; $i++) {
			if    ($values[$i] =~ /^ref\(record (\d+)\)$/) { $values[$i] = $records[$1 - 1]; }
			elsif ($values[$i] =~ s/^"(.*)"$/$1/)          { $values[$i] =~ s/""/"/g; }
		}
		if ($ref eq 'undef')     { $record = undef; }
		elsif ($ref eq '')       { $record = $values[0]; }
		elsif ($ref eq 'SCALAR') { $record = \$values[0]; }
		elsif ($ref eq 'ARRAY')  { $record = [@values]; }
		elsif ($ref eq 'HASH')   { $record = {@values}; }
		$self->{'data'}->{$key} = $record if ($key);
		$records[$i] = $record;
	}
}

sub list_session_files {
	my $self     = shift;
	my $database = $self->{"data_dir"};
	$database    =~ s/\/?$/\//;
	my @files;
	foreach my $file (glob($database.'*')) {
		next unless (-f $file);
		my $session_id = substr($file, length($database));
		next unless ($self->is_valid_id($session_id));
		push(@files, $file);
	}
	return @files;
}

sub list_session_id {
	my $self       = shift;
	my $database   = $self->{"data_dir"};
	$database      =~ s/\/?$/\//;
	my @files      = $self->list_session_files;
	my @session_id = map { substr($_, length($database)) } @files;
	return @session_id;
}

sub delete_expired {
	my $self       = shift;
	my $limit_sec  = shift;
	return unless ($limit_sec and $limit_sec =~ /^\d+$/);
	my $limit_time = time() - $limit_sec;
	my @files      = $self->list_session_files;
	my @expireds   = grep { -f $_ and (stat($_))[9] < $limit_time } @files;
	my $result     = 1;
	foreach my $expired (@expireds) {
		unless (unlink($expired)) {
			$self->add_log(qq(Can't remove expired file "$expired": $!));
			$result = 0;
		}
	}
	return $result;
}

sub create_lock {
	my $self = shift;
	return 0 unless (open LOCK, '>'.$self->{'lock_file'});
	flock( LOCK, 2 );
	$self->{'lock_handle'} = *LOCK;
	return 1;
}

sub remove_lock {
	my $self = shift;
	flock( $self->{'lock_handle'}, 8);
	close $self->{'lock_handle'};
	my $result = unlink $self->{'lock_file'};
	return $result;
}

sub set_file_path {
	my $self = shift;
	foreach my $key (qw(data lock temp)) {
		my $directory = defined($self->{"${key}_dir"}) ? $self->{"${key}_dir"} : $self->{"data_dir"};
		$directory =~ s|\\|/|g;
		$directory =~ s|^([^/])|./$1|;
		$directory =~ s|/\.?/|/|g;
		$directory =~ s|/?$|/|;
		while ($directory =~ s!/(\.{3,}|[^/]*[^/\.][^/]*)/\.\./!/!) { 1; }
		$self->{"${key}_file"} = $directory.$self->{'data'}->{'_session_id'};
	}
	$self->{'lock_file'} = $self->{'data_file'}.'.lock';
	$self->{'temp_file'} = $self->{'data_file'}.'.temp';
}

sub add_log {
	my $self     = shift;
	if (@_) {
		push(@{$self->{'log'}}, @_);
		push(@log, @_);
	}
}

1;
