///////////////////////////////////////////////////////////////////////////////////
/// OpenGL Image (gli.g-truc.net)
///
/// Copyright (c) 2008 - 2015 G-Truc Creation (www.g-truc.net)
/// Permission is hereby granted, free of charge, to any person obtaining a copy
/// of this software and associated documentation files (the "Software"), to deal
/// in the Software without restriction, including without limitation the rights
/// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
/// copies of the Software, and to permit persons to whom the Software is
/// furnished to do so, subject to the following conditions:
/// 
/// The above copyright notice and this permission notice shall be included in
/// all copies or substantial portions of the Software.
/// 
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
/// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
/// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
/// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
/// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
/// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
/// THE SOFTWARE.
///
/// @ref core
/// @file gli/core/format.inl
/// @date 2012-10-16 / 2015-06-16
/// @author Christophe Riccio
///////////////////////////////////////////////////////////////////////////////////

namespace gli{
namespace detail
{
	enum
	{
		CAP_COMPRESSED_BIT = (1 << 0),
		CAP_PACKED_BIT = (1 << 1),
		CAP_NORMALIZED_BIT = (1 << 2),
		CAP_SCALED_BIT = (1 << 3),
		CAP_UNSIGNED_BIT = (1 << 4),
		CAP_SIGNED_BIT = (1 << 5),
		CAP_INTEGER_BIT = (1 << 6),
		CAP_FLOAT_BIT = (1 << 7),
		CAP_DEPTH_BIT = (1 << 8),
		CAP_STENCIL_BIT = (1 << 9),
		CAP_COLORSPACE_SRGB_BIT = (1 << 10),
		CAP_SWIZZLE_BIT = (1 << 11),
		CAP_LUMINANCE_ALPHA_BIT = (1 << 12)
	};

	struct formatInfo
	{
		std::uint8_t BlockSize;
		glm::u8vec3 BlockDimensions;
		std::uint8_t Component;
		std::uint16_t Flags;
	};

	inline formatInfo const & get_format_info(format Format)
	{
		static formatInfo const Table[] =
		{
			// unorm formats
			{  1, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_R8_UNORM,
			{  2, glm::u8vec3(1, 1, 1), 2, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RG8_UNORM,
			{  3, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGB8_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA8_UNORM,

			{  2, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_R16_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 2, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RG16_UNORM,
			{  6, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGB16_UNORM,
			{  8, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA16_UNORM,

			// snorm formats
			{  1, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_R8_SNORM,
			{  2, glm::u8vec3(1, 1, 1), 2, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_RG8_SNORM,
			{  3, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_RGB8_SNORM,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_RGBA8_SNORM,

			{  2, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_R16_SNORM,
			{  4, glm::u8vec3(1, 1, 1), 2, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_RG16_SNORM,
			{  6, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_RGB16_SNORM,
			{  8, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},	//FORMAT_RGBA16_SNORM,

			// Unsigned integer formats
			{  1, glm::u8vec3(1, 1, 1), 1, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_R8_UINT,
			{  2, glm::u8vec3(1, 1, 1), 2, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RG8_UINT,
			{  3, glm::u8vec3(1, 1, 1), 3, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB8_UINT,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA8_UINT,

			{  2, glm::u8vec3(1, 1, 1), 1, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_R16_UINT,
			{  4, glm::u8vec3(1, 1, 1), 2, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RG16_UINT,
			{  6, glm::u8vec3(1, 1, 1), 3, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB16_UINT,
			{  8, glm::u8vec3(1, 1, 1), 4, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA16_UINT,

			{  4, glm::u8vec3(1, 1, 1), 1, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_R32_UINT,
			{  8, glm::u8vec3(1, 1, 1), 2, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RG32_UINT,
			{ 12, glm::u8vec3(1, 1, 1), 3, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB32_UINT,
			{ 16, glm::u8vec3(1, 1, 1), 4, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA32_UINT,

			/// Signed integer formats
			{  1, glm::u8vec3(1, 1, 1), 1, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_R8_SINT,
			{  2, glm::u8vec3(1, 1, 1), 2, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RG8_SINT,
			{  3, glm::u8vec3(1, 1, 1), 3, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RGB8_SINT,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RGBA8_SINT,

			{  2, glm::u8vec3(1, 1, 1), 1, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_R16_SINT,
			{  4, glm::u8vec3(1, 1, 1), 2, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RG16_SINT,
			{  6, glm::u8vec3(1, 1, 1), 3, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RGB16_SINT,
			{  8, glm::u8vec3(1, 1, 1), 4, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RGBA16_SINT,

			{  4, glm::u8vec3(1, 1, 1), 1, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_R32_SINT,
			{  8, glm::u8vec3(1, 1, 1), 2, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RG32_SINT,
			{ 12, glm::u8vec3(1, 1, 1), 3, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RGB32_SINT,
			{ 16, glm::u8vec3(1, 1, 1), 4, CAP_INTEGER_BIT | CAP_SIGNED_BIT},		//FORMAT_RGBA32_SINT,

			/// Floating formats
			{  2, glm::u8vec3(1, 1, 1), 1, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_R16_SFLOAT,
			{  4, glm::u8vec3(1, 1, 1), 2, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_RG16_SFLOAT,
			{  6, glm::u8vec3(1, 1, 1), 3, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_RGB16_SFLOAT,
			{  8, glm::u8vec3(1, 1, 1), 4, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_RGBA16_SFLOAT,

			{  4, glm::u8vec3(1, 1, 1), 1, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_R32_SFLOAT,
			{  8, glm::u8vec3(1, 1, 1), 2, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_RG32_SFLOAT,
			{ 12, glm::u8vec3(1, 1, 1), 3, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_RGB32_SFLOAT,
			{ 16, glm::u8vec3(1, 1, 1), 4, CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_RGBA32_SFLOAT,

			/// sRGB formats
			{  3, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_COLORSPACE_SRGB_BIT},	//FORMAT_R8_SRGB,
			{  3, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_COLORSPACE_SRGB_BIT},	//FORMAT_RG8_SRGB,
			{  3, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_COLORSPACE_SRGB_BIT},	//FORMAT_RGB8_SRGB,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_COLORSPACE_SRGB_BIT},	//FORMAT_RGBA8_SRGB,

			/// Packed formats
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_PACKED_BIT},				//FORMAT_RGB10A2_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_INTEGER_BIT | CAP_UNSIGNED_BIT | CAP_PACKED_BIT},				//FORMAT_RGB10A2_UINT,
			{  4, glm::u8vec3(1, 1, 1), 3, CAP_PACKED_BIT | CAP_FLOAT_BIT | CAP_UNSIGNED_BIT},					//FORMAT_RGB9E5_UFLOAT,
			{  4, glm::u8vec3(1, 1, 1), 3, CAP_PACKED_BIT | CAP_FLOAT_BIT | CAP_SIGNED_BIT},					//FORMAT_RG11B10_UFLOAT,
			{  1, glm::u8vec3(1, 1, 1), 3, CAP_PACKED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},				//FORMAT_RG3B2_UNORM,
			{  2, glm::u8vec3(1, 1, 1), 3, CAP_PACKED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},				//FORMAT_R5G6B5_UNORM,
			{  2, glm::u8vec3(1, 1, 1), 4, CAP_PACKED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},				//FORMAT_RGB5A1_UNORM,
			{  2, glm::u8vec3(1, 1, 1), 4, CAP_PACKED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},				//FORMAT_RGBA4_UNORM,

			// Swizzle formats
			{  4, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_SWIZZLE_BIT},							//FORMAT_BGRX8_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_SWIZZLE_BIT},							//FORMAT_BGRA8_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 3, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_SWIZZLE_BIT | CAP_COLORSPACE_SRGB_BIT},	//FORMAT_BGRX8_SRGB,
			{  4, glm::u8vec3(1, 1, 1), 4, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_SWIZZLE_BIT | CAP_COLORSPACE_SRGB_BIT},	//FORMAT_BGRA8_SRGB,

			// Luminance Alpha formats
			{  1, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_LUMINANCE_ALPHA_BIT},	//FORMAT_L8_UNORM,
			{  1, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_LUMINANCE_ALPHA_BIT},	//FORMAT_A8_UNORM,
			{  2, glm::u8vec3(1, 1, 1), 2, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_LUMINANCE_ALPHA_BIT},	//FORMAT_LA8_UNORM,
			{  2, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_LUMINANCE_ALPHA_BIT},	//FORMAT_L16_UNORM,
			{  2, glm::u8vec3(1, 1, 1), 1, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_LUMINANCE_ALPHA_BIT},	//FORMAT_A16_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 2, CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT | CAP_LUMINANCE_ALPHA_BIT},	//FORMAT_LA16_UNORM,

			/// Depth formats
			{  2, glm::u8vec3(1, 1, 1), 1, CAP_DEPTH_BIT | CAP_INTEGER_BIT},					//FORMAT_D16_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 1, CAP_DEPTH_BIT | CAP_INTEGER_BIT},					//FORMAT_D24_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 2, CAP_DEPTH_BIT | CAP_STENCIL_BIT},					//FORMAT_D24S8_UNORM,
			{  4, glm::u8vec3(1, 1, 1), 1, CAP_DEPTH_BIT | CAP_FLOAT_BIT},						//FORMAT_D32_UFLOAT,
			{  8, glm::u8vec3(1, 1, 1), 2, CAP_DEPTH_BIT | CAP_STENCIL_BIT | CAP_FLOAT_BIT},	//FORMAT_D32_UFLOAT_S8_UNORM,

			/// Compressed formats
			{  8, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_DXT1_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_DXT1_UNORM,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_DXT3_UNORM,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_DXT5_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 1, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_R_ATI1N_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 1, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},		//FORMAT_R_ATI1N_SNORM,
			{ 16, glm::u8vec3(4, 4, 1), 2, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RG_ATI2N_UNORM,
			{ 16, glm::u8vec3(4, 4, 1), 2, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},		//FORMAT_RG_ATI2N_SNORM,
			{ 16, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_FLOAT_BIT | CAP_UNSIGNED_BIT},			//FORMAT_RGB_BP_UFLOAT,
			{ 16, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_FLOAT_BIT | CAP_SIGNED_BIT},			//FORMAT_RGB_BP_SFLOAT,
			{ 16, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_BP_UNORM,
			{ 32, glm::u8vec3(8, 8, 1), 3, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_PVRTC1_8X8_UNORM,
			{ 32, glm::u8vec3(16, 8, 1), 3, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGB_PVRTC1_16X8_UNORM,
			{ 32, glm::u8vec3(8, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_PVRTC1_8X8_UNORM,
			{ 32, glm::u8vec3(16, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_PVRTC1_16X8_UNORM,
			{ 32, glm::u8vec3(8, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_PVRTC2_8X8_UNORM,
			{ 32, glm::u8vec3(16, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_PVRTC2_16X8_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_ATC_UNORM,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ATC_EXPLICIT_UNORM,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ATC_INTERPOLATED_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_ETC_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_ETC2_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ETC2_PUNCHTHROUGH_UNORM,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ETC2_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 1, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_R11_EAC_UNORM,
			{  8, glm::u8vec3(4, 4, 1), 1, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},		//FORMAT_R11_EAC_SNORM,
			{ 16, glm::u8vec3(4, 4, 1), 2, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RG11_EAC_UNORM,
			{ 16, glm::u8vec3(4, 4, 1), 2, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_SIGNED_BIT},		//FORMAT_RG11_EAC_SNORM,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_4X4_UNORM,
			{ 16, glm::u8vec3(5, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_5X4_UNORM,
			{ 16, glm::u8vec3(5, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_5X5_UNORM,
			{ 16, glm::u8vec3(6, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_6X5_UNORM,
			{ 16, glm::u8vec3(6, 6, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_6X6_UNORM,
			{ 16, glm::u8vec3(8, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_8X5_UNORM,
			{ 16, glm::u8vec3(8, 6, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_8X6_UNORM,
			{ 16, glm::u8vec3(8, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC_8X8_UNORM,
			{ 16, glm::u8vec3(10, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_ASTC_10X5_UNORM,
			{ 16, glm::u8vec3(10, 6, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_ASTC_10X6_UNORM,
			{ 16, glm::u8vec3(10, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_ASTC_10X8_UNORM,
			{ 16, glm::u8vec3(10, 10, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_ASTC_10X10_UNORM,
			{ 16, glm::u8vec3(12, 10, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_ASTC_12X10_UNORM,
			{ 16, glm::u8vec3(12, 12, 1), 4, CAP_COMPRESSED_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},	//FORMAT_RGBA_ASTC_12X12_UNORM,

			// Compressed sRGB formats
			{  8, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_DXT1_SRGB,
			{  8, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_DXT1_SRGB,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_DXT3_SRGB,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_DXT5_SRGB,
			{ 16, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_BP_SRGB,
			{ 32, glm::u8vec3(8, 8, 1), 3, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_PVRTC1_8X8_SRGB,
			{ 32, glm::u8vec3(16, 8, 1), 3, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_PVRTC1_16X8_SRGB,
			{ 32, glm::u8vec3(8, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_PVRTC1_8X8_SRGB,
			{ 32, glm::u8vec3(16, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_PVRTC1_16X8_SRGB,
			{ 32, glm::u8vec3(8, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_PVRTC2_8X8_SRGB,
			{ 32, glm::u8vec3(16, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_PVRTC2_16X8_SRGB,
			{  8, glm::u8vec3(4, 4, 1), 3, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGB_ETC2_SRGB,
			{  8, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ETC2_PUNCHTHROUGH_SRGB,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ETC2_SRGB,
			{ 16, glm::u8vec3(4, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC4X4_SRGB,
			{ 16, glm::u8vec3(5, 4, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC5X4_SRGB,
			{ 16, glm::u8vec3(5, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC5X5_SRGB,
			{ 16, glm::u8vec3(6, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC6X5_SRGB,
			{ 16, glm::u8vec3(6, 6, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC6X6_SRGB,
			{ 16, glm::u8vec3(8, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC8X5_SRGB,
			{ 16, glm::u8vec3(8, 6, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC8X6_SRGB,
			{ 16, glm::u8vec3(8, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC8X8_SRGB,
			{ 16, glm::u8vec3(10, 5, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC10X5_SRGB,
			{ 16, glm::u8vec3(10, 6, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC10X6_SRGB,
			{ 16, glm::u8vec3(10, 8, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC10X8_SRGB,
			{ 16, glm::u8vec3(10, 10, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC10X10_SRGB,
			{ 16, glm::u8vec3(12, 10, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC12X10_SRGB,
			{ 16, glm::u8vec3(12, 12, 1), 4, CAP_COMPRESSED_BIT | CAP_COLORSPACE_SRGB_BIT | CAP_NORMALIZED_BIT | CAP_UNSIGNED_BIT},		//FORMAT_RGBA_ASTC12X12_SRGB,
		};

		GLM_STATIC_ASSERT(sizeof(Table) / sizeof(Table[0]) == FORMAT_COUNT, "GLI error: format descriptor list doesn't match number of supported formats");
		assert(Format != static_cast<format>(FORMAT_INVALID));

		return Table[Format];
	};

	inline std::uint32_t bits_per_pixel(format Format)
	{
		detail::formatInfo const & Info = detail::get_format_info(Format);

		return Info.BlockSize * 8 / (Info.BlockDimensions.x * Info.BlockDimensions.y * Info.BlockDimensions.z);
	}
}//namespace detail

	inline std::uint32_t block_size(format Format)
	{
		return detail::get_format_info(Format).BlockSize;
	}

	inline gli::dim3_t block_dimensions(format Format)
	{
		return gli::dim3_t(detail::get_format_info(Format).BlockDimensions);
	}

	inline std::uint32_t component_count(format Format)
	{
		return detail::get_format_info(Format).Component;
	}
}//namespace gli
