/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

/*
** DAV filesystem-based repository provider
*/

#include "apr.h"
#include "apr_file_io.h"
//#include "apr_arch_file_io.h"
#include "apr_strings.h"
#include "apr_portable.h"
#include "apr_thread_mutex.h"
//#include "apr_arch_inherit.h"

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>

/*
 * chown() and chmod() for WebDAV
 */
APR_DECLARE(apr_status_t)
apr_chown_diskquota(const char *filepath, const char* username)
{
    char currentdir[APR_PATH_MAX];
    struct stat filestat;
    struct passwd *user;
    uid_t new_uid = 0;
    static uid_t apache_uid = -1;
    static gid_t apache_gid = -1;
    int length;
    mode_t new_mode = S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP;

    FILE *f;

    /*
     * Get UID/GID of the Apache.
     */

    if (apache_uid == -1)
	apache_uid = getuid();
    if (apache_gid == -1)
	apache_gid = getgid();


    /*
     * Check username
     */
    if (username == NULL)
	return APR_SUCCESS; /* nothing to do */

    /*
     * Check filepath
     */

    if (filepath == NULL)
	return APR_EGENERAL;

    if (filepath[0] != '/')
	return APR_ERELATIVE;

    if (lstat(filepath, &filestat) == -1)
	return errno;

    if (S_ISDIR(filestat.st_mode)) {
	/* path is directory */
	new_mode |= S_IXUSR|S_IXGRP;
    } else
	/* is path a regular file? */
	if (! (S_ISREG(filestat.st_mode)))
	    return APR_EINVAL;

    /*
     * apache_chown_disquota targets a file/direcotry
     * that has same uid/gid of apache process
     */

    if (apache_uid != filestat.st_uid || 
	apache_gid != filestat.st_gid)
	return APR_EACCES;

    /*
     * check the directory that contains the specified file/directory.
     * That must same GID of the Apache process.
     */

    for (length = 1; length < APR_PATH_MAX; length++)
	if (filepath[length] == '\0') break;

    for (length--; filepath[length] == '/'; length--);

    for (; length > 0; length--)
	if (filepath[length] == '/') break;

    memcpy(currentdir, filepath, length);
    currentdir[length] = '\0';

    if (lstat(currentdir, &filestat) == -1)
	return errno;

    if (! S_ISDIR(filestat.st_mode))
	return APR_EINVAL;

    if (apache_gid != filestat.st_gid)
	return APR_EACCES;

    /*
     * Check username is an account of the OS.
     * (if not, then UID of current directory is used)
     */

    new_uid = filestat.st_uid;
    user = getpwnam(username);
    if (user != NULL)
	new_uid = user->pw_uid;

    /*
     * Change owner and permission
     */
    if (chmod(filepath, new_mode) == -1)
	return errno;

    if (lchown(filepath, new_uid, apache_gid) == -1)
	return errno;

    return APR_SUCCESS;
}

