/*
 * $Id: ERBErrorHandlerTest.java,v 1.4 2004/04/20 06:57:04 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests.publish;

import java.io.FileOutputStream;
import java.util.*;

import junit.framework.TestCase;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;

import com.narucy.webpub.core.publish.PublisherBuilder;
import com.narucy.webpub.core.tests.WebpubTestPlugin;

/**
 * <ul>
 * <li>erb-undef_val.html - erb inline error, undef value.
 * <li>erb-undef_method.html - erb inline error, undef method. (tests error marker line location.)
 * <li>erb-undef_method2.html - tests externer script error handling.
 * </ul>
 */
public class ERBErrorHandlerTest extends TestCase {

	IProject project;
	
	protected void setUp() throws Exception{
		project = WebpubTestPlugin.createTestProject("test_projects/ErrorHandling").getProject();
	}
	
	public void testCreateMarker() throws Exception {
		
		// undef variable
		IMarker[] markers = findMarker( project.getFile("ht_sources/erb-undef_val.html") );
		assertEquals(2, markers.length);
		
		assertEquals(PublisherBuilder.MARKER_MISSED_SRC, markers[0].getType());
		assertEquals(PublisherBuilder.MARKER_RUBY, markers[1].getType());
		
		String lines = markers[1].getAttribute(PublisherBuilder.ATTR_BACKTRACE, null);
		assertTrue(lines.indexOf("(erb):3:undefined local variable or method `a' for main:Object") != -1);
		assertEquals(markers[0].getAttribute(IMarker.MESSAGE), markers[1].getAttribute(IMarker.MESSAGE));
		
		// undef method
		markers = findMarker( project.getFile("ht_sources/erb-undef_method.html") );
		
		assertEquals(2, markers.length);
		assertEquals(PublisherBuilder.MARKER_MISSED_SRC, markers[0].getType());
		assertEquals(PublisherBuilder.MARKER_RUBY, markers[1].getType());
		
		lines = markers[1].getAttribute(PublisherBuilder.ATTR_BACKTRACE, null);
		assertTrue(lines.indexOf("(erb):4:in `y': undefined method `z' for main:Object") != -1);
		assertTrue(lines.indexOf("from (erb):3:in `x'") != -1);
		assertTrue(lines.indexOf("from (erb):5") != -1);
		
		// occastion undef method error in external script
		IFile htFile = project.getFile("ht_sources/erb-undef_method2.html");
		IMarker[] srcMarker = findMarker(htFile);
		assertEquals(1, srcMarker.length);
		assertEquals(PublisherBuilder.MARKER_MISSED_SRC, srcMarker[0].getType());
		
		IFile rubyFile = project.getFile("scripts/lib/undef_method_lib.rb");
		IMarker[] rubyMarker = findMarker(rubyFile);
		assertEquals(1, rubyMarker.length);
		assertEquals(PublisherBuilder.MARKER_RUBY, rubyMarker[0].getType());
		
		assertEquals(
				srcMarker[0].getAttribute(IMarker.MESSAGE),
				rubyMarker[0].getAttribute(IMarker.MESSAGE));
		assertTrue(
				rubyMarker[0].getAttribute(PublisherBuilder.ATTR_BACKTRACE).toString()
					.indexOf("in `yyy': undefined method `zzz' for main:Object") != -1);
		
		// delete marker
		FileOutputStream out = new FileOutputStream( htFile.getLocation().toFile() );
		out.close();
		htFile.refreshLocal(IResource.DEPTH_ZERO, null);
		htFile.getProject().build(IncrementalProjectBuilder.INCREMENTAL_BUILD, null);
		assertEquals(0, findMarker(htFile).length);
		assertEquals(0, findMarker(rubyFile).length);
		
	}

	IMarker[] findMarker(IResource r) throws CoreException{
		IMarker[] markers = r.findMarkers(null, false, IResource.DEPTH_ZERO);
		Arrays.sort(markers, new Comparator() {
			public boolean equals(Object obj) {
				return false;
			}
			public int compare(Object a, Object b) {
				try {
					return ((IMarker)a).getType().compareTo( ((IMarker)b).getType() );
				} catch (CoreException e) {
					return 0;
				}
			}
		});
		return markers;
	}
	
	protected void tearDown() throws Exception {
		WebpubTestPlugin.forceDelete(project);
	}

}
