/*
 * $Id: PublishDescriptionFactory.java,v 1.5 2004/04/01 05:48:08 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;

import java.io.IOException;
import java.util.*;
import java.util.regex.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

public class PublishDescriptionFactory {

	final static String
		PUBLISHCODE_BEGIN = "<?publish ",
		PUBLISHCODE_END = "?>";		
	
	final WebProject webProject;
	
	public PublishDescriptionFactory(WebProject webProject){
		this.webProject = webProject;
	}
	
	/**
	 * <p>
	 * To create PublishDescription from iterator. 
	 * This method for on memory document data.
	 * 
	 * <p>
	 * This method can not automatically set a publish location.
	 */
	public PublishDescription create(Iterator ite) {
		while(ite.hasNext()){
			String line = (String)ite.next();
			if(line == null){
				continue;
			}
			int begin = line.indexOf(PUBLISHCODE_BEGIN);
			int end = line.indexOf(PUBLISHCODE_END);
			if( begin != -1 && end != -1){
				return createFromMatcher(line.substring(begin + PUBLISHCODE_BEGIN.length(), end));
			}
		}
		return null;
	}

	boolean checkPublishSource(IResource res) throws CoreException {
		if( !(res instanceof IFile || res instanceof IFolder) || !res.isAccessible()){
			return false;
		}
		IContainer pubFolder = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER);
		for(IResource r = res; !(r instanceof IWorkspaceRoot); r = r.getParent()){
			if( pubFolder.equals(r) ){
				return false;
			}
		}
		String[] ignores = webProject.getArray(WebProject.KEY_IGNORE_FILE);
		for (int i = 0; i < ignores.length; i++) {
			if( new Wildcard(ignores[i]).match(res.getName()) ){
				return false;
			}
		}
		return true;
	}
	
	/**
	 * <p>
	 * Create PublishDescription (this method is define publish location from
	 * referer htFile location that process is not exec other method).
	 * 
	 * <p>
	 * Return null if do not found publish description.
	 */
	public PublishDescription create(IResource file) throws IOException, CoreException, IllegalConfigurationException {
		if( !checkPublishSource(file) ){
			return null;
		}
		
		PublishDescription desc = null;
		if(file instanceof IFile){
			desc = createFromFile((IFile)file);
		}
		if(desc == null){
			// if can not create specify html file.
			// search generator property in parent folder.
			desc = createFromParent(file);
		}
		if( desc != null){
			// if creation successfuly, initialize publish location.
			initPublishLocation(file, desc);
			if(desc.getPublishTo() == null){
				desc = null;
			}
		}
		return desc;
	}
	
	public PublishDescription createFromFile(IFile file) throws CoreException, IOException {
		WebProject wp = (WebProject)file.getProject().getNature(WebProject.ID_NATURE);
		
		if( !file.exists() || !file.isLocal(IResource.DEPTH_ZERO) || wp == null || !wp.isHTExtension(file.getFileExtension()) ){
			return null;
		}
		TextReader reader = null;
		try{
			reader = new TextReader(file.getContents());
			return create(reader);
		} finally{
			if(reader != null){
				reader.close();
			}
		}
	}
	
	void initPublishLocation(IResource source, PublishDescription desc) throws CoreException, IllegalConfigurationException {
		if( desc.getPublishTo() != null ){
			return;
		}
		
		IProject proj = source.getProject();
		WebProject webProj = (WebProject)proj.getNature(WebProject.ID_NATURE);
		if(webProj == null){
			// illegal placed publish description file (.publish)
			return;
		}
		IContainer publishFolder = webProj.getFolder(WebProject.KEY_PUBLISH_FOLDER);
		
		String publishPath = desc.getArgument("publish_to");
		if( publishPath != null){
			// first char is '/', remove this.
			if (publishPath.charAt(0) == '/'){
				publishPath = publishPath.substring(1);
			}
			
			// attribute is setted, referer this values.
			IResource res;
			Matcher m;
			if(publishPath.charAt(publishPath.length()-1) == '/'){
				// specified direcotry
				res = publishFolder.getFolder(new Path(publishPath));
			}else if ( (m = Pattern.compile("(.+)/\\*\\.(.+?)$").matcher(publishPath)).matches() ){
				// use wild card, provisional division.
				IFolder folder = publishFolder.getFolder(new Path(m.group(1)));
				String ext = source.getFileExtension();
				String publishFileName = source.getName().replaceFirst(ext + "$",  m.group(2));
				res = folder.getFile(publishFileName);
			}else{
				// specify file.
				res = publishFolder.getFile(new Path(publishPath));
			}
			desc.setPublishTo(res);
			desc.setArgument("publish_to", null);
		}else{
			IResource publishLocation = htSourceLocationToPublishLocation(source);
			if( publishLocation != null){
				desc.setPublishTo(publishLocation);
			}
		}
	}
	
	/**
	 * Returns copy targets (publish location) from specify resource location.
	 * If specify location is not able to define a publish location, return null.
	 */
	IResource htSourceLocationToPublishLocation(IResource res) throws CoreException{
		WebProject webProj = (WebProject)res.getProject().getNature(WebProject.ID_NATURE);
		
		String htSourceFolder = webProj.getFolder(WebProject.KEY_HTSOURCES_FOLDER).getFullPath().toString();
		String resPath = res.getFullPath().toString();
		
		if (resPath.indexOf(htSourceFolder) != -1){
			String relativePath = resPath.substring(
				htSourceFolder.toString().length()+1, resPath.length() );
			return webProj.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFile( new Path(relativePath) );
		}
		
		return null;

	}
	
	/**
	 * Creates publish description from parent folder on reflexive.
	 * If none looking publish description in html source folder, return null.
	 */
	PublishDescription createFromParent(IResource targetFile) throws CoreException {
		PublishDescription desc = null;
		
		IFile[] propFiles = findPublishPropertyFiles(targetFile);
		for (int i = 0; i < propFiles.length; i++) {
			desc = createFromPropertyFile(targetFile, propFiles[i]);
			if(desc != null){
				break;
			}
		}
		if( desc == null){
			desc = createDefaultPublishDescription(targetFile);
		}
		return desc;
	}
	
	public PublishDescriptionDefinition findMatchDefinition(IResource source) throws CoreException{
		IFile[] propFiles = findPublishPropertyFiles(source);
		for (int i = 0; i < propFiles.length; i++) {
			IFile propFile = propFiles[i];
			
			PublishDescriptionDefinition definition =
					PublishDescriptionDefinitionStore.getInstance().getDescriptionDefinication(propFile);
			
			if(definition != null && definition.getMatchIndex(source) != -1){
				return definition;
			}
		}
		return null;
	}
	
	public IFile[] findPublishPropertyFiles(IResource res) throws CoreException {
		if(!checkPublishSource(res) ){
			return null;
		}
		ArrayList files = new ArrayList();
		while((res = res.getParent()) instanceof IFolder){
			IResource propFile = ((IFolder)res).findMember(".publish");
			if(propFile instanceof IFile && propFile.exists()){
				files.add(propFile);
			}
		}
		return (IFile[]) files.toArray(new IFile[files.size()]);
	}

	/**
	 * Return null if can not publish location as specify file.
	 */
	PublishDescription createDefaultPublishDescription(IResource targetFile) throws CoreException{
		if(targetFile.isAccessible()){
			WebProject webProj = (WebProject)targetFile.getProject().getNature(WebProject.ID_NATURE);
			if( webProj != null){
				IContainer htSourceFolder = webProj.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
				for(IResource res = targetFile; (res = res.getParent()) instanceof IFolder; ){
					if (res.equals(htSourceFolder) ){
						IResource copyTarget = htSourceLocationToPublishLocation(targetFile);
						if( copyTarget != null ){
							PublishDescription desc = new PublishDescription("copy");
							desc.setPublishTo(copyTarget);
							return desc;
						}
					}
				}
			}
		}
		
		return null;
	}
	
	PublishDescription createFromPropertyFile(IResource source, IFile propFile) throws CoreException {
		PublishDescriptionDefinition definition = findMatchDefinition(source);
		if(definition != null){
			int index = definition.getMatchIndex(source);
			PublishDescription desc = new PublishDescription(definition.getPublishBy(index));
			
			Map args = definition.getArguments(index);
			Object[] keys = args.keySet().toArray();
			for (int j = 0; j < keys.length; j++) {
				String k = (String)keys[j];
				desc.setArgument(k, (String)args.get(k));
			}
			return desc;
		}
		return null;
	}
	
	/**
	 * Parse as follow style line, and return that created PublishDescriptor
	 * form line.
	 * <pre>
	 *   file="home3.rb" class="Home" method="home" publish_to="foo/home.html
	 * </pre>
	 */
	PublishDescription createFromMatcher(String code) {
		return createFromMap( lineToMap(code) );
	}
	
	static Map lineToMap(String code){
		Map map = new HashMap();
		Pattern  pattern = Pattern.compile("(.+?)\\s*=\\s*\"(.+?)\"");
		Matcher matcher = pattern.matcher(code);
		while (matcher.find()) {
			map.put( matcher.group(1).trim(), matcher.group(2));
		}
		return map;
	}

	/**
	 * Create PublishDescriptor from specify Map represents properties.
	 */
	PublishDescription createFromMap(Map map) {
		String publishBy = (String)map.remove("by");
		PublishDescription desc = new PublishDescription(publishBy);
		Object[] keys = map.keySet().toArray();
		for (int i = 0; i < keys.length; i++) {
			String k = (String)keys[i];
			desc.setArgument( k, (String)map.get(k) );
		}
		return desc;
	}

}

