/*
 * $Id: PublisherBuilder.java,v 1.14 2004/04/20 06:52:05 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;

import java.io.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

/**
 * PublisherBuilder invokes wp command.
 */
public class PublisherBuilder extends IncrementalProjectBuilder {

	final public static String
		ID_BUILDER = WebpubPlugin.ID_PLUGIN + ".publisherBuilder",
		MARKER_PUBLISH = WebpubPlugin.ID_PLUGIN + ".publish.marker",
		MARKER_MISSED_SRC = WebpubPlugin.ID_PLUGIN + ".publish.missedsrcmarker",
		MARKER_RUBY = WebpubPlugin.ID_PLUGIN + ".publish.rubymarker",
		ATTR_BACKTRACE = "backtrace",
		ATTR_RELATE_RESOURCE = "relateResource";
	
	static IPath publishersPluginEntryFile =
		WebpubPlugin.getDefault().getStateLocation().append("publisherPluginEntryFile");

	static {
		PublisherRegistory registory = PublisherRegistory.getInstance();
		PrintWriter writer = null;
		try{
			FileOutputStream out = new FileOutputStream(publishersPluginEntryFile.toFile());
			writer = new PrintWriter(new OutputStreamWriter(out));
			String[] byKeys = registory.getPublishByKeys();
			for (int i = 0; i < byKeys.length; i++) {
				String k = byKeys[i];
				writer.println(k);
				writer.println("\t" + registory.getPublishScriptFileLocation(k));
				writer.println("\t" + registory.getPublishClassName(k));
				writer.println();
			}
		} catch (FileNotFoundException e) {
			throw new RuntimeException(e);
		} finally{
			if(writer != null){
				writer.close();
			}
		}
	}

	String publishCommand = WebpubPlugin.getLocalPath("scripts/bin/wp");
	int readSleepMs = 10;
	IPath currnetLogFile;
	
	HashMap errorHandlers = new HashMap();

	public PublisherBuilder(){
		errorHandlers.put(null, new PublishErrorHandler());
		errorHandlers.put("amrita", new AmritaErrorHandler());
		errorHandlers.put("erb", new ERBErrorHandler());
	}

	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		IResourceDelta delta = getDelta(getProject());
		switch (kind) {
			case FULL_BUILD:
				fullBuild(monitor);
				break;
			case AUTO_BUILD:
			case INCREMENTAL_BUILD:
				if (delta == null) {
					fullBuild(monitor);
				} else {
					incrementalBuild(delta, monitor);
				}
				break;
			case CLEAN_BUILD:
				cleanBuild(monitor);
				break;
		}
		return null;
	}
	
	void fullBuild(IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("publish full sources", 1);
		try{
			IResource[] resources = findTargetResources();
			if(resources.length > 0){
				doPublish(resources, new SubProgressMonitor(monitor, 1));
			}
		}finally{
			monitor.done();
		}
	}

	/**
	 * Choose publishing atarget files these are all of files
	 * except of script folder and publisher.
	 */	
	IResource[] findTargetResources() throws CoreException {
		ArrayList dist = new ArrayList();
		collectPublishSources(getProject(), dist);
		return (IResource[])dist.toArray(new IResource[dist.size()]);
	}
	
	WebProject getWebProject() throws CoreException {
		return (WebProject)getProject().getNature(WebProject.ID_NATURE);
	}
	
	void collectPublishSources(IContainer folder, ArrayList dist) throws CoreException{
		IResource[] resources = folder.members();
		for (int i = 0; i < resources.length; i++) {
			IResource res = resources[i];
			try {
				if(getWebProject().getPublishDescriptionFactory().create(res) != null){
					dist.add(res);
				}
			} catch (IOException e) {
				WebpubPlugin.handleException(e);
			} catch (IllegalConfigurationException e) {
				WebpubPlugin.handleException(e);
			}
			if(res instanceof IContainer){
				collectPublishSources((IContainer)res, dist);
			}
		}
	}
	
	void cleanBuild(IProgressMonitor monitor) throws CoreException{
		try{
			monitor.beginTask("clean markers", 10000);
			IMarker[] markers = getProject().findMarkers(MARKER_PUBLISH, true, IResource.DEPTH_INFINITE);
			int ratio = 10000 / markers.length;
			for (int i = 0; i < markers.length; i++) {
				markers[i].delete();
				monitor.worked(ratio);
			}
		}finally{
			monitor.done();
		}
	}

	void incrementalBuild(IResourceDelta delta, IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("incremental publish", 1);
			
			final ArrayList publishTargets = new ArrayList();
			delta.accept(new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta delta) throws CoreException {
					int kind = delta.getKind();
					if( kind == IResourceDelta.ADDED || kind == IResourceDelta.CHANGED){
						IResource res = delta.getResource();
						try{
							if(getWebProject().getPublishDescriptionFactory().create(res) != null){
								publishTargets.add(res);
							}
						} catch(IOException e){
							WebpubPlugin.handleException(e);
						} catch (IllegalConfigurationException e) {
							WebpubPlugin.handleException(e);
						}
					}
					return true;
				}
			});
			if( publishTargets.size() > 0){
				doPublish( (IResource[])publishTargets.toArray(new IResource[publishTargets.size()]), new SubProgressMonitor(monitor, 1) );
			}
		}
		finally{
			monitor.done();
		}
	}
	
	void doPublish(IResource[] resources, IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("publishing", resources.length+1);
			
			IProject project = getProject();
			WebProject webProject = getWebProject();
			String encoding = webProject.getString(WebProject.KEY_ENCODING);
			
			File publishTargetFile = distPublishTargetListFile(resources);
			File[] logFiles = webProject.getLogStore().newLogFiles(new String[]{"stderr", "stdout"});
			
			String command =
				publishCommand +
				" -p" + publishersPluginEntryFile +
				" -w" + project.getWorkspace().getRoot().getLocation() +
				" -e" + logFiles[0] +
				" -t" + publishTargetFile;

			// exec ruby from system call
			// invoke stream reader thread thread
			Process process = WebpubPlugin.getDefault().rubyExec(command, project.getLocation());
			
			File unparsedLineFile = logFiles[1];
			PublishResultReader resultReader = new PublishResultReader(process.getInputStream(), encoding, unparsedLineFile);
			resultReader.start();
			
			// loop for waiting publishing process
			// exist loop when both stream is end or canceled form monitor.
			int workingCount = 0;
			while( !(resultReader.isDone()) && !monitor.isCanceled() ){
				Thread.sleep(readSleepMs);
				int count = resultReader.getFinishedCount();
				if( count > workingCount){
					monitor.worked(count - workingCount);
					workingCount = count;
				}
			}
			
			ArrayList errorResources = new ArrayList();
			// addition to task list if error exist.
			String[] errorFilePathes = resultReader.getErrorFiles();
			for (int i = 0; i < errorFilePathes.length; i++) {
				String errorPath = errorFilePathes[i];
				
				// search marker append target resource
				IResource from = getErrorSourceFile(resources, errorPath);
				
				if(from != null){
					RubyException[] exceptions = resultReader.getRubyException(errorPath);
					if(exceptions.length > 0){
						PublishErrorHandler errorHandler = getErrorHandler(from);
						for (int j=0; j<exceptions.length; j++) {
							errorHandler.createMarker(from, exceptions[j]);
						}
						errorResources.add(from);
					}
				}
			}
			
			// delete fixed markers of past build
			for(int i=0; i<resources.length; i++){
				IResource res = resources[i];
				if( !errorResources.contains(res) ){
					deleteMarker(res);
				}
			}
			
			// refresh project publish folder resources.
			webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER).refreshLocal(IResource.DEPTH_INFINITE, new SubProgressMonitor(monitor, 1));
		}
		catch(InterruptedException ex){
			WebpubPlugin.handleException(ex);
			
		}
		catch(IOException ex){
			IPath rubyInterpreter = WebpubPlugin.getDefault().getRubyCommandPath();
			String msg = !rubyInterpreter.toFile().exists() ?
				"Ruby interpreter is not exist: " + rubyInterpreter + " - Please settings in preference page \"Web Publisher > Ruby Interpreter\".":
				ex.getMessage();
			
			throw new CoreException(new Status(IStatus.ERROR, WebpubPlugin.ID_PLUGIN, IStatus.OK, msg, ex));
		}
		finally{
			monitor.done();
		}
	}
	
	/**
	 * get appropriate error handler
	 */
	PublishErrorHandler getErrorHandler(IResource res) throws IOException, CoreException {
		PublishErrorHandler errorHandler = (PublishErrorHandler)errorHandlers.get(null);
		try {
			WebProject webProject = (WebProject)res.getProject().getNature(WebProject.ID_NATURE);
			PublishDescription desc = webProject.getPublishDescriptionFactory().create(res);
			if(desc != null){
				String by = desc.getPublishBy();
				if(errorHandlers.containsKey(by)){
					errorHandler = (PublishErrorHandler)errorHandlers.get(by);
				}
			}
		} catch (IllegalConfigurationException e) {
		}
		return errorHandler;
	}
	
	void deleteMarker(IResource res) throws CoreException {
		IMarker[] markers = res.findMarkers(MARKER_PUBLISH, true, IResource.DEPTH_ZERO);
		for (int i = 0; i < markers.length; i++) {
			IMarker m = markers[i];
			String relateResourcePath = m.getAttribute(ATTR_RELATE_RESOURCE, null);
			if(relateResourcePath != null){
				deleteMarker( res.getWorkspace().getRoot().getFile(new Path(relateResourcePath)) );
			}
			m.delete();
		}
	}
	
	File distPublishTargetListFile(IResource[] resources) throws CoreException{
		File targets = getWebProject().getLogStore().getPublishTargetsLocation();
		
		PrintWriter writer = null;
		try{
			writer = new PrintWriter( new FileOutputStream(targets));
			for (int i = 0; i < resources.length; i++) {
				writer.println(resources[i].getLocation().toString());
			}
			return targets;
		} catch (FileNotFoundException e) {
			throw new RuntimeException(e);
		} finally{
			if(writer != null){
				writer.close();
			}
		}
	}
	
	IResource getErrorSourceFile( IResource[] resources, String resourcePath){
		for (int i = 0; i < resources.length; i++) {
			IResource res = resources[i];
			if( res.getLocation().toString().equals(resourcePath) ){
				return res;
			}
			
			// for cygwin, temporary way:
			// choose relative path from workspace name.
			// e.g, /cygdrive/c/workspace/fooproject/foofile to fooproject/foofile.
			// and choose that file from workspace root.
			IWorkspaceRoot root = res.getWorkspace().getRoot();
			String workspaceDirName = root.getLocation().lastSegment();
			int index = resourcePath.indexOf(workspaceDirName);
			if( index != -1){
				Path path = new Path( resourcePath.substring(index + workspaceDirName.length() ) );
				IFile f = root.getFile(path);
				if( f.exists() ){
					return f;
				}
			}
		}

		return null;
	}	
	

}

