/*
 * $Id: PublishConsoleViewTest.java,v 1.5 2004/04/24 05:04:36 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.tests.views;

import java.io.File;
import java.util.Date;

import junit.framework.TestCase;

import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.jface.action.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.IWorkbenchPage;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.publish.PublishLogStore;
import com.narucy.webpub.core.tests.WebpubTestPlugin;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.tests.UITestUtils;
import com.narucy.webpub.ui.views.PublishConsoleView;

/**
 * <ul>
 * <li>Local menubar display pull down menu that includes action list these item
 * are publish date and project name. Item selected when browse publish log in
 * a central text control. sorted in date.
 * <li>clear log command invoked when clear log files and pull down menu is
 * clear.
 * <li>publish builder invoked when update console view. if console view is
 * hidden, active console.
 * <li>pin control pressed when display pin icon clicked log.
 * </ul>
 */
public class PublishConsoleViewTest extends TestCase {

	IWorkbenchPage page;
	PublishConsoleView view;
	WebProject webProject;

	protected void setUp() throws Exception {
		page = WebpubUIPlugin.getActivePage();
		view = (PublishConsoleView)page.showView("com.narucy.webpub.ui.views.PublishConsoleView");
		view.setFocus();
		
		PublishLogStore.clearAllLogs();
		webProject = WebpubTestPlugin.createTestProject("test_projects/PublishLogStoreTest");
	}
	
	protected void tearDown() throws Exception {
		WebpubTestPlugin.forceDelete(webProject.getProject());
	}
	
	public void testMenuBehavior() throws Exception {
		Shell sh = new Shell();
		try{
			// tests  toolbar items
			IToolBarManager toolBar = view.getViewSite().getActionBars().getToolBarManager();
			IContributionItem[] toolBarItems = toolBar.getItems();
			assertEquals(2, toolBarItems.length);
			
			IAction pinAction = ((ActionContributionItem)toolBarItems[0]).getAction();
			assertEquals("Pin Console", pinAction.getText());
			pinAction.setChecked(false);
			
			IAction logsPulldownMenuAction = ((ActionContributionItem)toolBarItems[1]).getAction();
			assertEquals("Display Selected Logs", logsPulldownMenuAction.getText());
			
			// tests toolbar
			MenuItem[] items = logsPulldownMenuAction.getMenuCreator().getMenu(sh).getItems();
			
			assertEquals(4, items.length);
			assertTrue( items[0].getText().indexOf("stderr") != -1);
			assertTrue( items[1].getText().indexOf("stdout") != -1);
			assertTrue( (items[2].getStyle() & SWT.SEPARATOR) != 0);
			assertTrue( (items[3].getStyle() & SWT.SEPARATOR) == 0);
			assertTrue( items[3].getText().indexOf("Clear History") != -1);
			
			assertEquals(0, view.getText().length());
			assertTrue(view.getTitle().indexOf("not an available") != -1);
			
			items[0].notifyListeners(SWT.Selection, new Event());
			
			assertEquals("ABCDEF", view.getText().split("\\s")[0] );
			assertTrue(view.getTitle().indexOf("stderr") != -1);
			
			items[1].notifyListeners(SWT.Selection, new Event());
			assertEquals("abc", view.getText().split("\\s")[0] );
			assertTrue( view.getTitle().indexOf("out") != -1);
			
			// republish command invoke behavior
			fullBuild();
			
			MenuItem[] newItems = logsPulldownMenuAction.getMenuCreator().getMenu(sh).getItems();
			assertEquals(6, newItems.length);
			
			assertTrue( newItems[0].getText().indexOf("stderr") != -1);
			assertTrue( newItems[1].getText().indexOf("stdout") != -1);
			
			assertEquals(items[0].getText(), newItems[2].getText());
			assertEquals(items[1].getText(), newItems[3].getText());
			
			// clear log
			newItems[5].notifyListeners(SWT.Selection, null);
			
			newItems = logsPulldownMenuAction.getMenuCreator().getMenu(sh).getItems();
			assertEquals(2, newItems.length);
		} finally {
			sh.dispose();
		}
	}
	
	public void testPinAction() throws Exception {
		IContributionItem[] toolBarItems = view.getViewSite().getActionBars().getToolBarManager().getItems();
		IAction pinAction = ((ActionContributionItem)toolBarItems[0]).getAction();
		assertEquals("Pin Console", pinAction.getText());
		
		// browse stderr log on default.
		PublishLogStore logStore = webProject.getLogStore();
		File file = logStore.getLogFiles(logStore.getLogDates()[0] )[1];
		assertTrue(file.getName().indexOf("stdout") != -1);
		
		view.browse(file,  false);
		assertEquals("abc", view.getText().split("\\s")[0] );
		
		// auto refresh on resource chagne
		pinAction.setChecked(false);
		fullBuild();
		
		UITestUtils.whileReadAndDispatch();
		assertEquals("ABCDEF", view.getText().split("\\s")[0] );
		
		// pin action enabled when no change browse log
		view.browse(file,  false);
		assertEquals("abc", view.getText().split("\\s")[0] );
		pinAction.setChecked(true);
		
		assertEquals("abc", view.getText().split("\\s")[0] );
		assertEquals(file, view.getBrowseLogFile());
		
		// browse log button clicked when lastest item
		((ActionContributionItem)toolBarItems[1]).getAction().run();
		Date[] ds = logStore.getLogDates();
		assertEquals( logStore.getLogFiles( ds[ds.length-1] )[0].getName(), view.getBrowseLogFile().getName());
	}
	
	void fullBuild() throws Exception{
		webProject.getProject().build(IncrementalProjectBuilder.FULL_BUILD, null);
		UITestUtils.whileReadAndDispatch();
	}
	
}
