/*
 * $Id: ImgElementWizardPage.java,v 1.3 2004/04/14 03:15:02 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.editors.html.wizards;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.net.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.*;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.w3c.dom.Element;

import com.narucy.webpub.core.*;
import com.narucy.webpub.ui.*;

/**
 * 
 */
public class ImgElementWizardPage extends EditElementWizardPage {
	
	ImageBrowser imgBrowser;
	Text altText, imgWidth, imgHeight, previewImgWidth, previewImgHeight;

	Button browseButton;
	Button fitSizeButton;
	ScrolledComposite imgBrowserBase;
	WebResourceSelectionBlock block;

	ModifyListener modifyListener = new ModifyListener() {
		public void modifyText(ModifyEvent e) {
			refreshPreview();
		}
	};

	protected ImgElementWizardPage() {
		super("ImgElementWizardPage");
		setTitle("Image");
		setDescription("Edit image element for html.");

	}

	protected void createChildControl(Composite parent) throws CoreException {
		GridLayout layout = new GridLayout(4, false);
		layout.verticalSpacing = 9;
		parent.setLayout(layout);
		// imae resource selection
		block = new WebResourceSelectionBlock(getEditFile());
		
		final WebResourceLinkModel model = block.getModel();
		model.addListener( new WebResourceSelectionAdapter(this) {
			public void resourceChanged(IPath path, IResource existResource) {
				refreshPreview();
				if(model.getWebResource() != null){
					refreshImagePreview();
				}
			}
		});

		Control blockControl = block.createControl(parent);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 4;
		blockControl.setLayoutData(gd);

		// alternate text
		Label labe = new Label(parent, SWT.NONE);
		labe.setText("&Alternate Text:");

		altText = new Text(parent, SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		altText.setLayoutData(gd);
		altText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				refreshPreview();
			}
		});
		block.setColumnNameWidth(labe.computeSize(SWT.DEFAULT, SWT.DEFAULT).x);

		// image size.
		Group imgSizeSettings = new Group(parent, SWT.NONE);
		GridLayout imgGroupLayout = new GridLayout(3, false);
		imgGroupLayout.verticalSpacing = 9;
		imgSizeSettings.setLayout(imgGroupLayout);

		gd = new GridData();
		gd.widthHint = 240;
		gd.horizontalSpan = 2;
		gd.verticalAlignment = GridData.BEGINNING;
		imgSizeSettings.setLayoutData(gd);

		imgSizeSettings.setText("Image Size");
		imgWidth = createImageSizeText(imgSizeSettings, "&Width:");
		imgHeight = createImageSizeText(imgSizeSettings, "&Height:");
		previewImgWidth = createImageSizePreviewText(imgSizeSettings, "Preview Width:");
		previewImgHeight = createImageSizePreviewText(imgSizeSettings, "Preview Height:");
		
		gd = (GridData)previewImgHeight.getLayoutData();
		gd.horizontalSpan = 1;
		
		fitSizeButton = new Button(imgSizeSettings, SWT.PUSH);
		fitSizeButton.setLayoutData( new GridData(GridData.HORIZONTAL_ALIGN_END) );
		fitSizeButton.setText("Fit &Size");
		fitSizeButton.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				fitSize();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		// image preview
		Composite imgPreviewBase = new Composite(parent, SWT.NONE);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 2;
		imgPreviewBase.setLayoutData(gd);
		imgPreviewBase.setLayout(new GridLayout(2, true));

		labe = new Label(imgPreviewBase, SWT.NONE);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gd.verticalAlignment = GridData.END;
		labe.setLayoutData(gd);
		labe.setText("Image Preview:");

		Button button = new Button(imgPreviewBase, SWT.PUSH);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_END);
		button.setLayoutData(gd);
		button.setText("Refr&esh");
		button.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				invokeRefresh();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
		});


		imgBrowserBase = new ScrolledComposite(imgPreviewBase, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		imgBrowser = new ImageBrowser(imgBrowserBase);
		imgBrowserBase.setContent(imgBrowser);

		imgBrowserBase.setAlwaysShowScrollBars(false);

		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 2;
		imgBrowserBase.setLayoutData(gd);
		
		// values init
		if (getEditType() == MODIFY){
			Element elem = getParsedSelectionText();
			if(elem != null){
				model.setPath(elem.getAttribute("src"));
				altText.setText( HTMLUtilities.unescape(elem.getAttribute("alt")) );
				imgWidth.setText( elem.getAttribute("width"));
				imgHeight.setText(elem.getAttribute("height"));
			}
		}
		
		if( block.getModel().getWebResource() != null){
			refreshImagePreview();
		}
		
		block.forceFocus();
	}
	
	Text createImageSizeText(Composite base, String label){
		Label labe = new Label(base, SWT.NONE);
		labe.setText(label);
		Text text = new Text(base, SWT.BORDER);
		text.setText("0");
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		text.setLayoutData(gd);
		text.addVerifyListener(new NumVerifyListener());
		text.addModifyListener(modifyListener);
		
		return text;
	}
	
	static Text createImageSizePreviewText(Composite base, String label){
		Label labe = new Label(base, SWT.NONE);
		labe.setText("Preview Width:");
		Text text  = new Text(base, SWT.READ_ONLY);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		text.setLayoutData(gd);
		return text;
	}
	
	void invokeRefresh(){
		// resource is not local resource therefore load from exteral network,
		// call progress monitor dialog.
		if( block.getModel().getWebResource() == null){
			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getShell());
			try {
				dialog.run(true, true, new IRunnableWithProgress() {
					public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
						loadImageAsProgress(monitor);
					}
				});
			} catch (InvocationTargetException e1) {
				WebpubUIPlugin.handleException(e1);
			} catch (InterruptedException e1) {
				WebpubUIPlugin.handleException(e1);
			}
		}else{
			// if local resource, direct load.
			refreshImagePreview();
		}
	}
	
	void fitSize(){
		Point previewSize = getPreviewImaeSize();
		imgWidth.setText(String.valueOf(previewSize.x));
		imgHeight.setText(String.valueOf(previewSize.y));
	}
	
	void loadImageAsProgress(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
		try{
			monitor.beginTask("Load image datas", IProgressMonitor.UNKNOWN);
									
			final boolean[] done = new boolean[]{ false };
			Thread th = new Thread(){
			   public void run(){
				   getShell().getDisplay().syncExec(new Runnable() {
						public void run() {
							refreshImagePreview();
							done[0] = true;
						}
					});
				}
			};
			th.setPriority(Thread.MIN_PRIORITY);
			th.start();
			while( done[0] && !monitor.isCanceled()){
				Thread.sleep(100);
				monitor.worked(1);
			}
			th.join();
		} finally{
			monitor.done();
		}
	}
	
	void refreshImagePreview(){
		Image img = null;
		try {
			IResource res = block.getModel().getWebResource();	
			InputStream stream = null;
			if( res != null && res instanceof IFile){
				res.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor() );
				stream = ((IFile)res).getContents();
			}else{
				stream = new URL( block.getModel().getPath().toString() ).openStream();
			}
			img = imgBrowser.browseImage(stream);
		} catch(ConnectException ex){
			setMessage( ex.getMessage(), WARNING);
		} catch(FileNotFoundException ex){
			setMessage("Specify resource is not found", WARNING);
		} catch(SWTException ex){
			setMessage( ex.getMessage(), WARNING);
		} catch(UnknownHostException ex){
			setMessage( ex.getMessage(), WARNING);
		} catch(MalformedURLException ex){
			setMessage( ex.getMessage(), WARNING);
		} catch (IOException ex) {
			WebpubUIPlugin.handleException(ex);
		} catch (CoreException ex) {
			WebpubUIPlugin.handleException(ex);
		}
		if (img != null) {
			ImageData data = img.getImageData();
			String w = String.valueOf(data.width);
			previewImgWidth.setText(w);
			if( imgWidth.getText().equals("0") ){
				imgWidth.setText(w);
			}
			String h = String.valueOf(data.height);
			previewImgHeight.setText(h);
			if( imgHeight.getText().equals("0")){
				imgHeight.setText(h);
			}
			fitSizeButton.setEnabled(true);
		}else{
			previewImgWidth.setText("");
			previewImgHeight.setText("");
			
			imgBrowser.browseImage(null);
			fitSizeButton.setEnabled(false);
		}
		

		Point p = imgBrowser.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		imgBrowser.setSize(p);
		imgBrowserBase.setMinSize(p.x, p.y);
		imgBrowserBase.redraw(0, 0, p.x, p.y, true);
	}
	
	public String getPreviewText() {
		Point size = getImageSize();
		return
			"<img src=\"" + getImageUrl() + "\"" +
			" alt=\"" + HTMLUtilities.escape(getAltitudeText()) + "\"" +
			" width=\"" + size.x + "\"" +
			" height=\"" + size.y + "\" />";
	}
	
	String getImageUrl(){
		return (block != null && block.getModel().getPath() != null) ? block.getModel().getPath().toString() : "";
	}

	String getAltitudeText(){
		return (altText != null) ? altText.getText() : "";
	}
	
	Point getImageSize(){
		return new Point(
			(imgWidth != null) ? Integer.parseInt( imgWidth.getText()) : 0,
			(imgHeight != null) ? Integer.parseInt( imgHeight.getText()) : 0); 
	}
	
	Point getPreviewImaeSize(){
		return new Point(
			(previewImgWidth != null) ? Integer.parseInt( previewImgWidth.getText()) : 0,
			(previewImgHeight != null) ? Integer.parseInt( previewImgHeight.getText()) : 0); 
		
		
	}

}

class ImageBrowser extends Canvas {

	Image image = null;

	public ImageBrowser(Composite parent) {
		super(parent, SWT.NO_BACKGROUND);

		addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				dispose(e);
			}
		});

		addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				paint(e);
			}
		});
		
		parent.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				disposeImage();
			}
		});
	}

	void dispose(DisposeEvent e) {
		if (image != null) {
			image.dispose();
		}
	}

	public Point computeSize(int wHint, int hHint) {
		if (image == null) {
			return new Point(0, 0);
//			return super.computeSize(wHint, hHint);
		}
		ImageData data = image.getImageData();
		return new Point(data.width, data.height);
	}

	/**
	 * Returns create image, return null if image can not create as specify stream.
	 */
	public synchronized Image browseImage(InputStream stream) {
		Display display = getDisplay();
		disposeImage();
		
		if (stream != null) {
			image = new Image(display, stream);
		}
		
		return image;
	}

	void paint(PaintEvent e) {
		if (image == null) {
			return;
		}

		ImageData data = image.getImageData();
		int w = data.width;
		int h = data.height;

		boolean overWidth = (e.width > w);
		if (overWidth) {
			e.gc.fillRectangle(w, 0, e.width - w, h);
		}

		boolean overHeight = (e.height > h);
		if (overHeight) {
			e.gc.fillRectangle(0, h, w, e.height - h);
			if (overWidth) {
				e.gc.fillRectangle(w, h, e.width - w, e.height - h);
			}
		}
		e.gc.fillRectangle(0, 0, w, h);
		e.gc.drawImage(image, 0, 0, data.width, data.height, 0, 0, w, h);
	}
	
	void disposeImage(){
		if (image != null && !image.isDisposed()) {
			image.dispose();
		}
	}



}