/*
 * $Id: TocController.java,v 1.3 2004/04/14 23:19:11 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.editors.toc;

import java.util.*;
import java.util.regex.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;

import com.narucy.webpub.core.*;
import com.narucy.webpub.core.publish.SourceFileSeeker;
import com.narucy.webpub.core.toc.*;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * Controller for bridge of resources and toc.
 */
public class TocController {

	final static String ATTR_FULL_PATH = "FULL_PATH";

	Toc toc;
	IContainer publishFolder;
	
	public TocController(IContainer pubFolder, Toc toc) {
		this.publishFolder = pubFolder;
		this.toc = toc;
	}

	public Topic[] insertResourceToToc(IResource res, Topic parentTopic) throws CoreException {
		if(res instanceof IContainer){
			return insertContainerToToc((IContainer)res, parentTopic);
		}else{
			return new Topic[]{ insertFileToToc((IFile)res, parentTopic) };
		}
	}

	public Topic[] insertContainerToToc(IContainer res, Topic parentTopic) throws CoreException{
		ArrayList dist = new ArrayList();
		
		IContainer c = (IContainer)res;
		Topic newParentTopic = null;
		
		// find parent resource item.
		IFile parentFile = findDirectoryIndex(c);
		if(parentFile != null){
			newParentTopic = insertFileToToc(parentFile, parentTopic);
			dist.add(newParentTopic);
		}else{
			String fullPath = c.getFullPath().toString();
			Topic[] topics = toc.getAllTopics();
			// search same path item.
			for(int i=0; i<topics.length; i++){
				Topic t = (Topic)topics[i];
				if(fullPath.equals(t.getAttribute(ATTR_FULL_PATH))){
					newParentTopic = t;
					break;
				}
			}
			// create.
			if(newParentTopic == null){
				newParentTopic = parentTopic.createTopic(null, "Group of " + toBaseName(c));
				newParentTopic.setAttribute(ATTR_FULL_PATH, fullPath);
				dist.add(newParentTopic);
			}
		}
		
		// insert child resources to new parent topic
		IResource[] rs = c.members();
		for (int i = 0; i < rs.length; i++) {
			Topic[] added = insertResourceToToc(rs[i], newParentTopic);
			dist.addAll(Arrays.asList(added));
		}
		return (Topic[])dist.toArray(new Topic[dist.size()]);
	}

	public static IFile findDirectoryIndex(IContainer container) throws CoreException{
		WebProject wp = (WebProject)container.getProject().getNature(WebProject.ID_NATURE);
		
		IResource[] rs = container.members();
		String dirBaseName = toBaseName(container);
		for (int i = 0; i < rs.length; i++) {
			IResource r = rs[i];
			if(r instanceof IFile){
				String baseName = toBaseName(r);
				String[] files = wp.getArray(WebProject.KEY_DIRECTORY_INDEX);
				if(Arrays.binarySearch(files, baseName) > 0 || dirBaseName.equals(baseName) ){
					return (IFile)r;
				}
			}
		}
		return null;
	}
	
	public boolean isAlreadyAdded(IResource[] rs){
		return getAlreadyAdded(rs).length == rs.length;
	}
	
	public IResource[] getAlreadyAdded(IResource[] rs){
		ArrayList dist = new ArrayList();
		
		Topic[] topics = toc.getAllTopics();
		for (int i = 0; i < rs.length; i++) {
			IResource r = rs[i];
			boolean file = (r instanceof IFile);
			String p = file ? toHref((IFile)r) : r.getFullPath().toString();
			
			for (int j = 0; j < topics.length; j++) {
				Topic t = (Topic)topics[j];
				if( file ? p.equals(t.getHref()) : p.equals(t.getAttribute(ATTR_FULL_PATH)) ){
					dist.add(r);
					break;
				}
			}
		}
		
		return (IResource[])dist.toArray(new IResource[dist.size()]);
	}

	static String toBaseName(IResource res){
		String name = res.getName();
		int i = name.lastIndexOf('.');
		if(i != -1){
			return name.substring(0, i-1);
		}
		return name;
	}

	String toHref(IFile f){
		return f.getFullPath().removeFirstSegments(publishFolder.getFullPath().segmentCount()).toString();
	}

	public Topic insertFileToToc(IFile f, Topic parentTopic) throws CoreException {
		String p = toHref(f);
		if(toc.findTopic(p) == null){
			return parentTopic.createTopic(p, chooseTitle(f));
		}
		return null;
	}

	static String chooseTitle(IFile f) throws CoreException {
		Pattern pattern = Pattern.compile(".*<title.*>(.+)</title>.*");
		
		IFile from =SourceFileSeeker.findSource(f);
		if(from != null){
			WebProject wp = (WebProject)from.getProject().getNature(WebProject.ID_NATURE);
			String encoding = wp.getString(WebProject.KEY_ENCODING);
			TextReader reader = new TextReader(f.getContents(), encoding);
			try{
				while(reader.hasNext()){
					Matcher m = pattern.matcher(reader.nextLine());
					if(m.matches()){
						return m.group(1);
					}
				}
			}finally{
				if(reader != null){
					reader.close();
				}
			}
		}
		
		return null;
	}
	
	public Topic findNextSelectionTopic(Topic[] removeTopics){
		Topic[] allTopic = toc.getAllTopics();
		for(Topic topic = removeTopics[0]; (topic = topic.getPrevTopic(true)) != null; ){
			if(!isContain(removeTopics, topic)){
				return topic;
			}
		}
		for(Topic topic = removeTopics[0]; (topic = topic.getNextTopic(true)) != null; ){
			if(!isContain(removeTopics, topic)){
				return topic;
			}
		}
		return null;
	}

	public boolean isAllResourcesAdded(){
		IResource[] rs = getAllPublishResources();
		return getAlreadyAdded(rs).length == rs.length;
	}
	
	IResource[] getAllPublishResources(){
		ArrayList dist = new ArrayList();
		try {
			doFindResource(publishFolder, dist);
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
		return (IResource[])dist.toArray(new IResource[dist.size()]);
	}
	
	static void doFindResource(IContainer container, List dist) throws CoreException{
		IResource[] ress = container.members();
		for (int i = 0; i < ress.length; i++) {
			IResource r = ress[i];
			dist.add(r);
			if(r instanceof IContainer){
				doFindResource((IContainer)r, dist);
			}
		}
	}

	static boolean isContain(Topic[] removeTopic, Topic t){
		for (int i = 0; i < removeTopic.length; i++) {
			if( removeTopic[i].equals(t)){
				return true;
			}
		}
		return false;
	}
	
}
