/*
 * $Id: TocEditBlock.java,v 1.4 2004/04/14 23:19:11 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.editors.toc;

import java.util.*;
import java.util.List;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.model.*;

import com.narucy.webpub.core.toc.*;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.views.TocContentProvider;

/**
 * 
 */
public class TocEditBlock {

	Cursor cursor;

	TreeViewer pubFolderTree, tocTree;

	Button
		addButton, removeButton,
		addAllButton, removeAllButton,
		upButton, leftButton, rightButton, downButton,
		createEmptyButton;

	Composite base, buttonBase;

	Toc toc = null;
	IContainer publishFolder;
	TocController controller;
	
	TocListener tocListener = new TocListener() {
		public void topicChanged(TocEvent[] events) {
			base.getDisplay().asyncExec(new Runnable() {
				public void run() {
					if(!base.isDisposed()){
						tocTree.refresh();
						refreshButtonState();
					}
				}
			});
		}
	};

	Listener buttonListener = new Listener() {
		public void handleEvent(Event e) {
			Object[] expanded = tocTree.getExpandedElements();
			
			if(e.widget == addButton)
				handleAddButton();
			else if(e.widget == removeButton)
				handleRemoveButton();
			else if(e.widget == addAllButton)
				handleAddAllButton();
			else if(e.widget == removeAllButton)
				handleRemoveAllButton();
			else if(e.widget == upButton)
				handleUpButton();
			else if(e.widget == leftButton)
				handleLeftButton();
			else if(e.widget == rightButton)
				handleRightButton();
			else if(e.widget == downButton)
				handleDownButton();
			else if(e.widget == createEmptyButton)
				handleCreateEmptyButton();
			
			toc.fireChanged();
			
			// expand node
			Topic[] ts = toc.getAllTopics();
			for (int i = 0; i < ts.length; i++) {
				Topic t = ts[i];
				for (int j=0; j<expanded.length; j++) {
					if(t.equals(expanded[j])){
						tocTree.expandToLevel(t, 1);
					}
				}
			}
		}
	};
	
	public TocEditBlock(Composite parent) {
		base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(3, false));
		
		cursor = new Cursor(base.getDisplay(), SWT.CURSOR_SIZEWE);
		
		new Label(base, SWT.NONE).setText("Resources");
		new Label(base, SWT.NONE);
		new Label(base, SWT.NONE).setText("Sitemap");
		
		// create publish folder tree.
		pubFolderTree = new TreeViewer(base);
		pubFolderTree.getControl().setLayoutData(new GridData(GridData.FILL_VERTICAL));
		pubFolderTree.setLabelProvider(new WorkbenchLabelProvider());
		pubFolderTree.setContentProvider(new WorkbenchContentProvider());
		pubFolderTree.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshButtonState();
			}
		});
		
		// create buttns
		buttonBase = new Composite(base, SWT.NONE);
		GridLayout gl = new GridLayout(1, false);
		gl.marginHeight = 4;
		buttonBase.setLayout(gl);
		
		addButton = createButton(">", "Add");
		removeButton = createButton("<", "Removed");
		addAllButton = createButton(">>", "Add All");
		removeAllButton = createButton("<<", "Remove All");
		
		new Label(buttonBase, SWT.NONE);
		upButton = createButton("&Up");
		leftButton = createButton("&Left");
		rightButton = createButton("&Right");
		downButton = createButton("&Down");

		new Label(buttonBase, SWT.NONE);
		createEmptyButton = createButton("&Create Empty");

		// create toc tree.		
		tocTree = new TreeViewer(base);
		tocTree.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));
		tocTree.setLabelProvider(new WorkbenchLabelProvider(){
			public String decorateText(String text, Object e){
				String href = ((Topic)e).getHref();
				return (href != null) ?
					text + " (" + href + ")" : text;
			}
		});
		tocTree.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshButtonState();
			}
		});
		tocTree.setContentProvider(new TocContentProvider());
		tocTree.getControl().addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				if(toc != null){
					toc.removeTocListener(tocListener);
				}
				cursor.dispose();
			}
		});
		
		base.addMouseTrackListener(new MouseTrackListener() {
			public void mouseEnter(MouseEvent e) {
				mouseHover(e);
			}
			
			public void mouseExit(MouseEvent e) {
				base.setCursor(null);
			}
			
			public void mouseHover(MouseEvent e) {
				base.setCursor(isTreeSeparateSpace(e.x, e.y) ? cursor : null);
			}
			
			boolean isTreeSeparateSpace(int mouseX, int mouseY){
				Control
					pc = pubFolderTree.getControl(),
					tc = tocTree.getControl();
				
				Point
					pl = pc.getLocation(),
					pd = pc.getSize();
				
				int x = pl.x + pd.x;
				int y = pl.y;
				int w = tc.getLocation().x - x;
				int h = pd.y - y;
				
				return x < mouseX && mouseX < x+w && y < mouseY && mouseY < y+h;
			}
			
		});

		base.addMouseListener(new MouseAdapter() {
			int bx;

			public void mouseDown(MouseEvent e) {
				bx = e.x;
			}

			public void mouseUp(MouseEvent e) {
				Control control = pubFolderTree.getControl();
				GridData gd = (GridData)control.getLayoutData();
				if(gd.widthHint == SWT.DEFAULT){
					gd.widthHint = control.getSize().x;
				}
				gd.widthHint += (e.x - bx);
				base.layout();
			}
		});
		
		setEnable(false);
	}
	
	public void setInput(IContainer pubFolder, Toc t){
		if(toc == t || (t != null ? t.equals(toc) : toc.equals(t)) ){
			return;
		}
		
		if(toc != null){
			toc.removeTocListener(tocListener);
		}
		
		publishFolder = pubFolder;
		toc = t;
	
		if(publishFolder != null && toc != null){
			pubFolderTree.setInput(pubFolder);
			tocTree.setInput(toc);
			tocTree.expandAll();
			
			controller = new TocController(pubFolder, toc);
			toc.addTocListener(tocListener);
			
			setEnable(true);
			refreshButtonState();
		}else{
			setEnable(false);
		}
	}
	
	void setEnable(boolean b){
		notifyEnable(base, b);
	}
	
	static void notifyEnable(Composite p, boolean b){
		Control[] children = p.getChildren();
		for (int i = 0; i < children.length; i++) {
			Control control = children[i];
			control.setEnabled(b);
			if(control instanceof Composite){
				notifyEnable((Composite)control, b);
			}
		}
	}
	
	void refreshButtonState(){
		// add, remove button refresh.
		IResource[] resources = getSelectedResources();
		addButton.setEnabled( resources.length == 0 ? false : !controller.isAlreadyAdded(resources) );
		addAllButton.setEnabled( !controller.isAllResourcesAdded() );
		
		Topic[] topics = getSelectedTopics();
		removeButton.setEnabled(topics.length > 0);
		removeAllButton.setEnabled(toc.hasSubtopics());

		boolean selected = topics.length > 0;
		createEmptyButton.setEnabled(selected);
		
		boolean
			up = selected,
			left = selected,
			right = selected,
			down = selected;
		
		if(selected){
			for (int i = 0; i < topics.length; i++) {
				if(topics[i].getNextTopic(false) == null){
					down = false;
				}
				if(topics[i].getPrevTopic(false) == null){
					up = false;
					right = false;
				}
				if( topics[i].getParent() instanceof Toc){
					left = false;
				}
			}
		}
		upButton.setEnabled(up);
		leftButton.setEnabled(left);
		rightButton.setEnabled(right);
		downButton.setEnabled(down);
	}

	Button createButton(String name){
		return createButton(name, null);
	}
	
	Button createButton(String name, String toolTip){
		Button button = new Button(buttonBase, SWT.FLAT);
		button.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_END));
		button.setText(name);
		button.setToolTipText(toolTip);
		button.addListener( SWT.Selection, buttonListener);
		return button;
	}

	IResource[] getSelectedResources(){
		List l = ((IStructuredSelection) pubFolderTree.getSelection()).toList();
		return (IResource[])l.toArray(new IResource[l.size()]);
	}

	Topic[] getSelectedTopics(){
		List l = ((IStructuredSelection)tocTree.getSelection()).toList();
		return (Topic[])l.toArray( new Topic[l.size()]);
	}
	
	void handleCreateEmptyButton() {
		InputDialog dialog = new InputDialog(
			base.getShell(),
			"New Node",
			"Create empty node",
			null,
			null);
		
		if(dialog.open() == Window.OK){
			createEmptyNode(dialog.getValue());
		}
	}

	void createEmptyNode(String name){
		Topic[] ts = getSelectedTopics();
		Topic emptyNode = (ts.length > 0 ? ts[0] : toc).createTopic(null, name);
		tocTree.setSelection(new StructuredSelection(emptyNode));
	}

	void handleDownButton() {
		Topic[] topics = getSelectedTopics();
		for (int i = topics.length-1; i >= 0; i--) {
			topics[i].moveDown();
		}
	}

	void handleLeftButton(){
		Topic[] topics = getSelectedTopics();
		for (int i = 0; i < topics.length; i++) {
			topics[i].levelDown();
		}
	}
	
	void handleRightButton(){
		Topic[] topics = getSelectedTopics();
		for (int i = 0; i < topics.length; i++) {
			topics[i].levelUp();
		}
	}
	
	void handleUpButton() {
		Topic[] topics = getSelectedTopics();
		for (int i = 0; i < topics.length; i++) {
			topics[i].moveUp();
		}
	}

	void handleRemoveAllButton() {
		toc.removeSubtopics();
	}

	void handleRemoveButton() {
		Topic[] removeTopics = getSelectedTopics();
		for(int i=0; i<removeTopics.length; i++){
			Topic t = removeTopics[i];
			t.getParent().removeTopic(t);
		}
		Topic nextSelection = controller.findNextSelectionTopic(removeTopics);
		if(nextSelection != null){
			tocTree.setSelection(new StructuredSelection(nextSelection));
		}
	}

	void handleAddButton() {
		try {
			invokeInsertResourceToToc(getSelectedResources());
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
	}

	void handleAddAllButton() {
		try {
			invokeInsertResourceToToc(publishFolder.members());
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
	}

	void invokeInsertResourceToToc(IResource[] resources) throws CoreException{
		Topic[] topics = getSelectedTopics();
		
		ArrayList added = new ArrayList();
		Topic parent = topics.length > 0 ? topics[0] : toc;
		for (int i = 0; i < resources.length; i++) {
			added.addAll( Arrays.asList(controller.insertResourceToToc(resources[i], parent)) );
		}
		tocTree.setSelection(new StructuredSelection(added));
	}
	
	public void setBackground(Color color){
		base.setBackground(color);
		buttonBase.setBackground(color);
		Control[] controls = base.getChildren();
		for (int i = 0; i < controls.length; i++) {
			Control c = controls[i];
			if(c instanceof Label){
				((Label)c).setBackground(color);
			}
			
		}
	}
	
	public Composite getControl() {
		return base;
	}

	public TreeViewer getTocTree() {
		return tocTree;
	}

}