# = CGIKit - Web Application Framework
#
# == See also
# * www  - http://www.spice-of-life.net/download/cgikit/
# * mail - info@spice-of-life.net
#
# == License
# CGIKit is copyright (C) 2002- SPICE OF LIFE.
# It is a free software distributed under the Ruby license.
#
# == Author
# SUZUKI Tetsuya <suzuki@spice-of-life.net>


require 'stringio'
require 'webrick/httpservlet/abstract'

# Classes under CKAdapter are interface between CGIKit and web servers.
# 
# The role of adapter is to provide the consistent interface to 
# CKApplication. The difference of the relationship between 
# CGIKit and a web server is aboserbed by the obejct. 
module CKAdapter
	class CGI
		attr_accessor :headers, :params
		attr_reader :input, :output, :error

		CR  = "\r"
		LF  = "\n"
		EOL = CR + LF

		def initialize
			@headers = ENV
			@input   = $stdin
			@output  = $stdout
			@error   = $stderr
			_initialize_query
		end

		def create_request
			CKRequest.new( headers, params )
		end

		def create_response
			CKResponse.new headers
		end

		def run( request, response, &block )
			if request then params.update request end
			ckrequest  = create_request
			ckresponse = response || create_response

			if block_given?
				block.call( ckrequest, ckresponse )
			end

			print ckresponse.to_s
		end

		private

		# Copied from cgi.rb.
		# Other interfaces should override _parse* methods.
		def _initialize_query
			if ("POST" == @headers['REQUEST_METHOD']) and
				 %r|\Amultipart/form-data.*boundary=\"?([^\";,]+)\"?|n.match(@headers['CONTENT_TYPE'])
				boundary = $1.dup
				@params = _parse_multipart(boundary, Integer(@headers['CONTENT_LENGTH']))
				query = _parse_get
				@params.udate(CKRequest.parse_query_string(query))
		else
				case @headers['REQUEST_METHOD']
				when "HEAD" then query = _parse_head
				when "GET"  then query = _parse_get
				when "POST" then query = _parse_post
				else             query = _parse_else end
				@params = CKRequest.parse_query_string query
				if @headers['REQUEST_METHOD'] == "POST"
				  @params.update(CKRequest.parse_query_string(_parse_get))
				end
			end
		end

		def _parse_get
			@headers['QUERY_STRING'] or ""
		end

		alias :_parse_head :_parse_get

		def _parse_post
			@input.binmode
			@input.read(Integer(@headers['CONTENT_LENGTH'])) or ''
		end

		def _parse_else
			require "shellwords"

			string = unless ARGV.empty?
				ARGV.join(' ')
			else
				if STDIN.tty?
					STDERR.print(
						%|(offline mode: enter name=value pairs on standard input)\n|
					)
				end
				readlines.join(' ').gsub(/\n/n, '')
			end.gsub(/\\=/n, '%3D').gsub(/\\&/n, '%26')

			words = Shellwords.shellwords(string)

			if words.find{|x| /=/n.match(x) }
				words.join('&')
			else
				words.join('+')
			end
		end

		def _parse_multipart( boundary, content_length )
			params   = Hash.new([])
			boundary = "--" + boundary
			buf      = ""
			bufsize  = 10 * 1024

			# start multipart/form-data
			@input.binmode
			boundary_size   = boundary.size + EOL.size
			content_length -= boundary_size
			status          = @input.read boundary_size
			if status == nil then
				raise EOFError, "no content body"
			elsif boundary + EOL != status
				raise EOFError, "bad content body"
			end

			until content_length == -1
				head = nil
				data = CKByteData.new

				until head and /#{boundary}(?:#{EOL}|--)/n.match(buf)
					if (not head) and /#{EOL}#{EOL}/n.match(buf)
						buf = buf.sub(/\A((?:.|\n)*?#{EOL})#{EOL}/n) do
						head = $1.dup
						""
					end
					next
				end

				if head and ( (EOL + boundary + EOL).size < buf.size )
					data << buf[0 ... (buf.size - (EOL + boundary + EOL).size)]
					buf[0 ... (buf.size - (EOL + boundary + EOL).size)] = ""
				end

				if bufsize < content_length then
					c = @input.read(bufsize) or ''
				else
					c = @input.read(content_length) or ''
				end
				buf += c
				content_length -= c.size
			end

			buf = buf.sub(/\A((?:.|\n)*?)(?:#{EOL})?#{boundary}(#{EOL}|--)/n) do
				data << $1
				if "--" == $2
					content_length = -1
					end
					""
				end

				/Content-Disposition:.* filename="?([^\";]*)"?/ni.match(head)
				filename = ($1 || "").dup
				if /Mac/ni.match(@headers['HTTP_USER_AGENT']) and
					/Mozilla/ni.match(@headers['HTTP_USER_AGENT']) and
					(not /MSIE/ni.match(@headers['HTTP_USER_AGENT']))
					filename = CKUtilities.unescape_url filename
				end
				data.path = filename

				/Content-Type: ([^\r\n]*)/ni.match(head)
				if $1 then
					data.content_type = $1.dup
				else
					data = data.to_s
				end

				/Content-Disposition:.* name="?([^\";]*)"?/ni.match(head)
				name = $1.dup

				if params.has_key? name then
					params[name].push data
				else
					params[name] = [data]
				end
			end

			params
		end
	end


	class ModRuby < CGI
		def run( request, response, &block )
			if request then params.update request end
			ckrequest  = create_request
			ckresponse = response || create_response

			if block_given?
				block.call( ckrequest, ckresponse )
			end

			print ckresponse.status_line
			print ckresponse.to_s
		end
	end


	class Template < CGI
		def initialize
			@headers = ENV
			@input   = StringIO.new
			@output  = StringIO.new
			@error   = StringIO.new
		end

		def run( request, response, &block )
			ckrequest  = request || create_request
			ckresponse = response || create_response

			if block_given?
				block.call( ckrequest, ckresponse )
			end

			ckresponse
		end
	end


	# The adapter and WEBrick::CGIKitServlet are for WEBrick
	# writtend by Ken-ichi HASHIMOTO and Shinya Kawaji. Thanks!
	class WEBrick
		GENERAL_HEADER = %w[
			Cache-Control
			Connection
			Date
			Pragma
			Trailer
			Transfet-Encoding
			Upgrade
			Via
			Warning
		]

		RESPONSE_HEADER = %w[
			Accept-Ranges
			Age
			ETag
			Location
			Proxy-Authenticate
			Retry-After
			Server
			Vary
			WWW-Authenticate
		]

		ENTITY_HEADER = %w[
			Allow
			Content-Encoding
			Content-Language
			Content-Length
			Content-Location
			Content-MD5
			Content-Range
			Content-Disposition
			Expires
			Last-Modified
		]

		LEAVE_HEADER = {}
		(GENERAL_HEADER + RESPONSE_HEADER + ENTITY_HEADER).each{|key|
			LEAVE_HEADER[key] = true
		}

		def run(request, response)
			# set request
			headers = request.meta_vars
			form_values = Hash.new([])
			request.query.each{|key, val|
				form_values[key] = val.list
			}

			ckrequest  = CKRequest.new(headers, form_values)
			ckresponse = CKResponse.new(headers)

			# run
			if block_given?
				yield(ckrequest, ckresponse)
			end

			# set response
			response.request_http_version = ckresponse.http_version
			response.status = ckresponse.status
			ckresponse.headers.each{|key, val|
				if LEAVE_HEADER[key]
					response[key] = val
				end
			}
			response['Content-Type'] = ckresponse.headers['Content-Type']
			if ckresponse.encoding
				response['Content-Type'] << "; charset=#{ckresponse.encoding}"
			end
			ckresponse.cookies.each{|cookie|
				response.cookies << cookie
			}
			response.body = ckresponse.content
		end
	end

end


module WEBrick
	module CGIKitServlet
		class ApplicationHandler < HTTPServlet::AbstractServlet
			def initialize(server, app)
				@app = app
				@app.interface = CKAdapter::WEBrick
			end

			def do_GET(req, res)
				@app.run(req, res)
			end
			alias_method :do_POST, :do_GET
		end

		class PathHandler < ApplicationHandler
			def initialize(server, component_path)
				app = CKApplication.new
				app.component_path = component_path
				super(server, app)
			end
		end

		class HashHandler < ApplicationHandler
			def initialize(server, hash)
				app = CKApplication.new
				hash.each{|key, val|
					app.__send__("#{key}=", val)
				}
				super(server, app)
			end
		end
	end
end


# The super class of CGIKit error classes.
class CKError < StandardError; end


# A class for session management. CKSession objects have a hash of
# arbitary objects and information about browser name, IP address, etc.
# However, you can't set objects that can't be marshal ( IO, Proc, etc. )
# to the session with default database manager CKSessionStore::FileStore.
#
# == Programming Topics
# * SessionManagement[www.spice-of-life.net/download/cgikit/en/userguide/session.html]
class CKSession
	class << self
		def session_id?( session_id )
			if ( session_id == nil ) or ( session_id == "" ) then
				false
			elsif session_id =~ /([0-9A-Za-z]*)/ then
				true
			else
				false
			end
		end

		def create_session_id
			require 'digest/md5'
			md5 = Digest::MD5::new
			md5.update Time.now.to_s
			md5.update rand(0).to_s
			md5.update $$.to_s
			md5.hexdigest[0,16]
		end
	end

	# A hash of arbitary objects.
	attr_accessor :session

	# Session ID.
	attr_accessor :session_id

	# Creation time of the session.
	attr_accessor :creation_time

	# Last accessed time for the session.
	attr_accessor :last_accessed_time

	# Name of browser.
	attr_accessor :user_agent

	# IP address.
	attr_accessor :remote_addr


	def initialize( session_id = nil )
		unless CKSession.session_id? session_id then
			session_id     = CKSession.create_session_id
		end

		@session            = {}
		@session_id         = session_id
		@clear              = false
		@non_access_time    = 0
		@creation_time      = Time.new
		@last_accessed_time = Time.new
	end

	public

	def fetch( key )
		restore unless @session
		@session[key]
	end

	def store( key, value )
		restore unless @session
		@session[key] = value
	end

	alias []  fetch
	alias []= store

	def remove( key )
		@session.delete key
	end

	def clear
		@session.clear
		@clear = true
	end

	# Returns true if "clear" method of the session is called.
	def clear?
		@clear
	end

	# Returns true if the browser is equal to one when the session created.
	def user_agent?( user_agent )
		if @user_agent == user_agent then
			true
		else
			false
		end
	end

	# Returns true if the IP address is equal to one when the session created.
	def remote_addr?( remote_addr )
		if @remote_addr == remote_addr then
			true
		else
			false
		end
	end

	# Returns true if the session isn't expired.
	def timeout?( timeout )
		non_access_time = @last_accessed_time - @creation_time

		if timeout == 0 then
			false
		elsif timeout <= non_access_time then
			true
		else
			false
		end
	end
end


# CKSessionStore is a class for saving session with database manager.
# However this class is for internal use. Use methods of CKApplication to
# manage sessions.
#
# == Customizing database manager
# Database manager objects, such as CKSessionStore::FileStore, save sessions.
# The objects has these 3 methods, implement the methods if you develop or
# customize database manager class.
#
# <b>save</b>::    Saves the session.
# <b>clear</b>::   Clear the session.
# <b>restore</b>:: Returns session restored from the saved. 
#
class CKSessionStore
	attr_accessor :database_manager

	def initialize( application )
		@application      = application
		@database_manager = @application.database_manager.new @application
	end

	def save( session )
		if session.clear? then
			@database_manager.clear session.session_id
		else
			@database_manager.save session
		end
	end

	# Restore a session.
	# Returns a restored session or nil if the session ID is not existed.
	def restore( session_id )
		@database_manager.restore session_id
	end

	def clear( session_id )
		@database_manager.clear session_id
	end


	# A class that saves sessions to files by marshaling.
	class FileStore
		TMPDIR = 'session'

		def initialize( application )
			@application = application
			@tmpdir      = File.join(@application.tmpdir, TMPDIR).untaint
		end

		def save( session )
			unless FileTest.directory? @tmpdir
				require 'ftools'
				File.makedirs @tmpdir
			end

			CKFileLock.exclusive_lock( _tmpfile(session.session_id) ) do | file |
				Marshal.dump( session, file )
			end
		end

		# Restore a session.
		#
		# <b>Returns</b>
		#
		# A restored session or nil if the session ID is not existed.
		def restore( session_id )
			session = nil
			if _exist? session_id then
				CKFileLock.shared_lock( _tmpfile(session_id) ) do | file |
					session = Marshal.load file
				end
				session.last_accessed_time = Time.new
			end
			session
		end

		private
		def _tmpfile( session_id )
			File.join( @tmpdir, session_id ).untaint
		end

		def _exist?( session_id )
			if FileTest.exist? _tmpfile( session_id )
				true
			else
				false
			end
		end

		public
		def clear( session_id )
			if FileTest.exist? _tmpfile( session_id )
				File.delete _tmpfile( session_id )
			end
		end
	end
end


# This class is one of the most important classes in CGIKit.
# CKApplication has two major roles. 
# 
# One role is that CKAplication provides parameters which 
# effect the whole behavior of CGIKit application.
# The other role is that CKApplication communicates with CKAdapter, 
# an interface between a web server and CGIKit.
# 
# When CGIKit receives a request from a client,
# CKApplication arranges its parameters and creates a CKAdapter object.
# Next, CKApplication gets a request object from the CKAdapter object. 
# In reponse to the request object, CKApplication creates 
# a response object and sends it to the CKAdapter object.  
#
# == Application Process
#
# === 1. Ready a request and response object
# First, CKApplication creates a CKAdapter object and gets a request
# ( CKRequest ) object from it. Then, CKApplication creates a response 
# ( CKResponse ) object. 
#
# === 2. Event loop
# Next, CKApplication goes into event loop. In this loop, CKApplication creates 
# components depending on the request and invokes the specified method of 
# the components until it returns an object whose class is not CKComponent.
#
# === 3. Return the response object to the CKAdapter object
# Finally, CKApplication converts the components created in event loop to HTML
# and adds the HTML to the CKResponse object created in 1.
# CKApplication sends the response object to the CKAdapter object.
# Then, the CKAdapter object displays the response object to a client.
# 
# == Programming Topics
# * Architecture[www.spice-of-life.net/download/cgikit/en/userguide/architecture.html]
# * SessionManagement[www.spice-of-life.net/download/cgikit/en/userguide/session.html]
class CKApplication
	class SessionAuthorizationError < CKError; end
	class SessionTimeoutError       < CKError; end

	# Main component. If the element ID  isn't specified, this component is shown.
	# The default value is 'MainPage'.
	attr_accessor :main

	# Locale of an application in a transaction. If the locale is specified,
	# CGIKit change a template for a component. The name of the template
	# includes the component name and the locale name. Also, the template name
	# is seprated by underscore("_"). For example, if the locale is "ja" and
	# the component name is "MainPage", the template name is "MainPage_ja.html".
	attr_accessor :locale

	# Main locale of an application in a transaction. If the master_locale
	# is equal to the specified locale, the components use templates whose name
	# doesn't include the locale name. For instance, if the master_locale is
	# "ja", the locale is "ja" and the component name is "MainPage",
	# the template name is "MainPage.html".
	attr_accessor :master_locale

	# Document root directory.
	attr_accessor :document_root

	# The file system path of the application.
	attr_accessor :path

	# The file system path for components. Components are searched under it.
	attr_accessor :component_path

	# Resource directory. It has files that don't be send to browser.
	attr_accessor :resources

	# Web server resources directory.
	# If "file" attribute of CKImage is set,
	# the element searches an image file from the directory.
	attr_accessor :web_server_resources

	# CKResourceManager object.
	attr_accessor :resource_manager

	# Adapter object.
	attr_accessor :adapter

	# Interface of adapter. The default value is an interface for CGI.
	attr_accessor :interface

	# HTTP request object ( CKRequest ).
	attr_accessor :request

	# HTTP response object ( CKResponse ).
	attr_accessor :response

	# Hash of logging options.
	#
	# level::         Log level. Select in CKLog::DEBUG, CKLog::INFO, CKLog::WARN,
	#                 CKLog::ERROR, CKLog::FATAL.
	# name::          Program name.
	# out::           Outputter.
	# file::          File name to output logs. The attribute has priority over "out".
	# max_file_size:: Max file size to log. Enables if "file" is setted.
	attr_accessor :log_options

	# Name of an error page component to show caught errors.
	attr_accessor :error_page

	# Temporary directory for CGIKit.
	attr_accessor :tmpdir

	# Session object. If session don't exist, creates a new session.
	attr_writer :session

	# Session ID.
	attr_accessor :session_id

	# Session key.
	attr_accessor :session_key

	# Seconds until the session has timed out.
	attr_accessor :timeout

	# Expiry date of cookie for session. If you set the value to nil,
	# session cookies will be invalid when closing browser.
	attr_accessor :session_cookie_expires

	# Enables or disables the use of URLs for storing session IDs.
	attr_accessor :store_in_url

	# Enables or disables the use of cookies for storing session IDs.
	attr_accessor :store_in_cookie

	# Enables or disables automatic session management.
	attr_accessor :manage_session

	# Enables or disables session authorization by browsers.
	attr_accessor :auth_by_user_agent

	# Enables or disables session authorization by IP addresses.
	attr_accessor :auth_by_remote_addr

	# Database manager class saving the session.
	attr_accessor :database_manager

	# Element ID ( CKElementID object ).
	attr_accessor :element_id

	# Session database object ( CKSessionStore ).
	attr_accessor :session_store

	# Character code to convert form data used in CKComponent#canvert_char_code.
	# Select these Japanese codes: 'jis', 'sjis' or 'euc'. If the value is nil,
	# raw form data is substituted for variables. The default value is nil.
	#
	# The attribute and method is setted for Japanese character codes by default.
	# If you convert form data to non-Japanese character codes, override the method.
	attr_accessor :char_code

	# Handled exception.
	attr_reader :error

	VERSION  = '1.2.1'

	class << self
		# Returns version of CGIKit.
		def version
			VERSION
		end
	end

	def initialize
		@path                   = $0
		@component_path         = Dir.pwd
		@main                   = 'MainPage'
		@error_page             = 'CKErrorPage'
		@tmpdir                 = './tmp' || ENV['TMP'] || ENV['TEMP']
		@session_key            = '_session_id'
		@manage_session         = false
		@timeout                = 60 * 60 * 24 * 7
		@session_cookie_expires = 60 * 60 * 24 * 7
		@store_in_url           = true
		@store_in_cookie        = true
		@auth_by_user_agent     = false
		@auth_by_remote_addr    = false
		@database_manager       = CKSessionStore::FileStore
		@log_options            = {}
		@char_code              = nil
		@resources              = './'

		# decides interface of adapter
		if defined?(MOD_RUBY) then
			@interface = CKAdapter::ModRuby
		else
			@interface = CKAdapter::CGI
		end

		CKElement.load_element_file(@component_path, 'CKErrorPage')

		init
	end

	# Hook method to initialize for convenience.
	def init; end

	# Returns the name of the application without file extension.
	def name
		File.basename( @path, '.*' )
	end

	# The application URL based on SCRIPT_NAME with session ID.
	def baseurl( session = false )
		if (session == true) and session? then
			@baseurl + "?#@session_key=#@session_id"
		else
			@baseurl
		end
	end

	# Creates a session.
	def create_session
		session             = CKSession.new
		session.user_agent  = @request.user_agent
		session.remote_addr = @request.remote_addr
		session
	end

	# Returns a restored session objects with the session ID.
	def restore_session( session_id )
		session_store.restore session_id
	end

	# Deletes the session.
	def clear_session( session )
		session.clear
		save_session session
		@session_id = nil
		@session = nil
	end

	# Saves the session, and set a cookie if "store_in_cookie" attribute is
	# setted. If "clear" method of the session is called, the session is deleted.
	def save_session( session )
		if session.clear? then
			if session.session_id == @session_id then
				_clear_session_cookie
			end
		elsif (@session_id == session.session_id) and @store_in_cookie then
			cookie = CKCookie.new(@session_key, @session_id)
			if @session_cookie_expires then
				cookie.expires = Time.new + @session_cookie_expires
			end
			@response.add_cookie cookie
		end

		session_store.save session
	end

	private

	def _clear_session_cookie
		@session_id = nil

		if @store_in_cookie then
			cookie         = CKCookie.new @session_key
			cookie.expires = Time.new - 60
			@response.add_cookie cookie
		end
	end

	public

	# Returns the session object. If the session isn't existed, returns a new session.
	def session
		@session ||= create_session
	end

	# Returns true if the session is existed.
	def session?
		if @session and @session_key and @session_id then
			true
		else
			false
		end
	end

	# Returns true if the locale is master locale.
	def master_locale?
		if @locale == @master_locale then
			true
		else
			false
		end
	end

	# Runs the application.
	# This method calls hook methods "pre_run" and "pre_respond".
	def run( request = nil, response = nil )
		@adapter = create_adapter
		@adapter.run( request, response ) do | ckrequest, ckresponse |
			@request  = ckrequest
			@response = ckresponse

			begin
				# trap generic components
				_ready_request_response
				_ready_session
				pre_action # hook

				unless ckresponse.redirect?
					_event_loop ckresponse
				end
			rescue Exception => e
				@error = e

				# trap custom components' error
				begin
					component = handle_error e
					ckresponse.content = component.to_s
				rescue Exception => e
					component = _default_error_page e
					ckresponse.content = component.to_s
				end
			end

			pre_respond # hook
		end
	end

	# Hook method called before generating specified component.
	# When calling the method, objects of request and session are already setted.
	def pre_action; end

	# Hook method called just before sending the response to browser.
	# When calling the method, a web page converted to HTML code
	# has already setted to the response.
	def pre_respond; end

	private

	def _ready_request_response
		@baseurl       = @request.script_name              unless @baseurl
		@document_root = @request.headers['DOCUMENT_ROOT'] unless @document_root
		@web_server_resources ||= @document_root
		@resource_manager       = CKResourceManager.new self
		unless @request.languages.empty? then
			@locale = @request.languages.first
		end

		if @element_id then
			@target = @element_id.component
		elsif @request['element_id'] then
			@element_id = CKElementID.new @request['element_id'].to_s
			@target     = @element_id.component
		else
			@target = @main.dup
		end
	end

	def _ready_session
		@session_store = CKSessionStore.new self

		# get the session ID.
		cookie    = @request.cookie(@session_key)
		id_cookie = cookie.value if cookie
		id_query  = @request[@session_key]
		unless CKSession.session_id? id_cookie then id_cookie   = nil       end
		unless CKSession.session_id? id_query  then id_query    = nil       end
		if     @store_in_cookie and id_cookie then @session_id = id_cookie
		elsif                       id_query  then @session_id = id_query  end

		# restore the session.
		if @session_id then
			@session = restore_session @session_id
		else
			@session_id = nil
		end

		# ready a new session in automatic session management
		if @manage_session and @session_id.nil? and @session.nil? then
			@session    = create_session
			@session_id = @session.session_id
			save_session @session

			if @store_in_url and (@store_in_cookie == false) then
				@response.set_redirect url(@element_id, @request.form_values)
			end
		end

		# check 1 - session ID and session
		if @session_id and @session.nil? then
			_session_timeout
		end

		# check 2 - authorization and timeout
		if @session then
			if (@auth_by_user_agent and \
			   (not (@session.user_agent? @request.user_agent))) or \
			   (@auth_by_remote_addr and \
			   (not (@session.remote_addr? @request.remote_addr))) then
				raise SessionAuthorizationError, 'Your session is not authorized.'
			elsif (@session.timeout? @timeout) then
				_session_timeout
			end
		end
	end

	def _session_timeout
		if @session then
			clear_session @session
		elsif @session_id then
			_clear_session_cookie
		end

		raise SessionTimeoutError, 'Your session has timed out.'
	end

	def _event_loop( response )
		result = page @target
		result.is_top = true

		begin
			component = result
			result    = component.run

			if @manage_session and @session then
				save_session @session
			end
		end while result.is_a? CKComponent

		# Recreates a main component if the component includes CKPartsMaker.
		if component.is_a? CKPartsMaker
			@target   = component.substitute_page || @main
			component = page @target
			component.run
		end

		# for displaying images
		element = nil
		if @element_id then
			if defs = component.definitions[@element_id.element] then
				element = defs['element']
			end
		end

		if (CKByteData === result) and (element == 'CKImage') then
			raw_mime = component.definitions[@element_id.element]['mime']
			unless raw_mime then
				mime = result.content_type
			else
				mime = component.parse_ckd_value raw_mime
			end
			response.headers['Content-Type'] = mime
			response.content = result.to_s
		else
			response.content = component.to_s
		end
	end

	public

	# Creates an adapter object.
	def create_adapter
		@interface.new
	end

	# Handles every errors and return an error page component.
	def handle_error( error )
		error_page = nil
		if ( error.class == SessionTimeoutError )       or \
		   ( error.class == SessionAuthorizationError ) then
			error_page = handle_session_error error
		end
		unless error_page.is_a? CKComponent then
			error_page       = page @error_page
			error_page.error = error
		end
		error_page
	end

	# Hook method to handle session errors.
	def handle_session_error( error ); end

	private

	# Return a default error page component.
	def _default_error_page( error )
		error_page       = page 'CKErrorPage'
		error_page.error = error
		error_page
	end

	public

	# Creates a specified page component.
	def page( name )
		CKElement.instance( name, self, nil, nil, nil )
	end

	# Creates a URL.
	def url( id = nil, query = nil, secure = false, direct = false )
		protocol = nil
		string   = '?'

		if @secure then
			protocol = 'https://'
		end

		string << "element_id=#{id};"  if id
		string << _query_string(query) if query
		string << _query_for_session   if direct != true
		string.chop!                   if string =~ /&$/

		url = ''
		if protocol then
			url << protocol
			url << @request.server_name if @request.server_name
		end
		url << self.baseurl if self.baseurl
		url << string
	end

	private

	def _query_string( hash )
		str = ''
		hash.each do | key, value |
			if key == @session_key then
				next
			elsif value.is_a? Array then
				value.each do | _value |
					str << "#{key}=#{CKUtilities.escape_url(_value)};"
				end
			else
				str << "#{key}=#{CKUtilities.escape_url(value)};"
			end
		end
		str
	end

	def _query_for_session
		if @store_in_url and @session_id and @session_key then
			"#@session_key=#@session_id;"
		else
			''
		end		
	end

end


# CKElementID is a class for element IDs.
# The object is used to specify an element when substituting form values and
# running an action.
#
# Element ID has 3 elements, these are "component, repetitions, element".
# "repetitions" are definition names enclosed the element with CKRepetition
# and indexes in the CKRepetition elements.
#
# The each elements are splitted by '.'. Definition names and indexes in
# repetitions are splitted by ':'.
# 
# * MainPage component
#  MainPage
#
# * String element of MainPage component
#  MainPage.String
#
# * String element in third repeat for Repetition element as 
#   CKRepetition of MainPage component 
#  MainPage.Repetition:3.String
#
# Component is required. If you specify elements in CKRepetition, 
# repetitions are required.
class CKElementID
	# Name of the component.
	attr_reader :component

	# Array of the repetitions.
	attr_accessor :repetitions

	# Name of the element.
	attr_accessor :element

	def initialize( string = nil )
		@repetitions = []
		if string then
			tokens     = string.split '.'
			@component = tokens.shift
			@element   = tokens.pop

			# parses repetitions
			token = repetition = nil
			unless tokens.empty?
				tokens.each do | token |
					repetition = token.split ':'
					@repetitions << [ repetition.first, repetition.last.to_i ]
				end
			end
		end
	end

	# Returns true if the ID is in repetitions.
	def repetitions?
		if @repetitions.empty? then
			false
		else
			true
		end
	end

	# Executes the block for every definition names and repetition indexes.
	def each
		definition = index = nil
		@repetitions.each do | name, index |
			yield name, index
		end
	end

	# Returns true if a component of the ID and specified are equal.
	def component?( name )
		if @component == name then
			true
		else
			false
		end
	end

	def component=( name )
		@component = name.split('::').last
	end

	# Returns the object as a string.
	def to_s
		str = @component + '.'
		if repetitions? then
			@repetitions.each do | name, index |
				str << "#{name}:#{index}."
			end
		end
		str << @element if @element
		str.chop! if str =~ /\.$/
		str
	end
end


# CKKeyValueCoding provides methods to access instance variables of the object.
# The methods try accessing instance variables by using accessor method.
# If it is failure, its try accessing directly.
#
# If a class method "access_instance_variables?" defines in the class and
# the method returns "true", the direct access is success. Or failure.
module CKKeyValueCoding
	class UnknownKeyError < CKError; end

	# Retrieves value of the instance variable with method chain.
	def retrieve_value( key )
		keypath = key.split '.'
		object  = self
		keypath.each do |path|
			if object.respond_to? path then
				object = object.__send__ path
			elsif _directly?(object) == true then
				object = object.instance_eval "@#{path}"
			else
				_raise_error(object, path)
			end
		end
		object
	end

	# Sets value for the instance variable with method chain.
	def take_value( key, value )
		keypath = key.split '.'
		object  = self
		keypath.each_with_index do |path, index|
			writer = "#{path}="

			if (index + 1) == keypath.size then
				# takes value for key
				if object.respond_to? writer then
					object.__send__(writer, value)
				elsif _directly?(object) == true then
					object.instance_eval "@#{path}=value"
				else
					_raise_error(object, writer)
				end
			else
				# get and set value for the object
				if object.respond_to? path then
					object = object.__send__ path
				elsif _directly?(object) == true then
					object = object.instance_eval "@#{path}"
				else
					_raise_error(object, path)
				end
			end
		end
	end

	alias []  retrieve_value
	alias []= take_value

	private

	def _directly?( object )
		if object.class.respond_to? 'access_instance_variables?' then
			object.class.access_instance_variables?
		else
			false
		end
	end

	def _raise_error( object, key )
		msg =  "This \"#{object.class}\" object does not have a method "
		msg << "\"#{key}\", nor an instance variable \"@#{key.sub(/=$/,'')}\"."
		raise UnknownKeyError, msg
	end

end


# The super class of dynamic element classes.
#
# Dynamic elements convert themselves to HTML.
# These are very important in development with CGIKit.
#
# CKElement returns HTML when to_s() is called.
# In components development, you bind elements to component's methods. 
# At runtime, the binding is executed by run(). 
# These two methods are expected to be overrided in the subclasses of CKElement.
#
# == Paths for searching elements and components
# CKElement objects, including components, are usually instantiated
# by CKElement.instance. This method loads elements/components and creates
# element objects. In this method, some paths are searched to require files.
#
# The searched paths:
# 1. CKApplicaton#component_path
# 2. ($LOAD_PATH)/cgikit/elements
# 3. ($LOAD_PATH)/cgikit/components
#
# The latter two paths are for extention elements or components.
# It is recommended that your own elements or components are installed
# in the 1st path.
#
# == Creation of custom elements
# If you create custom elements, the class of the custom elements must 
# inherit CKElement. Simple dynamic element can be created only 
# by overriding to_s(). to_s() is called when CGIKit
# converts the dynamic elements to HTML. If you add new attributes
# to your custom elements, run() must be overrided. 
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/elements.html]
class CKElement
	class UnknownElementError < CKError; end #:nodoc:
	class AttributeError      < CKError; end #:nodoc:

	attr_accessor :name
	attr_accessor :body
	attr_writer   :definition
	attr_accessor :parent
	attr_accessor :application

	# Array of repetitions.
	attr_accessor :repetitions

	# Index in the last repetition.
	attr_accessor :repetition_index

	class << self
		def instance( element, app, parent, name, body )
			body = '' unless body
			path = load_element_file(app.component_path, element)
			klass = app.class.const_get element
			klass.new(app, parent, name, body, path)
		end

		# Internal method.
		def load_element_file( component_path, element )
			component = "#{component_path}/#{element}/#{element}"

			# check paths for extension elements and components
			ext_element   = "cgikit/elements/#{element}"
			ext_component = "cgikit/components/#{element}/#{element}"
			paths         = [ ext_element, ext_component ]
			$LOAD_PATH.each do | load_path |
				paths.each do | path |
					ext_path = "#{load_path}/#{path}"
					if FileTest.exist? "#{ext_path}.rb".untaint
						require path.untaint
						return ext_path
					else
						next
					end
				end
			end

			# components
			if FileTest.exist? "#{component}.rb".untaint then
				require component.untaint
				return nil
			# starndard elements
			elsif Object.const_defined? element then
				return nil
			end

			raise UnknownElementError, "No such '#{element}' element or component."
		end

		# Internal method that will be overridden in subclasses.
		def bind_request( component, definition, value ); end

		# Return an array of names of the element's own attributes.
		# The value is used to generate optional HTML attributes.
		# Subclasses must override this method.
		def own_attributes
			[]
		end
	end

	# Returns the element ID.
	def element_id
		unless @element_id then
			@element_id             = CKElementID.new
			@element_id.component   = parent.class.to_s
			@element_id.element     = name

			unless repetitions.empty? then
				last_repetition = repetitions.pop
				last_repetition = [ last_repetition.first,repetition_index ]
				repetitions.push last_repetition
			end
			@element_id.repetitions = repetitions || []
		end
		@element_id
	end

	def initialize( app, parent, name, body, path = nil )
		@application      = app
		@parent           = parent
		@name             = name
		@body             = body
		@repetitions      = []
		@repetition_index = 0
	end

	# Fetches the result of the parent component's methods.
	# When is_binding is true, fetch() returns String without binding from . 
	def fetch( key, is_binding = true )
		data = definition[ key ]
		if is_binding == false
			data
		else
			parent.parse_ckd_value( data )
		end
	end

	# Return a definition hash of the element.
	def definition
		unless @definition then
			@definition = parent.definitions[name]
		end
		@definition
	end

	# Return a definition name of the element with the class name.
	def name_with_class
		"\"#{name}\" at #{self.class}"
	end

	# Internal method. This method will be overridden in subclasses.
	def run; end

	# Returns true if the element is a top level.
	def top_level?
		unless @parent then
			true
		else
			false
		end
	end

	# Return a string of values of optional attributes with
	# "other" attribute string excepting the element's own attributes.
	def other_attributes_string
		optional = ''
		each do |key, value|
			if (self.class.own_attributes.include?(key) == false) and \
				(value != nil) and (key != 'other') then
				value = parent.parse_ckd_value value
				optional << " #{key}=\"#{value}\""
			end
		end

		unless other = fetch('other') then
			other = ''
		else
			other = ' ' + other.to_s
		end			

		optional + other
	end

	# Calls block once for each attributes in definition
	# passing the key and value as parameters.
	def each
		definition.each do |key, value|
			unless (key == 'oid') or (key == 'element') then
				yield key, value
			end
		end
	end


	# ElementeAttribute module manages attributes of elements.
	# The module does that following:
	#
	# * add attributes for HTML common attributes to elements 
	# * add attributes for form to elements
	# * add "others" attributes to elements
	# * check attributes of elements
	#
	# == Added HTML common attributes
	# class, id, style, title, dir, lang, onclick, ondblcclick, onmousedown,
	# onmouseup, onmouseover, onmousemove, onmouseout, onkeypress,
	# onkeydown, onkeyup
	#
	# == Added form common attributes
	# enabled
	#
	# == "others" attributes
	#
	module ElementAttribute
		HTML_COMMON_ATTRIBUTES =
			[ 'class', 'id', 'style', 'title', 'dir', 'lang',
			  'onclick',     'ondblcclick', 'onmousedown', 'onmouseup',
			  'onmouseover', 'onmousemove', 'onmouseout',  'onkeypress',
			  'onkeydown',   'onkeyup' ]
		FORM_COMMON_ATTRIBUTES = ['enabled']
		OTHERS_ATTRIBUTE       = ['others']

		# Raises exception if the element has undefined attributes.
		def check_undefined_attributes( *attrs )
			attrs.flatten!
			attrs << 'oid'
			attrs << 'element'

			attr = name = nil
			definition.each_key do | attr |
				if ( attrs.include? attr ) == false then
					raise AttributeError, \
					"Attribute '#{attr}' is undefined attribute - #{name_with_class}"
				end
			end
		end

		# Raises an exception if the element doesn't have required attributes.
		def check_required_attributes( *patterns )
			requires = 0
			attrs    = []
			pattern  = attr = nil

			patterns.each do | pattern |
				pattern.each do | attr |
					requires += 1 if definition.include? attr
				end

				return true if requires == pattern.size
				requires = 0
				attrs << ( "'" + pattern.join(', ') + "'" )
			end

			message =  "#{self.class} requires #{attrs.join(' or ')} "
			message << "of attribute - #{name_with_class}"
			raise AttributeError, message
		end

		# Raises an exception if the element has attributes conflicting with 
		# each other.
		def check_conflicted_attributes( *attributes )
			combination = []
			attr = nil

			attributes.each do | attr |
				if definition.include? attr then
					combination << attr
				end
			end

			if combination.size >= 2 then
				_attr    = nil
				string   = ''
				pop1_str = combination.pop
				pop2_str = combination.pop
	
				combination.each do | _attr |
					string << "\"#{_attr}\", "
				end
				string << "\"#{pop2_str}\" or \"#{pop1_str}\""

				msg = "Exactly one of #{string} must be bound - "
				msg << name_with_class
				raise AttributeError, msg
			end
		end

		# Returns a string with HTML common attributes setted in the element.
		def string_for_html_common_attributes
			string = ''
			attr = value = nil
			HTML_COMMON_ATTRIBUTES.each do | attr |
				value = fetch attr
				if value then
					string << " #{attr}=\"#{value}\"" 
				end
			end
			string
		end

		# Returns a string with form common attributes setted in the element.
		def string_for_form_common_attributes
			string = ''
			attr = value = nil
			FORM_COMMON_ATTRIBUTES.each do | attr |
				value = fetch attr
				if ( attr == 'enabled' ) and ( value == false ) then
					string << " disabled" 
				elsif value and (attr != 'enabled') then
					string << " #{attr}=\"#{value}\"" 
				end
			end
			string
		end

		# Returns an array of names of HTML common attributes.
		def html_common_attributes
			HTML_COMMON_ATTRIBUTES
		end

		# Returns an array of names of form common attributes.
		def form_common_attributes
			FORM_COMMON_ATTRIBUTES
		end
	end


	# ElementState module manages states of elements.
	module ElementState
		class << self
			# Returns a hash parsed a string of state.
			def parse( component, string )
				hash     = {}
				splitted = []

				items = string.split ','
				items.each do | item |
					splitted = item.split ':'
					hash[splitted.first] = component.parse_ckd_value splitted.last
				end
				hash
			end
		end

		# Returns a hash of state for CKConditional.
		def state_for_conditional
			state( 'CKConditional', 'condition', 'negate' )
		end

		# Returns a hash of state.
		def state( element, *attributes )
			value  = nil
			values = []
			query  = {}

			parent.definitions.each do | key, definition |
				if definition['element'] == element then
					id           = CKElementID.new
					id.component = parent.class.to_s
					id.element   = definition['oid']

					values = []
					attributes.each do | attr |
						value = parent.parse_ckd_value( definition[attr] )
						values << "#{attr}:#{value.inspect}"
					end
					query[id.to_s] = values.join ','
				end
			end
			query
		end
	end

end



module CKValidatable
	def validate_with_format_key( key, format_key, pass_key )
		format = parse_ckd_value format_key
		validate(key, format, pass_key)
	end

	def validate( key, format, pass_key )
		qualifier = CKQualifier.format format

		if qualifier.eval? self then
			self[pass_key] = true
		else
			self[pass_key] = false
		end
	end
end


# When you define your component, its class must inherit CKComponent in code
# file. The component dynamically renders HTML with template and binding file. 
#
# A component has 3 files in a directory whose name is the same as the
# component's name. These three files composite Model-View-Controller
# architecture.
#
# Template(View)::      A HTML file includes CGIKit tags ( <cgikit> ). One
#                       CGIKit tag corresponds to one dynamic element.
#
# Binding(Controller):: A definition file for elements is used by a component.
#                       The suffix, ckd, is abbreviation of "CGIKit Definition".
#                       The definition file connects Template to Code.
#
# Code(Model)::         A Ruby source where a component's class is defined. 
#                       It must inherit CKComponent. The name of the component's
#                       class must be the same as the component name. 
#
# ex) An application with a MainPage component
#  /cgi-bin/cgikit-app
#    cgikit-app.cgi
#    /MainPage
#      MainPage.html
#      MainPage.ckd
#      MainPage.rb
#
# == Programming Topics
# * Architecture[www.spice-of-life.net/download/cgikit/en/userguide/architecture.html]
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKComponent < CKElement
	include CKKeyValueCoding, CKValidatable

	class FileNotFoundError < CKError; end #:nodoc:

	# Path for structual files of the component.
	attr_reader :path

	# Hash of definitions in a binding file of the component.	
	attr_accessor :definitions

	attr_accessor :is_top

	class << self
		# Enables accessing instance variables directly with value(), take_value().
		def access_instance_variables?
			true
		end
	end

	def initialize( app, parent, name, body, path = nil )
		super( app, parent, name, body, path )
		@path = path || "#{app.component_path}/#{_component_name}/#{_component_name}"
		@definitions = CKDefinition.parse_ckd_file definition_file
		@is_top = false
		init
	end

	# Hook method for initialization instead of initialize().
	# Form data is not setted when the method is called.
	def init; end

	# Returns a request object of the application.
	def request
		application.request
	end

	# Returns a response object of the application.
	def response
		application.response
	end

	# Returns the session object. If the session isn't existed, returns a new session.
	def session
		application.session
	end

	# Returns a resource manager object.
	def resource_manager
		application.resource_manager
	end

	# Internal method.
	def parse_ckd_value( string )
		object = nil
		if string =~ /\A([^"'].*)\[(.*)\]/ then
			variable = retrieve_value $1
			key      = parse_ckd_value $2
			object   = variable[key]
		elsif string == 'true'   then object = true
		elsif string == 'false'  then object = false
		elsif string == 'nil'    then object = nil
		elsif string =~ /"(.*)"/ then object = $1
		elsif string =~ /'(.*)'/ then object = $1
		elsif ((not string =~ /[\D]/) and (string =~ /[\d]/)) then
			object = string.to_i
		else
			object = retrieve_value string if string
		end

		object
	end

	# Internal method.
	def variable?( string )
		if ( string == 'true' )   or ( string == 'false'  ) or \
		   ( string == 'nil'  )   or ( string =~ /"(.*)"/ ) or \
		   ( string =~ /'(.*)'/ ) or \
		   ( ( not string =~ /[\D]/ ) and ( string =~ /[\d]/ ) ) or \
			( self.respond_to?("#{string}=") == false ) then
			false
		else
			true
		end
	end

	# Returns name of the template file.
	def template_file
		if application.master_locale? then
			"#{path}.html"
		else
			file = "#{path}_#{application.locale}.html".untaint
			if FileTest.exist? file then
				return file
			end

			unless request.languages.empty? then
				request.languages.each do |lang|
					file = "#{path}_#{lang}.html".untaint
					if FileTest.exist? file then
						return file
					end
				end
			end

			"#{path}.html"
		end
	end

	# Returns body of the template file as string.
	def template_string
		begin
			f = File.new template_file.untaint
			@template_string = f.read
			f.close
		rescue
			msg = "Can't read a template file - #{template_file}"
			raise FileNotFoundError, msg
		end

		@template_string
	end

	# Returns name of the definition( binding ) file.
	def definition_file
		"#{path}.ckd".untaint
	end

	# Returns body of the definition( binding ) file as string.
	def definition_string
		unless @definition_string then
			begin
				f = File.new definition_file
				@definition_string = f.readlines.join
				f.close
			rescue
				msg = "Can't read a binding file - #{definition_file} #{self.class}"
				raise FileNotFoundError, msg
			end
		end

		@definition_string
	end

	# Creates a specified page component.
	def page( page )
		CKElement.instance( page, application, parent, nil, nil )
	end

	# Invokes the action after binding component's variables to attributes
	# for definitions, values from request to component's variables.
	def run
		_bind_attribute
		_bind_request

		if is_top then
			pre_action
			result = _invoke_action
			post_action
		end

		result
	end

	private

	# Binds attribute setted in parent component to self.
	# This method is for nested components.
	def _bind_attribute
		if parent and definition then
			defaults = { "name" => true, "oid" => true, "element" => true }
			definition.each do | key, value |
				unless defaults[key] then
					take_value( key, parent.parse_ckd_value(value) )
				end
			end
		end
	end

	def _bind_request
		id = key = value = klass = klass_name = list_name = item_name = nil

		application.request.form_values.sort.each do | key, value |
			id = CKElementID.new key.to_s
			next unless _component_name == id.component

			if definitions[id.element] then
				klass_name  = definitions[id.element]['element']
			else
				# for radio buttons of CKRadioButton and CKGenericElement
				def_name, attr_value = value.last.split( '.', 2 )
				klass_name           = definitions[def_name]['element']
				id.element           = def_name
				value                = attr_value
			end

			_bind_item_in_repetitions id
			value = convert_char_code value

			CKElement.load_element_file( application.component_path, klass_name )
			klass = Object.const_get klass_name
			klass.bind_request( self, definitions[id.element], value, id )
		end
	end

	def _bind_item_in_repetitions( id )
		rep_name = index = list_name = item_name = list = nil

		# check repetitions reflexively
		if id.repetitions? then
			id.each do | rep_name, index |
				if definitions.include? rep_name then
					list_name = definitions[rep_name]['list']
					item_name = definitions[rep_name]['item']
				end

				if list_name and item_name then
					list = retrieve_value list_name

					if list then
						list.each_with_index do |_item, _index|
							if index == _index then
								take_value(item_name, _item)
								break
							end
						end
					end
				end
			end
		end
	end

	public

	# Hook method to convert character code for the form values.
	def convert_char_code( values )
		if code = application.char_code then
			require 'kconv'

			case code.downcase
			when 'jis'  then kcode = Kconv::JIS
			when 'sjis' then kcode = Kconv::SJIS
			when 'euc'  then kcode = Kconv::EUC
			else
				return values
			end

			k_value = []
			values.each do |value|
				if String === value then
					k_value << Kconv.kconv(value, kcode)
				else
					k_value << value
				end
			end
			values = k_value
		end
		values
	end

	private

	def _invoke_action
		result = nil
		if id = application.element_id then
			_bind_item_in_repetitions id

			# get action
			def_action = nil
			if id.component == _component_name then
				definitions.each_value do | definition |
					if definition['oid'] == id.element then
						case definition['element']
						when 'CKGenericElement' then
							def_action = 'invoke_action'
						when 'CKFrame' then
							def_action = 'value'
						when 'CKImage' then
							def_action = 'data'
							mime = parse_ckd_value definition['mime']
							response.headers['Content-Type'] = mime
						else
							def_action = 'action'
						end

						@action = definition[def_action]
					end
				end

				result = retrieve_value @action if @action
			end
		end

		result
	end

	# Returns name of the component in name space package.
	def _component_name
		self.class.to_s.split('::').last
	end

	public

	# Hook method called after setting form data, before invoking action.
	def pre_action; end

	# Hook method called after run() that are setting form data and invoking action.
	def post_action; end

	# Converts the component to HTML.
	def to_s
		parser = CKHTMLParser.new( self, template_string )
		parser.parse
	end
end

# CKString shows the result of binding as string.
#
# == Bindings
# Required attributes: <b>value</b>
#
# <b>value</b>::  Text to be displayed.   
# <b>escape</b>:: Escapes HTML control characters in "value" if the "escape"
#                 is true. The default value is true.
# <b>empty</b>::  Text that is substituted for "value"
#                 when the "value" is nil.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKString < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['value', 'escape', 'empty']
		end
	end

	def run	
		@attr_value  = fetch( 'value' )  # String
		@attr_escape = fetch( 'escape' ) # true or false
		@attr_empty  = fetch( 'empty' )  # String
		if @attr_escape == nil then
			@attr_escape = true
		end

		check_required_attributes( ['value'], ['empty'] )
	end

	def to_s
		to_s = nil

		if @attr_value then
			to_s = @attr_value
		elsif @attr_empty then
			to_s = @attr_empty
		end

		if to_s and (@attr_escape == true) then
			to_s = CKUtilities.escape_html to_s
		end

		to_s
	end
end


# CKHyperlink generates a hypertext link.
#
# == Bindings
# Required attributes: <b>action</b>, <b>href</b> or <b>page</b>.
#
# <b>action</b>::  Method to be invoked when the link is clicked.
# <b>enabled</b>:: Generates a non-active link if the value is false.
# <b>href</b>::    You specify the URL to other web page directly.
#                  This attribute prevails over "action" or "page" attribute.
# <b>page</b>::    Name of component to display when the link is clicked.
# <b>string</b>::  Text of the link. If the body of CKHyperlink tag is
#                  not empty, the body is displayed. For example,
#                  if the template includes "<cgikit name=link>foo</cgikit>",
#                  this element shows "foo" as the link.
# <b>target</b>::  target attribute of HTML's <a> tag. 
# <b>secure</b>::  Appends "https://" to the URL if the value is true.
#                  The default value is false.
# <b>query</b>::   Hash as the query string. The value of "query" attribute is
#                  converted to string. Then, the string is added to append
#                  the URL.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKHyperlink < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['action', 'enabled', 'href', 'page', 'string', 'target',
			 'secure', 'query']
		end
	end

	def run
		@attr_action  = fetch( 'action', false ) # CKComponent
		@attr_enabled = fetch( 'enabled' )       # true or false
		@attr_href    = fetch( 'href' )          # String
		@attr_page    = fetch( 'page' )          # String
		@attr_string  = fetch( 'string' )        # String
		@attr_target  = fetch( 'target' )        # String
		@attr_secure  = fetch( 'secure' )        # true or false
		@attr_query   = fetch( 'query' ) || {}   # Hash

		if @attr_string then
			@attr_string = @attr_string.to_s
		end

		check_required_attributes( ['action'], ['href'], ['page'] )
		check_conflicted_attributes( 'action', 'href', 'page' )
	end

	def to_s
		container = nil
		unless body.empty? then
			parser                  = CKHTMLParser.new( parent, body )
			parser.repetitions      = @repetitions
			parser.repetition_index = @repetition_index
			container               = parser.parse
		end

		# action
		if    @attr_page   then id = CKElementID.new @attr_page
		elsif @attr_action then id = element_id
		else                    id = nil end
		@attr_action = application.url( id, @attr_query, @attr_secure )

		# to_s
		to_s = '<a'
		if   @attr_href    then to_s << " href=\"#@attr_href\""
		else                    to_s << " href=\"#@attr_action\""   end
		if   @attr_target  then to_s << " target=\"#@attr_target\"" end
		to_s << other_attributes_string
		to_s << '>'
		if    @attr_string then to_s << @attr_string
		elsif container    then to_s << container end
		to_s << '</a>'
		
		if @attr_enabled == false then
			to_s = @attr_string
		end
		if to_s.nil? || to_s.length == 0 then
			to_s = container
		end

		to_s
	end
end


# Creates an image tag.
#
# == Bindings
# Required attributes: <b>file</b>, <b>src</b>, or <b>data</b>.
#
# <b>alt</b>::    Alternative text to the picture.
# <b>border</b>:: Size of image border.
# <b>width</b>::  Width of image.
# <b>height</b>:: Height of image.
# <b>file</b>::   Name of an image file in web server resource directory.
# <b>src</b>::    You specify an image file directly. This value prevails 
#                 over "file" attribute.
# <b>data</b>::   A CKByteData object to display as image.
#                 If you create the object without resource manager,
#                 You must use this with "mime" attribute.
# <b>mime</b>::   MIME type for a resource of "data" attribute.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKImage < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['alt', 'border', 'width', 'height', 'file', 'src', 'data', 'mime']
		end
	end

	def run
		@attr_alt    = fetch( 'alt' )         # String
		@attr_border = fetch( 'border' )      # Integer
		@attr_width  = fetch( 'width' )       # Integer
		@attr_height = fetch( 'height' )      # Integer
		@attr_file   = fetch( 'file' )        # String
		@attr_src    = fetch( 'src' )         # String
		@attr_data   = fetch( 'data', false ) # String

		check_required_attributes( ['file'], ['src'], ['data','mime'] )
		check_conflicted_attributes( 'file', 'src', 'data' )
	end

	def to_s
		if @attr_file then
			@attr_src = application.resource_manager.url @attr_file
		elsif @attr_data then
			@attr_src = application.url element_id
		end

		to_s = "<img src=\"#@attr_src\" alt=\"#@attr_alt\""
		if @attr_border then to_s << " border=\"#@attr_border\"" end
		if @attr_width  then to_s << " width=\"#@attr_width\"" end
		if @attr_height then to_s << " height=\"#@attr_height\"" end
		to_s << other_attributes_string
		to_s << '>'
	end
end


# Controls generating HTML.
#
# == Bindings
# Required attribute: <b>condition</b>
#
# <b>condition</b>:: If the value is true and "negate" is false, the body of the 
#                    CKCoditional tag is displayed.
# <b>negate</b>::    Inverts the meaning of the "condition".
#
# == Control table
#   condition  negate  result
#   true       false   show
#   false      false   not show
#   true       true    not show
#   false      false   show
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKConditional < CKElement
	include ElementAttribute, ElementState

	class << self
		def bind_request( component, definition, value, id )
			parsed = CKElement::ElementState.parse( component, value.first )

			# change if the definition is variable
			if component.variable? definition['condition'] then
				component.take_value( definition['condition'], parsed['condition'])
			end
			if component.variable? definition['negate'] then
				component.take_value( definition['negate'], parsed['negate'] )
			end
		end

		def own_attributes
			['condition', 'negate']
		end
	end

	def run
		@attr_condition = fetch( 'condition' )
		@attr_negate    = fetch( 'negate' )

		check_required_attributes( ['condition'] )
	end

	def to_s
		to_s = ''
		if ( ( ( not @attr_negate    ) and @attr_condition ) or \
		     ( ( not @attr_condition ) and @attr_negate    ) ) then
			parser                  = CKHTMLParser.new( parent, body )
			parser.repetitions      = @repetitions
			parser.repetition_index = @repetition_index
			to_s = parser.parse
		end
		to_s
	end
end


# A CKRepeition object repeats its contents. 
#
# == Bindings
# Required attributes: <b>list</b> and <b>item</b>, or <b>count</b>
#
# <b>count</b>:: CKRepeition repeats its contents this number of times.
#                The attribute conflicts with "list".
# <b>list</b>::  Array which is iterated through.
# <b>item</b>::  Current item when the list is iterated through.
# <b>index</b>:: Index of the current item.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKRepetition < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['count', 'list', 'item', 'index']
		end
	end

	def run
		@attr_count = fetch( 'count' )        # Integer
		@attr_item  = fetch( 'item', false )
		@attr_list  = fetch( 'list' )         # Enumerable
		@attr_index = fetch( 'index', false )

		@repetitions << [ @name, @repetition_index ]

		check_required_attributes( ['list', 'item'], ['count'] )
		check_conflicted_attributes( 'list', 'count' )
	end

 	def to_s
 		to_s  = ''

 		if @attr_list then
			@attr_list.each_with_index do | item, index |
				parent.take_value( @attr_item, item )
				to_s << _iteration( index )
			end
		elsif @attr_count then
			@attr_count.times do | index |
				to_s << _iteration( index )
 			end
 		end
 		to_s
 	end

	private
	def _iteration( index )
		parent.take_value( @attr_index, index ) if @attr_index
		parser                  = CKHTMLParser.new( parent, body )
		parser.repetitions      = @repetitions
		parser.repetition_index = index
		parser.parse
	end
end


# CKContent is used in nested components.
# This element tag in the template is replaced
# with a part of the template of its grandparent component.
#
# == Bindings
# CKContent doesn't have any attribute.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKContent < CKElement
	include ElementAttribute

	def to_s
		unless parent.parent
			return
		end

		parser = CKHTMLParser.new( parent.parent, parent.body )
		parser.parse
	end
end


# CKFrame generates frame tag in HTML.
#
# == Bindings
# Required attributes: <b>page</b>, <b>src</b> or <b>value</b>.
#
# The priorities of "src", "page" and "value" is "src", 
# "page", "value". 
# 
# <b>name</b>::  "name" attribute of HTML's <frame> tag.
# <b>page</b>::  Name of component that supplies the content
#                for the frame.
# <b>src</b>::   You specify the URL or file for the frame. 
# <b>value</b>:: Method that supplies the content. The parent of this element
#                must have the specified method.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKFrame < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['name', 'page', 'src', 'value']
		end
	end

	def run
		@attr_name  = fetch( 'name' )  # String
		@attr_page  = fetch( 'page' )  # String
		@attr_src   = fetch( 'src' )   # String
		@attr_value = fetch( 'value' ) # CKComponent

		check_required_attributes( ['page'], ['src'], ['value'] )
		check_conflicted_attributes( 'page', 'src', 'value' )
	end

	def to_s
		if @attr_src then
		elsif @attr_page then
			id = CKElementID.new @attr_page
			@attr_src = application.url id
		elsif @attr_value then
			@attr_src = application.url element_id
		end

		to_s =  "<frame name=\"#@attr_name\" src=\"#@attr_src\""
		to_s << other_attributes_string
		to_s << ">"
	end
end


# CKGenericElement generates generic HTML tags.
#
# == Bindings
#
# Required attributes: <b>tag</b>
#
# <b>tag</b>::           Name of the HTML tag. If the attribute is nil,
#                        body enclosed by the element or "string"
#                        attribute are displayed.
# <b>enabled</b>::       Enables or disables the tag. If the attribute is false,
#                        body enclosed by the element or "string"
#                        attribute are displayed.
# <b>string</b>::        String to display if body enclosed by the element
#                        isn't exist.
# <b>option</b>::        String to append for the open tag. For example,
#                        "checked" or "selected".
# <b>form_value</b>::    If the element is form, the attribute is setted
#                        form datas as a string.
# <b>form_values</b>::   If the element is form, the attribute is setted
#                        form datas as an array.
# <b>invoke_action</b>:: If the element is executable ( hyperlink, button,
#                        etc. ), the method is called when clicked.
#
# You can define other voluntary attributes.
# The attributes is appended to the tag in format as "attribute=value".
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKGenericElement < CKElement
	include ElementAttribute

	NO_CLOSE_TAGS = [ 'area', 'base', 'basefont', 'br', 'col', 'frame', 'hr',
	                  'img', 'input', 'link', 'map', 'meta', 'param' ]

	class << self
		def bind_request( component, definition, value, id )
			if definition['invoke_action'] then
				component.application.element_id = id
			end

			if action = definition['form_value'] then
				if id.repetitions? then
					list          = component.retrieve_value action
					list_id       = id.repetitions.last.last
					list[list_id] = value.first
				else
					component.take_value( action, value.first )
				end
			end

			if action = definition['form_values'] then
				if id.repetitions? then
					list          = component.retrieve_value action
					list_id       = id.repetitions.last.last
					list[list_id] = value
				else
					component.take_value( action, value )
				end
			end
		end

		def own_attributes
			['tag', 'enabled', 'string', 'option', 'form_value',
			 'form_values', 'invoke_action']
		end
	end

	def run
		@attr_name          = element_id                      # String
		@attr_tag           = fetch( 'tag' )                  # String
		@attr_enabled       = fetch( 'enabled' ) == false ? false : true # true/false
		@attr_string        = fetch( 'string' )               # String
		@attr_value         = fetch( 'value' )                # String
		@attr_form_value    = fetch( 'form_value' )           # String
		@attr_form_values   = fetch( 'form_values' )          # String
		@attr_invoke_action = fetch( 'invoke_action', false ) # String
		@attr_option        = fetch( 'option' )               # String

		check_required_attributes( ['tag'] )
	end

	def to_s
		string = nil
		if @attr_string then
			string = @attr_string
		elsif body.empty? == false then
			parser                  = CKHTMLParser.new( parent, body )
			parser.repetitions      = @repetitions
			parser.repetition_index = @repetition_index
			container               = parser.parse

			if container then string = container
			else              string = @attr_string end
		end

		to_s = value = href = nil
		if @attr_enabled == true then
			# hyperlink, <form> or <a>
			if    @attr_invoke_action   then href = application.url element_id
			elsif _href = fetch('href') then href = _href end

			# value
			if    @attr_value       then value = @attr_value
			elsif @attr_form_value  then value = @attr_form_value
			elsif @attr_form_values then
				value = @attr_form_values[@repetition_index]
			end

			# radiobutton
			if (@attr_tag.downcase == 'input') and \
				(fetch('type').downcase == 'radio') and (_name = fetch('name')) then
				@attr_name.element = _name
				if value == nil then value = 'on' end
				value = "#{definition['oid']}.#{value}"
			end

			to_s =  "<#@attr_tag name=\"#@attr_name\""
			to_s << " value=\"#{value}\"" if value
			to_s << " href=\"#{href}\""   if href
			to_s << " #@attr_option"      if @attr_option
			to_s << other_attributes_string
			to_s << '>'
			to_s << string if string

			unless NO_CLOSE_TAGS.include? @attr_tag then
				to_s << "</#@attr_tag>"
			end
		else
			to_s = string
		end

		to_s
	end
end


# Creates a fill-in form.
# Dynamic elements of form, for example, CKBrowser, CKCheckbox, CKRadioButton,
# CKPopUpButton, CKText, CKTextField, CKSubmitButton, CKResetButton and
# CKFileUpload are used within CKForm or HTML form.
#
# == Bindings
# Required attributes: none
#
# <b>method</b>::     Encode type to send form data.
#                     You can use "POST" or "GET" as the value.
# <b>enctype</b>::    Encode type for form data. When you use CKFileUpload,
#                     set this attribute to "multipart/form-data".
# <b>fileupload</b>:: If you set this attribute to true,
#                     "enctype" attribute is setted to "multipart/form-data".
#                     You can use this instead of "enctype" when using CKFileUpload.
# <b>href</b>::       URL to which the browser directs.
# <b>target</b>::     Frame in a frameset that receive the page.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKForm < CKElement
	include ElementState, ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			component.application.element_id = id
		end

		def own_attributes
			['method', 'enctype', 'fileupload', 'href', 'target']
		end
	end

	def run
		@attr_method     = fetch('method') || 'post'              # String
		@attr_enctype    = fetch 'enctype'                        # String
		@attr_href       = fetch('href')   || application.baseurl # String
		@attr_target     = fetch 'target'                         # String
		@attr_query      = fetch('query')  || {}                  # Hash
		@attr_fileupload = fetch 'fileupload'                     # true / false

		unless ( @attr_method.downcase == 'post' ) or \
		       ( @attr_method.downcase == 'get'  ) then
			@attr_method = 'post'
		end

		if @attr_fileupload == true then
			@attr_enctype = "multipart/form-data"
		end

		@attr_query.update state_for_conditional
	end

	def to_s
		parser               = CKHTMLParser.new( parent, body )
		value                = parser.parse
		element_id           = CKElementID.new
		element_id.component = parent.class.to_s
		element_id.element   = name
		
		to_s =  "<form method=\"#@attr_method\""
		to_s << " action=\"#@attr_href\""     if @attr_href
		to_s << " target=\"#@attr_target\""   if @attr_target
		to_s << " enctype=\"#@attr_enctype\"" if @attr_enctype
		to_s << other_attributes_string
		to_s << ">\n<div>\n"
		to_s << "<input type=\"hidden\" name=\"element_id\" "
		to_s << "value=\"#{element_id}\">\n"

		if application.session? then
			to_s << "<input type=\"hidden\" "
			to_s << "name=\"#{application.session_key}\" "
			to_s << "value=\"#{application.session_id}\">\n"
		end
		if @attr_query then
			to_s << _create_hidden_field( @attr_query )
		end

		to_s << "</div>\n#{value}\n</form>\n"
	end

	private

	def _create_hidden_field( query )
		fields = ''
		query.each do | key, value |
			fields << \
				"<input type=\"hidden\" name=\"#{key}\" value=\"#{value}\">\n"
		end
		fields
	end
end


# Creates a list whose multiple items can be selected.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>list</b>
#
# <b>escape</b>::   Escapes HTML control characters in "value"
#                   if the "escape" is true. The default value is true.
# <b>list</b>::     Array which is iterated through.
# <b>values</b>::   Array which is value for each "value" attributes of
#                   <option> elements.
# <b>selected</b>:: Items which are chosen from the list.
# <b>multiple</b>:: Multiple items of the list can be selected
#                   if the value is true.
# <b>size</b>::     Size of item in appearance.
# <b>enabled</b>::  If the value is false, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKBrowser < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if component.parse_ckd_value definition['escape'] then
				value.each do | _value |
					_value = CKUtilities.unescape_html _value
				end
			end

			# set value as Array
			component.take_value( definition['selected'], value )
		end

		def own_attributes
			['escape', 'list', 'values', 'selected', 'multiple', 'size', 'enabled']
		end
	end

	def run
		@attr_name     = element_id
		@attr_escape   = fetch( 'escape' )
		@attr_list     = fetch( 'list' )
		@attr_values   = fetch( 'values' )
		@attr_selected = fetch( 'selected' ) || []
		@attr_multiple = fetch( 'multiple' )
		@attr_size     = fetch( 'size' )

		@attr_escape = true if @attr_escape.nil?

		check_required_attributes( ['list'] )
	end

	def to_s
		to_s = '<select'
		if @attr_name     then to_s << " name=\"#@attr_name\"" end
		if @attr_size     then to_s << " size=\"#@attr_size\"" end
		if @attr_multiple then to_s << " multiple" end
		to_s << other_attributes_string
		to_s << ">\n"

		if @attr_list then
			item = index = value = nil
			@attr_list.each_with_index do | item, index |
				to_s << "<option"
				if (@attr_selected == item) or @attr_selected.include?(item) then
					to_s << " selected"
				end
				if @attr_values then
					value = @attr_values[index]
					if (@attr_selected == value) or @attr_selected.include?(value) then
						to_s << " selected"
					end
					if @attr_escape then
						value = CKUtilities.escape_html value
					end
					to_s << " value=\"#{value}\""
				end
				to_s << ">"

				if @attr_escape then
					item = CKUtilities.escape_html item
				end
				to_s << "#{item}</option>\n"
			end
		end

		to_s << '</select>'
	end
end


# Creates a checkbox.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attributes: <b>selection</b> and <b>value</b>, or <b>checked</b>
#
# You use this element in two ways. One it the way to use "checked" attribute. 
# The other is the way to use both "selection" and "value" attributes. 
#
# <b>checked</b>::   If neither "value" nor "selection" attribute is nil and
#                    the value of "selection" is equal to that of "value",
#                    the check box is checked.
# <b>value</b>::     When the check box is checked,
#                    the value of "value" attribute is set to the component by
#                    the method specified by "selection" attribute.
# <b>selection</b>:: Object that the user chose from the check box.
# <b>enabled</b>::   If the value is false, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKCheckbox < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if definition['checked']
				action = definition['checked']
				value  = true
			else
				action = definition['selection']
				value  = value.last
			end
			component.take_value( action, value )
		end

		def own_attributes
			['checked', 'value', 'selection', 'enabled']
		end
	end

	def run
		@attr_name      = element_id.to_s
		@attr_checked   = fetch( 'checked' )   # true or false
		@attr_value     = fetch( 'value' )     # String
		@attr_selection = fetch( 'selection' ) # String
		if ( ( @attr_value and @attr_selection ) and \
		     ( @attr_value ==  @attr_selection ) ) then
			@attr_checked = true
		end

		if application.request.form_values[element_id.to_s].empty? then
			@attr_checked = false
			if attr_name = fetch('checked', false) then
				parent.take_value(attr_name, false)
			end
		end

		check_required_attributes( ['selection','value'], ['checked'] )
		check_conflicted_attributes( 'selection', 'checked' )
		check_conflicted_attributes( 'value', 'checked' )
	end

	def to_s
		to_s =  "<input type=\"checkbox\" name=\"#@attr_name\""
		to_s << %Q' value="#{@attr_value || 1}"'
		if @attr_checked then to_s << " checked" end
		to_s << other_attributes_string
		to_s << '>'
	end
end


# Creates a pop-up menu.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>list</b>
#
# <b>escape</b>::   Escapes HTML control characters in the items of the list
#                   if the "escape" is true. The default value is true.
# <b>list</b>::     Array which is iterated through.
# <b>values</b>::   Array which is value for each "value" attributes of
#                   <option> elements.
# <b>default</b>::  The first item if no item is selected.
# <b>selected</b>:: Item that are chosen from the selection list.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKPopUpButton < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if component.parse_ckd_value definition['escape'] then
				value = CKUtilities.unescape_html value.last
			else
				value = value.last
			end

			component.take_value( definition['selected'], value )
		end

		def own_attributes
			['escape', 'list', 'values', 'default', 'selected', 'enabled']
		end
	end

	def run
		@attr_name     = element_id.to_s
		@attr_escape   = fetch( 'escape' ) || true # true or false
		@attr_list     = fetch( 'list' )           # Enumerable
		@attr_values   = fetch( 'values' )         # Enumerable
		@attr_default  = fetch( 'default' )        # String
		@attr_selected = fetch( 'selected' )       # String

		check_required_attributes( ['list'] )
	end

	def to_s
		to_s = '<select'
		if @attr_name then to_s << " name=\"#@attr_name\"" end
		to_s << other_attributes_string
		to_s << ">\n"

		if @attr_default then
			to_s << '<option value=""'
			if @attr_selected == nil then
				to_s << " selected"
			end
			to_s << ">#@attr_default</option>\n"
		end

		if @attr_list then
			item = index = value = nil
			@attr_list.each_with_index do | item, index |
				to_s << "<option"
				if @attr_selected == item then
					to_s << " selected"
				end
				if @attr_values then
					value = @attr_values[index].to_s
					if @attr_selected == value then
						to_s << " selected"
					end
					if @attr_escape then
						value = CKUtilities.escape_html value
					end
					to_s << " value=\"#{value}\""
				end
				to_s << ">"

				# escape
				if @attr_escape then
					item = CKUtilities.escape_html item
				end
				to_s << "#{item}</option>\n"
			end
		end

		to_s << '</select>'
	end
end


# Creates a radio button.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attributes: <b>selection</b> and <b>value</b>, or <b>checked</b>
#
# <b>name</b>::      Name that identifies the radio button's group.
# <b>checked</b>::   If neither "value" nor "selection" attribute is nil and
#                    the value of "selection" is equal to that of "value",
#                    the check box is checked.
# <b>value</b>::     When the check box is checked,
#                    the value of "value" attribute is set to the component by
#                    the method specified by "selection" attribute.
# <b>selection</b>:: Object that the user chose from the check box.
# <b>enabled</b>::   If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKRadioButton < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if definition['checked'] then
				action = definition['checked']
				value  = true
			else
				action = definition['selection']
			end
			component.take_value( action, value )
		end

		def own_attributes
			['name', 'checked', 'value', 'selection', 'enabled']
		end
	end

	def element_id
		id = super
		id.element = @attr_name
		id
	end

	def run
		@attr_name      = fetch( 'name' )      # String
		@attr_checked   = fetch( 'checked' )   # true or false
		@attr_value     = fetch( 'value' )     # String
		@attr_selection = fetch( 'selection' ) # String
		if ( ( @attr_value and @attr_selection ) and \
			( @attr_value == @attr_selection ) ) then
			@attr_checked = true
		end

		check_required_attributes( ['selection','value'], ['checked'] )
		check_conflicted_attributes( 'selection', 'checked' )
		check_conflicted_attributes( 'value', 'checked' )
	end

	def to_s
		value = "#{definition['oid']}.#@attr_value"
		to_s  = "<input type=\"radio\" name=\"#{element_id}\" value=\"#{value}\""
		to_s << " checked"  if @attr_checked
		to_s << other_attributes_string
		to_s << ">"
	end
end


# Creates a reset button.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attributes: none
#
# <b>value</b>:: Title of the button.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKResetButton < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['value', 'enabled']
		end
	end

	def run
		@attr_value = fetch( 'value' )    # String
		unless @attr_value then @attr_value = 'Reset' end
	end

	def to_s
		to_s  = "<input type=\"reset\" value=\"#@attr_value\""
		to_s << other_attributes_string
		to_s << ">"
		to_s
	end
end


# Creates a submit button.
# This element must be used within HTML form.
#
# == Bindings
# Required attributes: none
#
# <b>action</b>::   Method to invoke when the button is clicked.
# <b>value</b>::    Title of the button.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#                   In addition, it doesn't send the form data 
#                   although the button is clicked.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKSubmitButton < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			component.application.element_id = id
		end

		def own_attributes
			['action', 'value', 'enabled']
		end
	end

	def run
		@attr_name     = element_id
		@attr_action   = fetch( 'action', false ) # String
		@attr_value    = fetch( 'value' )         # String
		if @attr_value == nil then @attr_value = 'Submit' end
	end

	def to_s
		to_s = "<input type=\"submit\" name=\"#@attr_name\" value=\"#@attr_value\""
		to_s << other_attributes_string
		to_s << ">"
	end
end


# Creates a text area.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>value</b>
#
# <b>value</b>::    Value of the text area. 
# <b>columns</b>::  Column size.
# <b>rows</b>::     Row size.
# <b>validate</b>:: Format string to validate input value.
# <b>pass</b>::     If validating is passed, the value is true.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKText < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			key = definition['value']
			component[key] = value.last

			if (format_key = definition['validate']) and \
			   (pass_key = definition['pass']) then
				component.validate_with_format_key(key, format_key, pass_key)
			end
		end

		def own_attributes
			['value', 'columns', 'rows', 'enabled', 'validate', 'pass']
		end
	end

	def run
		@attr_name     = element_id
		@attr_value    = fetch( 'value' )    # String
		@attr_columns  = fetch( 'columns' )  # Integer
		@attr_rows     = fetch( 'rows' )     # Integer

		check_required_attributes( ['value'] )
	end

	def to_s
		to_s = "<textarea name=\"#@attr_name\""
		if @attr_columns  then to_s << " cols=\"#@attr_columns\"" end
		if @attr_rows     then to_s << " rows=\"#@attr_rows\""    end
		to_s << other_attributes_string
		to_s << '>'
		if @attr_value   then to_s << CKUtilities.escape_html(@attr_value) end
		to_s << '</textarea>'
	end
end


# Creates a text input field.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>value</b>
#
# <b>value</b>::     Value of the text field. If you set an accessor method to
#                    this element, the form data is set to a component automatically
#                    by the method.
# <b>type</b>::      Type of the text field. "text" is for a normal text input field,
#                    "password" is for a password field and 
#                    "hidden" is for a hidden field.
# <b>size</b>::      Size of the text field.
# <b>maxlength</b>:: Max length of data for the text field.
# <b>validate</b>::  Format string to validate input value.
# <b>pass</b>::      If validating is passed, the value is true.
# <b>enabled</b>::   If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKTextField < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			key = definition['value']
			component[key] = value.last

			if (format_key = definition['validate']) and \
			   (pass_key = definition['pass']) then
				component.validate_with_format_key(key, format_key, pass_key)
			end
		end

		def own_attributes
			['value', 'type', 'size', 'maxlength', 'enabled', 'validate', 'pass']
		end
	end

	def run
		@attr_name      = element_id.to_s
		@attr_type      = fetch( 'type' )      # String
		@attr_value     = fetch( 'value' )     # String
		@attr_size      = fetch( 'size' )      # Integer
		@attr_maxlength = fetch( 'maxlength' ) # Integer
		if @attr_type == nil then @attr_type = 'text' end

		check_required_attributes( ['value'] )

		@attr_value = CKUtilities.escape_html @attr_value
	end

	def to_s
		to_s = "<input type=\"#@attr_type\""
		if @attr_name      then to_s << " name=\"#@attr_name\""           end
		if @attr_value     then to_s << " value=\"#@attr_value\""         end
		if @attr_size      then to_s << " size=\"#@attr_size\""           end
		if @attr_maxlength then to_s << " maxlength=\"#@attr_maxlength\"" end
		to_s << other_attributes_string
		to_s << '>'
	end
end


# CKFileUpload generates an input form to upload files.
# To use this, set "enctype" attribute of CKForm to "multipart/form-data" or
# "fileupload" attribute to true.
#
# == Bindings
# Required attribute: <b>data</b> and <b>file</b>
#
# <b>data</b>::    Variable of the attribute is set the uploaded file as
#                  a CKByteData object. 
# <b>file</b>::    Path of the uploaded file.
# <b>enabled</b>:: If the value is false, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKFileUpload < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			data = value.last
			if data.is_a? CKByteData then
				component.take_value( definition['data'], data )
				component.take_value( definition['file'], data.path )
			end
		end

		def own_attributes
			['data', 'file', 'enabled']
		end
	end

	def run
		check_required_attributes( ['data', 'file'] )
	end

	def to_s
		to_s  = "<input type=\"file\" name=\"#{element_id}\""
		to_s << other_attributes_string
		to_s << ">"
		to_s
	end
end


# CKPartsMaker is a Ruby module to deal with a component as parts of web page.
# A component which includes this module isn't displayed even if CGIKit recieves
# requests to show it as the target component ( CKApplication#target ).
#
# A name of parts component is recommended to have "Parts" at
# the end of the name to distinguish it from page component.
#
# == Usage
# Subclass of CKComponent includes CKPartsMaker.
#
#  class AnyParts < CKComponent
#    include CKPartsMaker
#    ...
#  end
#
module CKPartsMaker
	# When CGIKit recieves requests to show component parts,
	# CGIKit shows the page specified by this attribute.
	# A main page of an application is displayed when the value is not defined.
	attr_accessor :substitute_page
end


require 'strscan'

# CKDefinition parses .ckd ( CGIKit Definition ) file.
# CKComponent stores the result of parse and uses it for binding.
#
# The format of CKD file is very different from Ruby or HTML
# Here, the format of CKD file is described as pseudo-BNF like Ruby's document.
#
#   STMTS           : (STMT)*
#   STMT            : ELEMENT_NAME ':' ELEMENT_TYPE '{' ATTRS '}'
#   ATTRS           : (ATTRIBUTE '=' VALUE TERM)*
#   ATTRIBUTE       : ruby_method_name
#   VALUE           : METHODS 
#                   | LITERAL 
#   METHODS         : ruby_method_name('.'ruby_method_name)*
#                   | METHODS BLITERAL
#   LITERAL         : 'true'
#                   | 'false'
#                   | DIGIT_SEQUENCE
#                   | STRING
#   BLITERAL        : ARRAY
#                   : HASH
#   ARRAY           : '[' DIGIT_SEQUENCE ']'
#   HASH            : '[' LITERAL ']'
#   DIGIT_SEQUENCE  : DIGIT+
#   DIGIT           : [0-9]
#   STRING          : ''' any_character* '''
#                   | '"' any_character* '"'
#   ELEMENT_NAME    : (LETTER | DIGIT | '_')+
#   ELEMENT_TYPE    : ruby_class_name
#   LETTER          : [A-Za-z]
#   TERM            : ';'
#                   | '\n'
#
# ruby_method_name means Ruby's Method Name: For example, 'foo', 'to_s' and 'is_a?'.
# Also, ruby_class means Ruby's Class Name: For instance, 'CKString' and 'CKComponent'.
#
# CKDefinition regards a part which begins from '#' to the end of the line as comment.
# In the next case, '# foo bar comment' is not interpreted by CKDefinition.
#
#   foo : CKString {
#     # foo bar comment
#     value = "foo";
#   }
#
class CKDefinition
	class CKDParseError < CKError ; end #:nodoc:

	class << self
		def parse_ckd_file(filename)
			filename.untaint
			string = nil

			File.open(filename) do |f|
				string = f.read
			end

			parse(string, filename)
		end

		def parse(string, filename = '')
			scanner = StringScanner.new( string, false )

			# The way to use these local variables is bad.			
			definitions     = {}
			in_attrs        = false
			line_num        = 1			
			current_element = nil
			element_name    = nil
			element_type    = nil
			term            = "(;|\n)"

			# in the future, we may have to change this Proc object.
			pos_proc = proc do 
				"#{filename}:#{line_num}: "
			end
			
			while scanner.rest?
				# line break
				if scanner.skip(/(\r)?\n/) then
					line_num += 1
					next
				end

				# skip commnet or space
				if scanner.skip(/(?:#.*$|[ \r\f\t]+)/) then
					next
				end

				if in_attrs then
					# attribute
					if scanner.skip(/([a-zA-Z0-9_]+[\!\?]?)[ \r\f\t]*=[ \r\f\t]*/)
						key = scanner[1]

						# array and hash literal
						if scanner.skip(/^([^"'][^"';\n]*)\[/) then
							list   = nil
							object = scanner[1]

							if scanner.match?(/^(["'])/) then
								# string literal
								close = scanner[1]
								if	   close == '"' and scanner.skip(/"(.*?)"\]/)
								elsif	close == "'" and scanner.skip(/'(.*?)'\]/)
								else
									raise CKDParseError, \
										pos_proc.call << scanner.peek(60).inspect << \
										"\nString literal error. Or, \";\" doesn't exist."
								end
								list = "\"#{scanner[1]}\""
							else
								if scanner.skip(/([a-zA-Z0-9_\.\!\^?]+)\][ \r\t\f]*#{term}/)
									list = scanner[1]
								else
									raise CKDParseError, pos_proc.call << \
										scanner.peek(60).inspect << \
										"\nArray or Hash literal error. Or, \";\" doesn't exist.\n"
								end
							end
							value = "#{object}[#{list}]"
							line_num += value.count("\n")

						# value of attribute
						elsif scanner.match?(/^(["'])/) then
							# string lietral
							close = scanner[1]
							if	   close == '"' and scanner.skip(/"(.*?)"[ \r\f\t]*#{term}/m)
							elsif	close == "'" and scanner.skip(/'(.*?)'[ \r\f\t]*#{term}/m)
							else
								raise CKDParseError, \
									pos_proc.call << scanner.peek(60).inspect << \
									"\nString literal error. Or, \";\" doesn't exist."
							end

							value = "\"#{scanner[1]}\""
							line_num += value.count("\n")
						else
							# others
							#
							# This regexp is not accurate.
							if scanner.skip(/([a-zA-Z0-9_\.\!\^?]+)[ \r\t\f]*#{term}/)
								value = scanner[1]
							else
								raise CKDParseError, pos_proc.call << \
									scanner.peek(60).inspect << \
									"\nMethod name error. Or, \";\" doesn't exist." "\n" 
							end
						end

						current_element[key] = value			
						next
					end

					# end of definition
					if scanner.skip(/\s*\}/)
						definitions[element_name] = current_element
						current_element = nil
						in_attrs = false
						next
					end

					# skips space and separator.
					#
					# (ex.)
					# foo : CKString{;}
					if scanner.skip(/[ \r\t\f]*#{term}/)
						next
					end
				else
					# extracts name of the element and its class
					# class's regexp is not precise.
				if scanner.skip(/([a-zA-Z0-9_]+)\s*:\s*([a-zA-Z0-9_]+)\s*\{/)
					element_name = scanner[1]
					element_type = scanner[2]
					current_element = {}
					current_element['oid'] = element_name
					current_element['element'] = element_type

					if definitions.key? element_name then
						raise CKDParseError, pos_proc.call << \
							scanner.peek(60).inspect << \
							"\n'#{element_name}' definition is already existed." "\n" 
					end

			
					in_attrs = true
					next
				end
			end

			def_str = _pretty_print(definitions)
				raise CKDParseError, pos_proc.call << scanner.peek(60).inspect << \
					"\nNot match any rule.\n\n" << def_str << "\n" 
			end

			# Now, there is no data to be parsed.			
			if in_attrs 
				raise CKDParseError, pos_proc.call << \
					"The last element is not enclosed.\n\n"
			end
			
			definitions
		end

		def pretty_print(defs)
			print _pretty_print(defs)
		end

		private
		def _pretty_print(defs)
			s = ''
			indent = '		'

			defs.keys.sort.each do |name|
				attrs = defs[name]
				s << name
				s << ":\n"
	
				attrs.keys.sort.each do |k|
					v = attrs[k]
					s << indent 
					s << k 
					s << ': '
					s << v.inspect
					s << "\n"
				end

				s << "\n"
			end

			s
		end
	end
end


# CKHTMLParser parses a template of a component.
# The template is HTML, except one fact that it has
# "<cgikit>" tag. You can use two types of formats as CGIKit tag,
# "<cgikit>...</cgikit>" and "<cgikit />". The difference of the
# formats is whether the tag's body is empty or not. So, if you write
# "<cgikit></cgikit>", it has the same meaning as "<cgikit />".
#
# The CGIKit tag has only one attribute, "name". The "name" attribute is
# a name of CGIKit element. The "name" attribute is composed of "[a-zA-Z0-9_]".
# You can enclose the value of the "name" attribute by double quotation mark.
# And, The value is case-sensitive.
#
# For instance, these have the same meaning.
#
#   <cgikit name=foo></cgikit>
#   <cgikit name=foo />
#   <cgikit name="foo"></cgikit>
#   <cgikit name="foo" />
#
# On the other hand, these have different meanings.
#   <cgikit name=Foo />
#   <cgikit name="foo" />
#   <cgikit name=FOO></cgikit>
#
# Comment of HTML is interpreted by CKHTMLParser because it is
# necessary to include something like CSS and Javascript.
# If you want to comment out some parts of a template, you use "<!--- ... --->".
# The format is like HTML comments, but with an additional hyphen. 
#
#   <html>
#   <head>
#   </head>
#   <body>
#   <!-- HTML comment. This is interpreted. -->
#   <!--- CGIKit comment. This is not interpreted.
#     <!-- HTML comment. Because this is in CGIKit comment,
#          this is not also interpreted. -->
#   --->
#   </body>
#   </html>
#
# As the template is explained in the example, the first HTML comment
# is interpreted and the second HTML comment is not interpreted.
#
# CGIKit comment is not allowed to be nested. So, the next example
# is not allowed.
#
#   <!--- foo bar
#    <!--- foo bar --->
#   --->
class CKHTMLParser

	class CKHTMLParseError < CKError; end #:nodoc:
	
	attr_accessor :application, :html_string, :parent, :repetitions, \
	              :repetition_index

	def initialize( parent, string )
		@parent      = parent
		@application = parent.application
		@html_string = string
		@scanner     = StringScanner.new(string, false)
	end

	def parse
		contents = ''

		each_block_string do |str, kind|
			case kind 
			when :normal
				contents << str
			when :cgikit_tag
				element = _create_element( *_extract_name_body(str) )
				if parsed = element.to_s then 
					contents << parsed 
				end
			when :comment
			else
				raise CKHTMLParseError, "BUGS: CKHTMLParser#parse unknown token"
			end	
		end

		contents
	end

	private
	def _create_element( name, body )
		element_def = parent.definitions[ name ]
		if element_def.nil?
			raise CKHTMLParseError, "Can't create element: " << \
			                        "#{name} of #{parent.name_with_class}"
		end
	
		element = element_def[ 'element' ]
		object = CKElement.instance( element, @application, @parent, name, body )
		if @repetitions then
			object.repetitions      = @repetitions
			object.repetition_index = @repetition_index
		end
		object.run
		object
	end

	public
	def each_block_string
		rough_reg = /<(?:\!|c|\/c)/i #/
		
		comment_reg     = /<\!---[^-](.*?)[^-]--->/mi #/
		cktag_reg       = /<cgikit\s+[^>]*?\/>/i #/
		start_cktag_reg = /<cgikit\s+[^>]*?>/mi	
		end_cktag_reg   = /<\/cgikit\s*>/i #/		
		other_reg       = /(?:.+?(?=<(?:\!|c|\/c))|.+)/mi #/
		
		nest         = 0
		nested_cktag = ''
		str          = nil

		while @scanner.rest?
			# comment or cgikit tag
			if @scanner.match?(rough_reg)
				if str = @scanner.scan(comment_reg)
					yield str, :comment
					next
				end

				if str = @scanner.scan(cktag_reg)
					if nest > 0
						nested_cktag << str
					else
						yield str, :cgikit_tag
					end
					next
				end
		
				if str = @scanner.scan(start_cktag_reg) 	
					nested_cktag << str
					nest += 1
					next
				end
		
				if str = @scanner.scan(end_cktag_reg) 	
					nested_cktag << str
					nest -= 1
		
					if nest == 0
						yield nested_cktag, :cgikit_tag
						nested_cktag = ''
					elsif nest < 0
						raise CKHTMLParseError, \
							"too many end-tags(</cgikit>) in #{@parent.template_file}."
					end
		
					next	
				end
			end
			
			if str = @scanner.scan(other_reg)
				if nest > 0
					nested_cktag << str
				else
					yield str, :normal
				end
				next
			end

			raise CKHTMLParseError, "not match any rule in #{@parent.template_file}."
		end

		if nest != 0
			raise CKHTMLParseError, \
				"a cgikit tag is not closed in #{@parent.template_file}."
		end
	end

	private
	def _extract_name(name)
		if /(['"])(.*?)\1/ =~ name
			name = $2
		end

		name
	end

	def _extract_name_body(str)		
		if str =~ /\A<cgikit\s+name\s*=\s*(.*?)\s*>(.*)<\/cgikit>/im	#/
			name = _extract_name($1)
			body = $2

			return name, body
		elsif str =~ /\A<cgikit\s+name\s*=\s*(.*?)\s*\/>/im	#/
			name = _extract_name($1)
			
			return name, ''
		else
			raise CKHTMLParseError, "can't extract name and body: " << str.inspect
		end
	end
end


# The super class of HTTP Request-Response classes.
class CKMessage
	# HTML content.
	attr_accessor :content

	# Hash of HTTP headers.
	attr_accessor :headers

	# HTTP version. The default value is "1.1".
	attr_accessor :http_version

	# The encoding used for the content.
	attr_accessor :encoding

	# Array of CKCookie objects.
	attr_reader   :cookies

	EOL = "\r\n"

	def initialize( headers = nil )
		@headers      = headers || {}
		@http_version = "1.1"
		@cookies      = []
	end

	# Adds a cookie object.
	def add_cookie( cookie )
		@cookies << cookie
	end

	# Removes the specified cookie in the cookies.
	def remove_cookie( cookie )
		@cookies.delete cookie
	end

	# Returns HTTP version.
	def http_version_line
		"HTTP/#@http_version"
	end
end


# CKRequest represents requests of HTTP headers.
#
# CKRequest returns form parameters and cookies.
# But, in many cases, this class isn't used directly to get parameters 
# because CGIKit sets these parameters to components 
# automatically in the process of instanciating components.
#
# == Getting a request object
# You can get a request object by CKApplication#request or CKComponent#request.
#
# == Parameters
# CKRequest has some methods to access parameters.
#
# <b>form_values[key]</b>::       Returns an array of parameters.
#                                 If key is missing, returns an empty array.
# <b>form_value(key), [key]</b>:: This method returns a first object of
#                                 an array of parameters.
#                                 If key is missing, returns nil.
#
# == Cookies
# Cookies is an array which includes some CKCookie objects.
# See CKCookie section about details of cookie.
#
# <b>cookie(key)</b>::        Returns a CKCookie object whose key is the 
#                             same as the argument.
# <b>cookies</b>::            Returns an array of CKCookie objects.
# <b>cookie_value(key)</b>::  Returns the value of CKCookie object
#                             whose key is the the same as the argument.
# <b>cookie_values(key)</b>:: If the argument is nil (by default, argument is
#                             nil.), this method returns an array which has
#                             all the values of cookies. Otherwise, it returns
#                             an array which has the values of cookies
#                             specified by the argument.
#
# == HTTP Headers
# Some of HTTP headers are defined by instance methods.
# Call CKRequest#headers to get other CGI environment variables.
#
# <b>headers[key]</b>::      Returns a value of HTTP header.
# <b>accept</b>::            HTTP_ACCEPT
# <b>accept_charset</b>::    HTTP_ACCEPT_CHARSET 
# <b>accept_language</b>::   HTTP_ACCEPT_LANGUAGE
# <b>auth_type</b>::         AUTH_TYPE
# <b>content_length</b>::    CONTENT_LENGTH
# <b>content_type</b>::      CONTENT_TYPE
# <b>from</b>::              HTTP_FROM    
# <b>gateway_interface</b>:: GATEWAY_INTERFACE
# <b>path_info</b>::         PATH_INFO
# <b>path_translated</b>::   PATH_TRANSLATED
# <b>query_string</b>::      QUERY_STRING
# <b>raw_cookie</b>::        HTTP_COOKIE
# <b>referer</b>::           HTTP_REFERER
# <b>remote_addr</b>::       REMOTE_ADDR
# <b>remote_host</b>::       HTTP_HOST
# <b>remote_ident</b>::      REMOTE_IDENT
# <b>remote_user</b>::       REMOTE_USER
# <b>request_method</b>::    REQUEST_METHOD
# <b>script_name</b>::       SCRIPT_NAME
# <b>server_name</b>::       SERVER_NAME
# <b>server_port</b>::       SERVER_PORT
# <b>server_protocol</b>::   SERVER_PROTOCOL
# <b>server_software</b>::   SERVER_SOFTWARE
# <b>uri</b>, <b>url</b>::   REQUEST_URI
# <b>user_agent</b>::        HTTP_USER_AGENT
class CKRequest < CKMessage
	# Array of parameters.
	attr_accessor :form_values

	class << self
		# Parse query string and return a hash of parameters.
		def parse_query_string( query )
			params = Hash.new([])
			query.split(/[&;]/n).each do |pairs|
				key, value = pairs.split('=',2).collect{|v| CKUtilities.unescape_url(v) }
				if params.has_key?(key)
					params[key].push(value)
				else
					params[key] = [value]
				end
			end
			params
		end
	end

	def initialize( headers = nil, form_values = nil )
		super headers
		@cookies     = CKCookie.parse_raw_cookie @headers['HTTP_COOKIE']
		@form_values = form_values || Hash.new([])
	end

	# Returns a first object of an array of parameters. If key is missing, returns nil.
	def form_value( key )
		if @form_values.key? key
			@form_values[key].first
		else
			nil
		end
	end

	alias [] form_value

	# Returns a CKCookie object whose key is the same as the argument.
	def cookie( key )
		@cookies.each { | cookie |
			if cookie.name == key
				return cookie
			end
		}
		nil
	end

	# Returns the value of CKCookie object whose key is the the same as the argument.
	def cookie_value( key )
		@cookies.each { | cookie |
			if cookie.name == key
				return cookie.value
			end
		}
		nil
	end

	# If the argument is nil (by default, argument is nil.),
	# this method returns an array which has all the values of cookies. Otherwise,
	# it returns an array which has the values of cookies specified by the argument.
	def cookie_values( key = nil )
		if key then
			_cookie_values_for_key( @cookies, key )
		else
			_cookie_values @cookies
		end
	end

	# Returns an array of languages for localization.
	# The order is the preferred order of languages.
	def languages
		langs   = []
		quality = {}

		if accept_language then
			accept_language.scan(/\s*([^,;]*)\s*;\s*q\s*=\s*([^,]*),|([^,]*),/) do |match|
				lang = match.pop  || match.shift
				q    = (match.pop || 1).to_f

				if lang =~ /([^-]*)-(.*)/ then lang = $1 end

				if quality[lang] then
					if q > quality[lang] then
						quality[lang] = q
					end
				else
					quality[lang] = q
				end
			end

			sorted = quality.sort do |a,b|
				if    a.last <  b.last then  1
				elsif a.last == b.last then  0
				elsif a.last >  b.last then -1
				end
			end

			sorted.each do |lang|
				langs << lang.first
			end
		end

		langs
	end

	def accept;            self.headers['HTTP_ACCEPT']          end
	def accept_charset;    self.headers['HTTP_ACCEPT_CHARSET']  end
	def accept_language;   self.headers['HTTP_ACCEPT_LANGUAGE'] end
	def auth_type;         self.headers['AUTH_TYPE']            end
	def content_length;    self.headers['CONTENT_LENGTH']       end
	def content_type;      self.headers['CONTENT_TYPE']         end
	def from;              self.headers['HTTP_FROM']            end
	def gateway_interface; self.headers['GATEWAY_INTERFACE']    end
	def path_info;         self.headers['PATH_INFO']            end
	def path_translated;   self.headers['PATH_TRANSLATED']      end
	def query_string;      self.headers['QUERY_STRING']         end
	def raw_cookie;        self.headers['HTTP_COOKIE']          end
	def referer;           self.headers['HTTP_REFERER']         end
	def remote_addr;       self.headers['REMOTE_ADDR']          end
	def remote_host;       self.headers['HTTP_HOST']            end
	def remote_ident;      self.headers['REMOTE_IDENT']         end
	def remote_user;       self.headers['REMOTE_USER']          end
	def request_method;    self.headers['REQUEST_METHOD']       end
	def script_name;       self.headers['SCRIPT_NAME']          end
	def server_name;       self.headers['SERVER_NAME']          end
	def server_port;       self.headers['SERVER_PORT']          end
	def server_protocol;   self.headers['SERVER_PROTOCOL']      end
	def server_software;   self.headers['SERVER_SOFTWARE']      end
	def uri;               self.headers['REQUEST_URI']          end
	def user_agent;        self.headers['HTTP_USER_AGENT']      end

	alias url uri

	private

	def _cookie_values( cookies )
		values = {}
		cookies.each { | cookie |
			values[ cookie.name ] = cookie.value
		}
		values
	end

	def _cookie_values_for_key( cookies, key )
		values = []
		cookies.each { | cookie |
			if cookie.name == key
				values << cookie.value
			end
		}
		values
	end
end


# A response object that is sent to a browser by a CKAdapter object.
#
# == Getting a response object
# You can get a response object by CKApplication#response or CKComponent#response.
#
# == Setting headers for a response object
# To send HTTP response headers, append a pair of key and value to headers.
# For example,
#
#  application.response.headers['Content-Type'] = 'text/html'
#
class CKResponse < CKMessage
	# Status code in HTTP. Default status is 200 ( OK ).
	attr_accessor :status

	STATUS = { 
		100 => 'Continue',
		101 => 'Switching Protocols',
		200 => 'OK',
		201 => 'Created',
		202 => 'Accepted',
		203 => 'Non-Authoritative Information',
		204 => 'No Content',
		205 => 'Reset Content',
		206 => 'Partial Content',
		300 => 'Multiple Choices',
		301 => 'Moved Permanently',
		302 => 'Found',
		303 => 'See Other',
		304 => 'Not Modified',
		305 => 'Use Proxy',
		307 => 'Temporary Redirect',
		400 => 'Bad Request',
		401 => 'Unauthorized',
		402 => 'Payment Required',
		403 => 'Forbidden',
		404 => 'Not Found',
		405 => 'Method Not Allowed',
		406 => 'Not Acceptable',
		407 => 'Proxy Authentication Required',
		408 => 'Request Timeout',
		409 => 'Conflict',
		410 => 'Gone',
		411 => 'Length Required',
		412 => 'Precondition Failed',
		413 => 'Request Entity Too Large',
		414 => 'Request-URI Too Long',
		415 => 'Unsupported Media Type',
		416 => 'Requested Range Not Satisfiable',
		417 => 'Expectation Failed',
		500 => 'Internal Server Error',
		501 => 'Not Implemented',
		502 => 'Bad Gateway',
		503 => 'Service Unavailable',
		504 => 'Gateway Timeout',
		505 => 'HTTP Version Not Supported'
	}

	def initialize( headers = nil )
		super
		@status = 200
		@headers['Content-Type'] = 'text/html'
	end

	def to_s
		response = ''
		response << ( __cookie || '' )
		response << ( _general_header  || '' )
		response << ( _response_header || '' )
		response << _entity_header
		response << EOL
		response << ( self.content     || '' ) unless redirect?
		response
	end

	private

	def _general_header
		header = ''
		header << ( __header('Cache-Control') || '' )
		header << ( __header('Connection') || '' )
		header << ( __header('Date') || '' )
		header << ( __header('Pragma') || '' )
		header << ( __header('Trailer') || '' )
		header << ( __header('Transfet-Encoding') || '' )
		header << ( __header('Upgrade') || '' )
		header << ( __header('Via') || '' )
		header << ( __header('Warning') || '' )
	end

	def _response_header
		header = ''
		header << ( __header('Accept-Ranges') || '' )
		header << ( __header('Age') || '' )
		header << ( __header('ETag') || '' )
		header << ( __header('Location') || '' )
		header << ( __header('Proxy-Authenticate') || '' )
		header << ( __header('Retry-After') || '' )
		header << ( __header('Server') || '' )
		header << ( __header('Vary') || '' )
		header << ( __header('WWW-Authenticate') || '' )
	end

	def _entity_header
		header = ''
		header << ( __header('Allow') || '' )
		header << ( __header('Content-Encoding') || '' )
		header << ( __header('Content-Language') || '' )
		header << ( __header('Content-Length') || '' )
		header << ( __header('Content-Location') || '' )
		header << ( __header('Content-MD5') || '' )
		header << ( __header('Content-Range') || '' )
		header << __content_type
		header << ( __header('Content-Disposition') || '' )
		header << ( __header('Expires') || '' )
		header << ( __header('Last-Modified') || '' )
	end

	def __header( header )
		"#{header}: #{self.headers[header]}#{EOL}" if self.headers[header]
	end

 	def __content_type
		header = "Content-Type: #{self.headers['Content-Type']}"
		header << "; charset=#{self.encoding}" if self.encoding
		header << EOL
	end

	def __cookie
		return if @cookies.empty?

		header = ''
		@cookies.each { | cookie |
			header << "Set-Cookie: #{cookie.to_s}"
			header << EOL
		}
		header
	end

	public

	def status_line
		"#{http_version_line} #@status #{STATUS[@status]}#{EOL}"
	end

	# Sends a temporary redirect response to the client using the specified URL.
	def set_redirect( url )
		@status = 302
		@headers['Location'] = url
	end

	# Returns true if the response is setted redirect.
	def redirect?
		if ( @status == 302 ) or ( @status == 307 ) then
			true
		else
			false
		end
	end
end


# CKCookie is a class for cookie.
# To send cookies to a browser needs to create cookie objects 
# and set them to a response object. Instead of creating cookie objects,
# you can also get cookie objects from a request object.
#
# CKCookie objects have a pair of a cookie name and value.
# If you make the objects have multiple values for one name, 
# you must write code by yourself. 
# 
# == Controlling cookie objects
#
# === Creating cookies
# Give arguments of initialize() a name or a pair of name/value.
# The value of cookie is omittable.
#
#  cookie = CKCookie.new( name, value )
#
# === Getting cookies from a request object
# CKRequest has some methods for getting cookies.
# The methods are cookie(key), cookies, cookie_value(key), cookie_values(key).
# See also CKRequest.
#
# === Setting cookies to a response object
# CKResponse has methods for setting cookies. These methods are 
# defined in CKMessage, the superclass of CKResponse.
# Use add_cookie(cookie) and remove_cookie(cookie).
class CKCookie
	# Name of the cookie.
	attr_accessor :name

	# Value of the cookie.
	attr_accessor :value

	# Restricts the cookie in the site.
	attr_accessor :path

	# Domain that can receive the cookie.
	attr_accessor :domain

	# Expiry date. You set Time object to the cookie object.
	# The value is formatted when the cookie is returned.
	attr_accessor :expires

	# Decides whether the cookie is encrypted or not.
	attr_accessor :secure

	class << self
		# Parse raw cookie string and return an array of cookies.
 		def parse_raw_cookie( raw_cookie )
			cookies = []
			return cookies unless raw_cookie

			raw_cookie.split('; ').each do |pairs|
				name, value = pairs.split('=',2)
				name  = CKUtilities.unescape_url name
				value = CKUtilities.unescape_url value

				cookies << CKCookie.new( name, value )
			end

			cookies
		end
	end

	def initialize( name, value = nil, domain = nil, path = nil, secure = false )
		@name   = name
		@value  = value
		@domain = domain
		@path   = path
		@secure = secure
	end

	def to_s
		buf = "#@name="
		if @value   then buf << CKUtilities.escape_url(@value.to_s) end
		if @domain  then buf << "; domain=#@domain" end
		if @path	   then buf << "; path=#@path" end
		if @expires then buf << "; expires=#{CKUtilities.date(@expires)}" end
		if @secure == true then buf << '; secure' end
		buf
	end
end


# CKUtilitis is a module wihch collects utility methods based on cgi.rb.
module CKUtilities
	CR            = "\015"
	LF            = "\012"
	EOL           = CR + LF
	RFC822_DAYS   = %w[ Sun Mon Tue Wed Thu Fri Sat ]
	RFC822_MONTHS = %w[ Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ]

	# Returns an encoded string for URL.
	def escape_url( string )
		string.to_s.gsub( /([^ a-zA-Z0-9_.-]+)/n ) do
			'%' + $1.unpack( 'H2' * $1.size ).join( '%' ).upcase
		end.tr( ' ', '+' )
	end

	# Returns a string decoded from URL.
	def unescape_url( string )
		string.to_s.tr( '+', ' ' ).gsub( /((?:%[0-9a-fA-F]{2})+)/n ) do
			[ $1.delete( '%' ) ].pack( 'H*' )
		end
	end

	# Escapes HTML control characters.
	def escape_html( string )
		string.to_s.gsub(/&/n, '&amp;').
			gsub(/\"/n, '&quot;').
			gsub(/>/n, '&gt;').
			gsub(/</n, '&lt;').
			gsub(/'/n, '&#39;')
	end

	# Unescapes HTML control characters.
	def unescape_html( string )
  		string.to_s.gsub(/&(.*?);/n) do
	    	match = $1.dup
	    	case match
    		when /\Aamp\z/ni           then '&'
			when /\Aquot\z/ni          then '"'
			when /\Agt\z/ni            then '>'
			when /\Alt\z/ni            then '<'
			when /\A#0*(\d+)\z/n       then
				if Integer($1) < 256
					Integer($1).chr
				else
					if Integer($1) < 65536 and \
						($KCODE[0] == ?u or $KCODE[0] == ?U)
						[Integer($1)].pack("U")
					else
						"&##{$1};"
					end
				end
			when /\A#x([0-9a-f]+)\z/ni then
				if $1.hex < 256
					$1.hex.chr
				else
					if $1.hex < 65536 and \
						($KCODE[0] == ?u or $KCODE[0] == ?U)
						[$1.hex].pack("U")
					else
						"&#x#{$1};"
					end
				end
			else
				"&#{match};"
			end
		end
	end

	# Formats Time object in RFC1123.
	# For example, "Sat, 1 Jan 2000 00:00:00 GMT".
	def date( time )
		t = time.clone.gmtime
		return format("%s, %.2d %s %.4d %.2d:%.2d:%.2d GMT",
			RFC822_DAYS[t.wday], t.day, RFC822_MONTHS[t.month-1], t.year,
			t.hour, t.min, t.sec)
	end

	module_function :escape_url
	module_function :unescape_url
	module_function :escape_html
	module_function :unescape_html
	module_function :date
end


# CKLog is a simple logging class.
#
# == Debug level
# DEBUG < INFO < WARN < ERROR < FATAL
class CKLog

	DEBUG          = 1
	INFO           = 2
	WARN           = 3
	ERROR          = 4
	FATAL          = 5
	DEFAULT_LEVEL  = DEBUG

	# Log level. The default level is DEBUG.
	attr_accessor :level

	# Outputter. The default outputter is standard error.
	attr_accessor :out

	# Program name. The default name is "CGIKit" and the version.
	attr_accessor :name

	# Max file size to log. If size of file to output is over the max file size,
	# exception FileSizeError is raised.
	attr_accessor :max_file_size

	class FileSizeError < CKError; end #:nodoc:

	class << self
		# Outputs a debug string to STDERR.
		# A backtrace is sent if "exception" is given an error object.
		def debug( message, exception = nil )
			message = "[#{Time.now}] DEBUG - #{CKApplication.version}: #{message}\n"
			if exception then
				message << exception.backtrace.join << "\n"
			end
			$stderr.write message
		end
	end

	def initialize( options = {} )
		@level         = options['level']  || DEFAULT_LEVEL
		@name          = options['name']   || CKApplication.version
		@max_file_size = options['max_file_size']
		@file          = options['file']
		@out           = options['out'] || $stderr
	end

	def add( level, message = nil, &block )
		if message.nil? and block_given? then
			message = yield
		end

		message = _format_message( level, @name, message )

		if @file then
			if _max_file_size? message then
				raise FileSizeError, "#@file: Can't write log. The file size is limit."
			else
				CKFileLock.exclusive_lock(@file, 'a') do |f|
					f.write message
				end
			end
		else
			@out.write message
		end
	end

	private
	def _format_message( level, name, message )
		case level
		when DEBUG then level_s = 'DEBUG'
		when INFO  then level_s = ' INFO'
		when WARN  then level_s = ' WARN'
		when ERROR then level_s = 'ERROR'
		when FATAL then level_s = 'FATAL'
		end

		"[#{Time.now}] #{level_s} - #{name}: #{message}\n"
	end

	def _max_file_size?( message )
		if @max_file_size and FileTest.exist?(@file) then
			@max_file_size <= (File.size(@file) + message.size)
		end
	end

	public
	def close
		@out.close
	end

	def debug( message = nil, &block )
		add( DEBUG, message, &block ) if debug?
	end

	def info( message = nil, &block )
		add( INFO, message, &block ) if info?
	end

	def warn( message = nil, &block )
		add( WARN, message, &block ) if warn?
	end

	def error( message = nil, &block )
		add( ERROR, message, &block ) if error?
	end

	def fatal( message = nil, &block )
		add( FATAL, message, &block ) if fatal?
	end

	def debug?
		@level <= DEBUG
	end

	def info?
		@level <= INFO
	end

	def warn?
		@level <= WARN
	end

	def error?
		@level <= ERROR
	end

	def fatal?
		@level <= FATAL
	end

end


# CKFileLock is for locking files.
class CKFileLock
	# Creates a shared file lock on a file.
	def self.shared_lock( filename, mode = 'r' )
		File.open( filename, mode ) do | io |
			io.flock File::LOCK_SH
			yield io
			io.flock File::LOCK_UN
		end
	end

	# Creates a exclusive file lock on a file.
	def self.exclusive_lock( filename, mode = 'w' )
		File.open( filename, mode ) do | io |
			io.flock File::LOCK_EX
			yield io
			io.flock File::LOCK_UN
		end
	end
end


# CKByteData objects manage bytes.
class CKByteData
	# Path of a file saving the bytes.
	attr_accessor :path

	# Content type of the bytes.
	attr_accessor :content_type

	class << self
		# Create an instance from specified IO object.
		# If you give this a File object, sets "path" attribute of the instance.
		def new_with_io( io, offset = nil, count = nil )
			io.pos = offset if offset
			bytes  = new io.read(count)
			if io.respond_to? 'path' then
				bytes.path = io.path
			end
			bytes
		end

		# Create an instance from specified file.
		def new_with_file( filename )
			f          = File.open filename
			bytes      = new f.read
			bytes.path = f.path
			f.close
			bytes
		end
	end

	public

	def initialize( string = nil )
		@bytes = string || ''
	end

	# Returns bytes with spcified length or whole length if you omit it.
	def bytes( length = nil )
		if length then
			@bytes.slice( 0, length )
		else
			@bytes.to_s
		end
	end

	# Executes the block for every byte.
	def each
		byte = nil
		@bytes.each_byte do | byte |
			yield byte
		end
	end

	# Returns true if the bytes of each objects are equal.
	def ==( bytes )
		if @bytes == bytes.bytes then
			true
		else
			false
		end
	end

	# Returns length of the bytes.
	def length
		@bytes.size
	end

	# Appends bytes to the bytes.
	def <<( bytes )
		if bytes.is_a? CKByteData then
			@bytes << bytes.bytes
		else
			@bytes << bytes
		end
	end

	# Writes the bytes to a specified file.
	def write_to_file( filename, lock = true )
		if lock then
			CKFileLock.exclusive_lock(filename, 'w+') do |file|
				file.write to_s
			end
		else
			File.open(filename, 'w+') do |file|
				file.write to_s
			end
		end			
	end

	# Returns the object as a string.
	def to_s
		@bytes.to_s
	end
end


# CKResourceManager class manages resources of an application.
class CKResourceManager
	MIME = {
		'ez'      => 'application/andrew-inset',
		'hqx'     => 'application/mac-binhex40',
		'cpt'     => 'application/mac-compactpro',
		'doc'     => 'application/msword',
		'bin'     => 'application/octet-stream',
		'dms'     => 'application/octet-stream',
		'lha'     => 'application/octet-stream',
		'lzh'     => 'application/octet-stream',
		'exe'     => 'application/octet-stream',
		'class'   => 'application/octet-stream',
		'so'      => 'application/octet-stream',
		'dll'     => 'application/octet-stream',
		'oda'     => 'application/oda',
		'pdf'     => 'application/pdf',
		'ai'      => 'application/postscript',
		'eps'     => 'application/postscript',
		'ps'      => 'application/postscript',
		'smi'     => 'application/smil',
		'smil'    => 'application/smil',
		'mif'     => 'application/vnd.mif',
		'xls'     => 'application/vnd.ms-excel',
		'ppt'     => 'application/vnd.ms-powerpoint',
		'wbxml'   => 'application/vnd.wap.wbxml',
		'wmlc'    => 'application/vnd.wap.wmlc',
		'wmlsc'   => 'application/vnd.wap.wmlscriptc',
		'bcpio'   => 'application/x-bcpio',
		'vcd'     => 'application/x-cdlink',
		'pgn'     => 'application/x-chess-pgn',
		'cpio'    => 'application/x-cpio',
		'csh'     => 'application/x-csh',
		'dcr'     => 'application/x-director',
		'dir'     => 'application/x-director',
		'dxr'     => 'application/x-director',
		'dvi'     => 'application/x-dvi',
		'spl'     => 'application/x-futuresplash',
		'gtar'    => 'application/x-gtar',
		'hdf'     => 'application/x-hdf',
		'js'      => 'application/x-javascript',
		'skp'     => 'application/x-koan',
		'skd'     => 'application/x-koan',
		'skt'     => 'application/x-koan',
		'skm'     => 'application/x-koan',
		'latex'   => 'application/x-latex',
		'nc'      => 'application/x-netcdf',
		'cdf'     => 'application/x-netcdf',
		'sh'      => 'application/x-sh',
		'shar'    => 'application/x-shar',
		'swf'     => 'application/x-shockwave-flash',
		'sit'     => 'application/x-stuffit',
		'sv4cpio' => 'application/x-sv4cpio',
		'sv4crc'  => 'application/x-sv4crc',
		'tar'     => 'application/x-tar',
		'tcl'     => 'application/x-tcl',
		'tex'     => 'application/x-tex',
		'texinfo' => 'application/x-texinfo',
		'texi'    => 'application/x-texinfo',
		't'       => 'application/x-troff',
		'tr'      => 'application/x-troff',
		'roff'    => 'application/x-troff',
		'man'     => 'application/x-troff-man',
		'me'      => 'application/x-troff-me',
		'ms'      => 'application/x-troff-ms',
		'ustar'   => 'application/x-ustar',
		'src'     => 'application/x-wais-source',
		'xhtml'   => 'application/xhtml+xml',
		'xht'     => 'application/xhtml+xml',
		'zip'     => 'application/zip',
		'au'      => 'audio/basic',
		'snd'     => 'audio/basic',
		'mid'     => 'audio/midi',
		'midi'    => 'audio/midi',
		'kar'     => 'audio/midi',
		'mpga'    => 'audio/mpeg',
		'mp2'     => 'audio/mpeg',
		'mp3'     => 'audio/mpeg',
		'aif'     => 'audio/x-aiff',
		'aiff'    => 'audio/x-aiff',
		'aifc'    => 'audio/x-aiff',
		'm3u'     => 'audio/x-mpegurl',
		'ram'     => 'audio/x-pn-realaudio',
		'rm'      => 'audio/x-pn-realaudio',
		'rpm'     => 'audio/x-pn-realaudio-plugin',
		'ra'      => 'audio/x-realaudio',
		'wav'     => 'audio/x-wav',
		'pdb'     => 'chemical/x-pdb',
		'xyz'     => 'chemical/x-xyz',
		'bmp'     => 'image/bmp',
		'gif'     => 'image/gif',
		'ief'     => 'image/ief',
		'jpeg'    => 'image/jpeg',
		'jpg'     => 'image/jpeg',
		'jpe'     => 'image/jpeg',
		'png'     => 'image/png',
		'tiff'    => 'image/tiff',
		'tif'     => 'image/tiff',
		'djvu'    => 'image/vnd.djvu',
		'djv'     => 'image/vnd.djvu',
		'wbmp'    => 'image/vnd.wap.wbmp',
		'ras'     => 'image/x-cmu-raster',
		'pnm'     => 'image/x-portable-anymap',
		'pbm'     => 'image/x-portable-bitmap',
		'pgm'     => 'image/x-portable-graymap',
		'ppm'     => 'image/x-portable-pixmap',
		'rgb'     => 'image/x-rgb',
		'xbm'     => 'image/x-xbitmap',
		'xpm'     => 'image/x-xpixmap',
		'xwd'     => 'image/x-xwindowdump',
		'igs'     => 'model/iges',
		'iges'    => 'model/iges',
		'msh'     => 'model/mesh',
		'mesh'    => 'model/mesh',
		'silo'    => 'model/mesh',
		'wrl'     => 'model/vrml',
		'vrml'    => 'model/vrml',
		'css'     => 'text/css',
		'html'    => 'text/html',
		'htm'     => 'text/html',
		'asc'     => 'text/plain',
		'txt'     => 'text/plain',
		'rtx'     => 'text/richtext',
		'rtf'     => 'text/rtf',
		'sgml'    => 'text/sgml',
		'sgm'     => 'text/sgml',
		'tsv'     => 'text/tab-separated-values',
		'wml'     => 'text/vnd.wap.wml',
		'wmls'    => 'text/vnd.wap.wmlscript',
		'etx'     => 'text/x-setext',
		'xml'     => 'text/xml',
		'xsl'     => 'text/xml',
		'mpeg'    => 'video/mpeg',
		'mpg'     => 'video/mpeg',
		'mpe'     => 'video/mpeg',
		'qt'      => 'video/quicktime',
		'mov'     => 'video/quicktime',
		'mxu'     => 'video/vnd.mpegurl',
		'avi'     => 'video/x-msvideo',
		'movie'   => 'video/x-sgi-movie',
		'ice'     => 'x-conference/x-cooltalk'
	}

	def initialize( application )
		@application          = application
		@resources            = @application.resources
		@web_server_resources = @application.web_server_resources
		@document_root        = @application.document_root
	end

	# Returns the public URL for the specified resource
	# when it is under the web server resources directory.
	# Otherwise returns nil.
	def url( name )
		if filepath = path(name) then
			absolute = File.expand_path @web_server_resources
			if filepath =~ /^#{absolute}/ then
				url = filepath.sub(@document_root, '')
				return url
			end
		end

		nil
	end

	# Returns the file path of the specified resource.
	def path( name )
		resourcedir = [@resources, @web_server_resources]

		resourcedir.each do |dir|
			filepath = File.join(dir, name).untaint
			if FileTest.exist? filepath then
				return File.expand_path(filepath).untaint
			end
		end
		nil
	end

	# Returns a CKByteData object for the specified resource.
	def bytedata( name )
		if filepath = path(name) then
			data = CKByteData.new_with_file filepath
			data.content_type = content_type filepath
			data
		else
			nil
		end
	end

	# Finds the content type for extension of the specified path.
	# If the path don't have extension, returns nil.
	def content_type( path )
		# for ruby 1.6
		base = File.basename path
		type = nil
		if base =~ /\.([^\.]+)$/ then
			type = MIME[$1]
		end
		type
	end
end


require 'set'

class CKQualifierParser
	class ParseError < StandardError; end #:nodoc:

	attr_reader :qualifier

	def initialize( format )
		tokens     = _analyze format.dup
		@qualifier = _parse tokens
	end

	private

	def _analyze( format )
		format    = "(#{format})"
		scanner   = StringScanner.new format
		qualifier = nil
		tokens    = []
		op_reg    = /\A(==|!=|>=|<=|>|<|=~)/im

		until scanner.eos? do
			scanner.skip /\A[\s]+/

			if str = scanner.scan(/\A(\(|\))/) then
				tokens << str
			elsif str = scanner.scan(op_reg) then
				tokens << CKQualifier.operator_symbol(str)
			elsif str = scanner.scan(/\A\d+\.\d+/) then
				tokens << str.to_f
			elsif str = scanner.scan(/\A\d+/) then
				tokens << str.to_i
			elsif scanner.match?(/\Atrue\W/) then
				scanner.scan /\Atrue/
				tokens << true
			elsif scanner.match?(/\Afalse\W/) then
				scanner.scan /\Afalse/
				tokens << false
			elsif scanner.match?(/\Anil\W/) then
				scanner.scan /\Anil/
				tokens << nil
			elsif scanner.scan(/\A\//) then
				tokens << _parse_regexp(scanner)
			elsif str = scanner.scan(/\A'(([^'\\]|\\.)*)'/) then
				tokens << scanner[0]
			elsif str = scanner.scan(/\A"(([^"\\]|\\.)*)"/) then
				tokens << scanner[0]
			else
				str = scanner.scan /\A[^\s\(\)]+/
				tokens << str
			end
		end

		tokens
	end

	def _parse_regexp( scanner )
		regstr = ''
		char = nil
		pre_char = nil
		option = nil

		until scanner.eos? do
			char = scanner.getch

			if (pre_char != '\\') and (char == '/') then
				option = scanner.scan(/\A\w+/)
				break
			end

			regstr << char
			pre_char = char
		end

		Regexp.new(regstr, option)
	end

	def _parse( tokens )
		op_stack  = []
		out_stack = []
		op = left = right = q = nil

		reg_and = /\Aand\Z/mi
		reg_or  = /\Aor\Z/mi
		reg_not = /\Anot\Z/mi

		tokens.each do |token|
			case token
			when '('
				op_stack << token
			when ')'
				until op_stack.last == '(' do
					op    = op_stack.pop
					right = out_stack.pop
					left  = out_stack.pop

					case op
					when Symbol
						if Regexp === right then
							q = CKKeyValueQualifier.new(left, op, right)
						elsif right =~ /\A'(([^'\\]|\\.)*)'/ then
							q = CKKeyValueQualifier.new(left, op, $1)
						elsif right =~ /\A"(([^"\\]|\\.)*)"/ then
							q = CKKeyValueQualifier.new(left, op, $1)
						elsif (Numeric === right) or (right == true) or \
							(right == false) or right.nil? then
							q = CKKeyValueQualifier.new(left, op, right)
						else
							q = CKKeyComparisonQualifier.new(left, op, right)
						end
					when reg_and
						if CKAndQualifier === right then
							right.qualifiers.unshift left
							q = right
						else
							q = CKAndQualifier.new [left, right]
						end
					when reg_or
						if CKOrQualifier === right then
							right.qualifiers.unshift left
							q = right
						else
							q = CKOrQualifier.new [left, right]
						end
					when reg_not
						q = CKNotQualifier.new right
					end
					out_stack << q
				end
				op_stack.pop		
			when reg_and
				op_stack << token
			when reg_or
				op_stack << token
			when reg_not
				op_stack << token
			when Symbol
				op_stack << token
			else
				out_stack << token
			end
		end

		result = out_stack.pop
		unless out_stack.empty? and op_stack.empty? then
			raise ParseError, 'parse error'
		end

		result
	end
end


class CKQualifier
	module ComparisonSupport
		def coerce( left, right )
			coerced = left

			case right
			when String
				coerced = left.to_s
			when Integer
				coerced = left.to_i
			when Float
				coerced = left.to_f
			when NilClass
				if left == '' then
					coerced = nil
				end
			when Regexp
				coerced = left.to_s
			end

			coerced
		end

		def compare( left, right, symbol )
			coerced = coerce(left, right)
			__send__(symbol, coerced, right)
		end

		def equal?( left, right )
			left == right
		end

		def not_equal?( left, right )
			left != right
		end

		def greater?( left, right )
			left > right
		end

		def greater_or_equal?( left, right )
			left >= right
		end

		def less?( left, right )
			left < right
		end

		def less_or_equal?( left, right )
			left <= right
		end

		def match?( left, right )
			if left =~ right then
				true
			else
				false
			end
		end
	end

	class UnknownKeyError < StandardError; end #:nodoc:

	extend ComparisonSupport

	EQUAL            = :'equal?'
	NOT_EQUAL        = :'not_equal?'
	GREATER          = :'greater?'
	GREATER_OR_EQUAL = :'greater_or_equal?'
	LESS             = :'less?'
	LESS_OR_EQUAL    = :'less_or_equal?'
	MATCH            = :'match?'

	class << self
		OPERATORS   = ['==', '!=', '<', '<=', '>', '>=', '=~']

		def new_with_format( format )
			parser = CKQualifierParser.new(format)
			parser.qualifier
		end

		alias format new_with_format

		def operator_symbol( string )
			case string.upcase
			when '==' then EQUAL
			when '!=' then NOT_EQUAL
			when '>'  then GREATER
			when '>=' then GREATER_OR_EQUAL
			when '<'  then LESS
			when '<=' then LESS_OR_EQUAL
			when '=~' then MATCH
			end
		end

		def operator_string( operator )
			case operator
			when EQUAL            then '=='
			when NOT_EQUAL        then '!='
			when GREATER          then '>'
			when GREATER_OR_EQUAL then '>='
			when LESS             then '<'
			when LESS_OR_EQUAL    then '<='
			when MATCH            then '=~'
			end
		end

		def operators
			OPERATORS
		end

	end

	def qualifier_keys
		set = Set.new
		add_qualifier_keys set
		set
	end

	# abstract - subclasses must override it
	def add_qualifier_keys( set ); end

	alias inspect to_s
end


class CKKeyValueQualifier < CKQualifier
	attr_reader :key, :value, :symbol

	def initialize( key, symbol, value )
		super()
		@key = key
		@symbol = symbol
		@value = value
	end

	def add_qualifier_keys( set )
		set << @key
	end

	def ==( other )
		bool = false
		if CKKeyValueQualifier === other then
			if (@key == other.key) and (@symbol == other.symbol) and \
				(@value == other.value) then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		CKQualifier.compare(object[@key], @value, @symbol)
	end

	public

	def to_s
		op = CKQualifier.operator_string @symbol
		if String === @value then
			value_s = "'#@value'"
		else
			value_s = @value
		end

		"(#@key #{op} #{value_s})"
	end
end


class CKKeyComparisonQualifier < CKQualifier
	attr_reader :left, :symbol, :right

	def initialize( left, symbol, right )
		super()
		@left   = left
		@symbol = symbol
		@right  = right
	end

	def add_qualifier_keys( set )
		set << @left
	end

	def ==( other )
		bool = false
		if CKKeyComparisonQualifier === other then
			if (@left == other.left) and (@symbol == other.symbol) and \
				(@right == other.right) then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		CKQualifier.compare(object[@left], object[@right], @symbol)
	end

	def to_s
		op = CKQualifier.operator_string @symbol
		"(#@left #{op} #@right)"
	end
end


class CKAndQualifier < CKQualifier
	attr_reader :qualifiers

	def initialize( qualifiers )
		super()
		@qualifiers = qualifiers
	end

	def each
		qualifiers.each do |qualifier|
			yield qualifier
		end
	end

	def add_qualifier_keys( set )
		@qualifiers.each do |qualifier|
			qualifier.add_qualifier_keys set
		end
	end

	def ==( other )
		bool = false
		if CKAndQualifier === other then
			if @qualifiers == other.qualifiers then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		@qualifiers.each do |qualifier|
			unless qualifier.eval? object then
				return false
			end
		end
		true
	end

	def to_s
		str = '('
		@qualifiers.each do |q|
			str << q.to_s
			unless @qualifiers.last == q then
				str << " AND "
			end
		end
		str << ')'
		str
	end
end


class CKOrQualifier < CKQualifier
	attr_reader :qualifiers

	def initialize( qualifiers )
		super()
		@qualifiers = qualifiers
	end

	def each
		qualifiers.each do |qualifier|
			yield qualifier
		end
	end

	def add_qualifier_keys( set )
		@qualifiers.each do |qualifier|
			qualifier.add_qualifier_keys set
		end
	end

	def ==( other )
		bool = false
		if CKOrQualifier === other then
			if @qualifiers == other.qualifiers then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		@qualifiers.each do |qualifier|
			if qualifier.eval? object then
				return true
			end
		end
		false
	end

	def to_s
		str = '('
		@qualifiers.each do |q|
			str << q.to_s
			unless @qualifiers.last == q then
				str << " OR "
			end
		end
		str << ')'
		str
	end
end


class CKNotQualifier < CKQualifier
	attr_reader :qualifier

	def initialize( qualifier )
		super()
		@qualifier = qualifier
	end

	def add_qualifier_keys( set )
		qualifier.add_qualifier_keys set
	end

	def ==( other )
		bool = false
		if CKNotQualifier === other then
			if @qualifier == other.qualifier then
				bool = true
			end
		end

		bool
	end

	def eval?( object )
		unless @qualifier.eval? object then
			true
		else
			false
		end
	end

	def to_s
		"(NOT #{qualifier})"
	end
end

