# = CSVParser - CSV format parser / generator
# CSVParser is a library to parse and generate CSV(Comma Sepalated Value) format 
# text data. It can do TSV(Tab Sepalated Value) format text data with tab as sepalate
# delimiter.
#
# == License
# CSVParser is a free software distributed under the Ruby license. 
#
# == Author
# SUZUKI Tetsuya <suzuki@spice-of-life.net> 


require 'delegate'

# CSVParser is a class to parse and generate CSV format.
# Instance methods of Array delegates to lines attribute.
class CSVParser < DelegateClass(Array)
	include Enumerable

	# String to parse.
	attr_accessor :string

	# Array of CSV lines(CSVLine objects).
	attr_accessor :lines

	class << self
		# Generates an instance parsed contents of the file.
		def new_with_file( filename, space = false, sepalator = ',' )
			file   = File.new filename
			string = file.read
			file.close
			CSVParser.new( string, space, sepalator )
		end

		# Returns an array parsed the string.
		def parse( string, space = false, sepalator = ',' )
			in_field  = :IN_FIELD
			in_qfield = :IN_QFIELD

			status = in_field
			csv    = string.split //
			parsed = []
	
			char = index = nil
			line = ''

			csv.each_with_index do | char, index |
				case status
				when in_field
					case char
					when '"'
						line << char
						status = in_qfield
					when "\r"
						unless csv[index+1] == "\n" then
							parsed << _parse_line(line,space,sepalator) unless line.empty?
							line = ''
						end
					when "\n"
						parsed << _parse_line(line,space,sepalator) unless line.empty?
						line = ''
					else
						line << char
					end
				when in_qfield
					case char
					when '"'
						line << char
						status = in_field
					else
						line << char
					end
				end
			end

			parsed << _parse_line(line,space,sepalator) unless line.empty?
			parsed
		end

		private
		def _parse_line( string, space, sepalator )
			in_field  = :IN_FIELD
			in_qfield = :IN_QFIELD

			status = in_field
			csv    = string.split //
			parsed = []
	
			char = index = match = nil
			field = ''
			quote = 0
			is_quote = false

			csv.each_with_index do | char, index |
				case status
				when in_field
					case char
					when '"'
						unless is_quote then
							field = ''
							is_quote = true
						else
							quote += 1
						end
						status = in_qfield

						if (quote > 0) and ((quote%2) == 0) then
							field << '"'
						end
					when sepalator
						if space then
							if match = /[ ]+$/.match(field) then field = match.pre_match end
							if match = /^[ ]+/.match(field) then field = match.post_match end
						end
						parsed << field
						field = ''
						is_quote = false
						quote = 0
					else
						field << char unless is_quote
					end
				when in_qfield
					case char
					when '"'
						quote += 1
						status = in_field
					else
						quote = 0
						field << char
					end
				end
			end

			if space then
				if match = /[ ]+$/.match(field) then field = match.pre_match end
				if match = /^[ ]+/.match(field) then field = match.post_match end
			end
			parsed << field

			parsed
		end

		public

		# Returns an array parsed contents of the file.
		def parse_with_file( filename, space = false, sepalator = ',' )
			file   = File.new filename
			string = file.read
			file.close
			CSVParser.parse( string, space, sepalator )
		end
	end

	# Generates an instance parsed the string sepalated by the sepalator.
	# If the space is true, deletes spaces of fields at start and end.
	def initialize( string = '', space = false, sepalator = ',' )
		@string = string
		@lines = []

		lines = CSVParser.parse( @string, space, sepalator )
		line  = nil
		lines.each do | line |
			@lines << CSVLine.new(line)
		end

		super @lines
	end

	# Executes the block for every field.
	def each_field
		line = field = nil
		@lines.each do | line |
			line.each do | field |
				yield field
			end
		end
	end

	# Writes saving data to the specified file.
	def write_to_file( filename, enquote = false, eol = "\r\n", sepalator = ',' )
		str = to_s( enquote, eol, sepalator )
		f = File.new( filename, 'w' )
		f.write str
		f.close
	end

	# Returns true if self is equal to the CSVParser object.
	def ==( csvparser )
		if @lines == csvparser.lines then
			true
		else
			false
		end
	end

	# Returns a string converted to CSV format with the sepalator and eol(end of line)
	# character. If enquote is true, each fields are enclosed with double quotes.
	def to_s( enquote = false, eol = "\r\n", sepalator = ',' )
		str = ''
		@lines.each do | line |
			str << line.to_s(enquote,sepalator)
			str << eol
		end
		str
	end
end


# CSVLine is a class to manage CSV line.
# Instance methods of Array delegates to fields attribute.
class CSVLine < DelegateClass(Array)

	# Array of CSV fields.
	attr_accessor :fields

	class << self
		# Returns a string escaped CSV control characters(double quotes and returns).
		# If enquote is true, the string are enclosed with double quotes. Or the string
		# are enclosed when it includes CSV control characters or the sepalator.
		def escape( string, enquote = false, sepalator = ',' )
			escaped = string.dup
			if (escaped =~ /[#{sepalator}"\r\n]/) or enquote then
				escaped.gsub!( '"', '""' )
				escaped = '"' + escaped + '"'
			end
			escaped
		end
	end

	# Generates an instance with the array as fields.
	def initialize( array = [] )
		@fields = []

		field = nil
		array.each do | field |
			@fields << field
		end

		super @fields
	end

	# Returns true if self is equal to the CSVLine object.
	def ==( csvline )
		if @fields == csvline.fields then
			true
		else
			false
		end
	end

	# Returns a string converted to CSV format with the sepalator.
	# If enquote is true, each fields are enclosed with double quotes.
	def to_s( enquote = false, sepalator = ',' )
		str  = ''
		field = nil

		@fields.each do | field |
			str << CSVLine.escape(field,enquote,sepalator)
			str << sepalator
		end
		str.chop!
		str
	end
end

