//
// Copyright (C) 1999-2006 WideStudio/MWT Project Team
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//

#include <WScom.h>
#ifndef NO_SOCKET
#include <WStcpcom.h>
#include <WSCstring.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/in.h>


#ifndef O_NONBLOCK
#include <sys/ioctl.h>
#endif

#if !defined(FD_SET)
#include <sys/select.h>
#endif

#define WS_READ_SIZE     1024
#define WS_READ_MIN_SIZE  256
#define WS_READ_ERR_SIZE  256
#define WS_MAX_PATH_LEN  2048
#define WS_MAX_URLS        64

static char WS_CONTENT_TYPE[]="Content-type:";
static char WS_CONTENT_LEN[]="Content-length:";
static char WS_POST_CONTENT_TYPE[]="application/x-www-form-urlencoded";

static void SetSocketBlockingState(int* sock,int blocking){
  int ret;
#ifdef O_NONBLOCK
  int flags = fcntl(*sock, F_GETFL);
  ret = fcntl(*sock, F_SETFL, blocking ? (flags & (~O_NONBLOCK)) : (flags | O_NONBLOCK));
#else
  int val=(!n_blocking);
  ret = ioctl(*sock, FIONBIO, &val);
#endif
  if (ret == -1){
    fprintf(stderr, "Connection error. <%s>\n", (blocking ? "blocking" : "non-blocking"));
  }
}

static long _connect_to(char* host,long port,int* sock){
  int status = WS_NO_ERR;
  struct sockaddr_in addr;
  struct sockaddr_in *sin=(&addr);
  struct hostent *hostent=NULL;

  if (*host >= '0' && *host <= '9'){
    sin->sin_addr.s_addr = inet_addr(host);
  }else{
    hostent = gethostbyname(host);
    if (hostent == NULL){
      fprintf(stderr, "Cannot find Internet node name.\n");
      return WS_HOSTN_ERR;
    }
    memcpy(&sin->sin_addr, hostent->h_addr, hostent->h_length);
  }
  sin->sin_family = AF_INET;
  sin->sin_port = htons((WSCushort)port);
  *sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);

  SetSocketBlockingState(sock, False);

  status = connect(*sock, (struct sockaddr*)&addr, sizeof(addr));
#ifdef SUN
  if ((status < 0) && (errno==EINPROGRESS || errno==EAGAIN)){
#else //SUN
  if ((status < 0) && (errno == EINPROGRESS)){
#endif //SUN
    struct timeval timeout;
    int ret=0;

    timeout.tv_sec = 0;
    timeout.tv_usec = 100000;
    while (ret <= 0){
      fd_set writefds;

      FD_ZERO(&writefds);
      FD_SET(*sock, &writefds);
      ret = select(FD_SETSIZE, NULL, &writefds, NULL, &timeout);
      if ((ret < 0)&&(errno != EALREADY)){
        status = ret;
        break;
      }else if (ret > 0){
        status = connect(*sock, (struct sockaddr*)&addr,sizeof(addr));
        if ((status < 0)&&(errno == EISCONN)){
          status = WS_NO_ERR;
        }
        if (errno == EALREADY){
          ret = 0;
        }else{
          break;
        }
      }else{
        status = connect(*sock, (struct sockaddr*)&addr, sizeof(addr));
#ifdef SUN
        if ((status < 0) && (errno != EALREADY) && (errno != EISCONN) && (errno != EAGAIN)){
#else //SUN
        if ((status < 0) && (errno != EALREADY) && (errno != EISCONN)){
#endif //SUN
               break;
         }
      }
    }
  }
  if (status >= 0){
    SetSocketBlockingState(sock, True);
  }else{
    close(*sock);
  }
  return status;
}


#ifdef DS
extern "C"{
  extern int gethostname(char*,WSCuint);
}
#endif

#if 0 /*SUN*/
extern "C"{
  extern int gethostname(char*,WSCuint);
}
#endif


static char _ws_http_ver[]="HTTP/1.0";
static char _client_name[1024];
static char _user_name_buf[1024];

static long _cgi_query        = False;
static char *_fname_cgi_query = NULL;

#ifdef DS90
extern "C"{
  extern int gethostname(char*,WSCuint);
}
#endif
#if 0 /*SUN*/
extern "C"{
  extern int gethostname(char*,WSCuint);
}
#endif


struct WSCurlCache {
   long   fl;
   long   rbuf_size;
   char*  url_base_name;
   char*  rbuf;
   char*  content_type;
   struct WSCurlCache* child;
   struct WSCurlCache* parent;
   WSCurlCache(){
     fl = 0;
     rbuf_size = 0;
     rbuf = NULL;
     content_type = NULL;
     url_base_name = NULL;
     child = NULL;
     parent = NULL;
   };
   ~WSCurlCache(){
     if (rbuf != NULL){
       delete rbuf;
     }
     if (content_type != NULL){
       delete content_type;
     }
     if (url_base_name != NULL){
       delete url_base_name;
     }
   };
};

static char          _default_file[] = "index.html";
static char          _http_proxy[WS_MAX_PATH_LEN+1];
static WSCurlCache* _first_url_cache = NULL;
static WSCurlCache* _last_url_cache  = NULL;
static long          _max_url_cache   = WS_MAX_URLS;
static long          _current_url_cache   = 0;
static long          _http_initialized = 0;
static long          _url_cache_initialized = 0;

void _get_user_name(char* buffer,long buf_size){
   char user_name[1024];
   long total=0;

   sprintf(user_name, "ws@");
   total = strlen(user_name);
   if (gethostname(&user_name[total], sizeof(user_name)-1-total) < 0){
      sprintf(&user_name[total], "UNKNOWN");
   }else{
      struct hostent *p_hostent=gethostbyname(&user_name[total]);

      if (p_hostent != NULL && p_hostent->h_name != NULL &&
            *p_hostent->h_name != 0){
         strcpy(&user_name[total], p_hostent->h_name);
      }
   }
   strncpy(buffer, user_name, buf_size);
}

static char *_get_url_name(char *url){
   char *ptr = strchr(url, '#');

   if (ptr != NULL){
      char *return_buf;

      *ptr = 0;
      return_buf = WSGFstrdup(url);
      *ptr = '#';
      return return_buf;
   }
   return WSGFstrdup(url);
}
static void _init_cache(){
  if (_url_cache_initialized == 0){
    _url_cache_initialized = 1;
    _first_url_cache = NULL;
    _last_url_cache = NULL;
    _current_url_cache = 0;
  }
}

static void _remove_cache( struct WSCurlCache *url){
  if (url == NULL){
    return;
  }
  if (url->child == NULL){
    _last_url_cache = url->parent;
  }else{
    url->child->parent = url->parent;
  }
  if (url->parent == NULL){
    _first_url_cache = url->child;
  }else{
    url->parent->child = url->child;
  }
  url->parent = url->child = NULL;
  _current_url_cache--;
}


static void _add_cache(struct WSCurlCache* parent_url,
                           struct WSCurlCache* child_url,
                           struct WSCurlCache* url){
  if (parent_url == NULL){
    _first_url_cache = url;
  }else{
    parent_url->child = url;
  }
  if (child_url == NULL){
    _last_url_cache = url;
  }else{
    child_url->parent = url;
  }
  url->parent = parent_url;
  url->child = child_url;
  _current_url_cache++;
}

static WSCurlCache *_search_cache(char* url_name,long update_lru){
  char *url_base_name = _get_url_name(url_name);
  struct WSCurlCache *url;

  _init_cache();
  if (url_base_name == NULL){
    return NULL;
  }
  for (url=_last_url_cache; url != NULL; url=url->parent){
    if (strcmp(url_base_name, url->url_base_name) == 0){
      delete url_base_name;
      if (update_lru){
        _remove_cache(url);
        _add_cache(_last_url_cache, NULL, url);
      }
      return url;
    }
  }
  delete url_base_name;
  return NULL;
}

static char *_get_proxy_info(char* proxy_spec,long def_port,long* pn_port){
   char *colon_ptr=strchr(proxy_spec, ':'), *return_buf=NULL;

   if (colon_ptr == NULL){
      *pn_port = def_port;
      return_buf = WSGFstrdup(proxy_spec);
   }else{
      *colon_ptr = 0;
      *pn_port = atoi(&colon_ptr[1]);
      return_buf = WSGFstrdup(proxy_spec);
      *colon_ptr = ':';
   }
   return return_buf;
}

static long _usr_agent_name_inited = False;

static void _init_client_name(){
   if (_usr_agent_name_inited){
     return;
   }
   _usr_agent_name_inited = True;
   sprintf(_client_name,"ws/web");
   _get_user_name(_user_name_buf, sizeof(_user_name_buf));
}

static char *accept_strs[] = {
  "text/plain",
  "text/html",
  "application/x-ws",
  "*/*",
  NULL
};

static char *_add_acstr(char* buf){
  char **ptr;
  WSCstring tmp;
  tmp.setString(buf);
  delete buf;
  for (ptr=accept_strs; *ptr != NULL; ptr++){
    WSCstring tmp2;
    tmp2.setString(tmp.getString());
    tmp2.addString("Accept: ");
    tmp2.addString(*ptr);
    tmp2.addString("\r\n");
    tmp.setString(tmp2.getString());
  }
  buf = WSGFstrdup(tmp.getString());
  return buf;
}

static char *_add_string(char* buf,char* name,char* value){
  WSCstring tmp;
  if (buf != NULL){
    tmp.setString(buf);
    delete buf;
    tmp.addString(name);
    tmp.addString(": ");
    tmp.addString(value);
    tmp.addString("\r\n");
    return WSGFstrdup(tmp.getString());
  }else{
    tmp.setString(name);
    tmp.addString(": ");
    tmp.addString(value);
    tmp.addString("\r\n");
    return WSGFstrdup(tmp.getString());
  }
}

static char *_add_uastr(char* buf){
  _init_client_name();
  return _add_string(buf, "User-Agent", _client_name);
}

static char *_add_from(char* buf){
   _init_client_name();
   return _add_string(buf, "From", _user_name_buf);
}

static char *_add_typestr(char* buf){
   return _add_string(buf, "Content-type", WS_POST_CONTENT_TYPE);
}

static char* _add_lenstr(char* buf,long clen){
   char len_str[20];
   sprintf(len_str, "%1d", clen);
   return _add_string(buf, "Content-length", len_str);
}

static char* _add_connect_str(char* buf,FILE* fp,long clen){
  long buf_len=strlen(buf);
  long new_len=buf_len+clen+1;
  char read_buf[1024];
  char* ret_buf = new char[new_len];
  if (ret_buf == NULL){
    return NULL;
  }
  strcpy(ret_buf,buf);
  delete buf;
  long total_len=0;
  long read_len;
  while ((read_len=fread(read_buf, sizeof(char), sizeof(read_buf),fp)) > 0){
    if (read_len+total_len > clen){
      read_len = clen-total_len;
      fprintf(stderr, "Lines too long in _add_connect_str().\n");
    }
    strncpy(&ret_buf[buf_len+total_len], read_buf, read_len);
    total_len += read_len;
  }
  ret_buf[buf_len + clen] = 0;
  return ret_buf;
}

static char* add_linefeed(char* buf){
  if (buf != NULL){
    WSCstring tmp;
    tmp.setString(buf);
    tmp.addString("\r\n");
    delete buf;
    return WSGFstrdup(tmp.getString());
  }else{
    return WSGFstrdup("\r\n");
  }
}

static long _write_data(int sock,char* path){
  long status=0;
  long total_size=0;
  char *buf=new char[strlen(path)+64];
  FILE *fp=NULL;

  if (buf == NULL){
    fprintf(stderr, "Memory allocation failed.\n");
    return WS_MEM_ERR;
  }
  if (_cgi_query){
    sprintf(buf, "POST %s %s\r\n", path, _ws_http_ver);
  }else{
    sprintf(buf, "GET %s %s\r\n", path, _ws_http_ver);
  }
  buf=_add_acstr(buf);
  buf=_add_uastr(buf);
  buf=_add_from(buf);

  if (_cgi_query && _fname_cgi_query != NULL){
    long read_len;
    char read_buf[1024];

    if ((fp=fopen(_fname_cgi_query, "r")) == NULL){
      fprintf(stderr, "Failed to open file:%s.\n",_fname_cgi_query);
      return WS_READ_ERR;
    }
    while ((read_len=fread(read_buf, sizeof(char), sizeof(read_buf),fp)) > 0){
      total_size += read_len;
    }
    rewind(fp);
    buf=_add_typestr(buf);
    buf=_add_lenstr(buf, total_size);
  }
  buf=add_linefeed(buf);
  if (fp != NULL){
    buf = _add_connect_str(buf, fp, total_size);
    fclose(fp);
    if (buf == NULL){
      fprintf(stderr, "Memory allocation failed.\n");
      return WS_MEM_ERR;
    }
  }

  status = WS_NO_ERR;
  int ret = write(sock, buf, (int)strlen(buf));
  if (ret <= 0){
    if (ret == 0){
      fprintf(stderr, "0 bytes written.\n");
    }else if ((errno == ENOTCONN || errno == ECONNRESET || errno == EPIPE)){
      fprintf(stderr, "Network write error.\n");
      status = WS_WRITE_ERR;
    }
  }
  delete buf;

  if (status != WS_NO_ERR){
    WSMFtrace("HTTP: Failed to send requests.\n");
  }
  return status;
}

char *_http_extxt(char* buf,long* bsize,long* fl,char** ctype){
  char* ptr = strchr(buf, '\n');
  char* content_type = NULL;
  long  clen = -1;
  long  cl_len = strlen(WS_CONTENT_LEN);
  long  ct_len = strlen(WS_CONTENT_TYPE);
  long  text_fl = False;

  if (bsize != NULL){
    *bsize = 0;
  }
  if (fl != NULL){
    *fl = False;
  }
  if (ctype != NULL){
    *ctype = NULL;
  }
  char* current_ptr = buf;
  while (ptr != NULL){
    char *parent_ptr = ptr;
    if (parent_ptr != current_ptr && *(--parent_ptr) == '\r'){
      *parent_ptr = 0;
    }else{
      parent_ptr = NULL;
      *ptr = 0;
    }
    if (*current_ptr == 0){
      if (parent_ptr == NULL){
        *ptr = '\n';
      }else{
        *parent_ptr = '\r';
      }
      current_ptr = &ptr[1];
      break;
    }
    if (content_type == NULL &&
        WSGFstrnccmp(current_ptr, WS_CONTENT_TYPE, ct_len) == 0){
      content_type = WSGFstrdup(&current_ptr[ct_len]);
      if (content_type != NULL){
        WSGFtrimSpace(content_type);
        if (ctype != NULL){
          *ctype = WSGFstrdup(content_type);
        }
        if (WSGFstrnccmp(content_type, "text/", 5) == 0){
          if (strcmp(&content_type[5], "html") == 0){
            if (fl != NULL){
              *fl = True;
            }
          }
          text_fl = True;
        }
      }
    }else
    if (clen == -1 && WSGFstrnccmp(current_ptr,WS_CONTENT_LEN,cl_len) == 0){
      char *tstr = WSGFstrdup(&current_ptr[cl_len]);
      if (tstr != NULL){
        WSGFtrimSpace(tstr);
        clen = atoi(tstr);
        delete tstr;
      }
    }
    if (parent_ptr == NULL){
      *ptr = '\n';
    }else{
      *parent_ptr = '\r';
    }
    current_ptr = &ptr[1];
    ptr = strchr(current_ptr, '\n');
  }
  if (content_type != NULL){
    delete content_type;
  }
  if (text_fl){
    long buf_len = strlen(current_ptr);
    char *return_buf;

    if (clen == (-1)){
      clen = buf_len;
      return_buf = new char[buf_len+1];
    }else{
      return_buf = new char[clen+1];
    }

    if (return_buf == NULL){
      WSMFtrace("Memory allocation failed.\n");
      return NULL;
    }
    if (buf_len <= clen){
      memcpy(return_buf, current_ptr, clen);
    }else{
      while (buf_len > clen){
      if (*current_ptr == '\r' || *current_ptr == '\n'){
        current_ptr++;
        buf_len--;
      }else{
        break;
      }
    }
    memcpy(return_buf, current_ptr, clen);
  }
  return_buf[clen] = 0;
  if (bsize != NULL) *bsize = (clen+1);
    return return_buf;
  }else if (clen != (-1)){
    char *return_buf= new char[clen+1];

    if (return_buf == NULL){
       WSMFtrace("Memory allocation failed.\n");
       return NULL;
    }
    memcpy(return_buf, current_ptr, clen);
    return_buf[clen] = 0;
    if (bsize != NULL) *bsize = (clen+1);
    return return_buf;
  }
  return NULL;
}

static long _read_data(int sock,char** pbuffer,long* bsize){
  long  size = WS_READ_SIZE;
  long  len  = 0;
  char* buf = new char[size];

  memset(buf,0,sizeof(char)*size);
  if (bsize != NULL){
    *bsize = 0;
  }
  *pbuffer = NULL;
  if (buf == NULL){
    WSMFtrace("Memory allocation failed.\n");
    return WS_MEM_ERR;
  }

  long  status = WS_NO_ERR;
  long  eof_fl = False;
  while (status == WS_NO_ERR && !eof_fl){
    long read_len;
    if (size - len < WS_READ_MIN_SIZE){
      size += WS_READ_SIZE;
      char* new_buf = new char[size];
      memcpy(new_buf,buf,size-WS_READ_SIZE);
      delete buf;
      buf = new_buf;
    }
    read_len = read(sock, &buf[len], size-len-1);
    if (read_len <= 0){
      if (read_len < 0 && (errno == ENOTCONN || errno == ECONNRESET ||
          errno == EPIPE)){
        WSMFtrace("Network read error.\n");
        status = WS_READ;
      }else if (read_len < 0){
        WSMFtrace("Network error.\n");
        status = WS_NET_ERR;
      }
      eof_fl = True;
    }else{
      len += read_len;
    }
  }

  if (status == WS_NO_ERR){
    buf[len] = 0;
    *pbuffer = buf;
    if (bsize != NULL){
      *bsize = (len+1);
    }
  }else{
    if (buf != NULL){
      delete buf;
    }
    WSMFtrace("HTTP: error encountered in receiving responses.\n");
  }
  return status;
}

static long _get_url_data(char* url,char** protocol,char** host,long* port,char** path){
  char *ptr = strchr(url, ':');
  if (ptr == NULL){
    return WS_FORMAT_ERR;
  }

  *protocol = NULL;
  *host = NULL;
  *path = NULL;

  *ptr = 0;
  *protocol = WSGFstrdup(url);
  *ptr++ = ':';
  if (strncmp(ptr, "//", 2) != 0){
    *path = WSGFstrdup(ptr);
    *host = WSGFstrdup("localhost");
  }else{
    char *port_ptr = NULL;
    char *host_ptr = &ptr[2];

    if ((ptr=strchr(host_ptr, '/')) == NULL){
      *path = WSGFstrdup("");
    }else{
      *path = WSGFstrdup(ptr);
      *ptr = 0;
    }
    if ((port_ptr=strchr(host_ptr, ':')) == NULL){
      *host = WSGFstrdup(host_ptr);
    }else{
      *port_ptr = 0;
      *port = (long)atoi(&port_ptr[1]);
      *host = WSGFstrdup(host_ptr);
      *port_ptr = ':';
      if (*port <= 0){
        return WS_FORMAT_ERR;
      }
    }
    if (ptr != NULL){
      *ptr = '/';
    }
  }
  return WS_NO_ERR;
}

void _update_cache(char* url_name,char* rbuf,char* ctype, long rbuf_size, long fl){
  if (rbuf == NULL){
    rbuf = WSGFstrdup("");
  }
  if (ctype == NULL){
    ctype = WSGFstrdup("");
  }
  char *url_base_name = _get_url_name(url_name);
  WSCurlCache *url;

  _init_cache();
  if (url_base_name == NULL){
    return;
  }
  for (url=_last_url_cache; url != NULL; url=url->parent){
    if (strcmp(url_base_name, url->url_base_name) == 0){
      break;
    }
  }
  if (url != NULL){
    _remove_cache(url);
    delete url;
  }else{
    if (_current_url_cache >= _max_url_cache){
      url = _first_url_cache;
      _remove_cache(url);
      if (url != NULL){
        delete url;
      }
    }
  }
  url = new WSCurlCache;
  if (url == NULL){
    WSMFtrace("Memory Allocate error\n");
    delete url_base_name;
    return;
  }
  url->fl = fl;
  url->rbuf_size = rbuf_size;
  url->rbuf = WSGFstrdup(rbuf);
  url->content_type = WSGFstrdup(ctype);
  url->url_base_name = url_base_name;
  _add_cache(_last_url_cache, NULL, url);
}

long WSGFloadRemoteFile(char* url,char** pbuffer,char** ctype,
                         long* bsize,long* fl,long reload){
  char* buf = NULL;
  char* proxy_host = NULL;
  struct WSCurlCache *curl = NULL;

  if (_http_initialized == 0){
    char* httpp = getenv("WSHTPROXY");
    if (httpp != NULL){
      strcpy(_http_proxy,httpp);
    }
    _http_initialized = 1;
  }
  if (bsize != NULL){
    *bsize = 0;
  }
  if (fl != NULL){
    *fl = False;
  }
  *pbuffer = NULL;

  if (!reload){
    curl = _search_cache(url, True);
  }
  if (curl != NULL && curl->rbuf_size > 0 && curl->rbuf != NULL){
    *pbuffer = new char[curl->rbuf_size];
    memset(*pbuffer,0,curl->rbuf_size);
    if (*pbuffer == NULL){
      WSMFtrace("Memory allocate error!\n");
    }
    memcpy(*pbuffer, curl->rbuf, curl->rbuf_size);
    *bsize = curl->rbuf_size;
    if (ctype != NULL){
      *ctype = WSGFstrdup(curl->content_type);
    }
    *fl = curl->fl;
    return True;
  }

  char* protocol = NULL;
  char* host = NULL;
  char* path = NULL;
  char  port_str[20];
  port_str[0] = 0;
  long  port = 0;
  long  proxy_port = 0;
  long status = _get_url_data(url, &protocol, &host, &port, &path);
  if (status != WS_NO_ERR){
    if (host != NULL){
      delete host;
    }
    if (protocol != NULL){
      delete protocol;
    }
    if (path != NULL){
      delete path;
    }
    WSMFtrace("Invalid URL format. <%s>\n", url);
    return False;
  }

  long  size = 0;
  int   sock = 0;
  if (WSGFstricmp(protocol, "http") == 0){
    long len=strlen(path);
    if (path[len-1] == '/'){
      char *work = new char[len+strlen(_default_file)+1];
      memset(work,0, len+strlen(_default_file)+1);
      if (work == NULL){
        WSMFtrace("Memory allocate error.\n");
        return False;
      }
      sprintf(work, "%s%s", path, _default_file);
      delete path;
      path = work;
    }
    if (port == 0){
      port = 80;
    }

    if (*_http_proxy == 0){
      sprintf(port_str, "%1d", port);
      status = _connect_to(host, port, &sock);
    }else{
      proxy_host = _get_proxy_info(_http_proxy, 80, &proxy_port);
      sprintf(port_str, "%1d", proxy_port);
      if (proxy_host == NULL){
        status = -1;
      }else{
        status = _connect_to(proxy_host, proxy_port, &sock);
      }
    }
    if (status == WS_INTR){
      WSMFtrace("HTTP: Connection interrupted.\n");
    }else if (status < 0){
      if (*_http_proxy == 0){
        WSMFtrace("HTTP: Failed to connect to %s%s%s.\n", host,
              port==80 ? "" : ":", port==80 ? "" : port_str);
      }else{
        WSMFtrace("HTTP: Failed to connect to %s%s%s.\n", proxy_host,
                  proxy_port==80 ? "" : ":", proxy_port==80 ? "" : port_str);
      }
    }else if (status == WS_NO_ERR){
//      if (_write_data(sock, (*_http_proxy==0 ? path : url)) == WS_NO_ERR){
      if (_write_data(sock,url) == WS_NO_ERR){
        if ((status=_read_data(sock, &buf, &size)) == WS_NO_ERR &&
             buf != NULL && *buf != 0){
          *pbuffer = _http_extxt(buf, bsize, fl,ctype);
          delete buf;
        }else if (status == WS_INTR){
          WSMFtrace("HTTP: connection longerrupted.\n");
        }else if (buf == NULL || *buf == 0){
          WSMFtrace( "HTTP: Failed to retrieve file from the server.\n");
          WSMFtrace( "Can not get %s.\n", url);
        }else{
          if (*_http_proxy == 0){
            WSMFtrace("HTTP: Failed to talk to %s%s%s.\n",
                   host, port==80 ? "" : ":", port==80 ? "" : port_str);
          }else{
            WSMFtrace("HTTP: Failed to talk to %s%s%s.\n",
                   proxy_host, proxy_port==80 ? "" : ":",
                   proxy_port==80 ? "" : port_str);
          }
        }
      }
      close(sock);
    }else{
      if (*_http_proxy == 0){
        WSMFtrace("HTTP: Failed to connect to %s%s%s.\n", host,
               port==80 ? "" : ":", port==80 ? "" : port_str);
      }else{
        WSMFtrace("HTTP: Failed to connect to %s%s%s.\n", proxy_host,
                proxy_port==80 ? "" : ":", proxy_port==80 ? "" : port_str);
      }
    }
    if (status == WS_NO_ERR){
      _update_cache(url, *pbuffer,
      ctype==(char**)NULL ? (char*)NULL : *ctype,*bsize, *fl);
    }
  }else if (WSGFstricmp(protocol, "ftp") == 0){
    WSMFtrace("Unsupported the ftp protocol.\n");
  }else{
    WSMFtrace("Unsupported the '%s' protocol.\n", url);
  }
  if (host != NULL){
    delete host;
  }
  if (path != NULL){
    delete path;
  }
  if (protocol != NULL){
    delete protocol;
  }
  if (proxy_host != NULL){
    delete proxy_host;
  }
  return True;
}

#endif //NO_SOCKET
