//
// Copyright (C) 1999-2006 WideStudio/MWT Project Team
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//

#include "xunicode.h"

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

static char* xu_lang = NULL;
static char xu_language[3] = {0, 0, 0};

char* XUGetLang(){
  if (xu_lang == NULL){
    return "C";
  }
  return xu_lang;
} 

static int xu_locale_encoding = XU_CONV_NONE;
BOOL xunicode_enable = FALSE;
int XUGetLocale(){
  return xu_locale_encoding;
}
#ifdef WIN32
#define Bool char
#define True  1
#define False 0
#else
#include <X11/Xlocale.h>
#include <locale.h>




/* link list */
static XUInfoDisplay* xu_info_display = NULL;
static XUInfoFontSetting* xu_info_font_setting = NULL;

/* font set */
static int xu_fontset_count = 0;
static int xu_fontset_encoding[] = {0, 0, 0, 0, 0};
#endif

#ifndef WIN32
/* charsets */
static char* xu_charset[XU_FONT_COUNT] =
{
  NULL,
  "iso8859-1",
  "iso8859-2",
  "iso8859-3",
  "iso8859-4",
  "iso8859-5",
  "iso8859-6",
  "iso8859-7",
  "iso8859-8",
  "iso8859-9",
  "iso8859-10",
  NULL,
  NULL,
  "iso8859-13",
  "iso8859-14",
  "iso8859-15",
  "iso10646-1",
  "koi8-r",
  "jisx0201.1976-0",
  "jisx0208.1983-0",
  "ksc5601.1987-0",
  "gb2312.1980-0",
  "big5-0"
};
#endif /*not WIN32*/

/* encodings */
#include "iso8859.h"
#include "koi8r.h"
#include "jis0208.h"
#include "ksc5601.h"
#include "gb2312.h"
#include "big5.h"
#include "cp1251.h"
static XUChar table_rev_latin2 [ 1024];
static XUChar table_rev_koi8r_1[ 1024];
static XUChar table_rev_koi8r_2[ 1024];
static XUChar table_rev_iso8859[65536];
static XUChar table_rev_jis0208[65536];
static XUChar table_rev_ksc5601[65536];
static XUChar table_rev_gb2312 [65536];
static XUChar table_rev_big5   [65536];
static XUChar table_rev_cp1251_1[ 1024];
static XUChar table_rev_cp1251_2[ 1024];


static void XUInitTable();
static void XUInitLocale();
static void XUInitEncoding();
static Bool XUIsEncoding(const char* enc);
static int XUGetISO8859();
#ifndef WIN32
static void XUInitFontSet();
static void XUInitSetting();
static Bool XUReadSetting(const char* fn);
static void XUSetSetting(char* data);
static void XUStripSpace(char* text);
#endif

char* _lang_ptr = NULL;
void XUInit()
{
  if(xunicode_enable) return;
  
#ifdef WIN32
  xunicode_enable = TRUE;
  long lid = GetSystemDefaultLCID();
  if ((lid & 0x3ff) == LANG_JAPANESE){
    xu_locale_encoding = XU_CONV_SJIS;
    xu_lang = "ja_JP.sjis";
  }else if ((lid & 0x3ff) == LANG_KOREAN){
    xu_locale_encoding = XU_CONV_EUCKR;
    xu_lang = "ko_KR.euckr";
  }else if ((lid & 0x3ff) == LANG_RUSSIAN){
    xu_locale_encoding = XU_CONV_CP1251;
    xu_lang = "ru_RU";
  }else if ((lid & 0x3ff) == LANG_CHINESE){
    if (SUBLANGID(lid) == SUBLANG_CHINESE_SIMPLIFIED){
      xu_locale_encoding = XU_CONV_EUCCN;
      xu_lang = "zh_CN";
    }else if (SUBLANGID(lid) == SUBLANG_CHINESE_TRADITIONAL){
      xu_locale_encoding = XU_CONV_BIG5;
      xu_lang = "zh_TW";
    }
  }else{
    xu_locale_encoding = XU_CONV_LOCALE;
  }
#ifndef _WSWINCE
  char* xu_lang = getenv("LANG");
#else
  char* xu_lang = "";
#endif
  if(xu_lang){
    xu_locale_encoding = XU_CONV_NONE;
    XUInitLocale();
  }
//--- [20041020]
  else if ( xu_locale_encoding == XU_CONV_LOCALE){
#ifndef _WSWINCE
    char* country;
    long lid = GetSystemDefaultLCID();
    _lang_ptr = (char*)malloc(11*sizeof(char));
    lcid2locale(lid, &country);
    sprintf(_lang_ptr,"LANG=%s",country);

    // Here corrections can be included

    // All spanish countries are translated
    // into es_ES
    if (!strncmp(country,"es",2)){
      free(_lang_ptr);
      putenv("LANG=es_ES");

    // After all corrections, this is the default

    }else{
      putenv(_lang_ptr);
    }
    
    free(country);
#endif
    xu_locale_encoding = XU_CONV_NONE;
    XUInitLocale();
  }
//---
//printf("XXZZlang encoding %d\n",xu_locale_encoding);
#else
  xunicode_enable = True;
  XUInitLocale();
  XUInitSetting();
#endif
  XUInitTable();
extern void WSGFsetSystemDefaultEncoding(long);
  WSGFsetSystemDefaultEncoding(xu_locale_encoding);
}


static void XUInitTable()
{
  int i, j;
  XUChar c;
  XUChar c1;
  XUChar c2;
  
  for(i = 0; i < 1024; i++) {
    table_rev_latin2 [i] = 0;
    table_rev_koi8r_1[i] = 0;
    table_rev_koi8r_2[i] = 0;
    table_rev_cp1251_1[i] = 0;
    table_rev_cp1251_2[i] = 0;
  }
  
  for(i = 0; i < 128; i++) {
    c = table_iso8859[1].data[i];
    if(c) table_rev_latin2[c] = i + 0x0080;
  }
  
  for(i = 0; i < 128; i++) {
    c = table_koi8r[i];
    if(0x0080 <= c && c < 0x0480) {
      table_rev_koi8r_1[c - 0x0080] = i + 0x0080;
    } else if(0x2200 <= c && c < 0x2600) {
      table_rev_koi8r_2[c - 0x2200] = i + 0x0080;
    }
    c = table_cp1251[i];
    if(0x0080 <= c && c < 0x0480) {
      table_rev_cp1251_1[c - 0x0080] = i + 0x0080;
    } else if(0x2200 <= c && c < 0x2600) {
      table_rev_cp1251_2[c - 0x2200] = i + 0x0080;
    }
  }
  
  for(i = 0; i < 65536; i++) table_rev_iso8859[i] = 0;
  for(i = 0; i < 65536; i++) table_rev_jis0208[i] = 0;
  for(i = 0; i < 65536; i++) table_rev_ksc5601[i] = 0;
  for(i = 0; i < 65536; i++) table_rev_gb2312 [i] = 0;
  for(i = 0; i < 65536; i++) table_rev_big5   [i] = 0;

  for(i = 9; i >= 1; i--) {
    c = (i << 8) + 128;
    for(j = 0; j < 128; j++, c++) {
      table_rev_iso8859[table_iso8859[i - 1].data[j]] = c;
    }
  }
  
  c1 = 0x21;
  c2 = 0x21;
  for(i = 0; i < MAX_JIS_CHARS; i++) {
    c = table_jis0208[i];
    if(c && table_rev_jis0208[c] == 0) table_rev_jis0208[c] = (c1 << 8) + c2;
    c2++;
    if(c2 == 0x7f) {
      c2 = 0x21;
      c1++;
    }
  }
  
  c1 = 0x21;
  c2 = 0x21;
  for(i = 0; i < 8178; i++) {
    c = table_gb2312[i];
    if(c) table_rev_gb2312[c] = (c1 << 8) + c2;
    c2++;
    if(c2 == 0x7f) {
      c2 = 0x21;
      c1++;
    }
  }
  
  c1 = 0xa1;
  c2 = 0x40;
  for(i = 0; i < 13973; i++) {
    c = table_big5[i];
    if(c) table_rev_big5[c] = (c1 << 8) + c2;
    c2++;
    if(c2 == 0x7f) {
      c2 = 0xa1;
    } else if(c2 == 0xff) {
      c2 = 0x40;
      c1++;
    }
  }
  
  c1 = 0x81;
  c2 = 0x41;
  for(i = 0; i < 12816; i++) {
    c = table_ksc5601[i];
    if(c) table_rev_ksc5601[c] = (c1 << 8) + c2;
    c2++;
    if(c2 == 0x5b) {
      c2 = 0x61;
    } else if(c2 == 0x7b) {
      c2 = 0x81;
    } else if(c2 == 0xff) {
      c2 = 0x41;
      c1++;
    }
  }
  
  c1 = 0xca;
  c2 = 0xa1;
  for(i = 0; i < 4888; i++) {
    c = table_ksc5601_hanja[i];
    if(c) table_rev_ksc5601[c] = (c1 << 8) + c2;
    c2++;
    if(c2 == 0xff) {
      c2 = 0xa1;
      c1++;
    }
  }
}


static void XUInitLocale()
{
#ifndef _WSWINCE
  xu_lang = getenv("LANG");
#else
  xu_lang = NULL;
#endif
  if(!xu_lang) xu_lang = "C";
  strncpy(xu_language, xu_lang, 2);
  
  XUInitEncoding();
  
#ifndef WIN32
  if(setlocale(LC_ALL, "")) XUInitFontSet();
  XSetLocaleModifiers("");
#endif // WIN32
}


static void XUInitEncoding()
{
#if 0
  if(XUIsEncoding("utf8")) {
    xu_locale_encoding = XU_CONV_UTF8;
  } else if(XUIsEncoding("sjis") || XUIsEncoding("shiftjis")) {
    xu_locale_encoding = XU_CONV_SJIS;
  } else if(XUIsEncoding("eucjp")) {
    xu_locale_encoding = XU_CONV_EUCJP;
  } else if(XUIsEncoding("euckr")) {
    xu_locale_encoding = XU_CONV_EUCKR;
  } else if(XUIsEncoding("euccn")) {
    xu_locale_encoding = XU_CONV_EUCCN;
  } else if(XUIsEncoding("big5")) {
    xu_locale_encoding = XU_CONV_BIG5;
  } else if(XUIsEncoding("koi8r")) {
    xu_locale_encoding = XU_CONV_KOI8R;
  } else if(XUIsEncoding("iso8859") || XUIsEncoding("dis8859")) {
    xu_locale_encoding = XUGetISO8859();
  }
  if(xu_locale_encoding != XU_CONV_NONE) return;
  
  if(!strcmp(xu_language, "cs")) xu_locale_encoding = XU_CONV_ISO8859(2);
  if(!strcmp(xu_language, "he")) xu_locale_encoding = XU_CONV_ISO8859(8);
  if(!strcmp(xu_language, "hr")) xu_locale_encoding = XU_CONV_ISO8859(2);
  if(!strcmp(xu_language, "hu")) xu_locale_encoding = XU_CONV_ISO8859(2);
  if(!strcmp(xu_language, "pl")) xu_locale_encoding = XU_CONV_ISO8859(2);
  if(!strcmp(xu_language, "ro")) xu_locale_encoding = XU_CONV_ISO8859(2);
  if(!strcmp(xu_language, "ru")) xu_locale_encoding = XU_CONV_KOI8R;
  if(!strcmp(xu_language, "ja")) xu_locale_encoding = XU_CONV_EUCJP;
  if(!strcmp(xu_language, "ko")) xu_locale_encoding = XU_CONV_EUCKR;
  if(!strcmp(xu_language, "zh")) xu_locale_encoding = XU_CONV_EUCCN;
  if(xu_locale_encoding != XU_CONV_NONE) return;
  
  xu_locale_encoding = XU_CONV_ISO8859(1);
#endif
//--- [20041020]
  // Encodings for languages with Euro symbol
  if(!strcmp(xu_lang, "es_ES")) xu_locale_encoding = -1; // Spanish-Castilian
  if(!strcmp(xu_lang, "ca_ES")) xu_locale_encoding = -1; // Catalan/Valencian (Spain)
  if(!strcmp(xu_lang, "eu_ES")) xu_locale_encoding = -1; // Basque (Spain)
  if(!strcmp(xu_lang, "fr_FR")) xu_locale_encoding = -1; // French (France)
  if(!strcmp(xu_lang, "fr_BE")) xu_locale_encoding = -1; // French (Belgium)
  if(!strcmp(xu_lang, "fr_LU")) xu_locale_encoding = -1; // French (Luxembourg)
  if(!strcmp(xu_lang, "en_GB")) xu_locale_encoding = -1; // English (UK)
  if(!strcmp(xu_lang, "en_IE")) xu_locale_encoding = -1; // English (Ireland)
  if(!strcmp(xu_lang, "de_DE")) xu_locale_encoding = -1; // German (Germany)
  if(!strcmp(xu_lang, "de_AT")) xu_locale_encoding = -1; // German (Austria)
  if(!strcmp(xu_lang, "de_LU")) xu_locale_encoding = -1; // German (Luxembourg)
  if(!strcmp(xu_lang, "it_IT")) xu_locale_encoding = -1; // Italian
  if(!strcmp(xu_lang, "pt_PT")) xu_locale_encoding = -1; // Portuguese
  if(!strcmp(xu_lang, "nl_NL")) xu_locale_encoding = -1; // Dutch (Netherlands)
  if(!strcmp(xu_lang, "nl_BE")) xu_locale_encoding = -1; // Dutch (Belgium)
  if (xu_locale_encoding != XU_CONV_NONE) return;
//---
extern long WSGFgetEncoding(char*);
  xu_locale_encoding = WSGFgetEncoding(xu_lang);
}

#ifndef WIN32

static void XUInitFontSet()
{
  if(!strcmp(xu_language, "ja")) {
    xu_fontset_count = 3;
    xu_fontset_encoding[0] = XU_FONT_ISO8859(1);
    xu_fontset_encoding[1] = XU_FONT_JIS0201;
    xu_fontset_encoding[2] = XU_FONT_JIS0208;
  } else if(!strcmp(xu_language, "ko")) {
    xu_fontset_count = 2;
    xu_fontset_encoding[0] = XU_FONT_ISO8859(1);
    xu_fontset_encoding[1] = XU_FONT_KSC5601;
  } else if(!strncmp(xu_lang, "zh_TW", 5)) {
    xu_fontset_count = 2;
    xu_fontset_encoding[0] = XU_FONT_ISO8859(1);
    xu_fontset_encoding[1] = XU_FONT_BIG5;
  } else if(!strcmp(xu_language, "zh")) {
    xu_fontset_count = 2;
    xu_fontset_encoding[0] = XU_FONT_ISO8859(1);
    xu_fontset_encoding[1] = XU_FONT_GB2312;
  }
}
#endif //WIN32

/**
  Other -> Unicode
*/

XUChar XUCharEncode(const char* text, int max, int code,int* rlen)
{
  XUChar ret = 0;
  XUChar c1;
  XUChar c2;
  XUChar num;
  *rlen = 1;

  if(max == 0 || !*text) return 0;
  
  if(code == XU_CONV_LOCALE) {
#ifdef WIN32
    if(xu_locale_encoding == XU_CONV_LOCALE) {
      MultiByteToWideChar(CP_ACP, 0, text, max, &ret, 1);
      return ret;
    }
    code = xu_locale_encoding;
#else
    code = xu_locale_encoding;
#endif
  }
  if(code <= XU_CONV_NONE) return '?';
  if(code == XU_CONV_UTF8) return XUutf8CharEncode(text, max,rlen);
  
  c1 = (XUChar)(unsigned char)*text;
  if(c1 < 0x80) return c1;
  if(code <= XU_CONV_ISO8859(15)) {
    ret = table_iso8859[code - XU_CONV_ISO8859(1)].data[c1 - 0x80];
    if(ret) return ret;
    return '?';
  } else if(code == XU_CONV_KOI8R) {
    ret = table_koi8r[c1 - 0x80];
    if(ret) return ret;
    return '?';
  } else if(code == XU_CONV_CP1251) {
    ret = table_cp1251[c1 - 0x80];
    if(ret) return ret;
    return '?';
  }
  if(code == XU_CONV_SJIS) {
    if(0xa1 <= c1 && c1 <= 0xdf) return c1 + 0xfec0;
  }
  if(max == 1) return '?';
  text++;
  
  c2 = (XUChar)(unsigned char)*text;
  if(!c2) return '?';
  if(code == XU_CONV_EUCJP) {
    if(c1 == 0x8e){
      *rlen = 2;
      return c2 + 0xfec0;
    }
//    if(c1 < 0xa1 || c1 > 0xf4) return '?';
    if(c1 < 0xa1 || c1 > 0xfc) return '?';
    if(c2 < 0xa1 || c2 > 0xfe) return '?';
    *rlen = 2;
    ret = table_jis0208[(c1 - 0xa1) * 94 + (c2 - 0xa1)];
  } else if(code == XU_CONV_SJIS) {
    if(c1 == 0x80 || c1 == 0xa0 || c1 > 0xfc) return '?';
    if(c2  < 0x40 || c2 == 0x7f || c2 > 0xfc) return '?';
    if(c1 >= 0xe0) c1 -= 0x40;
    if(c2 > 0x7e) c2--;
    *rlen = 2;
    num = (c1 - 0x81) * 188 + (c2 - 0x40);
    if(num < MAX_JIS_CHARS) ret = table_jis0208[num];
  } else if(code == XU_CONV_EUCKR) {
    if(0x81 <= c1 && c1 <= 0xc8) {
      if(c2 < 0x41 || c2 > 0xfe) return '?';
      if(0x5a < c2 && c2 < 0x61) return '?';
      if(0x7a < c2 && c2 < 0x81) return '?';
      if(c2 > 0x7a) c2 -= 6;
      if(c2 > 0x5a) c2 -= 6;
      *rlen = 2;
      ret = table_ksc5601[(c1 - 0x81) * 178 + (c2 - 0x41)];
    } else if(0xca <= c1 && c1 <= 0xfd) {
      if(c2 < 0xa1 || c2 > 0xfe) return '?';
      *rlen = 2;
      ret = table_ksc5601_hanja[(c1 - 0xca) * 94 + (c2 - 0xa1)];
    }
  } else if(code == XU_CONV_EUCCN) {
    if(c1 < 0xa1 || c1 > 0xf7) return '?';
    if(c2 < 0xa1 || c2 > 0xfe) return '?';
    *rlen = 2;
    ret = table_gb2312[(c1 - 0xa1) * 94 + (c2 - 0xa1)];
  } else if(code == XU_CONV_BIG5) {
    if(c1 < 0xa1 || c1 > 0xf9) return '?';
    if(c2 < 0x40 || c2 > 0xfe) return '?';
    if(0x7e < c2 && c2 < 0xa1) return '?';
    if(c2 > 0x7e) c2 -= 0x22;
    *rlen = 2;
    ret = table_big5[(c1 - 0xa1) * 157 + (c2 - 0x40)];
  }
  if(ret) return ret;
  
  *rlen = 1;
  return '?';
}


/**
  Unicode -> Other
*/

int XUCharDecode(char* dest, int max, XUChar ch, int code)
{
  XUChar ret = 0;
  XUChar r1;
  XUChar r2;
  XUChar i;
  XUChar* table;
  
  if(code == XU_CONV_LOCALE) {
#ifdef WIN32
    if(xu_locale_encoding == XU_CONV_LOCALE) {
      WideCharToMultiByte(CP_ACP, 0, &ch, 1, dest, max, NULL, NULL);
      return WideCharToMultiByte(CP_ACP, 0, &ch, 1, NULL, 0, NULL, NULL);
    }
    code = xu_locale_encoding;
#else
    code = xu_locale_encoding;
#endif
  } else if(code == XU_CONV_NONE) {
    if(max >= 1) dest[0] = '?';
    return 1;
  } else if(code == XU_CONV_UTF8) {
    return XUutf8CharDecode(dest, max, ch);
  }
  
  if(ch < 0x0080) {
    if(max >= 1) dest[0] = (char)ch;
    return 1;
  }
  if(code <= XU_CONV_ISO8859(15)) {
    if(ch < 0x00a1) {
      ret = ch;
    } else {
      table = table_iso8859[code - XU_CONV_ISO8859(1)].data;
      for(i = 0x21; i < 0x80; i++) {
        if(ch == table[i]) {
          ret = i + 0x80;
          break;
        }
      }
    }
    if(!ret) ret = '?';
    if(max >= 1) dest[0] = (char)ret;
    return 1;
  } else if(code == XU_CONV_KOI8R) {
    if(0x0080 <= ch && ch < 0x0480) {
      ret = table_rev_koi8r_1[ch - 0x0080];
    } else if(0x2200 <= ch && ch < 0x2600) {
      ret = table_rev_koi8r_2[ch - 0x2200];
    }
    if(!ret) ret = '?';
    if(max >= 1) dest[0] = (char)ret;
    return 1;
  } else if(code == XU_CONV_CP1251) {
    if(0x0080 <= ch && ch < 0x0480) {
      ret = table_rev_cp1251_1[ch - 0x0080];
    } else if(0x2200 <= ch && ch < 0x2600) {
      ret = table_rev_cp1251_2[ch - 0x2200];
    }
    if(!ret) ret = '?';
    if(max >= 1) dest[0] = (char)ret;
    return 1;
  }
  
  if(code == XU_CONV_EUCJP) {
    if(0xff61 <= ch && ch <= 0xff9f) {
      ret = (ch - 0xfec0) + 0x8e00;
    } else {
      ret = table_rev_jis0208[ch];
      if(ret) ret += 0x8080;
    }
  } else if(code == XU_CONV_SJIS) {
    if(0xff61 <= ch && ch <= 0xff9f) {
      if(max >= 1) dest[0] = (char)(ch - 0xfec0);
      return 1;
    } else {
      ret = table_rev_jis0208[ch];
      r1 = (ret >> 8  ) - 0x21;
      r2 = (ret & 0xff) - 0x21;
      if(r1 & 1) r2 += 94;
      r1 = (r1 >> 1) + 0x81;
      if(r1 > 0x9f) r1 += 0x40;
      r2 += 0x40;
      if(r2 > 0x7e) r2++;
      ret = (r1 << 8) + r2;
    }
  } else if(code == XU_CONV_EUCKR) {
    ret = table_rev_ksc5601[ch];
  } else if(code == XU_CONV_EUCCN) {
    ret = table_rev_gb2312[ch];
    if(ret) ret += 0x8080;
  } else if(code == XU_CONV_BIG5) {
    ret = table_rev_big5[ch];
  }
  if(!ret) {
    if(max >= 1) dest[0] = '?';
    return 1;
  }
  
  if(max >= 2) {
    dest[0] = (char)(ret >> 8);
    dest[1] = (char)(ret & 0xff);
  }
  return 2;
}


int XUCharLen(const char* text, int max, int code)
{
  XUChar c1;
  XUChar c2;
  
  if(max == 0 || !*text) return 1;
  
  if(code == XU_CONV_LOCALE) {
#ifdef WIN32
    if(xu_locale_encoding == XU_CONV_LOCALE) {
      return MultiByteToWideChar(CP_ACP, 0, text, max, NULL, 0);
    }
    code = xu_locale_encoding;
#else
    code = xu_locale_encoding;
#endif
  }
  if(code == XU_CONV_UTF8  ) return XUutf8CharLen(text, max);
  if(code <= XU_CONV_KOI8R ) return 1;
  
  c1 = (XUChar)(unsigned char)*text;
  if(code == XU_CONV_SJIS) {
    if(0xa1 <= c1 && c1 <= 0xdf) return 1;
  }
  if(max == 1) return 1;
#if 1 //added by hirabayashi
  if(c1 < 0x80) return 1;
#endif
  text++;
  
  c2 = (XUChar)(unsigned char)*text;
  if(!c2) return 1;
  if(code == XU_CONV_EUCJP) {
    if(c1 == 0x8e) return 2;
//    if(c1 < 0xa1 || c1 > 0xf4) return 1;
    if(c1 < 0xa1 || c1 > 0xfc) return 1;
    if(c2 < 0xa1 || c2 > 0xfe) return 1;
    return 2;
  } else if(code == XU_CONV_SJIS) {
    if(c1 == 0x80 || c1 == 0xa0 || c1 > 0xfc) return 1;
    if(c2  < 0x40 || c2 == 0x7f || c2 > 0xfc) return 1;
    return 2;
  } else if(code == XU_CONV_EUCKR) {
    if(0x81 <= c1 && c1 <= 0xc8) {
      if(c2 < 0x41 || c2 > 0xfe) return 1;
      if(0x5a < c2 && c2 < 0x61) return 1;
      if(0x7a < c2 && c2 < 0x81) return 1;
      return 2;
    } else if(0xca <= c1 && c1 <= 0xfd) {
      if(c2 < 0xa1 || c2 > 0xfe) return 1;
      return 2;
    }
  } else if(code == XU_CONV_EUCCN) {
    if(c1 < 0xa1 || c1 > 0xf7) return 1;
    if(c2 < 0xa1 || c2 > 0xfe) return 1;
    return 2;
  } else if(code == XU_CONV_BIG5) {
    if(c1 < 0xa1 || c1 > 0xf9) return 1;
    if(c2 < 0x40 || c2 > 0xfe) return 1;
    if(0x7e < c2 && c2 < 0xa1) return 1;
    return 2;
  }
  
  return 1;
}


/**
  UTF-8 -> Unicode
*/

XUChar XUutf8CharEncode(const char* text, int max,int* rlen)
{
  XUChar c[6];
  *rlen = 1;

  if(max == 0 || !*text) return 0;
  
  c[0] = (XUChar)(unsigned char)*text;
  if(max == 1 || c[0] < 0xc0 || c[0] > 0xfd) return c[0];
  text++;
  
  c[1] = (XUChar)(unsigned char)*text;
  if((c[1] & 0xc0) != 0x80) return c[0];
  c[1] = c[1] & 0x3f;
  if((c[0] & 0xe0) == 0xc0) {
    if(c[0] < 0xc2) return c[0];
    *rlen = 2;
    return ((c[0] & 0x1f) << 6) + c[1];
  }
  if(max == 2) return c[0];
  text++;
  
  c[2] = (XUChar)(unsigned char)*text;
  if((c[2] & 0xc0) != 0x80) return c[0];
  c[2] = c[2] & 0x3f;
  if((c[0] & 0xf0) == 0xe0) {
    if(c[1] < 0x20 && c[0] < 0xe1) return c[0];
    *rlen = 3;
    return ((c[0] & 0x0f) << 12) + (c[1] << 6) + c[2];
  }
  if(max == 3) return c[0];
  text++;
  
  c[3] = (XUChar)(unsigned char)*text;
  if((c[3] & 0xc0) != 0x80) return c[0];
  c[3] = c[3] & 0x3f;
  if((c[0] & 0xf8) == 0xf0) {
    if(c[1] < 0x10 && c[0] < 0xf1) return c[0];
    *rlen = 4;
    return ((c[1] & 0x0f) << 12) + (c[2] << 6) + c[3];
  }
  if(max == 4) return c[0];
  text++;
  
  c[4] = (XUChar)(unsigned char)*text;
  if((c[4] & 0xc0) != 0x80) return c[0];
  c[4] = c[4] & 0x3f;
  if((c[0] & 0xfc) == 0xf8) {
    if(c[1] < 0x08 && c[0] < 0xf9) return c[0];
    *rlen = 5;
    return ((c[2] & 0x0f) << 12) + (c[3] << 6) + c[4];
  }
  if(max == 5) return c[0];
  text++;
  
  c[5] = (XUChar)(unsigned char)*text;
  if((c[5] & 0xc0) != 0x80) return c[0];
  c[5] = c[5] & 0x3f;
  if(c[1] < 0x04 && c[0] < 0xfd) return c[0];
  *rlen = 6;
  return ((c[3] & 0x0f) << 12) + (c[4] << 6) + c[5];
}


/**
  Unicode -> UTF-8
*/

int XUutf8CharDecode(char* dest, int max, XUChar ch)
{
  if(ch < 0x0080) {
    if(max >= 1) dest[0] = (char)ch;
    if(max >= 2) dest[1] = '\0';
    return 1;
  }
  
  if(ch < 0x0800) {
    if(max >= 1) dest[0] = (char)(0xc0 + ((ch >> 6) & 0x001f));
    if(max >= 2) dest[1] = (char)(0x80 + (ch & 0x003f));
    if(max >= 3) dest[2] = '\0';
    return 2;
  }
  
  if(max >= 1) dest[0] = (char)(0xe0 + ((ch >> 12) & 0x000f));
  if(max >= 2) dest[1] = (char)(0x80 + ((ch >>  6) & 0x003f));
  if(max >= 3) dest[2] = (char)(0x80 + (ch & 0x003f));
  if(max >= 4) dest[3] = '\0';
  return 3;
}


int XUutf8CharLen(const char* text, int max)
{
  XUChar c1;
  XUChar c2;
  XUChar ch;
  
  if(max == 0 || !*text) return 1;
  
  c1 = (XUChar)(unsigned char)*text;
  if(max == 1 || c1 < 0xc0 || c1 > 0xfd) return 1;
  text++;
  
  c2 = (XUChar)(unsigned char)*text;
  if((c2 & 0xc0) != 0x80) return 1;
  c2 = c2 & 0x3f;
  if((c1 & 0xe0) == 0xc0) {
    if(c1 < 0xc2) return 1;
    return 2;
  }
  if(max == 2) return 1;
  text++;
  
  ch = (XUChar)(unsigned char)*text;
  if((ch & 0xc0) != 0x80) return 1;
  if((c1 & 0xf0) == 0xe0) {
    if(c2 < 0x20 && c1 < 0xe1) return 1;
    return 3;
  }
  if(max == 3) return 1;
  text++;
  
  ch = (XUChar)(unsigned char)*text;
  if((ch & 0xc0) != 0x80) return 1;
  if((c1 & 0xf8) == 0xf0) {
    if(c2 < 0x10 && c1 < 0xf1) return 1;
    return 4;
  }
  if(max == 4) return 1;
  text++;
  
  ch = (XUChar)(unsigned char)*text;
  if((ch & 0xc0) != 0x80) return 1;
  if((c1 & 0xfc) == 0xf8) {
    if(c2 < 0x08 && c1 < 0xf9) return 1;
    return 5;
  }
  if(max == 5) return 1;
  text++;
  
  ch = (XUChar)(unsigned char)*text;
  if((ch & 0xc0) != 0x80) return 1;
  if(c2 < 0x04 && c1 < 0xfd) return 1;
  return 6;
}


int XUutf8CharRLen(const char* text, int max)
{
  XUChar c1;
  XUChar c2;
  
  if(max == 0) return 1;
  text--;
  
  c1 = (XUChar)(unsigned char)*text;
  if(max == 1 || (c1 & 0xc0) != 0x80) return 1;
  text--;
  
  c1 = (XUChar)(unsigned char)*text;
  if((c1 & 0xe0) == 0xc0) {
    if(c1 < 0xc2) return 1;
    return 2;
  }
  if(max == 2 || (c1 & 0xc0) != 0x80) return 1;
  text--;
  
  c2 = c1 & 0x3f;
  c1 = (XUChar)(unsigned char)*text;
  if((c1 & 0xf0) == 0xe0) {
    if(c2 < 0x20 && c1 < 0xe1) return 1;
    return 3;
  }
  if(max == 3 || (c1 & 0xc0) != 0x80) return 1;
  text--;
  
  c2 = c1 & 0x3f;
  c1 = (XUChar)(unsigned char)*text;
  if((c1 & 0xf8) == 0xf0) {
    if(c2 < 0x10 && c1 < 0xf1) return 1;
    return 4;
  }
  if(max == 4 || (c1 & 0xc0) != 0x80) return 1;
  text--;
  
  c2 = c1 & 0x3f;
  c1 = (XUChar)(unsigned char)*text;
  if((c1 & 0xfc) == 0xf8) {
    if(c2 < 0x08 && c1 < 0xf9) return 1;
    return 5;
  }
  if(max == 5 || (c1 & 0xc0) != 0x80) return 1;
  text--;
  
  c2 = c1 & 0x3f;
  c1 = (XUChar)(unsigned char)*text;
  if((c1 & 0xfe) == 0xfc) {
    if(c2 < 0x04 && c1 < 0xfd) return 1;
    return 6;
  }
  return 1;
}


int XUStrLen(const XUChar* text)
{
  int ret = 0;
  
  while(*text) {
    text++;
    ret++;
  }
  return ret;
}


/**
  Other -> Unicode(UCS-2)
*/

int XUEncode(XUChar* dest, int max, const char* text, int length, int code)
{
  int ret = 0;
  int chlen;
  
  if(length == 0) {
    if(max > 0) *dest = 0;
    return 0;
  }
  
#ifdef WIN32
  if(code == XU_CONV_LOCALE) {
    if(xu_locale_encoding == XU_CONV_LOCALE) {
      ret = MultiByteToWideChar(CP_ACP, 0, text, length, NULL, 0);
      MultiByteToWideChar(CP_ACP, 0, text, length, dest, max);
      if(ret < max) dest[ret] = L'\0';
      return ret;
    }
  }
#endif

  for(;;) {
    if(length < 0 && !*text) break;
    if(ret < max) {
      *dest = XUCharEncode(text, length, code,&chlen);
      dest++;
    }else{
      chlen = XUCharLen(text, length, code);
    }
    ret++;
    text += chlen;
    if(length > 0) {
      length -= chlen;
      if(length < 1) break;
    }
  }
  
  if(ret < max) *dest = 0;
  return ret;
}


/**
  Unicode(UCS-2) -> Other
*/

int XUDecode(char* dest, int max, const XUChar* text, int length, int code)
{
  int ret = 0;
  int chlen;
  
  if(length == 0) {
    if(max > 0) *dest = 0;
    return 0;
  }
  
#ifdef WIN32
  if(code == XU_CONV_LOCALE) {
    if(xu_locale_encoding == XU_CONV_LOCALE) {
      ret = WideCharToMultiByte(CP_ACP, 0, text, length, NULL, 0, NULL, NULL);
      WideCharToMultiByte(CP_ACP, 0, text, length, dest, max, NULL, NULL);
      if(ret < max) dest[ret] = '\0';
      return ret;
    }
  }
#endif
  
  for(;;) {
    if(length < 0 && !*text) break;
    chlen = XUCharDecode(dest, max, *text, code);
    ret += chlen;
    if(max > 0) {
      dest += chlen;
      max -= chlen;
    }
    text++;
    if(length > 0) {
      length--;
      if(length < 1) break;
    }
  }
  
  if(max > 0) *dest = 0;
  return ret;
}


int XULen(const char* text, int max, int code)
{
  int ret = 0;
  int chlen;
  
  if(max == 0) return 0;
  
#ifdef WIN32
  if(code == XU_CONV_LOCALE) {
    if(xu_locale_encoding == XU_CONV_LOCALE) {
      return MultiByteToWideChar(CP_ACP, 0, text, max, NULL, 0);
    }
  }
#endif
  
  for(;;) {
    if(max < 0 && !*text) break;
    ret++;
    chlen = XUCharLen(text, max, code);
    text += chlen;
    if(max > 0) {
      max -= chlen;
      if(max < 1) break;
    }
  }
  return ret;
}


/**
  UTF-8 -> UCS-2
*/

int XUutf8Encode(XUChar* dest, int max, const char* text, int length)
{
  int ret = 0;
  int chlen;
  
  if(length == 0) {
    if(max > 0) *dest = 0;
    return 0;
  }
  
  for(;;) {
    if(length < 0 && !*text) break;
    if(ret < max) {
      *dest = XUutf8CharEncode(text, length,&chlen);
      dest++;
    }else{
      chlen = XUutf8CharLen(text, length);
    }
    ret++;
    text += chlen;
    if(length > 0) {
      length -= chlen;
      if(length < 1) break;
    }
  }
  
  if(ret < max) *dest = 0;
  return ret;
}


/**
  UCS-2 -> UTF-8
*/

int XUutf8Decode(char* dest, int max, const XUChar* text, int length)
{
  int ret = 0;
  int chlen;
  
  if(length == 0) {
    if(max > 0) *dest = 0;
    return 0;
  }
  
  for(;;) {
    if(length < 0 && !*text) break;
    chlen = XUutf8CharDecode(dest, max, *text);
    ret += chlen;
    if(max > 0) {
      dest += chlen;
      max -= chlen;
    }
    text++;
    if(length > 0) {
      length--;
      if(length < 1) break;
    }
  }
  
  if(max > 0) *dest = 0;
  return ret;
}


int XUutf8Len(const char* text, int max)
{
  int ret = 0;
  int chlen;
  
  if(max == 0) return 0;
  
  for(;;) {
    if(max < 0 && !*text) break;
    ret++;
    chlen = XUutf8CharLen(text, max);
    text += chlen;
    if(max > 0) {
      max -= chlen;
      if(max < 1) break;
    }
  }
  return ret;
}


int XUCodeConv(char* dest, int max, int codeTo, const char* text, int length, int codeFrom)
{
  XUChar* buf;
  int len1;
  int len2;
  
  if(length < 0) length = strlen(text);
  buf = (XUChar*)malloc(sizeof(XUChar) * length);
  len1 = XUEncode(buf, length, text, length, codeFrom);
  len2 = XUDecode(dest, max, buf, len1, codeTo);
  free(buf);
  return len2;
}

//--- [20041020]
bool lcid2locale(long lcid, char **localeid)
{
/*
    This functions translate lcid code from Windows
    into locale code with language_COUNTRY.
    If lcid is located, localeid will have got
    locale code and return will be true, if
    lcid is not located, localeid will have got
    "C" as value and return will be false
*/
    *localeid = (char*)malloc(6*sizeof(char));

switch(lcid)
{
case 0x0401: strcpy(*localeid,"ar_SA"); break; // Arabic (Saudi Arabia) - SAUDI ARABIA
case 0x0801: strcpy(*localeid,"ar_IQ"); break; // Arabic (Iraq) - IRAQ
case 0x0c01: strcpy(*localeid,"ar_EG"); break; // Arabic (Egypt) - EGYPT
case 0x1001: strcpy(*localeid,"ar_LY"); break; // Arabic (Libya) - LYBIAN ARAB JAMAHIRIYA
case 0x1401: strcpy(*localeid,"ar_DZ"); break; // Arabic (Algeria) - ALGERIA
case 0x1801: strcpy(*localeid,"ar_MA"); break; // Arabic (Morocco) - MOROCCO
case 0x1c01: strcpy(*localeid,"ar_TN"); break; // Arabic (Tunisia) - TUNISIA
case 0x2001: strcpy(*localeid,"ar_OM"); break; // Arabic (Oman) - OMAN
case 0x2401: strcpy(*localeid,"ar_YE"); break; // Arabic (Yemen) - YEMEN
case 0x2801: strcpy(*localeid,"ar_SY"); break; // Arabic (Syria) - SYRIAN ARAB REPUBLIC
case 0x2c01: strcpy(*localeid,"ar_JO"); break; // Arabic (Jordan) - JORDAN
case 0x3001: strcpy(*localeid,"ar_LB"); break; // Arabic (Lebanon) - LEBANON
case 0x3401: strcpy(*localeid,"ar_KW"); break; // Arabic (Kuwait) - KUWAIT
case 0x3801: strcpy(*localeid,"ar_AE"); break; // Arabic (U.A.E.) - UNITED ARAB EMIRATES
case 0x3c01: strcpy(*localeid,"ar_BH"); break; // Arabic (Bahrain) - BAHRAIN
case 0x4001: strcpy(*localeid,"ar_QA"); break; // Arabic (Qatar) - QATAR
case 0x0402: strcpy(*localeid,"bg_BG"); break; // Bulgarian - BULGARIA
case 0x0403: strcpy(*localeid,"ca_ES"); break; // Catalan - SPAIN
case 0x0404: strcpy(*localeid,"zh_TW"); break; // Chinese (Taiwan) - TAIWAN, PROVINCE OF CHINA
case 0x0804: strcpy(*localeid,"zh_CN"); break; // Chinese (PRC) - CHINA
case 0x0c04: strcpy(*localeid,"zh_HK"); break; // Chinese (Hong Kong) - HONG KONG
case 0x1004: strcpy(*localeid,"zh_SG"); break; // Chinese (Singapore) - SINGAPORE
case 0x0405: strcpy(*localeid,"cs_CZ"); break; // Czech - CZECH REPUBLIC
case 0x0406: strcpy(*localeid,"da_DK"); break; // Danish - DENMARK
case 0x0407: strcpy(*localeid,"de_DE"); break; // German (Standard) - GERMANY
case 0x0807: strcpy(*localeid,"de_CH"); break; // German (Swiss) - SWITZERLAND
case 0x0c07: strcpy(*localeid,"de_AT"); break; // German (Austrian) - AUSTRIA
case 0x1007: strcpy(*localeid,"de_LU"); break; // German (Luxembourg) - LUXEMBOURG
case 0x1407: strcpy(*localeid,"de_LI"); break; // German (Liechtenstein) - LIECHTENSTEIN
case 0x0408: strcpy(*localeid,"el_GR"); break; // Greek - GREECE
case 0x0409: strcpy(*localeid,"en_US"); break; // English (United States) - UNITED STATES
case 0x0809: strcpy(*localeid,"en_GB"); break; // English (United Kingdom) - UNITED KINGDOM
case 0x0c09: strcpy(*localeid,"en_AU"); break; // English (Australian) - AUSTRALIA
case 0x1009: strcpy(*localeid,"en_CA"); break; // English (Canadian) - CANADA
case 0x1409: strcpy(*localeid,"en_NZ"); break; // English (New Zealand) - NEW ZEALAND
case 0x1809: strcpy(*localeid,"en_IE"); break; // English (Ireland) - IRELAND
case 0x1c09: strcpy(*localeid,"en_ZA"); break; // English (South Africa) - SOUTH AFRICA
case 0x2009: strcpy(*localeid,"en_JM"); break; // English (Jamaica) - JAMAICA
case 0x2409: strcpy(*localeid,"en_??"); break; // English (Caribbean) - ??
case 0x2809: strcpy(*localeid,"en_BZ"); break; // English (Belize) - BELIZE
case 0x2c09: strcpy(*localeid,"en_TT"); break; // English (Trinidad) - TRINIDAD AND TOBAGO
case 0x040a: strcpy(*localeid,"es_ES"); break; // Spanish (Traditional Sort) - SPAIN
case 0x080a: strcpy(*localeid,"es_MX"); break; // Spanish (Mexican) - MEXICO
case 0x0c0a: strcpy(*localeid,"es_ES"); break; // Spanish (Modern Sort) - SPAIN
case 0x100a: strcpy(*localeid,"es_GT"); break; // Spanish (Guatemala) - GUATEMALA
case 0x140a: strcpy(*localeid,"es_CR"); break; // Spanish (Costa Rica) - COSTA RICA
case 0x180a: strcpy(*localeid,"es_PA"); break; // Spanish (Panama) - PANAMA
case 0x1c0a: strcpy(*localeid,"es_DO"); break; // Spanish (Dominican Republic) - DOMINICAN REPUBLIC
case 0x200a: strcpy(*localeid,"es_VE"); break; // Spanish (Venezuela) - VENEZUELA
case 0x240a: strcpy(*localeid,"es_CO"); break; // Spanish (Colombia) - COLOMBIA
case 0x280a: strcpy(*localeid,"es_PE"); break; // Spanish (Peru) - PERU
case 0x2c0a: strcpy(*localeid,"es_AR"); break; // Spanish (Argentina) - ARGENTINA
case 0x300a: strcpy(*localeid,"es_EC"); break; // Spanish (Ecuador) - ECUADOR
case 0x340a: strcpy(*localeid,"es_CL"); break; // Spanish (Chile) - CHILE
case 0x380a: strcpy(*localeid,"es_UY"); break; // Spanish (Uruguay) - URUGUAY
case 0x3c0a: strcpy(*localeid,"es_PY"); break; // Spanish (Paraguay) - PARAGUAY
case 0x400a: strcpy(*localeid,"es_BO"); break; // Spanish (Bolivia) - BOLIVIA
case 0x440a: strcpy(*localeid,"es_SV"); break; // Spanish (El Salvador) - EL SALVADOR
case 0x480a: strcpy(*localeid,"es_HN"); break; // Spanish (Honduras) - HONDURAS
case 0x4c0a: strcpy(*localeid,"es_NI"); break; // Spanish (Nicaragua) - NICARAGUA
case 0x500a: strcpy(*localeid,"es_PR"); break; // Spanish (Puerto Rico) - PUERTO RICO
case 0x040b: strcpy(*localeid,"fi_FI"); break; // Finnish - FINLAND
case 0x040c: strcpy(*localeid,"fr_FR"); break; // French (Standard) - FRANCE
case 0x080c: strcpy(*localeid,"fr_BE"); break; // French (Belgian) - BELGIUM
case 0x0c0c: strcpy(*localeid,"fr_CA"); break; // French (Canadian) - CANADA
case 0x100c: strcpy(*localeid,"fr_CH"); break; // French (Swiss) - SWITZERLAND
case 0x140c: strcpy(*localeid,"fr_LU"); break; // French (Luxembourg) - LUXEMBOURG
case 0x040d: strcpy(*localeid,"he_IL"); break; // Hebrew - ISRAEL
case 0x040e: strcpy(*localeid,"hu_HU"); break; // Hungarian - HUNGARY
case 0x040f: strcpy(*localeid,"is_IS"); break; // Icelandic - ICELAND
case 0x0410: strcpy(*localeid,"it_IT"); break; // Italian (Standard) - ITALY
case 0x0810: strcpy(*localeid,"it_CH"); break; // Italian (Swiss) - SWITZERLAND
case 0x0411: strcpy(*localeid,"ja_JP"); break; // Japanese - JAPAN
case 0x0412: strcpy(*localeid,"ko_KP"); break; // Korean - DEMOCRATIC PEOPLE'S REPUBLIC OF KOREA
case 0x0812: strcpy(*localeid,"ko_KR"); break; // Korean (JoHab) - REPUBLIC OF KOREA
case 0x0413: strcpy(*localeid,"nl_NL"); break; // Dutch (Standard) - NETHERLANDS
case 0x0813: strcpy(*localeid,"nl_BE"); break; // Dutch (Belgian) - BELGIUM
case 0x0414: strcpy(*localeid,"nb_NO"); break; // Norwegian (Bokmal) - NORWAY
case 0x0814: strcpy(*localeid,"nn_NO"); break; // Norwegian (Nynorsk) - NORWAY
case 0x0415: strcpy(*localeid,"pl_PL"); break; // Polish - POLAND
case 0x0416: strcpy(*localeid,"pt_BR"); break; // Portuguese (Brazilian) - BRAZIL
case 0x0816: strcpy(*localeid,"pt_PT"); break; // Portuguese (Standard) - PORTUGAL
case 0x0418: strcpy(*localeid,"ro_RO"); break; // Romanian - ROMANIA
case 0x0419: strcpy(*localeid,"ru_RU"); break; // Russian - RUSSIAN FEDERATION
case 0x041a: strcpy(*localeid,"hr_HR"); break; // Croatian - CROATIA
case 0x0c1a: strcpy(*localeid,"sr_CS"); break; // Serbian - SERBIA AND MONTENEGRO
case 0x041b: strcpy(*localeid,"sk_SK"); break; // Slovak - SLOVAKIA
case 0x041c: strcpy(*localeid,"sq_AL"); break; // Albanian - ALBANIA
case 0x041d: strcpy(*localeid,"sv_SE"); break; // Swedish - SWEDEN
case 0x081d: strcpy(*localeid,"sv_FI"); break; // Swedish (Finland) - FINLAND
case 0x041e: strcpy(*localeid,"th_TH"); break; // Thai - THAILAND
case 0x041f: strcpy(*localeid,"tr_TR"); break; // Turkish - TURKEY
case 0x0421: strcpy(*localeid,"id_ID"); break; // Indonesian - INDONESIA
case 0x0422: strcpy(*localeid,"uk_UA"); break; // Ukrainian - UKRAINE
case 0x0423: strcpy(*localeid,"be_BY"); break; // Belarusian - BELARUS
case 0x0424: strcpy(*localeid,"sl_SI"); break; // Slovenian - SLOVENIA
case 0x0425: strcpy(*localeid,"et_EE"); break; // Estonian - ESTONIA
case 0x0426: strcpy(*localeid,"lv_LV"); break; // Latvian - LATVIA
case 0x0427: strcpy(*localeid,"lt_LT"); break; // Lithuanian - LITHUANIA
case 0x081a: strcpy(*localeid,"sr_CS"); break; // Serbian - SERBIA AND MONTENEGRO
case 0x0429: strcpy(*localeid,"fa_IR"); break; // Farsi - ISLAMIC REPUBLIC OF IRAN
case 0x042d: strcpy(*localeid,"eu_ES"); break; // Basque - SPAIN
case 0x0436: strcpy(*localeid,"af_ZA"); break; // Afrikaans - SOUTH AFRICA
case 0x0438: strcpy(*localeid,"fo_FO"); break; // Faeroese - FAROE ISLANDS
}

if (strlen(*localeid) > 0){
    return true;
}else{
    strcpy(*localeid,"C");
    return false;
}

};
//---
