package jp.ac.dendai.cdl.mori.wikie.main;

import java.io.*;
import java.util.*;

import jp.ac.dendai.cdl.mori.wikie.io.*;

import org.apache.hadoop.conf.*;
import org.apache.hadoop.fs.*;
import org.apache.hadoop.io.*;
import org.apache.hadoop.mapred.*;
import org.apache.hadoop.util.*;

/**
 * メインクラス。
 * このクラスからWik-IEを動作させる。
 * @author Mori
 *
 */
public class WikIE extends Configured implements Tool {
    /**
     * MapperクラスとReducerクラスの設定ファイル。
     */
    public static final String MAPRED_CONF_FILE_PATH = "mapred.conf";
    /**
     * 言語のプレフィックスを記述したファイル。
     */
    public static final String LANG_DAT = "wiki_lang.dat";
    /**
     * Wikipediaプロジェクトのプレフィックスを記述したファイル。
     */
    public static final String PROJECT_DAT = "wiki_project.dat";

    public static final String ENTRY = "entry";
    public static final String LEAF = "leaf";
    public static final String NODE = "node";
    public static final String EDGE = "edge";
    public static final String OTHER = "other";
    public static final String TARGET = "target";
    public static final String REDIRECT = "redirect";
    public static final String HYPERNYM = "hypernym";

    public static final String UTF8 = "UTF8";

    public static final String MEDIAWIKI_ELEMENT = "mediawiki";
    public static final String BASE_ELEMENT = "base";
    public static final String NAMESPACES_ELEMENT = "namespaces";
    public static final String NAMESPACE_ELEMENT = "namespace";
    public static final String PAGE_ELEMENT = "page";
    public static final String TITLE_ELEMENT = "title";
    public static final String ID_ELEMENT = "id";
    public static final String REVISION_ELEMENT = "revision";
    public static final String TIMESTAMP_ELEMENT = "timestamp";
    public static final String CONTRIBUTOR_ELEMENT = "contributor";
    public static final String USERNAME_ELEMENT = "username";
    public static final String IP_ELEMENT = "ip";
    public static final String TEXT_ELEMENT = "text";

    public static final String KEY_ATTRIBUTE = "key";

    public static final int LEAF_KIND = 1;
    public static final int NODE_KIND = 2;
    public static final int REDIRECT_KIND = 3;
    public static final int HYPERNYM_KIND = 4;

    public static int ARTICLE_NS_NUM = 0;
    public static int IMAGE_NS_NUM = 6;
    public static int CATEGORY_NS_NUM = 14;

    public static final String PROP_START_TAG = "wikie.io.startTag";
    public static final String PROP_END_TAG = "wikie.io.endTag";
    public static final String PROP_RESOURCE = "wikie.map.resource";
    public static final String PROP_FUNC = "wikie.job.func";
    public static final String PROP_ISBN_HEADER = "wikie.map.isbnheader";
    public static final String PROP_PROJECT = "wikie.project";
    public static final String PROP_LANG = "wikie.lang";

    public static void main(String[] args) throws Exception {
        ToolRunner.run(new Configuration(), new WikIE(), args);
    }

    @Override
    public int run(String[] args) throws Exception {
        JobConf conf = createJobConf(args);
        if (conf != null) {
            JobClient.runJob(conf);
            return 0;
        }
        return -1;
    }

    /**
     * 実行するJobConfを生成する。
     * @param args　コマンドライン引数
     * @return 生成したJobConf
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public JobConf createJobConf(String[] args)
    throws IOException, ClassNotFoundException {
        JobConf conf = new JobConf(WikIE.class);
        if (args.length >= 3) {
            conf.set(WikIE.PROP_FUNC, args[0]);
            conf.setInputFormat(XMLInputFormat.class);
            conf.setOutputFormat(TextOutputFormat.class);
            conf.setOutputKeyClass(Text.class);
            conf.setOutputValueClass(Text.class);
            FileInputFormat.setInputPaths(conf, new Path(args[1]));
            FileOutputFormat.setOutputPath(conf, new Path(args[2]));
            conf.set(PROP_RESOURCE, args[1]);
            conf.set(PROP_START_TAG, WikIE.PAGE_ELEMENT);
            conf.set(PROP_END_TAG, WikIE.PAGE_ELEMENT);
            conf.set(PROP_PROJECT, createProjectSet(PROJECT_DAT));
            conf.set(PROP_LANG, createProjectSet(LANG_DAT));

            if (args.length >= 4) {
                try {
                    conf.setNumReduceTasks(Integer.parseInt(args[3]));
                }
                catch (NumberFormatException e) {
                    System.out.println("reduceタスク数は整数値を指定してください。");
                    return null;
                }
            }

            if (!setMapRed(conf, MAPRED_CONF_FILE_PATH)) {
                return null;
            }

            return conf;
        }
        else {
            printFunc(MAPRED_CONF_FILE_PATH);
            return null;
        }
    }

    /**
     * 設定ファイルmapred.confからMapperクラスとReducerクラスを設定する。
     * @param conf 実行するJobConf
     * @param confFilePathStr 設定ファイルのパス
     * @return 設定に成功すればtrue, 失敗したらfalse
     * @throws IOException
     * @throws ClassNotFoundException
     */
    private boolean setMapRed(JobConf conf, String confFilePathStr)
    throws IOException, ClassNotFoundException {
        String func = conf.get(WikIE.PROP_FUNC);
        FileInputStream is = new FileInputStream(new File(confFilePathStr));
        BufferedReader reader = new BufferedReader(new InputStreamReader(is, WikIE.UTF8));
        String line = new String();
        Set<String> funcSet = new TreeSet<String>();
        while ((line = reader.readLine()) != null) {
            String[] column = line.split("\t");
            funcSet.add(column[0]);
            if ((column[0]).equals(func)) {
                Class mapper = Class.forName(column[1]);
                Class reducer = Class.forName(column[2]);
                conf.setMapperClass(mapper);
                conf.setReducerClass(reducer);
                reader.close();
                return true;
            }
        }
        reader.close();
        printFunc(confFilePathStr);
        return false;
    }

    /**
     * 使いかたを出力する。
     * @param confFilePathStr　設定ファイルのパス
     * @throws IOException
     */
    private void printFunc(String confFilePathStr)
    throws IOException {
        FileInputStream is = new FileInputStream(new File(confFilePathStr));
        BufferedReader reader = new BufferedReader(new InputStreamReader(is, WikIE.UTF8));
        String line = new String();
        System.out.println("以下のオプションが実行可能です。");
        while ((line = reader.readLine()) != null) {
            String[] column = line.split("\t");
            System.out.println(column[0]);
        }
        reader.close();
        System.out.println("Wik-IEには次のように引数を与えてください（Reduceタスク数は省略可）。");
        System.out.println("機能 データファイルのパス 出力ディレクトリのパス Reduceタスク数");
    }

    /**
     * プロジェクトプレフィックスを読み込んで設定する。
     * @param projectConfFilePath
     * @return プレフィックスをタブ区切りしたString
     * @throws UnsupportedEncodingException
     * @throws IOException
     */
    public static String createProjectSet(String projectConfFilePath)
    throws UnsupportedEncodingException, IOException {
        StringBuffer result = new StringBuffer();
        BufferedReader reader = new BufferedReader(
                new InputStreamReader(new FileInputStream(new File(projectConfFilePath)), WikIE.UTF8));
        String line = new String();
        while ((line = reader.readLine()) != null) {
            result.append(line.trim() + "\t");
        }
        reader.close();
        return result.toString().trim();
    }

    /**
     * 言語プレフィックスを読み込んで設定する。
     * @param langConfFilePath
     * @return プレフィックスをタブ区切りしたString
     * @throws UnsupportedEncodingException
     * @throws IOException
     */
    public static String createlangSet(String langConfFilePath)
    throws UnsupportedEncodingException, IOException {
        StringBuffer result = new StringBuffer();
        BufferedReader reader = new BufferedReader(
                new InputStreamReader(new FileInputStream(new File(langConfFilePath)), WikIE.UTF8));
        String line = new String();
        while ((line = reader.readLine()) != null) {
            result.append(line.trim() + "\t");
        }
        reader.close();
        return result.toString().trim();
    }

}
