package jp.ac.dendai.cdl.mori.wikie.mapred;

import info.bliki.wiki.model.*;

import java.io.*;
import java.util.*;

import javax.xml.parsers.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;
import jp.ac.dendai.cdl.mori.wikie.parser.*;
import jp.ac.dendai.cdl.mori.wikie.util.*;

import org.apache.hadoop.io.*;
import org.apache.hadoop.mapred.*;
import org.xml.sax.*;

/**
 * page要素を扱うMapperの基底クラス。
 * このクラスをオーバーライドしてMapperを作る。
 * @author Mori
 *
 */
public abstract class WMapper extends MapReduceBase
implements Mapper<LongWritable, Text, Text, Text> {
    /**
     * タイトル等の正規化に使うWNormalizer
     */
    protected WNormalizer normalizer;
    /**
     * リンク等の取得に使うWLinkUtils
     */
    protected WLinkUtils linkUtils;
    /**
     * page要素の解析に使うSAXParser
     */
    protected SAXParser parser;
    /**
     * Wiki記法のパースに使うWikiModel。WikiModelはBlikiのライブラリ。
     * <a href="http://matheclipse.org/doc/bliki/index.html">Java Wikipedia API (Bliki engine)のAPI</a>
     */
    protected WikiModel model;

    @Override
    public void configure(JobConf job) {
        super.configure(job);
        try {
            parser = SAXParserFactory.newInstance().newSAXParser();
            Set<String> projectSet = new TreeSet<String>();
            for (String p : job.get(WikIE.PROP_PROJECT).split("\t")) {
                projectSet.add(p);
            }
            Set<String> langSet = new TreeSet<String>();
            for (String l : job.get(WikIE.PROP_LANG).split("\t")) {
                langSet.add(l);
            }
            normalizer = new WNormalizer(job.get(WikIE.PROP_RESOURCE), projectSet, langSet);
            linkUtils = new WLinkUtils(normalizer);
            model = new WikiModel(normalizer.getBaseURL() + "${image}", normalizer.getBaseURL() + "${title}");
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * XMLのpage要素をText型で与えてSAXで解析する。
     * page要素はMapperクラスのmapメソッドが入力として渡されるTextオブジェクトをそのまま渡す。
     * @param text XMLのpage要素
     * @return 解析したHandler
     * @throws IOException
     * @throws SAXException
     */
    protected WPageElementHandler createPageHandler(Text text)
    throws IOException, SAXException {
        WPageElementHandler page = new WPageElementHandler();
        parser.parse(new InputSource(new StringReader(text.toString())), page);
        return page;
    }
}
