package jp.ac.dendai.cdl.mori.wikie.util;

import java.io.*;
import java.net.*;
import java.util.*;
import java.util.Map.*;
import java.util.regex.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;

import org.apache.commons.codec.*;
import org.apache.commons.codec.net.*;
import org.apache.commons.lang.*;
import org.apache.hadoop.conf.*;
import org.apache.hadoop.fs.*;
import org.apache.hadoop.fs.FileSystem;

/**
 * Wikipediaのエントリのタイトルの正規化に関するクラス
 * @author mori
 *
 */
public class WNormalizer {
    /**
     * 言語略記の一覧ページ
     */
    public static final String LANG_URL = "http://en.wikipedia.org/wiki/Special:Export/List_of_Wikipedias";
    /**
     * ベースURL
     */
    private String baseURL;
    /**
     * 名前空間がkey, 番号がvalueのMap
     */
    private Map<String, Integer> nsNumberMap;
    /**
     * 言語略記のSet
     */
    private Set<String> langSet;
    /**
     * プロジェクトプレフィックスのSet
     */
    private Set<String> projectSet;

    public WNormalizer(String sourceFilePathStr, Set<String> projectSet, Set<String> langSet)
    throws IOException, UnsupportedEncodingException {
        this.nsNumberMap = createNSNumberMap(sourceFilePathStr);
        this.langSet = langSet;
        this.projectSet = projectSet;
    }

    /**
     * 処理するXMLファイルのヘッダー部分から各種設定を行う
     * @param xmlFilePath
     * @return 名前空間のMap
     * @throws IOException
     * @throws UnsupportedEncodingException
     */
    public Map<String, Integer> createNSNumberMap(String xmlFilePath)
    throws IOException, UnsupportedEncodingException {
        Configuration conf = new Configuration();
        FileSystem fs = FileSystem.get(conf);
        FSDataInputStream fsdis = fs.open(new Path(xmlFilePath));

        Map<String, Integer> result = new HashMap<String, Integer>();

        BufferedReader reader = new BufferedReader(new InputStreamReader(fsdis, WikIE.UTF8));
        String line = new String();
        result.put("", WikIE.ARTICLE_NS_NUM);
        result.put("image", WikIE.IMAGE_NS_NUM);
        while (!(line = reader.readLine().trim()).equals("</" + WikIE.NAMESPACES_ELEMENT + ">")) {
            Pattern pattern = Pattern.compile("<" + WikIE.NAMESPACE_ELEMENT + " +key=\"(-*[0-9]+)\">(.*?)" + "</" + WikIE.NAMESPACE_ELEMENT + ">");
            Matcher matcher = pattern.matcher(line);
            if (matcher.find()) {
                int number = Integer.parseInt(matcher.group(1));
                String namespace = matcher.group(2).toLowerCase();
                result.put(namespace, number);
            }
            else {
                pattern = Pattern.compile("<base>(http://.+?\\.wikipedia\\.org/wiki/).+?</base>", Pattern.CASE_INSENSITIVE);
                matcher = pattern.matcher(line);
                if (matcher.find()) {
                    baseURL = new String(matcher.group(1));
                }
            }
        }
        reader.close();
        return result;
    }


    /**
     * 処理中のWikipediaのベースURLを取得する。
     * @return ベースURL.<br>
     *          日本語版の場合http://ja.wikipedia.org/wiki/
     */
    public String getBaseURL() {
        return baseURL;
    }

    /**
     * 言語リストを設定する。<br>
     * languageURLのページから抽出する。
     */
    public Set<String> createLangSet(String langURL)
    throws UnsupportedEncodingException, IOException {
        Set<String> result = new TreeSet<String>();
        result.add("en");
        BufferedReader reader = new BufferedReader(
                new InputStreamReader(
                        new URL(langURL).openStream(), WikIE.UTF8));
        String line = new String();
        while ((line = reader.readLine()) != null) {
            Pattern p = Pattern.compile("\\| *\\[\\[:(.*?):\\|(.*?)\\]\\].*?");
            Matcher m = p.matcher(line);
            if (m.find()) {
                result.add(m.group(1));
            }
        }
        reader.close();
        return result;
    }

    /**
     * Wikipediaのタイトルルールを適用した文字列を返す
     * @param str
     * @return WikipediaのWikipediaのタイトルルールを適用した文字列.
     *          <ul>
     *              <li>両端のスペースは削除</li>
     *              <li>アンダースコアはスペースに変換</li>
     *              <li>2個以上連続したスペースは1個に変換</li>
     *          </ul>
     */
    public static String wTitleRule(String str) {
        str = str.trim();
        str = str.replaceAll("_", " ");
        str = str.replaceAll(" {2,}", " ");
        while (StringUtils.isNotBlank(str) && str.charAt(0) == ':') {
            str = str.replaceFirst(":", "");
        }
        return str.trim();
    }

    /**
     * 制御文字を削除した文字列を返す
     * @param text
     * @return
     */
    public static String deleteNonPrintingChar(String text) {
        //制御文字を除去する。(MediaWikiにもある処理）
        char[] target = {'\n', '\r', '\u0000', '\u200e', '\u200f', '\u2028', '\u2029'};
        for (char c : target) {
            text = text.replaceAll(Character.toString(c), "");
        }
        return text;
    }

    /**
     * 文字列中にWikipediaのタイトルに使えない文字がないか調べる。
     * タイトルとして使えない文字列は、<br>
     * # < > [ ] | { }
     * @param title 調べたい文字列
     * @return 不正な文字がなければtrue, あればfalse
     */
    public static boolean isCorrectTitle(String title) {
        char[] ngc = new char[] {'#', '<', '>', '[', ']', '|', '{', '}'};
        for (char n : ngc) {
            if (title.indexOf(n) != -1) {
                return false;
            }
        }
        return true;
    }

    /**
     * URLエンコードとHTMLエスケープを処理した文字列を返す
     * @param str
     * @return
     */
    public static String decode(String str) {
        URLCodec codec = new URLCodec(WikIE.UTF8);
        Pattern pattern = Pattern.compile("(%[%a-zA-Z0-9]+)");
        Matcher matcher = pattern.matcher(str);
        while (matcher.find()) {
            try {
                String encoded = matcher.group(1);
                String decoded = codec.decode(encoded);
                if (decoded.length() < encoded.length()) {
                    try {
                        str = str.replaceAll(encoded, Matcher.quoteReplacement(decoded));
                    }
                    catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
            catch (DecoderException e) {
            }
            catch (IllegalArgumentException e) {
                e.printStackTrace();
            }
        }
        while (!str.equals(StringEscapeUtils.unescapeHtml(str))) {
            str = StringEscapeUtils.unescapeHtml(str);
        }

        return str;
    }

    /**
     * セクションリンクをデコードした文字列を返す
     * @param str
     * @return
     */
    public static String decodeSectionLink(String str) {
        return decode(str.replaceAll("\\.", "%"));
    }

    /**
     * タイトル文字列を正規化してWEntryで返す
     * @param target タイトル文字列
     * @return 正規化したタイトル文字列を設定したWEntry
     */
    public WEntry normalize(String target) {
        target = decode(wTitleRule(target));
        if (StringUtils.isBlank(target)) {
            return new WEntry();
        }
        String[] part = StringUtils.splitPreserveAllTokens(target, ":");

        int nameIndex = getNameStartIndex(part);
        int nsIndex = getNSStartIndex(part, nameIndex);
        int lim = nsIndex > nameIndex ? nsIndex : nameIndex;
        int langIndex = getLangStartIndex(part, lim);
        lim = langIndex > lim ? langIndex : lim;
        int prIndex = getProjectStartIndex(part, lim);

        return createWEntry(part, prIndex, langIndex, nsIndex, nameIndex);
    }

    /**
     * WEntryオブジェクトを生成する。
     * 各プレフィックスは小文字、記事名は先頭が大文字でそれ以降は小文字に統一される。
     * @param part エントリタイトルをコロン(:)でsplitした配列。
     * @param name 名前部分の始まりのインデックス。
     * @param ns　名前空間部分の始まりのインデックス。
     * @param lang　言語部分の始まりのインデックス。
     * @param pr　プロジェクト部分の始まりのインデックス。
     * @return　生成したWEntryオブジェクト
     */
    public WEntry createWEntry(String[] part, int pr, int lang, int ns, int name) {
        StringBuffer nameStr = new StringBuffer();
        for (int i = name ; i < part.length; i++) {
            nameStr.append(part[i] + ":");
        }
        nameStr.deleteCharAt(nameStr.length() -1);
        String n = StringUtils.capitalize(nameStr.toString().trim());
        String nsStr = new String();
        int nsNumber = 0;
        if (ns != -1) {
            nsStr = part[ns].toLowerCase().trim();
            nsNumber = nsNumberMap.get(nsStr);
        }
        String langStr = new String();
        if (lang != -1) {
            langStr = part[lang].toLowerCase().trim();
        }
        String prStr = new String();
        if (pr != -1) {
            prStr = part[pr].toLowerCase().trim();
        }
        return new WEntry(prStr, langStr, nsStr, n, nsNumber);
    }

    /**
     * タイトルの記事名部分の始まりのインデックスを取得する。
     * @param part エントリタイトルをコロン(:)でsplitした配列。
     * @return
     */
    public int getNameStartIndex(String[] part) {
        int pr = -1;
        int lang = -1;
        int ns = -1;
        for (int i = 0; i < part.length; i++) {
            String p = part[i].toLowerCase().trim();
            if (projectSet.contains(p) && pr == -1 && lang == -1 && ns == -1) {
                pr = i;
            }
            else if (langSet.contains(p) && lang == -1 && ns == -1 ) {
                lang = i;
            }
            else if (nsNumberMap.containsKey(p) && ns == -1) {
                ns = i;
            }
            else if (!langSet.contains(p) && !nsNumberMap.containsKey(p) && !projectSet.contains(p) ||
                      ns != -1 || lang != -1 || pr != -1) {
                return i;
            }
        }
        return part.length-1;
    }

    /**
     * タイトルの名前空間分の始まりのインデックスを取得する。
     * @param part　エントリタイトルをコロン(:)でsplitした配列。
     * @param lim　終了インデックス。これ以下は探さない。
     * @return
     */
    public int getNSStartIndex(String[] part, int lim) {
        if (lim == -1) {
            lim = part.length;
        }
        for (int i = 0; i < lim; i++) {
            String p = part[i].toLowerCase().trim();
            if (nsNumberMap.containsKey(p)) {
                return i;
            }
        }
        return -1;
    }

    /**
     * タイトルの言語部分の始まりのインデックスを取得する。
     * @param part　エントリタイトルをコロン(:)でsplitした配列。
     * @param lim　終了インデックス。これ以下は探さない。
     * @return
     */
    public int getLangStartIndex(String[] part, int lim) {
        if (lim == -1) {
            lim = part.length;
        }
        for (int i = 0; i < lim; i++) {
            String p = part[i].toLowerCase().trim();
            if (langSet.contains(p)) {
                return i;
            }
        }
        return -1;
    }

    /**
     * タイトルのプロジェクト部分の始まりのインデックスを取得する。
     * @param part　エントリタイトルをコロン(:)でsplitした配列。
     * @param lim　終了インデックス。これ以下は探さない。
     * @return
     */
    public int getProjectStartIndex(String[] part, int lim) {
        if (lim == -1) {
            lim = part.length;
        }
        for (int i = 0; i < lim; i++) {
            String p = part[i].toLowerCase().trim();
            if (projectSet.contains(p)) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 文字列が言語プレフィックスか判定する
     * @param str
     * @return 言語プレフィックスならtrue, そうでなければfalse
     */
    public boolean isLangPrefix(String str) {
        return this.langSet.contains(str);
    }

    /**
     * 文字列が名前空間プレフィックスか判定する
     * @param str
     * @return 名前空間プレフィックスならtrue, そうでなければfalse
     */
    public boolean isNamespacePrefix(String str) {
        return this.nsNumberMap.containsKey(str);
    }

    /**
     * 文字列がプロジェクトプレフィックスか判定する
     * @param str
     * @return プロジェクトプレフィックスならtrue, そうでなければfalse
     */
    public boolean isProjectPrefix(String str) {
        return this.projectSet.contains(str);
    }

    /**
     * プロジェクトプレフィックスのSetのIteratorを取得する。
     * @return
     */
    public Iterator<String> projectItr() {
        return projectSet.iterator();
    }

    /**
     * 言語プレフィックスのSetのIteratorを取得する。
     * @return
     */
    public Iterator<String> langItr() {
        return langSet.iterator();
    }

    /**
     * 名前空間MapのEntrySetのIteratorを取得する。
     * @return
     */
    public Iterator<Entry<String, Integer>> nsNumberItr() {
        return nsNumberMap.entrySet().iterator();
    }
}
