
/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    winsyn_c.c - Windows synthesizer interface
        Copyright (c) 2002 2003 Keishi Suenaga <s_keishi@mutt.freemail.ne.jp>

    I referenced following sources.
        alsaseq_c.c - ALSA sequencer server interface
            Copyright (c) 2000  Takashi Iwai <tiwai@suse.de>
        readmidi.c


    DESCRIPTION
    ===========

    This interface provides a Windows MIDI device interface which receives
    events and plays it in real-time.  On this mode, TiMidity works
    purely as software (real-time) MIDI render.

    For invoking Windows synthesizer interface, run timidity as folows:
      % timidity -iW    (interactively select an Input MIDI device)
    or
      % timidity -iW 2  (connect to MIDI device No. 2)

    TiMidity loads instruments dynamically at each time a PRM_CHANGE
    event is received.  It sometimes causes a noise.
    If you are using a low power machine, invoke timidity as follows:
      % timidity -s 11025 -iW        (set sampling freq. to 11025Hz)
    or
      % timidity -EFreverb=0 -iW    (disable MIDI reverb effect control)

    TiMidity keeps all loaded instruments during executing.

    To use TiMidity as output device, you need a MIDI loopback device.
    I use MIDI Yoke. It can freely be obtained MIDI-OX site
    (http://www.midiox.com).
*/
//#define  USE_PORTMIDI 1
//#define USE_GTK_GUI 1

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */


#include "rtsyn.h"
#ifdef USE_GTK_GUI
#include "wsgtk_main.h"
#endif

#ifndef __W32__
#include <stdio.h>
#include <termios.h>
//#include <term.h>
#include <unistd.h>
#endif

/* for dumb interface */
static int dumb_f;
static int ctl_read_blocked(int32 *valp);
static void ctl_total_time(int tt);
static void ctl_file_name(char *name);
static void ctl_current_time(int secs, int v);

#ifndef __W32__
static struct termios initial_settings, new_settings;
static int peek_character = -1;
#endif

extern int volatile stream_max_compute;	// play_event()  compute_data() ŌvZő厞
int rtsyn_seq_quit=1;
char dumb_ctl='\0';

static int ctl_open(int using_stdin, int using_stdout);
static void ctl_close(void);
static int ctl_read(int32 *valp);
static int cmsg(int type, int verbosity_level, char *fmt, ...);
static void ctl_event(CtlEvent *e);
static void ctl_pass_playing_list(int n, char *args[]);

#ifndef __W32__
static void init_keybord(void);
static void close_keybord(void);
static int kbhit(void);
static char readch(void);
#endif

#define ctl plugin_control_mode

ControlMode ctl=
{
    "plugin interface", 'p',
    1,0,0,
    0,
    ctl_open,
    ctl_close,
    ctl_pass_playing_list,
    ctl_read,
    cmsg,
    ctl_event
};

static int32 event_time_offset;
static FILE *outfp;

/*ARGSUSED*/

static int ctl_open(int using_stdin, int using_stdout)
{
	ctl.opened = 1;
	ctl.flags &= ~(CTLF_LIST_RANDOM|CTLF_LIST_SORT);
	if (using_stdout)
		outfp = stderr;
	else
		outfp = stdout;
	return 0;
}

static void ctl_close(void)
{
  fflush(outfp);
  ctl.opened=0;
}

static int ctl_read(int32 *valp)
{
	if(dumb_f==1){
    	return ctl_read_blocked(valp);
	}else{
    	return RC_NONE;
	}
}

static int cmsg(int type, int verbosity_level, char *fmt, ...)
{

	va_list ap;

  if ((type==CMSG_TEXT || type==CMSG_INFO || type==CMSG_WARNING) &&
      ctl.verbosity<verbosity_level)
    return 0;
  va_start(ap, fmt);
  if(type == CMSG_WARNING || type == CMSG_ERROR || type == CMSG_FATAL)
      dumb_error_count++;
  if (!ctl.opened)
    {
      vfprintf(stderr, fmt, ap);
      fputs(NLS, stderr);
    }
  else
    {
      vfprintf(outfp, fmt, ap);
      fputs(NLS, outfp);
      fflush(outfp);
    }
  va_end(ap);
    return 0;
}

static void ctl_event(CtlEvent *e)
{
	if(dumb_f==1){
    	switch(e->type){
      		case CTLE_NOW_LOADING:
				ctl_file_name((char *)e->v1);
				break;
      		case CTLE_PLAY_START:
				ctl_total_time((int)e->v1);
			break;
      		case CTLE_CURRENT_TIME:
			ctl_current_time((int)e->v1, (int)e->v2);
			break;
    	}
	}
}

static void doit(void);


static void ctl_pass_playing_list(int n, char *args[])
{
	int i, j,devnum,devok;
	unsigned int port=0 ;
	int started;
	char cbuf[80];

if( (args[0][0]!='r')||( args[0][1]!='t') ){ //dumb interface
	rtsyn_seq_quit=0;
	dumb_pass_playing_list(n-1, &(args[1]));
	rtsyn_seq_quit=1;
	return;
}

	rtsyn_init();
	rtsyn_get_port_list();

	if(n > MAX_PORT ){
		printf( "Usage: timidity -iW [Midi interface No s]\n");
		return;
	}
    n--;
	args=&(args[1]);
	if(n>0){
		port=0;
		while(port<n && n!=0){
			if( (portID[port] = atoi(args[port]))==0 ){
				n=0;
			}else{
				devok=0;
				for(i=0;i<rtsyn_nportlist;i++){
					sscanf( rtsyn_portlist[i],"%d:%s",&devnum,cbuf);
					if(devnum==portID[port]) devok=1;
				}
				if(devok==0){
					n=0;
				}
			}
		port++;
		}
	}
	if(n==0){
		rtsyn_portnumber=0;
	}else{
		rtsyn_portnumber=port;
	}

	for(port=0;port<rtsyn_portnumber;port++){
		portID[port]=portID[port]-1;
	}

	
	if(0==rtsyn_synth_start()){
		rtsyn_seq_quit=0;
		while(rtsyn_seq_quit==0) {
			doit();
		}
		rtsyn_synth_stop();
	}
	rtsyn_close();
	return;
}


static void doit(void)
{
	while(rtsyn_seq_quit==0){
		rtsyn_play_some_data();
		rtsyn_play_calculate();
		sleep(0);
	}
}

/*********************************
 *                               *
 * functions for dumb inter face *
 *                               *
 *********************************/
static int ctl_read_blocked(int32 *valp)
{
  unsigned char c;
	c=dumb_ctl; dumb_ctl='\0';
    switch(c){
      case 'h':
      case '?':
        /*ctl_help_mode();*/
        return RC_NONE;
        
      case 'V':
        *valp=10;
        return RC_CHANGE_VOLUME;
      case 'v':
        *valp=-10;
        return RC_CHANGE_VOLUME;
      case 'q':
        return RC_QUIT;
      case 'n':
        return RC_NEXT;
      case 'p':
        return RC_REALLY_PREVIOUS;
      case 'r':
        return RC_SYNC_RESTART;
      case 'f':
        *valp=play_mode->rate;
        return RC_FORWARD;
      case 'b':
        *valp=play_mode->rate;
        return RC_BACK;
      case ' ':
		return RC_TOGGLE_PAUSE; /* Now supported by this improved timidity */
    }
  return RC_NONE;
}
		
static void ctl_total_time(int tt)
{
  int mins, secs;
  if (ctl.trace_playing)
    {
      secs=tt/play_mode->rate;
      mins=secs/60;
      secs-=mins*60;
      fprintf(outfp, "Total playing time: %3d min %02d s\n", mins, secs);
    }
}

static void ctl_file_name(char *name)
{
  if (ctl.verbosity>=0 || ctl.trace_playing)
    fprintf(outfp, "Playing %s\n", name);
}

static void ctl_current_time(int secs, int v)
{
  int mins;
  if (ctl.trace_playing)
    {
      mins=secs/60;
      secs-=mins*60;
      fprintf(outfp, "\r%3d:%02d", mins, secs);
      fflush(outfp);
    }
}

/*
 * interface_<id>_loader();
 */
ControlMode *interface_p_loader(void)
{
    return &ctl;
}


