/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    effect.c - To apply sound effects.
    Mainly written by Masanao Izumo <mo@goice.co.jp>

 * Interfaces:
 * void init_effect(tmdy_struct_ex_t *tmdy_struct, void);
 * do_effect(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count);
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#define USE_MT_RAND
#ifdef USE_MT_RAND
#include "mt19937ar.h"
#endif
#define RAND_MAX 0xffffffff

#include "tmdy_struct.h"
#include "timidity.h"
#include "tables.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "output.h"
#include "reverb.h"

#include  "aq.h"

#define SIDE_CONTI_SEC	10
#define CHANGE_SEC		2.0

#define NS_AMP_MAX	((int32) 0x0fffffff)
#define NS_AMP_MIN	((int32) 0x8fffffff)

static void effect_left_right_delay(tmdy_struct_ex_t *tmdy_struct, int32 *, int32);
static void init_ns_tap(tmdy_struct_ex_t *tmdy_struct);
static void init_ns_tap16(tmdy_struct_ex_t *tmdy_struct);
static void ns_shaping8(tmdy_struct_ex_t *tmdy_struct, int32 *, int32);
static void ns_shaping16(tmdy_struct_ex_t *tmdy_struct, int32 *, int32);
static void ns_shaping16_trad(tmdy_struct_ex_t *tmdy_struct, int32 *, int32);
static void ns_shaping16_9(tmdy_struct_ex_t *tmdy_struct, int32 *, int32);
static inline unsigned long frand(tmdy_struct_ex_t *tmdy_struct);
static inline int32 my_mod(tmdy_struct_ex_t *tmdy_struct, int32, int32);


static float ns9_coef[9] = {
	2.412f, -3.370f, 3.937f, -4.174f, 3.353f,
	-2.205f, 1.281f, -0.569f, 0.0847f
};


void init_effect(tmdy_struct_ex_t *tmdy_struct)
{
	effect_left_right_delay(tmdy_struct, NULL, 0);
	init_ns_tap(tmdy_struct);
	tmdy_struct->reverb->init_reverb(tmdy_struct, (tmdy_struct->output->play_mode)->rate);
	tmdy_struct->reverb->init_ch_delay(tmdy_struct);
	
	tmdy_struct->reverb->init_ch_chorus(tmdy_struct);
	tmdy_struct->reverb->init_eq(tmdy_struct);
	tmdy_struct->reverb->init_insertion_effect(tmdy_struct);
	
}

/*
 * Left & Right Delay Effect
 */
static void effect_left_right_delay(tmdy_struct_ex_t *tmdy_struct, int32 *buff, int32 count)
{
	static int32 prev[AUDIO_BUFFER_SIZE * 2];
	int32 save[AUDIO_BUFFER_SIZE * 2];
	int32 pi, i, j, k, v, backoff;
	int b;
	static int turn_counter = 0, tc;
	static int status;
	static double rate0, rate1, dr;
	int32 *p;
	
	if (buff == NULL) {
		memset(prev, 0, sizeof(prev));
		return;
	}
	if ((tmdy_struct->output->play_mode)->encoding & PE_MONO)
		return;
	if ((tmdy_struct->timidity_main->effect_lr_mode) == 0 || (tmdy_struct->timidity_main->effect_lr_mode) == 1 || (tmdy_struct->timidity_main->effect_lr_mode) == 2)
		b = (tmdy_struct->timidity_main->effect_lr_mode);
	else
		return;
	count *= 2;
	backoff = 2 * (int) ((tmdy_struct->output->play_mode)->rate * (tmdy_struct->timidity_main->effect_lr_delay_msec) / 1000.0);
	if (backoff == 0)
		return;
	if (backoff > count)
		backoff = count;
	if (count < audio_buffer_size * 2) {
		memset(buff + count, 0, 4 * (audio_buffer_size * 2 - count));
		count = audio_buffer_size * 2;
	}
	memcpy(save, buff, 4 * count);
	pi = count - backoff;
	if (b == 2) {
		if (turn_counter == 0) {
			turn_counter = SIDE_CONTI_SEC * (tmdy_struct->output->play_mode)->rate;
			/* status: 0 -> 2 -> 3 -> 1 -> 4 -> 5 -> 0 -> ...
			 * status left   right
			 * 0      -      +		(right)
			 * 1      +      -		(left)
			 * 2     -> +    +		(right -> center)
			 * 3      +     -> -	(center -> left)
			 * 4     -> -    -		(left -> center)
			 * 5      -     -> +	(center -> right)
			 */
			status = 0;
			tc = 0;
		}
		p = prev;
		for (i = 0; i < count; i += 2, pi += 2) {
			if (i < backoff)
				p = prev;
			else if (p == prev) {
				pi = 0;
				p = save;
			}
			if (status < 2)
				buff[i + status] = p[pi + status];
			else if (status < 4) {
				j = (status & 1);
				v = (int32) (rate0 * buff[i + j] + rate1 * p[pi + j]);
				buff[i + j] = v;
				rate0 += dr, rate1 -= dr;
			} else {
				j = (status & 1);
				k = ! j;
				v = (int32) (rate0 * buff[i + j] + rate1 * p[pi + j]);
				buff[i + j] = v;
				buff[i + k] = p[pi + k];
				rate0 += dr, rate1 -= dr;
			}
			tc++;
			if (tc == turn_counter) {
				tc = 0;
				switch (status) {
				case 0:
					status = 2;
					turn_counter = (CHANGE_SEC / 2.0) * (tmdy_struct->output->play_mode)->rate;
					rate0 = 0.0;
					rate1 = 1.0;
					dr = 1.0 / turn_counter;
					break;
				case 2:
					status = 3;
					rate0 = 1.0;
					rate1 = 0.0;
					dr = -1.0 / turn_counter;
					break;
				case 3:
					status = 1;
					turn_counter = SIDE_CONTI_SEC * (tmdy_struct->output->play_mode)->rate;
					break;
				case 1:
					status = 4;
					turn_counter = (CHANGE_SEC / 2.0) * (tmdy_struct->output->play_mode)->rate;
					rate0 = 1.0;
					rate1 = 0.0;
					dr = -1.0 / turn_counter;
					break;
				case 4:
					status = 5;
					turn_counter = (CHANGE_SEC / 2.0) * (tmdy_struct->output->play_mode)->rate;
					rate0 = 0.0;
					rate1 = 1.0;
					dr = 1.0 / turn_counter;
					break;
				case 5:
					status = 0;
					turn_counter = SIDE_CONTI_SEC * (tmdy_struct->output->play_mode)->rate;
					break;
				}
			}
		}
	} else {
		for (i = 0; i < backoff; i += 2, pi += 2)
			buff[b + i] = prev[b + pi];
		for (pi = 0; i < count; i += 2, pi += 2)
			buff[b + i] = save[b + pi];
	}
	memcpy(prev + count - backoff, save + count - backoff, 4 * backoff);
}

/* Noise Shaping filter from
 * Kunihiko IMAI <imai@leo.ec.t.kanazawa-u.ac.jp>
 * (Modified by Masanao Izumo <mo@goice.co.jp>)
 */
static void init_ns_tap(tmdy_struct_ex_t *tmdy_struct)
{
	memset(tmdy_struct->aq->ns_z0, 0, sizeof(tmdy_struct->aq->ns_z0));
	memset(tmdy_struct->aq->ns_z1, 0, sizeof(tmdy_struct->aq->ns_z0));
	if ((tmdy_struct->output->play_mode)->encoding & PE_16BIT)
		init_ns_tap16(tmdy_struct);
}

static void init_ns_tap16(tmdy_struct_ex_t *tmdy_struct)
{
	int i;
#ifdef USE_MT_RAND
	unsigned long init[4] = { 0x123, 0x234, 0x345, 0x456 };
	unsigned long length = 4;
#endif
	
#ifdef USE_MT_RAND
	init_by_array(init, length);
#endif
	tmdy_struct->aq->ns9_order = 9;
	for (i = 0; i < tmdy_struct->aq->ns9_order; i++)
		tmdy_struct->aq->ns9_c[i] = TIM_FSCALE(ns9_coef[i], 24);
	memset(tmdy_struct->aq->ns9_ehl, 0, sizeof(tmdy_struct->aq->ns9_ehl));
	memset(tmdy_struct->aq->ns9_ehr, 0, sizeof(tmdy_struct->aq->ns9_ehr));
	tmdy_struct->aq->ns9_histposl = tmdy_struct->aq->ns9_histposr = 8;
	tmdy_struct->aq->ns9_r1l = tmdy_struct->aq->ns9_r2l = tmdy_struct->aq->ns9_r1r = tmdy_struct->aq->ns9_r2r = 0;
}

void do_effect(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	/* reverb in mono */
	if (tmdy_struct->playmidi->opt_reverb_control && ((tmdy_struct->output->play_mode)->encoding & PE_MONO))
		tmdy_struct->reverb->do_mono_reverb(tmdy_struct, buf, count);
#ifndef USE_DSP_EFFECTS	/* do_compute_data_midi() already applied them */
	/* for static reverb / chorus level */
	if (tmdy_struct->playmidi->opt_reverb_control < 0 || tmdy_struct->playmidi->opt_chorus_control < 0) {
		tmdy_struct->reverb->set_dry_signal(tmdy_struct, buf,2 * count);
		if (tmdy_struct->playmidi->opt_chorus_control < 0)
			tmdy_struct->reverb->set_ch_chorus(tmdy_struct, buf, 2 * count, -tmdy_struct->playmidi->opt_chorus_control);
		if (tmdy_struct->playmidi->opt_reverb_control < 0)
			tmdy_struct->reverb->set_ch_reverb(tmdy_struct, buf, 2 * count, -tmdy_struct->playmidi->opt_reverb_control);
		tmdy_struct->reverb->mix_dry_signal(tmdy_struct, buf, 2 * count);
		if (tmdy_struct->playmidi->opt_chorus_control < 0)
			tmdy_struct->reverb->do_ch_chorus(tmdy_struct, buf, 2 * count);
		if (tmdy_struct->playmidi->opt_reverb_control < 0)
			tmdy_struct->reverb->do_ch_reverb(tmdy_struct, buf, 2 * count);
	}
#endif /* USE_DSP_EFFECTS */
	/* L/R Delay */
	effect_left_right_delay(tmdy_struct, buf, count);
	/* Noise shaping filter must apply at last */
	if (! ((tmdy_struct->output->play_mode)->encoding & (PE_16BIT | PE_ULAW | PE_ALAW)))
		ns_shaping8(tmdy_struct, buf, count);
	else if ((tmdy_struct->output->play_mode)->encoding & PE_16BIT)
		ns_shaping16(tmdy_struct, buf, count);
}

static void ns_shaping8(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
	int32 l, i, ll;
	int32 ns_tap_0, ns_tap_1, ns_tap_2, ns_tap_3;
	
	switch (tmdy_struct->aq->noise_sharp_type) {
	case 1:
		ns_tap_0 = 1;
		ns_tap_1 = 0;
		ns_tap_2 = 0;
		ns_tap_3 = 0;
		break;
	case 2:
		ns_tap_0 = -2;
		ns_tap_1 = 1;
		ns_tap_2 = 0;
		ns_tap_3 = 0;
		break;
	case 3:
		ns_tap_0 = 3;
		ns_tap_1 = -3;
		ns_tap_2 = 1;
		ns_tap_3 = 0;
		break;
	case 4:
		ns_tap_0 = -4;
		ns_tap_1 = 6;
		ns_tap_2 = -4;
		ns_tap_3 = 1;
		break;
	default:
		return;
	}
	for (i = 0; i < c; i++) {
		/* applied noise-shaping filter */
		if (lp[i] > NS_AMP_MAX)
			lp[i] = NS_AMP_MAX;
		else if (lp[i] < NS_AMP_MIN)
			lp[i] = NS_AMP_MIN;
		ll = lp[i] + ns_tap_0 * tmdy_struct->aq->ns_z0[0] + ns_tap_1 * tmdy_struct->aq->ns_z0[1]
				+ ns_tap_2 * tmdy_struct->aq->ns_z0[2] + ns_tap_3 * tmdy_struct->aq->ns_z0[3];
		l = ll >> (32 - 8 - GUARD_BITS);
		lp[i] = l << (32 - 8 - GUARD_BITS);
		tmdy_struct->aq->ns_z0[3] = tmdy_struct->aq->ns_z0[2], tmdy_struct->aq->ns_z0[2] = tmdy_struct->aq->ns_z0[1], tmdy_struct->aq->ns_z0[1] = tmdy_struct->aq->ns_z0[0];
		tmdy_struct->aq->ns_z0[0] = ll - l * (1U << (32 - 8 - GUARD_BITS));
		if ((tmdy_struct->output->play_mode)->encoding & PE_MONO)
			continue;
		i++;
		if (lp[i] > NS_AMP_MAX)
			lp[i] = NS_AMP_MAX;
		else if (lp[i] < NS_AMP_MIN)
			lp[i] = NS_AMP_MIN;
		ll = lp[i] + ns_tap_0 * tmdy_struct->aq->ns_z1[0] + ns_tap_1 * tmdy_struct->aq->ns_z1[1]
				+ ns_tap_2 * tmdy_struct->aq->ns_z1[2] + ns_tap_3 * tmdy_struct->aq->ns_z1[3];
		l = ll >> (32 - 8 - GUARD_BITS);
		lp[i] = l << (32 - 8 - GUARD_BITS);
		tmdy_struct->aq->ns_z1[3] = tmdy_struct->aq->ns_z1[2], tmdy_struct->aq->ns_z1[2] = tmdy_struct->aq->ns_z1[1], tmdy_struct->aq->ns_z1[1] = tmdy_struct->aq->ns_z1[0];
		tmdy_struct->aq->ns_z1[0] = ll - l * (1U << (32 - 8 - GUARD_BITS));
	}
}

static void ns_shaping16(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
	switch (tmdy_struct->aq->noise_sharp_type) {
	case 1:
		ns_shaping16_trad(tmdy_struct, lp, c * 2);
		break;
	case 2:
	case 3:
	case 4:
		ns_shaping16_9(tmdy_struct, lp, c * 2);
		break;
	}
}

static void ns_shaping16_trad(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
	int32 l, i, ll;
	int32 ns_tap_0, ns_tap_1, ns_tap_2, ns_tap_3;
	
	ns_tap_0 = -4;
	ns_tap_1 = 6;
	ns_tap_2 = -4;
	ns_tap_3 = 1;
	for (i = 0; i < c; i++) {
		/* applied noise-shaping filter */
		if (lp[i] > NS_AMP_MAX)
			lp[i] = NS_AMP_MAX;
		else if (lp[i] < NS_AMP_MIN)
			lp[i] = NS_AMP_MIN;
		ll = lp[i] + ns_tap_0 * tmdy_struct->aq->ns_z0[0] + ns_tap_1 * tmdy_struct->aq->ns_z0[1]
				+ ns_tap_2 * tmdy_struct->aq->ns_z0[2] + ns_tap_3 * tmdy_struct->aq->ns_z0[3];
		l = ll >> (32 - 16 - GUARD_BITS);
		lp[i] = l << (32 - 16 - GUARD_BITS);
		tmdy_struct->aq->ns_z0[3] = tmdy_struct->aq->ns_z0[2], tmdy_struct->aq->ns_z0[2] = tmdy_struct->aq->ns_z0[1], tmdy_struct->aq->ns_z0[1] = tmdy_struct->aq->ns_z0[0];
		tmdy_struct->aq->ns_z0[0] = ll - l * (1U << (32 - 16 - GUARD_BITS));
		if ((tmdy_struct->output->play_mode)->encoding & PE_MONO)
			continue;
		i++;
		if (lp[i] > NS_AMP_MAX)
			lp[i] = NS_AMP_MAX;
		else if (lp[i] < NS_AMP_MIN)
			lp[i] = NS_AMP_MIN;
		ll = lp[i] + ns_tap_0 * tmdy_struct->aq->ns_z1[0] + ns_tap_1 * tmdy_struct->aq->ns_z1[1]
				+ ns_tap_2 * tmdy_struct->aq->ns_z1[2] + ns_tap_3 * tmdy_struct->aq->ns_z1[3];
		l = ll >> (32 - 16 - GUARD_BITS);
		lp[i] = l << (32 - 16 - GUARD_BITS);
		tmdy_struct->aq->ns_z1[3] = tmdy_struct->aq->ns_z1[2], tmdy_struct->aq->ns_z1[2] = tmdy_struct->aq->ns_z1[1], tmdy_struct->aq->ns_z1[1] = tmdy_struct->aq->ns_z1[0];
		tmdy_struct->aq->ns_z1[0] = ll - l * (1U << (32 - 16 - GUARD_BITS));
	}
}

static void ns_shaping16_9(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
	int32 i, l, sample, output;
	
	for (i = 0; i < c; i++) {
		/* left channel */
		tmdy_struct->aq->ns9_r2l = tmdy_struct->aq->ns9_r1l;
		tmdy_struct->aq->ns9_r1l = frand(tmdy_struct);
		lp[i] = (lp[i] > NS_AMP_MAX) ? NS_AMP_MAX
				: (lp[i] < NS_AMP_MIN) ? NS_AMP_MIN : lp[i];
#if OPT_MODE != 0
		sample = lp[i] - imuldiv24(tmdy_struct->aq->ns9_c[8], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 8])
				- imuldiv24(tmdy_struct->aq->ns9_c[7], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 7])
				- imuldiv24(tmdy_struct->aq->ns9_c[6], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 6])
				- imuldiv24(tmdy_struct->aq->ns9_c[5], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 5])
				- imuldiv24(tmdy_struct->aq->ns9_c[4], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 4])
				- imuldiv24(tmdy_struct->aq->ns9_c[3], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 3])
				- imuldiv24(tmdy_struct->aq->ns9_c[2], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 2])
				- imuldiv24(tmdy_struct->aq->ns9_c[1], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 1])
				- imuldiv24(tmdy_struct->aq->ns9_c[0], tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl]);
#else
		sample = lp[i] - tmdy_struct->aq->ns9_coef[8] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 8]
				- tmdy_struct->aq->ns9_coef[7] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 7]
				- tmdy_struct->aq->ns9_coef[6] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 6]
				- tmdy_struct->aq->ns9_coef[5] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 5]
				- tmdy_struct->aq->ns9_coef[4] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 4]
				- tmdy_struct->aq->ns9_coef[3] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 3]
				- tmdy_struct->aq->ns9_coef[2] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 2]
				- tmdy_struct->aq->ns9_coef[1] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 1]
				- tmdy_struct->aq->ns9_coef[0] * tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl];
#endif
		l = sample >> (32 - 16 - GUARD_BITS);
		output = l * (1U << (32 - 16 - GUARD_BITS))
				+ ((tmdy_struct->aq->ns9_r1l - tmdy_struct->aq->ns9_r2l) >> 30);
		tmdy_struct->aq->ns9_histposl = my_mod(tmdy_struct, (tmdy_struct->aq->ns9_histposl + 8), tmdy_struct->aq->ns9_order);
		tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl + 9] = tmdy_struct->aq->ns9_ehl[tmdy_struct->aq->ns9_histposl] = output - sample;
		lp[i] = output;
		/* right channel */
		i++;
		tmdy_struct->aq->ns9_r2r = tmdy_struct->aq->ns9_r1r;
		tmdy_struct->aq->ns9_r1r = frand(tmdy_struct);
		lp[i] = (lp[i] > NS_AMP_MAX) ? NS_AMP_MAX
				: (lp[i] < NS_AMP_MIN) ? NS_AMP_MIN : lp[i];
#if OPT_MODE != 0
		sample = lp[i] - imuldiv24(tmdy_struct->aq->ns9_c[8], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 8])
				- imuldiv24(tmdy_struct->aq->ns9_c[7], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 7])
				- imuldiv24(tmdy_struct->aq->ns9_c[6], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 6])
				- imuldiv24(tmdy_struct->aq->ns9_c[5], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 5])
				- imuldiv24(tmdy_struct->aq->ns9_c[4], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 4])
				- imuldiv24(tmdy_struct->aq->ns9_c[3], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 3])
				- imuldiv24(tmdy_struct->aq->ns9_c[2], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 2])
				- imuldiv24(tmdy_struct->aq->ns9_c[1], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 1])
				- imuldiv24(tmdy_struct->aq->ns9_c[0], tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr]);
#else
		sample = lp[i] - tmdy_struct->aq->ns9_coef[8] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 8]
				- tmdy_struct->aq->ns9_coef[7] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 7]
				- tmdy_struct->aq->ns9_coef[6] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 6]
				- tmdy_struct->aq->ns9_coef[5] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 5]
				- tmdy_struct->aq->ns9_coef[4] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 4]
				- tmdy_struct->aq->ns9_coef[3] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 3]
				- tmdy_struct->aq->ns9_coef[2] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 2]
				- tmdy_struct->aq->ns9_coef[1] * tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 1]
				- tmdy_struct->aq->ns9_coef[0]*tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr];
#endif
		l = sample >> (32 - 16 - GUARD_BITS);
		output = l * (1U << (32 - 16 - GUARD_BITS))
				+ ((tmdy_struct->aq->ns9_r1r - tmdy_struct->aq->ns9_r2r) >> 30);
		tmdy_struct->aq->ns9_histposr = my_mod(tmdy_struct, (tmdy_struct->aq->ns9_histposr + 8), tmdy_struct->aq->ns9_order);
		tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr + 9] = tmdy_struct->aq->ns9_ehr[tmdy_struct->aq->ns9_histposr] = output - sample;
		lp[i] = output;
	}
}

static inline unsigned long frand(tmdy_struct_ex_t *tmdy_struct)
{
#ifdef USE_MT_RAND
	return genrand_int32();
#else
	static unsigned long a = 0xdeadbeef;
	
	a = a * 140359821 + 1;
	return a;
#endif
}

static inline int32 my_mod(tmdy_struct_ex_t *tmdy_struct, int32 x, int32 n)
{
	if (x > n)
		x -= n;
	return x;
}

