/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * EAW -- Quick and dirty hack to convert the TiMidity events generated from
 * a MOD file into events suitable for writing to a midi file, then write
 * the midi file.
 *
 * DONE LIST (various odd things needed for MOD->MIDI format conversion):
 *  1) MIDI does not support pitch bends > 2 octaves...
 *     scale pitch bends for a sensitivy of 24 instead of 128,
 *     pitch bends > 2 octaves wrapped back an octave and continue as normal
 *  2) moved all drum voices onto the drum channel, and all non-drum voices
 *     off of the drum channel to the first available channel (which could be
 *     a channel that has been freed by moving all of it's drum voices off
 *     of it)
 *  3) moved/duped events on to other channels to handle the drum related stuff
 *  4) force all drums to center pan, since drums originally on different
 *     channels could mess up each other's pans when they are merged on to a
 *     single channel
 *  5) added in extra stuff to keep track of and scale expression events as
 *     necessary, so that each sample can have it's own amplification setting
 *  6) turn all drum channel expression stuff into initial note volumes
 *  7) experimented with turning expression events into keypressure events,
 *     but this didn't sound good, because my XG card doesn't handle increases
 *     in keypressure, and the expression events are also needed to boost the
 *     volumes of note decays (since keypressure doesn't work on dead notes)
 *  8) emit chords for chord samples
 *  9) issue bank change events, as specified in the cfg file
 * 10) ignore redundant program changes, which often happen from drum moves
 * 11) ability to silence a sample, so it emits no note or program events
 * 12) issue port events to make midi with > 16 channels
 * 13) skip over the drum channel on the 2nd port
 * 14) kill non-looped mod samples when the play past sample end
 * 15) extended bend range to 4 octaves using note+pitchbend shifting
 * 16) automatic sample chord/pitch assignment and cfg file generation !!! :)
 * 17) can offset pitchbends to undo out of tune sample tuning (detuning)
 * 18) converts linear mod volumes into non-linear midi volumes
 * 19) other more minor things that may or may not be commented
 *
 * TODO LIST (likely to be done eventually)
 *  1) corre(TMDY_CONTROLS->ctl)y implement fine tuning tweaks via extra pitch bends
 *  2) maybe issue a SYSEX event to make channel 26 non-drum for regular use?
 *  3) clean up the code some more
 *
 * WISH LIST (far less likely to ever be done):
 *  1) make an alternate output mode that outputs a separate track for every
 *     combination of sample and channel it is used on
 *  2) possibly have an automated channel merger, but it might have to strip
 *     out some pans, expression events, and pitch bends to do it
 *  3) maybe handle > 4 octave pitch bends in a better way
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "tmdy_struct.h"
#include "timidity.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "output.h"
#include "controls.h"
#include "tables.h"
#include "freq.h"

#include "m2m.h"
#include "m2m_prv.h"





static const unsigned char init_header[14] = { 0x4D, 0x54, 0x68, 0x64,
    0x00, 0x00, 0x00, 0x06,
    0x00, 0x01, 0x00, 0x00,
    0x00, 0x00
};
const unsigned char init_mtrk[4] = { 0x4D, 0x54, 0x72, 0x6B };

static const char init_chord_letters[4] = { 'M', 'm', 'd', 'f' };



static const char init_vol_nonlin_to_lin[128][2] = {
      0, 127,   7, 125,  11, 120,  14, 121,  16, 126,  19, 121,  21, 122,
     23, 122,  25, 122,  26, 126,  28, 125,  30, 123,  31, 126,  33, 124,
     34, 126,  36, 124,  37, 125,  38, 126,  40, 124,  41, 125,  42, 126,
     43, 127,  45, 125,  46, 125,  47, 126,  48, 126,  49, 127,  50, 127,
     52, 125,  53, 125,  54, 125,  55, 125,  56, 126,  57, 126,  58, 126,
     59, 126,  60, 126,  61, 126,  62, 126,  63, 126,  64, 126,  65, 126,
     66, 126,  67, 125,  68, 125,  69, 125,  69, 127,  70, 127,  71, 126,
     72, 126,  73, 126,  74, 126,  75, 126,  76, 125,  76, 127,  77, 127,
     78, 126,  79, 126,  80, 126,  81, 126,  81, 127,  82, 126,  83, 126,
     84, 126,  85, 126,  85, 127,  86, 126,  87, 126,  88, 126,  88, 127,
     89, 127,  90, 126,  91, 126,  91, 127,  92, 127,  93, 126,  94, 126,
     94, 127,  95, 127,  96, 126,  97, 126,  97, 127,  98, 126,  99, 126,
    100, 126, 100, 127, 101, 126, 102, 126, 102, 127, 103, 126, 104, 126,
    104, 127, 105, 127, 106, 126, 106, 127, 107, 127, 108, 126, 108, 127,
    109, 127, 110, 126, 110, 127, 111, 127, 112, 126, 112, 127, 113, 127,
    114, 126, 114, 127, 115, 127, 116, 126, 116, 127, 117, 126, 118, 126,
    118, 127, 119, 126, 120, 126, 120, 127, 121, 126, 121, 127, 122, 126,
    123, 126, 123, 127, 124, 126, 124, 127, 125, 127, 126, 126, 126, 127,
    127, 126, 127, 127 };

/* generate the names of the output and cfg files from the input file */
/* modified from auto_wav_output_open() in wave_a.c */
static int auto_names(tmdy_struct_ex_t* tmdy_struct, const char *input_filename)
{
    char *ext, *p;

    TMDY_M2M->outname = (char *) TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->outname,
			       sizeof(char) * (strlen(input_filename) + 5));
    TMDY_M2M->cfgname = (char *) TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->cfgname,
			       sizeof(char) * (strlen(input_filename) + 5));

    strcpy(TMDY_M2M->outname, input_filename);
    if ((ext = strrchr(TMDY_M2M->outname, '.')) == NULL)
	ext = TMDY_M2M->outname + strlen(TMDY_M2M->outname);
    else
    {
	/* strip ".???" */
	*ext = '\0';
    }

    /* replace '.' and '#' before ext */
    for (p = TMDY_M2M->outname; p < ext; p++)
	if (*p == '.' || *p == '#')
	    *p = '_';

    strcpy(TMDY_M2M->cfgname, TMDY_M2M->outname);
    strcat(TMDY_M2M->outname, ".mid");
    strcat(TMDY_M2M->cfgname, ".m2m");

    TMDY_M2M->actual_outname = TMDY_M2M->outname;

    return 0;
}



void initialize_m2m_stuff(tmdy_struct_ex_t* tmdy_struct)
{

    int i;

    memset(TMDY_M2M->track_events, 0, 34 * sizeof(unsigned char *));
    memset(TMDY_M2M->last_track_event_time, 0, 34 * sizeof(uint32));
    memset(TMDY_M2M->track_size, 0, 34 * sizeof(uint32));
    memset(TMDY_M2M->tracks_enabled, 0, 34 * sizeof(int));
    memset(TMDY_M2M->tracks_useless, 0, 34 * sizeof(int));
    memset(TMDY_M2M->current_track_sample, 0, 34 * sizeof(int));
    memset(TMDY_M2M->orig_track_expr, 0, 34 * sizeof(int));
    memset(TMDY_M2M->current_channel_expr, 0, 34 * sizeof(int));
    memset(TMDY_M2M->current_channel_bank, 0, 34 * sizeof(int));
    memset(TMDY_M2M->current_channel_program, 0, 34 * sizeof(int));
    memset(TMDY_M2M->current_channel_note, 0, 34 * sizeof(int));
    memset(TMDY_M2M->current_track_note, 0, 34 * sizeof(int));

    memset(TMDY_M2M->banks, 0, 256 * sizeof(int));
    memset(TMDY_M2M->transpose, 0, 256 * sizeof(int));
    memset(TMDY_M2M->is_drum_sample, 0, 256 * sizeof(int));
    memset(TMDY_M2M->silent_samples, 0, 256 * sizeof(int));
    memset(TMDY_M2M->fine_tune, 0, 256 * sizeof(int));

    /* get the names of the output and cfg files */
    auto_names(tmdy_struct, TMDY_READMIDI->current_file_info->filename);
    if ((TMDY_OUTPUT->play_mode)->name != NULL)
	TMDY_M2M->actual_outname = (TMDY_OUTPUT->play_mode)->name;
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "Output %s", TMDY_M2M->actual_outname);

    for (i = 0; i < 256; i++)
    {
	TMDY_M2M->sample_to_program[i] = i;
	if (i > 127)
	    TMDY_M2M->sample_to_program[i] = i - 127;
	TMDY_M2M->sample_chords[i] = -1;
	TMDY_M2M->vol_amp[i] = 100;
    }
    for (i = 0; i < 34; i++)
    {
	TMDY_M2M->tracks_useless[i] = 1;
	TMDY_M2M->current_track_sample[i] = 255;
	TMDY_M2M->current_channel_note[i] = -1;
	TMDY_M2M->current_track_note[i] = -1;
    }

    /* support to increase the number of divisions, this may be useful */
    TMDY_M2M->orig_divisions = TMDY_READMIDI->current_file_info->divisions;
    TMDY_M2M->divisions = 480;		/* maximum number of divisions */
    TMDY_M2M->divisions_ratio = (TMDY_M2M->divisions / TMDY_M2M->orig_divisions);

    TMDY_M2M->num_tracks = 0;
    TMDY_M2M->tempo = 500000;
    TMDY_M2M->maxtime = 0;
    TMDY_M2M->num_killed_early = 0;
    TMDY_M2M->num_big_pitch_slides = 0;
    TMDY_M2M->num_huge_pitch_slides = 0;
    TMDY_M2M->pb_sensitivity = 24;
    TMDY_M2M->old_pb_sensitivity = 128;
    TMDY_M2M->notes_per_pb = 24.0 / 8191.0;
    TMDY_M2M->pb_per_note = 8191.0 / 24.0;
    TMDY_M2M->rpn_msb = 0;
    TMDY_M2M->rpn_lsb = 0;
    TMDY_M2M->min_enabled_track = -1;
    TMDY_M2M->first_free_track = -1;
    TMDY_M2M->non_drums_on_drums = 0;

    for (i = 1, TMDY_M2M->maxsample = 0; i < 256; i++)
    {
	if (TMDY_INSTRUM->special_patch[i])
	    TMDY_M2M->maxsample = i;
    }
}



int create_m2m_cfg_file(tmdy_struct_ex_t* tmdy_struct, char *cfgname)
{

    FILE *cfgout;
    int i, chord, chord_type, chord_subtype;
    char line[81];
    char chord_str[3];
    char program_str[17];

    cfgout = fopen(cfgname, "wb");
    if (!cfgout)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
		  "Could not open cfg file %s for writing", cfgname);
	return 0;
    }

    fprintf(cfgout, "%s\t%s\t\t%s\t%s\t%s\n\n",
    	    "# Sample", "Program", "Transpose", "FineTuning", "%Volume");

    for (i = 1; i <= TMDY_M2M->maxsample; i++)
    {
	memset(chord_str, 0, 3 * sizeof(char));

	if (TMDY_INSTRUM->special_patch[i])
	{
	    chord = TMDY_M2M->sample_chords[i];
	    if (chord >= 0)
	    {
		chord_type = chord / 3;
		chord_subtype = chord % 3;
		chord_str[0] = TMDY_M2M->chord_letters[chord_type];
		if (chord_subtype)
		    chord_str[1] = '0' + chord_subtype;
	    }

	    sprintf(program_str, "%d%s", TMDY_M2M->sample_to_program[i], chord_str);

	    sprintf(line, "%d\t\t%s\t\t%d\t\t!%.6f\t100\n",
		    i, program_str, TMDY_M2M->transpose[i],
		    TMDY_M2M->fine_tune[i] * TMDY_M2M->notes_per_pb);
	}
	else
	{
	    sprintf(line, "# %d unused\n", i);
	}

	fprintf(cfgout, "%s", line);
    }
    fclose(cfgout);

    return 1;
}



void read_m2m_cfg_file(tmdy_struct_ex_t* tmdy_struct)
{

    FILE *mod_cfg_file;
    float freq;
    int i, pitch;
    char *chord_str;
    int chord, chord_type, chord_subtype;
    char line[81];
    int mod_sample, program, trans, amp;
    char program_str[20], finetune_str[20];
    char *slash_pos;

    /* read in cfg file stuff, create new one if doesn't exist */
    mod_cfg_file = fopen(TMDY_M2M->cfgname, "rb");
    if (!mod_cfg_file)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
		  "Couldn't open '%s' cfg file.  Creating %s ...", TMDY_M2M->cfgname,
		  TMDY_M2M->cfgname);
	for (i = 1; i <= TMDY_M2M->maxsample; i++)
	{
	    if (TMDY_INSTRUM->special_patch[i])
	    {
		chord = -1;
		freq = TMDY_FREQ->freq_fourier(tmdy_struct, TMDY_INSTRUM->special_patch[i]->sample, &chord);
		pitch = TMDY_FREQ->assign_pitch_to_freq(tmdy_struct, freq);
		TMDY_M2M->fine_tune[i] = (-36.37631656f +
				 17.31234049f * log(freq) - pitch) *
				 TMDY_M2M->pb_per_note;

		sprintf(line,
			"Sample %3d Freq %10.4f Pitch %3d Transpose %4d",
			i, freq, pitch, 24 + pitch - 60);
		if (chord >= 0)
		{
		    sprintf(line, "%s Chord %c Subtype %d",
			    line, TMDY_M2M->chord_letters[chord / 3], chord % 3);
		}

		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "%s", line);

		TMDY_M2M->transpose[i] = 24 + pitch - 60;
		TMDY_M2M->sample_chords[i] = chord;
	    }
	}
	create_m2m_cfg_file(tmdy_struct, TMDY_M2M->cfgname);
	mod_cfg_file = fopen(TMDY_M2M->cfgname, "rb");
    }

    if (mod_cfg_file)
    {
	while (fgets(TMDY_M2M->linestring, 256, mod_cfg_file) != NULL)
	{
	    if (*TMDY_M2M->linestring == '#' || *TMDY_M2M->linestring == '\n' ||
		*TMDY_M2M->linestring == '\r')
		    continue;

	    sscanf(TMDY_M2M->linestring, "%d %s %d %s %d\n",
		   &mod_sample, program_str, &trans, finetune_str, &amp);

	    if (strchr(program_str, '!'))
	    	TMDY_M2M->silent_samples[mod_sample] = 1;

	    program = labs(atoi(program_str));
	    if ((slash_pos = strchr(program_str, '/')))
	    {
		TMDY_M2M->banks[mod_sample] = program;
		program = labs(atoi(slash_pos + 1));
	    }
	    TMDY_M2M->sample_to_program[mod_sample] = program;

	    TMDY_M2M->transpose[mod_sample] = trans;

	    if (strchr(finetune_str, '!'))
		TMDY_M2M->fine_tune[mod_sample] = 0;
	    else
		TMDY_M2M->fine_tune[mod_sample] = atof(finetune_str) * TMDY_M2M->pb_per_note;

	    TMDY_M2M->vol_amp[mod_sample] = amp;

	    if (strchr(program_str, '*'))
		TMDY_M2M->is_drum_sample[mod_sample] = 1;
	    else if ((chord_str = strchr(program_str, 'M')))
	    {
		chord_type = strchr(TMDY_M2M->chord_letters, 'M') - TMDY_M2M->chord_letters;
		chord_subtype = atoi(chord_str + 1);
		TMDY_M2M->sample_chords[mod_sample] = chord_type * 3 + chord_subtype;
	    }
	    else if ((chord_str = strchr(program_str, 'm')))
	    {
		chord_type = strchr(TMDY_M2M->chord_letters, 'm') - TMDY_M2M->chord_letters;
		chord_subtype = atoi(chord_str + 1);
		TMDY_M2M->sample_chords[mod_sample] = chord_type * 3 + chord_subtype;
	    }
	    else if ((chord_str = strchr(program_str, 'd')))
	    {
		chord_type = strchr(TMDY_M2M->chord_letters, 'd') - TMDY_M2M->chord_letters;
		chord_subtype = atoi(chord_str + 1);
		TMDY_M2M->sample_chords[mod_sample] = chord_type * 3 + chord_subtype;
	    }
	    else if ((chord_str = strchr(program_str, 'f')))
	    {
		chord_type = strchr(TMDY_M2M->chord_letters, 'f') - TMDY_M2M->chord_letters;
		chord_subtype = atoi(chord_str + 1);
		TMDY_M2M->sample_chords[mod_sample] = chord_type * 3 + chord_subtype;
	    }
	}
	fclose(mod_cfg_file);
    }
    else
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
		  "Couldn't open mod cfg file!  Proceeding without it.");
}



/* Fill the delta time byte array used in the mod->midi output routine */
int set_dt_array(tmdy_struct_ex_t* tmdy_struct, unsigned char *dt_array, int32 delta_time)
{
    int num_dt_bytes = 0, i = 0;
    int32 dt_byte;

    if (delta_time < 0)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "WTF?  Delta Time = %ld",
		  delta_time);
	delta_time = 0;
    }

    /* Delta time is variable length, max of 4 bytes long.
       Each byte contains 7 bits of the number.
       Every byte but the last one has an MSB of 1, while the last byte
       has an MSB of 0 to indicate it is the last byte.
     */

    dt_byte = (delta_time & 0x0FFFFFFF) >> 21;
    if (dt_byte)
    {
	TMDY_M2M->dt_array[i++] = dt_byte | 0x80;
	num_dt_bytes = 4;
    }

    dt_byte = (delta_time & 0x001FFFFF) >> 14;
    if (dt_byte || num_dt_bytes)
    {
	TMDY_M2M->dt_array[i++] = dt_byte | 0x80;
	if (!num_dt_bytes)
	    num_dt_bytes = 3;
    }

    dt_byte = (delta_time & 0x00003FFF) >> 7;
    if (dt_byte || num_dt_bytes)
    {
	TMDY_M2M->dt_array[i++] = dt_byte | 0x80;
	if (!num_dt_bytes)
	    num_dt_bytes = 2;
    }

    dt_byte = delta_time & 0x0000007F;
    TMDY_M2M->dt_array[i] = dt_byte;
    if (!num_dt_bytes)
	num_dt_bytes = 1;

    return num_dt_bytes;
}



void scan_ahead_for_m2m_tweaks(tmdy_struct_ex_t* tmdy_struct, MidiEvent * ev, int midi_ch, int midi_note,
			       int samplenum)
{

    int ch, event_type, init_ch, init_note, init_velocity;
    int32 bend = 0, lowbend = 0, highbend = 0;
    int32 pb_offset1 = 0, pb_offset2 = 0;
    int note_offset1 = 0, note_offset2 = 0;
    uint32 init_time, cur_time, old_time, cut_time = 0;
    Sample *sp;
    double a, bent_length = 0, delta_length;
    uint32 length;
    float root_freq, pitch, freq;

    init_time = cur_time = old_time = ev->time;
    init_ch = ev->channel;
    init_note = ev->a;
    init_velocity = ev->b;
    sp = TMDY_INSTRUM->special_patch[samplenum]->sample;
    root_freq = TMDY_FREQ->pitch_freq_table[36];
    length = sp->data_length >> FRACTION_BITS;

    /* check for a pitch bend prior to the note */
    if ((ev-1)->type == ME_PITCHWHEEL && (ev-1)->channel == init_ch)
    {
	bend = (ev-1)->b * 128 + (ev-1)->a;
	bend = ROUND((bend - 0x2000) * TMDY_M2M->old_pb_sensitivity /
	             (float) TMDY_M2M->pb_sensitivity);
	bend += TMDY_M2M->fine_tune[samplenum];

	if (bend < lowbend)
	    lowbend = bend;
	if (bend > highbend)
	    highbend = bend;

	pitch = init_note + TMDY_M2M->notes_per_pb * bend;
	freq = 13.75 * exp((pitch - 9) * 0.05776226505f);
    }
    else
        freq = TMDY_FREQ->pitch_freq_table[init_note];

    a = (freq * sp->sample_rate) /
	(root_freq * (TMDY_OUTPUT->play_mode)->rate);

    for (++ev; ev->type != ME_EOT; ev++)
    {
	event_type = ev->type;
	if (event_type != ME_ALL_NOTES_OFF &&
	    event_type != ME_NOTEOFF && event_type != ME_PITCHWHEEL)
		continue;

	ch = ev->channel;
	if (ch != init_ch)
	    continue;

	cur_time = ev->time;
	if (event_type == ME_ALL_NOTES_OFF)
	    break;

	if (event_type == ME_NOTEOFF)
	{
	    if (ev->a == init_note)
		break;
	    continue;
	}

	delta_length = (cur_time - old_time) * a;
	if (!cut_time && bent_length + delta_length > length)
	{
	    cut_time = ROUND(cur_time - (bent_length + delta_length - length) /
			     delta_length * (cur_time - old_time));
	}
	bent_length += delta_length;
	old_time = cur_time;

	bend = ev->b * 128 + ev->a;
	bend = ROUND((bend - 0x2000) * TMDY_M2M->old_pb_sensitivity /
	             (float) TMDY_M2M->pb_sensitivity);
	bend += TMDY_M2M->fine_tune[samplenum];

	if (bend < lowbend)
	    lowbend = bend;
	if (bend > highbend)
	    highbend = bend;
	    
	pitch = init_note + TMDY_M2M->notes_per_pb * bend;
	freq = 13.75 * exp((pitch - 9) * 0.05776226505f);

	a = (freq * sp->sample_rate) /
	    (root_freq * (TMDY_OUTPUT->play_mode)->rate);
    }

    delta_length = (cur_time - old_time) * a;
    if (!cut_time && bent_length + delta_length > length)
    {
	cut_time = ROUND(cur_time - (bent_length + delta_length - length) /
			 delta_length * (cur_time - old_time));
    }
    bent_length += delta_length;

    if (highbend > 8191)
    {
	pb_offset1 = highbend - 8191;
	note_offset1 = ceil(TMDY_M2M->notes_per_pb * pb_offset1);
	pb_offset1 = -note_offset1 * TMDY_M2M->pb_per_note;
    }
    if (lowbend < -8191)
    {
	pb_offset2 = lowbend - -8191;
	note_offset2 = floor(TMDY_M2M->notes_per_pb * pb_offset2);
	pb_offset2 = -note_offset2 * TMDY_M2M->pb_per_note;
    }

    if (note_offset1 > -note_offset2)
    {
	TMDY_M2M->tweak_note_offset[midi_ch] = note_offset1;
	TMDY_M2M->tweak_pb_offset[midi_ch] = pb_offset1;
    }
    else
    {
	TMDY_M2M->tweak_note_offset[midi_ch] = note_offset2;
	TMDY_M2M->tweak_pb_offset[midi_ch] = pb_offset2;
    }

    if (note_offset1 || note_offset2)
	TMDY_M2M->num_big_pitch_slides++;
    if (highbend - lowbend > 16382)
	TMDY_M2M->num_huge_pitch_slides++;

    /* hmm, need to kill this one early */
    TMDY_M2M->kill_early_time[init_ch] = 0;
    if (!(sp->modes & MODES_LOOPING))
    {
	if (bent_length > length)
	{
	    TMDY_M2M->kill_early_note[init_ch] = midi_note + TMDY_M2M->tweak_note_offset[midi_ch];
	    TMDY_M2M->kill_early_velocity[init_ch] = init_velocity;
	    TMDY_M2M->kill_early_time[init_ch] = cut_time;
	    TMDY_M2M->kill_early_ch[init_ch] = midi_ch;
	}
    }
}



void m2m_kill_notes_early(tmdy_struct_ex_t* tmdy_struct, MidiEvent * ev, double time)
{

    int i, j;
    int chord, chord_type, chord_subtype;
    int extra, newnote;
    uint32 kill_time;
    int kill_sample;
    int kill_ch;
    int kill_n;

    /* kill notes early if unlooped samples are causing problems */
    for (j = 0; j < 34; j++)
    {
	if (TMDY_M2M->kill_early_time[j] && TMDY_M2M->kill_early_time[j] <= ev->time)
	{
	    kill_sample = TMDY_M2M->current_track_sample[j];
	    kill_ch = TMDY_M2M->kill_early_ch[j];
	    kill_time = ROUND(time - ((ev->time - TMDY_M2M->kill_early_time[j]) *
			      TMDY_M2M->divisions_ratio) / TMDY_M2M->samples_per_tick);

	    /* looks like we got screwed by a TMDY_M2M->tempo event, so skip it */
	    if (kill_time < TMDY_M2M->last_track_event_time[kill_ch])
	    {
		TMDY_M2M->kill_early_time[j] = 0;
		continue;
	    }
	    /* yet another TMDY_M2M->tempo muck up */
	    if (TMDY_M2M->current_channel_note[kill_ch] != TMDY_M2M->kill_early_note[j])
	    {
		TMDY_M2M->kill_early_time[j] = 0;
		continue;
	    }
	    if (kill_time > TMDY_M2M->maxtime)
		TMDY_M2M->maxtime = kill_time;

	    kill_n = 3;
	    newnote = TMDY_M2M->kill_early_note[j];
	    while (newnote > 127)
	    	newnote -= 12;
	    while (newnote < 0)
	    	newnote += 12;
	    TMDY_M2M->event[0] = 0x80 | (kill_ch & 0x0F);
	    TMDY_M2M->event[1] = newnote;
	    TMDY_M2M->event[2] = TMDY_M2M->kill_early_velocity[j];
	    TMDY_M2M->current_track_note[j] = -1;
	    TMDY_M2M->current_channel_note[kill_ch] = -1;

	    /* resize the track event array */
	    TMDY_M2M->length = TMDY_M2M->track_size[kill_ch];
	    TMDY_M2M->num_dt_bytes = set_dt_array(tmdy_struct, TMDY_M2M->dt_array, kill_time -
					TMDY_M2M->last_track_event_time[kill_ch]);
	    TMDY_M2M->track_size[kill_ch] += kill_n + TMDY_M2M->num_dt_bytes;
	    TMDY_M2M->track_events[kill_ch] =
		TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->track_events[kill_ch],
			TMDY_M2M->track_size[kill_ch] * sizeof(unsigned char));

	    /* save the delta_time */
	    TMDY_M2M->p_track_event = TMDY_M2M->track_events[kill_ch] + TMDY_M2M->length;
	    for (i = 0; i < TMDY_M2M->num_dt_bytes; i++)
	    {
		TMDY_M2M->p_track_event[i] = TMDY_M2M->dt_array[i];
	    }

	    /* save the events */
	    TMDY_M2M->p_track_event += TMDY_M2M->num_dt_bytes;
	    for (i = 0; i < kill_n; i++)
	    {
		TMDY_M2M->p_track_event[i] = TMDY_M2M->event[i];
	    }

	    /* spawn extra events for chords */
	    chord = TMDY_M2M->sample_chords[kill_sample];
	    if (chord >= 0)
	    {
		extra = 2;
		TMDY_M2M->length = TMDY_M2M->track_size[kill_ch];
		TMDY_M2M->track_size[kill_ch] += 4 * extra;
		TMDY_M2M->track_events[kill_ch] =
		    TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->track_events[kill_ch],
			    TMDY_M2M->track_size[kill_ch] * sizeof(unsigned char));
		TMDY_M2M->p_track_event = TMDY_M2M->track_events[kill_ch] + TMDY_M2M->length;
		for (i = 0; i < 3; i++)
		{
		    chord_type = chord / 3;
		    chord_subtype = chord % 3;
		    newnote = TMDY_M2M->event[kill_n - 2] +
			TMDY_FREQ->chord_table[chord_type][chord_subtype][i];
		    if (newnote == TMDY_M2M->event[kill_n - 2])
			continue;
		    while (newnote > 127)
		    	newnote -= 12;
		    while (newnote < 0)
		    	newnote += 12;
		    TMDY_M2M->p_track_event[0] = 0x00;
		    TMDY_M2M->p_track_event[1] = TMDY_M2M->event[kill_n - 3];
		    TMDY_M2M->p_track_event[2] = newnote;
		    TMDY_M2M->p_track_event[3] = TMDY_M2M->event[kill_n - 1];
		    TMDY_M2M->p_track_event += 4;
		}
	    }

	    TMDY_M2M->last_track_event_time[kill_ch] = kill_time;
	    TMDY_M2M->kill_early_time[j] = 0;
	    TMDY_M2M->num_killed_early++;
	}
    }
}



void m2m_prescan(tmdy_struct_ex_t* tmdy_struct, MidiEvent * ev)
{

    int i, ch;

    /* find out which tracks will wind up with notes on them */
    for (; ev->type != ME_EOT; ev++)
    {
	if (ev->type == ME_NOTEON || ev->type == ME_SET_PATCH)
	{
	    ch = ev->channel;
	    if (ch >= 25)
		ch++;

	    if (ev->type == ME_NOTEON)
	    {
		if (TMDY_M2M->silent_samples[TMDY_M2M->current_track_sample[ch]])
		    continue;
		TMDY_M2M->tracks_useless[ch] = 0;

		/* move drums to drum channel */
		if (TMDY_M2M->is_drum_sample[TMDY_M2M->current_track_sample[ch]])
		    ch = 9;
		else if (ch == 9)
		{
		    TMDY_M2M->non_drums_on_drums = 1;
		    continue;
		}

		if (!TMDY_M2M->tracks_enabled[ch])
		{
		    TMDY_M2M->tracks_enabled[ch] = 1;
		    TMDY_M2M->num_tracks++;
		}
	    }
	    else
		TMDY_M2M->current_track_sample[ch] = ev->a;
	}
    }
    for (i = 0; i < 34; i++)
    {
	if (!TMDY_M2M->tracks_enabled[i])
	{
	    if (i != 9 && i != 25 && TMDY_M2M->first_free_track < 0)
		TMDY_M2M->first_free_track = i;
	}
	else
	{
	    if (TMDY_M2M->min_enabled_track < 0)
		TMDY_M2M->min_enabled_track = i;
	    TMDY_M2M->max_enabled_track = i;
	}
    }

    /* all tracks were filled, set it to the last track anyways */
    if (TMDY_M2M->first_free_track < 0)
	TMDY_M2M->first_free_track = 63;

    /* we're going to add another track to move stuff off of the drums */
    if (TMDY_M2M->non_drums_on_drums)
    {
	TMDY_M2M->tracks_enabled[TMDY_M2M->first_free_track] = 1;
	TMDY_M2M->num_tracks++;
    }

    /* re-initialize to unseen sample number to make sure it's not a drum */
    for (i = 0; i < 34; i++)
    {
	TMDY_M2M->current_track_sample[i] = 255;
    }

    /* Initialize Port Numbers for channels > 15 */
    for (i = 0; i < 34; i++)
    {
	if (TMDY_M2M->tracks_enabled[i])
	{
	    TMDY_M2M->length = TMDY_M2M->track_size[i];
	    TMDY_M2M->track_size[i] += 5;
	    TMDY_M2M->track_events[i] = TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->track_events[i], TMDY_M2M->track_size[i] *
				      sizeof(unsigned char));
	    TMDY_M2M->p_track_event = TMDY_M2M->track_events[i] + TMDY_M2M->length;
	    /* Port Change Event */
	    TMDY_M2M->p_track_event[0] = 0x00;
	    TMDY_M2M->p_track_event[1] = 0xFF;
	    TMDY_M2M->p_track_event[2] = 0x21;
	    TMDY_M2M->p_track_event[3] = 0x01;
	    TMDY_M2M->p_track_event[4] = i / 16;
	}
    }

    /* Issue Initial Drum Stuff */
    TMDY_M2M->length = TMDY_M2M->track_size[9];
    TMDY_M2M->track_size[9] += 15;
    TMDY_M2M->track_events[9] = TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->track_events[9], TMDY_M2M->track_size[9] *
			      sizeof(unsigned char));
    TMDY_M2M->p_track_event = TMDY_M2M->track_events[9] + TMDY_M2M->length;
    /* program change to Standard Drums */
    TMDY_M2M->p_track_event[0] = 0x00;
    TMDY_M2M->p_track_event[1] = 0xC9;
    TMDY_M2M->p_track_event[2] = 0x00;
    /* set initial volume */
    TMDY_M2M->p_track_event[3] = 0x00;
    TMDY_M2M->p_track_event[4] = 0xB9;
    TMDY_M2M->p_track_event[5] = 0x07;
    TMDY_M2M->p_track_event[6] = 127;
    /* set initial expression */
    TMDY_M2M->p_track_event[7] = 0x00;
    TMDY_M2M->p_track_event[8] = 0xB9;
    TMDY_M2M->p_track_event[9] = 0x0B;
    TMDY_M2M->p_track_event[10] = 127;
    /* set center pan */
    TMDY_M2M->p_track_event[11] = 0x00;
    TMDY_M2M->p_track_event[12] = 0xB9;
    TMDY_M2M->p_track_event[13] = 0x0A;
    TMDY_M2M->p_track_event[14] = 64;
}



void m2m_process_events(tmdy_struct_ex_t* tmdy_struct, MidiEvent * ev)
{

    int i;
    int moved_to_drums;
    int event_type;
    int skip_ch_expr_flag;
    uint32 oldtime = 0, deltatime;
    double time = 0;
    int chord, chord_type, chord_subtype;
    int ch, n, old_ch, newnote, mod_sample, expression;
    int extra;

    /* go through the list for real this time */
    for (; ev->type != ME_EOT; ev++)
    {
	/* convert timidity times to midi event times */
	TMDY_M2M->samples_per_tick = (double) (TMDY_OUTPUT->play_mode)->rate * (double) TMDY_M2M->tempo /
	    (double) 1000000 / (double) TMDY_M2M->orig_divisions;
	deltatime = ev->time - oldtime;
	oldtime = ev->time;
	time += (deltatime * TMDY_M2M->divisions_ratio) / TMDY_M2M->samples_per_tick;
	if (time > TMDY_M2M->maxtime)
	    TMDY_M2M->maxtime = time;

	m2m_kill_notes_early(tmdy_struct, ev, time);

	n = 0;
	ch = ev->channel;
	if (ch >= 25)
	    ch++;
	if (ev->type != ME_TEMPO && TMDY_M2M->tracks_useless[ch])
	    continue;
	if (ev->type != ME_TEMPO)
	    mod_sample = TMDY_M2M->current_track_sample[ch];

	/* skip silent sample events */
	if (TMDY_M2M->silent_samples[mod_sample] &&
	    (ev->type == ME_NOTEON || ev->type == ME_NOTEOFF ||
	     ev->type == ME_KEYPRESSURE)) continue;
	if (ev->type == ME_SET_PATCH && TMDY_M2M->silent_samples[ev->a])
	{
	    continue;
	}

	if (ev->type == ME_EXPRESSION && TMDY_M2M->silent_samples[ev->a] &&
	    !TMDY_M2M->current_track_note[ch])
	{
	    TMDY_M2M->orig_track_expr[ch] = ev->a;
	    continue;
	}

	skip_ch_expr_flag = 0;
	moved_to_drums = 0;
	old_ch = ch;
	if (ev->type != ME_TEMPO && ev->type != ME_SET_PATCH)
	{
	    /* move non-drums off to first free channel, drums to drum channel */
	    if (ch == 9 &&
		(!TMDY_M2M->is_drum_sample[mod_sample] ||
		 (ev->type != ME_NOTEON &&
		  ev->type != ME_NOTEOFF && ev->type != ME_KEYPRESSURE)))
	    {
		ch = TMDY_M2M->first_free_track;
	    }
	    else if (TMDY_M2M->is_drum_sample[mod_sample])
	    {
		if (ch != 9)
		{
		    ch = 9;
		    moved_to_drums = 1;
		}
	    }
	}

	event_type = ev->type;
	switch (ev->type)
	{
	case ME_NOTEOFF:
	    n = 3;

	    if (TMDY_M2M->is_drum_sample[mod_sample])
		newnote = TMDY_M2M->sample_to_program[mod_sample];
	    else
	    {
		newnote = ev->a + TMDY_M2M->transpose[mod_sample];
		newnote += TMDY_M2M->tweak_note_offset[ch];
	    }
	    while (newnote > 127)
	    	newnote -= 12;
	    while (newnote < 0)
	    	newnote += 12;

	    TMDY_M2M->event[0] = 0x80 | (ch & 0x0F);
	    TMDY_M2M->event[1] = newnote;
	    TMDY_M2M->event[2] = ev->b;

	    /* only issue a NOTEOFF is there is a note playing on the ch */
	    if (ch != 9 && TMDY_M2M->current_channel_note[ch] == -1)
		n = 0;

	    TMDY_M2M->current_channel_note[ch] = -1;
	    TMDY_M2M->current_track_note[old_ch] = -1;
	    break;

	case ME_NOTEON:
	    n = 3;

	    if (TMDY_M2M->is_drum_sample[mod_sample])
		newnote = TMDY_M2M->sample_to_program[mod_sample];
	    else
	    {
		newnote = ev->a + TMDY_M2M->transpose[mod_sample];
		scan_ahead_for_m2m_tweaks(tmdy_struct, ev, ch, newnote, mod_sample);
		newnote += TMDY_M2M->tweak_note_offset[ch];
	    }
	    while (newnote > 127)
	    	newnote -= 12;
	    while (newnote < 0)
	    	newnote += 12;

	    TMDY_M2M->event[0] = 0x90 | (ch & 0x0F);
	    TMDY_M2M->event[1] = newnote;
	    TMDY_M2M->event[2] = ev->b;

	    expression = ROUND(TMDY_M2M->orig_track_expr[old_ch] *
			       TMDY_M2M->vol_amp[mod_sample] / 100.0);

	    /* max expression at 127 */
	    if (expression > 127)
		expression = 127;

	    if (TMDY_M2M->is_drum_sample[mod_sample])
	    {
		TMDY_M2M->event[2] = expression;
	    }
	    /* current expression may not be what's wanted for the sample */
	    /* HACK -- insert a prior expression event */
	    else if (expression != TMDY_M2M->current_channel_expr[ch])
	    {
	    	/* NOTEON event */
		n = 11;
		TMDY_M2M->event[7] = 0x00;
		TMDY_M2M->event[8] = TMDY_M2M->event[0];
		TMDY_M2M->event[9] = TMDY_M2M->event[1];
		TMDY_M2M->event[10] = TMDY_M2M->event[2];

		/* non-linear expression event */
		TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
		TMDY_M2M->event[1] = 0x0B;
		TMDY_M2M->event[2] = TMDY_M2M->vol_nonlin_to_lin[expression][0];

		/* non-linear volume event */
		TMDY_M2M->event[3] = 0x00;
		TMDY_M2M->event[4] = 0xB0 | (ch & 0x0F);
		TMDY_M2M->event[5] = 0x07;
		TMDY_M2M->event[6] = TMDY_M2M->vol_nonlin_to_lin[expression][1];
		TMDY_M2M->current_channel_expr[ch] = expression;
	    }

	    TMDY_M2M->current_channel_note[ch] = newnote;
	    TMDY_M2M->current_track_note[old_ch] = newnote;
	    break;

	case ME_KEYPRESSURE:
	    n = 3;

	    if (TMDY_M2M->is_drum_sample[mod_sample])
		newnote = TMDY_M2M->sample_to_program[mod_sample];
	    else
	    {
		newnote = ev->a + TMDY_M2M->transpose[mod_sample];
		newnote += TMDY_M2M->tweak_note_offset[ch];
	    }
	    while (newnote > 127)
	    	newnote -= 12;
	    while (newnote < 0)
	    	newnote += 12;

	    TMDY_M2M->event[0] = 0xA0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = newnote;
	    TMDY_M2M->event[2] = ev->b;
	    break;

	case ME_SET_PATCH:
	    n = 2;
	    TMDY_M2M->current_track_sample[old_ch] = ev->a;

	    if (TMDY_M2M->is_drum_sample[ev->a])
	    {
		ch = 9;
		TMDY_M2M->current_channel_program[ch] = TMDY_M2M->sample_to_program[ev->a];

		/* don't emit any event at all */
		n = 0;

		/* change drum banks if necessary */
		if (TMDY_M2M->banks[ev->a] != TMDY_M2M->current_channel_bank[ch])
		{
		    if (n)
			TMDY_M2M->event[n++] = 0x00;
		    TMDY_M2M->event[n++] = 0xC0 | (ch & 0x0F);
		    TMDY_M2M->event[n++] = TMDY_M2M->current_channel_bank[ch] = TMDY_M2M->banks[ev->a];
		}
	    }
	    else
	    {
		if (ch == 9)
		    ch = TMDY_M2M->first_free_track;

		/* program already set, no need to change it */
		if (TMDY_M2M->sample_to_program[ev->a] == TMDY_M2M->current_channel_program[ch] &&
		    TMDY_M2M->banks[ev->a] == TMDY_M2M->current_channel_bank[ch])
		{
		    n = 0;
		}
		/* no need to change bank, it's already set correctly */
		else if (TMDY_M2M->banks[ev->a] == TMDY_M2M->current_channel_bank[ch])
		{
		    TMDY_M2M->event[0] = 0xC0 | (ch & 0x0F);
		    TMDY_M2M->event[1] = TMDY_M2M->sample_to_program[ev->a];
		}
		/* need to change bank to that of the new instrument */
		else
		{
		    n = 10;
		    /* Bank Select MSB */
		    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
		    TMDY_M2M->event[1] = 0x00;
		    TMDY_M2M->event[2] = TMDY_M2M->banks[ev->a];
		    TMDY_M2M->current_channel_bank[ch] = TMDY_M2M->banks[ev->a];

		    /* Bank Select LSB */
		    TMDY_M2M->event[3] = 0x00;
		    TMDY_M2M->event[4] = 0xB0 | (ch & 0x0F);
		    TMDY_M2M->event[5] = 0x20;
		    TMDY_M2M->event[6] = 0;

		    /* Program Change */
		    TMDY_M2M->event[7] = 0x00;
		    TMDY_M2M->event[8] = 0xC0 | (ch & 0x0F);
		    TMDY_M2M->event[9] = TMDY_M2M->sample_to_program[ev->a];
		}
		TMDY_M2M->current_channel_program[ch] = TMDY_M2M->sample_to_program[ev->a];
	    }
	    break;

	case ME_PITCHWHEEL:
	    n = 3;
	    TMDY_M2M->event[0] = 0xE0 | (ch & 0x0F);

	    /* max pitch bend sensitivity is 24, scale stuff */
	    TMDY_M2M->value = ev->b * 128 + ev->a;
	    TMDY_M2M->value = ROUND((TMDY_M2M->value - 0x2000) * TMDY_M2M->old_pb_sensitivity /
	                  (float) TMDY_M2M->pb_sensitivity);
	    TMDY_M2M->value += TMDY_M2M->fine_tune[mod_sample];
	    TMDY_M2M->value += TMDY_M2M->tweak_pb_offset[ch];

	    /* fudge stuff by an octave multiple if it's out of bounds */
	    /* this hasn't been tested much, but it seems to sound OK */
	    if (TMDY_M2M->value > 8191)
	    {
		TMDY_M2M->value = (TMDY_M2M->value % 8191) % 4095 + 4096;
	    }
	    else if (TMDY_M2M->value < -8191)
	    {
		TMDY_M2M->value = -TMDY_M2M->value;
		TMDY_M2M->value = (TMDY_M2M->value % 8191) % 4095 + 4096;
		TMDY_M2M->value = -TMDY_M2M->value;
	    }

	    /* add the offset back in, save it */
	    TMDY_M2M->value += 0x2000;
	    TMDY_M2M->event[1] = TMDY_M2M->value & 0x7F;
	    TMDY_M2M->event[2] = TMDY_M2M->value >> 7;
	    break;

	case ME_DATA_ENTRY_MSB:
	    n = 3;
	    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = 0x06;
	    TMDY_M2M->event[2] = ev->a;

	    /* pitch sensitivity maxes out at 24, not 128 */
	    if (TMDY_M2M->rpn_msb == 0 && TMDY_M2M->rpn_lsb == 0)
	    {
		TMDY_M2M->old_pb_sensitivity = TMDY_M2M->pb_sensitivity = ev->a;
		if (TMDY_M2M->pb_sensitivity > MAX_PB_SENSITIVITY)
		    TMDY_M2M->pb_sensitivity = MAX_PB_SENSITIVITY;
		TMDY_M2M->event[2] = TMDY_M2M->pb_sensitivity;
		TMDY_M2M->notes_per_pb = TMDY_M2M->pb_sensitivity / 8191.0;
		TMDY_M2M->pb_per_note = 8191.0 / TMDY_M2M->pb_sensitivity;
	    }
	    break;

	case ME_MAINVOLUME:
	    n = 3;
	    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = 0x07;
	    TMDY_M2M->event[2] = ev->a;
	    break;

	case ME_PAN:
	    n = 3;
	    TMDY_M2M->track_pans[ch] = ev->a;
	    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = 0x0A;
	    TMDY_M2M->event[2] = ev->a;
	    break;

	case ME_EXPRESSION:
	    n = 7;

	    TMDY_M2M->orig_track_expr[old_ch] = ev->a;

	    expression = ROUND(ev->a * TMDY_M2M->vol_amp[mod_sample] / 100.0);

	    /* max expression at 127 */
	    if (expression > 127)
		expression = 127;

	    if (TMDY_M2M->current_channel_expr[ch] == expression)
		skip_ch_expr_flag = 1;
	    else
		TMDY_M2M->current_channel_expr[ch] = expression;

#ifdef MUTATE_EXPRESSION_TO_KEYPRESSURE
	    /* HACK - mutate it into a KEYPRESSUE event */
	    /* but only if there's a note playing */
	    if (TMDY_M2M->current_track_note[old_ch] >= 0) {
	    	event_type = ME_KEYPRESSURE;
		TMDY_M2M->event[0] = 0xA0 | (ch & 0x0F);
		TMDY_M2M->event[1] = TMDY_M2M->current_track_note[old_ch];
		TMDY_M2M->event[2] = expression;
	    }
	    else
	    	n = 0;
#endif

	    /* non-linear expression event */
	    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = 0x0B;
	    TMDY_M2M->event[2] = TMDY_M2M->vol_nonlin_to_lin[expression][0];

	    /* non-linear volume event */
	    TMDY_M2M->event[3] = 0x00;
	    TMDY_M2M->event[4] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[5] = 0x07;
	    TMDY_M2M->event[6] = TMDY_M2M->vol_nonlin_to_lin[expression][1];
	    break;

	case ME_RPN_LSB:
	    n = 3;
	    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = 0x64;
	    TMDY_M2M->event[2] = ev->a;
	    TMDY_M2M->rpn_lsb = ev->a;
	    break;

	case ME_RPN_MSB:
	    n = 3;
	    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = 0x65;
	    TMDY_M2M->event[2] = ev->a;
	    TMDY_M2M->rpn_msb = ev->a;
	    break;

	case ME_TEMPO:
	    n = 6;
	    TMDY_M2M->event[0] = 0xFF;
	    TMDY_M2M->event[1] = 0x51;
	    TMDY_M2M->event[2] = 3;
	    TMDY_M2M->event[3] = ev->a;
	    TMDY_M2M->event[4] = ev->b;
	    TMDY_M2M->event[5] = ch;
	    TMDY_M2M->tempo = ch + ev->b * 256 + ev->a * 65536;
	    break;

	case ME_ALL_NOTES_OFF:
	    n = 3;
	    TMDY_M2M->event[0] = 0xB0 | (ch & 0x0F);
	    TMDY_M2M->event[1] = 123;
	    TMDY_M2M->event[2] = 0;
	    break;

/*	case ME_DRUMPART:
	    break;
*/
	}

	/* I here by decree that all TMDY_M2M->tempo events shall go on the first ch */
	if (event_type == ME_TEMPO)
	    ch = TMDY_M2M->min_enabled_track;

	/* ah ha, we shall keep this event */
	if (n)
	{
	    if (!(ch == 9 && (event_type == ME_EXPRESSION ||
	    		      event_type == ME_PITCHWHEEL ||
	    		      event_type == ME_PAN)) && !skip_ch_expr_flag)
	    {
		/* resize the track event array */
		TMDY_M2M->length = TMDY_M2M->track_size[ch];
		TMDY_M2M->num_dt_bytes = set_dt_array(tmdy_struct, TMDY_M2M->dt_array, time -
					    TMDY_M2M->last_track_event_time[ch]);
		TMDY_M2M->track_size[ch] += n + TMDY_M2M->num_dt_bytes;
		TMDY_M2M->track_events[ch] = TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->track_events[ch], TMDY_M2M->track_size[ch] *
					   sizeof(unsigned char));

		/* save the delta_time */
		TMDY_M2M->p_track_event = TMDY_M2M->track_events[ch] + TMDY_M2M->length;
		for (i = 0; i < TMDY_M2M->num_dt_bytes; i++)
		{
		    TMDY_M2M->p_track_event[i] = TMDY_M2M->dt_array[i];
		}

		/* save the events */
		TMDY_M2M->p_track_event += TMDY_M2M->num_dt_bytes;
		for (i = 0; i < n; i++)
		{
		    TMDY_M2M->p_track_event[i] = TMDY_M2M->event[i];
		}

		/* spawn extra events for chords */
		/* don't forget that there could be a preceeding expr event */
		chord = TMDY_M2M->sample_chords[mod_sample];
		if (chord >= 0 &&
		    (event_type == ME_NOTEON ||
		     event_type == ME_NOTEOFF || event_type == ME_KEYPRESSURE))
		{
		    extra = 2;
		    TMDY_M2M->length = TMDY_M2M->track_size[ch];
		    TMDY_M2M->track_size[ch] += 4 * extra;
		    TMDY_M2M->track_events[ch] =
			TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->track_events[ch],
				TMDY_M2M->track_size[ch] * sizeof(unsigned char));
		    TMDY_M2M->p_track_event = TMDY_M2M->track_events[ch] + TMDY_M2M->length;
		    for (i = 0; i < 3; i++)
		    {
			chord_type = chord / 3;
			chord_subtype = chord % 3;
			newnote = TMDY_M2M->event[n - 2] +
			    TMDY_FREQ->chord_table[chord_type][chord_subtype][i];
			if (newnote == TMDY_M2M->event[n - 2])
			    continue;
			while (newnote > 127)
			    newnote -= 12;
			while (newnote < 0)
			    newnote += 12;
			TMDY_M2M->p_track_event[0] = 0x00;
			TMDY_M2M->p_track_event[1] = TMDY_M2M->event[n - 3];
			TMDY_M2M->p_track_event[2] = newnote;
			TMDY_M2M->p_track_event[3] = TMDY_M2M->event[n - 1];
			TMDY_M2M->p_track_event += 4;
		    }
		}

		TMDY_M2M->last_track_event_time[ch] = time;
	    }

	    /* moved it for drums, issue control events to old channel too */
	    if (moved_to_drums && old_ch != TMDY_M2M->first_free_track &&
		event_type != ME_NOTEON &&
		event_type != ME_NOTEOFF &&
		event_type != ME_KEYPRESSURE &&
		!(event_type == ME_EXPRESSION &&
		  (TMDY_M2M->current_channel_note[old_ch] < 0 ||
		   TMDY_M2M->current_channel_expr[old_ch] == expression)) &&
		event_type != ME_TEMPO)
	    {
		/* resize the track event array */
		TMDY_M2M->length = TMDY_M2M->track_size[old_ch];
		TMDY_M2M->num_dt_bytes = set_dt_array(tmdy_struct, TMDY_M2M->dt_array, time -
					    TMDY_M2M->last_track_event_time[old_ch]);
		TMDY_M2M->track_size[old_ch] += n + TMDY_M2M->num_dt_bytes;
		TMDY_M2M->track_events[old_ch] = TMDY_COMMON->safe_realloc(tmdy_struct, TMDY_M2M->track_events[old_ch],
					       TMDY_M2M->track_size[old_ch] *
					       sizeof(unsigned char));

		/* save the delta_time */
		TMDY_M2M->p_track_event = TMDY_M2M->track_events[old_ch] + TMDY_M2M->length;
		for (i = 0; i < TMDY_M2M->num_dt_bytes; i++)
		{
		    TMDY_M2M->p_track_event[i] = TMDY_M2M->dt_array[i];
		}

		/* replace channel nibble with old channel */
		TMDY_M2M->event[0] = (TMDY_M2M->event[0] & 0xF0) | (old_ch & 0x0F);

		/* save the events */
		TMDY_M2M->p_track_event += TMDY_M2M->num_dt_bytes;
		for (i = 0; i < n; i++)
		{
		    TMDY_M2M->p_track_event[i] = TMDY_M2M->event[i];
		}
		TMDY_M2M->last_track_event_time[old_ch] = time;
	    }
	}
    }
}



void m2m_output_midi_file(tmdy_struct_ex_t* tmdy_struct)
{

    FILE *outfile;
    int extra;
    int i, j;

    outfile = fopen(TMDY_M2M->actual_outname, "wb");
    if (!outfile)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
		  "Uh oh, can't open '%s' output file.  Bombing out...",
		  TMDY_M2M->actual_outname);
	return;
    }

    /* finish up the file header */
    TMDY_M2M->header[10] = (TMDY_M2M->num_tracks & 0xFF00) >> 8;
    TMDY_M2M->header[11] = TMDY_M2M->num_tracks & 0x00FF;
    TMDY_M2M->header[12] = (TMDY_M2M->divisions & 0xFF00) >> 8;
    TMDY_M2M->header[13] = TMDY_M2M->divisions & 0x00FF;

    /* output the file header */
    for (i = 0; i < 14; i++)
    {
	fprintf(outfile, "%c", TMDY_M2M->header[i]);
    }

    /* output each track */
    for (i = 0; i < 34; i++)
    {
	if (!TMDY_M2M->tracks_enabled[i])
	    continue;

	/* do the track header */
	for (j = 0; j < 4; j++)
	{
	    fprintf(outfile, "%c", TMDY_M2M->mtrk[j]);
	}

	TMDY_M2M->length = TMDY_M2M->track_size[i] + 4;
	extra = 4;

	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "Track %d Size %d", i, TMDY_M2M->length);
	fprintf(outfile, "%c", (TMDY_M2M->length & 0xFF000000) >> 24);
	fprintf(outfile, "%c", (TMDY_M2M->length & 0x00FF0000) >> 16);
	fprintf(outfile, "%c", (TMDY_M2M->length & 0x0000FF00) >> 8);
	fprintf(outfile, "%c", TMDY_M2M->length & 0x000000FF);

	/* write the events */
	TMDY_M2M->p_track_event = TMDY_M2M->track_events[i];
	for (j = 0; j < TMDY_M2M->length - extra; j++, TMDY_M2M->p_track_event++)
	{
	    fprintf(outfile, "%c", *TMDY_M2M->p_track_event);
	}

	/* write the terminal event */
	fprintf(outfile, "%c", 0x00);
	fprintf(outfile, "%c", 0xFF);
	fprintf(outfile, "%c", 0x2F);
	fprintf(outfile, "%c", 0x00);
    }
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "Number of tracks actually used: %d",
	      TMDY_M2M->num_tracks);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "Track accepting drum refugees: %d",
	      TMDY_M2M->first_free_track);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
	      "Number of unlooped notes killed early: %ld", TMDY_M2M->num_killed_early);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
	      "Number of pitch slides > 2 octaves: %ld", TMDY_M2M->num_big_pitch_slides);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
	      "Number of pitch slides > 4 octaves: %ld",
	      TMDY_M2M->num_huge_pitch_slides);

    fclose(outfile);
}



void convert_mod_to_midi_file(tmdy_struct_ex_t* tmdy_struct, MidiEvent * ev)
{

    int i;

    TMDY_READMIDI->change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);

    initialize_m2m_stuff(tmdy_struct);

    /* this either isn't a MOD, or it doesn't have any samples... */
    if (!TMDY_M2M->maxsample)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL,
		  "Aborting!  This doesn't look like a MOD file!");
	return;
    }

    read_m2m_cfg_file(tmdy_struct);

    m2m_prescan(tmdy_struct, ev);
    m2m_process_events(tmdy_struct, ev);
    m2m_output_midi_file(tmdy_struct);

    /* free track event arrays */
    for (i = 0; i < 34; i++) {
	if (TMDY_M2M->track_events[i])
	    free(TMDY_M2M->track_events[i]);
    }
}




m2m_ex_t* new_m2m(tmdy_struct_ex_t *tmdy_struct){
	int i, j;
	m2m_ex_t* m2m_ex;

	m2m_ex=(m2m_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(m2m_ex_t));
	
	m2m_ex->convert_mod_to_midi_file=convert_mod_to_midi_file;
	
	for(i=0;i<14;i++){
		m2m_ex->header[i]=init_header[i];
	}
	for(i=0;i<4;i++){
		m2m_ex->mtrk[i]=init_mtrk[i];
	}
	for(i=0;i<4;i++){
		m2m_ex->chord_letters[i]=init_chord_letters[i];
	}
	for(i=0;i<128;i++){
		for(j=0;j<2;j++){
			m2m_ex->vol_nonlin_to_lin[i][j]=init_vol_nonlin_to_lin[i][j];
		}
	}
	m2m_ex->outname = NULL;
	m2m_ex->cfgname = NULL;
	m2m_ex->num_tracks = 0;
	m2m_ex->tempo = 500000;
	m2m_ex-> maxtime = 0;
	m2m_ex->num_killed_early = 0;
	m2m_ex->num_big_pitch_slides = 0;
	m2m_ex->num_huge_pitch_slides = 0;
	m2m_ex-> pb_sensitivity = 24;
	m2m_ex->old_pb_sensitivity = 128;
	m2m_ex->notes_per_pb = 24.0 / 8191.0;
	m2m_ex->pb_per_note = 8191.0 / 24.0;
	m2m_ex->rpn_msb = 0;
	m2m_ex->rpn_lsb = 0;
	m2m_ex->min_enabled_track = -1;
	m2m_ex->first_free_track = -1;
	m2m_ex->non_drums_on_drums = 0;
	
	return m2m_ex;
}
void destroy_m2m(m2m_ex_t* m2m){	
	free(m2m);
}
