/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * REVERB EFFECT FOR TIMIDITY++-1.X (Version 0.06e  1999/1/28)
 * 
 * Copyright (C) 1997,1998,1999  Masaki Kiryu <mkiryu@usa.net>
 *                           (http://w3mb.kcom.ne.jp/~mkiryu/)
 *
 * reverb.c  -- main reverb engine.
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include "tmdy_struct.h"
#include "timidity.h"
#include "controls.h"
#include "tables.h"
#include "common.h"
#include "output.h"
#include "reverb.h"
#include "instrum.h"
#include "playmidi.h"
#include <math.h>
#include <stdlib.h>

#include "reverb_prv.h"




/*								//guha
enum play_system_modes
{
    DEFAULT_SYSTEM_MODE,
    GM_SYSTEM_MODE,
    GS_SYSTEM_MODE,
    XG_SYSTEM_MODE
};
*/
//extern int play_system_mode;






#define rev_memset(xx)     memset(xx,0,sizeof(xx));

#define rev_ptinc() \
REVERB->spt0++; if(REVERB->spt0 == REVERB->def_rpt0) REVERB->spt0 = 0;\
REVERB->spt1++; if(REVERB->spt1 == REVERB->rpt1) REVERB->spt1 = 0;\
REVERB->spt2++; if(REVERB->spt2 == REVERB->rpt2) REVERB->spt2 = 0;\
REVERB->spt3++; if(REVERB->spt3 == REVERB->rpt3) REVERB->spt3 = 0;

#define MASTER_CHORUS_LEVEL 1.7
#define MASTER_DELAY_LEVEL 3.25

static void do_shelving_filter(tmdy_struct_ex_t *tmdy_struct, register int32 *, int32, int32 *, int32 *);
static void do_freeverb(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count);
static void alloc_revmodel(tmdy_struct_ex_t *tmdy_struct);

#if OPT_MODE != 0 && _MSC_VER
void set_dry_signal(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count)
{
	int32 *dbuf = REVERB->direct_buffer;
	_asm {
		mov		ecx, [count]
		mov		esi, [buf]
		test	ecx, ecx
		jz		short L2
		mov		edi, [dbuf]
L1:		mov		eax, [esi]
		mov		ebx, [edi]
		add		esi, 4
		add		ebx, eax
		mov		[edi], ebx
		add		edi, 4
		dec		ecx
		jnz		L1
L2:
	}
}
#else
void set_dry_signal(tmdy_struct_ex_t *tmdy_struct, register int32 *buf, int32 n)
{
#if USE_ALTIVEC
  if(is_altivec_available()) {
    v_set_dry_signal(REVERB->direct_buffer,buf,n);
  } else {
#endif
    register int32 i;
  	register int32 *dbuf = REVERB->direct_buffer;
  	
    for(i=n-1;i>=0;i--)
    {
    	dbuf[i] += buf[i];
    }
#if USE_ALTIVEC
  }
#endif
}
#endif

void mix_dry_signal(tmdy_struct_ex_t *tmdy_struct, register int32 *buf, int32 n)
{
 	memcpy(buf,REVERB->direct_buffer,sizeof(int32) * n);
	memset(REVERB->direct_buffer,0,sizeof(int32) * n);
}

#if OPT_MODE != 0
#if _MSC_VER
void set_ch_reverb(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count, int32 level)
{
	int32 *dbuf = REVERB->effect_buffer;
	level = TIM_FSCALE(level / 127.0 * REVERB->REV_INP_LEV, 24);

	_asm {
		mov		ecx, [count]
		mov		esi, [buf]
		mov		ebx, [level]
		test	ecx, ecx
		jz		short L2
		mov		edi, [dbuf]
L1:		mov		eax, [esi]
		imul	ebx
		shr		eax, 24
		shl		edx, 8
		or		eax, edx	/* u */
		mov		edx, [edi]	/* v */
		add		esi, 4		/* u */	
		add		edx, eax	/* v */
		mov		[edi], edx	/* u */
		add		edi, 4		/* v */
		dec		ecx			/* u */
		jnz		L1			/* v */
L2:
	}
}
#else
void set_ch_reverb(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count, int32 level)
{
    int32 i, *dbuf = REVERB->effect_buffer;
    level = TIM_FSCALE(level / 127.0 * REVERB->REV_INP_LEV, 24);

	for(i=count-1;i>=0;i--) {dbuf[i] += imuldiv24(buf[i], level);}
}
#endif	/* _MSC_VER */
#else
void set_ch_reverb(tmdy_struct_ex_t *tmdy_struct, register int32 *sbuffer, int32 n, int32 level)
{
    register int32  i;
    FLOAT_T send_level = (FLOAT_T)level / 127.0 * REVERB->REV_INP_LEV;
	
	for(i = 0; i < n; i++)
    {
        REVERB->effect_buffer[i] += sbuffer[i] * send_level;
    }
}
#endif /* OPT_MODE != 0 */


#if OPT_MODE != 0
void do_standard_reverb(tmdy_struct_ex_t *tmdy_struct, register int32 *comp, int32 n)
{
	int32 i, fixp, s, t;

    for(i = 0; i < n; i++)
    {
        /* L */
        fixp = REVERB->effect_buffer[i];
        REVERB->effect_buffer[i] = 0;

        REVERB->LPFL = imuldiv24(REVERB->LPFL,REV_LPF_LEV) + imuldiv24(REVERB->buf2_L[REVERB->spt2] + REVERB->tb,REV_LPF_INP) + imuldiv24(REVERB->ta,REV_WIDTH);
        REVERB->ta = REVERB->buf3_L[REVERB->spt3];
        s  = REVERB->buf3_L[REVERB->spt3] = REVERB->buf0_L[REVERB->spt0];
        REVERB->buf0_L[REVERB->spt0] = -REVERB->LPFL;

        t = imuldiv24(REVERB->HPFL + fixp,REV_HPF_LEV);
        REVERB->HPFL = t - fixp;

        REVERB->buf2_L[REVERB->spt2] = imuldiv24(s - imuldiv24(fixp,REV_FBK_LEV),REV_CMIX_LEV);
        REVERB->tb = REVERB->buf1_L[REVERB->spt1];
        REVERB->buf1_L[REVERB->spt1] = t;

        REVERB->EPFL = imuldiv24(REVERB->EPFL,REV_EPF_LEV) + imuldiv24(REVERB->ta,REV_EPF_INP);
        comp[i] += REVERB->ta + REVERB->EPFL;

        /* R */
        fixp = REVERB->effect_buffer[++i];
        REVERB->effect_buffer[i] = 0;

        REVERB->LPFR = imuldiv24(REVERB->LPFR,REV_LPF_LEV) + imuldiv24(REVERB->buf2_R[REVERB->spt2] + REVERB->tb,REV_LPF_INP) + imuldiv24(REVERB->ta,REV_WIDTH);
        REVERB->ta = REVERB->buf3_R[REVERB->spt3];
        s  = REVERB->buf3_R[REVERB->spt3] = REVERB->buf0_R[REVERB->spt0];
        REVERB->buf0_R[REVERB->spt0] = REVERB->LPFR;

        t = imuldiv24(REVERB->HPFR + fixp,REV_HPF_LEV);
        REVERB->HPFR = t - fixp;

        REVERB->buf2_R[REVERB->spt2] = imuldiv24(s - imuldiv24(fixp,REV_FBK_LEV),REV_CMIX_LEV);
        REVERB->tb = REVERB->buf1_R[REVERB->spt1];
        REVERB->buf1_R[REVERB->spt1] = t;

        REVERB->EPFR = imuldiv24(REVERB->EPFR,REV_EPF_LEV) + imuldiv24(REVERB->ta,REV_EPF_INP);
        comp[i] += REVERB->ta + REVERB->EPFR;

        rev_ptinc();
    }
}
#else
void do_standard_reverb(tmdy_struct_ex_t *tmdy_struct, int32 *comp, int32 n)
{
    int32  fixp, s, t, i;

    for(i = 0; i < n; i++)
    {
        /* L */
        fixp = REVERB->effect_buffer[i];
        REVERB->effect_buffer[i] = 0;

        REVERB->LPFL = REVERB->LPFL*REV_LPF_LEV + (REVERB->buf2_L[REVERB->spt2]EDDD>tb)*REV_LPF_INP + REVERB->ta*REV_WIDTH;
        REVERB->ta = REVERB->buf3_L[REVERB->spt3];
        s  = REVERB->buf3_L[REVERB->spt3] = REVERB->buf0_L[REVERB->spt0];
        REVERB->buf0_L[REVERB->spt0] = -REVERB->LPFL;

        t = (REVERB->HPFL + fixp) * REV_HPF_LEV;
        REVERB->HPFL = t - fixp;

        REVERB->buf2_L[REVERB->spt2] = (s - fixp * REV_FBK_LEV) * REV_CMIX_LEV;
        REVERB->tb = REVERB->buf1_L[REVERB->spt1];
        REVERB->buf1_L[REVERB->spt1] = t;

        REVERB->EPFL = REVERB->EPFL * REV_EPF_LEV + REVERB->ta * REV_EPF_INP;
        comp[i] += REVERB->ta + REVERB->EPFL;
        REVERB->direct_buffer[i] = 0;

        /* R */
        fixp = REVERB->effect_buffer[++i];
        REVERB->effect_buffer[i] = 0;

        REVERB->LPFR = REVERB->LPFR*REV_LPF_LEV + (REVERB->buf2_R[REVERB->spt2]EDDD>tb)*REV_LPF_INP + REVERB->ta*REV_WIDTH;
        REVERB->ta = REVERB->buf3_R[REVERB->spt3];
        s  = REVERB->buf3_R[REVERB->spt3] = REVERB->buf0_R[REVERB->spt0];
        REVERB->buf0_R[REVERB->spt0] = REVERB->LPFR;

        t = (REVERB->HPFR + fixp) * REV_HPF_LEV;
        REVERB->HPFR = t - fixp;

        REVERB->buf2_R[REVERB->spt2] = (s - fixp * REV_FBK_LEV) * REV_CMIX_LEV;
        REVERB->tb = REVERB->buf1_R[REVERB->spt1];
        REVERB->buf1_R[REVERB->spt1] = t;

        REVERB->EPFR = REVERB->EPFR * REV_EPF_LEV + REVERB->ta * REV_EPF_INP;
        comp[i] += REVERB->ta + REVERB->EPFR;
        REVERB->direct_buffer[i] = 0;

        rev_ptinc();
    }
}
#endif /* OPT_MODE != 0 */

void do_ch_reverb(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	int i;
	if((TMDY_PLAYMIDI->opt_reverb_control == 3 || REVERB->opt_effect_quality >= 1) && REVERB->reverb_status.pre_lpf) {
		do_shelving_filter(tmdy_struct, REVERB->effect_buffer, count, REVERB->reverb_status.high_coef, REVERB->reverb_status.high_val);
	}
	if(TMDY_PLAYMIDI->opt_reverb_control == 3 || REVERB->opt_effect_quality >= 2) {
		do_freeverb(tmdy_struct, buf, count);
	} else {
		do_standard_reverb(tmdy_struct, buf, count);
	}
}

void do_reverb(tmdy_struct_ex_t *tmdy_struct, int32 *comp, int32 n)
{
    int32  fixp, s, t, i;

    for(i = 0; i < n; i++)
    {
        /* L */
        fixp = comp[i];

        REVERB->LPFL = REVERB->LPFL*REV_LPF_LEV + (REVERB->buf2_L[REVERB->spt2]+REVERB->tb)*REV_LPF_INP + REVERB->ta*REV_WIDTH;
        REVERB->ta = REVERB->buf3_L[REVERB->spt3];
        s  = REVERB->buf3_L[REVERB->spt3] = REVERB->buf0_L[REVERB->spt0];
        REVERB->buf0_L[REVERB->spt0] = -REVERB->LPFL;

        t = (REVERB->HPFL + fixp) * REV_HPF_LEV;
        REVERB->HPFL = t - fixp;

        REVERB->buf2_L[REVERB->spt2] = (s - fixp * REV_FBK_LEV) * REV_NMIX_LEV;
        REVERB->tb = REVERB->buf1_L[REVERB->spt1];
        REVERB->buf1_L[REVERB->spt1] = t;

        REVERB->EPFL = REVERB->EPFL * REV_EPF_LEV + REVERB->ta * REV_EPF_INP;
        comp[i] = REVERB->ta + REVERB->EPFL + fixp;

        /* R */
        fixp = comp[++i];

        REVERB->LPFR = REVERB->LPFR*REV_LPF_LEV + (REVERB->buf2_R[REVERB->spt2]+REVERB->tb)*REV_LPF_INP + REVERB->ta*REV_WIDTH;
        REVERB->ta = REVERB->buf3_R[REVERB->spt3];
        s  = REVERB->buf3_R[REVERB->spt3] = REVERB->buf0_R[REVERB->spt0];
        REVERB->buf0_R[REVERB->spt0] = REVERB->LPFR;

        t = (REVERB->HPFR + fixp) * REV_HPF_LEV;
        REVERB->HPFR = t - fixp;

        REVERB->buf2_R[REVERB->spt2] = (s - fixp * REV_FBK_LEV) * REV_NMIX_LEV;
        REVERB->tb = REVERB->buf1_R[REVERB->spt1];
        REVERB->buf1_R[REVERB->spt1] = t;

        REVERB->EPFR = REVERB->EPFR * REV_EPF_LEV + REVERB->ta * REV_EPF_INP;
        comp[i] = REVERB->ta + REVERB->EPFR + fixp;

        rev_ptinc();
    }
}

void do_mono_reverb(tmdy_struct_ex_t *tmdy_struct, int32 *comp, int32 n)
{
    int32  fixp, s, t, i;

    for(i = 0; i < n; i++)
    {
        /* L */
        fixp = comp[i] * REV_MONO_LEV;

        REVERB->LPFL = REVERB->LPFL*REV_LPF_LEV + (REVERB->buf2_L[REVERB->spt2]+REVERB->tb)*REV_LPF_INP + REVERB->ta*REV_WIDTH;
        REVERB->ta = REVERB->buf3_L[REVERB->spt3];
        s  = REVERB->buf3_L[REVERB->spt3] = REVERB->buf0_L[REVERB->spt0];
        REVERB->buf0_L[REVERB->spt0] = -REVERB->LPFL;

        t = (REVERB->HPFL + fixp) * REV_HPF_LEV;
        REVERB->HPFL = t - fixp;

        REVERB->buf2_L[REVERB->spt2] = (s - fixp * REV_FBK_LEV) * REV_NMIX_LEV;
        REVERB->tb = REVERB->buf1_L[REVERB->spt1];
        REVERB->buf1_L[REVERB->spt1] = t;

        /* R */
        REVERB->LPFR = REVERB->LPFR*REV_LPF_LEV + (REVERB->buf2_R[REVERB->spt2]+REVERB->tb)*REV_LPF_INP + REVERB->ta*REV_WIDTH;
        REVERB->ta = REVERB->buf3_R[REVERB->spt3];
        s  = REVERB->buf3_R[REVERB->spt3] = REVERB->buf0_R[REVERB->spt0];
        REVERB->buf0_R[REVERB->spt0] = REVERB->LPFR;

        t = (REVERB->HPFR + fixp) * REV_HPF_LEV;
        REVERB->HPFR = t - fixp;

        REVERB->buf2_R[REVERB->spt2] = (s - fixp * REV_FBK_LEV) * REV_NMIX_LEV;
        REVERB->tb = REVERB->buf1_R[REVERB->spt1];
        REVERB->buf1_R[REVERB->spt1] = t;

        REVERB->EPFR = REVERB->EPFR * REV_EPF_LEV + REVERB->ta * REV_EPF_INP;
        comp[i] = REVERB->ta + REVERB->EPFR + fixp;

        rev_ptinc();
    }
}

/* dummy */
void reverb_rc_event(tmdy_struct_ex_t *tmdy_struct, int rc, int32 val)
{
    switch(rc)
    {
      case RC_CHANGE_REV_EFFB:
        break;
      case RC_CHANGE_REV_TIME:
        break;
    }
}


/*                             */
/*   Delay (Celeste) Effect    */
/*                             */

void do_basic_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count);
void do_cross_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count);
void do_3tap_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count);

void init_ch_delay(tmdy_struct_ex_t *tmdy_struct)
{
#ifdef USE_DSP_EFFECT
	memset(REVERB->delay_buf0_L,0,sizeof(REVERB->delay_buf0_L));
	memset(REVERB->delay_buf0_R,0,sizeof(REVERB->delay_buf0_R));
	memset(REVERB->delay_effect_buffer,0,sizeof(REVERB->delay_effect_buffer));
	memset(REVERB->delay_status.high_val,0,sizeof(REVERB->delay_status.high_val));

	REVERB->delay_wpt0 = 0;
	REVERB->delay_spt0 = 0;
	REVERB->delay_spt1 = 0;
	REVERB->delay_spt2 = 0;
#endif /* USE_DSP_EFFECT */
}

#ifdef USE_DSP_EFFECT
void do_ch_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	if((TMDY_PLAYMIDI->opt_reverb_control == 3 || REVERB->opt_effect_quality >= 1) && REVERB->delay_status.pre_lpf) {
		do_shelving_filter(tmdy_struct, REVERB->delay_effect_buffer, count, REVERB->delay_status.high_coef, REVERB->delay_status.high_val);
	}

	switch(REVERB->delay_status.type) {
	case 1: do_3tap_delay(tmdy_struct, buf,count);
			break;
	case 2: do_cross_delay(tmdy_struct, buf,count);
			break;
	default: do_basic_delay(tmdy_struct, buf,count);
			break;
	}
}

#if OPT_MODE != 0
#if _MSC_VER
void set_ch_delay(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count, int32 level)
{
	int32 *dbuf = REVERB->delay_effect_buffer;
	level = level * 65536 / 127;

	_asm {
		mov		ecx, [count]
		mov		esi, [buf]
		mov		ebx, [level]
		test	ecx, ecx
		jz		short L2
		mov		edi, [dbuf]
L1:		mov		eax, [esi]
		imul	ebx
		shr		eax, 16
		shl		edx, 16
		or		eax, edx	/* u */
		mov		edx, [edi]	/* v */
		add		esi, 4		/* u */	
		add		edx, eax	/* v */
		mov		[edi], edx	/* u */
		add		edi, 4		/* v */
		dec		ecx			/* u */
		jnz		L1			/* v */
L2:
	}
}
#else
void set_ch_delay(tmdy_struct_ex_t *tmdy_struct, register int32 *sbuffer, int32 n, int32 level)
{
    register int32 i;
	int32 *buf = REVERB->delay_effect_buffer;
	level = level * 65536 / 127;

	for(i=n-1;i>=0;i--) {buf[i] += imuldiv16(sbuffer[i], level);}
}
#endif	/* _MSC_VER */
#else
void set_ch_delay(tmdy_struct_ex_t *tmdy_struct, register int32 *sbuffer, int32 n, int32 level)
{
    register int32 i;
	register int32 count = n;
    FLOAT_T send_level = (FLOAT_T)level / 127.0;

    for(i=0;i<count;i++)
    {
        REVERB->delay_effect_buffer[i] += sbuffer[i] * send_level;
    }
}
#endif /* OPT_MODE != 0 */

#if OPT_MODE != 0
void do_basic_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	register int32 n = count;
	int32 level,feedback,output,send_reverb;

	level = TIM_FSCALE(REVERB->delay_status.level_ratio_c * MASTER_DELAY_LEVEL, 24);
	feedback = TIM_FSCALE(REVERB->delay_status.feedback_ratio, 24);
	send_reverb = TIM_FSCALE(REVERB->delay_status.send_reverb_ratio * REVERB->REV_INP_LEV, 24);

	REVERB->delay_spt0 = REVERB->delay_wpt0 - REVERB->delay_status.sample_c;
	if(REVERB->delay_spt0 < 0) {REVERB->delay_spt0 += REVERB->delay_rpt0;}

	for(i=0;i<n;i++) {
		REVERB->delay_buf0_L[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[i] + imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt0], feedback);
		output = imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt0], level);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output, send_reverb);
		REVERB->delay_effect_buffer[i] = 0;

		REVERB->delay_buf0_R[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[++i] + imuldiv24(REVERB->delay_buf0_R[REVERB->delay_spt0], feedback);
		output = imuldiv24(REVERB->delay_buf0_R[REVERB->delay_spt0], level);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output, send_reverb);
		REVERB->delay_effect_buffer[i] = 0;

		if(++REVERB->delay_spt0 == REVERB->delay_rpt0) {REVERB->delay_spt0 = 0;}
		if(++REVERB->delay_wpt0 == REVERB->delay_rpt0) {REVERB->delay_wpt0 = 0;}
	}
}
#else
void do_basic_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	register int32 n = count;
	FLOAT_T level,feedback;

	level = REVERB->delay_status.level_ratio_c * MASTER_DELAY_LEVEL;
	feedback = REVERB->delay_status.feedback_ratio;

	REVERB->delay_spt0 = REVERB->delay_wpt0 - REVERB->delay_status.sample_c;
	if(REVERB->delay_spt0 < 0) {REVERB->delay_spt0 += REVERB->delay_rpt0;}

	for(i=0;i<n;i++) {
		REVERB->delay_buf0_L[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[i] + REVERB->delay_buf0_L[REVERB->delay_spt0] * feedback;
		buf[i] += REVERB->delay_buf0_L[REVERB->delay_spt0] * level;
		REVERB->delay_effect_buffer[i] = 0;

		REVERB->delay_buf0_R[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[++i] + REVERB->delay_buf0_R[REVERB->delay_spt0] * feedback;
		buf[i] += REVERB->delay_buf0_R[REVERB->delay_spt0] * level;
		REVERB->delay_effect_buffer[i] = 0;

		if(++REVERB->delay_spt0 == REVERB->delay_rpt0) {REVERB->delay_spt0 = 0;}
		if(++REVERB->delay_wpt0 == REVERB->delay_rpt0) {REVERB->delay_wpt0 = 0;}
	}
}
#endif /* OPT_MODE != 0 */

#if OPT_MODE != 0
void do_cross_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	register int32 n = count;
	int32 feedback,level_c,level_l,level_r,send_reverb,output;

	feedback = TIM_FSCALE(REVERB->delay_status.feedback_ratio, 24);
	level_c = TIM_FSCALE(REVERB->delay_status.level_ratio_c * MASTER_DELAY_LEVEL, 24);
	level_l = TIM_FSCALE(REVERB->delay_status.level_ratio_l * MASTER_DELAY_LEVEL, 24);
	level_r = TIM_FSCALE(REVERB->delay_status.level_ratio_r * MASTER_DELAY_LEVEL, 24);
	send_reverb = TIM_FSCALE(REVERB->delay_status.send_reverb_ratio * REVERB->REV_INP_LEV, 24);

	REVERB->delay_spt0 = REVERB->delay_wpt0 - REVERB->delay_status.sample_c;
	if(REVERB->delay_spt0 < 0) {REVERB->delay_spt0 += REVERB->delay_rpt0;}
	REVERB->delay_spt1 = REVERB->delay_wpt0 - REVERB->delay_status.sample_l;
	if(REVERB->delay_spt1 < 0) {REVERB->delay_spt1 += REVERB->delay_rpt0;}
	REVERB->delay_spt2 = REVERB->delay_wpt0 - REVERB->delay_status.sample_r;
	if(REVERB->delay_spt2 < 0) {REVERB->delay_spt2 += REVERB->delay_rpt0;}

	for(i=0;i<n;i++) {
		REVERB->delay_buf0_L[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[i] + imuldiv24(REVERB->delay_buf0_R[REVERB->delay_spt0],feedback);
		output = imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt0],level_c) + imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt1] + REVERB->delay_buf0_R[REVERB->delay_spt1],level_l);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output,send_reverb);
		REVERB->delay_effect_buffer[i] = 0;

		REVERB->delay_buf0_R[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[++i] + imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt0],feedback);
		output = imuldiv24(REVERB->delay_buf0_R[REVERB->delay_spt0],level_c) + imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt2] + REVERB->delay_buf0_R[REVERB->delay_spt2],level_r);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output,send_reverb);
		REVERB->delay_effect_buffer[i] = 0;

		if(++REVERB->delay_spt0 == REVERB->delay_rpt0) {REVERB->delay_spt0 = 0;}
		if(++REVERB->delay_spt1 == REVERB->delay_rpt0) {REVERB->delay_spt1 = 0;}
		if(++REVERB->delay_spt2 == REVERB->delay_rpt0) {REVERB->delay_spt2 = 0;}
		if(++REVERB->delay_wpt0 == REVERB->delay_rpt0) {REVERB->delay_wpt0 = 0;}
	}
}
#else
void do_cross_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	register int32 n = count;
	FLOAT_T feedback,level_c,level_l,level_r;

	feedback = REVERB->delay_status.feedback_ratio;
	level_c = REVERB->delay_status.level_ratio_c * MASTER_DELAY_LEVEL;
	level_l = REVERB->delay_status.level_ratio_l * MASTER_DELAY_LEVEL;
	level_r = REVERB->delay_status.level_ratio_r * MASTER_DELAY_LEVEL;

	REVERB->delay_spt0 = REVERB->delay_wpt0 - REVERB->delay_status.sample_c;
	if(REVERB->delay_spt0 < 0) {REVERB->delay_spt0 += REVERB->delay_rpt0;}
	REVERB->delay_spt1 = REVERB->delay_wpt0 - REVERB->delay_status.sample_l;
	if(REVERB->delay_spt1 < 0) {REVERB->delay_spt1 += REVERB->delay_rpt0;}
	REVERB->delay_spt2 = REVERB->delay_wpt0 - REVERB->delay_status.sample_r;
	if(REVERB->delay_spt2 < 0) {REVERB->delay_spt2 += REVERB->delay_rpt0;}

	for(i=0;i<n;i++) {
		REVERB->delay_buf0_L[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[i] + REVERB->delay_buf0_R[REVERB->delay_spt0] * feedback;
		buf[i] += REVERB->delay_buf0_L[REVERB->delay_spt0] * level_c + (REVERB->delay_buf0_L[REVERB->delay_spt1] + REVERB->delay_buf0_R[REVERB->delay_spt1]) * level_l;
		REVERB->delay_effect_buffer[i] = 0;

		REVERB->delay_buf0_R[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[++i] + REVERB->delay_buf0_L[REVERB->delay_spt0] * feedback;
		buf[i] += REVERB->delay_buf0_R[REVERB->delay_spt0] * level_c + (REVERB->delay_buf0_L[REVERB->delay_spt2] + REVERB->delay_buf0_R[REVERB->delay_spt2]) * level_r;
		REVERB->delay_effect_buffer[i] = 0;

		if(++REVERB->delay_spt0 == REVERB->delay_rpt0) {REVERB->delay_spt0 = 0;}
		if(++REVERB->delay_spt1 == REVERB->delay_rpt0) {REVERB->delay_spt1 = 0;}
		if(++REVERB->delay_spt2 == REVERB->delay_rpt0) {REVERB->delay_spt2 = 0;}
		if(++REVERB->delay_wpt0 == REVERB->delay_rpt0) {REVERB->delay_wpt0 = 0;}
	}
}
#endif /* OPT_MODE != 0 */

#if OPT_MODE != 0
void do_3tap_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	register int32 n = count;
	int32 feedback,level_c,level_l,level_r,output,send_reverb;

	feedback = TIM_FSCALE(REVERB->delay_status.feedback_ratio, 24);
	level_c = TIM_FSCALE(REVERB->delay_status.level_ratio_c * MASTER_DELAY_LEVEL, 24);
	level_l = TIM_FSCALE(REVERB->delay_status.level_ratio_l * MASTER_DELAY_LEVEL, 24);
	level_r = TIM_FSCALE(REVERB->delay_status.level_ratio_r * MASTER_DELAY_LEVEL, 24);
	send_reverb = TIM_FSCALE(REVERB->delay_status.send_reverb_ratio * REVERB->REV_INP_LEV, 24);

	REVERB->delay_spt0 = REVERB->delay_wpt0 - REVERB->delay_status.sample_c;
	if(REVERB->delay_spt0 < 0) {REVERB->delay_spt0 += REVERB->delay_rpt0;}
	REVERB->delay_spt1 = REVERB->delay_wpt0 - REVERB->delay_status.sample_l;
	if(REVERB->delay_spt1 < 0) {REVERB->delay_spt1 += REVERB->delay_rpt0;}
	REVERB->delay_spt2 = REVERB->delay_wpt0 - REVERB->delay_status.sample_r;
	if(REVERB->delay_spt2 < 0) {REVERB->delay_spt2 += REVERB->delay_rpt0;}

	for(i=0;i<n;i++) {
		REVERB->delay_buf0_L[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[i] + imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt0],feedback);
		output = imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt0],level_c) + imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt1] + REVERB->delay_buf0_R[REVERB->delay_spt1],level_l);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output,send_reverb);
		REVERB->delay_effect_buffer[i] = 0;

		REVERB->delay_buf0_R[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[++i] + imuldiv24(REVERB->delay_buf0_R[REVERB->delay_spt0],feedback);
		output = imuldiv24(REVERB->delay_buf0_R[REVERB->delay_spt0],level_c) + imuldiv24(REVERB->delay_buf0_L[REVERB->delay_spt2] + REVERB->delay_buf0_R[REVERB->delay_spt2],level_r);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output,send_reverb);
		REVERB->delay_effect_buffer[i] = 0;

		if(++REVERB->delay_spt0 == REVERB->delay_rpt0) {REVERB->delay_spt0 = 0;}
		if(++REVERB->delay_spt1 == REVERB->delay_rpt0) {REVERB->delay_spt1 = 0;}
		if(++REVERB->delay_spt2 == REVERB->delay_rpt0) {REVERB->delay_spt2 = 0;}
		if(++REVERB->delay_wpt0 == REVERB->delay_rpt0) {REVERB->delay_wpt0 = 0;}
	}
}
#else
void do_3tap_delay(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	register int32 n = count;
	FLOAT_T feedback,level_c,level_l,level_r;

	feedback = REVERB->delay_status.feedback_ratio;
	level_c = REVERB->delay_status.level_ratio_c * MASTER_DELAY_LEVEL;
	level_l = REVERB->delay_status.level_ratio_l * MASTER_DELAY_LEVEL;
	level_r = REVERB->delay_status.level_ratio_r * MASTER_DELAY_LEVEL;

	REVERB->delay_spt0 = REVERB->delay_wpt0 - REVERB->delay_status.sample_c;
	if(REVERB->delay_spt0 < 0) {REVERB->delay_spt0 += REVERB->delay_rpt0;}
	REVERB->delay_spt1 = REVERB->delay_wpt0 - REVERB->delay_status.sample_l;
	if(REVERB->delay_spt1 < 0) {REVERB->delay_spt1 += REVERB->delay_rpt0;}
	REVERB->delay_spt2 = REVERB->delay_wpt0 - REVERB->delay_status.sample_r;
	if(REVERB->delay_spt2 < 0) {REVERB->delay_spt2 += REVERB->delay_rpt0;}

	for(i=0;i<n;i++) {
		REVERB->delay_buf0_L[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[i] + REVERB->delay_buf0_L[REVERB->delay_spt0] * feedback;
		buf[i] += REVERB->delay_buf0_L[REVERB->delay_spt0] * level_c + (REVERB->delay_buf0_L[REVERB->delay_spt1] + REVERB->delay_buf0_R[REVERB->delay_spt1]) * level_l;
		REVERB->delay_effect_buffer[i] = 0;

		REVERB->delay_buf0_R[REVERB->delay_wpt0] = REVERB->delay_effect_buffer[++i] + REVERB->delay_buf0_R[REVERB->delay_spt0] * feedback;
		buf[i] += REVERB->delay_buf0_R[REVERB->delay_spt0] * level_c + (REVERB->delay_buf0_L[REVERB->delay_spt2] + REVERB->delay_buf0_R[REVERB->delay_spt2]) * level_r;
		REVERB->delay_effect_buffer[i] = 0;

		if(++REVERB->delay_spt0 == REVERB->delay_rpt0) {REVERB->delay_spt0 = 0;}
		if(++REVERB->delay_spt1 == REVERB->delay_rpt0) {REVERB->delay_spt1 = 0;}
		if(++REVERB->delay_spt2 == REVERB->delay_rpt0) {REVERB->delay_spt2 = 0;}
		if(++REVERB->delay_wpt0 == REVERB->delay_rpt0) {REVERB->delay_wpt0 = 0;}
	}
}
#endif /* OPT_MODE != 0 */
#endif /* USE_DSP_EFFECT */


/*                             */
/*        Chorus Effect        */
/*                             */

void init_chorus_lfo(tmdy_struct_ex_t *tmdy_struct)
{
#ifdef USE_DSP_EFFECT
	int32 i;

	REVERB->chorus_cyc0 = REVERB->chorus_param.cycle_in_sample;
	if(REVERB->chorus_cyc0 == 0) {REVERB->chorus_cyc0 = 1;}

	for(i=0;i<SINE_CYCLE_LENGTH;i++) {
		REVERB->chorus_lfo0[i] = TIM_FSCALE((lookup_triangular(i) + 1.0) / 2, 24);
	}

	for(i=0;i<SINE_CYCLE_LENGTH;i++) {
		REVERB->chorus_lfo1[i] = TIM_FSCALE((lookup_triangular(i + SINE_CYCLE_LENGTH / 4) + 1.0) / 2, 24);
	}
#endif /* USE_DSP_EFFECT */
}

void init_ch_chorus(tmdy_struct_ex_t *tmdy_struct)
{
#ifdef USE_DSP_EFFECT
	memset(REVERB->chorus_buf0_L,0,sizeof(REVERB->chorus_buf0_L));
	memset(REVERB->chorus_buf0_R,0,sizeof(REVERB->chorus_buf0_R));
	memset(REVERB->chorus_effect_buffer,0,sizeof(REVERB->chorus_effect_buffer));
	memset(REVERB->chorus_param.high_val,0,sizeof(REVERB->chorus_param.high_val));

	REVERB->chorus_cnt0 = 0;
	REVERB->chorus_wpt0 = 0;
	REVERB->chorus_wpt1 = 0;
	REVERB->chorus_spt0 = 0;
	REVERB->chorus_spt1 = 0;
#endif /* USE_DSP_EFFECT */
}

#ifdef USE_DSP_EFFECT
void do_stereo_chorus(tmdy_struct_ex_t *tmdy_struct, int32* buf, register int32 count)
{
#if OPT_MODE != 0	/* fixed-point implementation */
	register int32 i;
	int32 level, feedback, send_reverb, send_delay, delay, depth, output, div, v1l, v1r, f0, f1;

	level = TIM_FSCALE(REVERB->chorus_param.level_ratio * MASTER_CHORUS_LEVEL, 24);
	feedback = TIM_FSCALE(REVERB->chorus_param.feedback_ratio, 24);
	send_reverb = TIM_FSCALE(REVERB->chorus_param.send_reverb_ratio * REVERB->REV_INP_LEV, 24);
	send_delay = TIM_FSCALE(REVERB->chorus_param.send_delay_ratio, 24);
	depth = REVERB->chorus_param.depth_in_sample;
	delay = REVERB->chorus_param.delay_in_sample;
	div = TIM_FSCALE((SINE_CYCLE_LENGTH - 1) / (double)REVERB->chorus_cyc0, 24) - 0.5;

	f0 = imuldiv16(REVERB->chorus_lfo0[imuldiv24(REVERB->chorus_cnt0, div)], depth);
	REVERB->chorus_spt0 = REVERB->chorus_wpt0 - delay - (f0 >> 8);
	f0 &= 0xFF;
	if(REVERB->chorus_spt0 < 0) {REVERB->chorus_spt0 += REVERB->chorus_rpt0;}
	f1 = imuldiv16(REVERB->chorus_lfo1[imuldiv24(REVERB->chorus_cnt0, div)], depth);
	REVERB->chorus_spt1 = REVERB->chorus_wpt0 - delay - (f1 >> 8);
	f1 &= 0xFF;
	if(REVERB->chorus_spt1 < 0) {REVERB->chorus_spt1 += REVERB->chorus_rpt0;}
	
	for(i=0;i<count;i++) {
		v1l = REVERB->chorus_buf0_L[REVERB->chorus_spt0];
		v1r = REVERB->chorus_buf0_R[REVERB->chorus_spt1];

		REVERB->chorus_wpt1 = REVERB->chorus_wpt0;
		if(++REVERB->chorus_wpt0 == REVERB->chorus_rpt0) {REVERB->chorus_wpt0 = 0;}
		f0 = imuldiv16(REVERB->chorus_lfo0[imuldiv24(REVERB->chorus_cnt0, div)], depth);
		REVERB->chorus_spt0 = REVERB->chorus_wpt0 - delay - (f0 >> 8);
		f0 &= 0xFF;
		if(REVERB->chorus_spt0 < 0) {REVERB->chorus_spt0 += REVERB->chorus_rpt0;}
		f1 = imuldiv16(REVERB->chorus_lfo1[imuldiv24(REVERB->chorus_cnt0, div)], depth);
		REVERB->chorus_spt1 = REVERB->chorus_wpt0 - delay - (f1 >> 8);
		f1 &= 0xFF;
		if(REVERB->chorus_spt1 < 0) {REVERB->chorus_spt1 += REVERB->chorus_rpt0;}
		if(++REVERB->chorus_cnt0 == REVERB->chorus_cyc0) {REVERB->chorus_cnt0 = 0;}

		output = v1l + imuldiv8(REVERB->chorus_buf0_L[REVERB->chorus_spt0] - v1l, f0);
		REVERB->chorus_buf0_L[REVERB->chorus_wpt1] = REVERB->chorus_effect_buffer[i] + imuldiv24(output, feedback);
		output = imuldiv24(output, level);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output, send_reverb);
		REVERB->delay_effect_buffer[i] += imuldiv24(output, send_delay);
		REVERB->chorus_effect_buffer[i] = 0;

		output = v1r + imuldiv8(REVERB->chorus_buf0_R[REVERB->chorus_spt1] - v1r, f1);
		REVERB->chorus_buf0_R[REVERB->chorus_wpt1] = REVERB->chorus_effect_buffer[++i] + imuldiv24(output, feedback);
		output = imuldiv24(output, level);
		buf[i] += output;
		REVERB->effect_buffer[i] += imuldiv24(output, send_reverb);
		REVERB->delay_effect_buffer[i] += imuldiv24(output, send_delay);
		REVERB->chorus_effect_buffer[i] = 0;
	}
#endif /* OPT_MODE != 0 */
}

#if OPT_MODE != 0	/* fixed-point implementation */
#if _MSC_VER
void set_ch_chorus(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count, int32 level)
{
	int32 *dbuf = REVERB->chorus_effect_buffer;
	level = level * 65536 / 127;

	_asm {
		mov		ecx, [count]
		mov		esi, [buf]
		mov		ebx, [level]
		test	ecx, ecx
		jz		short L2
		mov		edi, [dbuf]
L1:		mov		eax, [esi]
		imul	ebx
		shr		eax, 16
		shl		edx, 16
		or		eax, edx	/* u */
		mov		edx, [edi]	/* v */
		add		esi, 4		/* u */	
		add		edx, eax	/* v */
		mov		[edi], edx	/* u */
		add		edi, 4		/* v */
		dec		ecx			/* u */
		jnz		L1			/* v */
L2:
	}
}
#else
void set_ch_chorus(tmdy_struct_ex_t *tmdy_struct, register int32 *sbuffer,int32 n, int32 level)
{
    register int32 i;
	int32 *buf = REVERB->chorus_effect_buffer;
	level = level * 65536 / 127;

	for(i=n-1;i>=0;i--) {buf[i] += imuldiv16(sbuffer[i], level);}
}
#endif	/* _MSC_VER */
#else	/* floating-point implementation */
void set_ch_chorus(tmdy_struct_ex_t *tmdy_struct, register int32 *sbuffer,int32 n, int32 level)
{
    register int32 i;
    register int32 count = n;
    FLOAT_T send_level = (FLOAT_T)level / 127.0;

    for(i=0;i<count;i++)
    {
		REVERB->chorus_effect_buffer[i] += sbuffer[i] * send_level;
    }
}
#endif /* OPT_MODE != 0 */

void do_ch_chorus(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	if((TMDY_PLAYMIDI->opt_reverb_control == 3 || REVERB->opt_effect_quality >= 1) && REVERB->chorus_param.chorus_pre_lpf) {
		do_shelving_filter(tmdy_struct, REVERB->chorus_effect_buffer, count, REVERB->chorus_param.high_coef, REVERB->chorus_param.high_val);
	}
	do_stereo_chorus(tmdy_struct, buf, count);
}
#endif /* USE_DSP_EFFECT */


/*                             */
/*       EQ (Equalizer)        */
/*                             */

void init_eq(tmdy_struct_ex_t *tmdy_struct)
{
	memset(REVERB->eq_buffer, 0, sizeof(REVERB->eq_buffer));
	memset(REVERB->eq_status.low_val, 0, sizeof(REVERB->eq_status.low_val));
	memset(REVERB->eq_status.high_val, 0, sizeof(REVERB->eq_status.high_val));
}

void calc_lowshelf_coefs(tmdy_struct_ex_t *tmdy_struct, int32* coef, int32 cutoff_freq, FLOAT_T dbGain, int32 rate)
{
	FLOAT_T a0, a1, a2, b0, b1, b2, omega, sn, cs, A, beta;

	A = pow(10, dbGain / 40);
	omega = 2.0 * M_PI * (FLOAT_T)cutoff_freq / (FLOAT_T)rate;
	sn = sin(omega);
	cs = cos(omega);
	beta = sqrt(A + A);

	a0 = 1.0 / ((A + 1) + (A - 1) * cs + beta * sn);
	a1 = 2.0 * ((A - 1) + (A + 1) * cs);
	a2 = -((A + 1) + (A - 1) * cs - beta * sn);
	b0 = A * ((A + 1) - (A - 1) * cs + beta * sn);
	b1 = 2.0 * A * ((A - 1) - (A + 1) * cs);
	b2 = A * ((A + 1) - (A - 1) * cs - beta * sn);

	a1 *= a0;
	a2 *= a0;
	b1 *= a0;
	b2 *= a0;
	b0 *= a0;

	coef[0] = TIM_FSCALE(a1, 24);
	coef[1] = TIM_FSCALE(a2, 24);
	coef[2] = TIM_FSCALE(b0, 24);
	coef[3] = TIM_FSCALE(b1, 24);
	coef[4] = TIM_FSCALE(b2, 24);
}

void calc_highshelf_coefs(tmdy_struct_ex_t *tmdy_struct, int32* coef, int32 cutoff_freq, FLOAT_T dbGain, int32 rate)
{
	FLOAT_T a0, a1, a2, b0, b1, b2, omega, sn, cs, A, beta;

	A = pow(10, dbGain / 40);
	omega = 2.0 * M_PI * (FLOAT_T)cutoff_freq / (FLOAT_T)rate;
	sn = sin(omega);
	cs = cos(omega);
	beta = sqrt(A + A);

	a0 = 1.0 / ((A + 1) - (A - 1) * cs + beta * sn);
	a1 = (-2 * ((A - 1) - (A + 1) * cs));
	a2 = -((A + 1) - (A - 1) * cs - beta * sn);
	b0 = A * ((A + 1) + (A - 1) * cs + beta * sn);
	b1 = -2 * A * ((A - 1) + (A + 1) * cs);
	b2 = A * ((A + 1) + (A - 1) * cs - beta * sn);

	a1 *= a0;
	a2 *= a0;
	b0 *= a0;
	b1 *= a0;
	b2 *= a0;

	coef[0] = TIM_FSCALE(a1, 24);
	coef[1] = TIM_FSCALE(a2, 24);
	coef[2] = TIM_FSCALE(b0, 24);
	coef[3] = TIM_FSCALE(b1, 24);
	coef[4] = TIM_FSCALE(b2, 24);
}


static void do_shelving_filter(tmdy_struct_ex_t *tmdy_struct, register int32* buf, int32 count, int32* eq_coef, int32* eq_val)
{
#if OPT_MODE != 0
	register int32 i;
	int32 x1l, x2l, y1l, y2l, x1r, x2r, y1r, y2r, yout;
	int32 a1, a2, b0, b1, b2;

	a1 = eq_coef[0];
	a2 = eq_coef[1];
	b0 = eq_coef[2];
	b1 = eq_coef[3];
	b2 = eq_coef[4];

	x1l = eq_val[0];
	x2l = eq_val[1];
	y1l = eq_val[2];
	y2l = eq_val[3];
	x1r = eq_val[4];
	x2r = eq_val[5];
	y1r = eq_val[6];
	y2r = eq_val[7];

	for(i=0;i<count;i++) {
		yout = imuldiv24(buf[i], b0) + imuldiv24(x1l, b1) + imuldiv24(x2l, b2) + imuldiv24(y1l, a1) + imuldiv24(y2l, a2);
		x2l = x1l;
		x1l = buf[i];
		y2l = y1l;
		y1l = yout;
		buf[i] = yout;

		yout = imuldiv24(buf[++i], b0) + imuldiv24(x1r, b1) + imuldiv24(x2r, b2) + imuldiv24(y1r, a1) + imuldiv24(y2r, a2);
		x2r = x1r;
		x1r = buf[i];
		y2r = y1r;
		y1r = yout;
		buf[i] = yout;
	}

	eq_val[0] = x1l;
	eq_val[1] = x2l;
	eq_val[2] = y1l;
	eq_val[3] = y2l;
	eq_val[4] = x1r;
	eq_val[5] = x2r;
	eq_val[6] = y1r;
	eq_val[7] = y2r;
#endif /* OPT_MODE != 0 */
}

void do_ch_eq(tmdy_struct_ex_t *tmdy_struct, int32* buf,int32 n)
{
	register int32 i;
	register int32 count = n;

	do_shelving_filter(tmdy_struct, REVERB->eq_buffer,count,REVERB->eq_status.low_coef,REVERB->eq_status.low_val);
	do_shelving_filter(tmdy_struct, REVERB->eq_buffer,count,REVERB->eq_status.high_coef,REVERB->eq_status.high_val);

	for(i=0;i<count;i++) {
		buf[i] += REVERB->eq_buffer[i];
		REVERB->eq_buffer[i] = 0;
	}
}

#if OPT_MODE != 0
#if _MSC_VER
void set_ch_eq(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count)
{
	int32 *dbuf = REVERB->eq_buffer;
	_asm {
		mov		ecx, [count]
		mov		esi, [buf]
		test	ecx, ecx
		jz		short L2
		mov		edi, [dbuf]
L1:		mov		eax, [esi]
		mov		ebx, [edi]
		add		esi, 4
		add		ebx, eax
		mov		[edi], ebx
		add		edi, 4
		dec		ecx
		jnz		L1
L2:
	}
}
#else
void set_ch_eq(tmdy_struct_ex_t *tmdy_struct, register int32 *buf, int32 n)
{
    register int32 i;

    for(i=n-1;i>=0;i--)
    {
        REVERB->eq_buffer[i] += buf[i];
    }
}
#endif	/* _MSC_VER */
#else
void set_ch_eq(tmdy_struct_ex_t *tmdy_struct, register int32 *sbuffer, int32 n)
{
    register int32  i;
    
	for(i = 0; i < n; i++)
    {
        REVERB->eq_buffer[i] += sbuffer[i];
    }
}
#endif /* OPT_MODE != 0 */


/*                             */
/*   LPF for System Effects    */
/*                             */

void calc_lowpass_coefs_24db(tmdy_struct_ex_t *tmdy_struct, int32* lpf_coef,int32 cutoff_freq,int16 resonance,int32 rate)
{
	FLOAT_T c,a1,a2,a3,b1,b2,q;
	const FLOAT_T sqrt2 = 1.4142134;

	/*memset(lpf_coef, 0, sizeof(lpf_coef));*/

	q = sqrt2 - (sqrt2 - 0.1) * (FLOAT_T)resonance / 127;

	c = (cutoff_freq == 0) ? 0 : (1.0 / tan(M_PI * (FLOAT_T)cutoff_freq / (FLOAT_T)rate));

	a1 = 1.0 / (1.0 + q * c + c * c); 
	a2 = 2* a1; 
	a3 = a1; 
	b1 = -(2.0 * (1.0 - c * c) * a1); 
	b2 = -(1.0 - q * c + c * c) * a1; 

	lpf_coef[0] = TIM_FSCALE(a1, 24);
	lpf_coef[1] = TIM_FSCALE(a2, 24);
	lpf_coef[2] = TIM_FSCALE(a3, 24);
	lpf_coef[3] = TIM_FSCALE(b1, 24);
	lpf_coef[4] = TIM_FSCALE(b2, 24);
}

void do_lowpass_24db(tmdy_struct_ex_t *tmdy_struct, register int32* buf,int32 count,int32* lpf_coef,int32* lpf_val)
{
#if OPT_MODE != 0
	register int32 i,length;
	int32 x1l,x2l,y1l,y2l,x1r,x2r,y1r,y2r,yout;
	int32 a1,a2,a3,b1,b2;

	a1 = lpf_coef[0];
	a2 = lpf_coef[1];
	a3 = lpf_coef[2];
	b1 = lpf_coef[3];
	b2 = lpf_coef[4];

	length = count;

	x1l = lpf_val[0];
	x2l = lpf_val[1];
	y1l = lpf_val[2];
	y2l = lpf_val[3];
	x1r = lpf_val[4];
	x2r = lpf_val[5];
	y1r = lpf_val[6];
	y2r = lpf_val[7];

	for(i=0;i<length;i++) {
		yout = imuldiv24(buf[i] + x2l,a1) + imuldiv24(x1l,a2) + imuldiv24(y1l,b1) + imuldiv24(y2l,b2);
		x2l = x1l;
		x1l = buf[i];
		buf[i] = yout;
		y2l = y1l;
		y1l = yout;

		yout = imuldiv24(buf[++i] + x2r,a1) + imuldiv24(x1r,a2) + imuldiv24(y1r,b1) + imuldiv24(y2r,b2);
		x2r = x1r;
		x1r = buf[i];
		buf[i] = yout;
		y2r = y1r;
		y1r = yout;
	}

	lpf_val[0] = x1l;
	lpf_val[1] = x2l;
	lpf_val[2] = y1l;
	lpf_val[3] = y2l;
	lpf_val[4] = x1r;
	lpf_val[5] = x2r;
	lpf_val[6] = y1r;
	lpf_val[7] = y2r;
#endif /* OPT_MODE != 0 */
}


/*                             */
/* Insertion Effect (SC-88Pro) */
/*                             */


/* general-purpose volume stat. */
inline void do_volume_stat(tmdy_struct_ex_t *tmdy_struct, int32 sample,int32 *volume)
{
	*volume -= 200;
	if(sample < 0) {sample = -sample;}
	if(sample > *volume) {*volume = sample;}
}

/* general-purpose Panning */
/* pan = -1.0 ~ 1.0          */
static inline int32 do_left_panning(tmdy_struct_ex_t *tmdy_struct, int32 sample, FLOAT_T pan)
{
	return (int32)(sample - sample * pan);
}

static inline int32 do_right_panning(tmdy_struct_ex_t *tmdy_struct, int32 sample, FLOAT_T pan)
{
	return (int32)(sample + sample * pan);
}

/* general-purpose Distortion */
/* level = 0.0 ~ 1.0         */
/* volume = 0.0 ~ 1.0        */ 
inline int32 do_distortion(tmdy_struct_ex_t *tmdy_struct, int32 sample,FLOAT_T level,FLOAT_T volume,int32 max_volume)
{
	int32 od_clip = max_volume >> 2;
	sample *= level;
	if(sample > od_clip) {sample = od_clip;}
	else if(sample < -od_clip) {sample = -od_clip;}
	sample *= volume;
	sample <<= 2;
	return sample;
}

/* general-purpose Overdrive */
/* level = 0.0 ~ 1.0        */
/* volume = 0.0 ~ 1.0       */
inline int32 do_overdrive(tmdy_struct_ex_t *tmdy_struct, int32 sample,FLOAT_T level,FLOAT_T volume,int32 max_volume)
{
	int32 od_threshold = max_volume >> 1;

	if(sample < -od_threshold) {
		sample = (int32)(-pow((FLOAT_T)-sample / (FLOAT_T)max_volume, level) * volume * max_volume);
	} else if(sample > od_threshold) {
		sample = (int32)(pow((FLOAT_T)sample / (FLOAT_T)max_volume, level) * volume * max_volume);
	} else {
		sample *= volume;
	}

	return sample;
}

/* 0x0110: Overdrive */
void do_0110_overdrive(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	int32 n = count,output;
	FLOAT_T pan;
	FLOAT_T level,volume;

	volume = (FLOAT_T)gs_ieffect.parameter[19] / 127.0;
	level = (FLOAT_T)gs_ieffect.parameter[0] / 127.0;
	pan = (gs_ieffect.parameter[18] - 0x40) / 63.0;

	for(i=0;i<n;i++) {
		/* Left */
		output = buf[i];
		do_volume_stat(tmdy_struct, output,&REVERB->od_max_volume1);
		output = do_overdrive(tmdy_struct, output,level,volume,REVERB->od_max_volume1);
		buf[i] = do_left_panning(tmdy_struct, output,pan);

		/* Right */
		output = buf[++i];
		do_volume_stat(tmdy_struct, output,&REVERB->od_max_volume2);
		output = do_overdrive(tmdy_struct, output,level,volume,REVERB->od_max_volume2);
		buf[i] = do_right_panning(tmdy_struct, output,pan);
	}
}

/* 0x0111: Distortion */
void do_0111_distortion(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	int32 n = count,output;
	FLOAT_T pan;
	FLOAT_T level,volume;

	volume = (FLOAT_T)gs_ieffect.parameter[19] / 127.0;
	level = (FLOAT_T)gs_ieffect.parameter[0] / 127.0;
	pan = (gs_ieffect.parameter[18] - 0x40) / 63.0;

	for(i=0;i<n;i++) {
		/* Left */
		output = buf[i];
		do_volume_stat(tmdy_struct, output,&REVERB->od_max_volume1);
		output = do_distortion(tmdy_struct, output,level,volume,REVERB->od_max_volume1);
		buf[i] = do_left_panning(tmdy_struct, output,pan);

		/* Right */
		output = buf[++i];
		do_volume_stat(tmdy_struct, output,&REVERB->od_max_volume2);
		output = do_distortion(tmdy_struct, output,level,volume,REVERB->od_max_volume2);
		buf[i] = do_right_panning(tmdy_struct, output,pan);
	}
}

/* 0x1103: OD1 / OD2 */
void do_1103_dual_od(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count)
{
	register int32 i;
	int32 n = count,output1,output2,type;
	FLOAT_T pan1,pan2;
	FLOAT_T level1,level2,volume,volume1,volume2;
	int32 (*od1)(tmdy_struct_ex_t *tmdy_struct, int32,FLOAT_T,FLOAT_T,int32);
	int32 (*od2)(tmdy_struct_ex_t *tmdy_struct, int32,FLOAT_T,FLOAT_T,int32);

	volume = (FLOAT_T)gs_ieffect.parameter[19] / 127.0;
	volume1 = (FLOAT_T)gs_ieffect.parameter[16] / 127.0 * volume;
	volume2 = (FLOAT_T)gs_ieffect.parameter[18] / 127.0 * volume;
	level1 = (FLOAT_T)gs_ieffect.parameter[1] / 127.0;
	level2 = (FLOAT_T)gs_ieffect.parameter[6] / 127.0;
	pan1 = (gs_ieffect.parameter[15] - 0x40) / 63.0;
	pan2 = (gs_ieffect.parameter[17] - 0x40) / 63.0;

	type = gs_ieffect.parameter[0];
	if(type == 0) {od1 = do_overdrive;}
	else {od1 = do_distortion;}

	type = gs_ieffect.parameter[5];
	if(type == 0) {od2 = do_overdrive;}
	else {od2 = do_distortion;}

	for(i=0;i<n;i++) {
		/* Left */
		output1 = buf[i];
		do_volume_stat(tmdy_struct, output1,&REVERB->od_max_volume1);
		output1 = (*od1)(tmdy_struct, output1,level1,volume1,REVERB->od_max_volume1);

		/* Right */
		output2 = buf[++i];
		do_volume_stat(tmdy_struct, output2,&REVERB->od_max_volume2);
		output2 = (*od2)(tmdy_struct, output2,level2,volume2,REVERB->od_max_volume2);

		/* Mix */
		buf[i-1] = do_left_panning(tmdy_struct, output1,pan1) + do_left_panning(tmdy_struct, output2,pan2);
		buf[i] = do_right_panning(tmdy_struct, output1,pan1) + do_right_panning(tmdy_struct, output2,pan2);
	}
}

void init_insertion_effect(tmdy_struct_ex_t *tmdy_struct)
{
	REVERB->od_max_volume1 = 0;
	REVERB->od_max_volume2 = 0;
}

/* !!! rename this function to do_insertion_effect_gs() !!! */
void do_insertion_effect(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count)
{
	do_effect_list(tmdy_struct, buf, count, gs_ieffect.ef);
}


/*                             */
/* High Quality Reverb Effect  */
/*                             */



static void setbuf_allpass(tmdy_struct_ex_t *tmdy_struct, allpass *allpass, int32 size)
{
	if(allpass->buf != NULL) {free(allpass->buf);}
	allpass->buf = (int32 *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(int32) * size);
	if(allpass->buf == NULL) {return;}
	allpass->index = 0;
	allpass->size = size;
}

static void realloc_allpass(tmdy_struct_ex_t *tmdy_struct, allpass *allpass)
{
	if(allpass->buf != NULL) {free(allpass->buf);}
	allpass->buf = (int32 *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(int32) * allpass->size);
	if(allpass->buf == NULL) {return;}
	allpass->index = 0;
}

static void init_allpass(tmdy_struct_ex_t *tmdy_struct, allpass *allpass)
{
	memset(allpass->buf, 0, sizeof(int32) * allpass->size);
}



static void setbuf_comb(tmdy_struct_ex_t *tmdy_struct, comb *comb, int32 size)
{
	if(comb->buf != NULL) {free(comb->buf);}
	comb->buf = (int32 *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(int32) * size);
	if(comb->buf == NULL) {return;}
	comb->index = 0;
	comb->size = size;
	comb->filterstore = 0;
}

static void realloc_comb(tmdy_struct_ex_t *tmdy_struct, comb *comb)
{
	if(comb->buf != NULL) {free(comb->buf);}
	comb->buf = (int32 *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(int32) * comb->size);
	if(comb->buf == NULL) {return;}
	comb->index = 0;
	comb->filterstore = 0;
}

static void init_comb(tmdy_struct_ex_t *tmdy_struct, comb *comb)
{
	memset(comb->buf, 0, sizeof(int32) * comb->size);
}

static int init_combtunings[numcombs] = {1116, 1188, 1277, 1356, 1422, 1491, 1557, 1617};
static int init_allpasstunings[numallpasses] = {225, 341, 441, 556};

static inline int isprime(tmdy_struct_ex_t *tmdy_struct, int val)
{
  int i;

  if (val == 2) return 1;
  if (val & 1)
	{
	  for (i=3; i<(int)sqrt((double)val)+1; i+=2)
		{
		  if ((val%i) == 0) return 0;
		}
	  return 1; /* prime */
	}
  else return 0; /* even */
}

static double gs_revchar_to_roomsize(tmdy_struct_ex_t *tmdy_struct, int character)
{
	double rs;
	switch(character) {
	case 0: rs = 1.0;	break;	/* Room 1 */
	case 1: rs = 0.94;	break;	/* Room 2 */
	case 2: rs = 0.97;	break;	/* Room 3 */
	case 3: rs = 0.90;	break;	/* Hall 1 */
	case 4: rs = 0.85;	break;	/* Hall 2 */
	default: rs = 1.0;	break;	/* Plate, Delay, Panning Delay */
	}
	return rs;
}

static double gs_revchar_to_level(tmdy_struct_ex_t *tmdy_struct, int character)
{
	double level;
	switch(character) {
	case 0: level = 0.744025605;	break;	/* Room 1 */
	case 1: level = 1.224309745;	break;	/* Room 2 */
	case 2: level = 0.858592403;	break;	/* Room 3 */
	case 3: level = 1.0471802;	break;	/* Hall 1 */
	case 4: level = 1.0;	break;	/* Hall 2 */
	case 5: level = 0.865335496;	break;	/* Plate */
	default: level = 1.0;	break;	/* Delay, Panning Delay */
	}
	return level;
}

static double gs_revchar_to_rt(tmdy_struct_ex_t *tmdy_struct, int character)
{
	double rt;
	switch(character) {
	case 0: rt = 0.516850262;	break;	/* Room 1 */
	case 1: rt = 1.004226004;	break;	/* Room 2 */
	case 2: rt = 0.691046825;	break;	/* Room 3 */
	case 3: rt = 0.893006004;	break;	/* Hall 1 */
	case 4: rt = 1.0;	break;	/* Hall 2 */
	case 5: rt = 0.538476488;	break;	/* Plate */
	default: rt = 1.0;	break;	/* Delay, Panning Delay */
	}
	return rt;
}

static double gs_revchar_to_width(tmdy_struct_ex_t *tmdy_struct, int character)
{
	double width;
	switch(character) {
	case 0: width = 0.5;	break;	/* Room 1 */
	case 1: width = 0.5;	break;	/* Room 2 */
	case 2: width = 0.5;	break;	/* Room 3 */
	case 3: width = 0.5;	break;	/* Hall 1 */
	case 4: width = 0.5;	break;	/* Hall 2 */
	default: width = 0.5;	break;	/* Plate, Delay, Panning Delay */
	}
	return width;
}

static double gs_revchar_to_apfbk(tmdy_struct_ex_t *tmdy_struct, int character)
{
	double apf;
	switch(character) {
	case 0: apf = 0.7;	break;	/* Room 1 */
	case 1: apf = 0.7;	break;	/* Room 2 */
	case 2: apf = 0.7;	break;	/* Room 3 */
	case 3: apf = 0.6;	break;	/* Hall 1 */
	case 4: apf = 0.55;	break;	/* Hall 2 */
	default: apf = 0.55;	break;	/* Plate, Delay, Panning Delay */
	}
	return apf;
}

static void recalc_reverb_buffer(tmdy_struct_ex_t *tmdy_struct, revmodel_t *rev)
{
	int i;
	int32 tmpL, tmpR;
	double time;

	time = reverb_time_table[REVERB->reverb_status.time] * gs_revchar_to_rt(tmdy_struct, REVERB->reverb_status.character)
		/ (60 * REVERB->combtunings[numcombs - 1] / (-20 * log10(rev->roomsize1) * 44100.0));

	for(i = 0; i < numcombs; i++)
	{
		tmpL = REVERB->combtunings[i] * REVERB->sample_rate * time / 44100.0;
		tmpR = (REVERB->combtunings[i] + stereospread) * REVERB->sample_rate * time / 44100.0;
		if(tmpL < 10) tmpL = 10;
		if(tmpR < 10) tmpR = 10;
		while(!isprime(tmdy_struct, tmpL)) tmpL++;
		while(!isprime(tmdy_struct, tmpR)) tmpR++;
		rev->combL[i].size = tmpL;
		rev->combR[i].size = tmpR;
		realloc_comb(tmdy_struct, &rev->combL[i]);
		realloc_comb(tmdy_struct, &rev->combR[i]);
	}

	for(i = 0; i < numallpasses; i++)
	{
		tmpL = REVERB->allpasstunings[i] * REVERB->sample_rate * time / 44100.0;
		tmpR = (REVERB->allpasstunings[i] + stereospread) * REVERB->sample_rate * time / 44100.0;
		tmpL *= REVERB->sample_rate / 44100.0;
		tmpR *= REVERB->sample_rate / 44100.0;
		if(tmpL < 10) tmpL = 10;
		if(tmpR < 10) tmpR = 10;
		while(!isprime(tmdy_struct, tmpL)) tmpL++;
		while(!isprime(tmdy_struct, tmpR)) tmpR++;
		rev->allpassL[i].size = tmpL;
		rev->allpassR[i].size = tmpR;
		realloc_allpass(tmdy_struct, &rev->allpassL[i]);
		realloc_allpass(tmdy_struct, &rev->allpassR[i]);
	}
}

static void update_revmodel(tmdy_struct_ex_t *tmdy_struct, revmodel_t *rev)
{
	int i;
	double allpassfbk = gs_revchar_to_apfbk(tmdy_struct, REVERB->reverb_status.character), rtbase;

	rev->wet = (double)REVERB->reverb_status.level / 127.0 * gs_revchar_to_level(tmdy_struct, REVERB->reverb_status.character);
	rev->roomsize = gs_revchar_to_roomsize(tmdy_struct, REVERB->reverb_status.character) * scaleroom + offsetroom;
	rev->width = gs_revchar_to_width(tmdy_struct, REVERB->reverb_status.character);

	rev->wet1 = rev->width / 2 + 0.5f;
	rev->wet2 = (1 - rev->width) / 2;
	rev->roomsize1 = rev->roomsize;
	rev->damp1 = rev->damp;

	recalc_reverb_buffer(tmdy_struct, rev);
	rtbase = 1.0 / (44100.0 * reverb_time_table[REVERB->reverb_status.time] * gs_revchar_to_rt(tmdy_struct, REVERB->reverb_status.character));

	for(i = 0; i < numcombs; i++)
	{
		rev->combL[i].feedback = pow(10, -3 * (double)REVERB->combtunings[i] * rtbase);
		rev->combR[i].feedback = pow(10, -3 * (double)(REVERB->combtunings[i] /*+ stereospread*/) * rtbase);
		rev->combL[i].damp1 = rev->damp1;
		rev->combR[i].damp1 = rev->damp1;
		rev->combL[i].damp2 = 1 - rev->damp1;
		rev->combR[i].damp2 = 1 - rev->damp1;
		rev->combL[i].damp1i = TIM_FSCALE(rev->combL[i].damp1, 24);
		rev->combR[i].damp1i = TIM_FSCALE(rev->combR[i].damp1, 24);
		rev->combL[i].damp2i = TIM_FSCALE(rev->combL[i].damp2, 24);
		rev->combR[i].damp2i = TIM_FSCALE(rev->combR[i].damp2, 24);
		rev->combL[i].feedbacki = TIM_FSCALE(rev->combL[i].feedback, 24);
		rev->combR[i].feedbacki = TIM_FSCALE(rev->combR[i].feedback, 24);
	}

	for(i = 0; i < numallpasses; i++)
	{
		rev->allpassL[i].feedback = allpassfbk;
		rev->allpassR[i].feedback = allpassfbk;
		rev->allpassL[i].feedbacki = TIM_FSCALE(rev->allpassL[i].feedback, 24);
		rev->allpassR[i].feedbacki = TIM_FSCALE(rev->allpassR[i].feedback, 24);
	}

	rev->wet1i = TIM_FSCALE(rev->wet1, 24);
	rev->wet2i = TIM_FSCALE(rev->wet2, 24);
}

static void init_revmodel(tmdy_struct_ex_t *tmdy_struct, revmodel_t *rev)
{
	int i;
	for(i = 0; i < numcombs; i++) {
		init_comb(tmdy_struct, &rev->combL[i]);
		init_comb(tmdy_struct, &rev->combR[i]);
	}
	for(i = 0; i < numallpasses; i++) {
		init_allpass(tmdy_struct, &rev->allpassL[i]);
		init_allpass(tmdy_struct, &rev->allpassR[i]);
	}
}

static void alloc_revmodel(tmdy_struct_ex_t *tmdy_struct)
{
//	static int REVERB->revmodel_alloc_flag = 0;
	revmodel_t *rev;
	if(REVERB->revmodel_alloc_flag) {return;}
	if(REVERB->revmodel == NULL) {
		REVERB->revmodel = (revmodel_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(revmodel_t));
		if(REVERB->revmodel == NULL) {return;}
		memset(REVERB->revmodel, 0, sizeof(revmodel_t));
	}
	rev = REVERB->revmodel;

	setbuf_comb(tmdy_struct, &rev->combL[0], REVERB->combtunings[0]);
	setbuf_comb(tmdy_struct, &rev->combR[0], REVERB->combtunings[0] + stereospread);
	setbuf_comb(tmdy_struct, &rev->combL[1], REVERB->combtunings[1]);
	setbuf_comb(tmdy_struct, &rev->combR[1], REVERB->combtunings[1] + stereospread);
	setbuf_comb(tmdy_struct, &rev->combL[2], REVERB->combtunings[2]);
	setbuf_comb(tmdy_struct, &rev->combR[2], REVERB->combtunings[2] + stereospread);
	setbuf_comb(tmdy_struct, &rev->combL[3], REVERB->combtunings[3]);
	setbuf_comb(tmdy_struct, &rev->combR[3], REVERB->combtunings[3] + stereospread);
	setbuf_comb(tmdy_struct, &rev->combL[4], REVERB->combtunings[4]);
	setbuf_comb(tmdy_struct, &rev->combR[4], REVERB->combtunings[4] + stereospread);
	setbuf_comb(tmdy_struct, &rev->combL[5], REVERB->combtunings[5]);
	setbuf_comb(tmdy_struct, &rev->combR[5], REVERB->combtunings[5] + stereospread);
	setbuf_comb(tmdy_struct, &rev->combL[6], REVERB->combtunings[6]);
	setbuf_comb(tmdy_struct, &rev->combR[6], REVERB->combtunings[6] + stereospread);
	setbuf_comb(tmdy_struct, &rev->combL[7], REVERB->combtunings[7]);
	setbuf_comb(tmdy_struct, &rev->combR[7], REVERB->combtunings[7] + stereospread);

	setbuf_allpass(tmdy_struct, &rev->allpassL[0], REVERB->allpasstunings[0]);
	setbuf_allpass(tmdy_struct, &rev->allpassR[0], REVERB->allpasstunings[0] + stereospread);
	setbuf_allpass(tmdy_struct, &rev->allpassL[1], REVERB->allpasstunings[1]);
	setbuf_allpass(tmdy_struct, &rev->allpassR[1], REVERB->allpasstunings[1] + stereospread);
	setbuf_allpass(tmdy_struct, &rev->allpassL[2], REVERB->allpasstunings[2]);
	setbuf_allpass(tmdy_struct, &rev->allpassR[2], REVERB->allpasstunings[2] + stereospread);
	setbuf_allpass(tmdy_struct, &rev->allpassL[3], REVERB->allpasstunings[3]);
	setbuf_allpass(tmdy_struct, &rev->allpassR[3], REVERB->allpasstunings[3] + stereospread);

	rev->allpassL[0].feedback = initialallpassfbk;
	rev->allpassR[0].feedback = initialallpassfbk;
	rev->allpassL[1].feedback = initialallpassfbk;
	rev->allpassR[1].feedback = initialallpassfbk;
	rev->allpassL[2].feedback = initialallpassfbk;
	rev->allpassR[2].feedback = initialallpassfbk;
	rev->allpassL[3].feedback = initialallpassfbk;
	rev->allpassR[3].feedback = initialallpassfbk;

	rev->wet = initialwet * scalewet;
	rev->damp = initialdamp * scaledamp;
	rev->width = initialwidth;
	rev->roomsize = initialroom * scaleroom + offsetroom;

	REVERB->revmodel_alloc_flag = 1;
}

static void free_revmodel(tmdy_struct_ex_t *tmdy_struct)
{
	int i;
	if(REVERB->revmodel != NULL) {
		for(i = 0; i < numcombs; i++)
		{
			if(REVERB->revmodel->combL[i].buf != NULL)
				free(REVERB->revmodel->combL[i].buf);
			if(REVERB->revmodel->combR[i].buf != NULL)
				free(REVERB->revmodel->combR[i].buf);
		}
		for(i = 0; i < numallpasses; i++)
		{
			if(REVERB->revmodel->allpassL[i].buf != NULL)
				free(REVERB->revmodel->allpassL[i].buf);
			if(REVERB->revmodel->allpassR[i].buf != NULL)
				free(REVERB->revmodel->allpassR[i].buf);
		}
		free(REVERB->revmodel);
	}
}

#if OPT_MODE != 0	/* fixed-point implementation */
#define do_allpass(_stream, _apbuf, _apsize, _apindex, _apfeedback) \
{ \
	_rev_bufout = _apbuf[_apindex];	\
	_rev_output = -_stream + _rev_bufout;	\
	_apbuf[_apindex] = _stream + imuldiv24(_rev_bufout, _apfeedback);	\
	if(++_apindex >= _apsize) {	\
		_apindex = 0;	\
	}	\
	_stream = _rev_output;	\
}

#define do_comb(_input, _output, _cbuf, _csize, _cindex, _cdamp1, _cdamp2, _cfs, _cfeedback)	\
{	\
	_rev_output = _cbuf[_cindex];	\
	_cfs = imuldiv24(_rev_output, _cdamp2) + imuldiv24(_cfs, _cdamp1);	\
	_cbuf[_cindex] = _input + imuldiv24(_cfs, _cfeedback);	\
	if(++_cindex >= _csize) {	\
		_cindex = 0;	\
	}	\
	_output += _rev_output;	\
}

static void do_freeverb(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count)
{
	int32 _rev_bufout, _rev_output;
	int32 i, k = 0;
	int32 outl, outr, input;
	revmodel_t *rev = REVERB->revmodel;
	comb *combL = rev->combL, *combR = rev->combR;
	allpass *allpassL = rev->allpassL, *allpassR = rev->allpassR;

	if(rev == NULL) {
		for (k = 0; k < count; k++)
		{
			REVERB->effect_buffer[k] = 0;
		}
		return;
	}

	for (k = 0; k < count; k++)
	{
		outl = outr = 0;
		input = REVERB->effect_buffer[k] + REVERB->effect_buffer[k + 1];
		REVERB->effect_buffer[k] = REVERB->effect_buffer[k + 1] = 0;

		for (i = 0; i < numcombs; i++) {
			do_comb(input, outl, combL[i].buf, combL[i].size, combL[i].index,
				combL[i].damp1i, combL[i].damp2i, combL[i].filterstore, combL[i].feedbacki);
			do_comb(input, outr, combR[i].buf, combR[i].size, combR[i].index,
				combR[i].damp1i, combR[i].damp2i, combR[i].filterstore, combR[i].feedbacki);
		}
		for (i = 0; i < numallpasses; i++) {
			do_allpass(outl, allpassL[i].buf, allpassL[i].size, allpassL[i].index, allpassL[i].feedbacki);
			do_allpass(outr, allpassR[i].buf, allpassR[i].size, allpassR[i].index, allpassR[i].feedbacki);
		}
		buf[k] += imuldiv24(outl, rev->wet1i) + imuldiv24(outr, rev->wet2i);
		buf[k + 1] += imuldiv24(outr, rev->wet1i) + imuldiv24(outl, rev->wet2i);
		++k;
	}
}
#else	/* floating-point implementation */
#define do_allpass(_stream, _apbuf, _apsize, _apindex, _apfeedback) \
{ \
	_rev_bufout = _apbuf[_apindex];	\
	_rev_output = -_stream + _rev_bufout;	\
	_apbuf[_apindex] = _stream + (_rev_bufout * _apfeedback);	\
	if(++_apindex >= _apsize) {	\
		_apindex = 0;	\
	}	\
	_stream = _rev_output;	\
}

#define do_comb(_input, _output, _cbuf, _csize, _cindex, _cdamp1, _cdamp2, _cfs, _cfeedback)	\
{	\
	_rev_output = _cbuf[_cindex];	\
	_cfs = (_rev_output * _cdamp2) + (_cfs * _cdamp1);	\
	_cbuf[_cindex] = _input + (_cfs * _cfeedback);	\
	if(++_cindex >= _csize) {	\
		_cindex = 0;	\
	}	\
	_output += _rev_output;	\
}

static void do_freeverb(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count)
{
	int32 _rev_bufout, _rev_output;
	int32 i, k = 0;
	int32 outl, outr, input;
	revmodel_t *rev = REVERB->revmodel;

	if(rev == NULL) {
		for (k = 0; k < count; k++)
		{
			REVERB->effect_buffer[k] = 0;
		}
		return;
	}

	for (k = 0; k < count; k++)
	{
		outl = outr = 0;
		input = REVERB->effect_buffer[k] + REVERB->effect_buffer[k + 1];
		REVERB->effect_buffer[k] = REVERB->effect_buffer[k + 1] = 0;

		for (i = 0; i < numcombs; i++) {
			do_comb(input, outl, rev->combL[i].buf, rev->combL[i].size, rev->combL[i].index,
				rev->combL[i].damp1, rev->combL[i].damp2, rev->combL[i].filterstore, rev->combL[i].feedback);
			do_comb(input, outr, rev->combR[i].buf, rev->combR[i].size, rev->combR[i].index,
				rev->combR[i].damp1, rev->combR[i].damp2, rev->combR[i].filterstore, rev->combR[i].feedback);
		}
		for (i = 0; i < numallpasses; i++) {
			do_allpass(outl, rev->allpassL[i].buf, rev->allpassL[i].size, rev->allpassL[i].index, rev->allpassL[i].feedback);
			do_allpass(outr, rev->allpassR[i].buf, rev->allpassR[i].size, rev->allpassR[i].index, rev->allpassR[i].feedback);
		}
		buf[k] += outl * rev->wet1 + outr * rev->wet2;
		buf[k + 1] += outr * rev->wet1 + outl * rev->wet2;
		++k;
	}
}
#endif	/* OPT_MODE != 0 */

void init_reverb(tmdy_struct_ex_t *tmdy_struct, int32 output_rate)
{
	REVERB->sample_rate = output_rate;
	memset(REVERB->reverb_status.high_val, 0, sizeof(REVERB->reverb_status.high_val));
	/* Only initialize freeverb if stereo output */
        /* Old non-freeverb must be initialized for mono reverb not to crash */
        if(!((TMDY_OUTPUT->play_mode)->encoding & PE_MONO) &&
	   TMDY_PLAYMIDI->opt_reverb_control == 3 || REVERB->opt_effect_quality >= 2) {
		alloc_revmodel(tmdy_struct);
		update_revmodel(tmdy_struct, REVERB->revmodel);
		init_revmodel(tmdy_struct, REVERB->revmodel);
		memset(REVERB->effect_buffer, 0, REVERB->effect_bufsize);
		memset(REVERB->direct_buffer, 0, REVERB->direct_bufsize);
		REVERB->REV_INP_LEV = fixedgain * REVERB->revmodel->wet;
	} else {
		REVERB->ta = 0; REVERB->tb = 0;
		REVERB->HPFL = 0; REVERB->HPFR = 0;
		REVERB->LPFL = 0; REVERB->LPFR = 0;
		REVERB->EPFL = 0; REVERB->EPFR = 0;
		REVERB->spt0 = 0; REVERB->spt1 = 0;
		REVERB->spt2 = 0; REVERB->spt3 = 0;

		rev_memset(REVERB->buf0_L); rev_memset(REVERB->buf0_R);
		rev_memset(REVERB->buf1_L); rev_memset(REVERB->buf1_R);
		rev_memset(REVERB->buf2_L); rev_memset(REVERB->buf2_R);
		rev_memset(REVERB->buf3_L); rev_memset(REVERB->buf3_R);

		memset(REVERB->effect_buffer, 0, REVERB->effect_bufsize);
		memset(REVERB->direct_buffer, 0, REVERB->direct_bufsize);

		if(output_rate > 65000) output_rate=65000;
		else if(output_rate < 4000)  output_rate=4000;

		REVERB->def_rpt0 = REVERB->rpt0 = REV_VAL0 * output_rate / 1000;
		REVERB->def_rpt1 = REVERB->rpt1 = REV_VAL1 * output_rate / 1000;
		REVERB->def_rpt2 = REVERB->rpt2 = REV_VAL2 * output_rate / 1000;
		REVERB->def_rpt3 = REVERB->rpt3 = REV_VAL3 * output_rate / 1000;

		REVERB->REV_INP_LEV = 1.0;

		REVERB->rpt0 = REVERB->def_rpt0 * REVERB->reverb_status.time_ratio;
		REVERB->rpt1 = REVERB->def_rpt1 * REVERB->reverb_status.time_ratio;
		REVERB->rpt2 = REVERB->def_rpt2 * REVERB->reverb_status.time_ratio;
		REVERB->rpt3 = REVERB->def_rpt3 * REVERB->reverb_status.time_ratio;
		while(!isprime(tmdy_struct, REVERB->rpt0)) REVERB->rpt0++;
		while(!isprime(tmdy_struct, REVERB->rpt1)) REVERB->rpt1++;
		while(!isprime(tmdy_struct, REVERB->rpt2)) REVERB->rpt2++;
		while(!isprime(tmdy_struct, REVERB->rpt3)) REVERB->rpt3++;
	}
}

void free_effect_buffers(tmdy_struct_ex_t *tmdy_struct)
{
	free_revmodel(tmdy_struct);
}

/*                                                        */
/* new implementation for insertion and variation effect. */
/*               (under construction...)                  */
/*                                                        */

/*! allocate new effect item and add it into the tail of effect list.
    EffectList *efc: pointer to the top of effect list.
    int8 type: type of new effect item.
    void *info: pointer to infomation of new effect item. */
EffectList *push_effect(tmdy_struct_ex_t *tmdy_struct, EffectList *efc, int8 type, void *info)
{
	EffectList *eft, *efn;
	if(type == EFFECT_NONE) {return NULL;}
	efn = (EffectList *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(EffectList));
	efn->type = type;
	efn->next_ef = NULL;
	efn->info = info;
	convert_effect(tmdy_struct, efn);

	if(efc == NULL) {
		efc = efn;
	} else {
		eft = efc;
		while(eft->next_ef != NULL) {
			eft = eft->next_ef;
		}
		eft->next_ef = efn;
	}
	return efc;
}

/*! process all items of effect list. */
void do_effect_list(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count, EffectList *ef)
{
	EffectList *efc = ef;
	if(ef == NULL) {return;}
	while(efc != NULL && efc->do_effect != NULL)
	{
		(*efc->do_effect)(tmdy_struct, buf, count, efc);
		efc = efc->next_ef;
	}
}

/*! free all items of effect list. */
void free_effect_list(tmdy_struct_ex_t *tmdy_struct, EffectList *ef)
{
	EffectList *efc, *efn;
	efc = ef;
	if(efc == NULL) {return;}
	do {
		efn = efc->next_ef;
		efc->do_effect = NULL;
		if(efc->info != NULL) {free(efc->info);}
		free(efc);
	} while ((efc = efn) != NULL);
}

/*! general purpose 2-band equalizer engine. */
void do_eq2(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count, EffectList *ef)
{
	struct InfoEQ2 *eq = (struct InfoEQ2 *)ef->info;
	if(count == MAGIC_INIT_EFFECT_INFO) {
		calc_lowshelf_coefs(tmdy_struct, eq->low_coef, eq->low_freq, eq->low_gain, (TMDY_OUTPUT->play_mode)->rate);
		calc_highshelf_coefs(tmdy_struct, eq->high_coef, eq->high_freq, eq->high_gain, (TMDY_OUTPUT->play_mode)->rate);
		memset(eq->low_val, 0, sizeof(eq->low_val));
		memset(eq->high_val, 0, sizeof(eq->high_val));
		return;
	}
	if(eq->low_gain != 0) {
		do_shelving_filter(tmdy_struct, buf, count, eq->low_coef, eq->low_val);
	}
	if(eq->high_gain != 0) {
		do_shelving_filter(tmdy_struct, buf, count, eq->high_coef, eq->high_val);
	}
}

/*! assign effect engine according to effect type. */
void convert_effect(tmdy_struct_ex_t *tmdy_struct, EffectList *ef)
{
	ef->do_effect = NULL;
	switch(ef->type)
	{
	case EFFECT_NONE:
		break;
	case EFFECT_EQ2:
		ef->do_effect = do_eq2;
		break;
	default:
		break;
	}

	(*ef->do_effect)(tmdy_struct, NULL, MAGIC_INIT_EFFECT_INFO, ef);
}



reverb_ex_t* new_reverb(tmdy_struct_ex_t *tmdy_struct){
	int i;
	reverb_ex_t* reverb_ex;

	reverb_ex=(reverb_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(reverb_ex_t));
	
reverb_ex->opt_effect_quality = 0;
	
reverb_ex->convert_effect=convert_effect;
reverb_ex->push_effect=push_effect;
reverb_ex->do_effect_list=do_effect_list;
reverb_ex->free_effect_list=free_effect_list;


reverb_ex->set_dry_signal=set_dry_signal;
reverb_ex->mix_dry_signal=mix_dry_signal;

/* channel by channel reverberation effect */
reverb_ex->do_reverb=do_reverb;
reverb_ex->do_ch_reverb=do_ch_reverb;
reverb_ex->set_ch_reverb=set_ch_reverb;
reverb_ex->do_mono_reverb=do_mono_reverb;
reverb_ex->init_reverb=init_reverb;
reverb_ex->reverb_rc_event=reverb_rc_event;
//reverb_ex->recompute_reverb_value=recompute_reverb_value;

/* channel by channel delay effect */
reverb_ex->do_ch_delay=do_ch_delay;
reverb_ex->set_ch_delay=set_ch_delay;
reverb_ex->init_ch_delay=init_ch_delay;

/* channel by channel chorus effect */
reverb_ex->do_ch_chorus=do_ch_chorus;
reverb_ex->set_ch_chorus=set_ch_chorus;
reverb_ex->init_chorus_lfo=init_chorus_lfo;
reverb_ex->init_ch_chorus=init_ch_chorus;

/* channel by channel equalizer */
reverb_ex->init_eq=init_eq;
reverb_ex->set_ch_eq=set_ch_eq;
reverb_ex->do_ch_eq=do_ch_eq;
reverb_ex->calc_lowshelf_coefs=calc_lowshelf_coefs;
reverb_ex->calc_highshelf_coefs=calc_highshelf_coefs;

/* insertion effect */
reverb_ex->init_insertion_effect=init_insertion_effect;
reverb_ex->do_insertion_effect=do_insertion_effect;

/* lowpass filter for system effects */
reverb_ex->do_lowpass_24db=do_lowpass_24db;
reverb_ex->calc_lowpass_coefs_24db=calc_lowpass_coefs_24db;

reverb_ex->free_effect_buffers=free_effect_buffers;
	
/**** private ****/
reverb_ex->effect_bufsize = sizeof(REVERB->effect_buffer);
reverb_ex->direct_bufsize = sizeof(REVERB->direct_buffer);
reverb_ex->delay_rpt0 = DELAY_BUFFER_SIZE;
reverb_ex->chorus_rpt0 = CHORUS_BUFFER_SIZE;
reverb_ex->sample_rate = 44100;
reverb_ex->REV_INP_LEV = 1.0;
for(i=0;i<numcombs;i++){
	reverb_ex->combtunings[i] =init_combtunings[i];
}
for(i=0;i<numallpasses;i++){
	reverb_ex->allpasstunings[i]=init_allpasstunings[i];
}

reverb_ex->revmodel = NULL;
reverb_ex->revmodel_alloc_flag = 0;

	return reverb_ex;
}
void destroy_reverb(reverb_ex_t* reverb){	
	free(reverb);
}

